# +===========================================================================+
# |   Copyright (c) 2003 Oracle Corporation, Redwood Shores, California, USA
# |                         All Rights Reserved
# |                        Applications Division
# +===========================================================================+
# |
# | FILENAME
# |	TechstackDB.pm
# |
# | DESCRIPTION
# |	TXK Techstack database package
# |
# | USAGE
# |	See TechstackDB.html
# |
# | PLATFORM
# |
# | NOTES
# |	Validation routines _getTempTablespace and _getUtlRecompState can be
# |	improved by executing a single PL/SQL block, rather than running
# |	multiple SQL statements, which results in multiple SQL*Plus sessions.
# |
# +===========================================================================+

# $Header: emll/sysman/admin/scripts/TXK/TechstackDB.pm /main/3 2008/11/16 06:44:07 ndutko Exp $

package TXK::TechstackDB;

@ISA = qw( TXK::Techstack);

#-------------------------------------
# Standard Modules
#-------------------------------------

use strict;
use English;
use Carp;

require 5.005;

#-------------------------------------
# Public Constants
#-------------------------------------

	#
	# Applications User & Sys names
	#

use constant APPS_USER_KEY	=> "appsuser"           ;
use constant APPS_DEFAULT_USER	=> "apps"               ;
use constant APPS_PASSWORD_KEY  => "appspass"           ;
use constant SYS_PASSWORD_KEY   => "syspass"            ;

	#
	# Multi-purpose constants
	#

use constant NULL			=> ""			;
use constant TXK_CURRENT_ERROR		=> "currentError"	;
use constant TXK_CURRENT_ERROR_MSG	=> "currentErrorMsg"	;
use constant TXK_OUTPUT_MESSAGE		=> "outputMsg"		;
use constant TXK_CONFIG_MESSAGE		=> "configMsg"		;
use constant TXK_VALIDATE_MESSAGE	=> "validateMsg"	;
use constant SQLPLUS_ERROR		=> "*ERROR*"		;

use constant UTL_RECOMP_MISSING		=> "-1"			;
use constant UTL_RECOMP_UNPATCHED	=> "-2"			;

#-------------------------------------
# Package Specific Modules
#-------------------------------------

use TXK::IO();
use TXK::Log();
use TXK::OSD();
use TXK::XML();
use TXK::Util();
use TXK::Error();
use TXK::Runtime();
use TXK::Process();
use TXK::SQLPLUS();
use TXK::RunScript();
use TXK::Techstack();

## -------------------------------------
## Package Variables
## -------------------------------------

my $PACKAGE_ID = "TXK::TechstackDB";

## -------------------------------------
## Object Keys
## -------------------------------------

my $HOST_NAME      	= "hostName"		;
my $APPL_TOP	   	= "APPL_TOP"		;
my $APPLTMP	   	= "APPLTMP"		;
my $TWO_TASK            = "TWO_TASK"		;
my $ADCONFIG_ENABLED	= "adconfigEnabled"	;
my $IS_DB_NODE		= "isDBNode"		;
my $IS_ADMIN_NODE	= "isAdminNode"		;
my $APPS_USER	  	= APPS_USER_KEY		;
my $APPS_PASSWORD       = APPS_PASSWORD_KEY	;
my $SYS_PASSWORD        = SYS_PASSWORD_KEY	;
my $VALIDATE_MESSAGE	= TXK_VALIDATE_MESSAGE	;

my $DB_VERSION		= "DBVersion"		;
my $DB_ONE_OFF          = "oneoff"              ;
my $DB_PATCH_SET        = "patchset"            ;
my $NODE_TYPE           = "nodetype"            ;
my $PROFILE_NAME        = "profilename"         ;
my $DIR_NAME            = "directory"           ;
my $FILE_NAME           = "filename"            ;

my $DB8174_BUG_HASH     = {  2664217  => { 
                                           HP_UX       =>  [ '.kql.o.pre2809848_8.1.7.4.0' ],
					   IBM_AIX     =>  [ '.kql.o.pre2809845_8.1.7.4.0' ],
				           UNIX_Alpha  =>  [ '.kql.o.pre2809844_8.1.7.4.0' ],
				           Linux       =>  [ '.kql.o.pre2809842_8.1.7.4.0' ],
				           Solaris     =>  [ '.kql.o.pre2809846_8.1.7.4.0' ]   
					 },

			     1805407  => { 
				           HP_UX       =>  [ '.prvtread.plb.pre2807208_8.1.7.4.0'],
				           IBM_AIX     =>  [ '.prvtread.plb.pre2807213_8.1.7.4.0'],
				           UNIX_Alpha  =>  [ '.prvtread.plb.pre2807212_8.1.7.4.0'],
				           Linux       =>  [ '.prvtread.plb.pre2807210_8.1.7.4.0'],
				           Solaris     =>  [ '.prvtread.plb.pre2807214_8.1.7.4.0']
					 },
			    2436624 => {
				           HP_UX       =>  [ '.pha.o.pre3382259_8.1.7.4.0'],
				           IBM_AIX     =>  [ '.pha.o.pre3379447_8.1.7.4.0'],
					   IBM_AIX64   =>  [ '.pha.o.pre3377026_8.1.7.4.0'],
				           UNIX_Alpha  =>  [ '.pha.o.pre3382260_8.1.7.4.0'],
				           Linux       =>  [ '.pha.o.pre3382258_8.1.7.4.0'],
				           Solaris     =>  [ '.pha.o.pre3382265_8.1.7.4.0'],
				           Solaris64   =>  [ '.pha.o.pre3382255_8.1.7.4.0']
				       }
			 };


my $DB_PARAMETERS		= "DBParameters"	;
my $RBS_OVER_50MB		= "RbsOver50MB"		;
my $TEMP_TABLESPACE_SIZE	= "TempTablespaceSize"	;
my $PORTAL_VERSION		= "PortalVersion"	;
my $LOGIN_SERVER_VERSION	= "LoginServerVersion"	;
my $MATERIALIZED_VIEW_PATCH	= "MaterializedView"	;
my $UTL_RECOMP_PATCH		= "UtlRecomp"		;
my $SYSTEM_TABLESPACE_SIZE      = "SystemTS"            ;
my $SYSTEM_TABLESPACE_AUTOEX    = "SystemTSAutoEx"      ;
my $VALID_NODE_TYPES            = [ "ADMIN", "WEB", "CP","FORMS" ];
my $DB_OBJECT_NAME              = "OBJECT_NAME";
my $DB_SCHEMA_OWNER             = "OWNER";

#
# The following patterns used for pattern matching can change
#

my $PATCHSET_PATTERN_1		= qr/^(\d+\.\d+\.\d+)\.(.*)$/	;
   ## This regexp assumes output has been reformatted to with set colsep '|'
   ##  and set wrap off. Uses non-greedy matching to strip end whitespace.
my $SQLPLUS_PATTERN_1		= qr/^\s*([^|]*?)\s*\|\s*(.*?)\s*$/	;

## -------------------------------------
## Package Methods - Public
## -------------------------------------

sub new;

sub txkGetDBVersion;		   # return full DB version
sub txkGetDBMinorVersion; 	   # return 1st-3rd digits of db version
sub txkGetDBPatchsetLevel; 	   # return 4th+ digits of db version
sub txkGetDBParameter;		   # return value of init.ora setting read in
				   #  via _getDBParameters
sub txkGetRBSOver50MB;		   # return number of rollback segs > 50MB
sub txkGetTempTablespaceSize;	   # return size of temp tablespace in MB
sub txkGetPortalVersion;	   # return Portal version
sub txkGetLoginServerVersion;	   # return Login Server version

sub txkCheckCPUCount;		   # verify parallel_max_servers >= cpu count
sub txkCheckMaterializedViewPatch; # verify patch 2328821 was applied
sub txkCheckUtlRecompExists; 	   # verify utl_recomp exists
sub txkCheckUtlRecompPatch; 	   # verify utl_recomp patch 2949941 exists
sub txkGetSystemTableSpace;        # return system table space
sub getSQLValue;                   # return result from 1 row, 1 column select
sub getSQLHash;                    # return multi-row select w/ key-value as cols
sub txkGetNoOfNodesForNodeType ;   # return no of nodes of input node type
sub txkDoesProfileOptionExist;     # returns true/false if the input profile option exist
sub txkGetProfileOptionValue;      # returns the site level profile option value
sub txkCheckFileInDBNode;          # checks whether the file exist on the db node
sub txkCheckForSystemTblSpc;       # checks whether it is required to check for Sys Table Space.
sub txkIsSystemTblSpcAutoExtensible; # checks whether the system is autoextensible
sub validateDBPassword;            # validates DB passsword given user/password/two_task
sub removeFNDPreferences;          # removes preferences using fnd.preference
sub retrieveFNDPreferences;        # retrieves preferences from fnd.preference
sub txkVerifyIES;
## -------------------------------------
## Package Methods - Private
## -------------------------------------

sub _getSQLValue;		# return result from 1 row, 1 column select
sub _getSQLHash;		# return multi-row select w/ key-value as cols

sub _getDBVersion; 		# store version string
sub _getDBParameters;		# store V$PARAMETER values
sub _getRBSOver50MB;	  	# store number of rollbacks over 50MB
sub _getTempTablespace; 	# store size in MB; undef if missing
sub _getPortalVersion;	  	# store version string
sub _getLoginServerVersion;	# store version string

sub _getMaterializedViewPatch; 	# store PATCHED/UNPATCHED; undef if missing
sub _getUtlRecompState;  	# store PATCHED/UNPATCHED; undef if missing
sub _getSystemTableSpace;       # store system table space size
sub _isSysTblSpcAutoEx;        

## -------------------------------------
## Constructor
## -------------------------------------

sub new
{

  my $type = $ARG[0];
  my $args = $ARG[1];

  my $self = TXK::Techstack->new($args);

  bless $self, $PACKAGE_ID ;

  my %INIT_OBJ = (
		  PACKAGE_IDENT   		=> $PACKAGE_ID		,
		  $DB_VERSION			=> undef		,
		  $DB_PARAMETERS		=>
		      {
		       job_queue_processes	=> undef,
		       undo_management		=> undef,
		       cpu_count		=> undef,
		       parallel_max_servers	=> undef,
		       _b_tree_bitmap_plans     => undef
		      }                                                 ,
		  $RBS_OVER_50MB		=> undef		,
		  $TEMP_TABLESPACE_SIZE		=> undef		,
		  $PORTAL_VERSION		=> undef		,
		  $LOGIN_SERVER_VERSION		=> undef		,
		  $MATERIALIZED_VIEW_PATCH	=> undef		,
		  $UTL_RECOMP_PATCH		=> undef                ,
		  $SYSTEM_TABLESPACE_SIZE       => undef                ,
		  $SYSTEM_TABLESPACE_AUTOEX     => undef
                 );

  foreach my $key (keys %INIT_OBJ)
  {
     $self->{$key} = $INIT_OBJ{$key};
  }

  return $self;
}

######################################
# validateDBPassword : validates DB passsword given user/password/two_task
######################################

sub validateDBPassword
{
    my $self  = $ARG[0];
    my $args  = $ARG[1];

    TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

    TXK::Util->isValidArgs({args=>$args, reqd=> ["user" ,"password", "two_task"]} );

    my $plus = TXK::SQLPLUS->new();
    
    $plus->setRuntime( { runtimeObj => $args->{"runtimeObj"} } ) 
	if ( exists $args->{"runtimeObj"} );

    $plus->abortOnError({ enable => TXK::Util::FALSE });

    my $rc = $plus->setConnectInfo({ user=>$args->{user}, password=>$args->{password},
				     two_task=>$args->{two_task} } );

    $plus->abortOnError({ enable => TXK::Util::TRUE });

    return $rc;
}

######################################
# removeFNDPreferences : removes preferences from FND
######################################

sub removeFNDPreferences
{
    my $self  = $ARG[0];
    my $args  = $ARG[1];

    TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

    my $key    = 'prefs';
    my $module = "module";

    TXK::Util->isValidArgs({args=>$args, reqd=> ["$key","$module"]} );

    my $sqlcmd = "BEGIN\n";

    my $pref_name;

    foreach $pref_name ( @{$args->{$key}} )
     {
	 $sqlcmd .= "fnd_preference.remove('#INTERNAL', \'$args->{$module}\',\'$pref_name\');\n";
     }

    $sqlcmd .= "EXCEPTION\n".
	        "  WHEN OTHERS THEN\n".
		"  null;\n".
		"  END;";

    my $retVal = $self->getSQLValue($sqlcmd);

    if ($retVal eq TXK::TechstackDB::SQLPLUS_ERROR)
     {
	 return TXK::Error::FAIL;
     }

    return TXK::Error::SUCCESS;
}

######################################
# retrieveFNDPreferences : returns a hash 
######################################

sub retrieveFNDPreferences
{
    my $self  = $ARG[0];
    my $args  = $ARG[1];

    TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

    my $key = 'prefs';
    my $module = "module";

    TXK::Util->isValidArgs({args=>$args, reqd=> ["$key","$module"]} );

    my $pref_hash = {};

    my $pref_name;

    my $sqlcmd=<<EOF;
DECLARE
    prefvalue varchar2(100);
BEGIN
 prefvalue := fnd_preference.get('#INTERNAL', \'$args->{$module}\','PREFNAME');
 DBMS_OUTPUT.PUT_LINE(prefvalue);
EXCEPTION
  WHEN OTHERS THEN
    null;
END;
EOF

    my $sqlcmd_enc=<<EOF;
DECLARE
    prefvalue varchar2(100);
BEGIN
 prefvalue := fnd_preference.eget('#INTERNAL', \'$args->{$module}\','PREFNAME','KEY');
 DBMS_OUTPUT.PUT_LINE(prefvalue);
EXCEPTION
  WHEN OTHERS THEN
    null;
END;
EOF

    foreach $pref_name ( @{$args->{$key}} )
     {
	 my $sqlcmd_exec  = $sqlcmd;

	 if ( exists $args->{'encryptkey'} and ( $args->{'encryptkey'} ne ""))
	  {
	      $sqlcmd_exec = $sqlcmd_enc ;	      
	      my $enc_key  = $args->{'encryptkey'};
	      $sqlcmd_exec =~s/KEY/$enc_key/g;
	  }

	 $sqlcmd_exec     =~ s/PREFNAME/$pref_name/g ;

	 my $retVal = $self->getSQLValue($sqlcmd_exec);	 

	 if ($retVal eq TXK::TechstackDB::SQLPLUS_ERROR)
	  {
	      return TXK::Error::FAIL;
	  }

	 $pref_hash->{$pref_name} = $retVal;
      }

    return $pref_hash;
}

######################################
# storeFNDPreferences :
######################################

sub storeFNDPreferences
{
    my $self  = $ARG[0];
    my $args  = $ARG[1];

    TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

    my $key = 'prefs';
    my $module = "module";

    TXK::Util->isValidArgs({args=>$args, reqd=> ["$key","$module"]} );

    my $sqlcmd = "BEGIN\n";

    my $pref_name;

    foreach $pref_name ( keys %{$args->{$key}} )
     {
	 my $pref_value = $args->{$key}->{$pref_name};

	 if ( exists $args->{'encryptkey'} and ( $args->{'encryptkey'} ne ""))
	  {
	      my $ekey = $args->{'encryptkey'};

	      $sqlcmd .= "fnd_preference.eput('#INTERNAL', \'$args->{$module}\',\'$pref_name\',\'$pref_value\',\'$ekey\');\n";
	  }
	 else
	  {
	      $sqlcmd .= "fnd_preference.put('#INTERNAL', \'$args->{$module}\',\'$pref_name\',\'$pref_value\');\n";
	  }
     }

    $sqlcmd .= "EXCEPTION\n".
	        "  WHEN OTHERS THEN\n".
		"  null;\n".
		"  END;";

    my $retVal = $self->getSQLValue($sqlcmd);

    if ($retVal eq TXK::TechstackDB::SQLPLUS_ERROR)
     {
	 return TXK::Error::FAIL;
     }

    return TXK::Error::SUCCESS;
}


## -------------------------------------
## getSQLValue : public api : return result from 1 row, 1 column select
## -------------------------------------


sub getSQLValue
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=> { args => $args}});

  return $self->_getSQLValue($args);

}

## -------------------------------------
## _getSQLValue : return result from 1 row, 1 column select
## -------------------------------------

sub _getSQLValue
{
  my $self = shift;
  my $sqlcmd = shift;

  my $appltmp  = TXK::OSD->getEnvVar({ name => "APPLTMP"});
  my $log_f     = $appltmp . "/" . "sqlplus.txt";

  if ( $self->_txkIsDBContext() )
   {
       my $tempDir  = $self->txkGetContextVar({ oavar => 's_temp' });
       $log_f       = $tempDir . "/" . "sqlplus.txt";
   }
  
  my $logfile   = TXK::OSD->trDirPathToBase($log_f);

  my $plus = TXK::SQLPLUS->new();
  $plus->setConnectInfo({ user=>$self->{$APPS_USER},
			  password=>$self->{$APPS_PASSWORD},
                          two_task=>$self->{$TWO_TASK} });
  $plus->clearCommand();
  $plus->addCommand("set serveroutput on");
  $plus->addCommand("set heading off");
  $plus->addCommand("set feedback off");
  $plus->addCommand("spool $logfile");
  $plus->addCommand($sqlcmd);
  $plus->addCommand("/");
  $plus->addCommand("spool off");
  my $rc = $plus->execute({ showCommand=>TXK::Util::FALSE,
			    showOutput =>TXK::Util::FALSE });

  if (! $rc)
    {
      TXK::Error->stop("General Error = " , $plus->getError() )
	  unless ( $plus->getErrorOutput() );

      TXK::Error->stop("SQLPLUS Error = " . $plus->getErrorOutput())
	  if ( $plus->getErrorOutput() );

      return SQLPLUS_ERROR;
    }

  my $io = TXK::IO->new();
  $io->open({ fileName => $logfile });
  my $io_ref = $io->getFileHandle();
  my @io_data = <$io_ref>;
  $io->close();

  my $fsys = TXK::FileSys->new();
  $fsys->rmfile( { fileName => $logfile } ) or TXK::Error->printMsg( $fsys->getError());

  my $retval;
  foreach (@io_data)
    {
      next if (/^\s*$/); # skip blank lines
      chomp;
      s/^\s*//;          # remove leading whitespace
      s/\s*$//;          # remove trailing whitespace
      $retval = $_;
    }

  return $retval;
}

## -------------------------------------
## getSQLHash : public api : return multi-row select w/ key-value as cols
## -------------------------------------

sub getSQLHash
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>{args=>$args}});

  return $self->_getSQLHash($args);
}

## -------------------------------------
## _getSQLHash : return multi-row select w/ key-value as cols
## -------------------------------------

sub _getSQLHash
{
  my $self = shift;
  my $sqlcmd = shift;

  my $appltmp  = TXK::OSD->getEnvVar({ name => "APPLTMP"});
  my $log_f     = $appltmp . "/" . "sqlplus.txt";
  if ( $self->_txkIsDBContext() )
   {
       my $tempDir  = $self->txkGetContextVar({ oavar => 's_temp' });
       $log_f       = $tempDir . "/" . "sqlplus.txt";
   }
  my $logfile   = TXK::OSD->trDirPathToBase($log_f);

  my $plus = TXK::SQLPLUS->new();
  $plus->setConnectInfo({ user=>$self->{$APPS_USER},
			  password=>$self->{$APPS_PASSWORD},
                          two_task=>$self->{$TWO_TASK} });
  $plus->clearCommand();
  $plus->addCommand("set serveroutput on");
  $plus->addCommand("set heading off");
  $plus->addCommand("set feedback off");
  $plus->addCommand("set wrap off");
  $plus->addCommand("set pagesize 9999");
  $plus->addCommand("set linesize 200");
  $plus->addCommand("set colsep '|'");
  $plus->addCommand("spool $logfile");
  $plus->addCommand($sqlcmd);
  $plus->addCommand("/");
  $plus->addCommand("spool off");
  my $rc = $plus->execute({ showCommand=>TXK::Util::FALSE,
			    showOutput =>TXK::Util::FALSE });

  if (! $rc)
    {
      TXK::Error->stop("General Error = " , $plus->getError() )
	  unless ( $plus->getErrorOutput() );

      TXK::Error->stop("SQLPLUS Error = " . $plus->getErrorOutput())
	  if ( $plus->getErrorOutput() );

      return SQLPLUS_ERROR;
    }

  my $io = TXK::IO->new();
  $io->open({ fileName => $logfile });
  my $io_ref = $io->getFileHandle();
  my @io_data = <$io_ref>;
  $io->close();

  my $fsys = TXK::FileSys->new();
  $fsys->rmfile( { fileName => $logfile } ) or TXK::Error->printMsg( $fsys->getError());

  my %retval;

  foreach (@io_data)
    {
      next if (/^\s*$/); # skip blank lines
      chomp;
      m/$SQLPLUS_PATTERN_1/;
      $retval{$1}=$2;
    }

  return %retval;
}

## -------------------------------------
## txkGetDBVersion : return full DB version
## -------------------------------------

sub txkGetDBVersion
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  if ( !defined ( $self->{$DB_VERSION} ))
   {
     $self->_getDBVersion($args);
   }

  return $self->{$DB_VERSION};
}


## -------------------------------------
## _getDBVersion : store version string
## -------------------------------------

sub _getDBVersion
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  my $sqlcmd="declare \n".
    	     "  dbms_version varchar2(10); \n".
	     "  compat varchar2(128); \n".
	     "begin \n".
	     "  dbms_utility.db_version ( version => dbms_version , compatibility => compat); \n".
	     "  dbms_output.put_line(dbms_version); \n".
	     "end;";

  my $dbver = $self->_getSQLValue($sqlcmd);
  if (! $dbver || ($dbver eq SQLPLUS_ERROR))
    {
      $self->txkSetError({
			  error_status  => TXK::Error::FAIL,
			  error_message => "Unable to determine database version."
			 });
      return TXK::Error::FAIL;

    }
  else
    {
      $self->{$DB_VERSION}=$dbver;
    }

  return $self->{$DB_VERSION};
}

## -------------------------------------
## txkGetRBSOver50MB : return number of rollback segs > 50MB
## -------------------------------------

sub txkGetRBSOver50MB
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  if ( !defined ( $self->{$RBS_OVER_50MB}) )
   {
     $self->_getRBSOver50MB($args);
   }

  return $self->{$RBS_OVER_50MB};
}


## -------------------------------------
## _getRBSOver50MB : store number of rollbacks over 50MB
## -------------------------------------

sub _getRBSOver50MB
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  my $sqlcmd="SELECT Count(rssize) FROM v\$rollstat a, v\$rollname b ".
             "WHERE a.usn=b.usn AND b.name NOT IN ('SYSTEM','system') ".
	     "AND rssize > 52428800";

  my $rbscount = $self->_getSQLValue($sqlcmd);
  if (((! $rbscount) && ($rbscount ne "0")) ||
      ($rbscount eq SQLPLUS_ERROR))
    {
      $self->txkSetError({
			  error_status   => TXK::Error::FAIL,
			  error_message => "Unable to query rollback segment information.",
			 });
      return TXK::Error::FAIL;
    }
  else
    {
      $self->{$RBS_OVER_50MB}=$rbscount;
    }

  return $self->{$RBS_OVER_50MB};
}


## -------------------------------------
## txkGetTempTablespaceSize : return size of temp tablespace in MB
## -------------------------------------

# Returns -1 if temp tablespace is not defined for apps user
sub txkGetTempTablespaceSize
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  if ( !defined ( $self->{$TEMP_TABLESPACE_SIZE}) )
   {
     $self->_getTempTablespace($args);
   }

  return $self->{$TEMP_TABLESPACE_SIZE};
}

## -------------------------------------
## _getTempTablespace : store size in MB; -1 if missing
## -------------------------------------

sub _getTempTablespace
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  my $sqlcmd="SELECT temporary_tablespace FROM dba_users ".
             "WHERE Upper(username)=Upper('".$self->{$APPS_USER}."')";

  my $tbspName = $self->_getSQLValue($sqlcmd);
  if ($tbspName eq SQLPLUS_ERROR)
    {
      $self->txkSetError({
			  error_status   => TXK::Error::FAIL,
			  error_message => "Unable to identify temporary tablespace.",
			 });
      return TXK::Error::FAIL;
    }
  if (! $tbspName)
    {
      $self->{$TEMP_TABLESPACE_SIZE}=-1;
      return $self->{$TEMP_TABLESPACE_SIZE};
    }

  $sqlcmd="SELECT Sum(bytes)/(1024*1024) FROM dba_temp_files ".
          "WHERE tablespace_name = '$tbspName'";

  my $tbspSize = $self->_getSQLValue($sqlcmd);
  if (((! $tbspSize) && ($tbspSize ne "0")) ||
      ($tbspSize eq SQLPLUS_ERROR))
    {
      $self->txkSetError({
			  error_status   => TXK::Error::FAIL,
			  error_message => "Unable to obtain temporary tablespace size.",
			 });
      return TXK::Error::FAIL;
    }
  else
    {
      $self->{$TEMP_TABLESPACE_SIZE}=$tbspSize;
    }

  return $self->{$TEMP_TABLESPACE_SIZE};
}

## -------------------------------------
## txkGetDBParameter : return v$parameter value read in via _getDBParameters
## -------------------------------------

sub txkGetDBParameter
{
  my $self=$ARG[0];
  my $args=$ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  if (! defined ($args->{'dbparamname'}))
    {
      $self->txkSetError({
			  error_status   => TXK::Error::FAIL,
			  error_message => "txkGetDBParameter requires the name of a database parameter to check. Please check the caller's syntax.",
			 });
      return TXK::Error::FAIL;
    }

  my $paramToQuery=$args->{'dbparamname'};

  # If not defined, query from v$parameters and add to hash
  if (! defined ($self->{$DB_PARAMETERS}->{$paramToQuery}) )
    {
      $self->_getDBParameters($args);
    }

  # If still not defined, then it's a variable we're not currently tracking
  # (_getDBParameters should be updated). We could force a separate query from
  # v$parameter but this makes troubleshooting easier by requiring a stable
  # list of known values to test in _getDBParameters.
  if (! defined ($self->{$DB_PARAMETERS}->{$paramToQuery}) )
    {
      $self->txkSetError({
			  error_status   => TXK::Error::FAIL,
			  error_message => "Parameter $paramToQuery is currently not tested. Please add to the list in _getDBParamters and try again.",
			 });
      return TXK::Error::FAIL;
    }

  return $self->{$DB_PARAMETERS}->{$paramToQuery};
}


## -------------------------------------
## _getDBParameters : query assorted values from v$paramter table
## -------------------------------------

sub _getDBParameters
{
  my $self=$ARG[0];
  my $args=$ARG[1];

  my @db_params = ('job_queue_processes','undo_management','cpu_count',
		   'parallel_max_servers','o7_dictionary_accessibility','_b_tree_bitmap_plans');

  foreach my $key ( @db_params)  # initialize the values in case some of them are not set in the db
   {
        $self->{$DB_PARAMETERS}->{$key}= '';
   }
                                           
  my $sqlcmd="SELECT lower(name),value FROM v\$parameter WHERE lower(name) in ('".
             join("','",@db_params).
             "')";

  my %params=$self->_getSQLHash($sqlcmd);

  if (! %params) {
    $self->txkSetError({
			error_status   => TXK::Error::FAIL,
			error_message => "Unable to query parameter table.",
		       });
    return TXK::Error::FAIL;
  }

  foreach my $key (keys %params)
    {
	$self->{$DB_PARAMETERS}->{$key}=$params{$key};
    }

  return TXK::Error::SUCCESS;
}

## -------------------------------------
## txkDoesProfileOptionExist : returns true/false for the input profile
## -------------------------------------

sub txkDoesProfileOptionExist
{
  my $self=$ARG[0];
  my $args=$ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  TXK::Util->isValidArgs({ args=>$args,reqd=> ["$PROFILE_NAME"]} );

  my $profile_name = $args->{$PROFILE_NAME};

  my $sqlcmd=<<EOF;
DECLARE
  RETVAL NUMBER;
BEGIN

  SELECT COUNT(PROFIlE_OPTION_NAME) INTO RETVAL
  FROM FND_PROFILE_OPTIONS
  WHERE PROFILE_OPTION_NAME ='#PROFILE_NAME#';

  DBMS_OUTPUT.PUT_LINE(RETVAL);
  EXCEPTION
    WHEN OTHERS THEN
    NULL ;
END;
EOF

  $sqlcmd =~ s/#PROFILE_NAME#/$profile_name/g;
  
  my $no_of_options = $self->_getSQLValue($sqlcmd);
    
  return TXK::Util::TRUE if ( $no_of_options > 0 );

  return TXK::Util::FALSE;
}

## -------------------------------------
## txkGetProfileOptionValue : returns the profile option value at the site level
## -------------------------------------

sub txkGetProfileOptionValue
{
  my $self=$ARG[0];
  my $args=$ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  TXK::Util->isValidArgs({ args=>$args,reqd=> ["$PROFILE_NAME"]} );

  my $profile_name = $args->{$PROFILE_NAME};

  return unless ( $self->txkDoesProfileOptionExist({ $PROFILE_NAME => $profile_name }) );

  my $sqlcmd=<<EOF;
DECLARE
  MY_PROFILE_VALUE VARCHAR2(240);
BEGIN

  SELECT PROFILE_OPTION_VALUE INTO MY_PROFILE_VALUE
  FROM FND_PROFILE_OPTION_VALUES
  WHERE (APPLICATION_ID, PROFILE_OPTION_ID) IN (
		SELECT APPLICATION_ID, PROFILE_OPTION_ID
		FROM   FND_PROFILE_OPTIONS
                WHERE  APPLICATION_ID = 0
                AND    PROFILE_OPTION_NAME = UPPER('#PROFILE_NAME#'))
  AND LEVEL_ID = 10001;

  DBMS_OUTPUT.PUT_LINE(MY_PROFILE_VALUE);
  EXCEPTION
    WHEN OTHERS THEN
    NULL ;
END;
EOF

  $sqlcmd =~ s/#PROFILE_NAME#/$profile_name/g;

  my $profile_value = $self->_getSQLValue($sqlcmd);
    
  return $profile_value;

}

## -------------------------------------
## txkCheckFileInDBNode : checks whether a file exists in database node
## -------------------------------------

sub txkCheckFileInDBNode
{
  my $self=$ARG[0];
  my $args=$ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  TXK::Util->isValidArgs({ args=>$args,reqd=> ["$DIR_NAME","$FILE_NAME"]} );

  my $dir = $args->{$DIR_NAME};

  my $file = $args->{$FILE_NAME};

  my $dropSql=<<EOF;
BEGIN
  BEGIN
   EXECUTE IMMEDIATE 'DROP DIRECTORY TXK_TECHSTACKDB_DIR';
   EXCEPTION 
    WHEN OTHERS THEN NULL;
  END;
  BEGIN
   EXECUTE IMMEDIATE 'DROP TABLE TXK_TECHSTACKDB_TABLE';
   EXCEPTION 
    WHEN OTHERS THEN NULL;
   END;
END;
EOF

  $self->_getSQLValue($dropSql);

  my $createSql=<<EOF;
CREATE OR REPLACE DIRECTORY TXK_TECHSTACKDB_DIR AS '#DIR_NAME#';
CREATE TABLE TXK_TECHSTACKDB_TABLE ( FILENAME BFILE );
DELETE FROM TXK_TECHSTACKDB_TABLE;
INSERT INTO TXK_TECHSTACKDB_TABLE VALUES ( BFILENAME ( 'TXK_TECHSTACKDB_DIR', '#FILE_NAME#' ) );
  COMMIT;
EOF

  $createSql =~ s/#DIR_NAME#/$dir/g;

  $createSql =~ s/#FILE_NAME#/$file/g;

  $self->_getSQLValue($createSql);

  my $sqlcmd=<<EOF;
DECLARE
 FILE BFILE;
 EXIST NUMBER;
BEGIN
  SELECT FILENAME INTO FILE FROM TXK_TECHSTACKDB_TABLE;
  EXIST :=  DBMS_LOB.FILEEXISTS(FILE);
  DBMS_OUTPUT.PUT_LINE ( 'FILE EXISTS:' || EXIST );
  EXCEPTION 
   WHEN OTHERS
   THEN DBMS_OUTPUT.PUT_LINE ( 'DOES NOT EXIST' );
END;
EOF

  my $file_exists = $self->_getSQLValue($sqlcmd);

  $self->_getSQLValue($dropSql);

  return TXK::Util::TRUE if ( $file_exists =~ m/FILE EXISTS:1/ );

  return TXK::Util::FALSE;
}


## -------------------------------------
## txkGetNoOfNodesForNodeType : retrieves no. of nodes of a particular
##                              type from fnd_nodes table
## -------------------------------------

sub txkGetNoOfNodesForNodeType 
{
  my $self=$ARG[0];
  my $args=$ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  TXK::Util->isValidArgs({ args=>$args,reqd=> ["$NODE_TYPE"]} );

  my $node_type;

  my $valid = TXK::Util::FALSE;

  foreach $node_type ( @{$VALID_NODE_TYPES} )
   {
       if ( $args->{$NODE_TYPE} =~ m/^$node_type$/i )
        {
	    $valid = TXK::Util::TRUE;
	}
   }

  TXK::Error->stop("Invalid node type: $args->{$NODE_TYPE}") unless ( $valid );

  $node_type = $args->{$NODE_TYPE};

  my $sqlcmd=<<EOF;
DECLARE
  NO_OF_NODES NUMBER;
BEGIN

  SELECT COUNT(NODE_NAME) INTO NO_OF_NODES 
  FROM FND_NODES WHERE SUPPORT_#TYPE#='Y' 
  AND ( STATUS IS NULL OR STATUS IN ('y','Y') )
  GROUP BY SUPPORT_#TYPE#;

  DBMS_OUTPUT.PUT_LINE(NO_OF_NODES);
  EXCEPTION
    WHEN OTHERS THEN
    NULL ;
END;
EOF

  $sqlcmd =~ s/#TYPE#/$node_type/g;
  
  my $no_of_nodes = $self->_getSQLValue($sqlcmd);

  return $no_of_nodes;
}

## -------------------------------------
## txkGetOWA_UTILVersion : retrieves the own_util version
## -------------------------------------

sub txkGetOWA_UTILVersion
{
  my $self=$ARG[0];
  my $args=$ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  my $sqlcmd=<<EOF;
DECLARE
  OWA_UTIL_VERSION VARCHAR2(30);
BEGIN

  OWA_UTIL_VERSION := 'WebDB2.5';
  EXECUTE IMMEDIATE 'SELECT OWA_UTIL.GET_VERSION  FROM DUAL' INTO OWA_UTIL_VERSION ;

  DBMS_OUTPUT.PUT_LINE(OWA_UTIL_VERSION);
  EXCEPTION
    WHEN OTHERS THEN
    DBMS_OUTPUT.PUT_LINE(OWA_UTIL_VERSION);
END;
EOF

  my $owa_ver = $self->_getSQLValue($sqlcmd);

  return $owa_ver;
}

## -------------------------------------
## txkCheckDBObject : checks whether a DB object is present
## takes object_name and owner as the parameter
## returns true if the count(*) is greater than 0
## -------------------------------------

sub txkCheckDBObject
{
  my $self=$ARG[0];
  my $args=$ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  TXK::Util->isValidArgs({ args=>$args,reqd=> ["$DB_OBJECT_NAME","$DB_SCHEMA_OWNER"]} );  

  my $sqlcmd=<<EOF;
DECLARE
  NO_OF_ROWS VARCHAR2(30);
BEGIN

  SELECT COUNT(*) INTO NO_OF_ROWS FROM DBA_OBJECTS WHERE OBJECT_NAME=upper('put_object_name') AND OWNER=upper('put_owner');

  DBMS_OUTPUT.PUT_LINE(NO_OF_ROWS);
  EXCEPTION
    WHEN OTHERS THEN
    NULL ;
END;
EOF

  $sqlcmd =~ s/put_object_name/$args->{$DB_OBJECT_NAME}/g;
  $sqlcmd =~ s/put_owner/$args->{$DB_SCHEMA_OWNER}/g;

  my $no_rows = $self->_getSQLValue($sqlcmd);

  return "TRUE" if ($no_rows > 0);
  
  return "FALSE";
}

## -------------------------------------
## txkIsSystemTblSpcAutoExtensible : checks if System table space is autoextensible
## -------------------------------------

sub txkIsSystemTblSpcAutoExtensible
{
  my $self=$ARG[0];
  my $args=$ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  if ( ! defined $self->{$SYSTEM_TABLESPACE_AUTOEX} )
   {
       $self->_isSysTblSpcAutoEx($args);
   }

  return $self->{$SYSTEM_TABLESPACE_AUTOEX};
}

## -------------------------------------
## _isSysTblSpcAutoEx : checks if System table space is autoextensible
## -------------------------------------

sub _isSysTblSpcAutoEx
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  my $sqlcmd=<<EOF;
DECLARE
  AUTO_EXT_FILES NUMBER;
BEGIN
    SELECT COUNT(*) INTO AUTO_EXT_FILES FROM  DBA_DATA_FILES 
    WHERE TABLESPACE_NAME = 'SYSTEM' AND AUTOEXTENSIBLE='YES';
  DBMS_OUTPUT.PUT_LINE(AUTO_EXT_FILES);
  EXCEPTION
    WHEN OTHERS THEN
    NULL ;
END;
EOF

  my $noOfAutoExtFiles = $self->_getSQLValue($sqlcmd);

  if ($noOfAutoExtFiles eq SQLPLUS_ERROR)
    {
      $self->txkSetError({
			  error_status   => TXK::Error::FAIL,
			  error_message => "Unable to obtain System table space version.",
			 });
      return TXK::Error::FAIL;
    }

  $self->{$SYSTEM_TABLESPACE_AUTOEX}= TXK::Util::FALSE ;

  if ( $noOfAutoExtFiles > 0 )
   {
       $self->{$SYSTEM_TABLESPACE_AUTOEX}= TXK::Util::TRUE ;
   }
}

## -------------------------------------
## txkCheckForSystemTblSpc : return FALSE if Sys table space is autoextensible
## -------------------------------------

sub txkCheckForSystemTblSpc
{
  my $self=$ARG[0];
  my $args=$ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  return TXK::Util::FALSE if ( $self->txkIsSystemTblSpcAutoExtensible() eq TXK::Util::TRUE );

  return TXK::Util::TRUE;
}

## -------------------------------------
## txkGetSystemTableSpace : retrieves the System Table Space size
## -------------------------------------

sub txkGetSystemTableSpace
{
  my $self=$ARG[0];
  my $args=$ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  if ( ! defined ( $self->{$SYSTEM_TABLESPACE_SIZE} ) )
   {
       $self->_getSystemTableSpace();
   }

  return $self->{$SYSTEM_TABLESPACE_SIZE} ;
}

## -------------------------------------
## _getSystemTSSize : store System table space
## -------------------------------------

sub _getSystemTableSpace
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  my $sqlcmd=<<EOF;
DECLARE
  SYS_TABSPACE_SIZE NUMBER;
BEGIN

  SELECT SUM(BYTES)/(1024*1024) INTO SYS_TABSPACE_SIZE
  FROM DBA_DATA_FILES WHERE TABLESPACE_NAME=UPPER('system');

  DBMS_OUTPUT.PUT_LINE(SYS_TABSPACE_SIZE);
  EXCEPTION
    WHEN OTHERS THEN
    NULL ;
END;
EOF

  my $systemTS = $self->_getSQLValue($sqlcmd);

  if ($systemTS eq SQLPLUS_ERROR)
    {
      $self->txkSetError({
			  error_status   => TXK::Error::FAIL,
			  error_message => "Unable to obtain System table space version.",
			 });
      return TXK::Error::FAIL;
    }

  $self->{$SYSTEM_TABLESPACE_SIZE}=$systemTS;

}


## -------------------------------------
## txkGetPortalVersion : return Portal version
## -------------------------------------

sub txkGetPortalVersion
{
  my $self=$ARG[0];
  my $args=$ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  if (! defined ($self->{$PORTAL_VERSION}) )
    {
      $self->_getPortalVersion($args);
    }

  return $self->{$PORTAL_VERSION};
}


## -------------------------------------
## _getPortalVersion : store version string
## -------------------------------------

sub _getPortalVersion
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  my $sqlcmd=<<EOF;
DECLARE
  portal_user      VARCHAR2(64);
  portal_user_name VARCHAR2(64);
  portal_ver       VARCHAR2(10);
BEGIN
  EXECUTE IMMEDIATE 'SELECT fnd_oracle_schema.getouvalue(''PORTAL'') FROM dual' INTO portal_user;
  BEGIN
    EXECUTE IMMEDIATE 'SELECT user_name FROM fnd_user WHERE user_name LIKE upper('''||portal_user||''')' INTO portal_user_name;
    IF portal_user IS NOT NULL THEN
      EXECUTE IMMEDIATE 'SELECT version FROM '||portal_user||'.wwc_version\$' INTO portal_ver;
      dbms_output.put_line(portal_ver);
    END IF;
  EXCEPTION
    WHEN NO_DATA_FOUND THEN
      null;
  END;
EXCEPTION
  WHEN OTHERS THEN
    null;
END;
EOF

  my $portalVer = $self->_getSQLValue($sqlcmd);
  if ($portalVer eq SQLPLUS_ERROR)
    {
      $self->txkSetError({
			  error_status   => TXK::Error::FAIL,
			  error_message => "Unable to obtain Portal version.",
			 });
      return TXK::Error::FAIL;
    }
  $self->{$PORTAL_VERSION}=$portalVer;
}

## -------------------------------------
## txkGetLoginServerVersion : return Login Server version (@@ OID also?)
## -------------------------------------

sub txkGetLoginServerVersion
{
  my $self=$ARG[0];
  my $args=$ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  if (! defined ($self->{$LOGIN_SERVER_VERSION}) )
    {
      $self->_getLoginServerVersion($args);
    }

  return $self->{$LOGIN_SERVER_VERSION};
}


## -------------------------------------
## _getLoginServerVersion : store version string
## -------------------------------------

sub _getLoginServerVersion
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  my $sqlcmd=<<EOF;
DECLARE
  sso_user      VARCHAR2(64);
  sso_user_name VARCHAR2(64);
  sso_ver       VARCHAR2(10);
BEGIN
  EXECUTE IMMEDIATE 'SELECT fnd_oracle_schema.getouvalue(''LOGINSERVER'') FROM dual' INTO sso_user;
  BEGIN
    EXECUTE IMMEDIATE 'SELECT user_name FROM fnd_user WHERE user_name LIKE upper('''||sso_user||''')' INTO sso_user_name;
    IF sso_user IS NOT NULL THEN
      EXECUTE IMMEDIATE 'SELECT version FROM '||sso_user||'.wwc_version\$' INTO sso_ver;
      dbms_output.put_line(sso_ver);
    END IF;
  EXCEPTION
    WHEN NO_DATA_FOUND THEN
      null;
  END;
EXCEPTION
  WHEN OTHERS THEN
    null;
END;
EOF

  my $loginVer = $self->_getSQLValue($sqlcmd);
  if ($loginVer eq SQLPLUS_ERROR)
    {
      $self->txkSetError({
			  error_status   => TXK::Error::FAIL,
			  error_message => "Unable to obtain Login Server version.",
			 });
      return TXK::Error::FAIL;
    }
  $self->{$LOGIN_SERVER_VERSION}=$loginVer;

}


## -------------------------------------
## txkGetDBMinorVersion : 1st-3rd digits of db version
## -------------------------------------

sub txkGetDBMinorVersion
{
  my $self=$ARG[0];
  my $args=$ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  if (! defined ( $self->{$DB_VERSION} ))
    {
      $self->_getDBVersion($args);
    }
  (my $minorversion=$self->{$DB_VERSION})=~s/$PATCHSET_PATTERN_1/$1/;
  return $minorversion;
}


## -------------------------------------
## txkGetDBPatchsetLevel : return 4th+ digits
## -------------------------------------

sub txkGetDBPatchsetLevel
{
  my $self=$ARG[0];
  my $args=$ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  if (! defined ( $self->{$DB_VERSION} ))
    {
      $self->_getDBVersion($args);
    }

  (my $patchsetlevel=$self->{$DB_VERSION})=~s/$PATCHSET_PATTERN_1/$2/;
  return $patchsetlevel;
}

## -------------------------------------
## txkCheckCPUCount : verify parallel_max_servers >= cpu count
## -------------------------------------

# returns failure if below minimum value, warning if below recommended value
sub txkCheckCPUCount
{
  my $self=$ARG[0];
  my $args=$ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  if (! defined ($self->{$DB_PARAMETERS}->{'cpu_count'}) ||
      ! defined ($self->{$DB_PARAMETERS}->{'parallel_max_servers'}) )
    {
      $self->_getDBParameters($args);
    }

  my $cpus   = $self->{$DB_PARAMETERS}->{'cpu_count'};
  my $parmax = $self->{$DB_PARAMETERS}->{'parallel_max_servers'};

  if ($parmax < $cpus)
    {
      $self->{$VALIDATE_MESSAGE}.="\n[FAIL]\t The parallel_max_servers parameter is set to: $parmax\n\t".
	" You must increase the parallel_max_servers setting on your\n\t".
	" database to at least $cpus. For optimal performance, we recommend\n\t".
	" that you set this parameter to at least ".(2*$cpus).".\n";
      return TXK::Error::FAIL;
    }
  elsif ($parmax < 2*$cpus)
    {
      $self->{$VALIDATE_MESSAGE}.="\n[WARN]\t The parallel_max_servers parameter is set to: $parmax\n\t".
	" For optimal performance, we recommend that you set this parameter\n\t".
	" to at least ".(2*$cpus).".\n";
      return TXK::Error::WARNING;
    }
  else
    {
      $self->{$VALIDATE_MESSAGE}.="\n[PASS]\t The parallel_max_servers parameter is set to: $parmax\n";
      return TXK::Error::SUCCESS;
    }
}


## -------------------------------------
## txkCheckMaterializedViewPatch : verify patch 2328821 was applied
## -------------------------------------

sub txkCheckMaterializedViewPatch
{
  my $self=$ARG[0];
  my $args=$ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  if (! defined ($self->{$MATERIALIZED_VIEW_PATCH}) )
    {
      $self->_getMaterializedViewPatch($args);
    }

  if ($self->{$MATERIALIZED_VIEW_PATCH})
    {
      $self->{$VALIDATE_MESSAGE}.="\n[PASS]\t Materialized view patch 2328821 has been applied.\n";
      return TXK::Error::SUCCESS;
    }
  else
    {
      $self->{$VALIDATE_MESSAGE}.="\n[FAIL]\t Cannot confirm materialized view patch has been applied.\n\t".
	" 11.5.10 Maintenance Pack requires that materialized view patch\n\t".
	" 2328821 be applied to this database. Please refer to My Oracle Support\n\t".
	" Note 237371.1.\n";
      return TXK::Error::FAIL;
    }
}


## -------------------------------------
## _getMaterializedViewPatch : store PATCHED/UNPATCHED; undef if missing
## -------------------------------------

sub _getMaterializedViewPatch
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  my $dbver = $self->txkGetDBVersion();
  # only need to check for v9.2 and later
  if ($dbver < 9.2)
    {
      $self->{$MATERIALIZED_VIEW_PATCH}=TXK::Util::TRUE;
      return $self->{$MATERIALIZED_VIEW_PATCH};
    }

  my $sqlcmd="SELECT Count(column_name) FROM all_ind_columns ".
             "WHERE index_owner='SYS' AND index_name='I_SUMKEY\$_1'";

  my $colCount = $self->_getSQLValue($sqlcmd);

  if ($colCount eq SQLPLUS_ERROR)
    {
      $self->txkSetError({
			  error_status   => TXK::Error::FAIL,
			  error_message => "Unable to test for materialized view patch.",
			 });
      return TXK::Error::FAIL;
    }

  if ($colCount == 4)
    {
      $self->{$MATERIALIZED_VIEW_PATCH}=TXK::Util::TRUE;
    }
  else
    {
      $self->{$MATERIALIZED_VIEW_PATCH}=TXK::Util::FALSE;
    }

  return $self->{$MATERIALIZED_VIEW_PATCH};
}

## -------------------------------------
## txkCheckUtlRecompExists: verify utl_recomp exists
## -------------------------------------
# For self-diagnostics, call txkCheckUtlRecompPatch instead of this one.
sub txkCheckUtlRecompExists
{
  my $self=$ARG[0];
  my $args=$ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  if (! defined ($self->{$UTL_RECOMP_PATCH}) )
    {
      $self->_getUtlRecompState($args);
    }

  if (($self->{$UTL_RECOMP_PATCH} eq UTL_RECOMP_MISSING) ||
      ! $self->{$UTL_RECOMP_PATCH} )
    {
      $self->{$VALIDATE_MESSAGE}.="\n[FAIL]\t UTL_RECOMP package is not found in the database.\n\t".
	" This package will improve AutoPatch performance and is required\n\t".
	" for the 11.5.10 Maintenance Pack. Please install this package by\n\t".
	" running the following command from your database ORACLE_HOME:\n\t".
	"   sqlplus sys/<sys_pwd> \@\$ORACLE_HOME/rdbms/admin/utlrcmp.sql\n";
      return TXK::Error::FAIL;
    }

  return TXK::Error::SUCCESS;
}


## -------------------------------------
## txkCheckUtlRecompPatch : verify utl_recomp patch 2949941 exists
## -------------------------------------
# This is the only routine that needs to be called if the return value is
# not being checked.
sub txkCheckUtlRecompPatch
{
  my $self=$ARG[0];
  my $args=$ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  return if (! $self->txkCheckUtlRecompExists);

  if ($self->{$UTL_RECOMP_PATCH} eq UTL_RECOMP_UNPATCHED)
    {
      $self->{$VALIDATE_MESSAGE}.="\n[FAIL]\t UTL_RECOMP package exists, but patch 2949941 has not been applied.\n\t".
	" Please download patch 2949941 from My Oracle Support and apply it\n\t".
	" to this database.\n";
      return TXK::Error::FAIL;
    }
  else
    {
      $self->{$VALIDATE_MESSAGE}.="\n[PASS]\t UTL_RECOMP package exists and is at the right patch level.\n";
    }

  return TXK::Error::SUCCESS;
}


## -------------------------------------
## _getUtlRecompState : store PATCHED/UNPATCHED; undef if missing
## -------------------------------------

sub _getUtlRecompState
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  my $sqlcmd="SELECT 'OK' FROM dba_source WHERE name='UTL_RECOMP' ".
             "AND owner = 'SYS' AND type = 'PACKAGE BODY'";

  my $tableExists = $self->_getSQLValue($sqlcmd);

  if ($tableExists eq SQLPLUS_ERROR)
    {
      $self->txkSetError({
			  error_status   => TXK::Error::FAIL,
			  error_message => "Unable to test for utl_recomp package.",
			 });
      return TXK::Error::FAIL;
    }

  if (! $tableExists)
    {
      $self->{$UTL_RECOMP_PATCH}=UTL_RECOMP_MISSING;
      return $self->{$UTL_RECOMP_PATCH};
    }

  my $dbver = $self->txkGetDBVersion();
  # only need to check for patch in v9.2 ; not sure about 10g, though @@
  if ($dbver < 9.2)
    {
      $self->{$UTL_RECOMP_PATCH}=TXK::Util::TRUE;
      return $self->{$UTL_RECOMP_PATCH};
    }

  $sqlcmd="SELECT 'OK' FROM dba_source WHERE name='UTL_RECOMP' ".
          "AND owner = 'SYS' AND type = 'PACKAGE BODY' ".
	  "AND 0 < Instr(text,'dbms_rule.evaluate')";

  my $patchExists = $self->_getSQLValue($sqlcmd);

  if ($patchExists eq SQLPLUS_ERROR)
    {
      $self->txkSetError({
			  error_status   => TXK::Error::FAIL,
			  error_message => "Unable to test for utl_recomp patch.",
			 });
      return TXK::Error::FAIL;
    }

  if (! $patchExists)
    {
      $self->{$UTL_RECOMP_PATCH}=UTL_RECOMP_UNPATCHED;
    }
  else
    {
      $self->{$UTL_RECOMP_PATCH}=TXK::Util::TRUE;
    }

  return $self->{$UTL_RECOMP_PATCH};
}

## ------------------------------------------------------
## txkCheckDBPSVerFromInv
## input params is hash containing 'patchset'
## ------------------------------------------------------


sub txkCheckDBPSVerFromInv
{
    my $self = $ARG[0];

    my $args = $ARG[1];

    TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

    TXK::Util->isValidArgs({ args=>$args,reqd=> ["$DB_PATCH_SET"]} );
    
    my $patch_set = $args->{$DB_PATCH_SET};

    my $oracle_home = $self->_getDBOracleHome();

    my $fsys = TXK::FileSys->new();
    
    my $inventory_file = TXK::OSD->trDirPathToBase ( "$oracle_home"."/inventory/ContentsXML/comps.xml" );

    $fsys->access( { fileName => $inventory_file,
		     type     => TXK::FileSys::FILE,
		     checkMode=> TXK::FileSys::READ_ACCESS } )
	or ( TXK::Error->printMsg($fsys->getError()) and return TXK::Util::FALSE );
    
    my $io = TXK::IO->new();

    $io->open ( { fileName => $inventory_file,
		  mode     => TXK::IO::READ } );
    
    my $io_file_handle = $io->getFileHandle();

    my @io_data = <$io_file_handle>;

    $io->close();

    return TXK::Util::TRUE if ( scalar ( grep ( /^<PATCHSET NAME=\"oracle.server\" VER=\"$patch_set/, @io_data ) ) > 0 );

    return TXK::Util::FALSE;
}

## ------------------------------------------------------
## txkCheckDBOneoff - checks for DB one-off
## input params is hash containing 'dbver' and 'oneoff'
## ------------------------------------------------------

sub txkCheckDBOneOff
{
    my $self = $ARG[0];

    my $args = $ARG[1];

    TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

    TXK::Util->isValidArgs({args=>$args,
			    reqd=> ["$DB_VERSION","$DB_ONE_OFF"]} );

    my $platform = TXK::OSD->getAutoConfigName();    

    my $oneoff = $args->{$DB_ONE_OFF} ;

    # just add the PSE number if it has to the base bug as well

    $oneoff = "$oneoff\-$args->{$platform}" if ( exists $args->{$platform} );

    if ( $args->{$DB_VERSION} =~ m/^db817/ )
     {
	return $self->_txkCheckDB817OneOff($oneoff);
     }
    elsif ( $args->{$DB_VERSION} =~ m/^db920/ )
     {
	return $self->_txkCheckDB920OneOff($oneoff);	 
     }
    elsif ( $args->{$DB_VERSION} =~ m/^db101/ )
     {
	return $self->_txkCheckDB101OneOff($oneoff);	 
     }

    return TXK::Util::FALSE;
}


## ------------------------------------------------------
## _txkCheckDB817Oneoff - checks for DB one-off for 8.1.7
## input params is oneoff
## ------------------------------------------------------

sub _txkCheckDB817OneOff 
{
    my $self = $ARG[0];

    my $oneoff = $ARG[1];

    my $platform = TXK::OSD->getAutoConfigName();

    print ("TXK::Techstack::_txkCheckDB817OneOff: platform : $platform : oneoff : $oneoff\n");

    return TXK::Util::FALSE unless ( ( exists $DB8174_BUG_HASH->{$oneoff} ) and
				     ( exists $DB8174_BUG_HASH->{$oneoff}->{$platform} ) );

    my $fileList = $DB8174_BUG_HASH->{$oneoff}->{$platform};

    my $file;

    print ("TXK::Techstack::_txkCheckDB817OneOff: checking platform : $platform : oneoff : $oneoff\n");

    my $oracle_home = $self->_getDBOracleHome();

    my $fsys = TXK::FileSys->new();

    foreach $file ( @$fileList )
     {
	 $file = TXK::OSD->trDirPathToBase("$oracle_home"."/lib/"."$file") ;
	 
	 print ("TXK::Techstack::_txkCheckDB817OneOff: checking $file \n");

	 $fsys->access( { fileName => $file,
			  type     => TXK::FileSys::FILE,
			  checkMode=> TXK::FileSys::READ_ACCESS } )
	     or return TXK::Util::FALSE;
     }

    return TXK::Util::TRUE;
}

## ------------------------------------------------------
## _txkCheckDB920Oneoff - checks for DB one-off for 9.2.0
## input params is oneoff
## There may be multiple oneoffs listed with "-" as delimeter
## This is required to support 32/64 bit platform support
##
## under each "$ORACLE_HOME/.patch_storage" directory open up
## the "inventory" which contains base bug and the PSE bug no.
## 
## if found the base bug in inventory then return TRUE;
## ------------------------------------------------------


sub _txkCheckDB920OneOff
{
    my $self = $ARG[0];

    my $oneoffList = $ARG[1];

    my $oracle_home = $self->_getDBOracleHome();

    my $fsys = TXK::FileSys->new();

    return TXK::Util::FALSE unless ( ( $fsys->isDirectory({dirName=>$oracle_home} ) ) and 
				     ( $self->_txkIsDBContext() )  ) ;

    my $patch_storage_dir = TXK::OSD->trDirPathToBase($oracle_home.'/.patch_storage/');

    return TXK::Util::FALSE unless ( $fsys->isDirectory({dirName=>$patch_storage_dir}) );

    my @dirList = $fsys->getDirList( { dirName => $patch_storage_dir } );

    my $dir;

    foreach $dir ( @dirList )
     {
	 my $file = TXK::OSD->trDirPathToBase( $patch_storage_dir."/".$dir."/"."inventory" );

	 $fsys->access( { fileName => $file,
			  type     => TXK::FileSys::FILE,
			  checkMode=> TXK::FileSys::READ_ACCESS }) 
	     or next;

	 my $io = TXK::IO->new();

	 $io->open( { fileName => $file,
		      mode     => TXK::IO::READ } );

	 my $io_file_handle = $io->getFileHandle();

	 my @io_data = <$io_file_handle>;

	 $io->close();

	 my $oneoff;

	 foreach $oneoff ( split (/-/, $oneoffList ) )
	  {
	      return TXK::Util::TRUE if ( scalar ( grep ( /$oneoff/ , @io_data ) ) > 0 ) ;
	  }
     }

    return TXK::Util::FALSE;
}

## ------------------------------------------------------
## _txkCheckDB101Oneoff - checks for DB one-off for 10.1
## input params is oneoff
## ------------------------------------------------------

sub _txkCheckDB101OneOff 
{
    my $self = $ARG[0];

    my $oneoff = $ARG[1];

    return TXK::Util::FALSE;
}



## ------------------------------------------------------
## txkIsProductInstalled
## ------------------------------------------------------

sub txkIsProductInstalled
{
	my $self = shift;
	my $args = shift;
			
	my $retVal ;

	my $prod_arg = $args->{'prod'};

	my $sqlcmd=<<EOF;
	declare
	prod_install_status varchar2(1);
	prod_id             number;
	returned_status     varchar2(5);
	cursor get_status is
	select fnd_fpi.status 
	from   FND_PRODUCT_INSTALLATIONS FND_FPI,
	FND_ORACLE_USERID FND_FOU,
	FND_APPLICATION FND_FA
		where  fnd_fpi.application_id = fnd_fa.application_id
		and    fnd_fpi.oracle_id = fnd_fou.oracle_id
		and    fnd_fa.application_short_name = upper('#PROD_ARG#');
	begin
	returned_status := 'FALSE';
	open get_status;
	fetch get_status into prod_install_status;
	if get_status%NOTFOUND then
	returned_status := 'FALSE';
	else
	if (prod_install_status = 'I' or prod_install_status = 'S')
		then
		returned_status := 'TRUE';
	else
		returned_status := 'FALSE';
	end if;
	end if;
	close get_status;
	dbms_output.put_line(returned_status);
	exception
	when others then
	dbms_output.put_line('FALSE');
	end;
EOF

	$sqlcmd =~ s/#PROD_ARG#/$prod_arg/g;


	$retVal = $self->_getSQLValue($sqlcmd);
	return $retVal;

}



## ------------------------------------------------------
## txkVerifyAppsPatch
## ------------------------------------------------------

sub txkVerifyAppsPatch
{
	my $self = shift;
	my $args = shift;

	my $retVal;
	my $patchno = undef;
	my $prod = undef;

	TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

	TXK::Util->isValidArgs({ args=>$args,reqd=> ["$patchno"]} );
	$patchno = $args->{'patchno'};
	$prod = $args->{'prod'};
	my $status = $self->_verifyPatch($patchno);
	my $prereq_status;
                if(defined $prod)

                {

                        if($status  eq TXK::Error::SUCCESS)

                        {

                                $retVal = $self->txkIsProductInstalled({ prod => $prod });

                        }

                }
	return $retVal;

}


#------------------------------------------------------------------------------
# txkVerifyIES
#------------------------------------------------------------------------------
sub txkVerifyIES
{

        my $self   = $ARG[0];

        my $host_n = $self->txkGetContextVar({ oavar => 's_hostname'});

        my $prereq_iespatch = 3197756;

        my $is_installed = $self->txkIsProductInstalled({ prod => 'IES' });

        chomp($is_installed);

        if  ($is_installed eq TXK::Util::TRUE)
        {

                my $patch_number = $self->_isPatchApplied($prereq_iespatch);

                chomp($patch_number);
                if  ($patch_number ne $prereq_iespatch)
                {
                        return  TXK::Util::FALSE; #Warning condition
                }
                else
                {
                        return TXK::Util::TRUE; #Pass Condition
                }

        }

return TXK::Util::TRUE;

}

1;
