
# +===========================================================================+
# |   Copyright (c) 2003 Oracle Corporation, Redwood Shores, California, USA
# |                         All Rights Reserved
# |                        Applications Division
# +===========================================================================+
# |
# | FILENAME
# |   Util.pm
# |
# | DESCRIPTION
# |      TXK Util package
# |
# | USAGE
# |       See Util.html
# |
# | PLATFORM
# |
# | NOTES
# |
# +===========================================================================+

# $Header: Util.pm 03-aug-2005.08:54:54 ndutko Exp $

package TXK::Util;

@ISA = qw( TXK::Common );

######################################
# Standard Modules
######################################

use strict;
use English;
use Carp;

require 5.005;

######################################
# Public Constants
######################################

use constant TRUE    => "true";
use constant FALSE   => "0";

use constant UNKNOWN_PERL_VERSION => "0.0.0";

######################################
# Package Specific Modules
######################################

use TXK::Error();

######################################
# Package Variables 
######################################

my $PACKAGE_ID = "TXK::Util";

my $INVALID_INSTANCE_CALL =
    "This function can only be called as an instance method of ";
my $INVALID_CLASS_CALL =
    "This function can only be called as a class method of ";
my $INVALID_ARGUMENTS =
   "This function must be called with a HASH argument ";
my $MISSING_ARGUMENTS =
   "This function requires the following arguments: ";

######################################
# Object Keys
######################################

######################################
#
# Object Structure
# ----------------
#
#  Hash Array
#
######################################

######################################
# Package Methods 
#
# Public
#
#	new 	- build empty object
#
######################################

sub new;
sub DESTROY;
sub setDebug;
sub setAbortOnError;
sub getScalarArg;
sub getBooleanArg;
sub getTimestamp;
sub getFileNameTimestamp;
sub getString;
sub copyHash;
sub copyArray;
sub printDataStr;
sub getPerlVersion;
sub isPerl55;
sub isPerl56;
sub isPerl58;

######################################
# Package Methods
# 
# Private
#       All private methods are marked with a leading underscore.
#
######################################

sub _doCopyHash;
sub _doCopyArray;

######################################
# Constructor
######################################

sub new {
  my $type = $ARG[0];

  my $self = TXK::Common->new();

  bless $self, $PACKAGE_ID ;

  my $key;

  my %INIT_OBJ = (
                   PACKAGE_IDENT   => $PACKAGE_ID,
                 );

  foreach $key (keys %INIT_OBJ)
   {
     $self->{$key} = $INIT_OBJ{$key};
   }

  return $self;
}

######################################
# Destructor
######################################

sub DESTROY
{
}

######################################
# Common sub to set debug flag
######################################

sub setDebug
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  my $status = "0";

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID,
		         args=>$args});

  $status = $args->{'enable'} if exists $args->{'enable'};

  $status = "0" if ( $status ne "true" );

  return $status;
}

######################################
# Common sub to set abortOnError
######################################

sub setAbortOnError
{  
  my $self  = $ARG[0];
  my $args  = $ARG[1];        
             
  my $status;

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID,
                         args=>$args});

  $status = $args->{'enable'} if exists $args->{'enable'};

  $status = "true" unless ( $status eq "false" || $status eq "0" );
  $status = "0" if ( $status eq "false" );

  return $status;
}

######################################                                          # Common Routine to check for valid object
######################################

sub isValidObj
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Error->abort("TXK::UTIL->isValidObj requires a valid hash arg")
              if ( ref($args) ne "HASH" );

  TXK::Error->abort("isValidObj must have valid obj/mode/package args")
             unless ( exists($args->{'obj'}) && exists($args->{'package'}) );

  if ( $args->{'mode'} eq "class" )
   {

     TXK::Error->abort( { _msg => $INVALID_CLASS_CALL . $args->{'package'} ,
                          _caller_level => 2 } ) 
             if ( $args->{'obj'}  ne $args->{'package'} ) ;
   }
  else
   {

     my $obj = $args->{'obj'};

     my $TXKident = undef;

     $TXKident = $obj->{PACKAGE_IDENT} 
                 if ( uc(substr(ref($obj),0,3)) eq "TXK"  &&
                      exists($obj->{PACKAGE_IDENT}) );

     $TXKident = $args->{'package'} unless defined $TXKident;

     TXK::Error->abort( { _msg => $INVALID_INSTANCE_CALL . $TXKident,  
                          _caller_level => 2 } ) 
             if ( ref($args->{'obj'}) ne $TXKident );
   }

  if ( exists $args->{'args'} )
   {
         TXK::Error->abort( { _msg => $INVALID_ARGUMENTS ,
                              _caller_level => 2 } )
             if ( defined $args->{'args'} && ref($args->{'args'}) ne "HASH" );
   }
}

######################################                        
# isValidArgs
######################################

sub isValidArgs
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Error->abort("TXK::UTIL->isValidArgs requires a valid hash arg")
              if ( ref($args) ne "HASH" );

  TXK::Error->abort("isValidArgs must have valid args/reqd keys")
             unless ( exists($args->{'args'}) && exists($args->{'reqd'}) );

  TXK::Error->abort( { _msg => $INVALID_ARGUMENTS , _caller_level => 2 } ) 
              if ( ref($args->{'args'}) ne "HASH" );

  TXK::Error->abort("TXK::UTIL->isValidArgs : reqd value must be an array ref")
              if ( ref($args->{'reqd'}) ne "ARRAY" );

  my ($reqd,$key,$missing);

  $reqd = $args->{'reqd'};

  foreach $key ( @$reqd )
   {
     unless ( exists $args->{'args'}->{$key} )
      {
        $missing .= ", " if defined $missing;
        $missing .= $key;
      }
   }

  TXK::Error->abort( { _msg => $MISSING_ARGUMENTS . $missing ,
                       _caller_level => 2 } )
              if ( defined $missing );
}

######################################
# getScalarArg
######################################

sub getScalarArg
{
  my $self  = $ARG[0];
  my $id    = $ARG[1];
  my $args  = $ARG[2];

  TXK::Error->abort( { _msg => "Argument <${id}> must be a scalar",
                       _caller_level => 2 } )
         unless ( defined $args &&
                   ( ref($args) eq "SCALAR" ||
                     !ref($args) ) );

  my $str = ( ref($args) eq "SCALAR" ? $$args : $args );

  TXK::Error->abort( { _msg => "Argument <${id}> cannot be an empty string",
                       _caller_level => 2 } )
          if ( $str eq "" );

  return $str;
}

######################################
# getScalarRef
######################################

sub getScalarRef
{
  my $self  = $ARG[0];
  my $id    = $ARG[1];
  my $args  = $ARG[2];

  TXK::Error->abort( { _msg => "Argument <${id}> must be a scalar ref",
                       _caller_level => 2 } )
         unless ( defined $args && ref($args) eq "SCALAR" );

  return $args;
}

######################################
# getArrayRef
######################################

sub getArrayRef
{
  my $self  = $ARG[0];
  my $id    = $ARG[1];
  my $args  = $ARG[2];

  TXK::Error->abort( { _msg => "Argument <${id}> must be a array ref",
                       _caller_level => 2 } )
         unless ( defined $args && ref($args) eq "ARRAY" );

  return $args;
}

######################################
# getHashRef
######################################

sub getHashRef
{
  my $self  = $ARG[0];
  my $id    = $ARG[1];
  my $args  = $ARG[2];

  TXK::Error->abort( { _msg => "Argument <${id}> must be a hash ref",
                       _caller_level => 2 } )
         unless ( defined $args && ref($args) eq "HASH" );

  return $args;
}

######################################
# getBooleanArg
######################################

sub getBooleanArg
{
  my $self  = $ARG[0];
  my $id    = $ARG[1];
  my $args  = $ARG[2];

  TXK::Error->abort( { _msg => 
			   "Argument <${id}> must be either true,false or 0",
                       _caller_level => 2 } )
         unless ( defined $args &&
                   (  ( ref($args) eq "SCALAR"  &&
                           ( $$args eq "true" || $$args eq "false" || 
			     $$args eq "0"
                           ) 
                      ) ||
                      ( 
                        !ref($args)  &&
                           ( $args eq "true" || $args eq "false" || 
                             $args eq "0"
                           )
                      )
                   )
                );

  my $str = ( ref($args) eq "SCALAR" ? $$args : $args );

  $str = "0" if ( $str eq "false" );

  return $str;
}

######################################
# getTimestamp
######################################

sub getTimestamp
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  return scalar(localtime(time));
}

######################################
# getFileNameTimestamp
######################################

sub getFileNameTimestamp
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  my $name = join("_",split(/ /,TXK::Util->getTimestamp()));
  
  $name =~ s/\s/_/g;
  $name =~ s/:/_/g;

  return $name;
}

######################################
# getString
######################################

sub getString
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  my $msg = ( !ref($args)
                 ? $args
                 : ( ref($args) eq "SCALAR" ? $$args : "" )
            );

  return $msg;
}

sub copyHash
{
    my $self      = $ARG[0];
    my $FROM_HASH = $ARG[1];
    my $TO_HASH   = $ARG[2];

    my $DUP_REF_HASH = { };

    $DUP_REF_HASH->{$FROM_HASH} = $TO_HASH;

    $self->_doCopyHash($FROM_HASH,$TO_HASH,$DUP_REF_HASH);
}

sub copyArray
{
    my $self       = $ARG[0];
    my $FROM_ARRAY = $ARG[1];
    my $TO_ARRAY   = $ARG[2];

    my $DUP_REF_HASH = { };

    $DUP_REF_HASH->{$FROM_ARRAY} = $TO_ARRAY;

    $self->_doCopyArray($FROM_ARRAY,$TO_ARRAY,$DUP_REF_HASH);
}

sub printDataStr
{
 my $self = $ARG[0];
 my $data = $ARG[1];

 my $traversed = {} ;

 _doPrintDataStr($self,$data,ref($data),0,"",$traversed);
} 

sub getPerlVersion
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

# To keep everyone happy, the string representation will return
# the old style for 5.5.

  return ( TXK::Util->isPerl55() ? $PERL_VERSION : _doGetPerlVersion() );
}

sub isPerl55
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  return ( _doGetPerlVersion() =~ m/^5\.5/ 
              ? TXK::Util::TRUE : TXK::Util::FALSE );
}

sub isPerl56
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  return ( _doGetPerlVersion() =~ m/^5\.6/ 
              ? TXK::Util::TRUE : TXK::Util::FALSE );
}

sub isPerl58
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,mode=>"class",package=>$PACKAGE_ID});

  return ( _doGetPerlVersion() =~ m/^5\.8/ 
              ? TXK::Util::TRUE : TXK::Util::FALSE );
}

######################################
# End of Public methods
######################################

sub _doPrintDataStr
{
 my $self = $ARG[0];
 my $data = $ARG[1];
 my $type = $ARG[2];
 my $depth= $ARG[3];
 my $key  = $ARG[4];
 my $traversed = $ARG[5];

 my $entry ;

 if ( $type )
   {
     if ( exists($traversed->{$data}) )
     {
       _doPrint("Object $data already traversed",$depth,$key);

       $traversed->{$data} += 1;

       return;
     }

     $traversed->{$data} = 0;
   }

 if    ( $type  eq "CODE"  ||
         $type  eq "GLOB"  ||
         !$type
       )
   {
     _doPrint($data,$depth,$key);
   }
 elsif ( $type eq "ARRAY" )
   {
     _doPrint("** BEGIN ARRAY ** $data",$depth,$key);

     foreach $entry (@$data)
     {
       _doPrintDataStr($self,$entry,ref($entry),$depth+1,"",$traversed);
     }

   _doPrint("** END ARRAY **",$depth,$key);
   }
 elsif ( $type eq "HASH" )
   {
     _doPrint("** BEGIN HASH ** $data",$depth,$key);

     foreach $entry ( keys %$data )
     {

       _doPrintDataStr($self,$data->{$entry},ref($data->{$entry}),
                       $depth+1,$entry,$traversed);
     }

     _doPrint("** END HASH **",$depth,$key);
   }
 elsif ( $type eq "SCALAR" )
   {
     _doPrint($$data,$depth,$key);
   }
 elsif ( $type eq "REF" )
   {
     _doPrint("** BEGIN REF **",$depth,$key);

     _doPrintDataStr($self,$$data,ref($$data),$depth+1,"",$traversed);

     _doPrint("** END REF **",$depth,$key);
   }
 elsif ( $type =~ m/\w+/ ) # Must be an object
   {
     my $obj_type ;

     if ( (split('=',$data))[1] =~ m/^([A-Z]+)/ )
     {
       $obj_type = $1;
     }

     $obj_type = ""
                 unless ( $obj_type eq "CODE" || $obj_type eq "GLOB" ||
                         $obj_type eq "REF"  || $obj_type eq "ARRAY" ||
                         $obj_type eq "HASH" || $obj_type eq "SCALAR" );

     _doPrint("** BEGIN OBJECT $data - " . ($obj_type ? $obj_type : "**BAD**"),
             $depth,$key);

#    If traverse is 0, then it was set at this depth. Since we process
#    objects by recursing back to the same object, we must delete the
#    flag.

     delete $traversed->{$data}
           if ( exists($traversed->{$data}) && $traversed->{$data} == 0 );

     _doPrintDataStr($self,$data,$obj_type,$depth+1,"",$traversed);

     _doPrint("** END OBJECT $data",$depth,$key);
   }
}

sub _doPrint
{
 my $data      = $ARG[0];
 my $noOfSpaces = $ARG[1];
 my $key        = $ARG[2];

 print " " x $noOfSpaces , ( $key ? "KEY=$key : " : "" ), $data, "\n";
}

sub _doCopyHash
{
    my $self = $ARG[0];

    my $FROM_HASH = $ARG[1];

    my $TO_HASH =  $ARG[2];

    my $DUP_REF_HASH = $ARG[3];

    my $key;
    my $data_ref;

    foreach $key ( keys %{$FROM_HASH} )
    {
        $data_ref = ref($FROM_HASH->{$key});

        if ( ( $data_ref eq 'HASH' || $data_ref eq 'ARRAY' ) &&
             exists $DUP_REF_HASH->{$FROM_HASH->{$key}} )
         {
           $TO_HASH->{$key} = $DUP_REF_HASH->{$FROM_HASH->{$key}};
         }
	elsif ( $data_ref eq 'HASH'  )
	 {
 	     $TO_HASH->{$key} = {};

             $DUP_REF_HASH->{$FROM_HASH->{$key}} = $TO_HASH->{$key};

             $self->_doCopyHash( $FROM_HASH->{$key} , $TO_HASH->{$key} ,
                                 $DUP_REF_HASH);
	 }
	elsif ( $data_ref eq 'ARRAY' )
	 {
	     $TO_HASH->{$key} = [ ];

             $DUP_REF_HASH->{$FROM_HASH->{$key}} = $TO_HASH->{$key};

             $self->_doCopyArray( $FROM_HASH->{$key} , $TO_HASH->{$key} ,
                                  $DUP_REF_HASH);
	 }
	else
	 {
	    $TO_HASH->{$key} = $FROM_HASH->{$key};
	 }
    }
}

sub _doCopyArray
{
    my $self = $ARG[0];

    my $FROM_ARRAY = $ARG[1];

    my $TO_ARRAY   = $ARG[2];

    my $DUP_REF_HASH = $ARG[3];

    my $index = 0; 
    my $data_ref;

    while ( $index < scalar(@{$FROM_ARRAY} ) )
     {
         $data_ref = ref($FROM_ARRAY->[$index]);

         if ( ( $data_ref eq 'HASH' || $data_ref eq 'ARRAY' ) &&
              exists $DUP_REF_HASH->{$FROM_ARRAY->[$index]} )
          {
               $TO_ARRAY->[$index] = $DUP_REF_HASH->{$FROM_ARRAY->[$index]};
          }
         elsif ( $data_ref eq 'HASH'  )
	  {
	       $TO_ARRAY->[$index] = { };

               $DUP_REF_HASH->{$FROM_ARRAY->[$index]} = $TO_ARRAY->[$index];

               $self->_doCopyHash( $FROM_ARRAY->[$index] , $TO_ARRAY->[$index],
                                   $DUP_REF_HASH );
	  }
	 elsif (  $data_ref eq 'ARRAY'  )
	  {
	       $TO_ARRAY->[$index] = [ ];

               $DUP_REF_HASH->{$FROM_ARRAY->[$index]} = $TO_ARRAY->[$index];

               $self->_doCopyArray( $FROM_ARRAY->[$index] , $TO_ARRAY->[$index],
                                    $DUP_REF_HASH );
	  }
	 else
	  {
	      $TO_ARRAY->[$index] = $FROM_ARRAY->[$index];
	  }
	 $index++;
      }
}

sub  _doGetPerlVersion
{
  if ( $] eq $PERL_VERSION )  # Old format
   {
#     Must be 5.005 or lower

      if ( $PERL_VERSION =~ m/(\d+)\.(\d\d\d)(\d*)/ )
       {
         return sprintf("%d.%d.%d",$1,$2,$3*10);
       }
   }
  else
   {
#     We could just use unpack("CCC",$PERL_VERSION) but I guess this
#     would break if the version really was in UNICODE format. [ unpack
#     works fine with 5.8. ], Anyway, sprintf is the safest thing to do.

     return sprintf("%vd",$PERL_VERSION);
   }

  return TXK::Util::UNKNOWN_PERL_VERSION;
}

# ==========================================================================

1;
