#
# $Header: emll/sysman/admin/scripts/TXK/Validate.pm /main/3 2010/01/27 12:46:52 rlandows Exp $
# +===========================================================================+
# |   Copyright (c) 2002 Oracle Corporation, Redwood Shores, California, USA
# |                         All Rights Reserved
# |                        Applications Division
# +===========================================================================+
# |
# | FILENAME
# |   Validate.pm
# |
# | DESCRIPTION
# |   Performs validation depeding upon the xml definition and 
# |   generates  report.
# |
# |
# | PLATFORM
# |   Generic
# |
# +===========================================================================+
#

package TXK::Validate;

@ISA = qw( TXK::Common );

######################################
# Standard Modules
######################################

use strict;
use English;
use Carp;

require 5.005;

######################################
# Constants
######################################

use constant ACTION_SUCCESS			=> "PASS"		;
use constant ACTION_ALL_SUCCESS			=> "ALLPASS"		;
use constant ACTION_FAILURE			=> "FAIL"		;
use constant ACTION_WARNING			=> "WARN"		;
use constant MESSAGE_FILE_KEY                   => "messagefile"        ;
use constant HEADER_TAG                         => "Header"             ;
use constant TITLE_TAG                          => "Title"              ;
use constant TXK_REPORT_TAG                     => "TXKReport"          ;
use constant TXK_NODE_TAG                       => "Node"               ;
use constant REPORT_ACTION_TAG                  => "action"             ;
use constant STATUS_TAG                         => "status"             ;
use constant MESSAGE_TAG                        => "message"            ;
use constant ACTION_VERSION_TAG                 => "version"            ;
use constant SUMMARY_TAG                        => "Summary"            ;
use constant INVENTORY_TAG                      => "Inventory"          ;
use constant SECTION_TITLE                      => "Title"              ; 
use constant XML_DEFN_FILE                      => "xmlfile"            ; 
use constant APPS_CONTEXT                       => "APPS_CONTEXT"       ;
use constant DB_CONTEXT                         => "DB_CONTEXT"         ;
use constant ACTION_OBJECT_CLASS_TYPE           => "class"              ;
use constant ACTION_OBJECT_OBJECT_TYPE          => "object"             ;

######################################
# Package Specific Modules
######################################

use TXK::XML;
use TXK::Util;
use TXK::Techstack;
use TXK::TechstackDB;
use TXK::Reports;

######################################
# Package Variables 
######################################

my $PACKAGE_ID = "TXK::Validate";

#--------------------------------------------------------------------
# the keys of the TECHSTACK_NODETYPE_METHODS should match the 
# "type" attribute of the "NodeType"  element (case insensitive)
#--------------------------------------------------------------------

my $TECHSTACK_NODETYPE_METHODS = {  WEB    => "txkIsWebNode",
				    CONC   => "txkIsConcNode",
				    ADMIN  => "txkIsAdminNode",
				    FORMS  => "txkIsFormsNode",
#				    DB     => "txkIsDBNode" 
				    DB     => "txkIsDBContext" 
				   };

my $NODETYPES_FOR_ORDERING     = {};


#####################################
# Object Keys
#####################################

my $XML_DEFN                 = "XML Definition";
my $ACTION_DEFN              = "Action Definition";
my $XML_FILE                 = XML_DEFN_FILE;
my $DOC_LOADED               = "DocLoaded";
my $OBJECT_MAPPING           = "ObjectMap";
my $APPS_CONTEXT_FILE        = TXK::Techstack::APPS_CTX_FILE_KEY;
my $APPS_CONTEXT             = "appscontext";
my $FILE_SYS                 = "FileSys";
my $CONFIG_OPTIONS           = "configOptions";
my $OBJECT_CONTAINER_HASH    = "ObjectContaier";
my $METHOD_RETURN_CACHE      = "CacheForMethodReturnValues";
my $APPS_USER                = TXK::Techstack::APPS_USER_KEY;
my $APPS_PASSWORD            = TXK::Techstack::APPS_PASSWORD_KEY;
my $TECHSTACK                = "TXK::Techstack";
my $TECHSTACKDB              = "TXK::TechstackDB";
my $DEBUG                    = "DebugFlag";
my $MSGXML                   = "messagefile";
my $ALLSTATUS                = "allStatus";
my $REPORT                   = "report";
my $PROG_ARGS                = "program_arguments";
my $REPORT_NODE_HASH         = "report_node_hash";

my $REPORT_TITLE             = "Report Title";
my $REP_DEFN_TYPE            = "Type";
my $REP_HEADER               = "Report Header";
my $NODETYPES                = "NodeTypes";
my $NODEORDER                = "Order";

my $ACTION_LIST              = "ActionList";
my $REPORT_CTX               = "ReportContext";
my $REPORT_FILE              = "reportfile";
my $REPORT_TYPE              = "reporttype";
my $REPORT_XSL               = "xslfile";
my $XML_HEADER               = "XMLHeader";
my $SUMMARY_ALLPASS          = "ALLPASS";
my $SUMMARY_FAIL             = "FAIL";
my $SUMMARY_WARN             = "WARN";
my $SUMMARY_MSG              = "summarymesssage";
my $SUMMARY_STATUS_TEXT      = "StatusText";
my $SUMMARY_TITLE            = "Title";
my $SINGLE_VALUE_TABLE       = "SingleValue";
my $NAME_VALUE_PAIR_TABLE    = "NameValuePair";
my $CONTEXT_TYPE             = "ContextType";

#---------------------------------------------------------
# Each of the following should match the attribute name in the
# xml file for the action. If the tag is changed in the action
# definition or a new attribute is added in the definition
# the same change needs to be done below.
#---------------------------------------------------------

my $ACTION_NAME              = "Name";
my $ACTION_DESCRIPTION       = "Description";
my $ACTION_TYPE              = "Type";
my $ACTION_METHOD            = "Method";
my $ACTION_PLATFORM          = "Platform";
my $ACTION_OS_TYPE           = "ostype";
my $ACTION_OS_VERSION        = "osversion";
my $ACTION_OS_RELEASE        = "osrelease";
my $ACTION_RESULT_VAR        = "resultVar";
my $ACTION_INPUTPARAMS       = "InputParams";
my $ACTION_CONFIG_OPTION     = "ConfigOption";
my $ACTION_CONDITION         = "Condition";
my $ACTION_SUCCESS_VALUES    = "SuccessValues";
my $ACTION_WARN_VALUES       = "WarnValues";
my $ACTION_FAIL_VALUES       = "ErrorValues";
my $ACTION_VALIDATION_METHOD = "ValidationMethod";
my $ACTION_SUCCESS_MESSAGE   = "SuccessMsg";
my $ACTION_WARN_MESSAGE      = "WarningMsg";
my $ACTION_FAILURE_MESSAGE   = "FailureMsg"; 
my $ACTION_NODETYPE          = "Tier";
my $ACTION_VERSION           = "Version";
my $ACTION_FIX_METHOD        = "fixMethod";
my $ACTION_FIX_PROMPT        = "fixMethodPrompt";
my $ACTION_FIX_ARGS_INPUT    = "fixMethodArgs";
my $ACTION_FIX_ARGS          = "fixMethodArgsProcessed";
my $ACTION_RETURN_TYPE       = "returnType";
my $ACTION_FIX_METHOD_PARAMS = "fixMethodParams";
my $ACTION_OBJECT_TYPE       = "objecttype";

#------------------------------------------
# for the ACTION_DELIM, the default is "," which is
# defined in the new method. An xml definition
# can override it by specifying the ActionDelim 
# attribute for the action. useful if we want to 
# use comma itself as part of the value.
#--------------------------------------------------

my $ACTION_DELIM              = "ActionDelim";        # this defaults to comma
my $ACTION_VALUE_DELIM        = "ActionValueDelim";   # this defaults to a dot
my $ACTION_STATUS             = "Status";
my $ACTION_PROCESSED          = "ActionProcessed";
my $ACTION_RETURN_XML_TYPE    = "xmlnode";
my $ACTION_RETURN_ARRAY_TYPE  = "array";
my $ACTION_RETURN_HASH_TYPE   = "hash";


#-------------------------------------------------
# following values are generated. not present in 
# the XML definition.
#------------------------------------------------

my $ACTION_VALID             = "Valid";
my $ACTION_EXEC_METHOD       = "Method to be executed";

######################################
#
# XML TAG NAMES
######################################

my $NODE_TYPE_TAG         = "type";
my $NODE_HEADER           = "NodeHeader";
my $NODE_TABLE_TYPE_TAG   = "TableType";
my $COMPONENT_HEADER      = "ComponentHeader";
my $VERSION_HEADER        = "VersionHeader";
my $STATUS_HEADER         = "StatusHeader";

#####################################
#
# DEFAULTS
####################################

my $DEFAULT_MSGFILE          = TXK::OSD->trDirPathToBase(TXK::OSD->getEnvVar({name=>'FND_TOP'})."/html/txkMessages.xml");
my $STATUS_ALLPASS           = "ALLPASS";
my $STATUS_FAIL              = "FAIL";
my $STATUS_WARN              = "WARN";
my $INPUT_PARAM_DELIM        = "\,";
my $INPUT_PARAM_VALUE_DELIM  = "=";
my $TEMP_DIR                 = TXK::OSD->trDirPathToBase(TXK::OSD->getEnvVar({name=>'APPLTMP'}) . "/TXK" )  ;
my $DEFAULT_ARG_PROMPT       = "Enter the value for ";

my $DEFAULT_OBJECT_MAP       = TXK::OSD->trDirPathToBase(TXK::OSD->getEnvVar({name => 'FND_TOP'})."/html/txkValObjMap.xml");
my $DEFAULT_REP_VARS         = { 't_date'         => undef,
				 't_platform'     => undef,
				 't_os_release'   => undef,
				 't_context'      => undef,
				 't_repfile'      => undef,
				 't_nodetypes'    => undef,
			         't_defnfile'     => undef,
			         't_defnfile_ver' => undef,
			         't_ctxfile_ver'  => undef };

######################################
#
# Object Structure
# ----------------
#
#  Hash Array
#
#  The xml definition for the validation is of the following format.
#
#  <Techstack Title="MSG1234" Type="Validate">   <!-- Type can be "Validate" or "Inventory" 
#                                                     For type Inventory, there is no "Status" column and
#                                                     there is no "Summary Status" -->
#                                             
#     <ReportHeader>
#       MSG1234</ReportHeader>
#
#     <NodeTypes>
#         <tier type="Web"   Name="HTTP Server Node" NodeHeader="MSG1234" 
#                                                    ComponentHeader="MSG123" 
#                                                    VersionHeader="MSG123"
#                                                    StatusHeader="MSG123"
#                                                    Order="1"/>   <!--  The default order is any node  with FAIL 
#                                                                         will be first in the report. This can be
#                                                                         overridden with Order attribute  -->
#                                                                       
#                          
#         <tier type="Forms" Name="Forms Server Node" NodeHeader="MSG1234"
#                                                     ComponentHeader="MSG123" 
#                                                     VersionHeader="MSG123"
#                                                     StatusHeader="MSG123"/>
#
#         <tier type="Conc"  Name="Concurrent Processing Server Node" NodeHeader="MSG1234"
#                                                                     ComponentHeader="MSG123" 
#                                                                     VersionHeader="MSG123"
#                                                                     StatusHeader="MSG123"/>
#
#         <tier type="Admin" Name="Administration Server Node" NodeHeader="MSG1234"
#                                                              ComponentHeader="MSG123" 
#                                                              VersionHeader="MSG123"
#                                                              StatusHeader="MSG123"/>
#      </NodeTypes>
#
#      <ReportVariables>                    <-- this section will contain all the report variable that will be used for
#                                               the report. this should not contain the application context vars -->
#        <var1 oa_var='t_var'/>
#      </ReportVariables>
#
#  <-- currently the following variables will be set by this module.
#      the rest of the variables in the ReportVariables section
#      should be set by the respective  actions 
#
#     t_date       => to the current date
#     t_platform   => to the current platform 
#     t_context    => to the applications context file
#     t_repfile    => to the location of the output report
#     t_nodetypes  => to the comma separated node list
#     t_defnfile   => this is the definition file that was processed
#
#  -->
#       <-- the following values are sample only to demonstrate the parameters. -->
#
#        <action Description="Check Display" 
#            tier="Web,Forms,Conc"           <-- this should be one of the values in NodeTypes -->
#            type="TXK::Techstack"           <-- object type -->
#            objecttype="class"              <-- whether this is a "class" object or an instantiated object. 
#                                                default is "object". If the method needs to be called without
#                                                creating an object, then "class" needs to be specified. -->
#            method="checkDisplay"           <-- this method will be looked in an object mapping file to get the actual method -->
#            inputParams="context=host"      <-- must be a name=value pair. this will be converted to hash arg and be passed 
#                                                to the method -->
#            condition="isUnix"              <-- this method must return TXK::Util::TRUE for the action's method to be executed -->
#	     platform="UNIX"                 <-- platforms values should match that of AutoConfig platforms or supported by TXK::OSD 
#                                                "UNIX" can be used for all Unix systems and "NT" can be used for Windows systems -->
#            ostype="Red Hat"                <-- os type is supported initially for Red Hat and SuSE, throguh detection in OSD.pm -->
#            osversion="3.0"                 <-- os version is supported for Red Hat thorugh detection in OSD.pm. you can also give
#                                                3.0+  -->
#            osrelease="2.4.9-e.25kmap1enterprise" <-- os release is used from 'uname -r', implementation is OSD.pm -->
#	     resultvar="t_display"           <-- the return value of the method will be stored in this variable -->
#	     configOption="all"              <-- config options can be valid techstack options, ias10,http1312,ias1022,db817,db910,db920
#                                                and db101 until now (05/04). config options for db's will be all the digits
#                                                of their version prepended by 'db'. so the config option for 9.2.0.4.0 
#                                                will be 'db92040'and if
#                                                the action specified 'db920' it will still be valid as 'db9204' is valid.--> 
#	     successValues="0"               <-- the values will be processed in this order error->warning->success 
#	     warningValues="1"                   depending upon, in which list the returned value exists, the status will
#            errorValues="-1"                    be set to ACTION_FAILURE,ACTION_WARNING,ACTION_SUCCESS. 
#                                                you can specify "1.1.8+" or ">8000" or "<9213". -->
#	     successMsg="MSG-TXKV0001"       <-- this message will be set in the report if the status is ACTION_SUCCESS -->
#	     warningMsg="MSG-TXKV0003"
#	     failureMsg="MSG-TXKV0002"
#            version="s_display"             <-- this is the version information for this action to be displayed in the report.
#                                                if this is not present which is default then resultvar will be reported for version -->
#	     fixMethod="txkFixXMLParser"     <-- this method can be called to fix the error for this action --> 
#	     fixMethodPrompt="Do you want to run adadmin to generate jar files to include xmlParser904 jar:"
#                                            <-- this prompt will be shown to the user when given the choice of selection -->
#	     fixMethodArgs="syspass=Enter system passwd:,location=Enter oracle_home location:"
#                                            <-- if the fixMethod requires arguments, then this can be provided here in the form
#                                                <arg1>=<prompt for arg1>,<arg2>=<prompt for arg2> 
#                                                these will be passed to the method -->
#
#            <!-- for multiple return values -->
#            
#            returnType="array"             <-- here we can specify the returntype of the action. permissible valuues are
#                                               "array", "hash" and "xmlnode". if the returntype is array then a new table
#                                               i.e. "NODE" element with attribute TableType="SingleValue" will be created.
#                                               the "Title", "VersionHeader", "ComponentHeader" and "Statusheader" will be
#                                               if defined as attributes of action. otherwise they are taken from node-type.
#                                               in case of array, a table of single column will be created.
#                                               in case of hash, a table of two columns will be created. the TableType attribute
#                                               is set to "NameValuePair". 
#                                               in case of "xmlnode" type the entire return value is included in the report for
#                                               the action. there is no TableType attribute set for the "xmlnode". -->
#
#           <!-- following are valid only if returnType is either "array" , "hash" or "xmlnode" -->
#
#            NodeHeader="title for the table"   <-- this optionally replaces the title from the node type -->
#
#            ComponentHeader="technology component" <-- is reported when return type is "array" , "hash" or "xmlnode" -->
#
#            VersionHeader="component version"  <-- is reported when return type is  "hash" or "xmlnode" -->
#            StatusHeader="status"              <-- is reported when return type is "xmlnode" -->
# 
##----------------------------------------------------------------------------------------------------------
# sample code tested for "xmlnode" return type
#
# sub testXML
# { 
#    my $xml = TXK::XML->new();

#    $xml->addNode ( { 'node' => TXK::Validate::REPORT_ACTION_TAG } );

#    $xml->addNode ( { 'node'   => TXK::Validate::MESSAGE_TAG,
#                      'value'  => 'Version of custom component',
#                      'parent' => TXK::Validate::REPORT_ACTION_TAG } );

#    $xml->addNode ( { 'node'   => TXK::Validate::ACTION_VERSION_TAG,
#                      'value'  => 'a.b.c.d.e',
#                      'parent' => TXK::Validate::REPORT_ACTION_TAG } );

#    $xml->addNode ( { 'node'   => TXK::Validate::STATUS_TAG,
#                      'value'  => 'FAIL',
#                      'parent' => TXK::Validate::REPORT_ACTION_TAG } );
#    return $xml;
# }
##----------------------------------------------------------------------------------------------------------
#
#	     >CheckDisplay</action>
#
#      <Summary>                            <-- if all actions are success then ALLPASS message will be set in the summary of the report.
#         <APPS_CONTEXT>                    <!-- this represents the context type, currently only 
#                                                APPS_CONTEXT and DB_CONTEXT supported-->
#          <ALLPASS>MSG-TXKV0061</ALLPASS>       if there is one warning then the WARN message will be set in the summary. the same
#          <FAIL>MSG-TXKV0062</FAIL>             case for FAIL. -->
#          <WARN>MSG-TXKV0063</WARN>
#         </APPS_CONTEXT>
#      </Summary>
#  </Techstack>
#
#  An object mapping is used to find out the actual method name for this object.
#
#  <TXKRegObject>                              
#      <Object type="TXK::Techstack"
#          description="Implements Technology Stack Inventory"
#          owner="TXK" >
#           <Method name="getJDKVersionForWeb">txkGetJDKVersion</Method>
#      </Object>
#  </TXKRegObject>    
#
#
######################################

######################################
# Package Methods 
#
# Public
#
#	new 	- build empty object
#
######################################

sub new;
sub DESTROY;
sub init;

sub loadDocuments;
sub setDebug;

sub ProcessActions;
sub getAllStatus;
sub getReport;
sub instantiateReport;
sub generateReport;

sub getAutoFixActionListWithPrompt;
sub GetAutoFixActionsArgs;
sub getAutoFixActionList;
sub ProcessAutoFixActions;

######################################
# Package Methods
# 
# Private
#       All private methods are marked with a leading underscore.
#
######################################

sub _createActionList;
sub _instantiateReportDefaults;
sub _processPassword;
sub _getSortedNodes;
sub _getMessageNode;
sub _getDBConfigOption;
sub _checkValidNodeType;
sub _checkConfigOption( $ );
sub _checkValidPlatform ( $ );
sub _validateActions( $ );
sub _getNodeTextForAttr( $ );
sub _getActionXML ( $ );
sub _addActionToReportNode ( $$ );
sub _checkReturnValue($$$);
sub _readReportHeader( $ );
sub _readSummary( $ );
sub _readXMLHeader ( $ );
sub _addActionToTheList( $ );
sub _readNodeTypes( $ );
sub _readReportVariables( $ );
sub _orderNodes;

######################################
# Constructor
######################################

sub new
{
    my $type = $ARG[0];

    my $args = $ARG[1];

    my $self = TXK::Common->new();

    bless $self, $PACKAGE_ID ;

    my %INIT_OBJ =   (  $REPORT_TITLE            => undef ,
			$REP_DEFN_TYPE           => undef ,
			$REP_HEADER              => undef ,
			$NODETYPES               => {} ,
			$ACTION_LIST             => [ ] ,
		        $REPORT_CTX              => TXK::XML->new(),
			$XML_HEADER              => TXK::XML->new(),
			$MSGXML                  => TXK::XML->new(),
			$SUMMARY_ALLPASS         => undef ,
			$SUMMARY_FAIL            => undef ,
			$SUMMARY_MSG             => undef,
			$OBJECT_MAPPING          => $DEFAULT_OBJECT_MAP,
			$CONFIG_OPTIONS          => [ ],
			$ALLSTATUS               => $STATUS_ALLPASS,
		        $SUMMARY_WARN            => undef,
		        $XML_FILE                => undef,
		        $REPORT                  => TXK::XML->new(),
		        PACKAGE_IDENT            => $PACKAGE_ID,
		        $DOC_LOADED              => TXK::Util::FALSE,
		        $APPS_CONTEXT_FILE       => undef,
			$APPS_CONTEXT            => TXK::XML->new(),
			$FILE_SYS                => TXK::FileSys->new(),
		        $OBJECT_CONTAINER_HASH   => {},
			$METHOD_RETURN_CACHE     => {},
		        $DEBUG                   => exists $args->{$DEBUG} ? $args->{$DEBUG} : TXK::Util::FALSE,
		        $REPORT_FILE             => undef,
			$REPORT_TYPE             => undef,
			$REPORT_XSL              => undef,
			$SUMMARY_STATUS_TEXT     => undef,
			$SUMMARY_TITLE           => undef,
			$CONTEXT_TYPE            => undef,
			$PROG_ARGS               => {},
			$REPORT_NODE_HASH        => {},

   		        $ACTION_DEFN => { $ACTION_NAME              =>   undef ,
					  $ACTION_DESCRIPTION       =>   undef ,
					  $ACTION_TYPE              =>   undef ,
					  $ACTION_METHOD            =>   undef ,
					  $ACTION_PLATFORM          =>   undef ,
					  $ACTION_OS_TYPE           =>   undef ,
					  $ACTION_OS_VERSION        =>   undef ,
					  $ACTION_OS_RELEASE        =>   undef ,
					  $ACTION_RESULT_VAR        =>   undef ,
					  $ACTION_INPUTPARAMS       =>   undef ,
					  $ACTION_CONFIG_OPTION     =>   undef ,
					  $ACTION_CONDITION         =>   undef ,
					  $ACTION_SUCCESS_VALUES    =>   undef ,
					  $ACTION_WARN_VALUES       =>   undef ,
					  $ACTION_FAIL_VALUES       =>   undef ,
					  $ACTION_VALIDATION_METHOD =>   undef ,
					  $ACTION_SUCCESS_MESSAGE   =>   undef ,
					  $ACTION_WARN_MESSAGE      =>   undef ,
					  $ACTION_FAILURE_MESSAGE   =>   undef ,
				          $ACTION_NODETYPE          =>   undef ,
					  $ACTION_PROCESSED         =>   TXK::Util::FALSE,
				          $ACTION_VALID             =>   TXK::Util::FALSE,
					  $ACTION_DELIM             =>   ','  ,
					  $ACTION_VALUE_DELIM       =>   '\.' ,
					  $ACTION_FIX_METHOD        =>   undef,
					  $ACTION_FIX_PROMPT        =>   undef,
					  $ACTION_FIX_ARGS_INPUT    =>   undef,
					  $ACTION_FIX_ARGS          =>   {},
				          $ACTION_EXEC_METHOD       =>   undef,
					  $ACTION_RETURN_TYPE       =>   undef,
				          $ACTION_VERSION           =>   undef,
					  $ACTION_FIX_METHOD_PARAMS =>   undef,
					  $ACTION_OBJECT_TYPE       =>   ACTION_OBJECT_OBJECT_TYPE,
					  
					  #-------------------------------------------------------
					  # support for multiple return values for this action, 
					  # not used otherwise.
					  #-------------------------------------------------------

					  $COMPONENT_HEADER         =>   undef,
					  $STATUS_HEADER            =>   undef,
					  $VERSION_HEADER           =>   undef,
					  $NODE_HEADER              =>   undef  }


		      );

    TXK::Util->copyHash( \%INIT_OBJ , $self );

    return $self;
}


######################################
# Destructor
######################################

sub DESTROY 
{

}

sub loadDocuments
{
    my $self = $ARG[0];

    my $args = $ARG[1];

    TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

    TXK::Util->isValidArgs({args=>$args,
			    reqd=> ["$XML_FILE","$APPS_CONTEXT_FILE","$APPS_PASSWORD"]} );

    if ( defined $args->{'tempDir'} )
     {
	 $TEMP_DIR = $args->{'tempDir'};
     }


    ( $self->{$APPS_USER}, $self->{$APPS_PASSWORD} ) = $self->_processPassword($args->{$APPS_PASSWORD});
    
    my $fsys = $self->{$FILE_SYS};

    $fsys->access({fileName=>"$args->{$XML_FILE}",
		   type=>TXK::FileSys::FILE,
		   checkMode=>TXK::FileSys::READ_ACCESS} )
	or return TXK::Error::printMsg($fsys->getError());

    print ("CONTEXT FILE: $args->{$APPS_CONTEXT_FILE}\n") if ( $self->{$DEBUG} );

    $fsys->access({fileName=>"$args->{$APPS_CONTEXT_FILE}",
		   type=>TXK::FileSys::FILE,
		   checkMode=>TXK::FileSys::READ_ACCESS} )
	or  TXK::Error::abort($fsys->getError());

    $self->{$APPS_CONTEXT_FILE} = $args->{$APPS_CONTEXT_FILE};

    $self->{$XML_FILE} = $args->{$XML_FILE};

    print ("TXK::Validate:loadDocuments loading xml documents and creaing action list\n");

    $self->{$APPS_CONTEXT}->loadDocument( { file => $self->{$APPS_CONTEXT_FILE} } );

    # we should always get the APPS_USER from the context file but for internal
    # environments where the read only user is used 'apps_read_only' we should support
    # taking the username from the command line as well

    $self->{$APPS_USER} = $args->{$APPS_USER} 
                          unless ( ( defined($self->{$APPS_USER}) ) and 
				   ( exists $args->{$APPS_USER} ) ) ;    

    # if APPS_USER is still not defined then retrieve it from the context file

    $self->{$APPS_USER} = $self->{$APPS_CONTEXT}->getOAVar( 's_apps_user' )
                                     unless ( defined($self->{$APPS_USER})) ;

print "printing TEMP DIR IS $TEMP_DIR\n";
    if ( $TEMP_DIR =~ m?^[\/|\\]TXK$? ) # Fix for bug 5338626 
     {
       $TEMP_DIR = $self->{$APPS_CONTEXT}->getOAVar( 's_appltmp' ) ;
print "printing TEMP DIR IS $TEMP_DIR\n";
     }

    $self->{$OBJECT_MAPPING} = $args->{$OBJECT_MAPPING}
                 if ( exists  $args->{$OBJECT_MAPPING}  and
		      defined $args->{$OBJECT_MAPPING}   );

    # create the action list

    $self->{$DOC_LOADED} = TXK::Util::TRUE
	if ( $self->_createActionList() == TXK::Error::SUCCESS );

    # load the message file into the xml object

    my $msgFile = exists ( $args->{$MSGXML} ) ? $args->{$MSGXML} : $DEFAULT_MSGFILE;

    $self->{$MSGXML}->loadDocument({file=>$msgFile});

    print ("TXK::Validate:loadDocuments loading of xml document files completed successfully.\n");

    return TXK::Error::SUCCESS;
}

####################################
# set the debug flag
####################################

sub setDebug
{
    my $self = $ARG[0];

    my $debug = $ARG[1];

    TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

    $self->{$DEBUG} =  $debug if ( defined $debug );
}

####################################
# getActionListWithPrompt
#####################################


sub getAutoFixActionListWithPrompt
{
    my $self     = $ARG[0];

    my $autoFixActionHash = $self->getAutoFixActionList();

    my $action ;

    my $actionList = [ keys %$autoFixActionHash ];

    return [] if ( scalar(@$actionList) <= 0 ) ;

    print "Some of the errors can be fixed with the following actions :\n";

    my $corrective_action_no = 1;

    foreach $action ( keys %$autoFixActionHash )
     {
	 print "** BEGIN CORRECTIVE ACTION : $corrective_action_no **\n";
	 print ( $autoFixActionHash->{$action} ."\n" );
	 print "** END CORRECTIVE ACTION : $corrective_action_no **\n";

	 $corrective_action_no++;
     }

    print "Do you want to run [A]ll of the above corrective actions, or [C]hoose some of them or run [N]one:";

    my $answer = <STDIN>;

    chomp($answer);

    return [] if ( $answer =~ m/^N$/i );

    if ( $answer =~ m/^A$/i )
     {
	 return $actionList ;
     }

    elsif ($answer =~ m/^C$/i )
     {
	 $actionList  = [];

	 $corrective_action_no = 1;
	 
	 foreach $action ( keys %$autoFixActionHash )
	  {
	      print "** BEGIN CORRECTIVE ACTION : $corrective_action_no **\n";
	      print ( $autoFixActionHash->{$action} ."\n ");
	      print "** END CORRECTIVE ACTION : $corrective_action_no **\n";
              print ( "Choose <Yes/No>:");

	      $answer = <STDIN>; chomp ($answer);

	      push ( @$actionList, $action ) if ( ( $answer =~ m/^Y$/i ) or ( $answer =~ m/^YES$/i ) );
	  }
	 return $actionList;
     }

    print "No valid option selected. Using default [N].\n";

    return [];
}


####################################
# Get AutoFix Actions Args
#####################################

sub GetAutoFixActionsArgs
{
    my $self = $ARG[0];
    
    my $args = $ARG[1];

    TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

    TXK::Util->isValidArgs({args=>$args,
			    reqd=> ["actionlist"]} );

    TXK::Error->abort("TXK::Validate : arugument must be an array ref")
              if ( ref($args->{'actionlist'}) ne "ARRAY" );    

    my ( $autoFixAction , $action );

    foreach $autoFixAction ( @{$args->{'actionlist'}} )
     {
	 foreach $action ( @{$self->{$ACTION_LIST}} )
	  {
	      if ( $action->{$ACTION_FIX_METHOD} =~ m/^$autoFixAction$/  )
	       {
		   if ( $action->{$ACTION_FIX_ARGS_INPUT} ne '' ) 
		    {
			my $actionArgList = [split(',',$action->{$ACTION_FIX_ARGS_INPUT})];

			next unless ( scalar(@$actionArgList) >= 0 );

			my $fixArg;

			foreach $fixArg ( @$actionArgList )
			 {
			     my ( $inner_arg,$prompt );

			     ( $inner_arg, $prompt ) = split('=',$fixArg);

			     if ( $prompt eq '' )
			      {
				  print ( $DEFAULT_ARG_PROMPT . " $inner_arg:" );
			      }
			     else
			      {
				  print ( $prompt );
			      }

			     my $argValue = <STDIN>;

			     chomp($argValue);

			     $action->{$ACTION_FIX_ARGS}->{$inner_arg} = $argValue;			
			   }
		       }

		   #------------------------------------------------
		   # check if there are any fixMethodParams defined
		   #------------------------------------------------

		   if ( $action->{$ACTION_FIX_METHOD_PARAMS} ne '' )
		    {
			my $argList =  [ split ( "$INPUT_PARAM_DELIM",$action->{$ACTION_FIX_METHOD_PARAMS} ) ] ;
			
			my ( $arg , $key , $value);
			
			foreach $arg ( @$argList )
			 {
			     ( $key , $value ) = split ("$INPUT_PARAM_VALUE_DELIM",$arg);

			     $action->{$ACTION_FIX_ARGS}->{$key} = $value;

			 }			

		    }

	       }

	  }
     }

    return TXK::Error::SUCCESS;
}

####################################
# Process AutoFix Actions
#####################################

sub ProcessAutoFixActions
{
    my $self = $ARG[0];
    
    my $args = $ARG[1];

    print ("TXK::Validate::ProcessAutoFixActions \n") if ( $self->{$DEBUG} );

    TXK::Util->printDataStr($args) if ( $self->{$DEBUG} ) ;

    TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

    TXK::Error->abort("TXK::Validate : arugument must be an array ref")
              if ( ref($args) ne "ARRAY" );    

    my ( $autoFixAction , $action );

    my $objectContainerHash = $self->{$OBJECT_CONTAINER_HASH};

    my $noMethodsExecuted = 0 ;

    foreach $autoFixAction ( @$args )
     {
	 foreach $action ( @{$self->{$ACTION_LIST}} )
	  {
	      if ( $action->{$ACTION_FIX_METHOD} =~ m/^$autoFixAction$/ )
	       {

		   if ( $action->{$ACTION_VALID} )
		    {
			my $object = $objectContainerHash->{$action->{$ACTION_TYPE}};

			my $method = $action->{$ACTION_FIX_METHOD};

			my $fix_args   = $action->{$ACTION_FIX_ARGS};

			TXK::Util->copyHash( $self->{$PROG_ARGS} , $fix_args );

			print ( "TXK::Validate:ProcessAutoFixActions printing methods before calling $method\n") if ( $self->{$DEBUG} );

			TXK::Util->printDataStr($fix_args) if ( $self->{$DEBUG} );

			eval {
			    my $retVal = $object->$method($fix_args);
			};
			
			if ( $@ )
			 {
			     print ("TXK::Validate::ProcessFixActions:Error: in executing method $method\n");

			     print ( $@ );

			     next ;
			 }
			
			$noMethodsExecuted++;
		    }
	       }
	 }	 
     }
    return $noMethodsExecuted ;
}

#####################################
# Process Actions
#####################################

sub ProcessActions
{
    my $self = $ARG[0];

    my $args = $ARG[1];

    TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

    return unless ( $self->{$DOC_LOADED} eq TXK::Util::TRUE ) ;

    # first validate the actions

    TXK::Util->copyHash( $args->{'args'} , $self->{$PROG_ARGS} ) if ( exists  $args->{'args'} );

    $self->_validateActions();

    # set the ALLSTATUS to allpass since it will be FAIL or WARN later
    # depending on actions

    $self->{$ALLSTATUS} = $STATUS_ALLPASS;

    #------------------------------------------------------------------------
    # re-initialize the OBJECT CONTAINER HASH so that if any of these objects
    # cache any information, will be re-initialized.
    # done because of bug 3995645
    #-------------------------------------------------------------------------

    $self->{$OBJECT_CONTAINER_HASH} = {} ; 

    my $objectContainerHash = $self->{$OBJECT_CONTAINER_HASH};

    my $action;

    foreach $action ( @{$self->{$ACTION_LIST}} )
     {
	 if ( $action->{$ACTION_VALID} )
	  {
	      print ("Txk:Validate:ProcessActions $action->{$ACTION_NAME} is  valid. \n") if ( $self->{$DEBUG} );

	      my $noKey = "_no".$action->{$ACTION_NAME};

	      next if ( ( exists $args->{'args'}->{$noKey} ) and
			( $args->{'args'}->{$noKey} eq TXK::ARGS::YES  ) ) ;

	      unless ( exists $objectContainerHash->{$action->{$ACTION_TYPE}} )
	       {
		   #-------------------------------------------------
		   # Load the object dynamically
		   # This does not catch the error if the module 
		   # could not be loaded successfully, which is 
		   # caught in the new() method call.
		   #-------------------------------------------------

		   unless ( eval "require $action->{$ACTION_TYPE}" )
		    {
			TXK::Error::abort("Could not load : $@");
		    }

		   eval {

		       my $classType = ACTION_OBJECT_CLASS_TYPE ;

		       if ( $action->{$ACTION_OBJECT_TYPE} =~ m/^$classType$/i )
		        {
			    $objectContainerHash->{$action->{$ACTION_TYPE}} = $action->{$ACTION_TYPE};
			}
		       else
		        {
			    $objectContainerHash->{$action->{$ACTION_TYPE}} 
			                = $action->{$ACTION_TYPE}->new({ $APPS_CONTEXT_FILE => $self->{$APPS_CONTEXT_FILE},
                                                                        $APPS_USER     => $self->{$APPS_USER},
                                                                        $APPS_PASSWORD => $self->{$APPS_PASSWORD}
                                                                       });
			}

		   };

		   if ( $@ )
		    {
			TXK::Error::abort("Could not create instance of $action->{$ACTION_TYPE} : $@");
		    }

		   print ("an instance $action->{$ACTION_TYPE} created \n") if $self->{$DEBUG} ;
	       }

	      my $object = $objectContainerHash->{$action->{$ACTION_TYPE}};

	      my $method = $action->{$ACTION_EXEC_METHOD};

	      my $conditionMethod = $action->{$ACTION_CONDITION} ;

	      my $conditionReturnVal ;

	      #-----------------------------------------------------------
	      # check if there is a conditional method for this action.
	      # call the condition method first to check if this action
	      # needs to be processed. also we cache the result of each method
	      # so that we do not have to call the same method twice
	      #----------------------------------------------------------

	      if ( $conditionMethod and ! defined $self->{$METHOD_RETURN_CACHE}->{$conditionMethod} )
	       {
		   print ("TXK:Validate:ProcessActions: Executing conditional method - $conditionMethod \n");
		   
		   eval {

		       $conditionReturnVal = $object->$conditionMethod();
		   };

		   if ( $@ )
		    {
			print ("Error calling condition method $method for $action->{$ACTION_TYPE} : $@");

		        $conditionReturnVal = TXK::Util::FALSE ;
		    }

		   print ("TXK:Validate:ProcessActions: Conditional method - $conditionMethod comleted successfully.\n\n");
	       }
	      else
	       {
		   $conditionReturnVal = ( defined $self->{$METHOD_RETURN_CACHE}->{$conditionMethod} ) ? 
		                                   $self->{$METHOD_RETURN_CACHE}->{$conditionMethod} : 
						   TXK::Util::TRUE;

	       }

	      if ( $conditionReturnVal )
	       {
		   my $method_args = { };

		   TXK::Util->copyHash( $self->{$PROG_ARGS} , $method_args );

		   my $returnVal;
		   
		   #----------------------------------------------------------------------
		   # call the action method on the object for this action. if there are any
		   # input parameters specified then pass them as well. also we need to pass 
		   # the reports context which contains all the variables to this method.
		   # we are not passing the report ctx anymore. 
		   #-----------------------------------------------------------------------

		   #$args->{$REPORT_CTX} = $self->{$REPORT_CTX};

		   if ( defined $action->{$ACTION_INPUTPARAMS} )
		    {
			my $argList =  [ split ( "$INPUT_PARAM_DELIM",$action->{$ACTION_INPUTPARAMS} ) ] ;
			
			my ( $arg , $key , $value);
			
			foreach $arg ( @$argList )
			 {
			     ( $key , $value ) = split ("$INPUT_PARAM_VALUE_DELIM",$arg);

			     $method_args->{$key} = $value;

			     print ("Validate:ProcessActions: Input Params key is $key and value is $value \n") if ( $self->{$DEBUG} );
			 }
		    }
		       
		   eval {
		       
		       #-------------------------------------------------------------------------
		       # here we are not checking if this method's return value is in the cache
		       # since we are only caching the method names and not the arguments. the 
		       # cache applies more to conditional methods where no arguments are passed.
		       #--------------------------------------------------------------------------

		       print ("Validate:ProcessAction: printing action before invoking method.\n") if ( $self->{$DEBUG} );

		       TXK::Util->printDataStr($action) if ( $self->{$DEBUG} ) ;

		       print ("Validate:ProcessAction: printing arguments before invoking method.\n") if ( $self->{$DEBUG} );

		       TXK::Util->printDataStr($method_args) if ( $self->{$DEBUG} ) ;

		       print ("TXK:Validate:ProcessActions: Executing action method - $method \n");

		       $returnVal = $object->$method($method_args);
		   };
	      
		   if ( $@ )
		    {
			print ("Error invkoing : $@ \n");
		    }
		   else
		    {
			print ("TXK:Validate:ProcessActions: Action method - $method completed successfully.\n\n");

			$self->{$METHOD_RETURN_CACHE}->{$method} = $returnVal ;
			
			print ("ProcessAction: $method returned : $returnVal \n") if ( $self->{$DEBUG} );

			if ( ( $action->{$ACTION_RETURN_TYPE} =~ m/^$ACTION_RETURN_XML_TYPE$/i   ) or
			     ( $action->{$ACTION_RETURN_TYPE} =~ m/^$ACTION_RETURN_ARRAY_TYPE$/i ) or
			     ( $action->{$ACTION_RETURN_TYPE} =~ m/^$ACTION_RETURN_HASH_TYPE$/i  ) )
			 {
			     TXK::Util->isValidObj({obj=>$returnVal,package=>"TXK::XML"})
				 if ( $action->{$ACTION_RETURN_TYPE} =~ m/^$ACTION_RETURN_XML_TYPE$/i   ) ;

			     TXK::Error->stop("Return type mismatch for method $method") 
				 if ( ( $action->{$ACTION_RETURN_TYPE} =~ m/^$ACTION_RETURN_ARRAY_TYPE$/i   ) and
				      ( ref ($returnVal) ne 'ARRAY') );

			     TXK::Error->stop("Return type mismatch for method $method") 
				 if ( ( $action->{$ACTION_RETURN_TYPE} =~ m/^$ACTION_RETURN_HASH_TYPE$/i   ) and
				      ( ref ($returnVal) ne 'HASH') );

			     $action->{$ACTION_PROCESSED} = TXK::Util::TRUE;			     

			     $action->{$ACTION_STATUS}    = ACTION_SUCCESS;			     

			     $action->{$ACTION_VERSION}   = $returnVal ;

			     next;
			 }

			$self->{$REPORT_CTX}->setOAVar ({ $action->{$ACTION_RESULT_VAR} => $returnVal });

			if (  ( ! defined  $action->{$ACTION_VERSION} ) or
			      ( $action->{$ACTION_VERSION} =~ /.*/ ) ) 
			 {
			     #----------------------------------------------------------------------
			     # set it to the resultvar only if resultvar is defined for this action
			     # it may be the case that this is only a check method and does not need to
			     # report any version.
			     #-----------------------------------------------------------------------

			     $action->{$ACTION_VERSION} = $returnVal if ( defined $action->{$ACTION_RESULT_VAR} );
			 }

			$action->{$ACTION_PROCESSED} = TXK::Util::TRUE;

			#-------------------------------------------------------------------------------------
			# here we got the return value. validate the return value with the success,warning
			# or failure values. and then set the status of this action (PASS or FAIL or WARN etc)
			#--------------------------------------------------------------------------------------

			my $succList = [ split($action->{$ACTION_DELIM},$action->{$ACTION_SUCCESS_VALUES}) ];

			my $failList = [ split($action->{$ACTION_DELIM},$action->{$ACTION_FAIL_VALUES}) ];

			my $warnList = [ split($action->{$ACTION_DELIM},$action->{$ACTION_WARN_VALUES}) ];
			
			my ($succ,$warn,$fail);

			# first check the failure list

		        $fail = $self->_checkReturnValue($action,$failList,$returnVal);

		        $warn = $self->_checkReturnValue($action,$warnList,$returnVal);

			$succ = $self->_checkReturnValue($action,$succList,$returnVal);

			if ( scalar(@$failList) and $fail  )  
			 {
			     # this is a failure

			     $action->{$ACTION_STATUS} = ACTION_FAILURE;
			 }
			elsif ( scalar(@$warnList) and $warn )
			 { 
			     # this is a warning

			     $action->{$ACTION_STATUS} = ACTION_WARNING;
			 }
			elsif ( scalar (@$succList) and $succ )
			 {
			     # this is a success

			     $action->{$ACTION_STATUS} = ACTION_SUCCESS;
			 }
			elsif ( !scalar (@$failList) and defined $action->{$ACTION_FAILURE_MESSAGE} )
			 {
			     #------------------------------------------------------
			     # here the value does not fall into any categories
			     # but if there is a failure message defined for this
			     # then the status should be fail. this is the case where
			     # we might have only success values specified, but also
			     # a failure message defined. if the return value is not
			     # in the success list then we should show the result
			     # as a failue.
			     #--------------------------------------------------------

			     $action->{$ACTION_STATUS} = ACTION_FAILURE;
			 }
			elsif ( !scalar (@$warnList) and defined $action->{$ACTION_WARN_MESSAGE} )
			 {
	      		     $action->{$ACTION_STATUS} = ACTION_WARNING;
			 }
			elsif ( !scalar (@$succList) and defined $action->{$ACTION_SUCCESS_MESSAGE} )
			 {
	      		     $action->{$ACTION_STATUS} = ACTION_SUCCESS;
			 }
			elsif( !scalar(@$failList) and !scalar(@$warnList) and !scalar(@$succList) )
			 {
			     # this is the case where no values were specified 

			     $action->{$ACTION_STATUS} = ACTION_SUCCESS;
			 }
			else
			 {
			     # this is where  none of them has matching value for the returned result

			     $action->{$ACTION_STATUS} = ACTION_FAILURE;
			 }

			print ("ProcessAction:for $action->{$ACTION_EXEC_METHOD} succ: $succ fail: $fail warn: $warn \n") 
			                                                                           if ( $self->{$DEBUG} );

		       print ("\nValidate:ProcessAction: printing action AFTER invoking method.\n") if ( $self->{$DEBUG} );

		       TXK::Util->printDataStr($action) if ( $self->{$DEBUG} ) ;

		    }
	       }
	  }
	 else
	  {
	      print ("Txk:Validatate:ProcessActions $action->{$ACTION_NAME} is not valid\n") if ( $self->{$DEBUG} );
	  }
     }

    return TXK::Error::SUCCESS ;
}

#################################
# sub getAllStatus
################################

sub getAllStatus
{
    my $self = $ARG[0];

    TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

    return $self->{$ALLSTATUS};
}

################################
# sub getReport
# returns the xml report objet
###############################

sub getReport
{
    my $self = $ARG[0];

    TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

    return $self->{$REPORT};
}

################################
# sub instantiateReport
################################

sub instantiateReport
{
    my $self = $ARG[0];

    TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

    my $fsys = $self->{$FILE_SYS};


    print ("TXK::Validate::instantiateReport after instantiating defaults \n") if ( $self->{$DEBUG} );

    my $acfg         = TXK::AutoConfig->new();

    my $outFile      = TXK::OSD->trDirPathToBase( $TEMP_DIR. "/txkvalreport.out" );

    my $templateFile = TXK::OSD->trDirPathToBase( $TEMP_DIR. "/report.xml" );
print "templateFile is $templateFile\n";

    $self->{$REPORT}->storeDocument({ file  => $templateFile});

    #------------------------------------------------------
    # first instantiate with the applications context
    # then instantiate with the reports context
    #------------------------------------------------------

    my $xml = TXK::XML->new();

    $xml->loadDocument( { file => $self->{$APPS_CONTEXT_FILE} } );

    if ( $self->_getTechStackObj()->txkIsDBContext({ $APPS_CONTEXT_FILE => $self->{$APPS_CONTEXT_FILE} } ) )
     {
	 #--------------------------------------------------------
	 ## since the db context does not have s_dbdomain variable
	 #  we need to have this variable set to s_domainname
	 #--------------------------------------------------------

	 $xml->addNode( { 'node'     => 'dbdomain',
			  'value'    => $xml->getOAVar ( 's_domainname' ),
			  'parent'   => 'oa_host',
			  'attrlist' => { 'oa_var' => 's_dbdomain' }
		      } );

     }

    $acfg->instantiate( { template => $templateFile,
			  output   => $outFile,
			  xmlObj   => $xml
		         } );

    print ("TXK::Validate::instantiateReport after instantiating with apps context file  \n") if ( $self->{$DEBUG} );

    #-----------------------------------------------------------------
    # copy the out file to be the template again to instantiate again
    # with the report context
    #-----------------------------------------------------------------

    $fsys->copy ( { source => $outFile , dest => $templateFile } ); 

    $acfg->instantiate( { template => $templateFile,
			  output   => $outFile,
			  xmlObj   => $self->{$REPORT_CTX} 
		         } );

    print ("TXK::Validate::instantiateReport after instantiating with reports context.  \n") if ( $self->{$DEBUG} );

    $self->{$REPORT} = TXK::XML->new();

    $self->{$REPORT}->loadDocument( { file => $outFile} );

    unless ( $self->{$DEBUG} )
     {
	 
	 $fsys->rmfile ( { fileName => $templateFile } )
	     or print ("instantiateReport: could not remove $templateFile \n");

	 $fsys->rmfile ( { fileName => $outFile  } ) 
	     or print  ("instantiateReport: could not remove $outFile\n");
     }

    print ( "TXK::Validate::instantiateReport printing report. \n") if ( $self->{$DEBUG} );

    TXK::Util->printDataStr($self->{$REPORT_CTX} ) if ( $self->{$DEBUG} );

    my $txkReport = TXK::Reports->new();

    if ( $self->{$DEBUG} )
     {
        my $tempDir =  $self->_getTechStackObj()->getAppsTempDir({$APPS_CONTEXT_FILE => $self->{$APPS_CONTEXT_FILE} }) ;

        $self->{$REPORT}->storeDocument( { file => TXK::OSD->trDirPathToBase($tempDir."/txkvalreport.xml" ) } );
     }

    my $reportArgs = { "InputXML"     => $self->{$REPORT} , 
		       "OutputReport" => $self->{$REPORT_FILE},
		       "$REPORT_TYPE" => $self->{$REPORT_TYPE},
		       "techstack"    => $self->_getTechStackObj()
		       };

    if ( $self->{$REPORT_TYPE} eq TXK::Reports::REPORT_HTML )
     {
	 $reportArgs->{$REPORT_XSL} =  $self->{$REPORT_XSL};
     }

    $txkReport->loadDocument(  $reportArgs );

    return $txkReport;
}

#######################################
#sub _instantiateReportDefaults
#######################################

sub _instantiateReportDefaults
{
    my $self = $ARG[0];

    #-----------------------------------------------------------------------
    # instantiate the following variables:
    #     t_date       => to the current date
    #     t_platform   => to the current platform 
    #     t_context    => to the applications context file
    #     t_repfile    => to the location of the output report
    #     t_nodetypes  => to the comma separated node list
    #     t_defnfile   => this is the xml definition file that was processed
    #-------------------------------------------------------------------------

    my $fsys = TXK::FileSys->new();

    $DEFAULT_REP_VARS->{'t_date'}      =  TXK::Util->getTimestamp();

    $DEFAULT_REP_VARS->{'t_platform'}  = TXK::OSD->getAutoConfigName();

    $DEFAULT_REP_VARS->{'t_os_release'}= TXK::OSD->getOSRelease();

    $DEFAULT_REP_VARS->{'t_context'}   = $self->{$APPS_CONTEXT_FILE};

    $DEFAULT_REP_VARS->{'t_repfile'}   = $self->{$REPORT_FILE};

    $DEFAULT_REP_VARS->{'t_defnfile'}   = $self->{$XML_FILE};

    my ($version, @identData);

    $version = $fsys->ident({  fileName => "$self->{$XML_FILE}",
			       identData => \@identData
			    });

    if ( ( $version ne TXK::FileSys::NO_IDENT         ) and 
	 ( $version ne TXK::FileSys::MULTIPLE_IDENTS  ) ) 
     {  
	 $DEFAULT_REP_VARS->{'t_defnfile_ver'}  =  $version;
     }

    my $ctx_ver ;

    $ctx_ver = $fsys->ident({  fileName => "$self->{$APPS_CONTEXT_FILE}",
			       identData => \@identData
			    });

    if ( ( $ctx_ver ne TXK::FileSys::NO_IDENT         ) and 
	 ( $ctx_ver ne TXK::FileSys::MULTIPLE_IDENTS  ) ) 
     {
	 $DEFAULT_REP_VARS->{'t_ctxfile_ver'}  =  $ctx_ver;	 
     }

    my $validNodeList = "";

    my $nodeType;

    foreach $nodeType ( keys %{$self->{$NODETYPES}} )
     {
	 $validNodeList .= $self->{$NODETYPES}->{$nodeType}->{$NODE_TYPE_TAG} . ","

	  if ( $self->_checkValidNodeType ( [ $self->{$NODETYPES}->{$nodeType}->{$NODE_TYPE_TAG} ] ) );
     }

    $validNodeList =~ s/(")|(,$)//g ;

    print ("TXK::Validate::_instantiateReportDefaults validNodeList is: $validNodeList \n") if ( $self->{$DEBUG} ); 

    $DEFAULT_REP_VARS->{'t_nodetypes'} = $validNodeList ;

    my $key;

    foreach $key ( keys %$DEFAULT_REP_VARS )
     {
	 $self->{$REPORT_CTX}->setOAVar( {  $key => $DEFAULT_REP_VARS->{$key} } );
     }


  return TXK::Error::SUCCESS;
}


################################
# sub _processPassword
################################

sub _processPassword
{
    my $self = $ARG[0];

    my ($username,$passwd) ;

    #---------------------------------------------------------------
    # if this is called from OAM, then the password comes from STDIN
    # in <username>/<password> form. 
    #--------------------------------------------------------------

    if ( $ARG[1] =~ m/\// )
     {
	 ( $username, $passwd ) = split ( /\// , $ARG[1] );

	 print ( "Extracted Username : $username \n" ) if $self->{$DEBUG} ;
     }
    else
     {
	 $passwd = $ARG[1];

	 $username = undef;
     }
	 
    return ( $username,$passwd );
}

sub _checkReturnValue($$$)
{
    my $self = $ARG[0];

    my $action       = $ARG[1];

    my $valueList    = $ARG[2];

    my $valueToCheck = $ARG[3];

    my $value;

    my $returnVal = TXK::Util::FALSE ;

    print ("_checkReturnValue: value to check : $valueToCheck \n") if ( $self->{$DEBUG} );

    foreach $value ( @$valueList )
     {
	 my $plus = TXK::Util::FALSE;

	 $plus = TXK::Util::TRUE if ( $value =~ s/\+$// );

	 my $greaterThan = TXK::Util::FALSE ;

	 my $lessThan    = TXK::Util::FALSE ;

	 my $star        = TXK::Util::FALSE ;
	 
	 $greaterThan = TXK::Util::TRUE if ( $value =~ s/^>// ) ;

	 $lessThan    = TXK::Util::TRUE if ( $value =~ s/^<// ) ;

	 $star        = TXK::Util::TRUE if ( $value =~ s/(^\*|\*$)// );

	 print ("_checkReturnValue: $value valueToCheck : $valueToCheck \n") if ( $self->{$DEBUG} );

	 if ( $plus )
	  {
	      #--------------------------------------------------------------------
	      # we need to convert the underscores to a consistent delimiter.
	      # some values will contain underscores in it e.g. 1.3.1_07 and 
	      # this substitution is for them. but only change them for plus 
	      #---------------------------------------------------------------------

	      my $localValueToCheck = $valueToCheck;

	      $localValueToCheck =~ s/_/$action->{ACTION_VALUE_DELIM}/;

	      $value =~ s/_/$action->{ACTION_VALUE_DELIM}/;

	      #-----------------------------------------------------------------------
	      # we split the values using the ACTION_VALUE_DELIM
	      # then we check individual numbers/chars
	      # so we support only values of the format 1.2.3.4+ or 1.2.3.a+
	      # we do not support values 1.2.3.4a+. those values should be 
	      # checked exactly (without plus) and should be defined in the xml file
	      # accordingly
	      #
	      # TODO: Logic needs to be changed as implemented in _checkValidOSVersion method
	      #------------------------------------------------------------------------


	      my $list1 =  [ split(/$action->{$ACTION_VALUE_DELIM}/,$localValueToCheck) ];

	      my $list2 =  [ split(/$action->{$ACTION_VALUE_DELIM}/,$value) ];

	      my $index = 0;

	      while ( $index < scalar(@$list2) )
	       {
		   # check to see if this is an integer

		   print ("_checkReturnValue: comparing $list1->[$index] with $list2->[$index] \n") if ( $self->{$DEBUG} );

		   if ( $list1->[$index] =~ m/\d/ )
		    {
			$returnVal = ( $list1->[$index] >= $list2->[$index] ) ?
			                                      TXK::Util::TRUE :
							      TXK::Util::FALSE ;

		    }

		   # now check for the character ( case insensitive, convert to upper case )

		   else 
		    {
			$returnVal = (  ord(uc($list1->[$index])) >= ord(uc($list2->[$index])) ) ?
			                                                         TXK::Util::TRUE :
							                         TXK::Util::FALSE ;
		    }

		   print ("_checkReturnValue: after comparing returnValue is $returnVal \n") if ( $self->{$DEBUG} );

		   last unless ($returnVal); # no need to check further since this indicates the
		                             # the value does not match the required criteria 

		   $index++;
	       }

	  }
	 elsif( $greaterThan )
	  {
	      #-------------------------------------------------------------------------
	      # here we check when the return value is greater than the required value
	      # so it is imperative that it is a number
	      #------------------------------------------------------------------------

	      $returnVal = TXK::Util::TRUE if ( $valueToCheck > $value );

	  }
	 elsif ( $lessThan )
	  {
	      #-------------------------------------------------------------------------
	      # here we check when the return value is less than the required value
	      #------------------------------------------------------------------------

	      $returnVal = TXK::Util::TRUE if ( $valueToCheck < $value );
	  }
	 elsif ( $star )
	  {
	      $returnVal = ( $valueToCheck =~ m/$value/ ) ?
		                          TXK::Util::TRUE :
					  TXK::Util::FALSE;
	  }
	 else
	  {
	      $returnVal = ( $value =~ m/^$valueToCheck$/ ) ?
 		                            TXK::Util::TRUE :
					    TXK::Util::FALSE;
	  }

         return $returnVal if ( $returnVal ) # we found one match so return. no need to check
	                                     # other values.
     }

    return $returnVal ; # should equal to FALSE since we did not find any match.
}

########################################
#
# getAutoFixActionList
#
#######################################

sub getAutoFixActionList
{
    my $self = $ARG[0];

    my $args = $ARG[1];

    my $autoFixActionHash = {};

    TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

    my $action;

    foreach $action ( @{$self->{$ACTION_LIST}} )
     {
	 if ( ( $action->{$ACTION_PROCESSED} ) and
	      ( $action->{$ACTION_STATUS} eq ACTION_FAILURE  ) )
	      
	        #-------------------------------------------------
	        # only if the action has a failure for now.
		# or ( $action->{$ACTION_STATUS} eq ACTION_WARNING ) ) )
	        #-------------------------------------------------

	  {
	      #--------------------------------------------------------
	      # add to the list only if this action has a fixMethod the
	      #--------------------------------------------------------

	      if ( ( defined $action->{$ACTION_FIX_METHOD} ) and
		   ( $action->{$ACTION_FIX_METHOD} ne '' ) )
	       {
		   my $prompt = $action->{$ACTION_FIX_METHOD} ;

		   $prompt = $action->{$ACTION_FIX_PROMPT} if ( $action->{$ACTION_FIX_PROMPT} ne '' );

		   $autoFixActionHash->{$action->{$ACTION_FIX_METHOD}} = $prompt;
	       }
	  }
     }

    return $autoFixActionHash;
}


#########################################
# genereateReport
# 
#########################################

sub generateReport
{
    my $self = $ARG[0];

    my $args = $ARG[1];

    TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

    TXK::Util->isValidArgs({args=>$args,
			    reqd=> ["$REPORT_FILE","$REPORT_TYPE"]} );

    $self->{$REPORT_FILE} = $args->{$REPORT_FILE};

    $self->{$REPORT_TYPE} = $args->{$REPORT_TYPE};

    $self->{$REPORT_XSL} = $args->{$REPORT_XSL}
             if ( exists $args->{$REPORT_XSL} );

    my $report = TXK::XML->new();

    my $action;

    my $report = TXK::XML->new();

    my $xmlHeader = $self->{$XML_HEADER}->getNode({node=>'?xml'});

    $report->addNode({'node' => {'type' => 'xmlnode',
				 'value' => $xmlHeader }
		      });

    $report->addNode( {'node'     => TXK_REPORT_TAG,
		       'attrlist' => { 'Type' => $self->{$REP_DEFN_TYPE}} 
                    } );

    $report->addNode( {'node'   => TITLE_TAG,
		       'value'  => $self->{$MSGXML}->getNodeValue({'node'=>$self->{$REPORT_TITLE}}),
		       'parent' => TXK_REPORT_TAG });

    my $headerNode = TXK::XML->new();

    $headerNode->addNode( { 'node' => { 'type' => 'xmlnode',
				'value'=>$self->{$MSGXML}->getNode({'node'=>$self->{$REP_HEADER}}) }
			} );

    $report->addNode( {'node'   => { 'type'   => 'xmlnode',
				     'value'  => $headerNode->getNode({'node'=>HEADER_TAG}) },
		       'parent' => TXK_REPORT_TAG } );

    $self->{$SUMMARY_MSG} = $self->{$SUMMARY_ALLPASS};

    #-----------------------------------------------------------------
    # in order to prevent duplicate rows when this method
    # is called twice, we must initiliaze the REPORT_NODE_HASH here.
    #------------------------------------------------------------------

    $self->{$REPORT_NODE_HASH} = {};

    my $report_node_hash = $self->{$REPORT_NODE_HASH};

    TXK::Util->printDataStr($self->{$MSGXML}->getDocument()) if ( $self->{$DEBUG} );
    
    #-----------------------------------------------------------------
    # instantiate defaults here so that we can use the default values later
    #------------------------------------------------------------------

    if ( $self->_instantiateReportDefaults() ne TXK::Error::SUCCESS )
     {
	 TXK::Error->abort("TXK::Validate::instantiateReport: Error in instantiating report defaults.\n ");
     }

    #---------------------------------------------------------------------
    # here add all the actions to their respective node sections.
    #--------------------------------------------------------------------

    foreach $action ( @{$self->{$ACTION_LIST}} )
     {
	 print ("Validate:GenerateReport: Generating report for $action->{$ACTION_EXEC_METHOD}.\n") if ( $self->{$DEBUG} );

	 if ( $action->{$ACTION_PROCESSED} )
	  {
	      my $actionXML = $self->_getActionXML($action);

	      next unless( defined $actionXML );

	      $self->_addActionToReportNode($action,$actionXML);
	  }
	 else
	  {
	      print ("Validate:GenerateReport:  $action->{$ACTION_EXEC_METHOD} is not processed.\n") if ( $self->{$DEBUG} );

	      TXK::Util->printDataStr($action) if ( $self->{$DEBUG} ) ;
	  }
     }

    my $sortedNodes = $self->_getSortedNodes();

    my $nodeType;

    foreach $nodeType ( @$sortedNodes )
     {
	 my $root = $report_node_hash->{$nodeType}->{'xml'}->getRootNodeName() ;

	 unless ( $root eq '' )
	  {
	      my $node = $report_node_hash->{$nodeType}->{'xml'}->getNode( { 'node' => $root } );

	      #------------------------------------------------------------
	      # No need to report a node which does not have any actions
	      #------------------------------------------------------------

	      next unless ( scalar ( @{$node->{'body'}} ) > 0 );

	      $report->addNode ( { 'node' => {'type' =>'xmlnode',
					      'value'=> $node },
				   'parent' => TXK_REPORT_TAG
				   } );
	  }
     }

    #-------------------------------------------------------------------
    # for Inventory types, there is no need to add the summary tag.
    #-------------------------------------------------------------------

    unless ( $self->{$REP_DEFN_TYPE} eq INVENTORY_TAG )
     {

	 $report->addNode({'node'     => SUMMARY_TAG,
			   'parent'   => TXK_REPORT_TAG,
			   'attrlist' => { $SUMMARY_TITLE       =>  $self->{$MSGXML}->getNodeValue({'node'=>$self->{$SUMMARY_TITLE}}),
					   $SUMMARY_STATUS_TEXT => $self->{$MSGXML}->getNodeValue({'node'=>$self->{$SUMMARY_STATUS_TEXT}}) }  
		       });

	 $report->addNode( { 'node'  => STATUS_TAG,
			     'value' => $self->{$ALLSTATUS},
			     'parent'=> SUMMARY_TAG } );

	 $report->addNode( { 'node'  => { 'type' => 'xmlnode',
					  'value' => _getMessageNode($self->{$MSGXML}->getNode({'node'=>$self->{$SUMMARY_MSG}}))} ,
			     'parent'=> SUMMARY_TAG } );
     }

   $self->{$REPORT} = $report;

   return $self->instantiateReport();
}


##########################################################
## sub _getSortedNodes
##########################################################

sub _getSortedNodes
{
    my $self = $ARG[0];

    my $nodeType;

    my $report_node_hash = $self->{$REPORT_NODE_HASH};

    my $sortedNodes = [];  

    #---------------------------------------------------------------
    # we want to sort the nodes, putting the first one that has
    # FAILIURE or WARNING
    #----------------------------------------------------------------

    my $ordering = TXK::Util::FALSE;

    foreach $nodeType ( keys %$report_node_hash )
     {
	 if ( $self->{$NODETYPES}->{$nodeType}->{$NODEORDER} =~ /\d+/ )
	  {
	      $ordering = TXK::Util::TRUE;

	      $NODETYPES_FOR_ORDERING->{$nodeType} = $self->{$NODETYPES}->{$nodeType}->{$NODEORDER};
	  }

	 if ( $report_node_hash->{$nodeType}->{'nodestatus'} eq ACTION_FAILURE or
	      $report_node_hash->{$nodeType}->{'nodestatus'} eq ACTION_FAILURE )
	  {
	      push ( @$sortedNodes, $sortedNodes->[0] )
		  if ( scalar( @$sortedNodes) > 0 );

	      $sortedNodes->[0] = $nodeType;
	  }
	 else
	  {
	      push ( @$sortedNodes , $nodeType );
	  }
     }

    if ( $ordering )
     {
	 my @orderedNodes = sort _orderNodes ( keys %$report_node_hash );

	 return [ @orderedNodes ];
     }
    
    return $sortedNodes;
}


sub _orderNodes
{
    return $NODETYPES_FOR_ORDERING->{$a} <=> $NODETYPES_FOR_ORDERING->{$b};
}

##########################################################
## sub _addActionToReportNode
## adds actions to the report node in the REPORT_NODE_HASH
##########################################################

sub _addActionToReportNode ( $$ )
{
    my $self = $ARG[0];

    my $action = $ARG[1];

    my $actionXML = $ARG[2];

    my $report_node_hash = $self->{$REPORT_NODE_HASH};

    my $nodeType;

    #------------------------------------------------------
    # look for all the valid node types this action lists 
    # and add it to all those report sections
    #-----------------------------------------------------

    foreach $nodeType ( keys %{$self->{$NODETYPES}})
     {
	 print ("Validate:GenerateReport: processing nodetype $nodeType for $action->{$ACTION_EXEC_METHOD}\n") 
									  if ( $self->{$DEBUG} );

	 unless ( exists $report_node_hash->{$nodeType} )
	  {
	      $report_node_hash->{$nodeType}->{'xml'} = TXK::XML->new();

	      $report_node_hash->{$nodeType}->{'nodestatus'} = ACTION_SUCCESS;
	  }

	 my $actionNode;

	 foreach $actionNode ( split(",",$action->{$ACTION_NODETYPE} ) )
	  {
	      # $DEFAULT_REP_VARS->{'t_nodetypes'}) is the comma separated list of all the valid node types

	      if ( ( $nodeType =~ m/$actionNode/i ) and 
		   ( scalar ( grep ( /^$actionNode$/i,  split ( /,/ , $DEFAULT_REP_VARS->{'t_nodetypes'}) ) ) > 0 ) )
	       {
		   print ("Validate:GenerateReport: adding to nodetype $nodeType for $action->{$ACTION_EXEC_METHOD}\n") 
	       								  if ( $self->{$DEBUG} );

		   my $reportNode = $report_node_hash->{$nodeType}->{'xml'}->getNode( {'node'=> TXK_NODE_TAG } );

		   my ( $sectionTitle,$cmpHeader,$verHeader,$staHeader);

		   my $nodeAttrList = {};

		   $sectionTitle = $self->{$NODETYPES}->{$actionNode}->{$NODE_HEADER};

		   $sectionTitle = $self->{$MSGXML}->getNodeValue({'node'=>$sectionTitle});

		   $cmpHeader = $self->{$MSGXML}->getNodeValue
					     ({'node'=> $self->{$NODETYPES}->{$actionNode}->{$COMPONENT_HEADER}});

		   $verHeader = $self->{$MSGXML}->getNodeValue
					     ({'node'=> $self->{$NODETYPES}->{$actionNode}->{$VERSION_HEADER}});

		   $staHeader = $self->{$MSGXML}->getNodeValue
					     ({'node'=> $self->{$NODETYPES}->{$actionNode}->{$STATUS_HEADER}});

		   unless ( defined $reportNode  )
		    {
			$nodeAttrList = {    'Title'           => $sectionTitle,
			        	     'Type'            => $nodeType,
					     $COMPONENT_HEADER => $cmpHeader,
					     $VERSION_HEADER   => $verHeader,
					     $STATUS_HEADER    => $staHeader	  };

			$report_node_hash->{$nodeType}->{'xml'}->addNode
						    ( { 'node'     => TXK_NODE_TAG,
							'parent'   => TXK_REPORT_TAG,
							'attrlist' => $nodeAttrList
						     } );
		     }

		   my $reportNodeXML =  $report_node_hash->{$nodeType}->{'xml'};

	       #-------------------------------------------------------------------------------
	       # support for multiple values.
	       #-------------------------------------------------------------------------------

		   if ( ( $action->{$ACTION_RETURN_TYPE} =~ m/^$ACTION_RETURN_XML_TYPE$/i   ) or
			( $action->{$ACTION_RETURN_TYPE} =~ m/^$ACTION_RETURN_ARRAY_TYPE$/i ) or
			( $action->{$ACTION_RETURN_TYPE} =~ m/^$ACTION_RETURN_HASH_TYPE$/i  ) )
		    { 
			my $nodeTypeAction = "$nodeType-$action->{$ACTION_NAME}";

			unless ( exists $report_node_hash->{$nodeTypeAction} )
		 	 {
			     $report_node_hash->{"$nodeType-$action->{$ACTION_NAME}"}->{'xml'} = TXK::XML->new();
			 } 

			my $actionNodeXML = $report_node_hash->{$nodeTypeAction}->{'xml'} ;

			my ( $sectionTitleAction,$cmpHeaderAction,$verHeaderAction,$staHeaderAction);

			$sectionTitleAction = $action->{$NODE_HEADER};

			$sectionTitleAction = $self->{$MSGXML}->getNodeValue({'node'=>$sectionTitle});

			$cmpHeaderAction    = $self->{$MSGXML}->getNodeValue
					     ({'node'=> $action->{$COMPONENT_HEADER}});

			$verHeaderAction = $self->{$MSGXML}->getNodeValue
					     ({'node'=> $action->{$VERSION_HEADER}});

			$staHeaderAction = $self->{$MSGXML}->getNodeValue
					     ({'node'=> $action->{$STATUS_HEADER}});

			#--------------------------------------------------------------------
			# Fall back to the messages for node if they are not defined in
			# the action.
			#-------------------------------------------------------------------

			$sectionTitleAction = $sectionTitle if ( $sectionTitleAction eq '' );

			$cmpHeaderAction = $cmpHeader if ( $cmpHeaderAction eq '' );

			$verHeaderAction = $verHeader if ( $verHeaderAction eq '' );

			$staHeaderAction = $staHeader if ( $staHeaderAction eq '' );

			$nodeAttrList = {    'Title'           => $sectionTitleAction,
					     'Type'            => $nodeTypeAction,
					     $COMPONENT_HEADER => $cmpHeaderAction,
					     $VERSION_HEADER   => $verHeaderAction,
					     $STATUS_HEADER    => $staHeaderAction	  };

			if  ( $action->{$ACTION_RETURN_TYPE} =~ m/^$ACTION_RETURN_ARRAY_TYPE$/i ) 
			 {
			     $nodeAttrList->{$NODE_TABLE_TYPE_TAG} = $SINGLE_VALUE_TABLE;
			 }
			elsif ( $action->{$ACTION_RETURN_TYPE} =~ m/^$ACTION_RETURN_HASH_TYPE$/i  ) 
			 {
			     $nodeAttrList->{$NODE_TABLE_TYPE_TAG} = $NAME_VALUE_PAIR_TABLE;				  
			 }

			$actionNodeXML->addNode( { 'node'     => TXK_NODE_TAG,
						   'parent'   => TXK_REPORT_TAG,
						   'attrlist' => $nodeAttrList
					       } );

			$reportNodeXML = $actionNodeXML;
		    }

		   my $actionTree = $actionXML->getDocument();

		   my $element;

		   foreach $element ( @$actionTree )
		    {
			$reportNodeXML->addNode( { 'node'   => { 'type'   => 'xmlnode',
								 'value'  => $element },
						   'parent' => TXK_NODE_TAG
						} );
		    }

		   if ( ( $action->{$ACTION_STATUS} eq ACTION_FAILURE ) or 
			( $action->{$ACTION_STATUS} eq ACTION_WARNING )  )
		    {
			$report_node_hash->{$nodeType}->{'nodestatus'} = $action->{$ACTION_STATUS};
		    }
	       }
	      else
	       {
		   print ("Validate:GenerateReport: Not adding to nodetype $nodeType for $action->{$ACTION_EXEC_METHOD}\n") 
										  if ( $self->{$DEBUG} );
	       }
	  }
      }

    return TXK::Error::SUCCESS;
}

##########################################
# sub _getActionXML
##########################################

sub _getActionXML ( $ )
{
      my $self   = $ARG[0];

      my $action = $ARG[1];

      my $actionXML = TXK::XML->new();

      if ( $action->{$ACTION_RETURN_TYPE} =~ m/^$ACTION_RETURN_XML_TYPE$/i   )
       {
	   #----------------------------------------------------------------------
	   # if the return type is xml, then just return it. it is the calle's 
	   # responsibility to see that the xml is of the proper format
	   #----------------------------------------------------------------------

	   $actionXML->addNode( { 'node' => { 'type'  => 'xmlnode',
					      'value' => $self->{$ACTION_VERSION} }
			      } );

	   $action->{$ACTION_VERSION}->storeDocument( { file => TXK::OSD->getEnvVar({name=>'APPLTMP'})."/actionXML.xml"});

	   return $action->{$ACTION_VERSION} ;
					  
       }
      elsif( $action->{$ACTION_RETURN_TYPE} =~ m/^$ACTION_RETURN_ARRAY_TYPE$/i ) 
       {
	   #-----------------------------------------------------------------------
	   # if the return type is an array then add as many actions as there are rows
	   # and each action will have the message part populated with each row value
	   # no need to check the type of the object which is checked after the 
	   # execution time
	   #-----------------------------------------------------------------------	   

	   my $subActionList = $action->{$ACTION_VERSION};

	   my $subAction;

	   foreach $subAction ( @$subActionList )
	    {
		my $subActionXML = TXK::XML->new();

		$subActionXML->addNode( {'node'     => REPORT_ACTION_TAG,
				          'attrlist' => {'name' => $action->{$ACTION_NAME}}
				  } );

		$subActionXML->addNode( {'node'  => MESSAGE_TAG ,
					 'value' => $subAction  ,
					 'parent'=> REPORT_ACTION_TAG } );

		my $tree = $subActionXML->getDocument();

		$actionXML->addNode( { 'node'  => { 'type'  => 'xmlnode',
						    'value'  => $tree->[0] }
				   } );

	    }
	   
	   $actionXML->storeDocument( { file => TXK::OSD->getEnvVar({name=>'APPLTMP'})."/actionXML.xml"});

	   return $actionXML ;
	   
       }
      elsif( $action->{$ACTION_RETURN_TYPE} =~ m/^$ACTION_RETURN_HASH_TYPE$/i  )
       {
	   #-----------------------------------------------------------------------
	   # if the return type is a hash then add as many actions as there are keys
	   # and each action will have the message part populated with each key 
	   # and the version part will be populated with the value of that key.
	   # no need to check the type of the object which is checked after the 
	   # execution time
	   #-----------------------------------------------------------------------	   

	   my $subActionHash = $action->{$ACTION_VERSION};

	   my $subActionKey;

	   foreach $subActionKey ( keys %$subActionHash )
	    {
		my $subActionXML = TXK::XML->new();

		$subActionXML->addNode( {'node'     => REPORT_ACTION_TAG,
				          'attrlist' => {'name' => $action->{$ACTION_NAME}}
				  } );

		$subActionXML->addNode( {'node'  => MESSAGE_TAG ,
					 'value' => $subActionKey  ,
					 'parent'=> REPORT_ACTION_TAG } );

		$subActionXML->addNode( {'node'  => ACTION_VERSION_TAG ,
					 'value' => $subActionHash->{$subActionKey}  ,
					 'parent'=> REPORT_ACTION_TAG } );

		my $tree = $subActionXML->getDocument();

		$actionXML->addNode( { 'node'  => { 'type'  => 'xmlnode',
						    'value'  => $tree->[0] }
				   } );
	    }

	   $actionXML->storeDocument( { file => TXK::OSD->getEnvVar({name=>'APPLTMP'})."/actionXML.xml"});

	   return $actionXML ;	   

       }

      $actionXML->addNode( {'node'     => REPORT_ACTION_TAG,
			    'attrlist' => {'name' => $action->{$ACTION_NAME}}
			    } );

      #------------------------------------------------------
      # for Inventory types, there is no need to add the status
      # tag. 
      #------------------------------------------------------

      unless ( $self->{$REP_DEFN_TYPE} eq INVENTORY_TAG )
       {
	   $actionXML->addNode( { 'node'   => STATUS_TAG,
				  'value'  => $action->{$ACTION_STATUS},
				  'parent' => REPORT_ACTION_TAG
				  } );
       }

      my $setAllSummaryFail = TXK::Util::FALSE;

      if ( defined $action->{$ACTION_VERSION} )
       {
	   $actionXML->addNode( { 'node'   => ACTION_VERSION_TAG,
				  'value'  => $action->{$ACTION_VERSION} ,
				  'parent' => REPORT_ACTION_TAG
				  } );
       } 

      my $message = "Message Not Found";

      if ( $action->{$ACTION_STATUS} eq ACTION_SUCCESS )
       {
	   $message = $action->{$ACTION_SUCCESS_MESSAGE};
       }
      elsif ( $action->{$ACTION_STATUS} eq ACTION_FAILURE )
       {
	   $message = $action->{$ACTION_FAILURE_MESSAGE};

	   $setAllSummaryFail = TXK::Util::TRUE;

       }
      elsif ( $action->{$ACTION_STATUS} eq ACTION_WARNING )
       {
	   $message = $action->{$ACTION_WARN_MESSAGE};

	   #-----------------------------------------------------------
	   # only if the ALLSTATUS is not fail
	   #----------------------------------------------------------

	   unless ( $self->{$ALLSTATUS} eq $STATUS_FAIL )
	    {
		$self->{$SUMMARY_MSG} = $self->{$SUMMARY_WARN};
	    }

	   #$self->{$ALLSTATUS} = $STATUS_WARN

	   #    if ( $self->{$ALLSTATUS} =~ m/$STATUS_ALLPASS/ );
       }

      print ("Validate:GenerateReport: retrieving the message number $message for $action->{$ACTION_EXEC_METHOD}\n") 
									 if ( $self->{$DEBUG} );
      my $msgNode = $self->{$MSGXML}->getNode({'node'=>$message}) ;

      print ("Validate:generateReport: printing the message node before processing\n ") if ( $self->{$DEBUG} );

      TXK::Util->printDataStr($msgNode) if ( $self->{$DEBUG} );

      $message = _getMessageNode ( $msgNode ) ;

      print ("Validate:GenerateReport: printing the message for $action->{$ACTION_EXEC_METHOD}\n") 
									 if ( $self->{$DEBUG} );
      TXK::Util->printDataStr($message) if ( $self->{$DEBUG} );

      # do not want to report an action without any message

      return undef  unless ( scalar(@{$message->{'body'}} > 0 ) ); 

      #---------------------------------------------------
      # if the action does not have a failure message then
      # it has no right to make others suffer :)
      #----------------------------------------------------

      if ( $setAllSummaryFail )
       {
	   $self->{$SUMMARY_MSG} = $self->{$SUMMARY_FAIL};

	   $self->{$ALLSTATUS} = $STATUS_FAIL;
       }

      $actionXML->addNode( { 'node' => { 'type' => 'xmlnode',
					 'value'=> $message },
			     'parent' => REPORT_ACTION_TAG
			     } );

      return $actionXML ;
}


##########################################
# sub _getMessageNode
#
# the input to this method is an xml node 
# containing of the type
# <MSG-TXKV0049>
# This is a message</MSG-TXKV0049>
#
# OR
#
# <MSG-TXKV0049>
# <text>his is a message</text><url type="patch">2313432</url><text>Please apply this patch</text></MSG-TXKV0049>
#
# In the former case it returns  xml node <message>This is a message</message>
#
# In the later case it returns the xml node containing 
#
# <message><text>his is a message</text><url type="patch">2313432</url><text>Please apply this patch</text></message>
#
##########################################


sub _getMessageNode
{
    my $node = $ARG[0];

    my $xml = TXK::XML->new();

    $xml->addNode( {'node' => { 'type' => 'xmlnode',
				'value'=> $node } 
		 } );

    my $tree = $xml->getDocument();

    my ($element,$line);

    my $retXML = TXK::XML->new();

    $retXML->addNode( {'node' => MESSAGE_TAG } );

    foreach $element ( @{$tree->[0]->{'body'}} )
     {

	 if ( ref ($element) eq 'HASH' )  # this is an xml element
	  {
	    $retXML->addNode( { 'node' => { 'type' => 'xmlnode',
					    'value' => $element },
				'parent' => MESSAGE_TAG
			    } );
	  }
	 else # this is just a scalar
	  {
	      $line .= $element ;

	      $retXML->setNodeValue( { 'node'  => MESSAGE_TAG,
				       'value' => $line } );
	  }
     }

    return $retXML->getNode( {'node' => MESSAGE_TAG } ) ;
}

sub _getNodeTextForAttr( $ )
{
    my $self = $ARG[0];

    my $args = $ARG[1];

    TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

    TXK::Util->isValidArgs({args=>$args,reqd=>["tree","name","value","type"]});

    my $nodeText;

    my $tree = $args->{'tree'};

    my $attrName = $args->{'name'};

    my $attrValue = $args->{'value'};

    my $ObjectType = $args->{'type'};

    my $obj;

    foreach $obj ( @$tree )
     {
	 if ( $obj->{'is_inner'} )
	  {
	      $args->{'tree'} = $obj->{'body'};

	      my $type = $obj->{'names'}->[0]->{type};  

	      $type =~ s/(^"|"$)//g ;

	      if ( $ObjectType =~ m/^$type$/ )
	       {
		   $args->{HEAD_FOUND} = TXK::Util::TRUE;
	       }

	      $nodeText =  $self->_getNodeTextForAttr($args);

	      return $nodeText if ( $args->{HEAD_FOUND} ) ;

	  }
	 else
	  {
	      my $attr;

	      foreach $attr ( @{$obj->{'names'}} )
	       {
		   if ( exists $attr->{$attrName}                    and
			$attr->{$attrName} =~ m/^["]$attrValue["]$/  and
			exists $args->{HEAD_FOUND} )
		    {
			$nodeText = $obj->{'body'}->[0];

			return $nodeText;
		    }
	       }
	  }
     }

   return $nodeText;
}

##################################
## sub _validateActions( $ )
##################################

sub _validateActions( $ )
{
    my $self = $ARG[0];

    TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

    return unless ( $self->{$DOC_LOADED} eq TXK::Util::TRUE );

    my $action;

    my $index = 0;

    my $validActionHash = {};

    foreach $action ( @{$self->{$ACTION_LIST}} )
     {
	 print ("TXK:Validate:_validateActions validating $action->{$ACTION_NAME} \n") if ($self->{$DEBUG} );

	 #-----------------------------------------------------
	 # The following things need to be true before the action is valid
	 #
	 # 1. Valid Platform
	 # 2. Valid config option
	 # 3. Valid node type
	 # 4. Valid OS Type     (e.g. for Linux this can be Red Hat or SuSE)
	 # 5. Valid OS Version  (e.g for Red Hat this may be AS 2.1 or 3.0)
	 # 6. Valid OS Release  (result of uname -r command)
	 #-----------------------------------------------------


	 my $validPlatform     = TXK::Util::FALSE;

	 my $validConfigOption = TXK::Util::FALSE;

	 my $validTier         = TXK::Util::FALSE;

	 my $validOSType       = TXK::Util::FALSE;

	 my $validOSVersion    = TXK::Util::FALSE;

	 my $validOSRelease    = TXK::Util::FALSE;

	 $validPlatform = $self->_checkValidPlatform( [ split(',',$action->{$ACTION_PLATFORM} ) ] );

	 $validConfigOption = $self->_checkConfigOption($action);

	 $validTier = $self->_checkValidNodeType( [ split (',',$action->{$ACTION_NODETYPE}) ] );

	 $validOSType = $self->_checkValidOSType($action);

	 $validOSRelease = $self->_checkValidOSRelease($action);

	 $validOSVersion = $self->_checkValidOSVersion($action);

	 print ("TXK:Validate:_validateActions  for $action->{$ACTION_NAME} validPlatform=$validPlatform -\n".
                " validConfigOption=$validConfigOption validTier=$validTier \n" .
		" validOSType=$validOSType validOSVersion=$validOSVersion validOSRelease=$validOSRelease\n") if ($self->{$DEBUG} );


	 $action->{$ACTION_VALID} = TXK::Util::TRUE 
	 
	   if ( $validPlatform and $validConfigOption and $validTier and $validOSType and $validOSVersion and $validOSRelease ) ;

	 if ( $action->{$ACTION_VALID} eq TXK::Util::TRUE )
	  {
	      #-------------------------------------------------------------
	      # here we assume that there can be only two collisons.
	      # using this we can override one action over another
	      # e.g. we can have a generic action for platform="all"
	      # and then we can have platform specific actions
	      # for the same actionName which will override the action="all"
	      # for the same actionName.
	      #-------------------------------------------------------------

	      if ( exists ($validActionHash->{$action->{$ACTION_NAME}}) )
	       {
		   $validActionHash->{$action->{$ACTION_NAME}}->{'index2'} = $index ;

		   $validActionHash->{$action->{$ACTION_NAME}}->{'count'} += 1 ;
	       }
	      else
	       {
		   $validActionHash->{$action->{$ACTION_NAME}}->{'count'} = 1 ;

		   $validActionHash->{$action->{$ACTION_NAME}}->{'index1'} = $index ;
	       }
	  }

	 $index++;
     }

    $self->_removeDuplicatActions($validActionHash);
}

sub _checkValidOSType ( $ )
{
    my $self = $ARG[0];

    my $action = $ARG[1];

    return TXK::Util::TRUE if ( ( $action->{$ACTION_OS_TYPE} eq "" ) or 
				( $action->{$ACTION_OS_TYPE} eq TXK::OSD->getOSDistribution() ) );

    return TXK::Util::FALSE;
}

sub _checkValidOSVersion ( $ )
{
    my $self = $ARG[0];

    my $action = $ARG[1];

    return TXK::Util::TRUE if ( $action->{$ACTION_OS_VERSION} eq "" ) ;

    if ( $action->{$ACTION_OS_VERSION} =~ s/\+$// ) # there is a plus sign
     {
	 my @actionOSVersion = split ( /$action->{$ACTION_VALUE_DELIM}/ , $action->{$ACTION_OS_VERSION} );
	 
	 my @actualOSVersion = split ( /$action->{$ACTION_VALUE_DELIM}/ , TXK::OSD->getOSVersion() );

	 my $retVal = TXK::Util::FALSE;

	 my $index = 0;

	 my $actionValue;

	 foreach $actionValue ( @actionOSVersion )
	  {
	      return $retVal if ( $index > $#actualOSVersion );

	      return TXK::Util::FALSE if ( ( $actionValue =~ /\d/ ) and ( $actualOSVersion[$index] !~ /\d/ ) );

	      return TXK::Util::FALSE if ( ( $actionValue !~ /\d/ ) and ( $actualOSVersion[$index] =~ /\d/ ) );

	      if ( $actionValue =~ /\d/ )
	       {
		   return TXK::Util::TRUE if ( $actualOSVersion[$index] > $actionValue  );

		   $retVal = ( $actualOSVersion[$index] == $actionValue  ) ? 
		                                           TXK::Util::TRUE : 
							   TXK::Util::FALSE ;
	       }
	      else
	       {
		  return TXK::Util::TRUE if ( ord(uc($actualOSVersion[$index])) > ord(uc($actionValue))  );

		  $retVal = ( ord(uc($actualOSVersion[$index])) == ord(uc($actionValue))  ) ?
		                                                            TXK::Util::TRUE :
									    TXK::Util::FALSE ;
	       }

	      return $retVal unless ($retVal) ;

	      $index++;
	  }
	 return $retVal;
     }
    else
     {
	 return TXK::Util::TRUE if ( $action->{$ACTION_OS_VERSION} eq TXK::OSD->getOSVersion() ) ;
     }

    return TXK::Util::FALSE;
}

sub _checkValidOSRelease ( $ )
{
    my $self = $ARG[0];

    my $action = $ARG[1];

    return TXK::Util::TRUE if ( ( $action->{$ACTION_OS_RELEASE} eq ""  ) or 
				( $action->{$ACTION_OS_RELEASE} eq TXK::OSD->getOSRelease() ) );

    return TXK::Util::FALSE;
}

sub _checkValidPlatform ( $ )
{
    my $self = $ARG[0];

    my $platformList = $ARG[1];

    print ("TXK:Validate:_checkValidPlatform: printing platform list to validate\n") if ( $self->{$DEBUG} );

    TXK::Util->printDataStr($platformList) if ( $self->{$DEBUG} );

    #--------------------------------------------------------------
    # if there is no platform specified then it is TRUE for all
    #--------------------------------------------------------------

    return TXK::Util::TRUE unless ( scalar ( @$platformList ) > 0 );

    my $acfgName = TXK::OSD->getAutoConfigName() ;

    my $acfgGName = TXK::OSD->getAutoConfigGenericName(); 

    my $platform;

    my $validPlatform = TXK::Util::FALSE ;

    foreach $platform ( @$platformList )
     {
	 if (    ( $platform eq "" ) 
	      or ( $platform =~ m/^ALL$/i )
	      or ( $platform =~ m/^$acfgName$/i )
	      or ( $platform =~ m/^$acfgGName$/i ) )
	  {
	      $validPlatform = TXK::Util::TRUE ;	      
	  }
     }

    return $validPlatform;
}

sub _getDBConfigOption
{
    my $self = $ARG[0];

    my $dbVersion ;

    #------------------------------------
    # if the db is down then
    # no need to crash the application
    #------------------------------------

    eval { $dbVersion = $self->_getTechStackDBObj()->txkGetDBVersion(); };

    $dbVersion =~ s/\.//g;

    $dbVersion = "db".$dbVersion;

    return $dbVersion;
}

#--------------------------------------------
# sub _checkValidNodeType( $ )
# Takes a list of Nodes to be checked
#--------------------------------------------

sub _checkValidNodeType
{
    my $self = $ARG[0];

    my $actionNodeList = $ARG[1];

    my $validNodeList = [];

    my $txkObj = $self->_getTechStackObj();
    
    my ($nodeType,$validNode,$actionNode,$isThisNode);

    foreach $nodeType ( keys %{$self->{$NODETYPES}} )
     {
	 push (@$validNodeList,$self->{$NODETYPES}->{$nodeType}->{$NODE_TYPE_TAG});
     }

    #------------------------------------------------------------------------------------
    # if it is a db context then the validNode type should be DB. this is because
    # in the db context file all the variables tier type variables are true.
    # For example: <TIER_NODE oa_var="s_isConc">YES</TIER_NODE>
    #------------------------------------------------------------------------------------

    if ( $self->_getTechStackObj()->txkIsDBContext( { $APPS_CONTEXT_FILE => $self->{$APPS_CONTEXT_FILE} } ) )
     {
	 $validNodeList = [ "DB" ];
     }
    
    if ( $self->{$DEBUG} )
     {
	 print ( "TXK::Validate::_checkValidNodeType: printing validNodeList \n");

	 TXK::Util->printDataStr( $validNodeList ) ;

	 print ( "TXK::Validate::_checkValidNodeType: printing actionNodeList \n") ;

	 TXK::Util->printDataStr( $actionNodeList );
     }

    foreach $validNode ( @$validNodeList )
     {
	 $validNode =~ s/(^")|("$)//g;

	 foreach $actionNode ( @$actionNodeList )
	  {
	      $actionNode =~ s/(^")|("$)//g;

	      if ( $validNode =~ m/^$actionNode$/i )
	       {
		   my $upperValidNode = uc($validNode);

		   if ( exists  $TECHSTACK_NODETYPE_METHODS->{$upperValidNode} and 
		        defined $TECHSTACK_NODETYPE_METHODS->{$upperValidNode}      )
		    {
			$isThisNode = $TECHSTACK_NODETYPE_METHODS->{$upperValidNode}
		    }
		   else
		    {
			return TXK::Util::FALSE;
		    }

		   print (" TXK::Validate::_checkValidNodeType calling method $isThisNode on techstack object\n") if ( $self->{$DEBUG} );

		   my $returnValue = TXK::Util::FALSE;

		   eval {
		         $returnValue = $txkObj->$isThisNode( { $APPS_CONTEXT_FILE => $self->{$APPS_CONTEXT_FILE} }  ) ;
		   };

		   if ($@)
		    {
			TXK::Error::abort("Method execution error: $isThisNode : $@ ");
		    }
		   else
		    {
			print (" TXK::Validate::_checkValidNodeType returning value $returnValue \n") if ( $self->{$DEBUG} );

                        # return only if it is true otherwise check other nodes

			return $returnValue if ( $returnValue ); 
		    }
	       }
	  }
     }

    return TXK::Util::FALSE;
}

sub _checkConfigOption( $ )
{
    my $self = $ARG[0];

    my $action = $ARG[1];

    my $validConfigList = $self->{$CONFIG_OPTIONS};

    return TXK::Util::TRUE 
	if ( $action->{$ACTION_CONFIG_OPTION} eq "" );

    my @configOptionList = split(',',$action->{$ACTION_CONFIG_OPTION});

    my ($actionConfig, $validConfigOption);

    foreach $actionConfig (@configOptionList)
     {
	 print ("TXK:Validate:_checkConfigOption checking $actionConfig for $action->{$ACTION_NAME} \n") if ( $self->{$DEBUG} );

	 foreach $validConfigOption (@$validConfigList)
	  {
	      print ("TXK:Validate:_checkConfigOption config option in action: $actionConfig validConfigList: $validConfigOption -\n".
		     "for $action->{$ACTION_NAME} \n") if ( $self->{$DEBUG} );
	      
	      #----------------------------------------------------------------
	      # if the action config option starts with 'db' then we allow
	      # the distiction made between 'db9203' and 'db9204' 
	      # 
	      #---------------------------------------------------------------

	      if ( $actionConfig =~ m/^db/ )
	       {
		   return TXK::Util::TRUE 
		       if ( $validConfigOption =~ m/^$actionConfig/i );
	       }
	      else
	       {
		   return TXK::Util::TRUE 
		       if ( $actionConfig =~ m/^$validConfigOption$/i );
	       }

	  }
     }

    return TXK::Util::FALSE;
}

sub _removeDuplicatActions
{
    #-----------------------------------------------------------------------
    # how to determine which actions are duplicate
    # 1. foreach action check if there is another action with the same name
    # 2. foreach of five conditions to be valid from (method _validateAction)
    #     1. Valid Platform
    #     2. Valid config option
    #     3. Valid OS Type     (e.g. for Linux this can be Red Hat or SuSE)
    #     4. Valid OS Version  (e.g for Red Hat this may be AS 2.1 or 3.0)
    #     5. Valid OS Release  (result of uname -r command)
    #   if there is another valid action with the same name but with different
    #   configuration, then the one which is specific (not equals "all" or "UNIX") should be true.
    #   Note that we do not override node types here.
    #-----------------------------------------------------------------------

    my $self = $ARG[0];

    my $validActionHash = $ARG[1];

    my $actionName;

    my $acfgGName = TXK::OSD->getAutoConfigGenericName(); 

    foreach $actionName ( keys %{$validActionHash} )
     {
	 # take care of those actions which are valid more than once.

	 next unless ( $validActionHash->{$actionName}->{'count'} > 1 );

	 my $action1 = $self->{$ACTION_LIST}->[$validActionHash->{$actionName}->{'index1'}] ;

	 my $action2 = $self->{$ACTION_LIST}->[$validActionHash->{$actionName}->{'index2'}] ;

	 $action1->{$ACTION_VALID} = TXK::Util::FALSE

	     if (
		  ( ( ( $action1->{$ACTION_PLATFORM} eq "" ) or ( $action1->{$ACTION_PLATFORM} =~ m/^all$/i ) or
		      ( $action1->{$ACTION_PLATFORM} =~ m/^$acfgGName$/ )                                   ) and 
		  ( ( $action2->{$ACTION_PLATFORM} ne "" ) or ( $action2->{$ACTION_PLATFORM} !~ m/^all$/i ) or
                    ( $action2->{$ACTION_PLATFORM} !~ m/^$acfgGName$/)  ) )
		  or
		  ( ( ( $action1->{$ACTION_CONFIG_OPTION} eq "" ) or ( $action1->{$ACTION_CONFIG_OPTION} =~ m/^all$/i ) ) and 
		  ( ( $action2->{$ACTION_CONFIG_OPTION} ne "" ) or ( $action2->{$ACTION_CONFIG_OPTION} !~ m/^all$/i ) ) )
		  or
		  ( ( ( $action1->{$ACTION_OS_TYPE} eq "" ) or ( $action1->{$ACTION_OS_TYPE} =~ m/^all$/i ) ) and 
		  ( ( $action2->{$ACTION_OS_TYPE} ne "" ) or ( $action2->{$ACTION_OS_TYPE} !~ m/^all$/i ) ) )
		  or
		  ( ( ( $action1->{$ACTION_OS_VERSION} eq "" ) or ( $action1->{$ACTION_OS_VERSION} =~ m/^all$/i ) ) and 
		  ( ( $action2->{$ACTION_OS_VERSION} ne "" ) or ( $action2->{$ACTION_OS_VERSION} !~ m/^all$/i ) ) )
		  or
		  ( ( ( $action1->{$ACTION_OS_RELEASE} eq "" ) or ( $action1->{$ACTION_OS_RELEASE} =~ m/^all$/i ) ) and 
		  ( ( $action2->{$ACTION_OS_RELEASE} ne "" ) or ( $action2->{$ACTION_OS_RELEASE} !~ m/^all$/i ) ) )
		 );

	 # if action1 is still valid then action2 should be invalid

	 $action2->{$ACTION_VALID} = TXK::Util::FALSE if ( $action1->{$ACTION_VALID} );

	 if ( $self->{$DEBUG} )
	  {
	     print ("TXK::Techstack::_removeDuplicateActions: $action1->{$ACTION_NAME} validity is $action1->{$ACTION_VALID}\n".
		    "and $action2->{$ACTION_NAME} validity is $action2->{$ACTION_VALID}\n");
                     
	  }
     }
}

sub _createTechStackObj
{
    my $self = $ARG[0];

    $self->{$OBJECT_CONTAINER_HASH}->{$TECHSTACK} = $TECHSTACK->new ( { $APPS_CONTEXT_FILE => $self->{$APPS_CONTEXT_FILE},
                                                                        $APPS_USER     => $self->{$APPS_USER},
                                                                        $APPS_PASSWORD => $self->{$APPS_PASSWORD}
                                                                           } );
}

sub _createTechStackDBObj
{
    my $self = $ARG[0];

    $self->{$OBJECT_CONTAINER_HASH}->{$TECHSTACKDB} = $TECHSTACKDB->new ( 
								     { $APPS_CONTEXT_FILE => $self->{$APPS_CONTEXT_FILE},
                                                                       $APPS_USER     => $self->{$APPS_USER},
                                                                       $APPS_PASSWORD => $self->{$APPS_PASSWORD}
                                                                      } );
}


sub _getTechStackObj
{
    my $self = $ARG[0];
    
    $self->_createTechStackObj() if ( ! defined $self->{$OBJECT_CONTAINER_HASH}->{$TECHSTACK} );

    return $self->{$OBJECT_CONTAINER_HASH}->{$TECHSTACK};
}

sub _getTechStackDBObj
{
    my $self = $ARG[0];

    $self->_createTechStackDBObj() if ( ! defined $self->{$OBJECT_CONTAINER_HASH}->{$TECHSTACKDB} );

    return $self->{$OBJECT_CONTAINER_HASH}->{$TECHSTACKDB};
}


sub _createActionList 
{
    my $self = $ARG[0];

    my $xmlObj = TXK::XML->new();


    #-----------------------------------------------------------------------------
    # create the tech stack object first since we have the application context file.
    #-----------------------------------------------------------------------------

    $self->_createTechStackObj();

    $self->_createTechStackDBObj();    

    # set the context type

    $self->{$CONTEXT_TYPE} = APPS_CONTEXT;

    $self->{$CONTEXT_TYPE} = DB_CONTEXT 

    if ( $self->_getTechStackObj()->txkIsDBContext({ $APPS_CONTEXT_FILE => $self->{$APPS_CONTEXT_FILE} } ) ) ;

    #-------------------------------
    # Load the definition
    #-------------------------------

    $xmlObj->loadDocument( { file => $self->{$XML_FILE} } );

    my $xmlTree = $xmlObj->getDocument();

    my $node;

    foreach $node ( @{$xmlTree} )
     {
	if ( $node->{head} =~ /^\?xml$/ )
	 {
	     $self->_readXMLHeader( $node );
	 }
	else
	 {
	     my $xml = TXK::XML->new();

	     $xml->addNode( { 'node' => {'type' => 'xmlnode',
					 'value'=> $node }
			  } );

	     $self->{$REPORT_TITLE} =  $xml->getAttrValue( { 'node' => $xml->getRootNodeName(),
				                             'attrname'=> 'Title' } );

	     $self->{$REP_DEFN_TYPE} =  $xml->getAttrValue( { 'node' => $xml->getRootNodeName(),
				                             'attrname'=> 'Type' } );

	     my $element;

             foreach $element ( @{$node->{body}} )
	      {
		  if ( $element->{head} =~ m/^ReportHeader$/i )
		   {
		       $self->_readReportHeader($element);
		   }
		  elsif ( $element->{head} =~ m/^NodeTypes$/i )
		   {
		       $self->_readNodeTypes($element);
		   }
		  elsif ( $element->{head} =~ m/^ReportVariables$/i )
		   {
		       $self->_readReportVariables($element);
		   }
		  elsif ( $element->{head} =~ m/^action$/i )
		   {
		       $self->_addActionToTheList($element);
		   }
		  elsif ( $element->{head} =~ m/^Summary$/i )
		   {
		       $self->_readSummary($element);
		   }
	      }
	 }
     }
	 
     #-------------------------
     # Load the Object mapping
     #-------------------------

    $xmlObj->loadDocument( { file => $self->{$OBJECT_MAPPING} } );

    $xmlTree = $xmlObj->getDocument();

    my $action;

    foreach $action ( @{$self->{$ACTION_LIST}} )
     {

	 $action->{$ACTION_EXEC_METHOD} =  $action->{$ACTION_METHOD} ;


	 my $methodExec = $self->_getNodeTextForAttr ( { tree => $xmlTree,
							name => "name",
							value=>$action->{$ACTION_METHOD},
						        type =>$action->{$ACTION_TYPE}} );

	 $action->{$ACTION_EXEC_METHOD} = $methodExec
	                      if ( $methodExec ne "" );

	 my $fixMethod = $self->_getNodeTextForAttr ( { tree => $xmlTree,
							name => "name",
							value=>$action->{$ACTION_FIX_METHOD},
						        type =>$action->{$ACTION_TYPE}} );

	 $action->{$ACTION_FIX_METHOD} = $fixMethod
	                      if ( $fixMethod ne "" );

	 if ( defined $action->{$ACTION_CONDITION} )
	  {

	      my $conditionExec = $self->_getNodeTextForAttr ( { tree => $xmlTree,
								 name => "name",
								 value=>$action->{$ACTION_CONDITION},
								 type =>$action->{$ACTION_TYPE}} );
	      $action->{$ACTION_CONDITION} = $conditionExec
		                if ( $conditionExec ne "" );
	  }
     }

   #----------------------------------------------------------
   # get s_techstack value from the apps_context file
   #----------------------------------------------------------

    my $configList = [ "all" ];

    unless ( $self->_getTechStackObj()->txkIsDBContext({ $APPS_CONTEXT_FILE => $self->{$APPS_CONTEXT_FILE} } ) )
     {
	 $xmlObj->loadDocument( { file => $self->{$APPS_CONTEXT_FILE} } );

	 my $techstack = $xmlObj->getOAVar ( 's_techstack' );

	 print ("TXK:Validate:_createActionList: $techstack \n") if ( $self->{$DEBUG} );

	 push ( @$configList, $techstack );
     }

    #8627055, code below would try to get a db connection
    #what it retrieves is not needed anyhow.
 
    #push ( @$configList, $self->_getDBConfigOption() );

    #print (" @$configList \n") if ( $self->{$DEBUG} );

    $self->{$CONFIG_OPTIONS} = $configList ;

    return TXK::Error::SUCCESS ;
}

sub _readXMLHeader ( $ )
{
    my $self = $ARG[0];

    $self->{$XML_HEADER}->addNode ( { 'node' => { 'type' => 'xmlnode',
						 'value'=> $ARG[1] }
				  } );
}


sub _addActionToTheList( $ )
{
    my $self = $ARG[0];

    my $attribute;

    my $action =  { };

    TXK::Util->copyHash( $self->{$ACTION_DEFN}, $action );

    $action->{$ACTION_NAME} = $ARG[1]->{'body'}->[0];

    foreach $attribute ( @{$ARG[1]->{names}} )
     {
	 my ( $actionKey,$attrKey );

	 foreach $attrKey ( keys %$attribute )
	  {
	      foreach $actionKey ( keys %$action )
	       {
		  if ( $actionKey =~ m/^$attrKey$/i )
		   {
		       ( $action->{$actionKey} =   $attribute->{$attrKey} ) =~ s/(^"|"$)//g ;
		   }
	       }
	  }
     }
    
     push (@{$self->{$ACTION_LIST}}, $action);
}

sub _readSummary( $ )
{
    my $self = $ARG[0];
    
    my $allSummaryXML = TXK::XML->new();

    $allSummaryXML->addNode({'node' => {'type'  => 'xmlnode',
				        'value' => $ARG[1] }
   		         });

    my $summaryXML = TXK::XML->new();

    $summaryXML->addNode({ 'node' => {'type' => 'xmlnode',
				       'value'=> $allSummaryXML->getNode({ node=>"$self->{$CONTEXT_TYPE}"}) }
			     });
    
    #-----------------------------------------------
    # get summary message for the context type
    #-----------------------------------------------

    $self->{$SUMMARY_ALLPASS} = $summaryXML->getNodeValue( { node => $SUMMARY_ALLPASS } ) ;

    $self->{$SUMMARY_FAIL} = $summaryXML->getNodeValue( { node => $SUMMARY_FAIL } ) ;

    $self->{$SUMMARY_WARN} = $summaryXML->getNodeValue( { node => $SUMMARY_WARN } ) ;

    $self->{$SUMMARY_STATUS_TEXT} = $summaryXML->getAttrValue( { node     => 'Summary',
                                                                attrname => $SUMMARY_STATUS_TEXT } );

    $self->{$SUMMARY_TITLE} = $summaryXML->getAttrValue( { node     => 'Summary',
                                                          attrname => $SUMMARY_TITLE } );
}

sub _readReportHeader( $ )
{
    my $self = $ARG[0];

    $self->{$REP_HEADER} = $ARG[1]->{body}->[0];
}

sub _readNodeTypes( $ )
{
    my $self = $ARG[0];

    my $nodeList = $ARG[1]->{body};

    my $nodeElement;

    my $nodeTypes = $self->{$NODETYPES};

    foreach $nodeElement ( @$nodeList )
    {
	my  $innerKey;

	foreach $innerKey ($nodeElement->{names})
	 {
	     my $nodeList = [ ];

	     my ($node,$type);

	     foreach $node ( @$innerKey )
	      {
		  my $key;

		  foreach $key ( keys %$node )
		   {
		       if ( $key =~ m/^$NODE_TYPE_TAG$/i )
		        {
			    $type = $node->{$key};

			    $type =~ s/(^")|("$)//g;

			    $nodeTypes->{$type} = { };

			    $nodeTypes->{$type}->{$NODE_TYPE_TAG} = $type;			
		        }
		  }

		  foreach $key ( keys %$node)
		   {
		       unless ( $key =~ m/^$NODE_TYPE_TAG$/i )
		       {
			   $node->{$key} =~ s/(^"|"$)//g;

			   $nodeTypes->{$type}->{$key} = $node->{$key};
		       }
		   }
	      }
	 }
    }

    print ("TXK::Validate::_readNodeTypes printing DataStr for nodeTypes \n") if ( $self->{$DEBUG} ) ;

    TXK::Util->printDataStr( $nodeTypes ) if ( $self->{$DEBUG} );
}

sub _readReportVariables( $ )
{
    my $self = $ARG[0];

    my $xmlHeader = $self->{$XML_HEADER}->getDocument();

    print ("TXK::Validate::_readReportVariables printing Report Variables \n") if ( $self->{$DEBUG} );

    TXK::Util->printDataStr($ARG[1]) if ( $self->{$DEBUG} );

    #push ( @$xmlHeader, $ARG[1]);

    $self->{$REPORT_CTX}->addNode( { 'node'  =>  { 'type' => 'xmlnode',
						   'value'=> $xmlHeader->[0] }
				 } );

    $self->{$REPORT_CTX}->addNode( { 'node'  =>  { 'type' => 'xmlnode',
						   'value'=> $ARG[1] }
				 } );
}

#=================================================================
# End of script.
#=================================================================

1;
