
# +===========================================================================+
# |   Copyright (c) 2003 Oracle Corporation, Redwood Shores, California, USA
# |                         All Rights Reserved
# |                        Applications Division
# +===========================================================================+
# |
# | FILENAME
# |   XML.pm
# |
# | DESCRIPTION
# |      TXK XML package
# |
# | USAGE
# |       See XML.html
# |
# | PLATFORM
# |
# | NOTES
# |
# +===========================================================================+

# $Header: XML.pm 03-aug-2005.08:54:55 ndutko Exp $

package TXK::XML;

@ISA = qw( TXK::Common );

######################################
# Standard Modules
######################################

use strict;
use English;
use Carp;

require 5.005;

######################################
# Package Specific Modules
######################################

use TXK::Error();

######################################
# Public Constants
######################################

use constant HEAD        => 'head';
use constant BODY        => 'body';
use constant IS_INNER    => 'is_inner';
use constant IS_EMTPYTAG => 'is_emptytag';
use constant NAMES       => 'names';
use constant DELETED     => 'deleted';
use constant XMLNODE     => 'xmlnode';

######################################
# Package Variables 
######################################

my $PACKAGE_ID = "TXK::XML";

my $debug_parser = 0;

######################################
# Object Keys
######################################

my $FILE_NAME      = "file";
my $XML_TREE	   = "xmlTree";
my $DEBUG_PARSER   = "debug_parser";
my $OA_VAR         = "oa_var";
my $OA_VAR_APPLY   = "OAVars";

######################################
#
# Object Structure
# ----------------
#
#  Hash Array
#
#      'xml'  => Array of XML entries
#
#                  Each element contains :
#
#                  Hash Table:
#
#                      'head' => Scalar: Name of XML entry.
#                      'body' => Array: If body is just text,
#                                           [0] contains text string
#                                       else
#                                           Recursive Array of XML entries.
#                      'is_inner'
#			     => Boolean: True if an inner recursive entry.
#		       'is_emptytag'
#			     => Boolean: True if no XML body.
#
#                       names=> An array of hash entries for Name/Value Pairs.
#
#			       Each array element has a hash table with one
#			       key/value pair. An array is used to ensure
#			       correct ordering when writing out the
#			       XML structure.
#
#		        'deleted'
#			     => Deleted. Only used when storing doc.
#
#      'oa_var' => Shortcut hash table to get/set oa_vars
#		
#		  Key => oa_var name.
#	          Value => Reference to body text.
#
#      'debug_parser'
#	        => Boolean: Output debug information if true.
#
######################################

######################################
# Package Methods 
#
# Public
#
#	new 	- build empty object
#
#   Unless otherwise stated all other public methods pass one argument,
#   either a scalar or hash table/
#
#   to pass arguments.
#
#	load_doc
#		   file =>  	Name of XML file to load.
#	store_doc
#		   file =>	Name of output file.
#       get_oavar
#		   SCALAR 	OA_VAR to get. 
#	set_oavar
#		   Hash table of of oa_var name/value pairs.
#	print_oavar
#		   No args
#	debug_parser
#		   SCALAR 	Boolean to enable/disbale debug.
#
######################################

sub new;
sub DESTROY;
sub loadDocument;
sub storeDocument;
sub printDocument;
sub getDocument;
sub getXMLName;
sub setXMLName;
sub getOAVar;
sub setOAVar;
sub printOAVar;
sub getOAVarList;
sub applyOAVarDefaults;
sub cleanupOAVarOSD;
sub debugParser;

sub getAttrValue;
sub getAttrList;
sub getNodeValue;
sub setNodeValue;
sub getNode;
sub addNode;

# Old versions kept for backward compatibility.

sub load_doc;
sub store_doc;
sub print_oavar;
sub print_doc;
sub debug_parser;

######################################
# Package Methods
# 
# Private
#       All private methods are marked with a leading underscore.
#
######################################

sub _parse_comment;
sub _parse_names;
sub _parse_head;
sub _parse_body;
sub _parse_doc;
sub _do_store_doc;

sub _getAttrList;
sub _getNodeValue;
sub _setNodeValue;
sub _getNode;
sub _addNode;

######################################
# Constructor
######################################

sub new {
  my $type = $ARG[0];

  my $self = TXK::Common->new();

  bless $self, $PACKAGE_ID ;

  my $key;

  my %INIT_OBJ = (
                   PACKAGE_IDENT   => $PACKAGE_ID,
		   $FILE_NAME      => "",
		   $XML_TREE       => [ ],
                   $DEBUG_PARSER   => "0",
                   $OA_VAR         => { },
                 );

  foreach $key (keys %INIT_OBJ)
   {
     $self->{$key} = $INIT_OBJ{$key};
   }

  return $self;
}

######################################
# Destructor
######################################

sub DESTROY
{
}

######################################
# Load Document
######################################

sub loadString
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  my $xmldoc = $args;

  $xmldoc =~ s/\r//g;

  my @xmldoc = split // , $xmldoc ;

  $self->{$XML_TREE} = [ ];

  $debug_parser = ( $self->{$DEBUG_PARSER} ? 1 : 0 );

  _parse_doc($self->{$XML_TREE},\@xmldoc,0);

#    Load oa_var shortcut table

  $self->{$OA_VAR} = { };

  _load_oavar($self->{$XML_TREE},$self->{$OA_VAR});
}

sub loadDocument
{
  load_doc(@ARG);
}

sub load_doc 
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  TXK::Util->isValidArgs({args=>$args,reqd=>["$FILE_NAME"]});

  $self->{$FILE_NAME} = $args->{$FILE_NAME};

  open FXML , $self->{$FILE_NAME}
                              or 
            TXK::Error->abort( { error => "Cannot open XML file for load",
                                 file  => $self->{$FILE_NAME},
                                 errorno=>$ERRNO } );

  my @xmldoc = <FXML>;

  close FXML or
            TXK::Error->abort( { error => "Cannot close XML file for load",
                                 file  => $self->{$FILE_NAME},
                                 errorno=>$ERRNO } );

  my $xmldoc = join '' , @xmldoc ;

  $xmldoc =~ s/\r//g;

  @xmldoc = split // , $xmldoc ;
 
  $self->{$XML_TREE} = [ ];

  $debug_parser = ( $self->{$DEBUG_PARSER} ? 1 : 0 );
   
  _parse_doc($self->{$XML_TREE},\@xmldoc,0);

#    Load oa_var shortcut table

  $self->{$OA_VAR} = { };

  _load_oavar($self->{$XML_TREE},$self->{$OA_VAR});

}

######################################
# Store Document
######################################

sub storeDocument
{
  store_doc(@ARG);
}

sub store_doc 
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  TXK::Util->isValidArgs({args=>$args,reqd=>["$FILE_NAME"]});

  open FXML , ( ">" . $args->{$FILE_NAME} )
       or
            TXK::Error->abort( { error => "Cannot open XML file for store",
                                 file  => $args->{$FILE_NAME},
                                 errorno=>$ERRNO } );

  my $xmltree = $self->{$XML_TREE};
  my $fhandle = \*FXML;

  _do_store_doc($xmltree,$fhandle,0);

  close $fhandle or
           TXK::Error->abort( { error => "Cannot close XML file for store",
                                file  => $args->{$FILE_NAME},
                                errorno=>$ERRNO } );

}

######################################
# Print Doc
######################################

sub printDocument
{
  print_doc(@ARG);
}

sub print_doc
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  my $xmltree = $self->{$XML_TREE};

  print "\n <<PRINT DOC>> xmlref = $xmltree \n\n";

  _do_print_doc($xmltree,0);
}

######################################
# Set Debug Parser
######################################

sub debugParser
{
  debug_parser(@ARG);
}

sub debug_parser
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  $self->{$DEBUG_PARSER} = 1;

}

######################################
# getOAVar 
######################################

sub getOAVar 
{
  my $self  = $ARG[0];
  my $var_name = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return ${$self->{$OA_VAR}->{$var_name}}
                    if exists( $self->{$OA_VAR}->{$var_name} ) ;
  return undef;
}

######################################
# setOAVar
######################################

sub setOAVar
{
  my $self = $ARG[0];
  my $args = $ARG[1];

  my $name ;

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  foreach $name (keys %$args)
   {
     ${$self->{$OA_VAR}->{$name}} = $args->{$name} 
                    if exists( $self->{$OA_VAR}->{$name} ) ;
   }
}

######################################
# Get a list of OAVars
######################################

sub getOAVarList
{
  my $self  = $ARG[0];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return (keys %{$self->{$OA_VAR}});
}

######################################
# Print oa_vars
######################################

sub printOAVar
{
  print_oavar(@ARG);
}

sub print_oavar
{
  my $self  = $ARG[0];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  my $oa_var = $self->{$OA_VAR};
  my $key;

  foreach $key (keys %$oa_var)
   {
     my $str = $oa_var->{$key} ;

     print "OA_VAR KEY=$key VALUE=", $$str, " \n";
   }
}

######################################
# getDocument
######################################

sub getDocument
{
  my $self  = $ARG[0];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return $self->{$XML_TREE};
}

######################################
# applyOAVarDefaults
######################################

sub applyOAVarDefaults
{
  my $self  = $ARG[0];
  my $args  = $ARG[1];

  my %oaVars ;
  my $useOAVars = TXK::Util::FALSE;

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID,args=>$args});

  if ( defined $args && 
       exists $args->{$OA_VAR_APPLY} && ref($args->{$OA_VAR_APPLY}) eq "ARRAY" )
   {
     my $i;

     $useOAVars = TXK::Util::TRUE;

     foreach $i (@{$args->{$OA_VAR_APPLY}})
      {
        $oaVars{$i} = " ";
      }
   }

  _doApplyOAVarDefaults($self->{$XML_TREE},$useOAVars,\%oaVars);

  return TXK::Error::SUCCESS;
}

######################################
# cleanupOAVarOSD
######################################

sub cleanupOAVarOSD
{
  my $self  = $ARG[0];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  _doCleanupOAVarOSD($self->{$XML_TREE});

  return TXK::Error::SUCCESS;
}

######################################
# getXMLName
######################################

sub getXMLName
{
  my $self  = $ARG[0];
  my $names = $ARG[1];
  my $id    = $ARG[2];

  my $entry;
  my $data;

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  foreach $entry ( @$names )
   {
     if ( exists $entry->{$id} )
      {
        $data = $entry->{$id} ;
        $data =~ s/\'//g;
        $data =~ s/\"//g;

        return $data;
      }
   }

  return undef;
}

######################################
# setXMLName
######################################

sub setXMLName
{
  my $self  = $ARG[0];
  my $names = $ARG[1];
  my $id    = $ARG[2];
  my $value = $ARG[3];

  my $entry;
  my $data;

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  foreach $entry ( @$names )
   {
     $entry->{$id} = '"' . $value . '"' , return TXK::Error::SUCCESS
             if ( exists $entry->{$id} );
   }

  return undef;
}

######################################
# getNodeValue
# returns the value for the node
# that first matches 
######################################

sub getNodeValue
{
  my $self = $ARG[0];

  my $args = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  TXK::Util->isValidArgs({args=>$args,reqd=>["node"]});

  return _getNodeValue($self->{$XML_TREE} , $args->{'node'});
}

#####################################
# getAttrList
#####################################

sub getAttrList
{
  my $self = $ARG[0];

  my $args = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  TXK::Util->isValidArgs({args=>$args,reqd=>["node"]});

  return _getAttrList($self->{$XML_TREE} , $args->{'node'});
}

######################################
# getAttrValue
######################################

sub getAttrValue
{
  my $self = $ARG[0];

  my $args = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  TXK::Util->isValidArgs({args=>$args,reqd=>["node","attrname"]});

  my $attrList = $self->getAttrList( { 'node'=> $args->{'node'} } );

  if ( defined $attrList and 
       ref($attrList) eq 'ARRAY' )
   {
       my ($key,$element);

       foreach $element ( @$attrList )
        {
	    foreach $key ( keys %$element )
	     {
		 if ( $key =~ m/$args->{'attrname'}/i )
		  {
		      $element->{$key} =~ s/(^"|"$)//g ;

		      return $element->{$key};
		  }
	     }
	}
   }

 return undef;
}

######################################
# setNodeValue
# sets the value of the first node that
# matches.
######################################

sub setNodeValue
{
  my $self = $ARG[0];

  my $args = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  TXK::Util->isValidArgs({args=>$args,reqd=>["node","value"]});

  return _setNodeValue($self->{$XML_TREE} , $args->{'node'},$args->{'value'});

}

######################################
# getRootNodeName 
# returns the name of the RootNode
######################################

sub getRootNodeName
{
  my $self = $ARG[0];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  return @{$self->{$XML_TREE}}->[0]->{'head'} 

     if ( scalar ( @{$self->{$XML_TREE}} ) > 0 ) ;
}


######################################
# getNode 
# returns a hash that represents the node
######################################

sub getNode
{
  my $self = $ARG[0];

  my $args = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  TXK::Util->isValidArgs({args=>$args,reqd=>["node"]});

  return _getNode($self->{$XML_TREE} , $args->{'node'} );
}

######################################
# sub addNode
######################################

sub addNode
{
  my $self = $ARG[0];

  my $args = $ARG[1];

  TXK::Util->isValidObj({obj=>$self,package=>$PACKAGE_ID});

  TXK::Util->isValidArgs({args=>$args,reqd=>["node"]});

  $self->{$XML_TREE} = [ ]  unless ( scalar $self->{$XML_TREE} )  ;

  _addNode($self->{$XML_TREE} , $args->{'node'},$args->{'value'},$args->{'attrlist'},$args->{'parent'}); 

  _load_oavar($self->{$XML_TREE},$self->{$OA_VAR});
}


######################################
# End of public methods
######################################

# ==========================================================================

sub _parse_comment
{
  my $xmldoc  = $ARG[0];
  my $names   = $ARG[1];
  my $endtoken= $ARG[2];

  my $c;
  my @comment;

  for ( ;; )
   {
     $c = shift @$xmldoc; 

     do { push @comment, $c ;
          $c = shift @$xmldoc; } until ( $c eq '"' || $c eq '\'' || $c eq "-");

     last if (  (  $c eq "-"          && $endtoken eq "-->" &&
                   $$xmldoc[0] eq "-" && $$xmldoc[1] eq ">" )
              ||
                ( not defined $c )
             );
           
     if ( $c eq '"' || $c eq '\'' )
      {
        my $quotechar = $c;

        do { push @comment, $c ;
             $c = shift @$xmldoc; } while ( $c ne $quotechar );

        push @comment, $c ;
        $c = shift @$xmldoc;

        last if (  (  $c eq "-"          && $endtoken eq "-->" &&
                      $$xmldoc[0] eq "-" && $$xmldoc[1] eq ">" )
                 ||
                   ( not defined $c )
                );
      }
 
#    Push back for the next loop;

     unshift @$xmldoc, $c;
   }

  push @$names , { '__comment__' => (join '' , @comment) };

  splice @$xmldoc , 0 , length($endtoken)-1;

  my ($key,$entry);

  foreach $entry ( @$names )
   {
     foreach $key ( keys %$entry )
      {
        print "Key = $key Value = $entry->{$key} \n" if $debug_parser;
      }
   }

  return 0;		# No body for comments.
}

sub _parse_names
{
  my $xmldoc  = $ARG[0];
  my $names   = $ARG[1];
  my $endtoken= $ARG[2];

  my $c;
  my @varname;
  my @varvalue;

  for ( ;; )
   {
     do { $c = shift @$xmldoc; } while ( $c =~ m/\s/ );

     last if (  (  $c eq "?"          && $endtoken eq "?>" &&
                   $$xmldoc[0] eq ">" )
              ||
                (  $c eq ">"          && $endtoken eq ">" )
              ||
                (  $c eq '/'          && $endtoken eq ">" && 
                   $$xmldoc[0] eq ">" )
              ||
                ( not defined $c )
             );
           
     do { push @varname, $c ;
          $c = shift @$xmldoc; } while ( $c =~ m/\S/ && $c ne "=" );

     $c = shift @$xmldoc while ( $c ne "=" );

     do { $c = shift @$xmldoc; } while ( $c =~ m/\s/ );

     my $quotechar =  ( ( $c eq '"' || $c eq '\'' ) ? $c : " " );
   
     do { push @varvalue, $c ;
          $c = shift @$xmldoc; } 
        while ( ( $c =~ m/\S/ && $c ne ">" && $quotechar eq " " ) 
               ||
                ( $quotechar ne " " && $c ne $quotechar )
              );

     if ( $quotechar ne " " )
      {
        push @varvalue, $c ;
        $c = shift @$xmldoc;
      }

     push @$names , { (join '', @varname) => (join '' , @varvalue) };

     undef @varname;
     undef @varvalue;

#    Push back for the next loop;

     unshift @$xmldoc, $c;
   }

# Get rid of remaining chars.

  splice @$xmldoc , 0 , length($endtoken)-1;

  shift @$xmldoc if (  $c eq '/' ); # /> token

  my ($key,$entry);

  foreach $entry ( @$names )
   {
     foreach $key ( keys %$entry )
      {
        print "Key = $key Value = $entry->{$key} \n" if $debug_parser;
      }
   }

  print "Names endc = $c \n" if $debug_parser;

  return ( ( $c eq '/' || $c eq '-' || $c eq '?' ) 
		? 0 : 1 ) ;  # False = no body.

}

sub _parse_head
{
  my $xmltree = $ARG[0];
  my $xmldoc  = $ARG[1];

  my $c = 0;
  my $endtoken;
  my @head;
  my $rc;
 
  do { $c = shift @$xmldoc; } while ( defined $c && ( $c =~ m/\s/ ) );

  return undef unless defined $c;

  if ( $c =~ m/</ )
   {
     push @$xmltree , { head => " ",
                        body => [ ],
                        is_inner => 0,
                        is_emptytag => 0,
                        names=> [ ],
                      };

     do { $c = shift @$xmldoc; } while ( $c =~ m/\s/ );

     do { push @head, $c ;
          $c = shift @$xmldoc; } while ( $c =~ m/\S/ && $c ne '>' 
                                         &&  
                                       ( ! ($c eq '/' && $$xmldoc[0] eq '>')) );

     my $headstr = join '' , @head;

     my $idx = $#$xmltree;

     $$xmltree[$idx]->{'head'} = $headstr;
     
     if    ( $headstr =~ m/\?xml/ ) { $endtoken = "?>" ; }
     elsif ( $headstr =~ m/!--/ )   { $endtoken = "-->"; }
     else                           { $endtoken = ">"  ; }

     print "Head = " , $$xmltree[$idx]->{'head'} , 
           " Endtoken = $endtoken " , "\n"  if $debug_parser;

#  Push back whitespace

     unshift @$xmldoc, $c;

     if ( $endtoken eq "-->" )
      {
        $rc = _parse_comment($xmldoc,$$xmltree[$idx]->{'names'},$endtoken);
      }
     else
      {
        $rc = _parse_names($xmldoc,$$xmltree[$idx]->{'names'},$endtoken);
      }
   }

 return $rc;

}

sub _parse_body
{
  my $xmltree = $ARG[0];
  my $xmldoc  = $ARG[1];
  my $depth   = $ARG[2];

  my $c = 0;
  my @body;

  print "Parse Body \n" if $debug_parser;

  do { $c = shift @$xmldoc; } while ( defined $c && ( $c =~ m/\s/ ) );

  return undef unless defined $c;

  if ( $c eq '<' && $$xmldoc[0] eq '/' )	  # Empty body.
   {
     do { $c = shift @$xmldoc; } until ( $c eq '>' );

     $$xmltree[$#$xmltree]->{'is_emptytag'} = 1;

     return;
   }

  my $bodyarray = $$xmltree[$#$xmltree]->{'body'};

  if ( $c eq "<" ) 	# Recursive doc.
   {
     $$xmltree[$#$xmltree]->{'is_inner'} = 1;

     print "recursive ... \n" if $debug_parser;

     unshift @$xmldoc , $c; 		# Push back "<"
     _parse_doc($bodyarray,$xmldoc,$depth+1);	# Recurse to next level
   }
  else
   {
      # Any text until </keyword> . No need to check for keyword.

     for ( ;; )
      {

#	Drop through if first char is a quote.

        unless ( $c eq '"' || $c eq '\'' )
         {
           do { push @body, $c ;
                $c = shift @$xmldoc; } until ( $c eq '"' || $c eq '\'' || 
                                               $c eq "<");

           last if (  (  $c eq "<"  && $$xmldoc[0] eq '/' )
                    ||
                      ( not defined $c )
                   );
         }
           
        if ( $c eq '"' || $c eq '\'' )
         {
           my $quotechar = $c;

           do { push @body, $c ;
                $c = shift @$xmldoc; } until ( $c eq $quotechar  ||
                                               $c eq "<" );

           if ( $c eq $quotechar )
            {
              push @body, $c ;
              $c = shift @$xmldoc;
            }

           last if (  (  $c eq "<"  && $$xmldoc[0] eq '/' )
                    ||
                      ( not defined $c )
                   );
         }
 
#    	No need to push back for the next loop;
      }

     $bodyarray->[0] = join '' , @body;

     $bodyarray->[0] =~ s/\s+$//g;	# Remove trailing whitespace.

     return if not defined $c;

     do { $c = shift @$xmldoc; } until ( $c eq '>' );
   }

  print "Body = $bodyarray->[0] \n" if $debug_parser;

}

sub _parse_doc
{
  my $xmltree = $ARG[0];
  my $xmldoc  = $ARG[1];
  my $depth   = $ARG[2];

  my $rc;
  my $c;

  print "Parse Doc at depth = $depth \n" if $debug_parser;

  for ( ;; )
   {
     $rc = _parse_head($xmltree,$xmldoc);

     last if not defined $rc;

     _parse_body($xmltree,$xmldoc,$depth) if ( $rc );

#	After parse routines, $$xmldoc[0] points to next char after
#	closing '>'.

     do { $c = shift @$xmldoc } while ( defined $c && ( $c =~ m/\s/ ) );

     return if not defined $c;

     if ( $c eq "<" && $$xmldoc[0] eq '/' )
      {
        do { $c = shift @$xmldoc; } until ( $c eq '>' );
        last;
      }

     unshift @$xmldoc , $c;		# Push back '<'
   }
}

sub _load_oavar
{
  my $xmltree = $ARG[0];
  my $oa_hash = $ARG[1];

  my ($obj,$name,$key,$var_name,$var_value);

  foreach $obj (@$xmltree)
   {
     foreach $name ( @{$obj->{'names'}} )
      {
        foreach $key ( keys %$name )
         {
           if ( $key eq "oa_var" )
            {
              $var_name = $name->{$key};

              $var_name =~ s/\'//g;
              $var_name =~ s/\"//g;

#	Set emtpy body if not defined.

              $obj->{'body'}->[0] = " " if not defined ($obj->{'body'}->[0]) ;

              $var_value= \($obj->{'body'}->[0]);

              $oa_hash->{$var_name} = $var_value;
            }
         }
      }

     if ( $obj->{'is_inner'} )
      {
        _load_oavar($obj->{'body'},$oa_hash);
      }
   }
}

sub _doApplyOAVarDefaults
{
  my $xmltree = $ARG[0];
  my $useOAVars=$ARG[1];
  my $oaVars  = $ARG[2];

  my ($obj,$name,$name2,$var_name,$deflt,$l_var,$applyDefault);

  foreach $obj (@$xmltree)
   {
     foreach $name ( @{$obj->{'names'}} )
      {
        if ( exists($name->{"oa_var"}) )
         {
           $var_name = $name->{"oa_var"};

           $var_name =~ s/\'//g;
           $var_name =~ s/\"//g;

           foreach $name2 ( @{$obj->{'names'}} )
            {
              if ( exists ($name2->{'default'}) )
               {
                 if ( $obj->{'body'}->[0] eq ("%" . $var_name . "%")  )
                  { 
                    $deflt = $name2->{'default'};

                    $deflt =~ s/\'//g;
                    $deflt =~ s/\"//g;

                    $applyDefault = TXK::Util::TRUE;

                    if ( $useOAVars )
                     {
                       $applyDefault = TXK::Util::FALSE;

                       foreach $l_var (keys %$oaVars)
                        {
                          $applyDefault = TXK::Util::TRUE, last
                                   if ( $deflt =~ m/%${l_var}%/ );
                        }
                     }
 
                    $obj->{'body'}->[0] = $deflt if ( $applyDefault );
                  }
                 delete $name2->{'default'} ;
               }
            }
         }
      }

     if ( $obj->{'is_inner'} )
      {
        _doApplyOAVarDefaults($obj->{'body'},$useOAVars,$oaVars);
      }
   }
}

sub _doCleanupOAVarOSD
{
  my $xmltree = $ARG[0];

  my ($obj,$name,$osd_name);

  my ($found_oavar,$found_osd);

  foreach $obj (@$xmltree)
   {
     $found_oavar=$found_osd=TXK::Util::FALSE;

     foreach $name ( @{$obj->{'names'}} )
      {
        $found_oavar = TXK::Util::TRUE if ( exists($name->{"oa_var"}) );

        if ( exists($name->{"osd"}) )
         {
           $osd_name = $name->{"osd"};

           $osd_name =~ s/\'//g;
           $osd_name =~ s/\"//g;

           $found_osd = TXK::Util::TRUE 
               unless ( TXK::OSD->getAutoConfigName() eq $osd_name ||
                        TXK::OSD->getAutoConfigGenericName() eq uc($osd_name) );
         }
      }

     $obj->{'deleted'} = TXK::Util::TRUE if ( $found_oavar && $found_osd );
     
     if ( $obj->{'is_inner'} )
      {
        _doCleanupOAVarOSD($obj->{'body'});
      }
   }
}

sub _do_store_doc
{
  my $xmltree = $ARG[0];
  my $fhandle = $ARG[1];
  my $depth   = $ARG[2];

  my ($obj,$name,$key);
  my $indent = ( "   " x $depth );

  foreach $obj (@$xmltree)
   {
     next if ($obj->{'deleted'});

     print $fhandle $indent, "<" , $obj->{'head'} ;

     my $last_print_value ;	# Remember last printed value;

     foreach $name ( @{$obj->{'names'}} )
      {
        foreach $key ( keys %$name )
         {
           $last_print_value = $name->{$key};

           if ( $key eq "__comment__" )
            {
              print $fhandle " ", $name->{$key} ;
            }
           else
            {
              print $fhandle " ", ${key}, "=", $name->{$key} ;
            }
         }
      }

     print $fhandle " " unless (  ( not defined $last_print_value )
                                || 
                                  ( substr($last_print_value,-1,1) eq '"' ||
                                    substr($last_print_value,-1,1) eq '\'' 
                                  )
                               );

     if ( substr($obj->{'head'},0,1) eq "?" )
      {
        print $fhandle "?>\n";
        next;
      }
     elsif ( substr($obj->{'head'},0,3) eq "!--" )
        {
          print $fhandle "-->\n";
          next;
        }
     elsif ( $obj->{'is_inner'} )
        {
          print $fhandle ">";
        }
     elsif ( $obj->{'is_emptytag'} || ( defined $obj->{'body'}->[0] ) )
        {
          print $fhandle ">";
        }
     else
        {
          print $fhandle "/>\n";
          next;
        }
          
     if ( $obj->{'is_inner'} )
      {
        print $fhandle "\n";

        _do_store_doc($obj->{'body'},$fhandle,$depth+1);

        print $fhandle $indent, '</' , $obj->{'head'} , ">\n";
      }
     else
      {
        print $fhandle $obj->{'body'}->[0]  if defined $obj->{'body'}->[0];
        print $fhandle '</' , $obj->{'head'} , '>' ;
        print $fhandle "\n";
      }
   }
}

sub _do_print_doc
{
  my $xmltree = $ARG[0];
  my $depth   = $ARG[1];

  my ($obj,$name,$key);
  my $indent = ( "*" x $depth );

  foreach $obj (@$xmltree)
   {
     print $indent, $obj->{'head'} , "\n" ;

     my $x = $obj->{'names'};

     foreach $name ( @{$obj->{'names'}} )
      {
        foreach $key ( keys %$name )
         {
           print $indent, "  " , "$key = $name->{$key}" , "\n";
         }
      }

     if ( $obj->{'is_inner'} )
      {
        _do_print_doc($obj->{'body'},$depth+1);
      }
     else
      {
        print $indent, "    " , $obj->{'body'}->[0] , "\n" 
		if defined $obj->{'body'}->[0];
      }
   }
}

sub _getNode
{
  my $xmltree  = $ARG[0];
  
  my $node     = $ARG[1];
  
  my ($obj,$name,$key,$retVal);

  my $retNode = undef;

  foreach $obj (@$xmltree)
   {
     if ( $obj->{'head'} =~ m/^$node$/i )
      {
	  return $obj;
      }
     elsif ( $obj->{'is_inner'} )
      {
          $retNode = _getNode($obj->{'body'},$node);
          return $retNode if $retNode;              # return if node found.
      } 
   }

 return $retNode ;    
}

sub _addNode
{
  my $xmltree    = $ARG[0];
    
  my $node       = $ARG[1];

  my $value      = $ARG[2];

  my $attrList   = $ARG[3];

  my $parent     = $ARG[4];

  my ($obj,$name,$key,$retVal);

  my $parentNode = { };

  if ( scalar @$xmltree and ( $parent ne "") )
   {
      $parentNode = _getNode($xmltree,$parent);
   }

  if ( ($parentNode) and (scalar (keys %$parentNode)) )  # parent node exists
   {
       $parentNode->{'is_inner'} = 1;

       unless ( ref ($parentNode->{'body'}->[0]) =~ m/HASH/i )
        {
	    $parentNode->{'body'} = [];
        }

       $xmltree = $parentNode->{'body'};
   }

  if ( ref($node) =~ m/^HASH$/i and $node->{'type'} eq XMLNODE )
   {
       push @$xmltree, $node->{'value'};

       return;
   }

   push @$xmltree , { head => $node,
		      body => [ ],
	              is_inner => 0,
	              is_emptytag => 0,
		      names=> [ ],

	            };

  my $newNode = $xmltree->[scalar (@$xmltree) -1 ]; 

  if ( $value ne "" ) 
   { 
      $newNode->{'body'}->[0] = $value;

      $newNode->{'is_emptytag'} = 1;
   }
  else
   {
       $newNode->{'is_emptytag'} = 0;
   }

  if ( scalar (keys %$attrList) )
   {
       my $key;

       foreach $key ( keys %$attrList )
        {
	    # add double quotes if already not present

	    $attrList->{$key} =~ s/(^")|("$)// ;

	    push @{$newNode->{'names'}} , { $key => '"' .$attrList->{$key}. '"' } ;
	}
   }
}

sub _setNodeValue
{
  my $xmltree = $ARG[0];
    
  my $node    = $ARG[1];

  my $value   = $ARG[2];

  my ($obj,$name,$key,$retVal);

  my $retVal = TXK::Util::FALSE;

  foreach $obj (@$xmltree)
   {
     if ( $obj->{'head'} =~ m/^$node$/i )
      {
	  if ( $obj->{'is_inner'} )
	   {
	        return TXK::Util::FALSE; # since this is a root element
	   }
	  else
	   {
	        $obj->{'body'}->[0] = $value;

		return TXK::Util::TRUE;
	   }
      }
     elsif ( $obj->{'is_inner'} )
      {
	  $retVal = _setNodeValue($obj->{'body'},$node,$value);
      
          return $retVal if ( $retVal );    # return if found.
      } 
   }

 return $retVal ; 
}

sub _getNodeValue
{
  my $xmltree = $ARG[0];

  my $node    = $ARG[1];

  my ($obj,$name,$key,$value);

  foreach $obj (@$xmltree)
   {
     if ( $obj->{'head'} =~ m/^$node$/i )
      {
	  if ( $obj->{'is_inner'} )
	   {
	       $value =  ""; # since this is a root element
	   }
	  else
	   {
	       $value = $obj->{'body'}->[0];

	       return $value;
	  }
      }
     elsif ( $obj->{'is_inner'} )
      {
	  last unless ( $value eq "" ); # only when we do not have the value

          $value = _getNodeValue($obj->{'body'},$node);
      }
   }

 return $value; 
}

sub _getAttrList
{
  my $xmltree = $ARG[0];

  my $node    = $ARG[1];

  my ($obj,$name,$key,$atList);

  
  my $atList = [];

  $atList = $ARG[2] if ( defined $ARG[2] );

  foreach $obj (@$xmltree)
   {
     my $x = $obj->{'names'};

     if ( $obj->{'head'} =~ m/^$node$/i )
      {
	   TXK::Util->copyArray( $obj->{'names'},$atList ) ;

	   return $atList;
      }
     elsif ( $obj->{'is_inner'} )
      {
	  $atList =  _getAttrList($obj->{'body'},$node,$atList );

          return $atList if ( scalar(@$atList) > 0 ) ;  # Return if found
      } 
   }

 return $atList; 

}

1;

