/* Copyright (c) 2004, 2005, Oracle. All rights reserved.  */

/*

   NAME
     mod_chronos.c - Chronos module for Apache 2.0

   DESCRIPTION
     Implements an Apache 2.0 module to allow for chronos data to be collected.

   EXPORT FUNCTION(S)
     <external functions defined for use outside package - one-line descriptions>

   INTERNAL FUNCTION(S)
     <other external functions defined - one-line descriptions>

   STATIC FUNCTION(S)
     <static functions defined - one-line descriptions>

   NOTES
     Once the chronos module has been loaded, it can be configured by the
     following commands:
		 ChronosEnabled (YES|NO)
		   This rule allows you to specify to or not to monitor for urls on a host
     ChronosRule (SUBSTRING|PREFIX|SUFFIX|REGEX) string (YES|NO)
       By default, the chronos javascript tag is appended to the end of every 
       document of which the mime-type begins with "text/html".  This rule 
       allows you to give exceptions.  The first type of argument tells how to 
       interpret the string and the third argument tells whether or not to add
       the script for that rule.  The rules are interpreted in order and as 
       soon as a matching rule is found, the decision is returned.
       The settings for this rule are per virtual host.
     The rest of the module is a copy of mod_usertrack.  The key changes are:
       The default name of the cookie is "ORACLE_CMP_CHRONOS_GL" and the note 
         it generates is called "ChronosCookie"
       All of the commands for mod_usertrack have been prepended by Chronos.  
         For example, CookieDomain is now ChronosCookieDomain.

   MODIFIED   (MM/DD/YY)
	 yxie        01/24/05 - Make url comparison case insensitive
   yxie        01/24/05 - Add ChronosEnabled
   yxie        11/17/04 - Add suffix to chronos rule
   eporter     10/19/04 - eporter_filter
   eporter     08/12/04 - Creation

*/

/* Copyright 1999-2004 The Apache Software Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "apr_buckets.h"
#include "apr_strings.h"
#include "apr_env.h"
#include "ap_config.h"
#include "apr.h"
#include "apr_lib.h"
#include "apr_strings.h"

#define APR_WANT_STRFUNC
#include "apr_want.h"

#include "httpd.h"
#include "http_config.h"
#include "http_request.h"
#include "http_core.h"
#include "http_protocol.h"
#include "http_log.h"
#include "http_main.h"

#include "util_filter.h"
#include "util_script.h"

/*---------------------------------------------------------------------------
                     PRIVATE TYPES AND CONSTANTS
  ---------------------------------------------------------------------------*/

module AP_MODULE_DECLARE_DATA chronos_module;
#define FOOTER_STRING      "<SCRIPT SRC=\"/oracle_smp_chronos/oracle_smp_chronos.js\"></SCRIPT>"
#define CHRONOS_DATE       "chronos_date"
#define CHRONOS_TIME       "chronos_time"  
#define CHRONOS_FOOTER     "chronos_footer"   
const char ADD_FOOTER = 'Y', NO_FOOTER = 'N';

/* The possible types of Chronos Rules */
typedef enum {
    RT_PREFIX,
		RT_SUFFIX,
    RT_SUBSTRING,
    RT_REGEX
} rule_type; 

typedef struct chronos_rule chronos_rule;
struct chronos_rule{
  char *expr; /* the expression */
  regex_t *regexp;  /* used to find a match in the uri */
  rule_type type;
  int monitor; /* whether to monitor or not */
  chronos_rule *next;
};

typedef struct {
  int chronos_enabled;
  int expires;

  chronos_rule* rules;
} per_host_state;

typedef enum {
  CT_UNSET,
  CT_NETSCAPE,
  CT_COOKIE,
  CT_COOKIE2
} cookie_type_e;

typedef struct {
  int enabled;
  cookie_type_e style;
  char *cookie_name;
  char *cookie_domain;
  char *regexp_string;  /* used to compile regexp; save for debugging */
  regex_t *regexp;  /* used to find usertrack cookie in cookie header */
} cookie_dir_rec;

#define COOKIE_NAME "ORACLE_SMP_CHRONOS_GL" /* name of glue cookie */
#define COOKIE_NOTE "ChronosCookie"         /* note for logging purposes */

/*---------------------------------------------------------------------------
                     STATIC FUNCTION DECLARATIONS 
  ---------------------------------------------------------------------------*/

/*
 * Helper function to determine if string suffix_str is a suffix of 
 * string on the left of mark ? in string str1
 */

static int suffix(char* str1, char* suffix_str)
{

	int str1_len;
	int suffix_len;
	char* a;

	if (str1 == NULL || suffix_str == NULL) return 1;

	str1_len = strlen(str1);
	suffix_len = strlen(suffix_str);
	a = strchr(str1, '?');

	if (str1_len == 0 || suffix_len == 0) return 1;

	if (a != NULL)
		return strncasecmp(a-suffix_len, suffix_str, suffix_len);
	else
		return strncasecmp(str1+str1_len-suffix_len, suffix_str, suffix_len);

}

static void toUpper(char* str)
{
	int x = 'A' - 'a';
	for (; *str != '\0'; str++) 
	{
		if (*str >= 'a' && *str <= 'z')
			*str += x;
	}
}

static int monitor_uri(request_rec *r, per_host_state *phs) 
{
  chronos_rule *rules = phs->rules;
  regmatch_t regm[1];

	if (!phs->chronos_enabled)
		return 0;

  while(rules != NULL) 
  {
    if(rules->type == RT_PREFIX)
    {
      if(!strncasecmp(r->uri, rules->expr, strlen(rules->expr)))
	 return rules->monitor;
    }else if(rules->type == RT_SUFFIX)
		{
			if (suffix(r->uri, rules->expr) == 0)
	 return rules->monitor;		
    }else if(rules->type == RT_SUBSTRING)
    {
			toUpper(r->uri);
			toUpper(rules->expr);
      if(strstr(r->uri, rules->expr))
	return rules->monitor;
    }else if(rules->type == RT_REGEX)
    {
  /*ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "uri=%s, the regex is %s", r->uri, rules->expr);*/
			toUpper(r->uri);
      if(!ap_regexec(rules->regexp, r->uri, 1, regm, 0))
	return rules->monitor;
    }

    rules = rules->next;
  }
  
  /* if no rule matches, return true */
  return 1;
}

static int chronos_filter(ap_filter_t *f, apr_bucket_brigade *bb)
{
  apr_table_t *notes = f->r->notes;
  apr_bucket *e;
  const char *content_length_str;
  int content_length = 0;
  char new_content_length_str[128];
  apr_bucket *footerbucket;
  int add_footer;
  const char * footer_note;
  per_host_state *phs;

  /*char null = '\0';*/

  /* check to see if there is a CHRONOS_ADD_FOOTER note.  If there is one, we 
     have already decided whether or not to add a footer. */
  if((footer_note = apr_table_get(notes, CHRONOS_FOOTER)))
    add_footer = (*footer_note == ADD_FOOTER);
  else {
  /*char * type = (f->r->content_type != NULL) ? (char *)f->r->content_type : (char *)(&null);
  ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, f->r, "uri=%s, type is %s, chunked is %d", f->r->uri, type, f->r->chunked);*/
  
    phs = ap_get_module_config(f->r->server->module_config, &chronos_module);
    /* Check if the content type is text/html */
    if(f->r->content_type != NULL && !strncmp(f->r->content_type, "text/html", 
					      9) && monitor_uri(f->r, phs)) {
      apr_table_setn(notes, CHRONOS_FOOTER, &ADD_FOOTER);
      add_footer = 1;

      /* Adjust the content length if we can.  The call to change the Content
         Length MUST be done the first time the filter is called. */
      if((content_length_str = apr_table_get(f->r->headers_out, 
					     "Content-Length"))) {
	sscanf(content_length_str, "%d", &content_length);
	content_length += sizeof(FOOTER_STRING)-1;
	sprintf(new_content_length_str, "%d", content_length);
	apr_table_set(f->r->headers_out, "Content-Length", 
		      new_content_length_str);
	/*ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, f->r, "uri=%s, setting content-length to %d", f->r->uri, content_length);*/
      }
    }else {
      apr_table_setn(notes, CHRONOS_FOOTER, &NO_FOOTER);
      add_footer = 0;
    }
  }

  if(add_footer) { 
    e = APR_BRIGADE_LAST( bb);

    if (APR_BUCKET_IS_EOS(e)) {
      APR_BUCKET_REMOVE(e);
      footerbucket = apr_bucket_transient_create(FOOTER_STRING, 
                       sizeof(FOOTER_STRING)-1, f->c->bucket_alloc);
      /*insert the tag for the javascript, then reinsert the EOS */
      APR_BRIGADE_INSERT_TAIL(bb, footerbucket);
      APR_BRIGADE_INSERT_TAIL(bb, e);   
	/*ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, f->r, "uri=%s, adding footer now", f->r->uri);*/
    }
  }
  
  ap_pass_brigade(f->next, bb);
  return APR_SUCCESS;	
}



/**
 * Stores the date and time in the notes space.  These values 
 * are output in the chronos_log
 */
static int set_date(request_rec *r) {
  apr_table_t *notes = r->notes;
  apr_pool_t *pool = r->pool;
  apr_time_t request_time = r->request_time;
  struct tm *localTime;
  char tmp[32], *entryDate, *entryTime;
  unsigned long secs = (unsigned long) apr_time_sec(request_time);
  unsigned long usecs = (unsigned long) apr_time_usec(request_time);
  cookie_dir_rec *dcfg = ap_get_module_config(r->per_dir_config,
					      &chronos_module);

  /* if not enabled the date has already been set, return */
  if(!dcfg->enabled || apr_table_get(notes, CHRONOS_DATE))
    return OK;

  /* convert to a (struct tm*) */
  localTime = localtime(&secs); 

  /* convert to a date in the format yyyy-mm-dd */
  strftime(tmp, 32, "%Y-%m-%d", localTime);
  entryDate = apr_pstrdup(pool, tmp);
  
  /* print out the time in the format hh-mm-ss.usecs (usecs is 6-wide and 0-padded) */
  strftime(tmp, 32, "%T", localTime);
  entryTime = apr_psprintf(pool, "%s.%06d", tmp, (int)usecs);

  /* store these strings in notes */
  apr_table_setn(notes, CHRONOS_DATE, entryDate);
  apr_table_setn(notes, CHRONOS_TIME, entryTime);

  return OK;
}


static const char *set_chronos_rule(cmd_parms *cmd, void *mconfig,
				    const char* type, const char *expr, 
				    const char* monitor)
{
    per_host_state *phs;
    rule_type myRuleType;
    int monitoring;
    chronos_rule* new_rule, *tmp_rule;

    phs  = ap_get_module_config(cmd->server->module_config,
                                &chronos_module);

    if(!strcmp(type, "PREFIX"))
      myRuleType = RT_PREFIX;
		else if(!strcmp(type, "SUFFIX"))
			myRuleType = RT_SUFFIX;
    else if(!strcmp(type, "SUBSTRING"))
      myRuleType = RT_SUBSTRING;
    else if(!strcmp(type, "REGEX"))
      myRuleType = RT_REGEX;
    else
      return "The first argument for ChronosRule must be either PREFIX, SUFFIX, SUBSTRING or REGEX.";

    if(!strcmp(monitor, "YES"))
       monitoring = 1;
    else if(!strcmp(monitor, "NO"))
       monitoring = 0;
    else
      return "The third arguement for ChronosRule must be either YES or NO.";

    new_rule = (chronos_rule *) apr_palloc(cmd->pool, sizeof(chronos_rule));
    new_rule->expr = apr_pstrdup(cmd->pool, expr);
    new_rule->regexp = ap_pregcomp(cmd->pool, expr, REG_EXTENDED);
    new_rule->type = myRuleType;
    new_rule->monitor = monitoring;
    new_rule->next = NULL;

    if(new_rule->regexp == NULL)
      return "Failed to compile regular expression";
    
    if(phs->rules == NULL)
      phs->rules = new_rule;
    else 
    {
      tmp_rule = phs->rules;
      while(tmp_rule->next != NULL)
	tmp_rule = tmp_rule->next;
      tmp_rule->next = new_rule;
    }

    return NULL;
}

static const char *set_chronos_enable(cmd_parms *cmd, void *mconfig, 
																		const char* enabled)
{
    per_host_state *phs;
    int enable;

    phs  = ap_get_module_config(cmd->server->module_config,
                                &chronos_module);

    if(!strcmp(enabled, "YES"))
       enable = 1;
    else if(!strcmp(enabled, "NO"))
       enable = 0;
    else
      return "The arguement for ChronosEnabled must be either YES or NO.";

		phs->chronos_enabled = enable;

    return NULL;
}


/* User Tracking Module (Was mod_cookies.c)
 *
 * *** IMPORTANT NOTE: This module is not designed to generate
 * *** cryptographically secure cookies.  This means you should not
 * *** use cookies generated by this module for authentication purposes
 *
 * This Apache module is designed to track users paths through a site.
 * It uses the client-side state ("Cookie") protocol developed by Netscape.
 * It is known to work on most browsers.
 *
 * Each time a page is requested we look to see if the browser is sending
 * us a Cookie: header that we previously generated.
 *
 * If we don't find one then the user hasn't been to this site since
 * starting their browser or their browser doesn't support cookies.  So
 * we generate a unique Cookie for the transaction and send it back to
 * the browser (via a "Set-Cookie" header)
 * Future requests from the same browser should keep the same Cookie line.
 *
 * By matching up all the requests with the same cookie you can
 * work out exactly what path a user took through your site.  To log
 * the cookie use the " %{Cookie}n " directive in a custom access log;
 *
 * Example 1 : If you currently use the standard Log file format (CLF)
 * and use the command "TransferLog somefilename", add the line
 *       LogFormat "%h %l %u %t \"%r\" %s %b %{Cookie}n"
 * to your config file.
 *
 * Example 2 : If you used to use the old "CookieLog" directive, you
 * can emulate it by adding the following command to your config file
 *       CustomLog filename "%{Cookie}n \"%r\" %t"
 *
 * Mark Cox, mjc@apache.org, 6 July 95
 *
 * This file replaces mod_cookies.c
 */


/* Make Cookie: Now we have to generate something that is going to be
 * pretty unique.  We can base it on the pid, time, hostip */

static void make_cookie(request_rec *r)
{
    per_host_state *phs = ap_get_module_config(r->server->module_config,
						 &chronos_module);
    /* 1024 == hardcoded constant */
    char cookiebuf[1024];
    char *new_cookie;
    const char *rname = ap_get_remote_host(r->connection, r->per_dir_config,
					   REMOTE_NAME, NULL);
    cookie_dir_rec *dcfg;

    dcfg = ap_get_module_config(r->per_dir_config, &chronos_module);

    /* XXX: hmm, this should really tie in with mod_unique_id */
    apr_snprintf(cookiebuf, sizeof(cookiebuf), "%s.%" APR_TIME_T_FMT, rname, 
                 apr_time_now());

    if (phs->expires) {

        /* Cookie with date; as strftime '%a, %d-%h-%y %H:%M:%S GMT' */
        new_cookie = apr_psprintf(r->pool, "%s=%s; path=/",
                                  dcfg->cookie_name, cookiebuf);

        if ((dcfg->style == CT_UNSET) || (dcfg->style == CT_NETSCAPE)) {
	    apr_time_exp_t tms;
            apr_time_exp_gmt(&tms, r->request_time 
                                 + apr_time_from_sec(phs->expires));
            new_cookie = apr_psprintf(r->pool,
                                       "%s; expires=%s, "
                                       "%.2d-%s-%.2d %.2d:%.2d:%.2d GMT",
                                       new_cookie, apr_day_snames[tms.tm_wday],
                                       tms.tm_mday,
                                       apr_month_snames[tms.tm_mon],
                                       tms.tm_year % 100,
                                       tms.tm_hour, tms.tm_min, tms.tm_sec);
        }
        else {
            new_cookie = apr_psprintf(r->pool, "%s; max-age=%d",
                                      new_cookie, phs->expires);
        }
    }
    else {
        new_cookie = apr_psprintf(r->pool, "%s=%s; path=/",
                                  dcfg->cookie_name, cookiebuf);
    }
    if (dcfg->cookie_domain != NULL) {
        new_cookie = apr_pstrcat(r->pool, new_cookie, "; domain=",
                                 dcfg->cookie_domain,
                                 (dcfg->style == CT_COOKIE2
                                  ? "; version=1"
                                  : ""),
                                 NULL);
    }

    apr_table_addn(r->headers_out,
                   (dcfg->style == CT_COOKIE2 ? "Set-Cookie2" : "Set-Cookie"),
                   new_cookie);
    apr_table_setn(r->notes, COOKIE_NOTE, apr_pstrdup(r->pool, cookiebuf));   /* log first time */
    return;
}

/* dcfg->regexp is "^cookie_name=([^;]+)|;[ \t]+cookie_name=([^;]+)",
 * which has three subexpressions, $0..$2 */
#define NUM_SUBS 3

static void set_and_comp_regexp(cookie_dir_rec *dcfg, 
                                apr_pool_t *p,
                                const char *cookie_name) 
{
    /* The goal is to end up with this regexp, 
     * ^cookie_name=([^;]+)|;[\t]+cookie_name=([^;]+) 
     * with cookie_name obviously substituted either
     * with the real cookie name set by the user in httpd.conf, or with the
     * default COOKIE_NAME. */
    dcfg->regexp_string = apr_pstrcat(p, "^", cookie_name, "=([^;]+)|;[ \t]+", cookie_name, "=([^;]+)", NULL);

    dcfg->regexp = ap_pregcomp(p, dcfg->regexp_string, REG_EXTENDED);
    ap_assert(dcfg->regexp != NULL);
}

static int spot_cookie(request_rec *r)
{
    cookie_dir_rec *dcfg = ap_get_module_config(r->per_dir_config,
						&chronos_module);
    const char *cookie_header;
    regmatch_t regm[NUM_SUBS];

    /* Do not run in subrequests */
    if (!dcfg->enabled || r->main) {
        return DECLINED;
    }

    if ((cookie_header = apr_table_get(r->headers_in, "Cookie"))) {
        if (!ap_regexec(dcfg->regexp, cookie_header, NUM_SUBS, regm, 0)) {
            char *cookieval = NULL;
            /* Our regexp,
             * ^cookie_name=([^;]+)|;[ \t]+cookie_name=([^;]+)
             * only allows for $1 or $2 to be available. ($0 is always
             * filled with the entire matched expression, not just
             * the part in parentheses.) So just check for either one
             * and assign to cookieval if present. */
            if (regm[1].rm_so != -1) {
                cookieval = ap_pregsub(r->pool, "$1", cookie_header,
                                       NUM_SUBS, regm);
            }
            if (regm[2].rm_so != -1) {
                cookieval = ap_pregsub(r->pool, "$2", cookie_header,
                                       NUM_SUBS, regm);
            }
            /* Set the cookie in a note, for logging */
            apr_table_setn(r->notes, COOKIE_NOTE, cookieval);

            return DECLINED;    /* There's already a cookie, no new one */
        }
    }
    make_cookie(r);
    return OK;                  /* We set our cookie */
}

static void *make_per_host_state(apr_pool_t *p, server_rec *s)
{
    per_host_state *phs =
    (per_host_state *) apr_palloc(p, sizeof(per_host_state));

    phs->expires = 0;

    phs->rules = NULL;

    return (void *) phs;
}

static void *make_cookie_dir(apr_pool_t *p, char *d)
{
    cookie_dir_rec *dcfg;

    dcfg = (cookie_dir_rec *) apr_pcalloc(p, sizeof(cookie_dir_rec));
    dcfg->cookie_name = COOKIE_NAME;
    dcfg->cookie_domain = NULL;
    dcfg->style = CT_UNSET;
    dcfg->enabled = 0;

    /* In case the user does not use the ChronosCookieName directive,
     * we need to compile the regexp for the default cookie name. */
    set_and_comp_regexp(dcfg, p, COOKIE_NAME);

    return dcfg;
}

static const char *set_cookie_enable(cmd_parms *cmd, void *mconfig, int arg)
{
    cookie_dir_rec *dcfg = mconfig;

    dcfg->enabled = arg;
    return NULL;
}

static const char *set_cookie_exp(cmd_parms *parms, void *dummy,
                                  const char *arg)
{
    per_host_state *phs;
    time_t factor, modifier = 0;
    time_t num = 0;
    char *word;

    phs  = ap_get_module_config(parms->server->module_config,
                                &chronos_module);
    /* The simple case first - all numbers (we assume) */
    if (apr_isdigit(arg[0]) && apr_isdigit(arg[strlen(arg) - 1])) {
        phs->expires = atol(arg);
        return NULL;
    }

    /*
     * The harder case - stolen from mod_expires 
     *
     * CookieExpires "[plus] {<num> <type>}*"
     */

    word = ap_getword_conf(parms->pool, &arg);
    if (!strncasecmp(word, "plus", 1)) {
        word = ap_getword_conf(parms->pool, &arg);
    };

    /* {<num> <type>}* */
    while (word[0]) {
        /* <num> */
	if (apr_isdigit(word[0]))
            num = atoi(word);
        else
            return "bad expires code, numeric value expected.";

        /* <type> */
        word = ap_getword_conf(parms->pool, &arg);
        if (!word[0])
            return "bad expires code, missing <type>";

        factor = 0;
        if (!strncasecmp(word, "years", 1))
            factor = 60 * 60 * 24 * 365;
        else if (!strncasecmp(word, "months", 2))
            factor = 60 * 60 * 24 * 30;
        else if (!strncasecmp(word, "weeks", 1))
            factor = 60 * 60 * 24 * 7;
        else if (!strncasecmp(word, "days", 1))
            factor = 60 * 60 * 24;
        else if (!strncasecmp(word, "hours", 1))
            factor = 60 * 60;
        else if (!strncasecmp(word, "minutes", 2))
            factor = 60;
        else if (!strncasecmp(word, "seconds", 1))
            factor = 1;
        else
            return "bad expires code, unrecognized type";

        modifier = modifier + factor * num;

        /* next <num> */
        word = ap_getword_conf(parms->pool, &arg);
    }

    phs->expires = modifier;

    return NULL;
}

static const char *set_cookie_name(cmd_parms *cmd, void *mconfig,
                                   const char *name)
{
    cookie_dir_rec *dcfg = (cookie_dir_rec *) mconfig;

    dcfg->cookie_name = apr_pstrdup(cmd->pool, name);

    set_and_comp_regexp(dcfg, cmd->pool, name);

    if (dcfg->regexp == NULL) {
        return "Regular expression could not be compiled.";
    }
    if (dcfg->regexp->re_nsub + 1 != NUM_SUBS) {
        return apr_pstrcat(cmd->pool, "Invalid cookie name \"",
                           name, "\"", NULL);
    }

    return NULL;
}

/*
 * Set the value for the 'Domain=' attribute.
 */
static const char *set_cookie_domain(cmd_parms *cmd, void *mconfig,
                                     const char *name)
{
    cookie_dir_rec *dcfg;

    dcfg = (cookie_dir_rec *) mconfig;

    /*
     * Apply the restrictions on cookie domain attributes.
     */
    if (strlen(name) == 0) {
        return "CookieDomain values may not be null";
    }
    if (name[0] != '.') {
        return "CookieDomain values must begin with a dot";
    }
    if (ap_strchr_c(&name[1], '.') == NULL) {
        return "CookieDomain values must contain at least one embedded dot";
    }

    dcfg->cookie_domain = apr_pstrdup(cmd->pool, name);
    return NULL;
}

/*
 * Make a note of the cookie style we should use.
 */
static const char *set_cookie_style(cmd_parms *cmd, void *mconfig,
                                    const char *name)
{
    cookie_dir_rec *dcfg;

    dcfg = (cookie_dir_rec *) mconfig;

    if (strcasecmp(name, "Netscape") == 0) {
        dcfg->style = CT_NETSCAPE;
    }
    else if ((strcasecmp(name, "Cookie") == 0)
             || (strcasecmp(name, "RFC2109") == 0)) {
        dcfg->style = CT_COOKIE;
    }
    else if ((strcasecmp(name, "Cookie2") == 0)
             || (strcasecmp(name, "RFC2965") == 0)) {
        dcfg->style = CT_COOKIE2;
    }
    else {
        return apr_psprintf(cmd->pool, "Invalid %s keyword: '%s'",
                            cmd->cmd->name, name);
    }

    return NULL;
}

static const command_rec chronos_cmds[] = {
    AP_INIT_TAKE1("ChronosCookieExpires", set_cookie_exp, NULL, OR_FILEINFO,
                  "an expiry date code"),
    AP_INIT_TAKE1("ChronosCookieDomain", set_cookie_domain, NULL, OR_FILEINFO,
                  "domain to which this cookie applies"),
    AP_INIT_TAKE1("ChronosCookieStyle", set_cookie_style, NULL, OR_FILEINFO,
                  "'Netscape', 'Cookie' (RFC2109), or 'Cookie2' (RFC2965)"),
    AP_INIT_TAKE1("ChronosCookieName", set_cookie_name, NULL, OR_FILEINFO,
                  "name of the tracking cookie"),
    AP_INIT_FLAG("ChronosModuleSupport", set_cookie_enable, NULL, OR_FILEINFO,
                 "whether or not to enable cookies and set date for logging"),
    AP_INIT_TAKE3("ChronosRule", set_chronos_rule, NULL, OR_FILEINFO,
                  "rule for matching URLs"),
    AP_INIT_TAKE1("ChronosEnabled", set_chronos_enable, NULL, OR_FILEINFO,
                  "whether or not to enable chronos"),
    {NULL}
};

/*
 * The hooks...
 */

static void register_hooks(apr_pool_t *p)
{
    ap_hook_fixups(spot_cookie,NULL,NULL,APR_HOOK_MIDDLE);

  /* Register my date-setting function to run after the request has been
     received by Apache, but before any processing is done on it. */
  ap_hook_post_read_request(set_date, NULL, NULL, APR_HOOK_MIDDLE);
  /* Register is at the level AP_FTYPE_CONTENT_SET+1 so that it will be called 
     after the other filters that change the content. */
  ap_register_output_filter("CHRONOS", chronos_filter, NULL, AP_FTYPE_CONTENT_SET+1);
}

module AP_MODULE_DECLARE_DATA chronos_module = {
    STANDARD20_MODULE_STUFF,
    make_cookie_dir,            /* dir config creater */
    NULL,                       /* dir merger --- default is to override */
    make_per_host_state,        /* server config */
    NULL,                       /* merge server configs */
    chronos_cmds,            /* command apr_table_t */
    register_hooks		/* register hooks */
};

/* end of file mod_chronos.c */

