#!/usr/local/bin/perl
# 
# $Header: emll/sysman/admin/scripts/beehive/bhv_config.pl /main/1 2009/06/27 00:41:48 rgorle Exp $
#
# bhv_config.pl
# 
# Copyright (c) 2006, 2009, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      bhv_config.pl - <one-line expansion of the name>
#
#    DESCRIPTION
#      <short description of component this file declares/defines>
#
#    NOTES
#      <other useful comments, qualifications, etc.>
#
#    MODIFIED   (MM/DD/YY)
#    rgorle      05/11/07 - Creation 
# 
# Include Libraries
use Getopt::Std;
use Text::ParseWords;
use XML::Parser;
use File::stat;
use strict;

#---------------------------------------------------------------------------------------------
# Global variables
#---------------------------------------------------------------------------------------------
my $oraHome                 = "";
my $config_file             = "";
my $input_file              = "";
my $input_obj_id            = "";
my $input_obj_type          = "";
my $midtier_id              = "";
my $input_parse_obj = {};
my $sys_obj_type_map = {};
my $sys_obj_map = {};
my $output_target_list = [];

#---------------------------------------------------------------------------------------------
# Main Logic
#---------------------------------------------------------------------------------------------
# this is to execute the script from terminal
#processCommandLine();
#exit(0);

#---------------------------------------------------------------------------------------------
# main function to be called from other functions
#---------------------------------------------------------------------------------------------
sub getOutputTargets
{
    my ($tmp_input_file, $tmp_input_obj_id, $tmp_input_obj_type, $tmp_oracle_home) = @_;
    $input_file = $tmp_input_file;
    $input_obj_id = $tmp_input_obj_id;
    $input_obj_type = $tmp_input_obj_type;
    $oraHome = $tmp_oracle_home;

    my $output_msg = process();
    return $output_msg;
}
#---------------------------------------------------------------------------------------------
# Parses the command line parameters  and updates the global variables
#---------------------------------------------------------------------------------------------
sub processCommandLine
{
    my %args_hash;
    getopts('i:m:t:o:', \%args_hash);
    $input_file = (exists $args_hash{i} ) ? trim($args_hash{i}) : "";
    $input_obj_id = (exists $args_hash{m} ) ? trim($args_hash{m}) : "";
    $input_obj_type = (exists $args_hash{t} ) ? trim($args_hash{t}) : "";
    $oraHome = (exists $args_hash{o} ) ? trim($args_hash{o}) : "";

    my $output_msg = process();
    print $output_msg;
}
#---------------------------------------------------------------------------------------------
# processing the input and provide the output 
#---------------------------------------------------------------------------------------------
sub process
{
    # check the input params
    my $output_msg = checkInputParams();
    return $output_msg if(length($output_msg) > 0);
    
    # parse the input file
    parseInputFile();
    # parse the output file
    parseConfigFile();
    # process the input file 
    $output_msg = processInputFile();
    return $output_msg;
}
#---------------------------------------------------------------------------------------------
# verify env parameters and set the path variables 
#---------------------------------------------------------------------------------------------
sub checkInputParams
{
    $oraHome = (defined $oraHome) ? trim($oraHome) : "";
    if($oraHome eq "")
    {
        return getWarningMsg("101","ORACLE_HOME not provided.");
    }

    # setting the config directory first, later add the file name
    my $config_dir = $oraHome. "/beehive/tmp/exported_configuration/";

    #replace \ with / in the input file
    $input_file =~ s/\\/\//g;
    $config_dir =~ s/\\/\//g;
    #replace /+ with / 
    $input_file =~ s/\/+/\//g;
    $config_dir =~ s/\/+/\//g;

    if (($^O =~ "Windows") || ($^O =~ "MSWin32"))  # windows platforms
    {
        # For Windows change the path  from / to \
        $input_file =~ s/\//\\\\/g;
        $config_dir =~ s/\//\\\\/g;
    }

    if (!((-e $input_file) && (-r $input_file))) {
        return getWarningMsg("101","Input File neither exist nor readible.");
    }
   
    # find the latest updated config file
    if (-d $config_dir) {
        my $latest;
        opendir DIR, $config_dir or return getWarningMsg("101","Can not open Configuration directory.");
        while(defined (my $file = readdir(DIR)) ) {
            if(($file =~ /exported_configuration_[0-9]*.xml/) && ($file =~ /xml$/) && (-e $config_dir.$file) && (-r $config_dir.$file)) {                
                my $mtime = stat($config_dir.$file)->mtime;
                if((!defined ($latest)) or ($mtime > $latest)) {
                    $config_file = $config_dir.$file;
                    $latest = $mtime;
                }
            }
        }
        closedir DIR;

        if(length($config_file) <= 0) {
            return getWarningMsg("101","Configuration File neither exist nor readible.");
        }
    } else {
        return getWarningMsg("101","Configuration directory does not exist.");
    }

    return "";
}
#---------------------------------------------------------------------------------------------
# Parse the input file and get the copy of the types 
#---------------------------------------------------------------------------------------------
sub parseInputFile {
    # parse the input file
    my $parser = new XML::Parser(Handlers => {Init => \&handle_Input_Init, Start => \&handle_Input_Start, End => \&handle_Input_End, Final => \&handle_Input_Final});
    $parser->parsefile($input_file);
}
#---------------------------------------------------------------------------------------------
sub handle_Input_Init {
    my $expat = shift;
    $expat->{Nodes} = [];
    $expat->{SysObjTypeMap} = ();
}
#---------------------------------------------------------------------------------------------
sub handle_Input_Start {
    (my $expat, my $tag, my %atts) = @_;

    if(defined $expat->{CurNode}) {
        push @{$expat->{Nodes}}, $expat->{CurNode}; 
    }

    if($tag eq "DiscoveryInput") {
        $expat->{CurNode} = ();
        $expat->{CurNode}->{Targets} = [];
    }elsif($tag eq "InputProperties") {
        $expat->{CurNode} = ();
        $expat->{CurNode}->{PropertyElements} = [];
    }elsif($tag eq "Target") {
        $expat->{CurNode} = ();
        $expat->{CurNode}->{Atts} = ();
        $expat->{CurNode}->{Targets} = [];
        $expat->{CurNode}->{MBeanElements} = [];
        $expat->{CurNode}->{PropertyElements} = [];
        $expat->{CurNode}->{PropertyListElements} = [];        
    }elsif($tag eq "MBean") {
        $expat->{CurNode} = ();
        $expat->{CurNode}->{Atts} = ();
        $expat->{CurNode}->{MBeanElements} = [];
        $expat->{CurNode}->{PropertyElements} = [];
        $expat->{CurNode}->{PropertyListElements} = [];        
    }elsif($tag eq "PropertyList") {
        $expat->{CurNode} = ();
        $expat->{CurNode}->{MBeanElements} = [];
        $expat->{CurNode}->{PropertyElements} = [];
    }elsif($tag eq "Property") {
        $expat->{CurNode} = ();
        $expat->{CurNode}->{Atts} = ();
    }

    if(exists $expat->{CurNode}->{Atts}) {
        foreach (keys %atts) {
            $expat->{CurNode}->{Atts}->{$_} = %atts->{$_};
        }
    }

    my $type = (exists %atts->{"beanType"}) ? trim(%atts->{"beanType"}) : "";
    if((length($type) > 0) && (!exists $expat->{SysObjTypeMap}->{$type})) {
        $expat->{SysObjTypeMap}->{$type} = [];
    }
    
    my $attName = (exists %atts->{"attName"}) ? trim(%atts->{"attName"}) : "";
    my @types = $attName =~ m/\((.*)\)/g;
    foreach $type (@types){
        $type = trim($type);
        if((length($type) > 0) && (!exists $expat->{SysObjTypeMap}->{$type})) {
            $expat->{SysObjTypeMap}->{$type} = [];
        }
    }
}
#---------------------------------------------------------------------------------------------
sub handle_Input_End {
    (my $expat,my $tag) = @_;

    return if($tag eq "DiscoveryInput");
    
    my $parentNode = pop @{$expat->{Nodes}};
    if($tag eq "InputProperties") {
        $parentNode->{InputProperties} = ();       
        foreach (@{$expat->{CurNode}->{PropertyElements}}) {
            my $currentProperty = $_;
            foreach (keys %{$currentProperty->{Atts}}) {
                my $key = $_;
                my $value = $currentProperty->{Atts}->{$key};
                $parentNode->{InputProperties}->{$key} = $value;
            }
        }        
    }elsif($tag eq "Target") {
        push @{$parentNode->{Targets}}, $expat->{CurNode}; 
    }elsif($tag eq "MBean") {
        push @{$parentNode->{MBeanElements}}, $expat->{CurNode}; 
    }elsif($tag eq "PropertyList") {
        push @{$parentNode->{PropertyListElements}}, $expat->{CurNode}; 
    }elsif($tag eq "Property") {
        push @{$parentNode->{PropertyElements}}, $expat->{CurNode}; 
    }

    if(exists $expat->{CurNode}->{Targets} && @{$expat->{CurNode}->{Targets}} <= 0){
        delete $expat->{CurNode}->{Targets};
    }
    if(exists $expat->{CurNode}->{MBeanElements} && @{$expat->{CurNode}->{MBeanElements}} <= 0){
        delete $expat->{CurNode}->{MBeanElements};
    }
    if(exists $expat->{CurNode}->{PropertyElements} && @{$expat->{CurNode}->{PropertyElements}} <= 0){
        delete $expat->{CurNode}->{PropertyElements};
    }
    if(exists $expat->{CurNode}->{PropertyListElements} && @{$expat->{CurNode}->{PropertyListElements}} <= 0){
        delete $expat->{CurNode}->{PropertyListElements};
    }

    $expat->{CurNode} = $parentNode;
}
#---------------------------------------------------------------------------------------------
sub handle_Input_Final {
    my $expat = shift;
    delete $expat->{Nodes};
    $sys_obj_type_map = delete $expat->{SysObjTypeMap};
    $input_parse_obj = delete $expat->{CurNode};
}
#---------------------------------------------------------------------------------------------
# Parse the config file. Use the stored types and parse only objects which corresponds to these 
# types. This will not save the unwanted remaining objects.
#---------------------------------------------------------------------------------------------
sub parseConfigFile {
    # parse the configuration file
    my $parser = new XML::Parser(Handlers => {Init => \&handle_Config_Init, Start => \&handle_Config_Start, Char => \&handle_Config_Char, End => \&handle_Config_End, Final => \&handle_Config_Final});
    $parser->parsefile($config_file);
}
#---------------------------------------------------------------------------------------------
sub handle_Config_Init {
    my $expat = shift;
    $expat->{SysObjMap} = ();
    $expat->{BeehiveInstanceId} = "";
}
#---------------------------------------------------------------------------------------------
sub handle_Config_Start {
    (my $expat, my $tag, my %atts) = @_;
    if($tag eq "system_object") {
        my $identifier =  trim(%atts->{"identifier"});
        my $type = get_type(%atts->{"system_class"});
        my $alias = %atts->{"alias"};
        if(exists $sys_obj_type_map->{$type}) {
            $expat->{CurObject} = ();
            $expat->{CurObject}->{type} = $type;
            $expat->{CurObject}->{properties} = ();

            $expat->{CurObject}->{properties}->{"Alias"} = [];
            push @{$expat->{CurObject}->{properties}->{"Alias"}}, $alias; 

            $expat->{SysObjMap}->{$identifier} = $expat->{CurObject};
            push @{$sys_obj_type_map->{$type}}, $identifier;
        }
    }elsif($tag eq "property") {
        if(defined $expat->{CurObject}) {            
            $expat->{CurProperty} = trim(%atts->{"name"});            
            $expat->{CurPropertyValue} = [];
            $expat->{CurObject}->{properties}->{$expat->{CurProperty}} = $expat->{CurPropertyValue};
        }
    }elsif($tag eq "value") {
        if(defined $expat->{CurPropertyValue}) {
            $expat->{ValidCurrentValue} = "true";
        }
    }elsif($tag eq "identification_of_exporting_component") {
        $expat->{BeehiveInstanceId} = trim(%atts->{"home_instance"});
    }
}
#---------------------------------------------------------------------------------------------
sub handle_Config_Char {
    my ($expat, $data) = @_;
    if((defined $expat->{ValidCurrentValue}) && ($expat->{ValidCurrentValue} eq "true")) {
        my $value = trim($data);
        if( length($value) > 0 ) {
            push @{$expat->{CurPropertyValue}}, $value;
        }
    }
}
#---------------------------------------------------------------------------------------------
sub handle_Config_End {
    (my $expat,my $tag) = @_;
    if($tag eq "system_object") {
        delete $expat->{CurObject};
    }elsif($tag eq "property") {
        delete $expat->{CurProperty};
        delete $expat->{CurPropertyValue};
    }elsif($tag eq "value") {
        delete $expat->{ValidCurrentValue};
    }
}
#---------------------------------------------------------------------------------------------
sub handle_Config_Final {
    my $expat = shift;
    $sys_obj_map = delete $expat->{SysObjMap};
    $midtier_id = delete $expat->{BeehiveInstanceId};
}
#---------------------------------------------------------------------------------------------
sub processInputFile {
    #process the targets based on the input file and configuration file
    processAllTargets($input_parse_obj->{Targets}, "   ", $output_target_list);

    # get the output targets
    my $tab_space = "  ";
    my $output_msg = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n";
    $output_msg = $output_msg."<DiscoveryResult>\n";
    $output_msg = $output_msg.printTargets($output_target_list, $tab_space);
    $output_msg = $output_msg."</DiscoveryResult>\n";

    return $output_msg;
}
#---------------------------------------------------------------------------------------------
sub processAllTargets {
    my ($inputTargets, $superIdentifier, $outputTargets) = @_;

    foreach (@$inputTargets) {
        my $inputTarget = $_;
        my @outputObjIds = ();        
        my $objType = (exists $inputTarget->{Atts}->{"beanType"}) ? $inputTarget->{Atts}->{"beanType"} : "";
        my $objId = (exists $inputTarget->{Atts}->{"beanName"}) ? $inputTarget->{Atts}->{"beanName"} : "";
        my $attName = (exists $inputTarget->{Atts}->{"attName"}) ? $inputTarget->{Atts}->{"attName"} : "";

        if(length($objId) > 0) {
            push @outputObjIds, $objId;
        }elsif (length($attName) > 0 && length($superIdentifier) > 0) {
            getPropValue($superIdentifier, $attName, \@outputObjIds);
        }elsif (length($objType) > 0) {
            getObjIdsByType($objType, \@outputObjIds);
        }

        foreach (@outputObjIds) {
            my $outputObjId = $_;
            my $tempOutputTargets = $outputTargets;            
            if(((length($input_obj_type) == 0) || ($input_obj_type eq $objType)) && ((length($input_obj_id) == 0) || ($input_obj_id eq $outputObjId))) {
                my $outputTarget = {};
                $outputTarget->{Atts} = {};
                $outputTarget->{Atts}->{beanType} = $objType;
                $outputTarget->{Targets} = [];
                $outputTarget->{PropertyElements} = [];                                     
                
                my $property = {};
                $property->{beanName} = [];
                
                push @{$property->{beanName}}, $outputObjId;                
                push @{$outputTarget->{PropertyElements}}, $property;
                push @{$outputTargets}, $outputTarget;
                                                                    
                processMBeanElement($inputTarget, $outputTarget, $outputObjId, "false", "");
                $tempOutputTargets = $outputTarget->{Targets};
            }
            
            processAllTargets($inputTarget->{Targets}, $outputObjId, $tempOutputTargets);
        }
    }
}
#---------------------------------------------------------------------------------------------
sub processMBeanElement {
    my ($inputTarget, $outputTarget, $outputObjId, $inPropertyList, $property) = @_;
    my $tempPropList = [];

    if(exists $inputTarget->{PropertyElements}) {
        my $propertyElements = $inputTarget->{PropertyElements};
        foreach (@$propertyElements) {
            my $propertyElement = $_;
            my $propName = (exists $propertyElement->{Atts}->{"name"}) ? $propertyElement->{Atts}->{"name"} : "";
            my $attName = (exists $propertyElement->{Atts}->{"attName"}) ? $propertyElement->{Atts}->{"attName"} : "";
            my $nameValue = [];
            if(length($attName) <= 0) {
                if(exists $input_parse_obj->{InputProperties}->{$propName}) {
                    push @{$nameValue},  $input_parse_obj->{InputProperties}->{$propName};
                }elsif($propName eq "isCurrentMidtier") {
                    if($outputObjId eq $midtier_id) {
                        push @{$nameValue},  "true";
                    }else {
                        push @{$nameValue},  "false";
                    }
                }
            }else {
                getPropValue($outputObjId, $attName, $nameValue);
                if(@$nameValue <= 0) {
                    push @{$nameValue}, "";
                }
            }        
            if($inPropertyList eq "true") {
                $property->{$propName} = $nameValue;
                push @$tempPropList, $propName;
            } else {
                my $property = {};
                $property->{$propName} = $nameValue;
                push @{$outputTarget->{PropertyElements}}, $property;
            }
        }
    }

    if($inPropertyList eq "false") {
        my $propertyListElements = $inputTarget->{PropertyListElements};
        foreach (@$propertyListElements) {
            my $propertyListElement = $_;
            processPropertyListElement($propertyListElement, $outputTarget, $outputObjId);        
        }
    }
    
    my $mBeanElements = $inputTarget->{MBeanElements};
    my $noInnerMBeans = "true";
    foreach (@$mBeanElements) {
        my $mBeanElement = $_;        
        my $attName = (exists $mBeanElement->{Atts}->{"attName"}) ? $mBeanElement->{Atts}->{"attName"} : "";
        if(length($attName) > 0) {
            my $mBeanObjIds = [];            
            getPropValue($outputObjId, $attName, $mBeanObjIds);
            foreach (@$mBeanObjIds) {
                my $mbeanObjId = $_;
                processMBeanElement($mBeanElement, $outputTarget, $mbeanObjId, $inPropertyList, $property);
            }            
            if(@$mBeanObjIds > 0) {
                $noInnerMBeans = "false";
            }
        }
    }
    
    if($inPropertyList eq "true") {
        if((@$mBeanElements <= 0) && ($noInnerMBeans eq "true")) {
            my $newProperty = {};
            foreach (keys %$property) {
                my $key = $_;
                my $value = $property->{$key};
                $newProperty->{$key} = $value;
            }
            push @{$outputTarget->{PropertyElements}}, $newProperty;
        }

        foreach (@$tempPropList) {
            my $tempPropName = $_;
            delete $property->{$tempPropName};
        }
    }
}
#---------------------------------------------------------------------------------------------
sub processPropertyListElement {
    my ($inputTarget, $outputTarget, $outputObjId) = @_;
    my $property = {};
   
    if(exists $inputTarget->{PropertyElements}) {
        my $propertyElements = $inputTarget->{PropertyElements};

        foreach (@$propertyElements) {
            my $propertyElement = $_;
            my $propName = (exists $propertyElement->{Atts}->{"name"}) ? $propertyElement->{Atts}->{"name"} : "";
            my $attName = (exists $propertyElement->{Atts}->{"attName"}) ? $propertyElement->{Atts}->{"attName"} : "";
            my $nameValue = [];
            if(length($attName) <= 0) {
                if(exists $input_parse_obj->{InputProperties}->{$propName}) {
                    push @{$nameValue},  $input_parse_obj->{InputProperties}->{$propName};
                }elsif($propName eq "isCurrentMidtier") {
                    if($outputObjId eq $midtier_id) {
                        push @{$nameValue},  "true";
                    }else {
                        push @{$nameValue},  "false";
                    }
                }
            }else {
                getPropValue($outputObjId, $attName, $nameValue);
                if(@$nameValue <= 0) {
                    push @{$nameValue}, "";
                }
            }                
            $property->{$propName} = $nameValue;
        }
    }

    my $mBeanElements = $inputTarget->{MBeanElements};
    foreach (@$mBeanElements) {
        my $mBeanElement = $_;        
        my $attName = (exists $mBeanElement->{Atts}->{"attName"}) ? $mBeanElement->{Atts}->{"attName"} : "";
        if(length($attName) > 0) {
            my $mBeanObjIds = [];            
            getPropValue($outputObjId, $attName, $mBeanObjIds);
            foreach (@$mBeanObjIds) {
                my $mbeanObjId = $_;
                processMBeanElement($mBeanElement, $outputTarget, $mbeanObjId, "true", $property);                
            }
        }
    }

    if(@$mBeanElements <= 0) {
        push @{$outputTarget->{PropertyElements}}, $property;
    }
}
#---------------------------------------------------------------------------------------------
sub getPropValue {
    my ($objIdentifier, $attName, $outputList) = @_;
    my @attribs = $attName =~ m/(\w+)[\((.*)\)]*/g;    
    $attName = $attribs[0];
    my $attType = "";
    if($#attribs > 0) {
        $attType = trim($attribs[$#attribs]);
    }
    if(exists $sys_obj_map->{$objIdentifier}) {
        if(exists $sys_obj_map->{$objIdentifier}->{properties}->{$attName}) {
            my $propValue =  $sys_obj_map->{$objIdentifier}->{properties}->{$attName};        
            if(length($attType)) {
                foreach (@$propValue) {
                    if((exists $sys_obj_map->{$_}) && ($sys_obj_map->{$_}->{type} eq $attType)) {
                        push @$outputList, $_;
                    }
                }
            }else {
                foreach (@$propValue) {
                    push @$outputList, $_;
                }
            }
        }
    }
}
#---------------------------------------------------------------------------------------------
sub getObjIdsByType {
    my ($objType, $outputList) = @_;
    if(exists $sys_obj_type_map->{$objType}) {
        my $objIds = $sys_obj_type_map->{$objType};
        foreach (@$objIds) {
            push @$outputList, $_;
        }
    }
}
#---------------------------------------------------------------------------------------------
sub printTargets {
    my ($outputTargets, $tab_space) = @_;
    my $output_msg = "";
    foreach (@$outputTargets) {
        my $outputTarget = $_;
        $output_msg = $output_msg.$tab_space."<Target beanType=\"".$outputTarget->{Atts}->{beanType}."\">\n";
        if(exists $outputTarget->{PropertyElements}) {
            my $propElements = $outputTarget->{PropertyElements};            
            foreach (@$propElements) {
                my $propElement = $_;
                if(keys %$propElement > 0) {
                    my $totalCount = 1;
                    my $sizes = [];
                    my $indices = [];
                    foreach (keys %$propElement) {
                        my $propValue = $propElement->{$_};
                        my $count = @$propValue;
                        $totalCount *= $count;
                        push @$sizes, $count;
                        push @$indices, 0;
                    }
                    my $nameCount = @$sizes;
                    $indices->[$nameCount-1] = -1;
                    for (my $i=0; $i<$totalCount; $i++) {
                        $output_msg = $output_msg.$tab_space."  "."<Property";
                        $indices->[$nameCount-1]++;
                        for (my $j=0; $j<($nameCount-1); $j++) {
                            my $lastIndex = $nameCount-1-$j;                            
                            if(($indices->[$lastIndex]/$sizes->[$lastIndex]) >= 1) {
                                $indices->[$lastIndex] = 0;
                                $indices->[$lastIndex-1]++;
                            }
                        }

                        my $j=0;                    
                        foreach (keys %$propElement) {
                            my $propKey = $_;
                            my $propValue = $propElement->{$propKey};
                            $output_msg = $output_msg." ".$propKey."=\"".$propValue->[$indices->[$j]]."\"";
                            $j++;
                        }
                        $output_msg = $output_msg."/>\n";
                    }
                }
            }
            $output_msg = $output_msg.printTargets($outputTarget->{Targets}, $tab_space."  ");
        }
        $output_msg = $output_msg.$tab_space."</Target>\n";
    }
    return $output_msg;
}
#---------------------------------------------------------------------------------------------
sub get_type {
    my ($input_var) = @_;
    my @tokens = &quotewords('\.', 0, $input_var);        
    my $type = $tokens[$#tokens];        
    @tokens = &quotewords('\$', 0, $type);
    $type = $tokens[$#tokens];
    return $type;
}
#---------------------------------------------------------------------------------------------
sub trim {
    my @out = @_;
    for(@out) {
        s/^\s+//;
        s/\s+$//;
    }
    return wantarray ? @out : $out[0];
}
#---------------------------------------------------------------------------------------------
sub getWarningMsg {
    (my $code, my $msg) = @_;
    my $warnmsg = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n". "<DiscoveryResult>\n"."  <Error code=\"$code\" message=\"$msg\"/>\n"."</DiscoveryResult>\n";
    return $warnmsg;
}
