#!/usr/local/bin/perl
# 
# $Header: emll/sysman/admin/scripts/ccr/ccr_common.pl /main/33 2012/06/14 16:46:36 jsutton Exp $
#
# ccr_common.pl
# 
# Copyright (c) 2005, 2012, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      ccr_common.pl - <one-line expansion of the name>
#
#    DESCRIPTION
#      <short description of component this file declares/defines>
#
#    NOTES
#      <other useful comments, qualifications, etc.>
#
#    MODIFIED   (MM/DD/YY)
#    fmorshed    10/04/12 - Change to use ORACLE_HOME (from LL_ORACLE_HOME).  Some scripts, at least
#                           emCCR, do not set LL_ORACLE_HOME!
#    fmorshed    10/04/12 - New method getStateDirs.  Knows how to deal with designated collector.
#    jsutton     03/29/12 - CDB/PDB support
#    ckalivar    11/04/11 - ER 12688918: add getCellHostInfoFromDBll for
#                           getting cell information for cell host collections
#    imunusam    08/26/11 - To Fix bug 12914611
#    imunusam    08/19/11 - to add missing declaration of File::stat
#    imunusam    07/05/11 - To move common code of cell discovery and
#                           host-dbmachine association
#    jsutton     04/18/11 - Verify UTF8 content
#    glavash     04/12/11 - remove full path for semd_common.pl rely on
#                           PERL5LIB
#    jsutton     02/25/11 - fix null string declarations
#    aghanti     10/26/10 - Bumping up META_VER
#    aghanti     10/14/10 - Bug 9033775 - Add sub llFileIsComplete() that
#                           checks if .ll*/.ccr/.emrep file has end marker
#    ckalivar    07/01/10 - Fix bug 9868596: added method
#                           getAppDBVersionFromDBll
#    rstorrie    06/22/10 - Addition of getRepVersionFromEMll
#    jsutton     05/21/10 - put reason in log
#    jsutton     04/15/10 - Fix expression to extract DB version from
#                           db_dbNInstanceInfo
#    nmittal     03/19/10 - removing next line while defining emclt files,
#                           otherwise it fails in windows
#    rstorrie    03/16/10 - Remove the simpleXpath call
#    rstorrie    03/09/10 - For bug8984993 add a common routine for calling
#                           emctl
#    ysun        02/10/10 - ignore exadata sql errors
#    aghanti     02/09/10 - Fixing the list of encoding mappings
#    glavash     08/29/09 - XbranchMerge glavash_bug-8850884 from
#                           st_emll_10.3.2
#    glavash     08/28/09 - fix 8850884 - no true/false missing db metrics
#    aghanti     05/26/09 - Created hashmap of characterset encoding names from
#                           database to perl
#    ysun        03/28/09 - add exadata metrics parsing
#    glavash     12/14/07 - add test of column count
#    glavash     12/12/07 - add escape logic
#    glavash     12/11/07 - add delimiter processing
#    jsutton     07/26/07 - Pick up existing shared OH work
#    jsutton     07/25/07 - Shared home work
#    dkapoor     01/31/07 - XbranchMerge dkapoor_bug-5857460 from
#                           st_emll_10.2.5
#    dkapoor     01/31/07 - don't return with -1
#    dkapoor     01/11/07 - compare sid ignore case on windows
#    pparida     09/19/06 - Fix bug 5214705: Return em_errors as separate rows
#                           instead of including them in em_result rows
#    dkapoor     08/16/06 - modularize config collection
#    dkapoor     06/06/06 - db_ll_pkgdefs name change 
#    dkapoor     12/20/05 - check for end of metric 
#    dkapoor     12/11/05 - impl getInitParams 
#    dkapoor     12/05/05 - Creation
# 
use Config;
use File::Spec;
use File::stat();
use Fcntl;
use Tie::File;

require "semd_common.pl";

strict;

#OCM LOG CATEGORY
my $LOG_CATEGORY = "OCM CONFIG COLLECTION: ";

my $ESCAPE_VERSION = "10.2.7.1.0";

# Hashmap of characterset encoding names from Database to Perl
my $DatabaseToPerlEncHashmap = {
    'AL16UTF16' => 'UTF-16BE',            # 16-bit Unicode Transformation Format - Big Endian
    'AL16UTF16LE' => 'UTF-16LE',          # 16-bit Unicode Transformation Format - Little Endian
    'AL32UTF8' => 'UTF-8',                # 8-bit Unicode Transformation Format
    'AR8ISO8859P6' => 'ISO-8859-6',       # ISO/IEC 8859 Latin/Arabic
    'AR8MSWIN1256' => 'WINDOWS-1256',     # Windows Arabic
    'BLT8ISO8859P13' => 'ISO-8859-13',    # ISO/IEC 8859 Latin-7/Baltic Rim
    'BLT8MSWIN1257' => 'WINDOWS-1257',    # Windows Baltic
    'BLT8PC775' => 'CP775',               # MS-DOS Baltic Rim
    'CDN8PC863' => 'CP863',               # MS-DOS Canadian French
    'CEL8ISO8859P14' => 'ISO-8859-14',    # ISO/IEC 8859 Latin-8/Celtic
    'CL8ISO8859P5' => 'ISO-8859-5',       # ISO/IEC 8859 Latin/Cyrillic
    'CL8KOI8R' => 'KOI8-R',               # KOI8 Cyrillic (Russian, Bulgarian)
    'CL8KOI8U' => 'KOI8-U',               # KOI8 Cyrillic (Ukranian)
    'CL8MSWIN1251' => 'WINDOWS-1251',     # Windows Cyrillic
    'EE8ISO8859P2' => 'ISO-8859-2',       # ISO/IEC 8859 Latin-2/Central European
    'EE8MSWIN1250' => 'WINDOWS-1250',     # Windows Latin 2
    'EE8PC852' => 'CP852',                # MS-DOS Slavic (Latin 2)
    'EL8ISO8859P7' => 'ISO-8859-7',       # ISO/IEC 8859 Latin/Greek
    'EL8MSWIN1253' => 'WINDOWS-1253',     # Windows Greek
    'EL8PC869' => 'CP869',                # IBM Modern Greek
    'HZ-GB-2312' => 'HZ-GB-2312',         # Hanzi (7-bit) Simplified Chinese (GB2312)
    'IS8PC861' => 'CP861',                # MS-DOS Icelandic
    'ISO2022-JP' => 'ISO-2022-JP',        # ISO/IEC 2022 - Japanese
    'ISO2022-KR' => 'ISO-2022-KR',        # ISO/IEC 2022 - Korean
    'IW8EBCDIC424' => 'CP424',            # IBM Hebrew
    'IW8EBCDIC424S' => 'CP424',           # IBM Hebrew
    'IW8ISO8859P8' => 'ISO-8859-8',       # ISO/IEC 8859 Latin/Hebrew (Logical Order) [Earlier mapping: ISO-8859-8-I]
    'IW8MSWIN1255' => 'WINDOWS-1255',     # Windows Hebrew
    'IW8PC1507' => 'CP862',               # MS-DOS Hebrew
    'JA16EUC' => 'EUC-JP',                # Extended Unix Code - Japanese
    'JA16EUCTILDE' => 'EUC-JP',           # Extended Unix Code - Japanese
    'JA16EUCYEN' => 'EUC-JP',             # Extended Unix Code - Japanese
    'JA16SJIS' => 'SHIFT_JIS',            # Shift JIS - Japanese
    'JA16SJISTILDE' => 'SHIFT_JIS',       # Shift JIS - Japanese
    'JA16SJISYEN' => 'SHIFT_JIS',         # Shift JIS - Japanese
    'JA16VMS' => 'EUC-JP',                # Extended Unix Code - Japanese
    'KO16KSC5601' => 'EUC-KR',            # Extended Unix Code - Korean
    'KO16MSWIN949' => 'EUC-KR',           # Extended Unix Code - Korean
    'N8PC865' => 'CP865',                 # MS-DOS Nordic
    'NE8ISO8859P10' => 'ISO-8859-10',     # ISO/IEC 8859 Latin-6/Nordic
    'NEE8ISO8859P4' => 'ISO-8859-4',      # ISO/IEC 8859 Latin-4/North European
    'RU8PC855' => 'CP855',                # IBM Cyrillic
    'RU8PC866' => 'CP866',                # MS-DOS Russian
    'SE8ISO8859P3' => 'ISO-8859-3',       # ISO/IEC 8859 Latin-3/South European
    'TH8TISASCII' => 'ISO-8859-11',       # ISO/IEC 8859 Latin/Thai [Earlier mapping: TIS-620]
    'TR8EBCDIC1026' => 'CP1026',          # EBCDIC
    'TR8EBCDIC1026S' => 'CP1026',         # EBCDIC
    'TR8MSWIN1254' => 'WINDOWS-1254',     # Windows Turkish
    'TR8PC857' => 'CP857',                # IBM Turkish
    'UCS2' => 'UTF-16',                   # 16-bit Unicode Transformation Format
    'US7ASCII' => 'US-ASCII',             # US ASCII
    'US8PC437' => 'CP437',                # MS-DOS United States
    'UTF16' => 'UTF-16',                  # 16-bit Unicode Transformation Format
    'UTF8' => 'UTF-8',                    # 8-bit Unicode Transformation Format
    'VN8MSWIN1258' => 'WINDOWS-1258',     # Windows Vietnamese
    'WE8DEC' => 'ISO-8859-1',             # (DEC) Extended ascii [Earlier mapping: DEC-MCS]
    'WE8EBCDIC1047' => 'CP1047',          # IBM Latin 1 [Earlier mapping: IBM1047]
    'WE8EBCDIC1047E' => 'CP1047',         # IBM Latin 1 [Earlier mapping: IBM1047]
    'WE8EBCDIC37' => 'CP37',              # EBCDIC USA, Canada, Australia, New Zealand, Netherlands, Brazil, Portugal
    'WE8EBCDIC37C' => 'CP37',             # EBCDIC USA, Canada, Australia, New Zealand, Netherlands, Brazil, Portugal
    'WE8EBCDIC500' => 'CP500',            # EBCDIC 500V1
    'WE8EBCDIC500C' => 'CP500',           # EBCDIC 500V1
    'WE8ISO8859P1' => 'ISO-8859-1',       # ISO/IEC 8859 Latin-1/Western European
    'WE8ISO8859P15' => 'ISO-8859-15',     # ISO/IEC 8859 Latin-9
    'WE8ISO8859P9' => 'ISO-8859-9',       # ISO/IEC 8859 Latin-5/Turkish
    'WE8MSWIN1252' => 'WINDOWS-1252',     # Windows Latin
    'WE8NCR4970' => 'ISO-8859-1',         # (DEC) Extended ascii [Earlier mapping: DEC-MCS]
    'WE8PC850' => 'CP850',                # MS-DOS Multilingual (Latin 1)
    'WE8PC858' => 'CP850',                # DOS Latin-1 + Euro <<approx>> [Earlier mapping: CP00858]
    'WE8PC860' => 'CP860',                # MS-DOS Portuguese
    'WE8ROMAN8' => 'HP-ROMAN8',           # HP Latin
    'ZHS16CGB231280' => 'GBK',            # Simplified Chinese (GB2312) - Extended
    'ZHS16GBK' => 'GBK',                  # Simplified Chinese (GB2312) - Extended
    'ZHT16BIG5' => 'BIG5-ETEN',           # Traditional Chinese + Extensions [Earlier mapping: BIG5]
    'ZHT16HKSCS' => 'BIG5-HKSCS',         # Traditional Chinese - Hong Kong Supplementary Character Set (Cantonese)
    'ZHT16MSWIN950' => 'CP950'            # Traditional Chinese + Microsoft vendor mappings [Earlier mapping: BIG5]
##
# The following encodings are valid mappings from database NLS charsets, but their perl counterparts are
# either obsoleted, not supported or not implemented (mostly by perl, and in one case, OCM)
# [Retaining them in this table for reference purposes only]
##
#    'AR8EBCDIC420S' => 'CP420',          # EBCDIC Arabic
#    'AR8EBCDICX' => 'CP420',             # EBCDIC Arabic
#    'D7DEC' => 'DIN_66003',              # ISO-IR-21/ISO646-DE German
#    'D7SIEMENS9780X' => 'DIN_66003',     # ISO-IR-21/ISO646-DE German
#    'D8EBCDIC1141' => 'CP01141',         # EBCDIC Austria, Germany
#    'D8EBCDIC273' => 'CP273',            # EBCDIC Austria, Germany
#    'DK7SIEMENS9780X' => 'NS_4551-1',    # ISO-646 Norwegian
#    'DK8EBCDIC1142' => 'CP01142',        # EBCDIC Denmark, Norway
#    'DK8EBCDIC277' => 'IBM277',          # CP277 - Norway, Denmark EBCDIC
#    'E7DEC' => 'ES',                     # ISO646-ES Spanish
#    'EE8EBCDIC870' => 'CP870',           # EBCDIC Latin-2
#    'EE8EBCDIC870C' => 'CP870',          # EBCDIC Latin-2
#    'EE8EBCDIC870S' => 'CP870',          # EBCDIC Latin-2
#    'EL8EBCDIC423R' => 'CP423',          # EBCDIC Greek
#    'EL8PC851' => 'CP851',               # EBCDIC Greek 1
#    'F7DEC' => 'NF_Z_62-010_(1973)',     # ISO-IR-25/ISO646-FR1 French 1
#    'F8EBCDIC1147' => 'CP01147',         # EBCDIC France
#    'F8EBCDIC297' => 'CP297',            # EBCDIC France
#    'I7DEC' => 'IT',                     # ISO646-IT Italian
#    'I7SIEMENS9780X' => 'IT',            # ISO646-IT Italian
#    'I8EBCDIC1144' => 'CP01144',         # EBCDIC Italy
#    'I8EBCDIC280' => 'CP280',            # EBCDIC Italy
#    'ISO2022-CN' => 'ISO-2022-CN',       # RFC1922 - Chinese CNS-11643-1 and -2
#    'N7SIEMENS9780X' => 'NS_4551-1',     # ISO-646 Norwegian
#    'S7SIEMENS9780X' => 'SEN_850200_C',  # ISO-IR-11/ISO646-SE2 Extended Swedish
#    'S8EBCDIC1143' => 'CP01143',         # EBCDIC Finland, Sweden
#    'S8EBCDIC278' => 'CP278',            # EBCDIC Finland, Sweden
#    'SF7ASCII' => 'SEN_850200_B',        # ISO-IR-10/ISO646-SE Basic Swedish
#    'WE8EBCDIC1140' => 'CP01140',        # EBCDIC USA, Canada, Australia, New Zealand, Netherlands
#    'WE8EBCDIC1140C' => 'CP01140',       # EBCDIC USA, Canada, Australia, New Zealand, Netherlands
#    'WE8EBCDIC1145' => 'CP01145',        # EBCDIC Spanish
#    'WE8EBCDIC1146' => 'CP01146',        # EBCDIC United Kingdom
#    'WE8EBCDIC1148' => 'CP01148',        # EBCDIC Latin-1
#    'WE8EBCDIC1148C' => 'CP01148',       # EBCDIC Latin-1
#    'WE8EBCDIC284' => 'CP284',           # EBCDIC Spanish
#    'WE8EBCDIC285' => 'CP285',           # EBCDIC United Kingdom
#    'WE8EBCDIC871' => 'CP871',           # EBCDIC Iceland
#    'WE8EBCDIC924' => 'CP00924',         # EBCDIC Latin-9
#    'WE8MACROMAN8' => 'MACINTOSH',       # No Mac support in OCM
#    'WE8MACROMAN8S' => 'MACINTOSH',      # No Mac support in OCM
#    'YUG7ASCII' => 'JUS_I.B1.002',       # ISO-IR-141/ISO646-YU Yugoslavian
#    'ZHS32GB18030' => 'GB18030'          # GBK + chinese traditional characters (Needs Encode::HanExtra from CPAN)
##
};

#For a given LL file, check if it has been completely written out
#Args: LL File
#Returns: "true" if ll file is complete, "false" otherwise

sub getStateDirs
{  
    my ($emdRoot,$hostName) = @_;
	my @stateDirs = ();
    if (defined $ENV{ORACLE_HOME} && -d  $ENV{ORACLE_HOME})   
    {
        my $oracleHome = $ENV{ORACLE_HOME};
        my $ccrConfigHome = $ENV{CCR_CONFIG_HOME};
        
        if (!defined $ENV{CCR_DESIGNATED_COLLECTOR_IS_COLLECTING_OTHER_HOMES})
        {
            EMD_PERL_DEBUG("getStateDirs called - NOT collecting other homes.");     
        	# Collector is not the designated (central or anonymous).  Hence, ccrConfigHome
        	# is set based on ORACLE_HOME and we can use it directly.
            push (@stateDirs, File::Spec->catdir($ccrConfigHome,"state"));
            push (@stateDirs, File::Spec->catdir($emdRoot,"state")) if ($emdRoot ne $ccrConfigHome);           
        }
        else
        {
            EMD_PERL_DEBUG("getStateDirs called - Collecting other homes.");     
            # Collector is the designated central or anonymous collector.  Hence, ccrConfigHome
            # is set based on the collector home (ccr home), not the target oracle home.  So, 
            # we cannot use it. But CCR_CENTRAL_PROPERTIES_CCH_FOR_OH is defined to point to
            # the CCR_CONFIG_HOME of the collector in the "other" OH for which designated collector
            # is collecting.  
            # 
            # Note:  Because designated collector never collects a home that has a collector whose
            # config home cannot be determined (that is, it was configured with ORACLE_CONFIG_HOME 
            # set and its cch is not recorded in central properties file), this code does not have 
            # to worry about that case!
        	my $cch = $ENV{CCR_CENTRAL_PROPERTIES_CCH_FOR_COLLECTOR_IN_OH};
        	if (-d  $cch)
        	{  
                push (@stateDirs, File::Spec->catdir($cch,"state"));
                my $ccrInOracleHome = File::Spec->catdir($oracleHome,"ccr");
                push (@stateDirs, File::Spec->catdir($ccrInOracleHome,"state")) if ($ccrInOracleHome ne $cch);  
            }         
        }
    }
    else
    {
        EMD_PERL_WARN("$LOG_CATEGORY ORACLE_HOME is not defined in getConfigDirs!");    	
    }
    return @stateDirs;
}

sub llFileIsComplete
{
  my ($configFile) = @_;
  my $lastLine = "";
  my @fileLines;

  if (!-e $configFile)
  {
    EMD_PERL_WARN("$LOG_CATEGORY File $configFile does not exist");
    return "false";
  }

  #Check if META_VER is > 10.3.5, return true for older versions
  my $metaVersion = getConfigProperty($configFile,"META_VER");
  if (compare_versions($metaVersion, "10.3.5.0.0") == -1)
  {
    EMD_PERL_WARN("$LOG_CATEGORY File $configFile was generated with OCM older than 10.3.5.0.0, ignoring EOF check.");
    return "true";
  }

  tie @fileLines, 'Tie::File', $configFile, mode => O_RDONLY or return "false";
  $lastLine = $fileLines[-1];
  chomp($lastLine);
  untie @fileLines;

  #check for end marker in the ll file
  if ( $lastLine eq "_CCR_EOF_")
  {
    return "true";
  }
  else
  {
    EMD_PERL_WARN("$LOG_CATEGORY File $configFile is incomplete. If you see this message frequently, there is probably a clash between the scheduled collection by the OCM collector and the collection job in the database. Please try changing the scheduled collection time for the OCM collector");
    return "false";
  }
}

#For a given LL file, get the db charset and map it to
#the corresponding perl encoding
#Args: LL File
#Returns: File open mode
sub getLLFileOpenMode
{
  my ($configFile) = @_;
  my $llOpenMode = "<"; #default mode if no encoding is determined
  my $llDbCharSet = getConfigProperty($configFile,"NLS_CHARACTERSET");
  if ($llDbCharSet ne "")
  {
    # Remove ^M chars if any
    $llDbCharSet =~ tr/\r\n/\n/;
    chomp($llDbCharSet);

    # Apparently UTF8 means slightly different things to different people. Trust but verify? 
    # Slurp the contents of the file, try a UTF-8 decode, and fall back to WE8ISO8859P1 (Latin1) on error.
    if ($llDbCharSet =~ "UTF8")
    {
      my $cfgText = "";
      {
        local( $/ ) ;
        open( my $fh, $configFile ) or die "sudden flaming death\n";
        $cfgText = <$fh>;
      }
      eval { $utf8 = decode( "UTF-8", $cfgText, Encode::FB_CROAK ) };
      if ( $@ ) 
      { # input was not utf8 (i.e. eval above CROAKed)
        $llDbCharSet = "WE8ISO8859P1";
      }
    }

    my $llDbPerlEnc = $DatabaseToPerlEncHashmap->{$llDbCharSet};
    if ($llDbPerlEnc ne "") 
    {
      $llOpenMode = "<:encoding($llDbPerlEnc)";
    }
  }
  return $llOpenMode;
}

# Set all output to be in UTF-8
binmode (STDOUT,":encoding(UTF-8)");
binmode (STDERR,":encoding(UTF-8)");

#Get database ll files
sub getLLFiles
{
  my ($emdRoot,$hostName) = @_;
  my @configDirs = getStateDirs($emdRoot,$hostName);
  my $configDir = @configDirs[0];
  EMD_PERL_DEBUG("$LOG_CATEGORY emdRoot=[$emdRoot] hostName=[$hostName]");
  my $DIR;
  opendir(DIR,$configDir);
  my @files = grep {/\.ll$/ }  readdir(DIR);
  closedir (DIR);

  my @confileFiles ;
  foreach my $file (@files)
  {
    my $fullFilePath = File::Spec->catdir($configDir,$file);
    EMD_PERL_DEBUG("$LOG_CATEGORY Days past since last modification on $fullFilePath=". (-M $fullFilePath));
    if( -M $fullFilePath >= 30)
    {
      #skip files more than 30 days old
      next;
    }
    push(@confileFiles,$fullFilePath);
  }
  return @confileFiles;
}

#Returns 1 if the file passed is a Database LL file,
#otherwise 0 is returned.
sub isDB_LLFile
{
  #Return 0 if the .ll file is an apps one
  return 0 if( basename($_[0]) =~ /(.*)-apps_(.*).ll$/ );
  #Otherwise, 1.
  return 1;
}

#Returns 1 if the file passed is a Database RAC LL file,
#otherwise 0 is returned.
sub isDB_RAC_LLFile
{
  #The file is RAC if its a DB file.
  if(isDB_LLFile($_[0]))
  {
    #And it ends with '-RAC'
    return 1 if( basename($_[0]) =~ /(.*)-RAC.ll$/ );
  }
  return 0;
}

#Get DB_NAME from the .ll file
sub getDBName
{
  my $db_name;
  my $configFile = $1 if (basename($_[0]) =~ /(.*).ll$/);
  EMD_PERL_DEBUG( "In getDBName (@_) configFile=$configFile");

  #The db name is in the following form in the ll file:
  #Non DB file and <db_name>-apps_.*.ll
  #RAC DB file and <db_name>-RAC.ll
  #DB file and <db_name>.ll
  if(!isDB_LLFile($_[0]) && $configFile =~ /(.*)-apps_(.*)$/)
  {
    $db_name = $1;
    EMD_PERL_DEBUG( "Apps db name db_name=$db_name");
  }
  elsif(isDB_LLFile($_[0])) 
  {
    if(isDB_RAC_LLFile($_[0]) && $configFile =~ /(.*)-RAC$/)
    {
      $db_name = $1;
      EMD_PERL_DEBUG( "RAC db name db_name=$db_name");
    }
    else
    {
      $db_name = $configFile;
      EMD_PERL_DEBUG( "Single instance db name db_name=$db_name");
    }
  }
  else
  {
    EMD_PERL_DEBUG("db NOT FOUND");
  }
  return $db_name;
}

#Get the SID with host name from a CCR config file
sub getSidsFromDBll
{
  my ($configFile) = @_;
  my $metric = "db_dbNInstanceInfo";
  my %sidTable;
  my $sid;
  my $found = 0;
  EMD_PERL_DEBUG("In getSidsFromDBll");

  my $dbScriptVersion = getConfigProperty($configFile,"META_VER");

  #Get the db charset and map it to corresponding perl encoding
  my $llOpenMode = getLLFileOpenMode($configFile);

  if(open(CONFIG_FILE_READER, $llOpenMode, $configFile))
  {
    my $line;
    my $endMarker;
    while($line = <CONFIG_FILE_READER>)
    {
      chomp($line);
      if($found)
      {
        if ($line =~ /^$endMarker/)
        {
          EMD_PERL_DEBUG("End marker found, find the next sid");
          $found = 0;
          $sid = "";
          #Go find the next one
          next; # found the metric end, check next
        }
        # skip lines with oddball/nonprintables?
        if ($line =~ /[^\x09\x20-\x7E]/)
        {
          EMD_PERL_DEBUG("Ignoring Result containing non printables :em_result=$line");
          next;
        }
        elsif($line !~ /^em_result=/)
        {
          EMD_PERL_DEBUG("Ignoring Result the result is not in new line:$line");
          next;
        }
        if (trim($sid) ne "")
        {
          #originally looked for 5 columns
          #  if ($line =~ /^em_result=(.*?)\|(.*?)\|(.*?)\|(.*?)\|(.*)/)
          my @col_values = parse_db_metric($line,$dbScriptVersion);
          if (scalar(@col_values) >= 5 )
          { 
            # host is 4th column 0 based index use 3 
            $sidTable{$sid} = $col_values[3]; 
            EMD_PERL_DEBUG("Found SID=$sid Host=$col_values[3]");
            $found = 0;
            #Found one, go find the next one
          }
        }
      }
      else
      {
        if ($line =~ /^$metric:(.*):Begin/)
        {
          # Found the start 
          if( $1 ne "")
          {
            #Found the metric start, 
            $found = 1;
            $sid = $1;
            $endMarker = "$metric:$1:End";
            EMD_PERL_DEBUG("Found SID=$sid");
            next;
          }
        }
      }
    }
  }
  close(CONFIG_FILE_READER);
  EMD_PERL_DEBUG("Out getSidsFromDBll");
  return %sidTable;
}

sub getConfigProperty
{
  my ($configFile,$property) = @_;
  my $err = '';
  my $value = '';
  my $CONFIG_FILE_READER;

  if(open(CONFIG_FILE_READER, $configFile))
  {
    my $line;
    while($line = <CONFIG_FILE_READER>)
    {
      chomp($line);
      if ($line =~ /^$property\s*=\s*(.+)/)
      {
        $value = $1;
        EMD_PERL_INFO("Found $property=$value");
        last; # found the metric end, exit the loop
      }
    }
  }
  else
  {
    $err = "Could not open the configuration file=\"$configFile\":$!\n";
    EMD_PERL_INFO($err);
  }
  close(CONFIG_FILE_READER);
  return ($err,$value);
}

sub getInstalledDBPL_SQLVersion
{
  my ($configFile) = @_;

  my $dbPL_SQLPackageFile = File::Spec->catdir($ENV{CCR_HOME},"admin","scripts","ocmdbd.sql");
  my $err = '';
  my $value = '';
  my $PLSQL_FILE_READER;

  if(open(PLSQL_FILE_READER, $dbPL_SQLPackageFile))
  {
    my $line;
    while($line = <PLSQL_FILE_READER>)
    {
      chomp($line);
      if ($line =~ /^ORACLE_DATABASE_META_VER (.*):=\s*\'(.+)\'\s*;/)
      {
        $value = $2;
        EMD_PERL_INFO("Found $property=$value");
        last; # found the metric end, exit the loop
      }
    }
  }
  else
  {
    $err = "Could not open the pl.sql file=\"$dbPL_SQLPackageFile\":$!\n";
    EMD_PERL_INFO($err);
  }
  close(PLSQL_FILE_READER);
  return ($err,$value);
}
#strips leading and trailing spaces and returns the string
sub trim 
{
  my $origStr = $_[0];
  #Strip trailing and leading
  $origStr =~ s/^\s*|\s*$//g;
  return $origStr;
}

sub getInstalledEBSDBPL_SQLVersion
{
  my ($configFile) = @_;

  my $sqlFile = File::Spec->catdir($ENV{CCR_HOME}, "admin", "scripts", "ebs_ll_pkgbodys.sql"); 
  my $err = '';
  my $value = '';
  my $INFILE;
  if (open(INFILE, $sqlFile)) 
  {
    my $line;
    while ($line = <INFILE>) 
    {
      chomp($line);
      if ($line =~ /^ORACLE_EBS_META_VER (.*):=\s*\'(.+)\'\s*;/) 
      {
        $value = $2;
        EMD_PERL_DEBUG("$CATEGORY Installed $sqlFile version= $value");
        last;
      }
    }
  }
  else
  {
    $err = "$CATEGORY Could not open PL/SQL file \"$sqlFile\":$!";
  }
  close(INFILE);
  return ($err, $value);
}

#Get the initialization parameter from a CCR config file
#Params:
#    configFile -- db configuration file to get init param value from
#    prams -- an array of parameters for which values are needed
#Output:
#    A hashtable filled with values of the uppercased parameters, otherwise '-' if not found in the config file. 
sub getInitParams 
{
  my ($configFile,@paramsArray) = @_;
  my $metric = "db_init_params";
  my $found = 0;
  EMD_PERL_DEBUG("In getInitParams: $configFile");
  my %params;

  my $dbScriptVersion = getConfigProperty($configFile,"META_VER");

  foreach $param ( @paramsArray)
  {
    EMD_PERL_DEBUG("param =$param ");
    #initialize values with "-"
    $params{uc $param} = '-';
  }

  #Get the db charset and map it to corresponding perl encoding
  my $llOpenMode = getLLFileOpenMode($configFile);

  if(open(CONFIG_FILE_READER, $llOpenMode, $configFile))
  {
    my $line;
    my $endMarker;

    #escapes_in_data will be >= 0 if there are escapes
    while($line = <CONFIG_FILE_READER>)
    {
      chomp($line);
      if($found)
      {
        if ($line =~ /^$endMarker/)
        {
          last; # found the metric end, exit the loop
        }
        elsif ($line =~ /[^\x09\x20-\x7E]/)
        { # skip lines with oddball/nonprintables?
          EMD_PERL_DEBUG("Ignoring Result containing non printables :em_result=$line");
          next;
        }
        elsif($line !~ /^em_result=/)
        {
          EMD_PERL_DEBUG("Ignoring Result the result is not in new line:$line");
          next;
        }
        my @col_values = parse_db_metric($line,$dbScriptVersion);
        #originally looked for at least 3 columns
        #if ($line =~ /^em_result=(.*?)\|(.*?)\|(.*?)/)
        if (scalar(@col_values) >= 3)
        {
          $param = uc trim($col_values[0]);
          if ($params{$param} eq "-")
          {
            $params{$param} = $col_values[1];
            EMD_PERL_DEBUG("Found Init param $param=$col_values[1]");
          }
        }
      }
      elsif ($line =~ /^$metric(.*):Begin/)
      {
        #Found the metric start, 
        $found = 1;
        $endMarker = "$metric$1:End";
        EMD_PERL_DEBUG("Found Start -> $1");
      }
    }
  }
  close(CONFIG_FILE_READER);
  EMD_PERL_DEBUG("Out getInitParams");
  return %params;
}

#Parses the ll file for a config metric.
#It prints the result for configuration collection.
sub printConfigMetricResult
{
  my ($configFile,$metric,$columnCount,$sid) = @_;
  handleConfigMetricResult($configFile,$metric,$columnCount,$sid,\&printResult);
}

#Parses the ll file for a cell metric.
#It prints the result for cell metric collection.
sub printCellMetricResult
{
  my ($configFile,$metric,$columnCount,$ip) = @_;
  handleCellMetricResult($configFile,$metric,$columnCount,$ip,\&printResult);
}

#Parses the ll file for a PDB config metric.
#It prints the result for configuration collection.
sub printPdbConfigMetricResult
{
  my ($configFile,$metric,$columnCount,$sid,$pdbName) = @_;
  handlePdbConfigMetricResult($configFile,$metric,$columnCount,$sid,$pdbName,\&printResult);
}

#Parses the ll file for a config metric.
#Client passes another function reference to handle output one 
#result at a time.
sub handleConfigMetricResult
{
    my ($configFile,$metric,$columnCount,$sid, $printFunc) = @_;
    handleBaseConfigMetricResult($configFile,$metric,$columnCount,$sid, $printFunc, "false", "");
}

sub handleCellMetricResult
{
    my ($configFile,$metric,$columnCount,$ip, $printFunc) = @_;
    handleBaseConfigMetricResult($configFile,$metric,$columnCount,$ip, $printFunc, "true", "");
}

sub handlePdbConfigMetricResult
{
    my ($configFile, $metric, $columnCount, $sid, $pdbName, $printFunc) = @_;
    handleBaseConfigMetricResult($configFile, $metric, $columnCount, $sid, $printFunc, "false", $pdbName);
}

sub handleBaseConfigMetricResult
{
  my ($configFile, $metric, $columnCount, $sid, $printFunc, $isCell, $pdbName) = @_;

  if(!defined $printFunc )
  {
    $printFunc = \&printResult;
  }
  EMD_PERL_INFO("In getConfigMetricResult for oh=[$oracle_home], configfile=[$configFile] and metric=[$metric]; pdbName=[$pdbName]");

  if (!defined $configFile )
  {
    EMD_PERL_ERROR("The OCM configuration file not passed.");
    #No need to print to std error
    #as there may be cases config file is not present and this method is called.
    #print STDERR "em_error=The OCM configuration file not passed.\n";
    exit (-1);
  }
  if( !(-f $configFile))
  {
    EMD_PERL_INFO("The OCM configuration file \"$configFile\" not found.");
    #No need to print to std error
    #as there may be cases config file is not present and this method is called.
    #print STDERR "em_error=The OCM configuration file \"$configFile\" not found.\n";
    exit ;
  }
  my $sid_qualifier = "";

  if ( $isCell eq "true" )
  {
      $sid_qualifier = "&$sid" if  (defined $sid && $sid ne "");
  }
  else
  {
      $sid_qualifier = ":$sid" if  (defined $sid && $sid ne "");
  }

  EMD_PERL_INFO("In getConfigMetricResult for SID=[$sid]");

  my $dbScriptVersion = getConfigProperty($configFile,"META_VER");

  #Get the db charset and map it to corresponding perl encoding
  my $llOpenMode = getLLFileOpenMode($configFile);
  my $found = 0;
  if(open(CONFIG_FILE_READER, $llOpenMode, $configFile))
  {
    my $line;
    my $endMarker;
    my $lastResult = '';

    while($line = <CONFIG_FILE_READER>)
    {
      chomp($line);
      if($found)
      {
        if ($line =~ /^$endMarker/)
        {
          $lastResult = escape_escape($lastResult,$dbScriptVersion);
          if ($pdbName ne "")
          {
            # match first column of result to PDB name, strip from output
            if ($lastResult =~ /^em_result=([^\|]+)\|(.*)/)
            {
              if ($pdbName eq $1)
              {
                 $lastResult = "em_result=$2";
                 $printFunc->($lastResult);
              }
            }
          }
          else
          {
            $printFunc->($lastResult);
          }
          last; # found the metric end, exit the loop
        }
        else
        {
          if(($line =~ /^em_result=/) || ($line =~ /^em_error=/))
          {
            #New result starts here, print the last result
            # and re-initialize the last result with this
            # line
            $lastResult = escape_escape($lastResult,$dbScriptVersion);
            if ($pdbName ne "")
            {
              # match first column of result to PDB name, strip from output
              if ($lastResult =~ /^em_result=([^\|]+)\|(.*)/)
              {
                if ($pdbName eq $1)
                {
                  $lastResult = "em_result=$2";
                  $printFunc->($lastResult);
                }
              }
            }
            else
            {
              $printFunc->($lastResult);
            }
            $lastResult=$line;
          }
          else
          {
            #Append the last result with
            #this line using a space character.
            $lastResult= $lastResult . " " . $line;
          }
          EMD_PERL_INFO("Processed:$line");
        }
      }
      else
      {
        if ($line =~ /^$metric(.*):Begin/)
        {
          # Found the start if (in precedence) 
          # - metric has a matching sid qualifier
          # - sid qualifier is empty  or 
          # - metric does not have sid qualifier or 
          my $metricSid = $1;
          if(get_osType() eq 'WIN')
          {
            #On windows, there can be case mismatch
            #between the passed sid and the one obtained from .ll file.
            #On windows, sid is case insensitive. Convert
            #to upper case before comparing.
            $metricSid = uc ($metricSid);
            $sid_qualifier = uc ($sid_qualifier);
          }
          if( $metricSid eq $sid_qualifier || 
              $sid_qualifier eq "" ||
              $metricSid eq "")
          {
            #Found the metric start, 
            $found = 1;
            $endMarker = "$metric$1:End";
            next;
          }
        }
      }
    }
  }
  else
  {
    EMD_PERL_ERROR("Could not open the config file");
    #No need to print to std error
    #as there may be cases config file is not present and this method is called.
    #print STDERR "em_error=Could not open the configuration file=\"$configFile\":$!\n";
    exit (-1);
  }
}

#prints the result
sub printResult
{
  my ($result) = @_;

  if(length($result) != 0)
  {
    print "$result\n";
    EMD_PERL_INFO("Printed:$result");
  }
}
   
my $db_version = '';

#Get the db version
sub getDBVersionFromDBll
{
    my ($configFile) = @_;
    $db_version = '';
    handleConfigMetricResult($configFile,"db_dbNInstanceInfo",
       15,'',\&handleInstanceInfo);
    my $ret_version = $db_version;
    $db_version = '';
    return  $ret_version;
}

#Handle the instance information metric
#to get the instance version number
sub handleInstanceInfo
{
  my ($result) = @_;
  EMD_PERL_DEBUG("In handleInstanceInfo: [$result]");
  if ($result =~ /em_result=([^\|]*\|){15}([^\|]*)/)
  {
     $db_version = $2;
     EMD_PERL_DEBUG("Found version=$db_version");
  }
}

my $appdbVersion = '';

#Get the app db version
sub getAppDBVersionFromDBll
{
    my ($configFile) = @_;
    #unset appdbVersin before parseAppDBVersion being called
    $appdbVersion = '';
    handleConfigMetricResult($configFile,"apps_summary",
                             5,'',\&parseAppDBVersion);
    my $ret_version = $appdbVersion;
    $appdbVersion='';
    return  $ret_version;
}

#Parse result of a metric in a result metric for appdbVersion
#stores the output version in appdbVersion, which should be set to '' before calling this method
#Sample apps_summary metric output
#apps_summary:Begin
#em_result=VISION|12.0.0|Y||Y
#apps_summary:End
sub parseAppDBVersion
{
  my ($result) = @_;
  EMD_PERL_DEBUG("In parseAppDBVersion: [$result]");
  #look for second column
  if ($result =~ /em_result=([^\|]*)\|([^\|]*)\|(.*)/)
  {
     $appdbVersion = $2;
     EMD_PERL_DEBUG("Found app db version=$appdbVersion");
  }

}

my $em_version = '';

#Get the em rep version
sub getEMVersionFromEMll
{
    my ($configFile) = @_;
    $em_version = '';
    handleConfigMetricResult($configFile,"emrep_config",
       2,'',\&handleEMInfo);
    my $ret_version = $em_version;
    $em_version = '';
    return  $ret_version;
}

#Get the EM version from the output
sub handleEMInfo
{
  my ($result) = @_;
  EMD_PERL_DEBUG("In handleEMInfo: [$result]");
  if ($result =~ /em_result=([^\|]*\|)([^\|]*)/)
  {
     $em_version = $2;
     EMD_PERL_DEBUG("Found version=$em_version");
  }
}

#compare versions
# compares version strings of the format
# "10.2.7.0.0" the key is the four '.''s
# returns 0 if the are equal
#         -1 if ver 1 is less than ver_2
#         1 if ver_1 is gtr than ver_2
sub compare_versions
{
  my ($ver_1,$ver_2) = @_;

  #break up input strings into their 5 parts
  my @val_1 = split(/\./,$ver_1);
  my @val_2 = split(/\./,$ver_2);

  my $i = 0;
  my $status;
  while ($i < 5)
  {
     if ($val_1[$i] == $val_2[$i])
     {
      $status = 0;
     }
     else
     {
        if ($val_1[$i] < $val_2[$i])
        {
          $status = -1;
          last;
        }
        else
        {
          $status = 1;
          last;
        }
     }
    $i++;
  }
  return $status;
}
#parse_db_metric
#
# given a db metric in the form of em_result=fi|fie|fo|fum
# return an array containing fi fie fo fum
#
# handles esacpe character of #
# this will allow for metric with a
# 2 columns 1=  bar|bar|bar and barnone to be logged as
# em_result=bar#|bar#|bar|barnone
# similarly the dbs script escapes the # as ##.
#
sub parse_db_metric
{
  my ($line,$version) = @_;
  my @values;
  my $esc_loc;
  my $column = 0;
  my $sep_loc = 0;

  # escape escape characters if need be
  $line = escape_escape($line,$version);

  # remove em_result=
  my $remnant = substr($line,index($line,'=')+1);

  $esc_loc = index($remnant,'#');
  # if no escape characters just split it
  if ($esc_loc == -1)
  {
      @values = split(/\|/,$remnant);
      return @values;
  }

  while ($sep_loc != -1)
  {
     $sep_loc = index($remnant,'|');
     $esc_loc = index($remnant,'#');
     # reached the end, last column has no delimiter
     if ($sep_loc == -1)
     {
      #if we have no more seperators but there are escapes, substitute
      #them all at once
      if ($esc_loc != -1)
      {
        $remnant =~ s/##/#/g;
      } 
      @values[$column] = $remnant;
      last;
     }
    
     #ignore the escape if it is past the seperator
     if ($esc_loc > $sep_loc)
     {
       $esc_loc = -1;
     } 

     if ($esc_loc != -1)
     {
       if ($sep_loc == $esc_loc+1)
       {
         #escape is escaping the separator
         if ($esc_loc != 0)
         {
            # append up to the escape and then a '|'
            $values[$column] = $values[$column] . substr($remnant,0,$esc_loc) . "|";
         }
         else
         {
            # first character is escape followed by sep append only sep
            $values[$column] = $values[$column] . "|";
         }
         # reset remnant, continue processing this column
         $remnant = substr($remnant,$sep_loc+1);
       }
       else  # this escape is escaping another esacpe
       {
         #copy the first escape, skip the second
         $values[$column] = $values[$column] . substr($remnant,0,$esc_loc+1);
         $remnant = substr($remnant,$esc_loc+2);
       }
    }
    else
    {
      # we have a seperator but no escape
      #no escape append this remnant to current column
      if ($sep_loc != 0)
      {
        $values[$column] = $values[$column] . substr($remnant,0,$sep_loc);
      }
      # reset remnant to rest of string
      $remnant = substr($remnant,$sep_loc+1);
      # bump column end of this column
      $column++;
    }
  }
return @values;
}
#
# if version is less than 10.2.7.1.0 replace
# all # with ## in line
#
sub escape_escape
{
  my ($line,$version) = @_;
  my $version_comp = compare_versions($version,$ESCAPE_VERSION);

  if ($version_comp == -1)
  {
    $line =~ s/#/##/g;
  }
  return $line;
}
#
# escape_pipe_and_pound
# take a string and turn # into ## and | into #|
#
sub escape_pipe_and_pound
{
   my ($value) = @_;

   $value =~ s/#/##/g;
   $value =~ s/\|/#\|/g;

   return $value;
}

###################################Exadata function#############
my @celltargets;
my $cellHostInfo;

#Parse result em_result=ip|cellname|id
sub handleCellTargetInfo
{
    my ($result) = @_;
    
    if ($result ne "")
    {
        EMD_PERL_DEBUG("In ************* handleCellTargetInfo***********: [$result]");
        if ($result !~ /em_error=/)
        {
            $result = substr($result,index($result,'=')+1); #strip off em_result=
            EMD_PERL_INFO("$result");
            push(@celltargets, $result);
        }
        EMD_PERL_DEBUG("End ************* handleCellTargetInfo***********");
    }
}

#Get the cell ip, name, id mapping from cell_list 
sub getCellTargetsInfoFromDBll
{
    my ($configFile) = @_;
    EMD_PERL_DEBUG("In ++++++++++++++++++++++ getCellTargetsInfoFromDBll +++++++++: [$configFile]");
    handleConfigMetricResult($configFile,"cell_list",
                             3,'',\&handleCellTargetInfo);
    EMD_PERL_DEBUG("In ++++++++++++++++++++++ End getCellTargetsInfoFromDBll +++++++++");
    return  @celltargets;
}

#Get the cell host information
sub getCellHostInfoFromDBll
{
    my ($configFile,$cellIp) = @_;
    EMD_PERL_DEBUG("In ++++++++++++++++++++++ getCellHostInfoFromDBll +++++++++: [$configFile]");
    handleCellMetricResult($configFile,"cell_config",
                             16,$cellIp,\&handleCellHostInfo);
    EMD_PERL_DEBUG("In ++++++++++++++++++++++ End getCellHostInfoFromDBll +++++++++");
    return  $cellHostInfo;
}
#Parse result em_result=12columns..|kernelVersion|rest of columns..
sub handleCellHostInfo
{
    my ($result) = @_;

    if ($result ne "")
    {
        EMD_PERL_DEBUG("In ************* handleCellHosInfo***********: [$result]");
        if ($result !~ /em_error=/)
        {
            $result = substr($result,index($result,'=')+1); #strip off em_result=
            EMD_PERL_INFO("$result");
            my @infos = split('\|',$result);
            $cellHostInfo= $infos[12];
        }
        EMD_PERL_DEBUG("End ************* handleCellHostInfo***********: [$cellHostInfo]");
    }
}

sub getAllLLfiles()
{
  my ($emdRoot,$hostName) = @_;
  if (defined $ENV{ORACLE_HOME} && -d  $ENV{ORACLE_HOME})
  {
    my ($perlHostName,$perlShortHostName,$perlIpAddr);    
    $perlHostName = lc(hostname());    
    ($perlShortHostName) = split /\./, $perlHostName;    
    $perlIpAddr = join(".",unpack("C4",(gethostbyname($perlHostName))[4]));

    my %allLLFiles;
    my %allCCRFiles;
    my @configDirs = getStateDirs($emdRoot,$hostName);
  
    foreach $configDir (@configDirs)
    {   
        EMD_PERL_DEBUG("Checking $configDir for ll/ccr files");
        opendir(DIR,$configDir);
        my @files = grep {/\.ll$|\.ccr$/ }
        readdir(DIR);
        foreach my $file (@files)
        {
            my $fullFilePath = File::Spec->catdir($configDir,$file);
            EMD_PERL_INFO("$LOG_CATEGORY Days past since last modification on $fullFilePath=". (-M $fullFilePath));
            if( -M $fullFilePath >= 30)
            {
                unlink($fullFilePath);
                EMD_PERL_INFO("$LOG_CATEGORY Deleted 30 day old file=$fullFilePath");
                next;
            }
        
            #
            # Check if the file has been completely written out (has an end marker)      
            # If it is incomplete, print a warning and ignore the file      
            #      
            next if (llFileIsComplete($fullFilePath) ne "true");
        
            #                   
            # split into LL and CCR files; where there are competing variants use the newer one.      
            #      
            if ($file =~ /(.*).ll$/)
            {
                $allLLFiles{uc($1)} = $fullFilePath;
            }
            if ($file =~ /(.*).ccr$/)
            {
                my $ccrFileHost = getConfigProperty($fullFilePath,"HOST_NAME");
                my ($ccrFileShortHost) = split /\./, $ccrFileHost;
                # verify the file was generated on this host
                EMD_PERL_INFO("$LOG_CATEGORY DB HostName: $ccrFileHost, $ccrFileShortHost; Perl HostName: $perlHostName, $perlShortHostName");
                if (($perlHostName ne $ccrFileHost) &&  ($perlShortHostName ne $ccrFileShortHost))
                {
                    # skip to the next file
                    EMD_PERL_INFO("$LOG_CATEGORY Skipping $file, no match across host names");
                    next;
                }
                $allCCRFiles{uc($1)} = $fullFilePath;
            }
        }
        closedir (DIR);
    }
   
    for my $llFile (keys %allLLFiles)
    {
        if (defined $allCCRFiles{$llFile})
        {
          # get full file paths and check mod dates; newer one wins
          my $stLL  = File::stat::stat($allLLFiles{$llFile});
          my $stCCR = File::stat::stat($allCCRFiles{$llFile});
          if ($stLL->mtime >= $stCCR->mtime)
          { delete($allCCRFiles{$llFile}); }
          else
          { delete($allLLFiles{$llFile}); }
        }
    }
    # merge the hashes into one; clean out the 'CCR files' hash
    @allLLFiles{keys %allCCRFiles} = values %allCCRFiles;
    %allCCRFiles = ();
    return (%allLLFiles);
  }
  #end of If
}

## This subroutine return common prefix part of two given string
sub findCommonPrefix
{  
        ($str1, $str2) = @_;
        $len1 = length $str1;
        $len2 = length $str2;
   
        @char1 = ( split(//, $str1));
        @char2 = ( split(//, $str2));
        $len = 0;
        $minlen = $len1 -1;        if($minlen gt $len2) { $minlen = $len2 -1; }   
        while( ($len <= $minlen) && ($char1[$len] eq $char2[$len]) )
        {
            $len = $len +1 ;
        }
     
        if($len gt 0)
        {
           $ret = substr($str1, 0, $len);
           return ($ret);
        }
        else
        {
           return "";
        }
}

###################################END of Exadata###############

# pdb
my @pdbtargets;

#Parse result em_result=name|con_uid|service_name
sub handlePDBTargetInfo
{
    my ($result) = @_;
    
    if ($result ne "")
    {
        EMD_PERL_DEBUG("In ************* handlePDBTargetInfo***********: [$result]");
        if ($result !~ /em_error=/)
        {
            $result = substr($result,index($result,'=')+1); #strip off em_result=
            EMD_PERL_INFO("$result");
            push(@pdbtargets, $result);
        }
        EMD_PERL_DEBUG("End ************* handlePDBTargetInfo***********");
    }
}

#Get the PDB info from oracle_pdb_list 
sub getPDBTargetsInfoFromDBll
{
    my ($configFile) = @_;
    EMD_PERL_DEBUG("In ++++++++++++++++++++++ getPDBTargetsInfoFromDBll +++++++++: [$configFile]");
    handleConfigMetricResult($configFile,"oracle_pdb_targets",
                             3,'',\&handlePDBTargetInfo);
    EMD_PERL_DEBUG("In ++++++++++++++++++++++ End getPDBTargetsInfoFromDBll +++++++++");
    return  @pdbtargets;
}


# pdb

# Routine for executing emctl, on Win32 this involves calling emctl.pl #####
# The routine is currently used by discvoery for getemhome ######
# and emd configuration for getversion ######

sub callEmctl
{
    my ($emctlParam) = @_;

    # Get the CONSOLE_CFG property from emctl, if emctl does not exist exit
    my $emctl_file = File::Spec->catfile( $ENV{LL_ORACLE_HOME}, "bin", "emctl");
    my $emctl_perl = File::Spec->catfile( $ENV{LL_ORACLE_HOME}, "bin", "emctl.pl");
    # If we are on windows then get emctl.bat rather than emctl
    my $OSNAME = $Config{'osname'};
    my $IsWin32 = ($OSNAME eq 'MSWin32');
    my $hsoe = "";

    if($IsWin32)
    {
        $emctl_file = File::Spec->catfile( $ENV{LL_ORACLE_HOME}, "bin", "emctl.bat" );
        open (EMCTLFILE, $emctl_file) or return '';
        my $line;

        while ($line = <EMCTLFILE>)
        {
           chomp $line;
           if ( $line =~ /^\s*set HOST_SID_OFFSET_ENABLED\s*=\s*(.*?)$/)
           {
              $hsoe = $1;
           }
        }
        close EMCTLFILE;
    }

    my $agent_only_install=0;
    if (-f $emctl_file)
    {
        open (EMCTLFILE, $emctl_file) or return '';
        # Check if CONSOLE_CFG is set to agent
        my @emctlData=<EMCTLFILE>;
        close EMCTLFILE;
        foreach (@emctlData)
        {
            if ((/^\s*CONSOLE_CFG\s*=\s*agent\s*$/) || (/^\s*set CONSOLE_CFG\s*=\s*agent\s*$/))
            {
                $agent_only_install=1;
                last;
            }
        }
    }

    # If we didnt find CONSOLE_CFG set to agent then exit
    # Currently this script is designed for grid control agents
    if ( $agent_only_install == 0 )
    {
        return '';
    }

    my @getemctl_output;

    # Use the emctl command to ?hsoeget the true home of the agent
    # This may not be ORACLE_HOME for dbconsole or RAC installs
    # On Windows this was found not to be possible
    # Therefore a direct call is made to emctl.pl to get the home

    if($IsWin32)
    {
        $ENV{EMDROOT}="$ENV{LL_ORACLE_HOME}";
        $ENV{CONSOLE_CFG}="agent";
        $ENV{PERL5LIB}="$ENV{PERL5LIB};$ENV{LL_ORACLE_HOME}/bin";
        $ENV{ORACLE_HOME}="$ENV{LL_ORACLE_HOME}";
        $ENV{JAVA_HOME}="$ENV{LL_ORACLE_HOME}/jdk";
        $ENV{HOST_OFFSET_ENABLED}=$hsoe;
        @getemctl_output= `$^X $emctl_perl $emctlParam`;
    }
    else
    {
        @getemctl_output= `$emctl_file $emctlParam`;
    }
    return @getemctl_output;
}
######## end of callEmctl #########################
1;
