#!/bin/sh
#
# $Header: cfgHostPatching.sh 10-may-2005.02:40:38 ranmath Exp $
#
# cfgHostPatching.sh
#
# Copyright (c) 2004, 2005, Oracle. All rights reserved.  
#
#    NAME
#      cfgHostPatching.sh - Configure Linux Host for Host Patching.
#
#    DESCRIPTION
#      Configures the Host to participate in Linux Host Patching.
#
#    NOTES
#      Only Linux Hosts are supported.  sudo and yum are pre-requisites.
#      The AIME run_as_root utility can optionally be used in case sudo
#      access is not available.  This is controlled by setting the 
#      T_USE_RUNASROOT variable.
#
#    MODIFIED   (MM/DD/YY)
#    ranmath     04/27/05 - er-4327827: Support SuSE. Make LSB-compliant.
#    ranmath     04/19/05 - bug-4306086: Fix update packages on reboot. 
#    tasingh     04/13/05 - Ensure that oraupdt service script has execute 
#                           permissions. 
#    ranmath     03/11/05 - bug-4234607: Create proper logs. 
#    ranmath     03/03/05 - bug-4216058: Check fine-grained sudo access. 
#    ranmath     01/17/05 - bug-4046545: Do not succeed if intermediate 
#                           commands fail. 
#    ranmath     11/25/04 - Check for T_USE_RUNASROOT in environment and use 
#                           run_as_root in case sudo fails and this variable
#                           is defined. 
#    ranmath     10/01/04 - Support removing the Host Patching configuration 
#                           from Hosts. Refactor script.
#    ranmath     09/13/04 - Creation
#


#
# Purpose:
#
#  Configures this Host for Host Patching via Oracle Enterprise Manager.
#  Can also be used to undo such a configuration.
#
# Usage:
#
#   cfgHostPatching.sh [-needReboot "pkg1,pkg2,..."] \
#     [-confFile file] [-reposURLs url1 url2 url3 ...]
#
#                     or
# 
#   cfgHostPatching.sh [-confFile file] [-undo]
#


# Global variables.

# The Linux variant in use.
# Currently supported values are "RHEL" and "SLES".
DISTRO="RHEL"

# Configuration file used for Host Patching.
CONF_FILE="../../config/hostpatch.conf"

# Log file for recording successful and failed updates.
LOG_FILE=`dirname $CONF_FILE`/../log/patching/hostpatch.log

# List of packages that can only be updated on a reboot.
NEED_REBOOT_PKGS=""

# File that stores the list of packages that can only be updated on
# a reboot.
NRP_LIST_FILE=`dirname $CONF_FILE`/needRebootPkgs.lst

# Oracle Enterprise Manager Host Patching service name.
# Need to install this only if NEED_REBOOT_PKGS becomes non-empty.
ORAUPDT_SVC=oraupdt

# Script implementing the Oracle Enterprise Manager Host Patching service.
ORAUPDT_SCR=/etc/init.d/$ORAUPDT_SVC

# The AIME run_as_root utility for regression testing, in case sudo fails.
RUNASROOT=/usr/local/packages/aime/emdw/run_as_root

# Use an alternative to sudo if it fails.
USE_SUDO_ALT=no


# Checks our pre-requisites.
#
checkPreReqs( )
{
  # We should be running on Linux.
  #
  HOST_OS=`uname -s`
  if test "$HOST_OS" != "Linux"; then
    echo ERROR: Only hosts running Linux are supported! 1>&2
    exit 1
  fi

  # We should be running a supported Linux variant.
  #
  if test -e /etc/redhat-release; then
    DISTRO="RHEL"
  elif test -e /etc/SuSE-release; then
    DISTRO="SLES"
  else
    echo ERROR: Sorry, this Linux variant is not supported yet! 1>&2
    exit 1
  fi

  # Find out the distribution-specific updater client.
  #
  if test "$DISTRO" = "RHEL"; then
    # We need YUM on Red Hat.
    #
    which yum >/dev/null 2>&1
    if test "$?" != "0"; then
      if test -x /usr/local/bin/yum; then
        YUM=/usr/local/bin/yum
      else
        echo ERROR: YUM is required for Linux Host Patching! 1>&2
        exit 1
      fi
    else
      YUM=`which yum`
    fi
  elif test "$DISTRO" = "SLES"; then
    # We need YaST Online Update (YOU) client on SuSE.
    #
    which online_update >/dev/null 2>&1
    if test "$?" != "0"; then
      echo ERROR: YaST Online Update is required for Linux Host Patching! 1>&2
      exit 1
    fi
    YOU=`which online_update`
  fi

  # We need "sudo" for Host Patching.
  #
  which sudo >/dev/null 2>&1
  if test "$?" != "0"; then
    if test -x /usr/local/bin/sudo; then
      SUDO=/usr/local/bin/sudo
    else
      echo ERROR: sudo is required for Linux Host Patching! 1>&2
      exit 1
    fi
  else
    SUDO=`which sudo`
  fi

  # We need sudo to work for this user.
  # If that fails, we will use run_as_root if available *and* only if
  # T_USE_RUNASROOT is defined.
  # (NOTE: $USER does not give the effective user name at all times.)
  #
  IDEXEC=`which id`
  echo Checking sudo access for user \"`$IDEXEC -un`\"...
  $SUDO -k
  echo "$PASSWORD" | $SUDO -S -p "" -v
  CAN_SUDO="$?"

  # Check fine-grained sudo root access to the commands we need.
  if test "$CAN_SUDO" = "0"; then
    if test "$DISTRO" = "RHEL"; then
      SUDO_CMDS="$YUM"
    elif test "$DISTRO" = "SLES"; then
      SUDO_CMDS="$YOU"
    fi
    SUDO_CMDS="$SUDO_CMDS /bin/cp /bin/rm /bin/chmod /sbin/chkconfig"

    for cmd in $SUDO_CMDS
    do
      # Note that all of our commands gladly accept the --version option,
      # *except* for the YaST Online Update (YOU) client and chkconfig
      # on SLES.
      echo "  Checking if user can run \"$cmd\" as root..."
      $SUDO -k
      if test "$cmd" = "$YOU"; then
          echo "$PASSWORD" | $SUDO -S -p "" $cmd -c >/dev/null
      elif test "$cmd" = "/sbin/chkconfig" -a "$DISTRO" = "SLES"; then
          echo "$PASSWORD" | $SUDO -S -p "" $cmd >/dev/null
      else
          echo "$PASSWORD" | $SUDO -S -p "" $cmd --version >/dev/null
      fi
      CAN_SUDO="$?"
      if test "$CAN_SUDO" != "0"; then
        break
      fi
    done
  fi

  if test "$CAN_SUDO" != "0"; then
    if test -z "$T_USE_RUNASROOT"; then
      echo ERROR: Linux Host Patching needs sudo to work for this user! 1>&2
      exit 1
    elif test -x $RUNASROOT; then
	USE_SUDO_ALT=yes
        echo Using \"$RUNASROOT\" instead of sudo...
    else
      echo ERROR: T_USE_RUNASROOT defined but \"$RUNASROOT\" unavailable! 1>&2
      exit 1
    fi
  fi
}

# Adds the Oracle Enterprise Manager Host Patching service.
# This is only needed if there are packages that can
# only be updated on a reboot.
#
addOraUpdtSvc( )
{
  echo Writing out the list of packages needing reboot \"$NRP_LIST_FILE\"...
  echo "$NEED_REBOOT_PKGS" | sed 's/\,/\n/g' >$NRP_LIST_FILE

  if test "$?" != "0"; then
    echo ERROR: Could not write out the list of packages needing reboot! 1>&2
    exit 1
  fi

  # Write out an /etc/init.d script that uses the above list.
  echo Configuring the Oracle Enterprise Manager Host Patching service \"$ORAUPDT_SVC\"...
  TMP_FILE=/tmp/`mktemp rebXXXXXX`
  cat <<"@EOF" >$TMP_FILE
#!/bin/sh

# Startup script for the Oracle Enterprise Manager Host Patching service.
# Copyright (c) 2004, 2005, Oracle. All rights reserved.  
#
# Metadata for chkconfig in RHEL, SLES, etc.
#
# chkconfig: 345 19 19
# description: Oracle Enterprise Manager Host Patching service.
#
# Metadata for other LSB-compliant distros.
#
### BEGIN INIT INFO
# Provides: oraupdt
# Required-Start: $local_fs $network
# Required-Stop: $local_fs $network
# Should-Start: $syslog $time $remote_fs
# Should-Stop: $remote_fs
# Default-Start: 3 4 5
# Default-Stop: 0 1 2 6
# Short-Description: Oracle Enterprise Manager Host Patching service.
# Description: Start Oracle Enterprise Manager Host Patching service 
#	to update those packages that can only be updated upon a reboot.
#	If such a package is successfully updated, the system is rebooted.
### END INIT INFO

# Source LSB init functions library.
. /lib/lsb/init-functions

# Lock file. If this is present, exit without running the update
# script. This is removed on a reboot or shutdown.
@EOF

  if test "$?" != "0"; then
    echo ERROR: Could not write a temporary file! 1>&2
    exit 1
  fi

  echo LOCK_FILE=/var/lock/subsys/$ORAUPDT_SVC >>$TMP_FILE

  cat <<"@EOF" >>$TMP_FILE

# The actual script to be run and its options.
@EOF

  # FIXME: This assumes we were run with an explicit path.
  # (Which we are when invoked by the intended EM Job.)
  MYDIR=`dirname $0`
  echo oraupdt=\"$MYDIR/updHostPackages.sh\" >>$TMP_FILE
  echo oraupdt_opts=\"-rebooting -confFile $CONF_FILE\" >>$TMP_FILE

  cat <<"@EOF" >>$TMP_FILE

# LSB runlevels for reboot and halt.
REBOOT_RL=6
HALT_RL=0

prog="Oracle Enterprise Manager Host Patching"

start( )
{
  echo -n $"Starting $prog: "
  if test -x "$oraupdt"; then
    if test -f "$LOCK_FILE"; then
      log_warning_msg "Lock file exists!"
      RETVAL=1
    else
      start_daemon "$oraupdt" $oraupdt_opts
      RETVAL=$?
      if test "$RETVAL" = "0"; then
        touch "$LOCK_FILE"
      fi
      echo
    fi
  else
    log_failure_msg "Updater script not found!"
    RETVAL=1
  fi
  return $RETVAL
}

stop( )
{
  echo -n $"Stopping $prog: "
  if test -f "$LOCK_FILE"; then
    if pidofproc "$oraupdt"; then
      killproc -TERM "$oraupdt"
    fi
    RUNLEVEL=`/sbin/runlevel | cut -d' ' -f2`
    if test $RUNLEVEL -eq $REBOOT_RL -o $RUNLEVEL -eq $HALT_RL; then
      /bin/rm -f $LOCK_FILE
    fi
    RETVAL=0
  else
    log_warning_msg "Lock file not found!"
    RETVAL=1
  fi
  return $RETVAL
}

status( )
{
  pidofproc "$oraupdt" >/dev/null 2>&1
  POP_RET=$?
  if test "$POP_RET" = "0"; then
    RETVAL=0
  elif test -f "$LOCK_FILE"; then
    RETVAL=2
  else
    RETVAL=3
  fi
  return $RETVAL
}

case "$1" in
    start)
      start
      ;;
    stop)
      stop
      ;;
    status)
      status
      ;;
    *)
      echo $"Usage: $0 start | stop | status"
      exit 1
esac

exit $?
@EOF

  # Set this script up as a service
  runPrivileged /bin/cp $TMP_FILE $ORAUPDT_SCR
  if test "$?" != "0"; then
    echo ERROR: Could not copy patching service script! 1>&2
    exit 1
  fi

  runPrivileged /bin/chmod 755 $ORAUPDT_SCR
  if test "$?" != "0"; then
    echo ERROR: Could not change permissions of patching service script! 1>&2
    exit 1
  fi

  runPrivileged /sbin/chkconfig --add $ORAUPDT_SVC
  if test "$?" != "0"; then
    echo ERROR: Could not install patching service! 1>&2
    exit 1
  fi

  /bin/rm $TMP_FILE
  if test "$?" != "0"; then
    echo ERROR: Could not remove temporary file! 1>&2
    exit 1
  fi
}

# Removes the OEM Host Patching service.
#
removeOraUpdtSvc( )
{
  if test -f $ORAUPDT_SCR; then
    echo Removing the Oracle Enterprise Manager Host Patching service...
    runPrivileged /sbin/chkconfig --del $ORAUPDT_SVC
    if test "$?" != "0"; then
      echo ERROR: Could not uninstall the patching service! 1>&2
      exit 1
    fi

    runPrivileged /bin/rm $ORAUPDT_SCR
    if test "$?" != "0"; then
      echo ERROR: Could not remove patching service script! 1>&2
      exit 1
    fi
  fi
 
  LOCK_FILE="/var/lock/subsys/$ORAUPDT_SVC"
  if test -f "$LOCK_FILE"; then
    echo Removing lock file \"$LOCK_FILE\"...
    runPrivileged /bin/rm "$LOCK_FILE"
    if test "$?" != "0"; then
      echo ERROR: Could not remove lock file \"$LOCK_FILE\"! 1>&2
      exit 1
    fi
  fi

  if test -f $NRP_LIST_FILE; then
    echo Removing list of pacakges needing reboot \"$NRP_LIST_FILE\"...
    /bin/rm $NRP_LIST_FILE
    if test "$?" != "0"; then
      echo ERROR: Could not remove list of packages needing reboot! 1>&2
      exit 1
    fi
  fi
}

# Writes the Host Patching configuration file for YUM (on RHEL).
#
writeYUMConfFile( )
{
  echo Writing out YUM configuration file \"$CONF_FILE\" for Host Patching...

  # Write out the header first.
  cat <<"@EOF" >$CONF_FILE
#
# YUM configuration file for Host Patching.
#

[main]
assumeyes=1
pkgpolicy=last
@EOF

  if test "$?" != "0"; then
    echo ERROR: Could not write out configuration file! 1>&2
    exit 1
  fi

  # Write out where to put logging information.
  echo logfile="$LOG_FILE" >>$CONF_FILE

  # Write out a section for each package repository URL.
  #
  # The names are generated like so because of the way
  # "pkgpolicy=last" works in YUM - see the manual 
  # page for yum.conf for details.
  #
  CTR=0
  while test -n "$1"; do
    echo "  Processing package repository at URL \"$1\"..." 
    REPOS_NO=`expr 99 - $CTR`
    echo >>$CONF_FILE
    echo \[package_repository_$REPOS_NO\] >>$CONF_FILE
    echo Name=PkgRepos$REPOS_NO >>$CONF_FILE
    echo baseurl=$1 >>$CONF_FILE
    echo gpgcheck=0 >>$CONF_FILE

    CTR=`expr $CTR + 1`
    if test "$CTR" = "100"; then
      echo ERROR: Too many package repository URLs! 1>&2
      exit 1
    fi

    shift
  done
}

# Writes the Host Patching configuration file for YaST Online Update (YOU)
# on SLES.
#
writeYOUConfFile( )
{
  echo Writing out YOU configuration file \"$CONF_FILE\" for Host Patching...

  # Write out a header.
  #
  cat <<@EOF >$CONF_FILE
#
# YaST Online Update (YOU) servers list for Host Patching.
#
@EOF

  # Write out a section for each package repository URL.
  #
  CTR=0
  while test -n "$1"; do
    echo "  Processing package repository at URL \"$1\"..." 
    REPOS_NO=`expr 99 - $CTR`
    echo >>$CONF_FILE
    echo \[package_repository_$REPOS_NO\] >>$CONF_FILE
    echo Name=PkgRepos$REPOS_NO >>$CONF_FILE
    echo baseurl=$1 >>$CONF_FILE

    CTR=`expr $CTR + 1`
    if test "$CTR" = "100"; then
      echo ERROR: Too many package repository URLs! 1>&2
      exit 1
    fi

    shift
  done
}

# Writes the configuration file for Host Patching.
#
writeConfFile( )
{
  if test "$DISTRO" = "RHEL"; then
    writeYUMConfFile $*
  else
    writeYOUConfFile $*
  fi
}

# Runs the given command with the given arguments as a privileged user.
#
runPrivileged( )
{
  if test "$USE_SUDO_ALT" != "yes"; then
    # Kill sudo password. This ensures that the password is not
    # inadvertently passed to the executing command in stdin.
    $SUDO -k
    echo "$PASSWORD" | $SUDO -S -p "" $*
  else
    $RUNASROOT "$*"
  fi
}


#
# The main processing.
#

# The password must be on our standard input.
read PASSWORD

# Check pre-requisites next (needs the password entered above).
checkPreReqs

# Process the command line arguments.
while test -n "$1"; do
  if test "$1" = "_dummy_"; then
    #
    # IGNORE.
    #
    # NOTE: The following is needed as we can't have an empty
    # if-then body for the Bourne shell.
    echo >/dev/null 2>&1
  elif test "$1" = "-needReboot"; then
    NEED_REBOOT_PKGS="$2"
    shift
  elif test "$1" = "-confFile"; then
    CONF_FILE="$2"
    NRP_LIST_FILE=`dirname $CONF_FILE`/needRebootPkgs.lst
    LOG_FILE=`dirname $CONF_FILE`/../log/patching/hostpatch.log
    shift
  elif test "$1" = "-undo"; then
    # Undo the changes we have made to this system.
    echo Undoing Host Patching configuration on this Host...

    if test "$DISTRO" = "RHEL"; then
      removeOraUpdtSvc
    fi

    if test -f $CONF_FILE; then
      echo Removing configuration file \"$CONF_FILE\"...
      /bin/rm $CONF_FILE
      if test "$?" != "0"; then
        echo ERROR: Could not remove existing configuration file! 1>&2
	exit 1
      fi
    fi

    echo Successfully undid Host Patching configuration on this Host!

    # NOTE: Do not process anything else.
    exit 0
  elif test "$1" = "-reposURLs"; then
    # NOTE: This must be the LAST command line option, if present.
    #

    shift

    # Write out the Host Patching configuration file.
    writeConfFile $*

    # Are there packages that can only be applied on a reboot?
    #
    if test "$DISTRO" = "RHEL"; then
      if test -n "$NEED_REBOOT_PKGS"; then
        addOraUpdtSvc
      elif test -f $NRP_LIST_FILE; then
        # We are reconfiguring the Host, but without need-reboot
        # packages list this time.
        removeOraUpdtSvc
      fi
    fi

    # Ensure that the logging folder exists.
    LOG_DIR=`dirname $LOG_FILE`
    if test ! -d "$LOG_DIR"; then
      echo Creating Host Patching logging folder \"$LOG_DIR\"...
      /bin/mkdir "$LOG_DIR"
      if test "$?" != "0"; then
	echo ERROR: Could not create logging folder \"$LOG_DIR\"! 1>&2
	exit 1
      fi
    fi

    echo "Done."

    # NOTE: Do not process anything else.
    exit 0
  else
    echo ERROR: Invalid command line argument \"$1\"! 1>&2
    exit 1
  fi
  
  # Move on to the next argument, if any.
  if test -n "$1"; then
    shift
  fi
done
