###################################################################### 
#!/usr/local/bin/perl
#
# $Header: chronos_run.pl 06-jun-2005.15:50:53 adosani Exp $
#
# chronos_run.pl
#
# Copyright (c) 2002, 2005, Oracle. All rights reserved.  
#
#    NAME
#      chronos_run.pl - script to provide chronos run metric 
#                       numbers to agent.
#
#    DESCRIPTION
#        This file appends a chronos_mining.pl generated file
#        and returns a table of values to emd for chronos run
#        metric.
#
#    NOTES
#        It is provided name of file created by chronos_mining.pl
#        for statistics for a run.
#
#    MODIFIED   (MM/DD/YY)
#    adosani     06/06/05 - make run out file name creation consistent with 
#                           CME 
#    adosani     01/28/05 - add option for apmeum log file rotation and number 
#                           and size of log files kept 
#    snakhoda    10/19/03 - fix silent fatal exit 
#    asawant     10/15/03 - Fix incorrect variable names
#    asawant     08/28/03 - Removing extra print line 
#    mashukla    06/04/03 - change log name
#    mashukla    06/03/03 - err msg changes
#    mashukla    04/23/03 - fix err msgs
#    mashukla    09/20/02 - mashukla_run_pl
#    mashukla    09/16/02 - fix returns
#    mashukla    09/12/02 - fix errors
#    mashukla    09/11/02 - fix debug info
#    mashukla    09/10/02 - add output file
#    mashukla    09/09/02 - Creation
#
###################################################################### 
use strict;
use Time::Local;
use Fcntl ':flock';
use Getopt::Long;                      # set up to accept user input
use File::Copy;
use ChronosLogging;
use ChronosTableOut;
###################################################################### 
# Main program.
######################################################################
#Constant definitions
my $SUCCESS = 10;
my $WARNING = 20;
my $FAILURE = 30;
my $CATASTROPHIC_FAILURE = 40;
my $DEBUG_FATAL = 0;
my $DEBUG_LOW = 1;
my $DEBUG_MEDIUM = 2;
my $DEBUG_HIGH = 3;

# status definitions from lower level subroutines
my $STATUS_WARN=2;
my $STATUS_SUCCESS=1;
my $STATUS_FAIL=0;
my $STATUS_ERROR=-1;

#same pathseperator for both nt and unix
my $pathSeperator="/";

#create a new debug file object for writing debug info
my $debug = new ChronosLogging();

#create a unique run number (this is unique for every run)
my $run_id = time();
my %ctx = ();

if (ProcessCommandLineInput(\%ctx) == $CATASTROPHIC_FAILURE)
{
  # exit with failure
  exit 1;
} else {
   if (AppendFile(\%ctx) == $CATASTROPHIC_FAILURE)
   {
     # exit with failure
     exit 1;
   }
}

# exit with success
exit 0;


#################### END of MAIN #####################################

###################################################################### 
# NAME: AppendFile()
###################################################################### 
sub AppendFile
{
  my ($ctxRef) = @_;
  my $inputParametersHashRef = $ctxRef->{COMMANDLINEINPUT};
  my $inFileEntry;

  #open input file craeted by CME

  unless (-e $inputParametersHashRef->{"inFile"}) 
  {
    $ctxRef->{"FILEVALUES"}->{"errorVal"} = "Run Metric Data File not available".
             $inputParametersHashRef->{"inFile"}." $!";
    $debug->PrintLog($DEBUG_LOW,$ctxRef->{"FILEVALUES"}->{"errorVal"},1);
    return $CATASTROPHIC_FAILURE;
  }

  if (!open(INFILE,$inputParametersHashRef->{"inFile"})) 
  {
    $ctxRef->{"FILEVALUES"}->{"errorVal"} = "Error opening Run Metric File".
             $inputParametersHashRef->{"inFile"}." $!";
    $debug->PrintLog($DEBUG_LOW,$ctxRef->{"FILEVALUES"}->{"errorVal"},1);
    return $CATASTROPHIC_FAILURE;
  }

  $inFileEntry=<INFILE>;

  # read the one line in file and close
  chomp ($inFileEntry);
  close (INFILE);

  #print to stdout so that emd can recieve table
  print "chronos_run_result=$inFileEntry\n";
  if (!unlink($inputParametersHashRef->{"inFile"}))
  {
    $ctxRef->{"FILEVALUES"}->{"errorVal"} = "Error failed to delete Run Metric File".
         $inputParametersHashRef->{"inFile"}." $!";
    $debug->PrintLog($DEBUG_LOW,$ctxRef->{"FILEVALUES"}->{"errorVal"});
  }
  $ctxRef->{"FILEVALUES"}->{"errorVal"} = "Successfully Reported Run Metric".
    " Output was: $inFileEntry";
  $debug->PrintLog($DEBUG_LOW,$ctxRef->{"FILEVALUES"}->{"errorVal"});
  return($SUCCESS);
}


###################################################################### 
# NAME: GetCurrentTime()
# OUTPUT: Current/Input paramerter Date/Time returned as MM/DD/YYYY HH:MM:SS
# FUNCTION: Get the current (or input parameter) date and time as a string
###################################################################### 
sub GetCurrentTime
{
  my ($my_time) = @_;
  unless(defined($my_time))
  {
    $my_time = time;
  }
  my ($sec, $min, $hour, $mday, $mon, $year, $wday, $yday, $isdst) = 
    localtime($my_time);
  $mon++;
  $year += 1900;
  return("$year/$mon/$mday $hour:$min:$sec");
}
###################################################################### 
# ProcessCommandLineInput : Processes command line parameters.
#                      Note that $workingDir, $privateDir and 
#                      $destinationDir are furthur qualified 
#                       by $targetName
# Context in: -
# Context out:
#                     $CommandLineInput.inFile
#                     $CommandLineInput.outFileName
#                     $CommandLineInput.inDirName
#                     $CommandLineInput.workingDirName
#                     $CommandLineInput.targetName
#                     $CommandLineInput.targetType
#                     $CommandLineInput.debug
#                     $CommandLineInput.chronosRoot
#                     $CommandLineInput.emdRoot
#
###################################################################### 
sub ProcessCommandLineInput
{
  my ($ctxRef) = @_;
  my $lockFileSufix=".lk";
  my %CommandLineInput;
  my $error="";
  my $status=$STATUS_SUCCESS;
  my $targetType_Name;
  my $chronosLogName="apmeum.log";
  my $sysFileDir="sysman";
  my $chronosLogDir="log";
  my $errorFileSufix="_chronos_run.trace";

  # a URI with any of these extensions is a page
  # NOTE: These strings  must be lower case !!!!
  # we use this table like a set with "if ( defined ..."
  # NOTE: there are other ways a hit can  be a page

  # put first command line parameter in variable progname
  my $progname = $0;
  # this is the first task of the script, so take a time stamp marking the
  # begining of run  
  $CommandLineInput{"startTime"}=$run_id;
  $CommandLineInput{"startTimeStamp"} = GetCurrentTime($run_id);

  # Keep arguments in a new var as GetOptions() will destroy @ARGV
  my @arguments = @ARGV;
 
  my %cmdLine = ();

  # This routine from Getopt package gets all the command line parameters
  GetOptions (\%cmdLine,
              "targetname=s",
              "targettype=s",
              "targetguid=s",
              "debug=i",
              "timezone=s",
              "emdURL=s",
              "trace!",
              "chronosroot=s",
              "emdroot=s",
	      "maxlogfilesizekb=i",
	      "numlogfiles=i",
	      "numtracefiles=i",
	      "logFile=s");

  # accept target name
  if ( $cmdLine{"targetname"} )
  {
    $CommandLineInput{"targetName"} = $cmdLine{"targetname"};
  } else {
    Usage($progname, 'Missing targetname') ;
    return $CATASTROPHIC_FAILURE;
  }
  # accept target type
  if ( $cmdLine{"targettype"} )
  {
    $CommandLineInput{"targetType"} = $cmdLine{"targettype"};
  } else {
    Usage($progname, 'Missing targettype') ;
    return $CATASTROPHIC_FAILURE;
  }
  # create merge of type and name to reduce concatanation
  $targetType_Name =
    $CommandLineInput{"targetType"}.'_'.$CommandLineInput{"targetName"};

  # replace nonstandard characters in targetType with '_' 
  # allowed characters include alphanumeric, '-', '_' and '.'

  $targetType_Name =~ s/[^\w\-\_\.]/_/g;

  # chronos root directory
  if ( $cmdLine{"chronosroot"} )
  {
    $CommandLineInput{"chronosRoot"} = $cmdLine{"chronosroot"};
  } else {
    Usage($progname, 'Missing chronosroot') ;
    return $CATASTROPHIC_FAILURE;
  }

  $CommandLineInput{inFile}=$CommandLineInput{"chronosRoot"}.$pathSeperator.'tmp'.
                   $pathSeperator.$targetType_Name.'_run.out';

  # emd root directory
  if ( $cmdLine{"emdroot"} )
  {
    $CommandLineInput{"emdRoot"} = $cmdLine{"emdroot"};
  } else {
    Usage($progname, 'Missing emdroot') ;
    return $CATASTROPHIC_FAILURE;
  }

  # Log File Name
  if(exists $cmdLine{'logFile'})
  {
    $CommandLineInput{'chronosLogFileName'} = $cmdLine{'logFile'};
  }
  else
  {
    $CommandLineInput{"chronosLogFileName"} = $chronosLogName;
  }

  # put chronos log file in {EMDROOT}/sysman/log directory
  $CommandLineInput{"chronosLogPath"} = $CommandLineInput{"emdRoot"}.
    $pathSeperator.$sysFileDir.$pathSeperator.$chronosLogDir.
                            $pathSeperator.$CommandLineInput{"chronosLogFileName"};

  # turn on debug info in chronos log at level provided in command line
  # parameter, default level is 1
  if (exists $cmdLine{"debug"} )
  {
    # make sure the debugLevel value is valid
    if (($cmdLine{"debug"} >= $DEBUG_LOW) && ($cmdLine{"debug"} <= $DEBUG_HIGH))
    {
      $CommandLineInput{"debugLevel"}=$cmdLine{"debug"};
    } else {
        Usage($progname, 'Invalid debug value') ;
        return $CATASTROPHIC_FAILURE;
    }
  } else {
    $CommandLineInput{"debugLevel"}=$DEBUG_MEDIUM;
  }

  # trace info turned on if debug level is high
  if ( exists $cmdLine{"trace"} )
  {
    $CommandLineInput{"traceInfo"} = $cmdLine{"trace"};
    $CommandLineInput{"errorFileName"} = $CommandLineInput{"chronosRoot"}.
        $pathSeperator.$targetType_Name.$errorFileSufix;
  } else {
    $CommandLineInput{"traceInfo"} = 0;
  }

  $CommandLineInput{inFile}=$CommandLineInput{"chronosRoot"}.$pathSeperator.'tmp'.
                   $pathSeperator.$targetType_Name.'_run.out';
  
# Max size (in KB) of rotated Log files (apmeum.log)
  if(exists $cmdLine{'maxlogfilesizekb'})
  {
    $CommandLineInput{'maxlogfilesizekb'} = $cmdLine{'maxlogfilesizekb'};
  }
  else
  {
    $CommandLineInput{"maxlogfilesizekb"} = 5120;
  }

  # Number of Log files to keep
  if(exists $cmdLine{'numlogfiles'})
  {
    $CommandLineInput{'numlogfiles'} = $cmdLine{'numlogfiles'};
  }
  else
  {
    $CommandLineInput{"numlogfiles"} = 10;
  }

  # Number of Trace files to keep
  if(exists $cmdLine{'numtracefiles'})
  {
    $CommandLineInput{'numtracefiles'} = $cmdLine{'numtracefiles'};
  }
  else
  {
    $CommandLineInput{"numtracefiles"} = 10;
  }

  # open debug file for appending debug info
  if (!$debug->Initialize(\$error,$CommandLineInput{"errorFileName"},
        $CommandLineInput{"chronosLogPath"},$CommandLineInput{"debugLevel"},
        $DEBUG_FATAL,$CommandLineInput{"traceInfo"},
        $CommandLineInput{"targetName"}.":".$CommandLineInput{"targetType"},
        $CommandLineInput{"startTimeStamp"},
	$CommandLineInput{"maxlogfilesizekb"}, $CommandLineInput{"numlogfiles"},
	$CommandLineInput{"numtracefiles"}))
  {
     $ctxRef->{"FILEVALUES"}->{"errorVal"}=
                       "Fatal error while initializing Logging/Tracing";
     $debug->PrintLog($DEBUG_FATAL, $ctxRef->{"FILEVALUES"}->{"errorVal"});
     return $CATASTROPHIC_FAILURE;
  }
  $debug->PrintLog($DEBUG_LOW,"Run Metric Collection:Started");
  $debug->PrintLog($DEBUG_LOW,"Run Metric Collection:Read parameters");
  $debug->PrintLog($DEBUG_LOW,$CommandLineInput{chronosRoot});
  #  make all command line input values hash part of main context hash
  $ctxRef->{"COMMANDLINEINPUT"}=\%CommandLineInput;
 
  return $SUCCESS;
}


###########################################################################
# Usage - print usage message
#
# IN:
#     $progname -- name of perl program
# OUT: -
###########################################################################
sub Usage 
{
    my ($progname,$err_msg)=@_;
    print STDERR <<EOM;

usage: $progname   [ -targetname  <target name>        ]
                           [ -targettype  <target type>        ]
                           [ -timezone    <local timezone>     ]
                           [ -emdURL      <local EMD's URL>    ]
			   [ -chronosroot <chronos' root>      ]
			   [ -emdroot	  <emd's root>         ]
                           [ -debug|nodebug                    ]
                           [ -trace                            ]
                           [ -debug 0|1|2                      ]

 Options	 Description

 -targetname     Name of target. (mandatory)
 -targettype     Type of target. (mandatory)
 -emdURL         The emd's URL. (mandatory)
 -chronosroot	 Path for chronos' root. (mandatory)
 -emdroot	 Path for emd's root. (mandatory)
 -outfile        The file name with path (mandatory).
 -timezone       The local time zone (e.g. PDT). (default is empty)
 -trace          Turn tracing on.
 -notrace        Turn tracing off (default).
 -debug          Set debugging level (0, 1, or 2). 2 is maximum detail. Default 
                 is 1.
 -maxlogfilesizekb Set the maximum size in KB of log file. Default 5 KB.
 -numlogfiles    Maximum number of log files to keep in disk. Default 10.
 -numtracefiles  Maximum number of trace files to keep in disk. Default 10.
 -logFile        Name of Log file to use. Default apmeum.log.

Example:

   $progname -targetname my_tgt -targettype my_tgt_type -timezone America/Los_Angeles PDT -emdURL http://somehost:8000/ -chronosroot /home/oracle/chronos -emdroot /home/oracle/emd

Error -> $err_msg

EOM
}

__END__

