/* This table stores the overall snapshot info */
CREATE TABLE dbsnmp.mgmt_snapshot(
       instance_number 		NUMBER NOT NULL,
       snap_id         		NUMBER NOT NULL,
       capture_time    		DATE NOT NULL,
       db_time         		NUMBER,
       cpu_time        		NUMBER,
       application_wait_time	NUMBER,
       cluster_wait_time	NUMBER,
       user_io_wait_time	NUMBER,
       concurrency_wait_time	NUMBER) NOLOGGING;
/* This table is used to store v$sql snapshot, while configuring baseline. */
CREATE TABLE dbsnmp.mgmt_snapshot_sql(
       snap_id         NUMBER NOT NULL,
       sql_id          VARCHAR2(13),
       hash_value      NUMBER NOT NULL,
       child_number    NUMBER NOT NULL,
       elapsed_time    NUMBER NOT NULL,
       executions      NUMBER NOT NULL) NOLOGGING;
/* This table stores the overall baseline info */
CREATE TABLE dbsnmp.mgmt_baseline(
       instance_number 		NUMBER NOT NULL,
       capture_time    		DATE NOT NULL,
       prev_capture_time 	DATE NOT NULL,
       db_time         		NUMBER,
       cpu_time                 NUMBER,
       application_wait_time    NUMBER,
       cluster_wait_time        NUMBER,
       user_io_wait_time        NUMBER,
       concurrency_wait_time    NUMBER);
/* The baseline table stores the identity of the SQL statements as well as
** the baseline elapsed_time_per_execution for those statements.
*/
CREATE TABLE dbsnmp.mgmt_baseline_sql(
       instance_number NUMBER NOT NULL,
       sql_text        VARCHAR2(1000),
       sql_id          VARCHAR2(13),
       hash_value      NUMBER NOT NULL,
       executions      NUMBER,
       elapsed_time    NUMBER,
       t_per_exec      NUMBER NOT NULL);
/* This table stores the overall capture info */
CREATE TABLE dbsnmp.mgmt_capture(
       instance_number 		NUMBER NOT NULL,
       capture_id      		NUMBER NOT NULL,
       capture_time    		DATE NOT NULL,
       db_time         		NUMBER,
       cpu_time                 NUMBER,
       application_wait_time    NUMBER,
       cluster_wait_time        NUMBER,
       user_io_wait_time        NUMBER,
       concurrency_wait_time    NUMBER);
/* Statistics are captured from v$sql at a regular interval and stored in
** the capture table. The columns capture_id and address together form
** the primary key. The elapsed_time and executions columns store cumulative
** values as opposed to deltas.
*/
CREATE TABLE dbsnmp.mgmt_capture_sql(
       capture_id      NUMBER NOT NULL,
       sql_id          VARCHAR2(13),
       hash_value      NUMBER NOT NULL,
       elapsed_time    NUMBER,
       executions      NUMBER);
/* Remember instance startup time in this table so that we can detect when
** the instance has been bounced.
*/
CREATE TABLE dbsnmp.mgmt_response_config(
       instance_number NUMBER NOT NULL,
       startup_time    DATE);
/* This table stores the overall latest metric info */
CREATE TABLE dbsnmp.mgmt_latest(
       instance_number 		NUMBER NOT NULL,
       capture_id      		NUMBER NOT NULL,
       capture_time    		DATE NOT NULL,
       prev_capture_time 	DATE NOT NULL,
       sql_response_time 	NUMBER NOT NULL,
       adjusted_sql_response_time NUMBER NOT NULL,
       baseline_sql_response_time NUMBER NOT NULL,
       relative_sql_response_time NUMBER NOT NULL,
       db_time         		NUMBER,
       cpu_time                 NUMBER,
       application_wait_time    NUMBER,
       cluster_wait_time        NUMBER,
       user_io_wait_time        NUMBER,
       concurrency_wait_time    NUMBER);
/* This table stores the sql info of the latest metric*/
CREATE TABLE dbsnmp.mgmt_latest_sql(
       capture_id      NUMBER NOT NULL,
       sql_id          VARCHAR2(13),
       hash_value      NUMBER NOT NULL,
       executions      NUMBER,
       elapsed_time    NUMBER,
       t_per_exec      NUMBER,
       adjusted_elapsed_time NUMBER);
/* This table stores the overall info for those metrics whose metric value
** exceeds the threshold.
*/
CREATE TABLE dbsnmp.mgmt_history(
       instance_number 		NUMBER NOT NULL,
       capture_id      		NUMBER NOT NULL,
       capture_time    		DATE NOT NULL,
       prev_capture_time 	DATE NOT NULL,
       sql_response_time 	NUMBER NOT NULL,
       adjusted_sql_response_time NUMBER NOT NULL,
       baseline_sql_response_time NUMBER NOT NULL,
       relative_sql_response_time NUMBER NOT NULL,
       db_time         		NUMBER,
       cpu_time                 NUMBER,
       application_wait_time    NUMBER,
       cluster_wait_time        NUMBER,
       user_io_wait_time        NUMBER,
       concurrency_wait_time    NUMBER);
/* this table is used to save the baseline and capture data when the sql 
** response time is exceeded four times (THRESHOLD_FOR_HISTORY) of the baseline. 
** We always keep no more than 25 collections for each instance.
*/
CREATE TABLE dbsnmp.mgmt_history_sql(
       capture_id       NUMBER NOT NULL,
       sql_id           VARCHAR2(13),
       hash_value       NUMBER NOT NULL,
       executions       NUMBER,
       elapsed_time     NUMBER,
       t_per_exec       NUMBER,
       adjusted_elapsed_time NUMBER);
CREATE GLOBAL TEMPORARY TABLE dbsnmp.mgmt_tempt_sql(
       sql_id          VARCHAR2(13),
       hash_value      NUMBER NOT NULL,
       elapsed_time    NUMBER NOT NULL,
       executions      NUMBER NOT NULL)
       ON COMMIT DELETE ROWS;
CREATE sequence dbsnmp.mgmt_response_capture_id 
       START WITH 1 INCREMENT BY 1 ORDER;
CREATE sequence dbsnmp.mgmt_response_snapshot_id 
       START WITH 1 INCREMENT BY 1 ORDER;
CREATE OR REPLACE VIEW dbsnmp.mgmt_response_baseline AS
   SELECT b.instance_number, s.sql_text, s.hash_value, v.address, s.t_per_exec
     FROM dbsnmp.mgmt_baseline b, dbsnmp.mgmt_baseline_sql s, v$sqlarea v
    WHERE b.instance_number = s.instance_number
      AND s.hash_value = v.hash_value;
CREATE OR REPLACE PACKAGE dbsnmp.mgmt_response wrapped 
0
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
3
9
9000000
1
4
0 
2a
2 :e:
1PACKAGE:
1MGMT_RESPONSE:
1TYPE:
1METRIC_CURSOR_TYPE:
1REF:
1CURSOR:
1FUNCTION:
1GETVERSION:
1RETURN:
1VARCHAR2:
1CREATE_BASELINE:
1SNAP_ID_1:
1NUMBER:
1N_STMTS:
1DELETE_SQL_FROM_BASELINE:
1HASH:
1DELETE_BASELINE:
1RESET_SNAPSHOT_STATS:
1RESET_CAPTURE_STATS:
1CAPTURE_STATS:
1TAKE_V$SQL_SNAPSHOT:
1MIN_EXECS:
1CALC_METRIC:
1METRIC:
1OUT:
1PURGE_OLD_STATS:
1PURGE_OLD_HISTORY:
1SAVE_TO_HISTORY:
1GET_METRIC_CURS:
1METRIC_CURS:
1GET_LATEST_CURS:
1GET_TIME_STATS:
1DB_TIME:
1CPU_TIME:
1APPLICATION_WAIT_TIME:
1CLUSTER_WAIT_TIME:
1USER_IO_WAIT_TIME:
1CONCURRENCY_WAIT_TIME:
1GETDURATION:
1START_DATE:
1TIMESTAMP:
1END_DATE:
0

0
0
a0
2
0 a0 97 a0 9d :2 a0 c8 77
a0 8d a0 b4 a0 2c 6a 9a
8f a0 b0 3d 8f a0 b0 3d
b4 55 6a 9a 8f a0 b0 3d
b4 55 6a 9a b4 55 6a 9a
8f a0 b0 3d b4 55 6a 9a
b4 55 6a 9a b4 55 6a 9a
8f a0 b0 3d b4 55 6a 9a
96 :2 a0 b0 54 b4 55 6a 9a
b4 55 6a 9a b4 55 6a 9a
b4 55 6a 9a 96 :2 a0 b0 54
b4 55 6a 9a 96 :2 a0 b0 54
b4 55 6a 9a 96 :2 a0 b0 54
96 :2 a0 b0 54 96 :2 a0 b0 54
96 :2 a0 b0 54 96 :2 a0 b0 54
96 :2 a0 b0 54 b4 55 6a a0
8d 8f a0 b0 3d 8f a0 b0
3d b4 :2 a0 2c 6a a0 :2 aa 59
58 1d 17 b5 
a0
2
0 3 7 11 25 19 1d 21
15 2c 30 44 48 49 4d 51
55 71 6d 6c 79 86 82 69
8e 81 93 97 9b b7 b3 7e
bf b2 c4 c8 cc af e0 e4
e8 104 100 ff 10c fc 111 115
119 12d 12e 132 136 14a 14b 14f
153 16f 16b 16a 177 167 17c 180
184 1a4 19c 1a0 19b 1ab 198 1b0
1b4 1b8 1cc 1cd 1d1 1d5 1e9 1ea
1ee 1f2 206 207 20b 20f 22f 227
22b 226 236 223 23b 23f 243 263
25b 25f 25a 26a 257 26f 273 277
297 28f 293 28e 29e 2af 2a7 2ab
28b 2b6 2c3 2bb 2bf 2a6 2ca 2db
2d3 2d7 2a3 2e2 2ef 2e7 2eb 2d2
2f6 307 2ff 303 2cf 30e 2fe 313
317 31b 31f 33b 337 2fb 343 34c
348 336 354 333 359 35d 361 365
369 36d 36f 371 374 377 378 381 
a0
2
0 1 10 4 9 1f 23 1f
:2 4 e 19 0 20 :2 4 e 1e
2b :2 1e 33 3e :2 33 1d :2 4 e
28 30 :2 28 27 :2 4 e 0 :2 4
e 23 30 :2 23 22 :2 4 e 0
:2 4 e 0 :2 4 e 22 2f :2 22
21 :2 4 e 1a 21 25 :2 1a 19
:2 4 e 0 :2 4 e 0 :2 4 e
0 :2 4 e 1e 2a 2e :2 1e 1d
:2 4 e 1e 2a 2e :2 1e 1d :2 4
e 1d 25 29 :2 1d 31 3a 3e
:2 31 46 5c 60 :2 46 8 1a 1e
:2 8 26 38 3c :2 26 8 1e 22
:2 8 1c :3 4 e 1a 28 :2 1a 33
3f :2 33 19 4a 51 :2 4 5 :7 1

a0
4
0 :2 1 :6 2 :3 3
0 :3 3 :c 4 :8 5
6 0 :2 6 :8 7
8 0 :2 8 9
0 :2 9 :8 a :9 b
c 0 :2 c d
0 :2 d e 0
:2 e :9 f :9 10 :10 11
:a 12 :5 13 :3 11 :f 14
15 :7 1 
383
4
:3 0 1 :3 0 2
:6 0 1 :2 0 3
:3 0 4 0 7
99 5 :3 0 6
:7 0 4 7 4
:4 0 7 :3 0 8
:a 0 f 2 :7 0
9 :4 0 a :3 0
c d 0 f
a e 0 99
b :a 0 1b 3
:7 0 5 7e 0
3 d :3 0 c
:7 0 13 12 :3 0
a af 0 7
d :3 0 e :7 0
17 16 :3 0 19
:2 0 1b 10 1a
0 99 f :a 0
23 4 :a 0 c
d :3 0 10 :7 0
1f 1e :3 0 21
:2 0 23 1c 22
0 99 11 :a 0
27 5 :7 0 25
:2 0 27 24 26
0 99 12 :a 0
2f 6 :7 0 10
:2 0 e d :3 0
c :7 0 2b 2a
:3 0 2d :2 0 2f
28 2e 0 99
13 :a 0 33 7
:8 0 31 :2 0 33
30 32 0 99
14 :a 0 37 8
:8 0 35 :2 0 37
34 36 0 99
15 :a 0 3f 9
:7 0 14 :2 0 12
d :3 0 16 :7 0
3b 3a :3 0 3d
:2 0 3f 38 3e
0 99 17 :a 0
48 a :7 0 18
:2 0 16 19 :3 0
d :3 0 18 :6 0
44 43 :3 0 46
:2 0 48 40 47
0 99 1a :a 0
4c b :8 0 4a
:2 0 4c 49 4b
0 99 1b :a 0
50 c :8 0 4e
:2 0 50 4d 4f
0 99 1c :a 0
54 d :8 0 52
:2 0 54 51 53
0 99 1d :a 0
5d e :7 0 1c
:2 0 1a 19 :3 0
4 :3 0 1e :6 0
59 58 :3 0 5b
:2 0 5d 55 5c
0 99 1f :a 0
66 f :7 0 20
:2 0 1e 19 :3 0
4 :3 0 1e :6 0
62 61 :3 0 64
:2 0 66 5e 65
0 99 20 :a 0
88 10 :7 0 24
2a3 0 22 19
:3 0 d :3 0 21
:6 0 6b 6a :3 0
28 2cf 0 26
19 :3 0 d :3 0
22 :6 0 70 6f
:3 0 19 :3 0 d
:3 0 23 :6 0 75
74 :3 0 2c 2fb
0 2a 19 :3 0
d :3 0 24 :6 0
7a 79 :3 0 19
:3 0 d :3 0 25
:6 0 7f 7e :3 0
35 333 0 2e
19 :3 0 d :3 0
26 :6 0 84 83
:3 0 86 :2 0 88
67 87 0 99
7 :3 0 27 :a 0
97 11 :7 0 39
:2 0 37 29 :3 0
28 :7 0 8d 8c
:3 0 29 :3 0 2a
:7 0 91 90 :3 0
9 :3 0 d :3 0
93 95 0 97
8a 96 0 99
2 :3 0 3c 9b
0 9b 99 9a
9c 2 9b 9e
0 9d 9c 9f
:8 0 
4e
4
:3 0 1 11 1
15 2 14 18
1 1d 1 20
1 29 1 2c
1 39 1 3c
1 41 1 45
1 56 1 5a
1 5f 1 63
1 68 1 6d
1 72 1 77
1 7c 1 81
6 6c 71 76
7b 80 85 1
8b 1 8f 2
8e 92 11 8
f 1b 23 27
2f 33 37 3f
48 4c 50 54
5d 66 88 97

1
4
0 
9e
0
1
14
11
22
0 1 1 1 1 1 1 1
1 1 1 1 1 1 1 1
1 0 0 0 
34 1 8
2 0 1
51 1 d
81 10 0
72 10 0
8b 11 0
5e 1 f
6d 10 0
24 1 5
5f f 0
56 e 0
28 1 6
4 1 0
8a 1 11
38 1 9
4d 1 c
30 1 7
8f 11 0
7c 10 0
29 6 0
11 3 0
68 10 0
55 1 e
49 1 b
10 1 3
1d 4 0
41 a 0
40 1 a
a 1 2
67 1 10
39 9 0
15 3 0
1c 1 4
77 10 0
0

/
CREATE OR REPLACE PACKAGE BODY dbsnmp.mgmt_response wrapped 
0
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
abcd
3
b
9000000
1
4
0 
15a
2 :e:
1PACKAGE:
1BODY:
1MGMT_RESPONSE:
1PACKAGE_VERSION:
1VARCHAR2:
130:
110.2.0.1.0:
1NUMBER_OF_STMTS_TO_TRACK:
1NUMBER:
125:
1MIN_EXECS:
15:
1MAX_NUMBER_OF_STMTS_IN_HISTORY:
1THRESHOLD_FOR_HISTORY:
1500:
1MIN_DELTA_EXECS:
150:
1COLLECTION_EMPTY:
1-:
10.1:
1NEED_MORE_CAPTURE:
10.2:
1BASELINE_EMPTY:
10.3:
1BASELINE_UNAVAILABLE:
10.4:
1FUNCTION:
1GETVERSION:
1RETURN:
1TAKE_V$SQL_SNAPSHOT:
1SNAPSHOT_ID:
1INSTANCE_NUM:
1SQL_ID:
113:
1CAPTURE_TIME:
1DATE:
1DB_VERSION:
1DB_TIME:
1CPU_TIME:
1APPLICATION_WAIT_TIME:
1CLUSTER_WAIT_TIME:
1USER_IO_WAIT_TIME:
1CONCURRENCY_WAIT_TIME:
1DBMS_OUTPUT:
1PUT_LINE:
1in take_v$sql_snapshot:
1INSTANCE_NUMBER:
1TO_NUMBER:
1TRIM:
1TRAILING:
1SUBSTR:
1VERSION:
1V$INSTANCE:
1SELECT instance_number, TO_NUMBER(TRIM(TRAILING '.' FROM SUBSTR(version, 1, 2+
1))):n   INTO instance_num, db_version:n   FROM v$instance:
1DBSNMP:
1MGMT_RESPONSE_SNAPSHOT_ID:
1NEXTVAL:
1DUAL:
1SELECT dbsnmp.mgmt_response_snapshot_id.nextval INTO snapshot_id FROM dual:
1SYSDATE:
1SELECT SYSDATE INTO capture_time FROM dual:
1>=:
110:
1GET_TIME_STATS:
1MGMT_SNAPSHOT:
1SNAP_ID:
1INSERT /* append */ INTO dbsnmp.mgmt_snapshot(instance_number, snap_id, captu+
1re_time, :n		db_time, cpu_time, application_wait_time, cluster_wait_time, use+
1r_io_wait_time, concurrency_wait_time):n      VALUES(instance_num, snapshot_i+
1d, capture_time, :n		db_time, cpu_time, application_wait_time, cluster_wait_t+
1ime, user_io_wait_time, concurrency_wait_time):
1INSERT /* append */ INTO dbsnmp.mgmt_snapshot(instance_number, snap_id, captu+
1re_time):n      VALUES(instance_num, snapshot_id, capture_time):
1MGMT_SNAPSHOT_SQL:
1HASH_VALUE:
1CHILD_NUMBER:
1ELAPSED_TIME:
1EXECUTIONS:
1V$SQLAREA:
1PARSING_USER_ID:
1USER_ID:
1DBA_USERS:
1USERNAME:
1INSERT /* append */ INTO dbsnmp.mgmt_snapshot_sql(:n          snap_id, sql_id+
1, hash_value, child_number, elapsed_time, executions):n      SELECT snapshot_+
1id, sql_id, hash_value, -1, elapsed_time, executions:n      FROM v$sqlarea:n +
1     WHERE executions > min_execs:n      AND parsing_user_id NOT IN (SELECT u+
1ser_id FROM dba_users:n                                  WHERE username IN ('+
1SYS','SYSMAN','DBSNMP')):
1MGMT_RESPONSE_CONFIG:
1DELETE FROM dbsnmp.mgmt_response_config:n      WHERE instance_number = instan+
1ce_num:
1STARTUP_TIME:
1INSERT INTO dbsnmp.mgmt_response_config(instance_number, startup_time):n     +
1 (SELECT instance_number, startup_time FROM v$instance):
1COMMIT:
1OUT:
1TYPE:
1STATCURTYP:
1REF:
1CURSOR:
1STATS_CUR:
1SQL_STMT:
1200:
1SELECT value FROM v$sysstat WHERE stat_id IN (24469293, 1099569955, 243203433+
17, 3332107451, 3649082374, 3868577743) ORDER BY stat_id:
1VALUE:
1V$SYSSTAT:
1OPEN:
1I:
11:
16:
1LOOP:
1EXIT:
1NOTFOUND:
12:
13:
14:
1CASE_NOT_FOUND:
1CLOSE:
1CREATE_BASELINE:
1SNAP_ID_1:
1N_STMTS:
1HASH:
1DE:
1DT:
1T:
1PREV_CAPTURE_TIME:
1PREV_DB_TIME:
1PREV_CPU_TIME:
1PREV_APPLICATION_WAIT_TIME:
1PREV_CLUSTER_WAIT_TIME:
1PREV_USER_IO_WAIT_TIME:
1PREV_CONCURRENCY_WAIT_TIME:
1DELTA_DB_TIME:
1DELTA_CPU_TIME:
1DELTA_APPLICATION_WAIT_TIME:
1DELTA_CLUSTER_WAIT_TIME:
1DELTA_USER_IO_WAIT_TIME:
1DELTA_CONCURRENCY_WAIT_TIME:
1C1:
1INST_NUM:
1ID_1:
1MIN_EXEC:
1DELTA_E:
1DELTA_T:
1T_PER_EXEC:
1SNAP1:
1SNAP2:
1SELECT sql_id, hash_value, delta_e, delta_t, delta_t/delta_e t_per_exec:n    +
1  FROM:n      (:n        SELECT snap1.sql_id,:n               snap1.hash_valu+
1e,                      :n               CASE WHEN snap2.executions >= snap1.+
1executions THEN snap2.executions - snap1.executions ELSE snap2.executions END+
1 delta_e,:n               CASE WHEN snap2.elapsed_time >= snap1.elapsed_time +
1THEN snap2.elapsed_time - snap1.elapsed_time ELSE snap2.elapsed_time END delt+
1a_t:n        FROM:n             (SELECT sql_id, hash_value, elapsed_time, exe+
1cutions:n              FROM dbsnmp.mgmt_snapshot_sql:n              WHERE sna+
1p_id = id_1) snap1,:n:n             (SELECT sql_id, hash_value, elapsed_time,+
1 executions:n              FROM v$sqlarea) snap2:n:n        WHERE snap1.hash_+
1value = snap2.hash_value :n      ):n      WHERE delta_e >= min_exec:n      OR+
1DER BY delta_e DESC:
1in create_baseline, snap_id = :
1||:
1RESET_SNAPSHOT_STATS:
1MGMT_BASELINE_SQL:
1DELETE FROM dbsnmp.mgmt_baseline_sql:n      WHERE instance_number = instance_+
1num:
1MGMT_BASELINE:
1DELETE FROM dbsnmp.mgmt_baseline:n      WHERE instance_number = instance_num:
1SELECT capture_time, db_time, cpu_time, application_wait_time, cluster_wait_t+
1ime, user_io_wait_time, concurrency_wait_time:n      INTO prev_capture_time, +
1prev_db_time, prev_cpu_time, prev_application_wait_time, prev_cluster_wait_ti+
1me, prev_user_io_wait_time, prev_concurrency_wait_time:n      FROM dbsnmp.mgm+
1t_snapshot:n      WHERE snap_id = snap_id_1:
1/:
1100:
1INSERT INTO dbsnmp.mgmt_baseline(instance_number, capture_time, prev_capture_+
1time, db_time, cpu_time, application_wait_time, cluster_wait_time, user_io_wa+
1it_time, concurrency_wait_time):n      VALUES(instance_num, capture_time, pre+
1v_capture_time, delta_db_time, delta_cpu_time, delta_application_wait_time, d+
1elta_cluster_wait_time, delta_user_io_wait_time, delta_concurrency_wait_time):
1SELECT capture_time:n      INTO prev_capture_time:n      FROM dbsnmp.mgmt_sna+
1pshot:n      WHERE snap_id = snap_id_1:
1INSERT INTO dbsnmp.mgmt_baseline(instance_number, capture_time, prev_capture_+
1time):n      VALUES(instance_num, capture_time, prev_capture_time):
1INSERT INTO dbsnmp.mgmt_baseline_sql(:n            instance_number, sql_id, h+
1ash_value, executions, elapsed_time, t_per_exec) :n            VALUES(instanc+
1e_num, sql_id, hash, de, dt, t):
1B:
1SQL_TEXT:
1V:
1UPDATE dbsnmp.mgmt_baseline_sql b SET b.sql_text =:n         (SELECT v.sql_te+
1xt FROM v$sqlarea v:n           WHERE v.hash_value = b.hash_value) :n      WH+
1ERE b.instance_number = instance_num:
1DELETE FROM dbsnmp.mgmt_snapshot_sql:n      WHERE snap_id = snap_id_1:
1DELETE FROM dbsnmp.mgmt_snapshot:n      WHERE instance_number = instance_num:
1DELETE_SQL_FROM_BASELINE:
1SELECT instance_number INTO instance_num FROM v$instance:
1DELETE FROM dbsnmp.mgmt_snapshot_sql:n      WHERE snap_id IN (SELECT snap_id+
1:n   			FROM dbsnmp.mgmt_snapshot:n   			WHERE instance_number = instance_num+
1):n        AND hash_value = hash:
1DELETE FROM dbsnmp.mgmt_baseline_sql:n      WHERE instance_number = instance_+
1num:n        AND hash_value = hash:
1MGMT_CAPTURE_SQL:
1CAPTURE_ID:
1MGMT_CAPTURE:
1DELETE FROM dbsnmp.mgmt_capture_sql:n      WHERE capture_id IN (SELECT captur+
1e_id FROM dbsnmp.mgmt_capture:n			   WHERE instance_number = instance_num):n +
1       AND hash_value = hash:
1MGMT_LATEST_SQL:
1MGMT_LATEST:
1DELETE FROM dbsnmp.mgmt_latest_sql:n      WHERE capture_id IN (SELECT capture+
1_id FROM dbsnmp.mgmt_latest:n                           WHERE instance_number+
1 = instance_num):n        AND hash_value = hash:
1MGMT_HISTORY_SQL:
1MGMT_HISTORY:
1DELETE FROM dbsnmp.mgmt_history_sql:n      WHERE capture_id IN (SELECT captur+
1e_id FROM dbsnmp.mgmt_history:n                           WHERE instance_numb+
1er = instance_num):n        AND hash_value = hash:
1DELETE_BASELINE:
1DELETE FROM dbsnmp.mgmt_snapshot_sql:n      WHERE snap_id IN (SELECT snap_id+
1:n   			FROM dbsnmp.mgmt_snapshot:n   			WHERE instance_number = instance_num+
1):
1DELETE FROM dbsnmp.mgmt_capture_sql:n      WHERE capture_id IN (SELECT captur+
1e_id FROM dbsnmp.mgmt_capture:n                           WHERE instance_numb+
1er = instance_num):
1DELETE FROM dbsnmp.mgmt_capture:n      WHERE instance_number = instance_num:
1DELETE FROM dbsnmp.mgmt_latest_sql:n      WHERE capture_id IN (SELECT capture+
1_id FROM dbsnmp.mgmt_latest:n                           WHERE instance_number+
1 = instance_num):
1DELETE FROM dbsnmp.mgmt_latest:n      WHERE instance_number = instance_num:
1DELETE FROM dbsnmp.mgmt_history_sql:n      WHERE capture_id IN (SELECT captur+
1e_id FROM dbsnmp.mgmt_history:n                           WHERE instance_numb+
1er = instance_num):
1DELETE FROM dbsnmp.mgmt_history:n      WHERE instance_number = instance_num:
1OLD_STARTUP_TIME:
1CUR_STARTUP_TIME:
1SELECT instance_number, startup_time, TO_NUMBER(TRIM(TRAILING '.' FROM SUBSTR+
1(version, 1, 2))):n    INTO instance_num, cur_startup_time, db_version:n    F+
1ROM v$instance:
1SELECT startup_time:n    INTO old_startup_time:n    FROM dbsnmp.mgmt_response+
1_config:n   WHERE instance_number = instance_num:
1!=:
1UPDATE dbsnmp.mgmt_snapshot_sql:n       SET elapsed_time=0, executions = 0:n +
1    WHERE snap_id = snap_id_1:
1UPDATE dbsnmp.mgmt_snapshot:n         SET db_time=0, cpu_time=0, application_+
1wait_time=0, cluster_wait_time=0, user_io_wait_time=0, concurrency_wait_time=+
10:n       WHERE snap_id = snap_id_1:
1DELETE:n      FROM dbsnmp.mgmt_response_config:n     WHERE instance_number = +
1instance_num:
1INSERT:n      INTO dbsnmp.mgmt_response_config(instance_number, startup_time)+
1:n    VALUES (instance_num, cur_startup_time):
1RESET_CAPTURE_STATS:
1UPDATE dbsnmp.mgmt_capture_sql:n       SET elapsed_time=0, executions = 0 :n +
1    WHERE capture_id IN (SELECT capture_id FROM dbsnmp.mgmt_capture:n			  WHE+
1RE instance_number = instance_num):
1UPDATE dbsnmp.mgmt_capture:n         SET db_time=0, cpu_time=0, application_w+
1ait_time=0, cluster_wait_time=0, user_io_wait_time=0, concurrency_wait_time=0+
1:n       WHERE instance_number = instance_num:
1CAPTURE_STATS:
1BASE_CUR:
1SELECT b.sql_id, b.hash_value FROM dbsnmp.mgmt_baseline_sql b:n              +
1 WHERE b.instance_number = instance_num:
1HASH_IN:
1SELECT * FROM v$sqlarea v WHERE v.hash_value = hash_in:
1STATS_REC:
1ROWTYPE:
1in capture_stats:
1SELECT instance_number, TO_NUMBER(TRIM(TRAILING '.' FROM SUBSTR(version, 1, 2+
1))):n   INTO instance_num, db_version :n   FROM v$instance:
1MGMT_RESPONSE_CAPTURE_ID:
1SELECT dbsnmp.mgmt_response_capture_id.nextval INTO capture_id FROM dual:
1BASE_REC:
1IS NOT NULL:
1SELECT sql_id INTO sql_id FROM dbsnmp.mgmt_baseline_sql :n             WHERE +
1hash_value=stats_rec.hash_value:n               AND instance_number=instance_+
1num:
1MGMT_TEMPT_SQL:
1INSERT INTO dbsnmp.mgmt_tempt_sql(sql_id, hash_value, elapsed_time, execution+
1s):n               VALUES (base_rec.sql_id, stats_rec.hash_value,  :n        +
1                    stats_rec.elapsed_time, stats_rec.executions):
1INSERT INTO dbsnmp.mgmt_capture(instance_number, capture_id, capture_time, db+
1_time, cpu_time, application_wait_time, cluster_wait_time, user_io_wait_time,+
1 concurrency_wait_time):n      VALUES(instance_num, capture_id, capture_time,+
1 db_time, cpu_time, application_wait_time, cluster_wait_time, user_io_wait_ti+
1me, concurrency_wait_time):
1INSERT INTO dbsnmp.mgmt_capture(instance_number, capture_id, capture_time):n +
1     VALUES(instance_num, capture_id, capture_time):
1SUM:
1INSERT INTO dbsnmp.mgmt_capture_sql(:n          capture_id, sql_id, hash_valu+
1e, elapsed_time, executions):n       SELECT capture_id, sql_id, hash_value, s+
1um(elapsed_time), sum(executions):n             FROM dbsnmp.mgmt_tempt_sql:n +
1            GROUP BY hash_value, sql_id:
1PURGE_OLD_STATS:
1ID_CURS:
1SELECT capture_id :n      FROM dbsnmp.mgmt_capture:n      WHERE instance_numb+
1er = inst_num:n      ORDER BY capture_id DESC:
1CAP_ID_1:
1CAP_ID_2:
1DELETE FROM dbsnmp.mgmt_capture_sql:n   WHERE capture_id IN (:n      SELECT c+
1apture_id:n      FROM dbsnmp.mgmt_capture:n      WHERE instance_number = inst+
1ance_num:n      AND capture_id < cap_id_2):
1DELETE FROM dbsnmp.mgmt_capture:n      WHERE instance_number = instance_num:n+
1        AND capture_id < cap_id_2:
1CALC_METRIC:
1METRIC:
1SELECT capture_id, capture_time, db_time, cpu_time, application_wait_time, cl+
1uster_wait_time, user_io_wait_time, concurrency_wait_time:n      FROM dbsnmp.+
1mgmt_capture:n      WHERE instance_number = inst_num:n      ORDER BY capture_+
1id DESC:
1DELTA_CURS:
1ID_2:
1CAPTURE1:
1CAPTURE2:
1BASELINE:
1SELECT capture1.hash_value, capture1.sql_id,:n             CASE WHEN capture1+
1.elapsed_time >= capture2.elapsed_time THEN capture1.elapsed_time - capture2.+
1elapsed_time ELSE capture1.elapsed_time END,:n             CASE WHEN capture1+
1.executions >= capture2.executions THEN capture1.executions - capture2.execut+
1ions ELSE capture1.executions END,:n             baseline.executions, baselin+
1e.elapsed_time:n      FROM (SELECT * FROM dbsnmp.mgmt_capture_sql WHERE captu+
1re_id = id_1) capture1,:n           (SELECT * FROM dbsnmp.mgmt_capture_sql WH+
1ERE capture_id = id_2) capture2,:n           dbsnmp.mgmt_baseline_sql baselin+
1e:n      WHERE capture1.hash_value = capture2.hash_value AND:n            cap+
1ture1.hash_value = baseline.hash_value AND:n            baseline.instance_num+
1ber = instance_num AND:n            (CASE WHEN capture1.executions >= capture+
12.executions THEN capture1.executions - capture2.executions ELSE capture1.exe+
1cutions END) >= MIN_DELTA_EXECS:
1BASELINE_EXEC:
1BASELINE_ELAP_TIME:
1H_VALUE:
1CAP_TIME_1:
1CAP_TIME_2:
1DB_TIME_1:
1CPU_TIME_1:
1APPLICATION_WAIT_TIME_1:
1CLUSTER_WAIT_TIME_1:
1USER_IO_WAIT_TIME_1:
1CONCURRENCY_WAIT_TIME_1:
1DB_TIME_2:
1CPU_TIME_2:
1APPLICATION_WAIT_TIME_2:
1CLUSTER_WAIT_TIME_2:
1USER_IO_WAIT_TIME_2:
1CONCURRENCY_WAIT_TIME_2:
1DELTA_TIME:
1DELTA_EXECS:
1ROW_COUNT:
1ADJ_ELAP_TIME:
1ADJ_ELAP_TIME_SUM:
1BASELINE_ELAP_TIME_SUM:
1ELAP_TIME_SUM:
1id1=:
1,id2=:
1DELETE FROM dbsnmp.mgmt_latest_sql:n   WHERE capture_id IN (SELECT capture_id+
1 FROM dbsnmp.mgmt_latest:n			WHERE instance_number = instance_num):
1DELETE FROM dbsnmp.mgmt_latest:n   WHERE instance_number = instance_num:
10:
1h_value=:
1,delta_t=:
1,delta_e=:
1,base_exec=:
1,base_elap_time=:
1*:
1adj_elap_time=:
1ROUND:
1+:
1ADJUSTED_ELAPSED_TIME:
1INSERT INTO dbsnmp.mgmt_latest_sql(:n	    capture_id, sql_id, hash_value, exe+
1cutions, elapsed_time, t_per_exec, adjusted_elapsed_time):n         VALUES(ca+
1p_id_1, sql_id, h_value, delta_execs, delta_time, t_per_exec, adj_elap_time):
1>:
1SQL_RESPONSE_TIME:
1ADJUSTED_SQL_RESPONSE_TIME:
1BASELINE_SQL_RESPONSE_TIME:
1RELATIVE_SQL_RESPONSE_TIME:
1INSERT INTO dbsnmp.mgmt_latest(instance_number, capture_id, capture_time, pre+
1v_capture_time, sql_response_time,:n		adjusted_sql_response_time, baseline_sq+
1l_response_time, relative_sql_response_time, :n		db_time, cpu_time, applicati+
1on_wait_time, cluster_wait_time, user_io_wait_time, concurrency_wait_time):n +
1     VALUES(instance_num, cap_id_1, cap_time_1, cap_time_2, elap_time_sum/100+
10000, adj_elap_time_sum/1000000, :n		baseline_elap_time_sum/1000000, metric, +
1:n		delta_db_time, delta_cpu_time, delta_application_wait_time, delta_cluster+
1_wait_time, delta_user_io_wait_time, delta_concurrency_wait_time):
1INSERT INTO dbsnmp.mgmt_latest(instance_number, capture_id, capture_time, pre+
1v_capture_time, sql_response_time,:n                adjusted_sql_response_tim+
1e, baseline_sql_response_time, relative_sql_response_time):n      VALUES(inst+
1ance_num, cap_id_1, cap_time_1, cap_time_2, elap_time_sum/1000000, adj_elap_t+
1ime_sum/1000000,:n                baseline_elap_time_sum/1000000, metric):
1SAVE_TO_HISTORY:
1PURGE_OLD_HISTORY:
1CAP_ID:
1in save_to_history:
1SELECT capture_id INTO cap_id:n   FROM dbsnmp.mgmt_latest:n   WHERE instance_+
1number = instance_num:
1DELETE FROM dbsnmp.mgmt_history WHERE capture_id = cap_id:
1DELETE FROM dbsnmp.mgmt_history_sql WHERE capture_id = cap_id:
1INSERT INTO dbsnmp.mgmt_history:n     SELECT * FROM dbsnmp.mgmt_latest:n     +
1WHERE capture_id = cap_id:
1INSERT INTO dbsnmp.mgmt_history_sql:n     SELECT * FROM dbsnmp.mgmt_latest_sq+
1l:n     WHERE capture_id = cap_id:
1SELECT capture_id :n      FROM dbsnmp.mgmt_history:n      WHERE instance_numb+
1er = inst_num:n      ORDER BY capture_id DESC:
1CURR_CAP_ID:
1in purge_old_history:
1row_count = :
1=:
1cap_id = :
1DELETE FROM dbsnmp.mgmt_history_sql:n      WHERE capture_id IN (:n         SE+
1LECT capture_id:n         FROM dbsnmp.mgmt_history:n         WHERE instance_n+
1umber = instance_num:n         AND capture_id <= cap_id):
1DELETE FROM dbsnmp.mgmt_history:n        WHERE instance_number = instance_num+
1:n        AND capture_id <= cap_id:
1GET_METRIC_CURS:
1METRIC_CURS:
1METRIC_CURSOR_TYPE:
1BASELINE_EXISTS:
1BASELINE_SQL_COUNT:
1SNAP_ID_2:
1CURR_TIME:
1PREV_TIME:
1DURATION:
1SELECT snap_id :n      FROM dbsnmp.mgmt_snapshot:n      WHERE instance_number+
1 = inst_num:n      ORDER BY snap_id DESC:
1SELECT 1 INTO baseline_exists FROM dual :n         WHERE EXISTS (SELECT * FRO+
1M dbsnmp.mgmt_baseline:n                       WHERE instance_number = instan+
1ce_num):
1NO_DATA_FOUND:
1baseline not configured:
1no v$sql snapshot; taking v$sql snapshot:
1SELECT SYSDATE INTO curr_time FROM dual:
1SELECT capture_time INTO prev_time FROM dbsnmp.mgmt_snapshot WHERE instance_n+
1umber = instance_num:
1GETDURATION:
1300:
1have v$sql snapshot; creating baseline:
1COUNT:
1SELECT count(*):n              INTO baseline_sql_count:n              FROM db+
1snmp.mgmt_baseline_sql:n             WHERE instance_number = instance_num:
1capturing stats:
1only one capture; so no metric yet:
1SELECT count(*):n        INTO baseline_sql_count:n        FROM dbsnmp.mgmt_ba+
1seline_sql:n       WHERE instance_number = instance_num:
1MAX:
1SELECT max(capture_time) INTO prev_time FROM dbsnmp.mgmt_capture WHERE instan+
1ce_number = instance_num:
1calculating metric:
1SELECT relative_sql_response_time:n                 INTO metric:n            +
1     FROM dbsnmp.mgmt_latest:n                WHERE instance_number = instanc+
1e_num:
1SELECT metric FROM dual:
1GET_LATEST_CURS:
1RELATIVE_RESPONSE_TIME:
1BASELINE_RESPONSE_TIME:
1RESPONSE_TIME:
1BASELINE_COUNT:
1CAPTURE_COUNT:
1SELECT count(*):n     INTO baseline_count:n     FROM dbsnmp.mgmt_baseline:n  +
1  WHERE instance_number = instance_num:
1SELECT count(*):n           INTO capture_count:n           FROM dbsnmp.mgmt_c+
1apture:n          WHERE instance_number = instance_num:
1SELECT relative_sql_response_time, baseline_sql_response_time, adjusted_sql_r+
1esponse_time:n              INTO relative_response_time, baseline_response_ti+
1me, response_time:n              FROM dbsnmp.mgmt_latest:n             WHERE +
1instance_number = instance_num:
1relative_sql_response_time=:
1baseline_sql_response_time=:
1sql_response_time=:
1SELECT relative_response_time, baseline_response_time, response_time FROM dua+
1l:
1START_DATE:
1TIMESTAMP:
1END_DATE:
1INTERVAL:
1DAY:
1SECOND:
1INTERVAL DAY TO SECOND:
1EXTRACT:
1 SYS$EXTRACT_FROM:
124:
1HOUR:
160:
1MINUTE:
0

0
0
da8
2
0 :2 a0 97 a3 a0 51 a5 1c
6e 81 b0 a3 a0 1c 51 81
b0 a3 a0 1c 51 81 b0 a3
a0 1c 51 81 b0 a3 a0 1c
51 81 b0 a3 a0 1c 51 81
b0 a3 a0 1c 7e 51 b4 2e
81 b0 a3 a0 1c 7e 51 b4
2e 81 b0 a3 a0 1c 7e 51
b4 2e 81 b0 a3 a0 1c 7e
51 b4 2e 81 b0 a0 8d a0
b4 a0 2c 6a :2 a0 65 b7 a4
a0 b1 11 68 4f 9a 8f a0
b0 3d b4 55 6a a3 a0 1c
81 b0 a3 a0 1c 81 b0 a3
a0 51 a5 1c 81 b0 a3 a0
1c 81 b0 a3 a0 1c 81 b0
a3 a0 1c 81 b0 a3 a0 1c
81 b0 a3 a0 1c 81 b0 a3
a0 1c 81 b0 a3 a0 1c 81
b0 a3 a0 1c 81 b0 :2 a0 6b
6e a5 57 :9 a0 12a :5 a0 12a :3 a0
12a a0 7e 51 b4 2e :7 a0 a5
57 :14 a0 12a b7 :8 a0 12a b7 :2 19
3c :14 a0 12a :4 a0 12a :7 a0 12a a0
12a b7 a4 b1 11 68 4f 9a
96 :2 a0 b0 54 96 :2 a0 b0 54
96 :2 a0 b0 54 96 :2 a0 b0 54
96 :2 a0 b0 54 96 :2 a0 b0 54
b4 55 6a a0 9d :2 a0 c8 77
a3 a0 1c 81 b0 a3 a0 51
a5 1c 6e 81 b0 a3 :2 a0 6b
:2 a0 f 1c 81 b0 :3 a0 11c 11d
91 :2 51 a0 63 37 :2 a0 e9 d3
:3 a0 f 2b a0 51 :2 a0 d b7
a6 9 51 :2 a0 d b7 a6 9
51 :2 a0 d b7 a6 9 51 :2 a0
d b7 a6 9 51 :2 a0 d b7
a6 9 51 :2 a0 d b7 a6 9
a0 62 b7 9 a4 14 b7 a0
47 :2 a0 e9 c1 b7 a4 b1 11
68 4f 9a 8f a0 b0 3d 8f
a0 b0 3d b4 55 6a a3 a0
1c 81 b0 a3 a0 1c 81 b0
a3 :2 a0 6b :2 a0 f 1c 81 b0
a3 a0 1c 81 b0 a3 a0 1c
81 b0 a3 a0 1c 81 b0 a3
a0 51 a5 1c 81 b0 a3 a0
1c 81 b0 a3 a0 1c 81 b0
a3 a0 1c 81 b0 a3 a0 1c
81 b0 a3 a0 1c 81 b0 a3
a0 1c 81 b0 a3 a0 1c 81
b0 a3 a0 1c 81 b0 a3 a0
1c 81 b0 a3 a0 1c 81 b0
a3 a0 1c 81 b0 a3 a0 1c
81 b0 a3 a0 1c 81 b0 a3
a0 1c 81 b0 a3 a0 1c 81
b0 a3 a0 1c 81 b0 a3 a0
1c 81 b0 a3 a0 1c 81 b0
a3 a0 1c 81 b0 a3 a0 1c
81 b0 a3 a0 1c 81 b0 a0
f4 8f a0 b0 3d 8f a0 b0
3d 8f a0 b0 3d b4 bf c8
:37 a0 12a bd b7 11 a4 b1 :2 a0
6b 6e 7e a0 b4 2e a5 57
:2 a0 a5 57 :9 a0 12a :4 a0 12a :4 a0
12a :3 a0 12a a0 7e 51 b4 2e
:7 a0 a5 57 :12 a0 12a :2 a0 7e b4
2e 5a :2 a0 7e a0 b4 2e 5a
7e 51 b4 2e d b7 :2 a0 7e
51 b4 2e d b7 :2 19 3c :2 a0
7e b4 2e 5a :2 a0 7e a0 b4
2e 5a 7e 51 b4 2e d b7
:2 a0 7e 51 b4 2e d b7 :2 19
3c :2 a0 7e b4 2e 5a :2 a0 7e
a0 b4 2e 5a 7e 51 b4 2e
d b7 :2 a0 7e 51 b4 2e d
b7 :2 19 3c :2 a0 7e b4 2e 5a
:2 a0 7e a0 b4 2e 5a 7e 51
b4 2e d b7 :2 a0 7e 51 b4
2e d b7 :2 19 3c :2 a0 7e b4
2e 5a :2 a0 7e a0 b4 2e 5a
7e 51 b4 2e d b7 :2 a0 7e
51 b4 2e d b7 :2 19 3c :2 a0
7e b4 2e 5a :2 a0 7e a0 b4
2e 5a 7e 51 b4 2e d b7
:2 a0 7e 51 b4 2e d b7 :2 19
3c :14 a0 12a b7 :6 a0 12a :8 a0 12a
b7 :2 19 3c :5 a0 a5 dd e9 91
51 :2 a0 63 37 :6 a0 e9 d3 5
:3 a0 f 2b :e a0 12a b7 a0 47
:2 a0 e9 c1 :10 a0 12a :4 a0 12a :4 a0
12a a0 12a b7 a4 b1 11 68
4f 9a 8f a0 b0 3d b4 55
6a a3 a0 1c 81 b0 :3 a0 12a
:a a0 12a :6 a0 12a :a a0 12a :a a0 12a
:a a0 12a a0 12a b7 a4 b1 11
68 4f 9a b4 55 6a a3 a0
1c 81 b0 :3 a0 12a :8 a0 12a :4 a0
12a :4 a0 12a :4 a0 12a :8 a0 12a :4 a0
12a :8 a0 12a :4 a0 12a :8 a0 12a :4 a0
12a a0 12a b7 a4 b1 11 68
4f 9a 8f a0 b0 3d b4 55
6a a3 a0 1c 81 b0 a3 a0
1c 81 b0 a3 a0 1c 81 b0
a3 a0 1c 81 b0 :b a0 12a :6 a0
12a :2 a0 7e b4 2e :6 a0 12a a0
7e 51 b4 2e :a a0 12a b7 19
3c :4 a0 12a :6 a0 12a b7 19 3c
a0 12a b7 a4 b1 11 68 4f
9a b4 55 6a a3 a0 1c 81
b0 a3 a0 1c 81 b0 a3 a0
1c 81 b0 a3 a0 1c 81 b0
:b a0 12a :6 a0 12a :2 a0 7e b4 2e
:a a0 12a a0 7e 51 b4 2e :a a0
12a b7 19 3c :4 a0 12a :6 a0 12a
b7 19 3c a0 12a b7 a4 b1
11 68 4f 9a b4 55 6a a0
f4 8f a0 b0 3d b4 bf c8
:a a0 12a bd b7 11 a4 b1 a0
f4 8f a0 b0 3d b4 bf c8
:5 a0 12a bd b7 11 a4 b1 a3
:2 a0 f 1c 81 b0 a3 a0 1c
81 b0 a3 a0 1c 81 b0 a3
a0 1c 81 b0 a3 a0 1c 81
b0 a3 a0 1c 81 b0 a3 a0
1c 81 b0 a3 a0 1c 81 b0
a3 a0 1c 81 b0 a3 a0 1c
81 b0 a3 a0 1c 81 b0 a3
a0 51 a5 1c 81 b0 :2 a0 6b
6e a5 57 a0 b4 57 :9 a0 12a
:5 a0 12a :3 a0 12a 91 :2 a0 a5 b
a0 37 :2 a0 6b 7e b4 2e 91
:3 a0 6b a5 b a0 37 :9 a0 12a
:e a0 12a b7 a0 47 b7 19 3c
b7 a0 47 a0 7e 51 b4 2e
:7 a0 a5 57 :14 a0 12a b7 :8 a0 12a
b7 :2 19 3c :12 a0 12a a0 12a b7
a4 b1 11 68 4f 9a b4 55
6a a0 f4 8f a0 b0 3d b4
bf c8 :6 a0 12a bd b7 11 a4
b1 a3 a0 1c 81 b0 a3 a0
1c 81 b0 a3 a0 1c 81 b0
:3 a0 12a :3 a0 a5 dd e9 :2 a0 e9
d3 :2 a0 f 5a :2 a0 e9 c1 a0
65 b7 19 3c :2 a0 e9 d3 :2 a0
f 5a :2 a0 e9 c1 a0 65 b7
19 3c :2 a0 e9 c1 :a a0 12a :6 a0
12a a0 12a b7 a4 b1 11 68
4f 9a 96 :2 a0 b0 54 b4 55
6a a0 f4 8f a0 b0 3d b4
bf c8 :d a0 12a bd b7 11 a4
b1 a0 f4 8f a0 b0 3d 8f
a0 b0 3d 8f a0 b0 3d b4
bf c8 :3f a0 12a bd b7 11 a4
b1 a3 a0 1c 81 b0 a3 a0
1c 81 b0 a3 :2 a0 6b :2 a0 f
1c 81 b0 a3 a0 51 a5 1c
81 b0 a3 a0 1c 81 b0 a3
a0 1c 81 b0 a3 a0 1c 81
b0 a3 a0 1c 81 b0 a3 a0
1c 81 b0 a3 a0 1c 81 b0
a3 a0 1c 81 b0 a3 a0 1c
81 b0 a3 a0 1c 81 b0 a3
a0 1c 81 b0 a3 a0 1c 81
b0 a3 a0 1c 81 b0 a3 a0
1c 81 b0 a3 a0 1c 81 b0
a3 a0 1c 81 b0 a3 a0 1c
81 b0 a3 a0 1c 81 b0 a3
a0 1c 81 b0 a3 a0 1c 81
b0 a3 a0 1c 81 b0 a3 a0
1c 81 b0 a3 a0 1c 81 b0
a3 a0 1c 81 b0 a3 a0 1c
81 b0 a3 a0 1c 81 b0 a3
a0 1c 81 b0 a3 a0 1c 81
b0 a3 a0 1c 81 b0 a3 a0
1c 81 b0 a3 a0 1c 81 b0
a3 a0 1c 81 b0 a3 a0 1c
81 b0 :2 a0 d :9 a0 12a :3 a0 a5
dd e9 :9 a0 e9 d3 5 :2 a0 f
5a :2 a0 e9 c1 a0 65 b7 19
3c :9 a0 e9 d3 5 :2 a0 f 5a
:2 a0 e9 c1 a0 65 b7 19 3c
:2 a0 e9 c1 :2 a0 6b 6e 7e a0
b4 2e 7e 6e b4 2e 7e a0
b4 2e a5 57 :8 a0 12a :4 a0 12a
a0 51 d a0 51 d a0 51
d a0 51 d :5 a0 a5 dd e9
:8 a0 e9 d3 5 :3 a0 f 2b :2 a0
6b 6e 7e a0 b4 2e 7e 6e
b4 2e 7e a0 b4 2e 7e 6e
b4 2e 7e a0 b4 2e 7e 6e
b4 2e 7e a0 b4 2e 7e 6e
b4 2e 7e a0 b4 2e a5 57
:2 a0 7e a0 b4 2e d :2 a0 7e
a0 b4 2e d :2 a0 6b 6e 7e
:2 a0 51 a5 b b4 2e a5 57
:2 a0 7e a0 b4 2e d :2 a0 7e
a0 b4 2e d :2 a0 7e a0 b4
2e d :2 a0 7e 51 b4 2e d
:10 a0 12a b7 a0 47 :2 a0 e9 c1
a0 7e 51 b4 2e :2 a0 7e a0
b4 2e 7e 51 b4 2e d b7
19 3c a0 7e 51 b4 2e :2 a0
7e b4 2e 5a :2 a0 7e a0 b4
2e 5a 7e 51 b4 2e d b7
:2 a0 7e 51 b4 2e d b7 :2 19
3c :2 a0 7e b4 2e 5a :2 a0 7e
a0 b4 2e 5a 7e 51 b4 2e
d b7 :2 a0 7e 51 b4 2e d
b7 :2 19 3c :2 a0 7e b4 2e 5a
:2 a0 7e a0 b4 2e 5a 7e 51
b4 2e d b7 :2 a0 7e 51 b4
2e d b7 :2 19 3c :2 a0 7e b4
2e 5a :2 a0 7e a0 b4 2e 5a
7e 51 b4 2e d b7 :2 a0 7e
51 b4 2e d b7 :2 19 3c :2 a0
7e b4 2e 5a :2 a0 7e a0 b4
2e 5a 7e 51 b4 2e d b7
:2 a0 7e 51 b4 2e d b7 :2 19
3c :2 a0 7e b4 2e 5a :2 a0 7e
a0 b4 2e 5a 7e 51 b4 2e
d b7 :2 a0 7e 51 b4 2e d
b7 :2 19 3c :1e a0 12a b7 :12 a0 12a
b7 :2 19 3c a0 12a :2 a0 7e b4
2e a0 57 b3 a0 57 b3 b7
19 3c b7 a4 b1 11 68 4f
9a b4 55 6a a3 a0 1c 81
b0 a3 a0 1c 81 b0 :2 a0 6b
6e a5 57 :3 a0 12a :6 a0 12a :4 a0
12a :4 a0 12a :6 a0 12a :6 a0 12a a0
12a b7 a4 b1 11 68 4f 9a
b4 55 6a a0 f4 8f a0 b0
3d b4 bf c8 :6 a0 12a bd b7
11 a4 b1 a3 a0 1c 81 b0
a3 a0 1c 81 b0 a3 a0 1c
81 b0 a3 a0 1c 81 b0 :2 a0
6b 6e a5 57 :3 a0 12a a0 51
d a0 7e 51 b4 2e d :3 a0
a5 dd e9 :3 a0 6b 6e 7e a0
b4 2e a5 57 :2 a0 e9 d3 :3 a0
f 2b :2 a0 7e b4 2e :2 a0 d
a0 2b b7 19 3c :2 a0 7e 51
b4 2e d b7 a0 47 :2 a0 e9
c1 :2 a0 6b 6e 7e a0 b4 2e
a5 57 a0 7e 51 b4 2e :a a0
12a :6 a0 12a a0 12a b7 19 3c
b7 a4 b1 11 68 4f 9a 96
:2 a0 b0 54 b4 55 6a a3 a0
1c 81 b0 a3 a0 1c 81 b0
a3 a0 1c 81 b0 a3 a0 1c
81 b0 a3 a0 1c 81 b0 a3
a0 1c 81 b0 a3 a0 1c 81
b0 a3 a0 1c 81 b0 a3 a0
1c 81 b0 a0 f4 8f a0 b0
3d b4 bf c8 :6 a0 12a bd b7
11 a4 b1 :3 a0 12a a0 51 d
:6 a0 12a b7 :2 a0 51 d b7 a6
9 a4 b1 11 4f a0 7e 51
b4 2e :2 a0 6b 6e a5 57 :3 a0
a5 dd e9 :2 a0 e9 d3 :2 a0 f
5a :2 a0 6b 6e a5 57 :2 a0 a5
57 :2 a0 e9 c1 :2 a0 d b7 :2 a0
e9 c1 :3 a0 12a :6 a0 12a :4 a0 a5
b d a0 7e 51 b4 2e :2 a0
6b 6e a5 57 :3 a0 a5 57 :6 a0
12a a0 7e 51 b4 2e :2 a0 6b
6e a5 57 a0 57 b3 :2 a0 6b
6e a5 57 :2 a0 d b7 :2 a0 d
b7 :2 19 3c b7 :2 a0 d b7 :2 19
3c b7 :2 19 3c b7 :6 a0 12a a0
7e 51 b4 2e :3 a0 12a :7 a0 12a
:4 a0 a5 b d a0 7e 51 b4
2e :2 a0 6b 6e a5 57 :2 a0 6b
a0 6b 57 b3 :2 a0 6b a0 6b
57 b3 :2 a0 6b 6e a5 57 :2 a0
6b a0 6b a0 a5 57 b7 :6 a0
12a b7 :3 a0 d b7 a6 9 a4
b1 11 4f b7 :2 19 3c b7 :2 a0
d b7 :2 19 3c b7 :2 19 3c :4 a0
12a e9 dd b7 a4 b1 11 68
4f 9a 96 :2 a0 b0 54 b4 55
6a a3 a0 1c 81 b0 a3 a0
1c 81 b0 a3 a0 1c 81 b0
a3 a0 1c 81 b0 a3 a0 1c
81 b0 a3 a0 1c 81 b0 a3
a0 1c 81 b0 a3 a0 1c 81
b0 :3 a0 12a a0 51 d a0 51
d :6 a0 12a a0 7e 51 b4 2e
:6 a0 12a a0 7e 51 b4 2e :6 a0
12a a0 7e 51 b4 2e :a a0 12a
b7 :2 a0 d b7 :2 19 3c b7 :2 a0
d b7 :2 19 3c b7 :2 a0 d b7
:2 19 3c :2 a0 6b 6e 7e a0 b4
2e a5 57 :2 a0 6b 6e 7e a0
b4 2e a5 57 :2 a0 6b 6e 7e
a0 b4 2e a5 57 :6 a0 12a e9
dd b7 a4 b1 11 68 4f a0
8d 8f a0 b0 3d 8f a0 b0
3d b4 :2 a0 2c 6a a3 :4 a0 63
1c 81 b0 :2 a0 7e a0 b4 2e
d :3 a0 6e :2 a0 b4 2e 7e 51
b4 2e 7e :2 a0 6e :2 a0 b4 2e
b4 2e 5a 7e 51 b4 2e 7e
:2 a0 6e :2 a0 b4 2e b4 2e 5a
7e 51 b4 2e 7e :2 a0 6e :2 a0
b4 2e b4 2e 65 b7 a4 b1
11 68 4f b1 b7 a4 11 a0
b1 56 4f 1d 17 b5 
da8
2
0 3 7 b 33 19 1d 20
21 29 2e 18 4f 3e 42 15
4a 3d 6b 5a 5e 3a 66 59
87 76 7a 56 82 75 a3 92
96 72 9e 91 bf ae b2 8e
ba ad e4 ca ce aa d6 d9
da df c9 109 ef f3 c6 fb
fe ff 104 ee 12e 114 118 eb
120 123 124 129 113 153 139 13d
110 145 148 149 14e 138 15a 15e
172 135 176 17a 17e 182 186 18a
18e 190 194 198 19a 1a6 1aa 1ac
1c8 1c4 1c3 1d0 1c0 1d5 1d9 1f2
1e1 1e5 1ed 1e0 20e 1fd 201 209
1dd 22a 215 219 21c 21d 225 1fc
246 235 239 241 1f9 25e 24d 251
259 234 27a 269 26d 275 231 292
281 285 28d 268 2ae 29d 2a1 2a9
265 2c6 2b5 2b9 2c1 29c 2e2 2d1
2d5 2dd 299 2fa 2e9 2ed 2f5 2d0
301 305 2cd 309 30e 30f 314 318
31c 320 324 328 32c 330 334 338
344 348 34c 350 354 358 364 368
36c 370 37c 380 383 386 387 38c
390 394 398 39c 3a0 3a4 3a8 3a9
3ae 3b2 3b6 3ba 3be 3c2 3c6 3ca
3ce 3d2 3d6 3da 3de 3e2 3e6 3ea
3ee 3f2 3f6 3fa 3fe 40a 40c 410
414 418 41c 420 424 428 42c 438
43a 43e 442 445 449 44d 451 455
459 45d 461 465 469 46d 471 475
479 47d 481 485 489 48d 491 495
4a1 4a5 4a9 4ad 4b1 4bd 4c1 4c5
4c9 4cd 4d1 4d5 4d9 4e5 4e9 4f5
4f7 4fb 4fd 509 50d 50f 52f 527
52b 526 536 547 53f 543 523 54e
55b 553 557 53e 562 573 56b 56f
53b 57a 587 57f 583 56a 58e 59f
597 59b 567 5a6 596 5ab 5af 5b3
5c7 5bb 5bf 5c3 5b7 5e3 5d2 5d6
5de 593 604 5ea 5ee 5f1 5f2 5fa
5ff 5d1 631 60f 613 5ce 617 61b
61f 624 62c 60e 638 63c 640 60b
644 648 64c 64f 652 656 65a 65c
660 664 669 66e 672 676 67a 67f
685 689 68c 690 694 698 69a 69b
6a0 6a3 6a7 6ab 6af 6b1 6b2 6b7
6ba 6be 6c2 6c6 6c8 6c9 6ce 6d1
6d5 6d9 6dd 6df 6e0 6e5 6e8 6ec
6f0 6f4 6f6 6f7 6fc 6ff 703 707
70b 70d 70e 713 717 71a 71c 721
725 72a 72c 730 737 73b 73f 744
746 748 74c 74e 75a 75e 760 77c
778 777 784 791 78d 774 799 78c
79e 7a2 7bb 7aa 7ae 7b6 789 7d3
7c2 7c6 7ce 7a9 800 7de 7e2 7a6
7e6 7ea 7ee 7f3 7fb 7dd 81c 80b
80f 817 7da 834 823 827 82f 80a
850 83f 843 84b 807 86c 857 85b
85e 85f 867 83e 888 877 87b 883
83b 8a0 88f 893 89b 876 8bc 8ab
8af 8b7 873 8d4 8c3 8c7 8cf 8aa
8f0 8df 8e3 8eb 8a7 908 8f7 8fb
903 8de 924 913 917 91f 8db 93c
92b 92f 937 912 958 947 94b 953
90f 970 95f 963 96b 946 98c 97b
97f 987 943 9a4 993 997 99f 97a
9c0 9af 9b3 9bb 977 9d8 9c7 9cb
9d3 9ae 9f4 9e3 9e7 9ef 9ab a0c
9fb 9ff a07 9e2 a28 a17 a1b a23
9df a40 a2f a33 a3b a16 a5c a4b
a4f a57 a13 a74 a63 a67 a6f a4a
a90 a7f a83 a8b a47 a7b a97 ab0
aac aab ab8 ac5 ac1 aa8 acd ad6
ad2 ac0 ade abd ae3 ae6 aea aee
af2 af6 afa afe b02 b06 b0a b0e
b12 b16 b1a b1e b22 b26 b2a b2e
b32 b36 b3a b3e b42 b46 b4a b4e
b52 b56 b5a b5e b62 b66 b6a b6e
b72 b76 b7a b7e b82 b86 b8a b8e
b92 b96 b9a b9e ba2 ba6 baa bae
bb2 bb6 bba bbe bc2 bc6 bd2 bd7
bd9 be5 be9 beb bef bf3 bf6 bfb
bfe c02 c03 c08 c09 c0e c12 c16
c17 c1c c20 c24 c28 c2c c30 c34
c38 c3c c40 c4c c50 c54 c58 c5c
c68 c6c c70 c74 c78 c84 c88 c8c
c90 c9c ca0 ca3 ca6 ca7 cac cb0
cb4 cb8 cbc cc0 cc4 cc8 cc9 cce
cd2 cd6 cda cde ce2 ce6 cea cee
cf2 cf6 cfa cfe d02 d06 d0a d0e
d12 d16 d22 d26 d2a d2d d2e d33
d36 d3a d3e d41 d45 d46 d4b d4e
d51 d54 d55 d5a d5e d60 d64 d68
d6b d6e d6f d74 d78 d7a d7e d82
d85 d89 d8d d90 d91 d96 d99 d9d
da1 da4 da8 da9 dae db1 db4 db7
db8 dbd dc1 dc3 dc7 dcb dce dd1
dd2 dd7 ddb ddd de1 de5 de8 dec
df0 df3 df4 df9 dfc e00 e04 e07
e0b e0c e11 e14 e17 e1a e1b e20
e24 e26 e2a e2e e31 e34 e35 e3a
e3e e40 e44 e48 e4b e4f e53 e56
e57 e5c e5f e63 e67 e6a e6e e6f
e74 e77 e7a e7d e7e e83 e87 e89
e8d e91 e94 e97 e98 e9d ea1 ea3
ea7 eab eae eb2 eb6 eb9 eba ebf
ec2 ec6 eca ecd ed1 ed2 ed7 eda
edd ee0 ee1 ee6 eea eec ef0 ef4
ef7 efa efb f00 f04 f06 f0a f0e
f11 f15 f19 f1c f1d f22 f25 f29
f2d f30 f34 f35 f3a f3d f40 f43
f44 f49 f4d f4f f53 f57 f5a f5d
f5e f63 f67 f69 f6d f71 f74 f78
f7c f80 f84 f88 f8c f90 f94 f98
f9c fa0 fa4 fa8 fac fb0 fb4 fb8
fbc fc0 fc4 fd0 fd2 fd6 fda fde
fe2 fe6 fea ff6 ffa ffe 1002 1006
100a 100e 1012 1016 1022 1024 1028 102c
102f 1033 1037 103b 103f 1043 104b 1046
104f 1053 1056 105a 105e 1062 1064 1068
106c 1070 1074 1078 107c 1081 1086 108a
108e 1092 1096 109b 10a1 10a5 10a9 10ad
10b1 10b5 10b9 10bd 10c1 10c5 10c9 10cd
10d1 10d5 10d9 10e5 10e7 10eb 10f2 10f6
10fa 10ff 1101 1105 1109 110d 1111 1115
1119 111d 1121 1125 1129 112d 1131 1135
1139 113d 1141 114d 1151 1155 1159 115d
1169 116d 1171 1175 1179 1185 1189 1195
1197 119b 119d 11a9 11ad 11af 11cb 11c7
11c6 11d3 11c3 11d8 11dc 11f5 11e4 11e8
11f0 11e3 11fc 1200 1204 1208 1214 1218
121c 1220 1224 1228 122c 1230 1234 1238
123c 1248 124c 1250 1254 1258 125c 1260
126c 1270 1274 1278 127c 1280 1284 1288
128c 1290 1294 12a0 12a4 12a8 12ac 12b0
12b4 12b8 12bc 12c0 12c4 12c8 12d4 12d8
12dc 12e0 12e4 12e8 12ec 12f0 12f4 12f8
12fc 1308 130c 11e0 1318 131c 131e 132a
132e 1330 1344 1345 1349 1362 1351 1355
135d 1350 1369 136d 1371 1375 1381 1385
1389 138d 1391 1395 1399 139d 13a1 13ad
13b1 13b5 13b9 13bd 13c9 13cd 13d1 13d5
13d9 13e5 13e9 13ed 13f1 13f5 1401 1405
1409 140d 1411 1415 1419 141d 1421 142d
1431 1435 1439 143d 1449 144d 1451 1455
1459 145d 1461 1465 1469 1475 1479 147d
1481 1485 1491 1495 1499 149d 14a1 14a5
14a9 14ad 14b1 14bd 14c1 14c5 14c9 14cd
14d9 14dd 134d 14e9 14ed 14ef 14fb 14ff
1501 151d 1519 1518 1525 1515 152a 152e
1547 1536 153a 1542 1535 1563 1552 1556
155e 1532 157b 156a 156e 1576 1551 1597
1586 158a 1592 154e 1582 159e 15a2 15a6
15aa 15ae 15b2 15b6 15ba 15be 15c2 15c6
15d2 15d6 15da 15de 15e2 15e6 15ea 15f6
15fa 15fe 1601 1602 1607 160b 160f 1613
1617 161b 161f 162b 162f 1632 1635 1636
163b 163f 1643 1647 164b 164f 1653 1657
165b 165f 1663 166f 1671 1675 1678 167c
1680 1684 1688 1694 1698 169c 16a0 16a4
16a8 16ac 16b8 16ba 16be 16c1 16c5 16d1
16d3 16d7 16d9 16e5 16e9 16eb 16ff 1700
1704 171d 170c 1710 1718 170b 1739 1728
172c 1734 1708 1751 1740 1744 174c 1727
176d 175c 1760 1768 1724 1758 1774 1778
177c 1780 1784 1788 178c 1790 1794 1798
179c 17a8 17ac 17b0 17b4 17b8 17bc 17c0
17cc 17d0 17d4 17d7 17d8 17dd 17e1 17e5
17e9 17ed 17f1 17f5 17f9 17fd 1801 1805
1811 1815 1818 181b 181c 1821 1825 1829
182d 1831 1835 1839 183d 1841 1845 1849
1855 1857 185b 185e 1862 1866 186a 186e
187a 187e 1882 1886 188a 188e 1892 189e
18a0 18a4 18a7 18ab 18b7 18b9 18bd 18bf
18cb 18cf 18d1 18e5 18e6 18ea 18ee 18f2
190b 1907 1906 1913 1903 1918 191b 191f
1923 1927 192b 192f 1933 1937 193b 193f
1943 1947 1953 1958 195a 1966 196a 196c
1970 1989 1985 1984 1991 1981 1996 1999
199d 19a1 19a5 19a9 19ad 19b1 19bd 19c2
19c4 19d0 19d4 19f4 19da 19de 19e2 19e7
19ef 19d9 1a10 19ff 1a03 1a0b 19d6 1a28
1a17 1a1b 1a23 19fe 1a44 1a33 1a37 1a3f
19fb 1a5c 1a4b 1a4f 1a57 1a32 1a78 1a67
1a6b 1a73 1a2f 1a90 1a7f 1a83 1a8b 1a66
1aac 1a9b 1a9f 1aa7 1a63 1ac4 1ab3 1ab7
1abf 1a9a 1ae0 1acf 1ad3 1adb 1a97 1af8
1ae7 1aeb 1af3 1ace 1b15 1b03 1acb 1b07
1b08 1b10 1b02 1b1c 1b20 1aff 1b24 1b29
1b2a 1b2f 1b33 1b34 1b39 1b3d 1b41 1b45
1b49 1b4d 1b51 1b55 1b59 1b5d 1b69 1b6d
1b71 1b75 1b79 1b7d 1b89 1b8d 1b91 1b95
1ba1 1ba5 1ba9 1bad 1bae 1bb0 1bb4 1bb6
1bba 1bbe 1bc1 1bc4 1bc5 1bca 1bce 1bd2
1bd6 1bda 1bdd 1bde 1be0 1be4 1be6 1bea
1bee 1bf2 1bf6 1bfa 1bfe 1c02 1c06 1c0a
1c16 1c1a 1c1e 1c22 1c26 1c2a 1c2e 1c32
1c36 1c3a 1c3e 1c42 1c46 1c4a 1c4e 1c5a
1c5c 1c60 1c67 1c69 1c6d 1c70 1c72 1c76
1c7d 1c81 1c84 1c87 1c88 1c8d 1c91 1c95
1c99 1c9d 1ca1 1ca5 1ca9 1caa 1caf 1cb3
1cb7 1cbb 1cbf 1cc3 1cc7 1ccb 1ccf 1cd3
1cd7 1cdb 1cdf 1ce3 1ce7 1ceb 1cef 1cf3
1cf7 1cfb 1cff 1d0b 1d0d 1d11 1d15 1d19
1d1d 1d21 1d25 1d29 1d2d 1d39 1d3b 1d3f
1d43 1d46 1d4a 1d4e 1d52 1d56 1d5a 1d5e
1d62 1d66 1d6a 1d6e 1d72 1d76 1d7a 1d7e
1d82 1d86 1d8a 1d8e 1d9a 1d9e 1daa 1dac
1db0 1db2 1dbe 1dc2 1dc4 1dd8 1dd9 1ddd
1de1 1de5 1dfe 1dfa 1df9 1e06 1df6 1e0b
1e0e 1e12 1e16 1e1a 1e1e 1e22 1e26 1e2a
1e36 1e3b 1e3d 1e49 1e4d 1e64 1e53 1e57
1e5f 1e52 1e80 1e6f 1e73 1e7b 1e4f 1e98
1e87 1e8b 1e93 1e6e 1e9f 1ea3 1ea7 1eab
1eb7 1ebb 1ebf 1e6b 1eca 1ec5 1ece 1ed2
1ed6 1edb 1ee0 1ee4 1ee8 1eed 1ef0 1ef4
1ef8 1efd 1eff 1f03 1f07 1f09 1f0d 1f10
1f14 1f18 1f1d 1f22 1f26 1f2a 1f2f 1f32
1f36 1f3a 1f3f 1f41 1f45 1f49 1f4b 1f4f
1f52 1f56 1f5a 1f5f 1f61 1f65 1f69 1f6d
1f71 1f75 1f79 1f7d 1f81 1f85 1f89 1f95
1f99 1f9d 1fa1 1fa5 1fa9 1fad 1fb9 1fbd
1fc9 1fcb 1fcf 1fd1 1fdd 1fe1 1fe3 2003
1ffb 1fff 1ffa 200a 1ff7 200f 2013 2017
201b 2034 2030 202f 203c 202c 2041 2044
2048 204c 2050 2054 2058 205c 2060 2064
2068 206c 2070 2074 2078 207c 2088 208d
208f 209b 209f 20a1 20a5 20be 20ba 20b9
20c6 20d3 20cf 20b6 20db 20e4 20e0 20ce
20ec 20cb 20f1 20f4 20f8 20fc 2100 2104
2108 210c 2110 2114 2118 211c 2120 2124
2128 212c 2130 2134 2138 213c 2140 2144
2148 214c 2150 2154 2158 215c 2160 2164
2168 216c 2170 2174 2178 217c 2180 2184
2188 218c 2190 2194 2198 219c 21a0 21a4
21a8 21ac 21b0 21b4 21b8 21bc 21c0 21c4
21c8 21cc 21d0 21d4 21d8 21dc 21e0 21e4
21e8 21ec 21f0 21f4 2200 2205 2207 2213
2217 222e 221d 2221 2229 221c 224a 2239
223d 2245 2219 2276 2251 2255 2259 225c
2260 2264 2269 2271 2238 2293 2281 2235
2285 2286 228e 2280 22af 229e 22a2 22aa
227d 22c7 22b6 22ba 22c2 229d 22e3 22d2
22d6 22de 229a 22fb 22ea 22ee 22f6 22d1
2317 2306 230a 2312 22ce 232f 231e 2322
232a 2305 234b 233a 233e 2346 2302 2363
2352 2356 235e 2339 237f 236e 2372 237a
2336 2397 2386 238a 2392 236d 23b3 23a2
23a6 23ae 236a 23cb 23ba 23be 23c6 23a1
23e7 23d6 23da 23e2 239e 23ff 23ee 23f2
23fa 23d5 241b 240a 240e 2416 23d2 2433
2422 2426 242e 2409 244f 243e 2442 244a
2406 2467 2456 245a 2462 243d 2483 2472
2476 247e 243a 249b 248a 248e 2496 2471
24b7 24a6 24aa 24b2 246e 24cf 24be 24c2
24ca 24a5 24eb 24da 24de 24e6 24a2 2503
24f2 24f6 24fe 24d9 251f 250e 2512 251a
24d6 2537 2526 252a 2532 250d 2553 2542
2546 254e 250a 256b 255a 255e 2566 2541
2587 2576 257a 2582 253e 259f 258e 2592
259a 2575 25bb 25aa 25ae 25b6 2572 25d3
25c2 25c6 25ce 25a9 25da 25de 25e2 25e6
25ea 25ee 25f2 25f6 25fa 25fe 2602 2606
260a 2616 261a 261e 25a6 2629 2624 262d
2631 2635 2639 263d 2641 2645 2649 264d
2651 2656 265b 265f 2663 2667 266c 266f
2673 2677 267c 267e 2682 2686 2688 268c
268f 2693 2697 269b 269f 26a3 26a7 26ab
26af 26b3 26b8 26bd 26c1 26c5 26c9 26ce
26d1 26d5 26d9 26de 26e0 26e4 26e8 26ea
26ee 26f1 26f5 26f9 26fe 2700 2704 2708
270b 2710 2713 2717 2718 271d 2720 2725
2726 272b 272e 2732 2733 2738 2739 273e
2742 2746 274a 274e 2752 2756 275a 275e
276a 276e 2772 2776 277a 2786 278a 278d
2791 2795 2798 279c 27a0 27a3 27a7 27ab
27ae 27b2 27b6 27ba 27be 27c2 27c6 27ce
27c9 27d2 27d6 27da 27de 27e2 27e6 27ea
27ee 27f2 27f7 27fc 2800 2804 2808 280c
2811 2817 281b 281f 2822 2827 282a 282e
282f 2834 2837 283c 283d 2842 2845 2849
284a 284f 2852 2857 2858 285d 2860 2864
2865 286a 286d 2872 2873 2878 287b 287f
2880 2885 2888 288d 288e 2893 2896 289a
289b 28a0 28a1 28a6 28aa 28ae 28b1 28b5
28b6 28bb 28bf 28c3 28c7 28ca 28ce 28cf
28d4 28d8 28dc 28e0 28e3 28e8 28eb 28ef
28f3 28f6 28f7 28f9 28fa 28ff 2900 2905
2909 290d 2910 2914 2915 291a 291e 2922
2926 2929 292d 292e 2933 2937 293b 293f
2942 2946 2947 294c 2950 2954 2958 295b
295e 295f 2964 2968 296c 2970 2974 2978
297c 2980 2984 2988 298c 2990 2994 2998
299c 29a0 29a4 29a8 29b4 29b6 29ba 29c1
29c5 29c9 29ce 29d0 29d4 29d7 29da 29db
29e0 29e4 29e8 29eb 29ef 29f0 29f5 29f8
29fb 29fc 2a01 2a05 2a07 2a0b 2a0e 2a12
2a15 2a18 2a19 2a1e 2a22 2a26 2a29 2a2a
2a2f 2a32 2a36 2a3a 2a3d 2a41 2a42 2a47
2a4a 2a4d 2a50 2a51 2a56 2a5a 2a5c 2a60
2a64 2a67 2a6a 2a6b 2a70 2a74 2a76 2a7a
2a7e 2a81 2a85 2a89 2a8c 2a8d 2a92 2a95
2a99 2a9d 2aa0 2aa4 2aa5 2aaa 2aad 2ab0
2ab3 2ab4 2ab9 2abd 2abf 2ac3 2ac7 2aca
2acd 2ace 2ad3 2ad7 2ad9 2add 2ae1 2ae4
2ae8 2aec 2aef 2af0 2af5 2af8 2afc 2b00
2b03 2b07 2b08 2b0d 2b10 2b13 2b16 2b17
2b1c 2b20 2b22 2b26 2b2a 2b2d 2b30 2b31
2b36 2b3a 2b3c 2b40 2b44 2b47 2b4b 2b4f
2b52 2b53 2b58 2b5b 2b5f 2b63 2b66 2b6a
2b6b 2b70 2b73 2b76 2b79 2b7a 2b7f 2b83
2b85 2b89 2b8d 2b90 2b93 2b94 2b99 2b9d
2b9f 2ba3 2ba7 2baa 2bae 2bb2 2bb5 2bb6
2bbb 2bbe 2bc2 2bc6 2bc9 2bcd 2bce 2bd3
2bd6 2bd9 2bdc 2bdd 2be2 2be6 2be8 2bec
2bf0 2bf3 2bf6 2bf7 2bfc 2c00 2c02 2c06
2c0a 2c0d 2c11 2c15 2c18 2c19 2c1e 2c21
2c25 2c29 2c2c 2c30 2c31 2c36 2c39 2c3c
2c3f 2c40 2c45 2c49 2c4b 2c4f 2c53 2c56
2c59 2c5a 2c5f 2c63 2c65 2c69 2c6d 2c70
2c74 2c78 2c7c 2c80 2c84 2c88 2c8c 2c90
2c94 2c98 2c9c 2ca0 2ca4 2ca8 2cac 2cb0
2cb4 2cb8 2cbc 2cc0 2cc4 2cc8 2ccc 2cd0
2cd4 2cd8 2cdc 2ce0 2ce4 2ce8 2cf4 2cf6
2cfa 2cfe 2d02 2d06 2d0a 2d0e 2d12 2d16
2d1a 2d1e 2d22 2d26 2d2a 2d2e 2d32 2d36
2d3a 2d3e 2d4a 2d4c 2d50 2d54 2d57 2d5b
2d67 2d6b 2d6f 2d72 2d73 2d78 2d7c 2d81
2d82 2d86 2d8b 2d8c 2d8e 2d92 2d95 2d97
2d9b 2d9d 2da9 2dad 2daf 2dc3 2dc4 2dc8
2de1 2dd0 2dd4 2ddc 2dcf 2dfd 2dec 2df0
2df8 2dcc 2de8 2e04 2e08 2e0b 2e10 2e11
2e16 2e1a 2e1e 2e22 2e2e 2e32 2e36 2e3a
2e3e 2e42 2e46 2e52 2e56 2e5a 2e5e 2e62
2e6e 2e72 2e76 2e7a 2e7e 2e8a 2e8e 2e92
2e96 2e9a 2e9e 2ea2 2eae 2eb2 2eb6 2eba
2ebe 2ec2 2ec6 2ed2 2ed6 2ee2 2ee4 2ee8
2eea 2ef6 2efa 2efc 2f10 2f11 2f15 2f19
2f1d 2f36 2f32 2f31 2f3e 2f2e 2f43 2f46
2f4a 2f4e 2f52 2f56 2f5a 2f5e 2f62 2f6e
2f73 2f75 2f81 2f85 2f9c 2f8b 2f8f 2f97
2f8a 2fb8 2fa7 2fab 2fb3 2f87 2fd0 2fbf
2fc3 2fcb 2fa6 2fec 2fdb 2fdf 2fe7 2fa3
2fd7 2ff3 2ff7 2ffa 2fff 3000 3005 3009
300d 3011 301d 3021 3024 3028 302c 302f
3032 3033 3038 303c 3040 3044 3048 3050
304b 3054 3058 305c 3060 3063 3068 306b
306f 3070 3075 3076 307b 307f 3083 3088
308d 3091 3095 3099 309e 30a4 30a8 30ac
30af 30b0 30b5 30b9 30bd 30c1 30c5 30cb
30cd 30d1 30d4 30d8 30dc 30df 30e2 30e3
30e8 30ec 30ee 30f2 30f9 30fd 3101 3106
3108 310c 3110 3113 3118 311b 311f 3120
3125 3126 312b 312f 3132 3135 3136 313b
313f 3143 3147 314b 314f 3153 3157 315b
315f 3163 316f 3173 3177 317b 317f 3183
3187 3193 3197 31a3 31a5 31a9 31ac 31ae
31b2 31b4 31c0 31c4 31c6 31e6 31de 31e2
31dd 31ed 31da 31f2 31f6 320f 31fe 3202
320a 31fd 322b 321a 321e 3226 31fa 3243
3232 3236 323e 3219 325f 324e 3252 325a
3216 3277 3266 326a 3272 324d 3293 3282
3286 328e 324a 32ab 329a 329e 32a6 3281
32c7 32b6 32ba 32c2 327e 32df 32ce 32d2
32da 32b5 32e6 32ea 3303 32ff 32b2 330b
32fe 32fb 3310 3314 3318 331c 3320 3324
3328 332c 3338 333d 333f 334b 334f 3351
3355 3359 335d 3369 336d 3370 3374 3378
337c 3380 3384 3388 338c 3398 339a 339e
33a2 33a5 33a9 33ab 33ac 33b1 33b5 33b7
33c3 33c5 33c9 33cc 33cf 33d0 33d5 33d9
33dd 33e0 33e5 33e6 33eb 33ef 33f3 33f7
33ff 33fa 3403 3407 340b 3410 3415 3419
341d 3422 3425 3429 342d 3430 3435 3436
343b 343f 3443 3444 3449 344d 3451 3456
3458 345c 3460 3464 3466 346a 346e 3473
3475 3479 347d 3481 348d 3491 3495 3499
349d 34a1 34a5 34b1 34b5 34b9 34bd 34c1
34c2 34c4 34c8 34cc 34cf 34d2 34d3 34d8
34dc 34e0 34e3 34e8 34e9 34ee 34f2 34f6
34fa 34fb 3500 3504 3508 350c 3510 3514
3518 3524 3528 352b 352e 352f 3534 3538
353c 353f 3544 3545 354a 354e 3553 3554
3558 355c 355f 3564 3565 356a 356e 3572
3576 3578 357c 3580 3584 3586 358a 358e
3591 3593 3597 359b 359f 35a1 35a5 35a9
35ac 35ae 35b2 35b6 35b9 35bb 35bf 35c3
35c7 35cb 35cf 35d3 35df 35e3 35e6 35e9
35ea 35ef 35f3 35f7 35fb 3607 360b 360f
3613 3617 361b 361f 3623 362f 3633 3637
363b 363f 3640 3642 3646 364a 364d 3650
3651 3656 365a 365e 3661 3666 3667 366c
3670 3674 3677 367b 367e 3683 3684 3688
368c 368f 3693 3696 369b 369c 36a0 36a4
36a7 36ac 36ad 36b2 36b6 36ba 36bd 36c1
36c4 36c8 36c9 36ce 36d0 36d4 36d8 36dc
36e0 36e4 36e8 36f4 36f6 36fa 36fe 3702
3706 3708 3709 370e 3712 3714 3720 3722
3724 3728 372c 372f 3731 3735 3739 373d
373f 3743 3747 374a 374c 3750 3754 3757
375b 375f 3763 3767 3773 3778 377c 377e
3782 3784 3790 3794 3796 37b6 37ae 37b2
37ad 37bd 37aa 37c2 37c6 37df 37ce 37d2
37da 37cd 37fb 37ea 37ee 37f6 37ca 3813
3802 3806 380e 37e9 382f 381e 3822 382a
37e6 3847 3836 383a 3842 381d 3863 3852
3856 385e 381a 387b 386a 386e 3876 3851
3897 3886 388a 3892 384e 3882 389e 38a2
38a6 38b2 38b6 38b9 38bd 38c1 38c4 38c8
38cc 38d0 38d4 38d8 38dc 38e0 38ec 38f0
38f3 38f6 38f7 38fc 3900 3904 3908 390c
3910 3914 3920 3924 3927 392a 392b 3930
3934 3938 393c 3940 3944 3948 3954 3958
395b 395e 395f 3964 3968 396c 3970 3974
3978 397c 3980 3984 3988 398c 3998 399a
399e 39a2 39a6 39a8 39ac 39b0 39b3 39b5
39b9 39bd 39c1 39c3 39c7 39cb 39ce 39d0
39d4 39d8 39dc 39de 39e2 39e6 39e9 39ed
39f1 39f4 39f9 39fc 3a00 3a01 3a06 3a07
3a0c 3a10 3a14 3a17 3a1c 3a1f 3a23 3a24
3a29 3a2a 3a2f 3a33 3a37 3a3a 3a3f 3a42
3a46 3a47 3a4c 3a4d 3a52 3a56 3a5a 3a5e
3a62 3a66 3a6a 3a76 3a7b 3a7f 3a81 3a85
3a87 3a93 3a97 3a99 3a9d 3ab9 3ab5 3ab4
3ac1 3ace 3aca 3ab1 3ad6 3ac9 3adb 3adf
3ae3 3ae7 3b10 3aef 3af3 3af7 3afb 3aff
3b03 3b0b 3ac6 3aeb 3b17 3b1b 3b1e 3b22
3b23 3b28 3b2c 3b30 3b34 3b38 3b3d 3b41
3b45 3b46 3b4b 3b4e 3b51 3b52 3b57 3b5a
3b5e 3b62 3b67 3b6b 3b6f 3b70 3b75 3b76
3b7b 3b7e 3b81 3b84 3b85 3b8a 3b8d 3b91
3b95 3b9a 3b9e 3ba2 3ba3 3ba8 3ba9 3bae
3bb1 3bb4 3bb7 3bb8 3bbd 3bc0 3bc4 3bc8
3bcd 3bd1 3bd5 3bd6 3bdb 3bdc 3be1 3be5
3be7 3beb 3bed 3bf9 3bfd 3bff 3c01 3c03
3c07 3c13 3c17 3c19 3c1c 3c1e 3c1f 3c28 
da8
2
0 1 9 15 1 11 1a 19
11 21 11 :2 1 :2 1a 24 1a :2 1
:2 b 15 b :2 1 :2 20 2a 20 :2 1
:2 17 21 17 :2 1 :2 11 1b 11 :2 1
:2 12 1c 1d :2 1c 12 :2 1 :2 13 1d
1e :2 1d 13 :2 1 :2 10 1a 1b :2 1a
10 :2 1 :2 16 20 21 :2 20 16 :2 1
a 15 0 1c :2 1 4 b 4
:2 1 5 :4 1 b 1f 2c :2 1f 1e
:2 1 4 :3 10 :2 4 :3 11 :2 4 b 14
13 :2 b :2 4 :3 11 :2 4 :3 f :2 4 :3 c
:2 4 :3 d :2 4 :3 1a :2 4 :3 16 :2 4 :3 16
:2 4 :3 1a :2 4 :2 10 19 :2 4 b 1c
26 2b 3d 44 9 17 9 4
b 12 2c 39 4a 4 b 18
2a 4 7 12 15 :2 12 7 16
1f 29 40 53 66 :2 7 20 27
35 46 4f 3 c 16 2d 40
53 e 1c 29 3 c 16 2d
40 53 7 18 20 27 35 46
4f e 1c 29 7 :4 4 1e 25
b 14 1c 28 36 44 e 1b
23 33 41 c d 1a b 2a
37 29 5 10 17 d 1f 4
10 17 2c 3d f 20 32 :3 4
:6 1 b 1a 22 26 :2 1a 2e 37
3b :2 2e 43 59 5d :2 43 5 17
1b :2 5 23 35 39 :2 23 5 1b
1f :2 5 19 :2 1 4 9 17 1b
17 :2 4 :3 e :2 4 d 16 15 d
1e d :2 4 a 14 a :2 1a :3 a
:2 4 9 :2 17 4 8 d 10 12
d 4 d 1c :3 7 11 1b 11
7 c f 16 22 16 11 :2 a
f 16 2f 16 11 :2 a f 16
2b 16 11 :2 a f 16 2b 16
11 :2 a b 12 1d 12 d :2 6
f 16 2f 16 11 :2 a :6 7 12
8 :2 4 a :2 4 :6 1 b 1b 28
:3 1b 26 :2 1b 1a :2 1 4 :3 11 :2 4
:3 9 :2 4 9 13 9 :2 1e :3 9 :2 4
:3 7 :2 4 :3 7 :2 4 :3 9 :2 4 b 14
13 :2 b :2 4 :3 11 :2 4 :3 16 :2 4 :3 f
:2 4 :3 c :2 4 :3 d :2 4 :3 1a :2 4 :3 16
:2 4 :3 16 :2 4 :3 1a :2 4 :3 11 :2 4 :3 12
:2 4 :3 1f :2 4 :3 1b :2 4 :3 1b :2 4 :3 1f
:2 4 :3 12 :2 4 :3 13 :2 4 :3 20 :2 4 :3 1c
:2 4 :3 1c :2 4 :3 20 :2 4 b e 1a
:3 e 16 :3 e 1a :2 e d :2 4 e
16 22 2b 34 3c 44 10 16
10 16 1a 20 2e 34 44 4a
57 5d 6d 73 82 1a 20 30
36 48 4e 5d 63 75 7b 8c
16 1e 2a 38 14 1b 15 1f
25 16 1e 2a 38 14 1f f
15 22 28 d 18 10 7 :6 4
:2 10 19 3a 3d :2 19 :3 4 19 :2 4
b 1c 26 2b 3d 44 9 17
9 4 10 17 d 1f 4 10
17 d 1f 4 b 18 2a 4
7 12 15 :2 12 7 16 1f 29
40 53 66 :2 7 e 1c 25 2f
46 59 6c c 1f 2d 3c 58
70 88 c 13 d 17 7 b
16 :3 13 :2 a 1c 24 26 :2 1c 1b
33 34 :2 1b a 24 a 1b 22
23 :2 1b a :4 7 b 17 :3 14 :2 a
1d 26 28 :2 1d 1c 36 37 :2 1c
a 26 a 1c 24 25 :2 1c a
:4 7 b 24 :3 21 :2 a 2a 40 42
:2 2a 29 5d 5e :2 29 a 40 a
29 3e 3f :2 29 a :4 7 b 20
:3 1d :2 a 26 38 3a :2 26 25 51
52 :2 25 a 38 a 25 36 37
:2 25 a :4 7 b 20 :3 1d :2 a 26
38 3a :2 26 25 51 52 :2 25 a
38 a 25 36 37 :2 25 a :4 7
b 24 :3 21 :2 a 2a 40 42 :2 2a
29 5d 5e :2 29 a 40 a 29
3e 3f :2 29 a :4 7 13 1a 28
39 47 5a 63 6d 84 97 aa
e 1c 2a 3d 4c 5c 79 92
ab 7 18 e :2 c 13 d 17
7 13 1a 28 39 47 e 1c
2a 7 :5 4 9 c 1a 25 :2 9
4 8 d 10 18 d 4 d
15 1d 23 27 2b :4 7 11 14
11 7 13 1a d 1e 26 32
3e 4c 14 22 2a 30 34 38
7 18 8 :2 4 a :2 4 b 12
24 2a 2c 12 14 22 2c 12
14 21 23 d f 21 4 10
17 d 17 4 10 17 d 1f
:3 4 :6 1 b 25 2d :2 25 24 :2 1
4 :3 11 4 b 20 32 4 10
17 d 20 c 13 d 1f d
1a 4 10 17 d 1f d 1a
4 10 17 d 23 33 3a d
1f d 1a 4 10 17 d 23
33 3a 22 34 d 1a 4 10
17 d 23 33 3a 22 34 d
1a :3 4 :6 1 b 0 :2 1 4 :3 11
4 b 20 32 4 10 17 d
20 c 13 d 1f 4 10 17
d 1f 4 10 17 d 1f 4
10 17 d 1f 4 10 17 d
23 33 3a 22 34 4 10 17
d 1f 4 10 17 d 23 33
3a 22 34 4 10 17 d 1f
4 10 17 d 23 33 3a 22
34 4 10 17 d 1f :3 4 :6 1
b 20 2d :2 20 1f :2 1 3 :3 14
:2 3 :3 14 :2 3 :3 14 :2 3 :3 14 3 a
1b 29 33 38 4a 51 a 18
2a a 3 :3 a 11 a 1c 3
6 1a :3 17 c 13 c 1c c
16 5 7 12 15 :2 12 e 15
e 19 25 3e 53 68 e 18
7 18 :2 4 c 13 c 1e fffa
c 13 28 39 d 1b fffa 2b
:4 3 :6 1 b 0 :2 1 3 :3 14 :2 3
:3 14 :2 3 :3 14 :2 3 :3 14 3 a 1b
29 33 38 4a 51 a 18 2a
a 3 :3 a 11 a 1c 3 6
1a :3 17 c 13 c 1c c 22
32 39 c 1e 5 7 12 15
:2 12 e 15 e 19 25 3e 53
68 e 20 7 18 :2 4 c 13
c 1e fffa c 13 28 39 d
1b fffa 2b :4 3 :6 1 b 0 :2 1
4 b 14 21 :2 14 13 :2 4 e
10 18 1a 2a 31 43 16 18
2a 7 :6 4 b 15 1d :2 15 14
:2 4 15 1f 27 29 36 7 :6 4
e 18 :3 e :2 4 :3 11 :2 4 :3 f :2 4
:3 11 :2 4 :3 f :2 4 :3 c :2 4 :3 d :2 4
:3 1a :2 4 :3 16 :2 4 :3 16 :2 4 :3 1a :2 4
b 14 13 :2 b :2 4 :2 10 19 :5 4
b 1c 26 2b 3d 44 9 17
9 4 b 12 2b 38 48 4
b 18 2a 4 8 14 1d :2 14
:2 4 a :2 13 :3 a e 1b 25 :2 2e
:2 1b :2 a 14 20 2c 33 14 1f
29 14 24 d 19 20 2f 37
43 51 18 21 29 33 1d 27
35 3f d a e a 2a :2 7
4 8 4 7 12 15 :2 12 7
16 1f 29 40 53 66 :2 7 13
1a 27 38 44 52 5b 65 7c
8f a2 e 1c 28 36 3f 49
60 73 86 7 18 13 1a 27
38 44 e 1c 28 7 :4 4 10
17 b 17 1f 2b 39 f 1b
23 2f 33 42 46 13 1a 17
23 :3 4 :6 1 b 0 :2 1 4 b
13 1f :2 13 12 :2 4 e c 13
d 1f 10 7 :6 4 :3 11 :2 4 :3 11
:2 4 :3 11 4 b 20 32 :2 4 9
11 :2 9 4 a 17 :2 4 8 10
8 :2 7 d :4 7 1a :2 4 a 17
:2 4 8 10 8 :2 7 d :4 7 1a
:3 4 a :2 4 10 17 a e c
13 d 1f b 18 4 10 17
d 1f d 1a :3 4 :6 1 b 17
1e 22 :2 17 16 :2 1 4 b 13
1f :2 13 12 :2 4 e 1a 28 31
3b 52 65 78 c 13 d 1f
10 7 :6 4 b 16 1e :2 16 26
2e :2 26 36 46 :2 36 15 :2 4 e
17 23 2c 18 21 31 3a 4c
55 64 6d 7f 88 18 21 2f
38 48 51 5e 67 77 80 e
17 23 2c 1b 22 39 46 4c
1b 22 39 46 4c c 13 25
d 16 23 2c d 16 23 2c
d 16 28 18 21 2f 38 48
51 5e 67 77 80 93 7 :6 4
:3 12 :2 4 :3 17 :2 4 d 17 d :2 22
:3 d :2 4 11 1a 19 :2 11 :2 4 :3 11
:2 4 :3 11 :2 4 :3 11 :2 4 :3 11 :2 4 :3 11
:2 4 :3 11 :2 4 :3 11 :2 4 :3 1c :2 4 :3 18
:2 4 :3 18 :2 4 :3 1c :2 4 :3 11 :2 4 :3 f
:2 4 :3 1c :2 4 :3 18 :2 4 :3 18 :2 4 :3 1c
:2 4 :3 12 :2 4 :3 13 :2 4 :3 20 :2 4 :3 1c
:2 4 :3 1c :2 4 :3 20 :2 4 :3 11 :2 4 :3 11
:2 4 :3 11 :2 4 :3 11 :2 4 :3 14 :2 4 :3 1a
:2 4 :3 1b :2 4 :3 13 :2 4 :3 11 :2 4 e
4 b 1c 26 2b 3d 44 9
17 9 :2 4 9 11 :2 9 4 a
17 21 2d 38 44 5d 72 87
:3 4 8 10 8 :2 7 d :4 7 1a
:2 4 a 17 21 2d 38 44 5d
72 87 :3 4 8 10 8 :2 7 d
:4 7 1a :3 4 a :3 4 :2 10 19 20
23 :2 19 2c 2f :2 19 37 3a :2 19
:2 4 10 17 a 20 30 37 a
1c 4 10 17 a 1c :2 4 11
:2 4 19 :2 4 1e :2 4 15 :2 4 9
14 1e 27 :2 9 :2 4 d 1d 26
2e 1d 2a 39 :4 7 11 1c 11
:2 7 :2 13 a 15 18 :2 a 20 :3 a
16 19 :2 a 24 27 :2 a 33 36
:2 a 42 :3 a 18 1b :2 a 29 2c
:2 a 3f 42 :2 a :2 7 a 18 22
23 :2 18 :2 a 1b 25 26 :2 1b :2 a
:2 16 1f 30 33 39 47 :2 33 :2 1f
:3 a 1f 31 33 :2 1f a 3 1d
34 36 :2 1d :2 3 14 22 24 :2 14
3 a 17 21 23 :2 17 a 16
1d 6 12 1a 26 32 40 4c
11 1b 23 2c 39 45 51 a
4 8 1 4 a :2 4 7 11
13 :2 11 7 11 23 25 :2 11 3c
3e :2 11 7 15 :2 4 7 12 15
:2 12 b 18 :3 15 :2 a 1c 26 28
:2 1c 1b 32 33 :2 1b a 23 a
1b 24 25 :2 1b a :4 7 b 19
:3 16 :2 a 1d 28 2a :2 1d 1c 35
36 :2 1c a 25 a 1c 26 27
:2 1c a :4 7 b 26 :3 23 :2 a 2a
42 44 :2 2a 29 5c 5d :2 29 a
3f a 29 40 41 :2 29 a :4 7
b 22 :3 1f :2 a 26 3a 3c :2 26
25 50 51 :2 25 a 37 a 25
38 39 :2 25 a :4 7 b 22 :3 1f
:2 a 26 3a 3c :2 26 25 50 51
:2 25 a 37 a 25 38 39 :2 25
a :4 7 b 26 :3 23 :2 a 2a 42
44 :2 2a 29 5c 5d :2 29 a 3f
a 29 40 41 :2 29 a :4 7 13
1a 26 37 43 51 64 3 1f
3b 3 c 16 2d 40 53 e
1c 26 32 3e 55 3 23 3
12 22 3f 58 71 7 18 13
1a 26 37 43 51 64 11 2d
49 e 1c 26 32 3e 55 11
31 7 :6 4 6 f :3 d :6 7 25
:2 3 :6 1 b 0 :2 1 4 :3 11 :2 4
:3 b :2 4 :2 10 19 :2 4 b 20 32
4 b 1b 9 10 a 1c 4
10 17 2a 37 4 10 17 2e
3b 4 10 17 14 1b c 19
4 10 17 14 1b c 19 :3 4
:6 1 b 0 :2 1 4 b 13 1f
:2 13 12 :2 4 e c 13 d 1f
10 7 :6 4 :3 f :2 4 :3 11 :2 4 :3 11
:2 4 :3 e :2 4 :2 10 19 :2 4 b 20
32 :2 4 11 :2 4 e f :2 e :2 4
9 11 :2 9 :2 4 5 :2 11 1a 29
2c :2 1a :2 5 b 18 :2 5 7 11
19 11 7 8 14 :3 12 7 11
:3 7 33 :3 5 12 1c 1e :2 12 5
4 8 1 4 a :3 4 :2 10 19
25 28 :2 19 :2 4 7 e 11 :2 e
13 1a d 11 f 16 10 22
e 1c 7 13 1a f 21 d
1b :3 7 13 :2 4 :6 1 b 1b 27
2b :2 1b 1a :2 1 4 :3 11 :2 4 :3 b
:2 4 :3 14 :2 4 :3 17 :2 4 :3 e :2 4 :3 e
:2 4 :3 e :2 4 :3 e :2 4 :3 d :2 4 b
13 1f :2 13 12 :2 4 e c 13
d 1f 10 7 :5 4 b 20 32
:2 4 17 4 15 2a 26 2d 1e
30 7 4 c a 1d a 1a
:2 7 4 :3 1 7 17 19 :2 17 7
:2 13 1c :3 7 c 14 :2 c 7 d
1a :2 7 b 13 b :2 a :2 16 1f
:3 a 1e :3 a 10 :3 a 14 a 1d
a 10 :2 a 11 1e 2d a 11
23 32 39 4d 5f :2 a 16 22
2d :2 16 a d 16 19 :2 16 d
:2 19 22 :3 d 1d 28 :2 d :3 14 1b
14 26 d 10 23 25 :2 23 10
:2 1c 25 :6 10 :2 1c 25 :3 10 1a 10
27 10 1a 10 :4 d 1d d 17
d :4 a :4 7 1b :3 e 15 e 20
7 a 1d 1f :2 1d 11 1e 2d
a 11 15 28 37 3e 51 63
:2 a 16 22 2d :2 16 a d 16
19 :2 16 d :2 19 22 :3 d :2 14 :2 22
:3 d :2 14 :2 22 :3 d :2 19 22 :3 d :2 14
:2 22 2e :2 d 1d :3 17 1e 17 29
10 d 15 13 1d 13 23 :2 10
d :7 a 21 a 14 a :4 7 :5 4
9 20 2c 19 4 19 :6 1 b
1b 27 2b :2 1b 1a :2 1 4 :3 11
:2 4 :3 1b :2 4 :3 1b :2 4 :3 12 :2 4 :3 11
:2 4 :3 13 :2 4 :3 17 :2 4 :3 12 4 b
20 32 :2 4 1e :2 4 15 4 :3 b
12 b 1d 4 7 16 18 :2 16
:3 e 15 e 20 7 a 1d 1f
:2 1d :3 11 18 11 23 a d 1b
1e :2 1b 14 30 4c 14 2c 44
14 1b 14 26 d 20 d 27
d :4 a 21 a 24 a :4 7 1a
7 21 7 :5 4 :2 10 19 37 3a
:2 19 :3 4 :2 10 19 37 3a :2 19 :3 4
:2 10 19 2e 31 :2 19 :3 4 9 20
38 50 63 19 4 19 :7 1 a
16 24 :2 16 2f 3b :2 2f 15 46
4d :2 1 4 d 16 1d :3 16 d
:2 4 10 19 1b :2 10 :2 4 d :2 15
1e :3 d 27 28 :2 d 2b d :2 15
1f :5 d c 29 2a :2 c 2d d
:2 15 21 :3 d :2 c b 2b 2c :2 b
2f d :2 15 21 :3 d :2 b 4 :a 1
5 :6 1 
da8
4
0 :3 1 :8 7 :6 8
:6 9 :6 a :6 b :6 c
:9 d :9 e :9 f :9 10
:3 13 0 :3 13 :3 15
:2 14 16 :4 13 :8 1c
:5 1d :5 1e :7 1f :5 22
:5 23 :5 24 :5 25 :5 26
:5 27 :5 28 :5 29 :6 2b
:6 2d :2 2e 2f 2d
:6 31 :4 33 :5 36 :9 37
:5 39 :6 3a :3 3b :6 3c
39 36 :5 3e :3 3f
3e 3d :3 36 :2 47
:6 48 :5 49 4a :2 4b
:3 4c 4d 47 :2 50
:2 51 50 :4 53 :3 54
53 :2 56 :2 2a :4 1c
:10 59 :a 5a :5 5b :3 59
:6 5c :5 5d :8 5e :a 5f
:5 61 :6 62 :4 63 :5 64
65 :7 66 :7 67 :7 68
:7 69 :7 6a :7 6b :6 65
62 6d 62 :4 6e
:2 60 :4 59 :5 7a :4 7b
:3 7a :5 7c :5 7d :a 7e
:5 7f :5 80 :5 81 :7 82
:5 83 :5 84 :5 85 :5 86
:5 87 :5 88 :5 89 :5 8a
:5 8b :5 8c :5 8d :5 8e
:5 8f :5 90 :5 91 :5 92
:5 93 :5 94 :5 95 :5 96
:5 97 :6 98 :4 99 :4 9a
:3 98 :7 9b :2 9e :2 9f
:b a0 :b a1 :4 a3 :2 a4
:3 a5 :4 a7 :2 a8 :4 aa
:2 ac ad 9b :5 98
:a af :4 b2 :6 b4 :2 b5
b6 b4 :2 b8 :2 b9
b8 :2 bb :2 bc bb
:4 be :5 c0 :9 c1 :7 c3
:7 c4 :2 c5 :2 c6 c3
:6 c8 :c c9 c8 :7 cb
ca :3 c8 :6 cd :c ce
cd :7 d0 cf :3 cd
:6 d2 :c d3 d2 :7 d5
d4 :3 d2 :6 d7 :c d8
d7 :7 da d9 :3 d7
:6 dc :c dd dc :7 df
de :3 dc :6 e1 :c e2
e1 :7 e4 e3 :3 e1
:b e7 :9 e8 e7 c0
ea eb :2 ec :2 ed
ea :5 ef :3 f0 ef
e9 :3 c0 :8 f4 :6 f5
:9 f6 :5 f7 :2 f8 :6 f9
:6 fa f8 f5 fb
f5 :4 fc :5 ff :4 100
:4 101 :3 102 ff :2 105
:2 106 105 :2 108 :2 109
108 :2 10a :2 ae :4 7a
:8 111 :5 112 :4 114 :2 116
:2 117 :2 118 :2 119 :2 11a
116 :2 11c :2 11d :2 11e
11c :2 120 :4 121 :2 122
:2 123 120 :2 125 :4 126
:2 127 :2 128 125 :2 12a
:4 12b :2 12c :2 12d 12a
:2 12f :2 113 :4 111 132
0 :2 132 :5 133 :4 135
:2 137 :2 138 :2 139 :2 13a
137 :2 13c :2 13d 13c
:2 13f :2 140 13f :2 142
:2 143 142 :2 145 :4 146
:2 147 145 :2 149 :2 14a
149 :2 14c :4 14d :2 14e
14c :2 150 :2 151 150
:2 153 :4 154 :2 155 153
:2 157 :2 158 157 :2 15a
:2 134 :4 132 :8 160 :5 162
:5 163 :5 164 :5 165 :7 16a
:3 16b 16c 16a 16f
170 :2 171 :2 172 16f
:5 174 :2 177 :2 178 :2 179
177 :5 17b :2 17d :6 17e
:2 17f 17d :3 17b :2 185
:2 186 185 :4 189 :2 18a
189 :3 174 :2 18e :2 167
:4 160 194 0 :2 194
:5 196 :5 197 :5 198 :5 199
:7 19e :3 19f 1a0 19e
1a3 1a4 :2 1a5 :2 1a6
1a3 :5 1a8 :2 1ab :2 1ac
:4 1ad :2 1ae 1ab :5 1b0
:2 1b2 :6 1b3 :2 1b4 1b2
:3 1b0 :2 1ba :2 1bb 1ba
:4 1be :2 1bf 1be :3 1a8
:2 1c3 :2 19b :4 194 1c9
0 :2 1c9 :9 1ca :7 1cb
:3 1cc 1cb :5 1ca :9 1cd
:6 1ce :5 1cd :7 1cf :5 1d0
:5 1d1 :5 1d2 :5 1d3 :5 1d4
:5 1d5 :5 1d6 :5 1d7 :5 1d8
:5 1d9 :7 1da :6 1dc :3 1df
:6 1e1 :2 1e2 1e3 1e1
:6 1e5 :4 1e7 :5 1ea 1eb
1ea :6 1ec :7 1ed 1ee
1ed :4 1ef :3 1f0 :2 1f1
1ef :6 1f2 :4 1f3 :4 1f4
1f2 1ee 1f5 1ed
:3 1ec 1eb 1f7 1ea
:5 1f9 :9 1fa :b 1fc :9 1fd
1fc 1f9 :5 1ff :3 200
1ff 1fe :3 1f9 :2 203
:5 204 :7 205 :2 206 :2 207
203 :2 208 :2 1db :4 1c9
20e 0 :2 20e :9 20f
210 :2 211 :2 212 213
210 :5 20f :5 214 :5 215
:5 216 :4 218 :6 21a :4 21b
:4 21c :4 21d :2 21e :3 21c
:4 220 :4 221 :4 222 :2 223
:3 221 :4 225 :2 227 228
229 :2 22a :2 22b :2 22c
227 :2 22e :2 22f :2 230
22e :2 232 :2 217 :4 20e
:9 23c :9 23d :8 23e :2 23f
:2 240 241 23e :5 23d
:11 242 :4 243 :a 244 :a 245
:4 246 :5 247 :5 248 :3 249
:4 24a :4 24b :3 24c :b 24d
243 :5 242 :5 24e :5 24f
:a 250 :7 251 :5 252 :5 253
:5 254 :5 255 :5 256 :5 257
:5 258 :5 259 :5 25a :5 25b
:5 25c :5 25d :5 25e :5 25f
:5 260 :5 261 :5 262 :5 263
:5 264 :5 265 :5 266 :5 267
:5 268 :5 269 :5 26a :5 26b
:5 26c :5 26d :5 26e :5 26f
:5 270 :5 271 :3 273 :6 275
:2 276 277 275 :6 27a
:c 27b :4 27c :4 27d :2 27e
:3 27c :c 280 :4 281 :4 282
:2 283 :3 281 :4 285 :12 286
:2 288 :4 289 :2 28a 288
:2 28c :2 28d 28c :3 28f
:3 290 :3 291 :3 292 :8 293
294 :4 296 :3 297 :3 296
:5 298 :3 29a :6 29b 29c
:2 29b :2 29c :2 29b :2 29c
:2 29b :2 29c :2 29b 29c
29d :2 29b :2 29d :2 29b
:2 29d :2 29b :2 29d :2 29b
:2 29a :7 2a2 :7 2a3 :e 2a4
:7 2a5 :7 2a6 :7 2a7 :7 2a8
:2 2aa :7 2ab :7 2ac 2aa
294 2af 272 :4 2b0
:5 2b2 :b 2b3 :3 2b2 :5 2b6
:6 2b7 :c 2b8 2b7 :7 2ba
2b9 :3 2b7 :6 2bc :c 2bd
2bc :7 2bf 2be :3 2bc
:6 2c1 :c 2c2 2c1 :7 2c4
2c3 :3 2c1 :6 2c6 :c 2c7
2c6 :7 2c9 2c8 :3 2c6
:6 2cb :c 2cc 2cb :7 2ce
2cd :3 2cb :6 2d0 :c 2d1
2d0 :7 2d3 2d2 :3 2d0
:7 2d6 :3 2d7 :6 2d8 :6 2d9
:2 2da :6 2db 2d6 2b6
:7 2dd :3 2de :6 2df :2 2e0
2dd 2dc :3 2b6 :2 2e2
:5 2e8 :3 2e9 :3 2ea :3 2e8
:2 272 :4 23c 2f3 0
:2 2f3 :5 2f4 :5 2f5 :6 2f7
:4 2f9 :2 2fb :2 2fc :2 2fd
2fb :5 2ff :5 301 :2 303
:2 304 :2 305 303 :2 307
:2 308 :2 309 307 :2 30b
:2 2f6 :4 2f3 312 0
:2 312 :9 313 314 :2 315
:2 316 317 314 :5 313
:5 318 :5 319 :5 31a :5 31b
:6 31d :4 31e :3 320 :6 321
:6 322 323 :a 324 :4 325
:5 326 :5 327 :3 328 :2 329
:3 327 :7 32b 323 32d
31c :4 32e :a 330 :5 331
:2 332 333 334 :2 335
:2 336 :2 337 332 :2 339
:2 33a :2 33b 339 :2 33c
:3 331 :2 31c :4 312 :9 343
:5 344 :5 345 :5 346 :5 347
:5 348 :5 349 :5 34a :5 34b
:5 34c :9 34d 34e :2 34f
:2 350 351 34e :5 34d
:4 353 :3 356 :2 358 :2 359
:2 35a 358 357 35c
:3 35d :3 35c 35b :3 352
:5 361 :6 363 :6 365 :4 367
:4 368 :6 36a :4 36b :4 36c
:3 36d 368 :4 370 :4 374
:7 375 :7 376 :5 377 :6 379
:5 37a 37c 37d :2 37e
:2 37f 37c :5 381 :6 383
:3 384 :6 387 :3 388 381
:3 38a 389 :3 381 377
:3 38d 38c :3 377 36e
:3 368 361 391 392
:2 393 :2 394 391 :5 396
:4 399 :8 39a :7 39b :5 39c
:6 39d :7 39e :7 39f :6 3a1
:8 3a2 39c 3a5 3a6
:2 3a7 :2 3a8 3a5 3a4
3aa :3 3ab :3 3aa 3a9
:4 3a3 :3 39c 396 :3 3af
3ae :3 396 390 :3 361
:7 3b3 :2 352 :4 343 :9 3bb
:5 3bc :5 3bd :5 3be :5 3bf
:5 3c0 :5 3c1 :5 3c2 :5 3c3
:4 3c5 :3 3c7 :3 3c8 3cb
3cc :2 3cd :2 3ce 3cb
:5 3d0 3d2 3d3 :2 3d4
:2 3d5 3d2 :5 3d7 3d9
3da :2 3db :2 3dc 3d9
:5 3de :3 3df :3 3e0 :2 3e1
:2 3e2 3df 3de :3 3e4
3e3 :3 3de 3d7 :3 3e7
3e6 :3 3d7 3d0 :3 3ea
3e9 :3 3d0 :a 3ed :a 3ee
:a 3ef :9 3f1 :2 3c4 :4 3bb
:f 3f5 :9 3f6 :7 3f8 :d 3f9
:7 3fa :3 3f9 :2 3fa :2 3f9
3fa :7 3fb :3 3f9 :2 3fb
:2 3f9 3fb :7 3fc :3 3f9
:2 3f7 :4 3f5 :4 13 401
:6 1 
3c2a
4
:3 0 1 :3 0 2
:3 0 3 :6 0 1
:2 0 a :2 0 :2 5
:3 0 6 :2 0 3
5 7 :6 0 7
:4 0 b 8 9
da2 0 4 :6 0
c :2 0 7 9
:3 0 d :7 0 11
e f da2 0
8 :6 0 a :2 0
:2 9 :3 0 13 :7 0
17 14 15 da2
0 b :6 0 f
:2 0 b 9 :3 0
19 :7 0 1d 1a
1b da2 0 d
:6 0 11 :2 0 d
9 :3 0 1f :7 0
23 20 21 da2
0 e :6 0 13
:2 0 f 9 :3 0
25 :7 0 29 26
27 da2 0 10
:6 0 13 :2 0 13
9 :3 0 2b :7 0
14 :2 0 11 2d
2f :3 0 32 2c
30 da2 0 12
:6 0 13 :2 0 17
9 :3 0 34 :7 0
16 :2 0 15 36
38 :3 0 3b 35
39 da2 0 15
:6 0 13 :2 0 1b
9 :3 0 3d :7 0
18 :2 0 19 3f
41 :3 0 44 3e
42 da2 0 17
:9 0 1f 9 :3 0
46 :7 0 1a :2 0
1d 48 4a :3 0
4d 47 4b da2
0 19 :6 0 1b
:3 0 1c :a 0 5d
2 :7 0 1d :3 0
5 :3 0 51 52
0 5d 4f 53
:2 0 1d :3 0 4
:3 0 56 :2 0 58
21 5c :3 0 5c
1c :4 0 5c 5b
58 59 :6 0 5d
1 0 4f 53
5c da2 :2 0 1e
:a 0 113 3 :7 0
25 :2 0 23 9
:3 0 b :7 0 62
61 :3 0 64 :2 0
113 5f 65 :2 0
29 1f9 0 27
9 :3 0 68 :7 0
6b 69 0 111
0 1f :6 0 2f
231 0 2d 9
:3 0 6d :7 0 70
6e 0 111 0
20 :6 0 5 :3 0
22 :2 0 2b 72
74 :6 0 77 75
0 111 0 21
:6 0 33 265 0
31 24 :3 0 79
:7 0 7c 7a 0
111 0 23 :6 0
9 :3 0 7e :7 0
81 7f 0 111
0 25 :6 0 37
299 0 35 9
:3 0 83 :7 0 86
84 0 111 0
26 :6 0 9 :3 0
88 :7 0 8b 89
0 111 0 27
:6 0 3b 2cd 0
39 9 :3 0 8d
:7 0 90 8e 0
111 0 28 :6 0
9 :3 0 92 :7 0
95 93 0 111
0 29 :6 0 a0
a1 0 3d 9
:3 0 97 :7 0 9a
98 0 111 0
2a :6 0 9 :3 0
9c :7 0 9f 9d
0 111 0 2b
:6 0 2c :3 0 2d
:3 0 2e :4 0 3f
a2 a4 :2 0 10f
2f :3 0 30 :3 0
31 :3 0 32 :3 0
33 :3 0 34 :3 0
20 :3 0 25 :3 0
35 :4 0 36 1
:8 0 10f 37 :3 0
38 :3 0 39 :3 0
1f :3 0 3a :4 0
3b 1 :8 0 10f
3c :3 0 23 :3 0
3a :4 0 3d 1
:8 0 10f 25 :3 0
3e :2 0 3f :2 0
43 bb bd :3 0
40 :3 0 26 :3 0
27 :3 0 28 :3 0
29 :3 0 2a :3 0
2b :3 0 46 bf
c6 :2 0 dd 37
:3 0 41 :3 0 2f
:3 0 42 :3 0 23
:3 0 26 :3 0 27
:3 0 28 :3 0 29
:3 0 2a :3 0 2b
:3 0 20 :3 0 1f
:3 0 23 :3 0 26
:3 0 27 :3 0 28
:3 0 29 :3 0 2a
:3 0 2b :4 0 43
1 :8 0 dd 4d
e8 37 :3 0 41
:3 0 2f :3 0 42
:3 0 23 :3 0 20
:3 0 1f :3 0 23
:4 0 44 1 :8 0
e7 50 e9 be
dd 0 ea 0
e7 0 ea 52
0 10f 37 :3 0
45 :3 0 42 :3 0
21 :3 0 46 :3 0
47 :3 0 48 :3 0
49 :3 0 1f :3 0
21 :3 0 46 :3 0
48 :3 0 49 :3 0
4a :3 0 49 :3 0
b :3 0 4b :3 0
4c :3 0 4d :3 0
4e :4 0 4f 1
:8 0 10f 37 :3 0
50 :3 0 2f :3 0
20 :4 0 51 1
:8 0 10f 37 :3 0
50 :3 0 2f :3 0
52 :3 0 2f :3 0
52 :3 0 35 :4 0
53 1 :8 0 10f
54 :4 0 54 1
:8 0 10f 55 112
:3 0 112 5f 112
111 10f 110 :6 0
113 1 0 5f
65 112 da2 :2 0
40 :a 0 1a4 4
:7 0 6d 53b 0
6b 55 :3 0 9
:3 0 26 :6 0 119
118 :3 0 71 567
0 6f 55 :3 0
9 :3 0 27 :6 0
11e 11d :3 0 55
:3 0 9 :3 0 28
:6 0 123 122 :3 0
75 593 0 73
55 :3 0 9 :3 0
29 :6 0 128 127
:3 0 55 :3 0 9
:3 0 2a :6 0 12d
12c :3 0 7e 5ce
0 77 55 :3 0
9 :3 0 2b :6 0
132 131 :3 0 134
:2 0 1a4 115 135
:2 0 56 :3 0 138
0 13b 1a2 58
:3 0 59 :7 0 57
13b 138 :4 0 14b
14c 0 82 57
:3 0 13e :7 0 141
13f 0 1a2 0
5a :6 0 5 :3 0
5c :2 0 80 143
145 :6 0 5d :4 0
149 146 147 1a2
0 5b :6 0 155
156 0 84 5f
:3 0 5e :2 0 4
56 :3 0 56 :2 0
1 14d 14f :3 0
150 :7 0 153 151
0 1a2 0 5e
:6 0 60 :3 0 5a
:3 0 5b :3 0 157
:2 0 1a0 61 :3 0
62 :2 0 63 :2 0
64 :3 0 15a 15b
:2 0 159 15d 5a
:3 0 5e :4 0 162
:2 0 199 15f 160
:3 0 65 :3 0 5a
:3 0 66 :3 0 164
165 :4 0 166 :3 0
199 61 :3 0 62
:2 0 27 :3 0 5e
:3 0 16a 16b 0
16d 86 16f 88
16e 16d :2 0 197
67 :2 0 28 :3 0
5e :3 0 171 172
0 174 8a 176
8c 175 174 :2 0
197 68 :2 0 29
:3 0 5e :3 0 178
179 0 17b 8e
17d 90 17c 17b
:2 0 197 69 :2 0
2a :3 0 5e :3 0
17f 180 0 182
92 184 94 183
182 :2 0 197 c
:2 0 26 :3 0 5e
:3 0 186 187 0
189 96 18b 98
18a 189 :2 0 197
63 :2 0 2b :3 0
5e :3 0 18d 18e
0 190 9a 192
9c 191 190 :2 0
197 6a :2 0 40
193 0 195 9e
196 0 195 :2 0
197 a0 :2 0 198
168 197 0 199
0 a8 19b 64
:3 0 15e 199 :4 0
1a0 6b :3 0 5a
:4 0 19f :2 0 1a0
19d 0 ac 1a3
:3 0 1a3 b0 1a3
1a2 1a0 1a1 :6 0
1a4 1 0 115
135 1a3 da2 :2 0
6c :a 0 40b 6
:7 0 b7 789 0
b5 9 :3 0 6d
:7 0 1a9 1a8 :3 0
bc 7a6 0 b9
9 :3 0 6e :7 0
1ad 1ac :3 0 1af
:2 0 40b 1a6 1b0
:2 0 1bd 1be 0
be 9 :3 0 1b3
:7 0 1b6 1b4 0
409 0 20 :6 0
9 :3 0 1b8 :7 0
1bb 1b9 0 409
0 61 :6 0 c2
807 0 c0 4a
:3 0 46 :2 0 4
56 :3 0 56 :2 0
1 1bf 1c1 :3 0
1c2 :7 0 1c5 1c3
0 409 0 6f
:6 0 c6 83b 0
c4 9 :3 0 1c7
:7 0 1ca 1c8 0
409 0 70 :6 0
9 :3 0 1cc :7 0
1cf 1cd 0 409
0 71 :6 0 cc
873 0 ca 9
:3 0 1d1 :7 0 1d4
1d2 0 409 0
72 :6 0 5 :3 0
22 :2 0 c8 1d6
1d8 :6 0 1db 1d9
0 409 0 21
:6 0 d0 8a7 0
ce 24 :3 0 1dd
:7 0 1e0 1de 0
409 0 23 :6 0
24 :3 0 1e2 :7 0
1e5 1e3 0 409
0 73 :6 0 d4
8db 0 d2 9
:3 0 1e7 :7 0 1ea
1e8 0 409 0
25 :6 0 9 :3 0
1ec :7 0 1ef 1ed
0 409 0 26
:6 0 d8 90f 0
d6 9 :3 0 1f1
:7 0 1f4 1f2 0
409 0 27 :6 0
9 :3 0 1f6 :7 0
1f9 1f7 0 409
0 28 :6 0 dc
943 0 da 9
:3 0 1fb :7 0 1fe
1fc 0 409 0
29 :6 0 9 :3 0
200 :7 0 203 201
0 409 0 2a
:6 0 e0 977 0
de 9 :3 0 205
:7 0 208 206 0
409 0 2b :6 0
9 :3 0 20a :7 0
20d 20b 0 409
0 74 :6 0 e4
9ab 0 e2 9
:3 0 20f :7 0 212
210 0 409 0
75 :6 0 9 :3 0
214 :7 0 217 215
0 409 0 76
:6 0 e8 9df 0
e6 9 :3 0 219
:7 0 21c 21a 0
409 0 77 :6 0
9 :3 0 21e :7 0
221 21f 0 409
0 78 :6 0 ec
a13 0 ea 9
:3 0 223 :7 0 226
224 0 409 0
79 :6 0 9 :3 0
228 :7 0 22b 229
0 409 0 7a
:6 0 f0 a47 0
ee 9 :3 0 22d
:7 0 230 22e 0
409 0 7b :6 0
9 :3 0 232 :7 0
235 233 0 409
0 7c :6 0 f4
a7b 0 f2 9
:3 0 237 :7 0 23a
238 0 409 0
7d :6 0 9 :3 0
23c :7 0 23f 23d
0 409 0 7e
:6 0 59 :3 0 9
:3 0 241 :7 0 244
242 0 409 0
7f :6 0 80 :a 0
7 28e :4 0 f8
abd 0 f6 9
:3 0 81 :7 0 249
248 :3 0 fc :2 0
fa 9 :3 0 82
:7 0 24d 24c :3 0
9 :3 0 83 :7 0
251 250 :3 0 246
255 0 253 :3 0
21 :3 0 46 :3 0
84 :3 0 85 :3 0
85 :3 0 84 :3 0
86 :3 0 87 :3 0
21 :3 0 87 :3 0
46 :3 0 88 :3 0
49 :3 0 87 :3 0
49 :3 0 88 :3 0
49 :3 0 87 :3 0
49 :3 0 88 :3 0
49 :3 0 84 :3 0
88 :3 0 48 :3 0
87 :3 0 48 :3 0
88 :3 0 48 :3 0
87 :3 0 48 :3 0
88 :3 0 48 :3 0
85 :3 0 21 :3 0
46 :3 0 48 :3 0
49 :3 0 37 :3 0
45 :3 0 42 :3 0
82 :3 0 87 :3 0
21 :3 0 46 :3 0
48 :3 0 49 :3 0
4a :3 0 88 :3 0
87 :3 0 46 :3 0
88 :3 0 46 :3 0
84 :3 0 83 :3 0
84 :4 0 89 1
:8 0 28f 246 255
290 0 409 100
290 292 28f 291
:6 0 28e 1 :6 0
290 2c :3 0 2d
:3 0 293 294 0
8a :4 0 8b :2 0
6d :3 0 102 297
299 :3 0 105 295
29b :2 0 407 8c
:3 0 6d :3 0 107
29d 29f :2 0 407
2f :3 0 30 :3 0
31 :3 0 32 :3 0
33 :3 0 34 :3 0
20 :3 0 25 :3 0
35 :4 0 36 1
:8 0 407 37 :3 0
8d :3 0 2f :3 0
20 :4 0 8e 1
:8 0 407 37 :3 0
8f :3 0 2f :3 0
20 :4 0 90 1
:8 0 407 3c :3 0
23 :3 0 3a :4 0
3d 1 :8 0 407
25 :3 0 3e :2 0
3f :2 0 10b 2ba
2bc :3 0 40 :3 0
26 :3 0 27 :3 0
28 :3 0 29 :3 0
2a :3 0 2b :3 0
10e 2be 2c5 :2 0
3a3 23 :3 0 26
:3 0 27 :3 0 28
:3 0 29 :3 0 2a
:3 0 2b :3 0 73
:3 0 74 :3 0 75
:3 0 76 :3 0 77
:3 0 78 :3 0 79
:3 0 37 :3 0 41
:3 0 42 :3 0 6d
:4 0 91 1 :8 0
3a3 26 :3 0 74
:3 0 3e :2 0 117
2dc 2dd :3 0 2de
:2 0 7a :3 0 26
:3 0 13 :2 0 74
:3 0 11a 2e2 2e4
:3 0 2e5 :2 0 92
:2 0 93 :2 0 11d
2e7 2e9 :3 0 2e0
2ea 0 2ec 120
2f5 7a :3 0 26
:3 0 92 :2 0 93
:2 0 122 2ef 2f1
:3 0 2ed 2f2 0
2f4 125 2f6 2df
2ec 0 2f7 0
2f4 0 2f7 127
0 3a3 27 :3 0
75 :3 0 3e :2 0
12c 2fa 2fb :3 0
2fc :2 0 7b :3 0
27 :3 0 13 :2 0
75 :3 0 12f 300
302 :3 0 303 :2 0
92 :2 0 93 :2 0
132 305 307 :3 0
2fe 308 0 30a
135 313 7b :3 0
27 :3 0 92 :2 0
93 :2 0 137 30d
30f :3 0 30b 310
0 312 13a 314
2fd 30a 0 315
0 312 0 315
13c 0 3a3 28
:3 0 76 :3 0 3e
:2 0 141 318 319
:3 0 31a :2 0 7c
:3 0 28 :3 0 13
:2 0 76 :3 0 144
31e 320 :3 0 321
:2 0 92 :2 0 93
:2 0 147 323 325
:3 0 31c 326 0
328 14a 331 7c
:3 0 28 :3 0 92
:2 0 93 :2 0 14c
32b 32d :3 0 329
32e 0 330 14f
332 31b 328 0
333 0 330 0
333 151 0 3a3
29 :3 0 77 :3 0
3e :2 0 156 336
337 :3 0 338 :2 0
7d :3 0 29 :3 0
13 :2 0 77 :3 0
159 33c 33e :3 0
33f :2 0 92 :2 0
93 :2 0 15c 341
343 :3 0 33a 344
0 346 15f 34f
7d :3 0 29 :3 0
92 :2 0 93 :2 0
161 349 34b :3 0
347 34c 0 34e
164 350 339 346
0 351 0 34e
0 351 166 0
3a3 2a :3 0 78
:3 0 3e :2 0 16b
354 355 :3 0 356
:2 0 7e :3 0 2a
:3 0 13 :2 0 78
:3 0 16e 35a 35c
:3 0 35d :2 0 92
:2 0 93 :2 0 171
35f 361 :3 0 358
362 0 364 174
36d 7e :3 0 2a
:3 0 92 :2 0 93
:2 0 176 367 369
:3 0 365 36a 0
36c 179 36e 357
364 0 36f 0
36c 0 36f 17b
0 3a3 2b :3 0
79 :3 0 3e :2 0
180 372 373 :3 0
374 :2 0 7f :3 0
2b :3 0 13 :2 0
79 :3 0 183 378
37a :3 0 37b :2 0
92 :2 0 93 :2 0
186 37d 37f :3 0
376 380 0 382
189 38b 7f :3 0
2b :3 0 92 :2 0
93 :2 0 18b 385
387 :3 0 383 388
0 38a 18e 38c
375 382 0 38d
0 38a 0 38d
190 0 3a3 37
:3 0 8f :3 0 2f
:3 0 23 :3 0 73
:3 0 26 :3 0 27
:3 0 28 :3 0 29
:3 0 2a :3 0 2b
:3 0 20 :3 0 23
:3 0 73 :3 0 7a
:3 0 7b :3 0 7c
:3 0 7d :3 0 7e
:3 0 7f :4 0 94
1 :8 0 3a3 193
3b5 23 :3 0 73
:3 0 37 :3 0 41
:3 0 42 :3 0 6d
:4 0 95 1 :8 0
3b4 37 :3 0 8f
:3 0 2f :3 0 23
:3 0 73 :3 0 20
:3 0 23 :3 0 73
:4 0 96 1 :8 0
3b4 19d 3b6 2bd
3a3 0 3b7 0
3b4 0 3b7 1a0
0 407 60 :3 0
80 :3 0 20 :3 0
6d :3 0 10 :3 0
1a3 3b9 3bd 0
3be :2 0 407 3b9
3bd :2 0 61 :3 0
62 :2 0 6e :3 0
64 :3 0 3c1 3c2
:2 0 3c0 3c4 80
:3 0 21 :3 0 6f
:3 0 70 :3 0 71
:3 0 72 :4 0 3cd
:2 0 3e3 3c6 3ce
:3 0 1a7 :3 0 65
:3 0 80 :3 0 66
:3 0 3d0 3d1 :4 0
3d2 :3 0 3e3 37
:3 0 8d :3 0 2f
:3 0 21 :3 0 46
:3 0 49 :3 0 48
:3 0 86 :3 0 20
:3 0 21 :3 0 6f
:3 0 70 :3 0 71
:3 0 72 :4 0 97
1 :8 0 3e3 1ad
3e5 64 :3 0 3c5
3e3 :4 0 407 6b
:3 0 80 :4 0 3e9
:2 0 407 3e7 0
37 :3 0 8d :3 0
98 :3 0 98 :3 0
99 :3 0 9a :3 0
99 :3 0 4a :3 0
9a :3 0 9a :3 0
46 :3 0 98 :3 0
46 :3 0 98 :3 0
2f :3 0 20 :4 0
9b 1 :8 0 407
37 :3 0 45 :3 0
42 :3 0 6d :4 0
9c 1 :8 0 407
37 :3 0 41 :3 0
2f :3 0 20 :4 0
9d 1 :8 0 407
54 :4 0 54 1
:8 0 407 1b1 40a
:3 0 40a 1c0 40a
409 407 408 :6 0
40b 1 0 1a6
1b0 40a da2 :2 0
9e :a 0 457 9
:7 0 1e0 :2 0 1de
9 :3 0 6f :7 0
410 40f :3 0 412
:2 0 457 40d 413
:2 0 1e4 456 0
1e2 9 :3 0 416
:7 0 419 417 0
455 0 20 :6 0
2f :3 0 20 :3 0
35 :4 0 9f 1
:8 0 453 37 :3 0
45 :3 0 42 :3 0
42 :3 0 37 :3 0
41 :3 0 2f :3 0
20 :3 0 46 :3 0
6f :4 0 a0 1
:8 0 453 37 :3 0
8d :3 0 2f :3 0
20 :3 0 46 :3 0
6f :4 0 a1 1
:8 0 453 37 :3 0
a2 :3 0 a3 :3 0
a3 :3 0 37 :3 0
a4 :3 0 2f :3 0
20 :3 0 46 :3 0
6f :4 0 a5 1
:8 0 453 37 :3 0
a6 :3 0 a3 :3 0
a3 :3 0 37 :3 0
a7 :3 0 2f :3 0
20 :3 0 46 :3 0
6f :4 0 a8 1
:8 0 453 37 :3 0
a9 :3 0 a3 :3 0
a3 :3 0 37 :3 0
aa :3 0 2f :3 0
20 :3 0 46 :3 0
6f :4 0 ab 1
:8 0 453 54 :4 0
54 1 :8 0 453
:3 0 456 1ec 456
455 453 454 :6 0
457 1 0 40d
413 456 da2 :2 0
ac :a 0 4ae a
:8 0 45a :2 0 4ae
459 45b :2 0 1f0
4ad 0 1ee 9
:3 0 45e :7 0 461
45f 0 4ac 0
20 :6 0 2f :3 0
20 :3 0 35 :4 0
9f 1 :8 0 4aa
37 :3 0 45 :3 0
42 :3 0 42 :3 0
37 :3 0 41 :3 0
2f :3 0 20 :4 0
ad 1 :8 0 4aa
37 :3 0 41 :3 0
2f :3 0 20 :4 0
9d 1 :8 0 4aa
37 :3 0 8f :3 0
2f :3 0 20 :4 0
90 1 :8 0 4aa
37 :3 0 8d :3 0
2f :3 0 20 :4 0
8e 1 :8 0 4aa
37 :3 0 a2 :3 0
a3 :3 0 a3 :3 0
37 :3 0 a4 :3 0
2f :3 0 20 :4 0
ae 1 :8 0 4aa
37 :3 0 a4 :3 0
2f :3 0 20 :4 0
af 1 :8 0 4aa
37 :3 0 a6 :3 0
a3 :3 0 a3 :3 0
37 :3 0 a7 :3 0
2f :3 0 20 :4 0
b0 1 :8 0 4aa
37 :3 0 a7 :3 0
2f :3 0 20 :4 0
b1 1 :8 0 4aa
37 :3 0 a9 :3 0
a3 :3 0 a3 :3 0
37 :3 0 aa :3 0
2f :3 0 20 :4 0
b2 1 :8 0 4aa
37 :3 0 aa :3 0
2f :3 0 20 :4 0
b3 1 :8 0 4aa
54 :4 0 54 1
:8 0 4aa :3 0 4ad
1fd 4ad 4ac 4aa
4ab :6 0 4ae 1
0 459 45b 4ad
da2 :2 0 8c :a 0
513 b :7 0 201
:2 0 1ff 9 :3 0
6d :7 0 4b3 4b2
:3 0 4b5 :2 0 513
4b0 4b6 :2 0 205
154e 0 203 24
:3 0 4b9 :7 0 4bc
4ba 0 511 0
b4 :6 0 209 1582
0 207 24 :3 0
4be :7 0 4c1 4bf
0 511 0 b5
:6 0 9 :3 0 4c3
:7 0 4c6 4c4 0
511 0 20 :6 0
2f :3 0 9 :3 0
4c8 :7 0 4cb 4c9
0 511 0 25
:6 0 52 :3 0 30
:3 0 31 :3 0 32
:3 0 33 :3 0 34
:3 0 20 :3 0 b5
:3 0 25 :3 0 35
:4 0 b6 1 :8 0
50f 52 :3 0 b4
:3 0 37 :3 0 50
:3 0 2f :3 0 20
:4 0 b7 1 :8 0
50f b5 :3 0 b4
:3 0 b8 :2 0 20d
4e1 4e2 :3 0 37
:3 0 45 :3 0 48
:3 0 49 :3 0 42
:3 0 6d :4 0 b9
1 :8 0 50a 25
:3 0 3e :2 0 3f
:2 0 212 4ec 4ee
:3 0 37 :3 0 41
:3 0 26 :3 0 27
:3 0 28 :3 0 29
:3 0 2a :3 0 2b
:3 0 42 :3 0 6d
:4 0 ba 1 :8 0
4fb 215 4fc 4ef
4fb 0 4fd 217
0 50a 37 :3 0
50 :3 0 2f :3 0
20 :4 0 bb 1
:8 0 50a 37 :3 0
50 :3 0 2f :3 0
52 :3 0 20 :3 0
b5 :4 0 bc 1
:8 0 50a 219 50b
4e3 50a 0 50c
21e 0 50f 54
:4 0 54 1 :8 0
50f 220 512 :3 0
512 225 512 511
50f 510 :6 0 513
1 0 4b0 4b6
512 da2 :2 0 bd
:a 0 578 c :8 0
516 :2 0 578 515
517 :2 0 22c 1724
0 22a 24 :3 0
51a :7 0 51d 51b
0 576 0 b4
:6 0 230 1758 0
22e 24 :3 0 51f
:7 0 522 520 0
576 0 b5 :6 0
9 :3 0 524 :7 0
527 525 0 576
0 20 :6 0 2f
:3 0 9 :3 0 529
:7 0 52c 52a 0
576 0 25 :6 0
52 :3 0 30 :3 0
31 :3 0 32 :3 0
33 :3 0 34 :3 0
20 :3 0 b5 :3 0
25 :3 0 35 :4 0
b6 1 :8 0 574
52 :3 0 b4 :3 0
37 :3 0 50 :3 0
2f :3 0 20 :4 0
b7 1 :8 0 574
b5 :3 0 b4 :3 0
b8 :2 0 234 542
543 :3 0 37 :3 0
a2 :3 0 48 :3 0
49 :3 0 a3 :3 0
a3 :3 0 37 :3 0
a4 :3 0 2f :3 0
20 :4 0 be 1
:8 0 56f 25 :3 0
3e :2 0 3f :2 0
239 551 553 :3 0
37 :3 0 a4 :3 0
26 :3 0 27 :3 0
28 :3 0 29 :3 0
2a :3 0 2b :3 0
2f :3 0 20 :4 0
bf 1 :8 0 560
23c 561 554 560
0 562 23e 0
56f 37 :3 0 50
:3 0 2f :3 0 20
:4 0 bb 1 :8 0
56f 37 :3 0 50
:3 0 2f :3 0 52
:3 0 20 :3 0 b5
:4 0 bc 1 :8 0
56f 240 570 544
56f 0 571 245
0 574 54 :4 0
54 1 :8 0 574
247 577 :3 0 577
24c 577 576 574
575 :6 0 578 1
0 515 517 577
da2 :2 0 c0 :a 0
68a d :8 0 57b
:2 0 68a 57a 57c
:2 0 59 :3 0 c1
:a 0 e 592 :4 0
253 :2 0 251 9
:3 0 20 :7 0 582
581 :3 0 57f 586
0 584 :3 0 98
:3 0 21 :3 0 98
:3 0 46 :3 0 37
:3 0 8d :3 0 98
:3 0 98 :3 0 2f
:3 0 20 :4 0 c2
1 :8 0 593 57f
586 594 0 688
255 594 596 593
595 :6 0 592 1
:6 0 594 59 :3 0
5a :a 0 f 5a6
:4 0 259 :2 0 257
9 :3 0 c3 :7 0
59b 59a :3 0 598
59f 0 59d :3 0
4a :3 0 9a :3 0
9a :3 0 46 :3 0
c3 :4 0 c4 1
:8 0 5a7 598 59f
5a8 0 688 25b
5a8 5aa 5a7 5a9
:6 0 5a6 1 :6 0
5a8 25f 19fb 0
25d 4a :3 0 c6
:3 0 5ac 5ad :3 0
5ae :7 0 5b1 5af
0 688 0 c5
:6 0 263 1a2f 0
261 9 :3 0 5b3
:7 0 5b6 5b4 0
688 0 20 :6 0
9 :3 0 5b8 :7 0
5bb 5b9 0 688
0 a3 :6 0 267
1a63 0 265 24
:3 0 5bd :7 0 5c0
5be 0 688 0
23 :6 0 9 :3 0
5c2 :7 0 5c5 5c3
0 688 0 25
:6 0 26b 1a97 0
269 9 :3 0 5c7
:7 0 5ca 5c8 0
688 0 26 :6 0
9 :3 0 5cc :7 0
5cf 5cd 0 688
0 27 :6 0 26f
1acb 0 26d 9
:3 0 5d1 :7 0 5d4
5d2 0 688 0
28 :6 0 9 :3 0
5d6 :7 0 5d9 5d7
0 688 0 29
:6 0 22 :2 0 271
9 :3 0 5db :7 0
5de 5dc 0 688
0 2a :6 0 9
:3 0 5e0 :7 0 5e3
5e1 0 688 0
2b :6 0 5eb 5ec
0 275 5 :3 0
273 5e5 5e7 :6 0
5ea 5e8 0 688
0 21 :6 0 2c
:3 0 2d :3 0 c7
:4 0 277 5ed 5ef
:2 0 686 bd :4 0
5f1 5f2 :2 0 686
2f :3 0 30 :3 0
31 :3 0 32 :3 0
33 :3 0 34 :3 0
20 :3 0 25 :3 0
35 :4 0 c8 1
:8 0 686 37 :3 0
c9 :3 0 39 :3 0
a3 :3 0 3a :4 0
ca 1 :8 0 686
3c :3 0 23 :3 0
3a :4 0 3d 1
:8 0 686 cb :3 0
c1 :3 0 20 :3 0
279 609 60b 64
:3 0 608 60c cb
:3 0 46 :3 0 60f
610 0 cc :2 0
27b 612 613 :3 0
c5 :3 0 5a :3 0
cb :3 0 46 :3 0
617 618 0 27d
616 61a 64 :3 0
615 61b 21 :3 0
21 :3 0 37 :3 0
8d :3 0 46 :3 0
c5 :3 0 46 :3 0
2f :3 0 20 :4 0
cd 1 :8 0 637
37 :3 0 ce :3 0
21 :3 0 46 :3 0
48 :3 0 49 :3 0
cb :3 0 21 :3 0
c5 :3 0 46 :3 0
c5 :3 0 48 :3 0
c5 :3 0 49 :4 0
cf 1 :8 0 637
27f 639 64 :3 0
61d 637 :4 0 63a
282 63b 614 63a
0 63c 284 0
63d 286 63f 64
:3 0 60e 63d :4 0
686 25 :3 0 3e
:2 0 3f :2 0 28a
641 643 :3 0 40
:3 0 26 :3 0 27
:3 0 28 :3 0 29
:3 0 2a :3 0 2b
:3 0 28d 645 64c
:2 0 663 37 :3 0
a4 :3 0 2f :3 0
a3 :3 0 23 :3 0
26 :3 0 27 :3 0
28 :3 0 29 :3 0
2a :3 0 2b :3 0
20 :3 0 a3 :3 0
23 :3 0 26 :3 0
27 :3 0 28 :3 0
29 :3 0 2a :3 0
2b :4 0 d0 1
:8 0 663 294 66e
37 :3 0 a4 :3 0
2f :3 0 a3 :3 0
23 :3 0 20 :3 0
a3 :3 0 23 :4 0
d1 1 :8 0 66d
297 66f 644 663
0 670 0 66d
0 670 299 0
686 37 :3 0 a2
:3 0 a3 :3 0 21
:3 0 46 :3 0 48
:3 0 49 :3 0 a3
:3 0 21 :3 0 46
:3 0 d2 :3 0 48
:3 0 d2 :3 0 49
:3 0 37 :3 0 ce
:3 0 46 :3 0 21
:4 0 d3 1 :8 0
686 54 :4 0 54
1 :8 0 686 29c
689 :3 0 689 2a6
689 688 686 687
:6 0 68a 1 0
57a 57c 689 da2
:2 0 d4 :a 0 6fc
12 :8 0 68d :2 0
6fc 68c 68e :2 0
59 :3 0 d5 :a 0
13 6a0 :4 0 2b7
:2 0 2b5 9 :3 0
81 :7 0 694 693
:3 0 691 698 0
696 :3 0 a3 :3 0
37 :3 0 a4 :3 0
2f :3 0 81 :3 0
a3 :4 0 d6 1
:8 0 6a1 691 698
6a2 0 6fa 2b9
6a2 6a4 6a1 6a3
:6 0 6a0 1 :6 0
6a2 2bd 1e6b 0
2bb 9 :3 0 6a6
:7 0 6a9 6a7 0
6fa 0 d7 :6 0
2c1 :2 0 2bf 9
:3 0 6ab :7 0 6ae
6ac 0 6fa 0
d8 :6 0 9 :3 0
6b0 :7 0 6b3 6b1
0 6fa 0 20
:6 0 2f :3 0 20
:3 0 35 :4 0 9f
1 :8 0 6f8 60
:3 0 d5 :3 0 20
:3 0 6b9 6bb 0
6bc :2 0 6f8 6b9
6bb :2 0 d5 :3 0
d7 :4 0 6c1 :2 0
6f8 6be 6bf :3 0
d5 :3 0 66 :3 0
6c2 6c3 :3 0 6c4
:2 0 6b :3 0 d5
:4 0 6c9 :2 0 6cc
6c7 0 1d :6 0
6cc 2c3 6cd 6c5
6cc 0 6ce 2c6
0 6f8 d5 :3 0
d8 :4 0 6d2 :2 0
6f8 6cf 6d0 :3 0
d5 :3 0 66 :3 0
6d3 6d4 :3 0 6d5
:2 0 6b :3 0 d5
:4 0 6da :2 0 6dd
6d8 0 1d :6 0
6dd 2c8 6de 6d6
6dd 0 6df 2cb
0 6f8 6b :3 0
d5 :4 0 6e3 :2 0
6f8 6e1 0 37
:3 0 a2 :3 0 a3
:3 0 a3 :3 0 37
:3 0 a4 :3 0 2f
:3 0 20 :3 0 a3
:3 0 d8 :4 0 d9
1 :8 0 6f8 37
:3 0 a4 :3 0 2f
:3 0 20 :3 0 a3
:3 0 d8 :4 0 da
1 :8 0 6f8 54
:4 0 54 1 :8 0
6f8 2cd 6fb :3 0
6fb 2d8 6fb 6fa
6f8 6f9 :6 0 6fc
1 0 68c 68e
6fb da2 :2 0 db
:a 0 a52 14 :7 0
2df :2 0 2dd 55
:3 0 9 :3 0 dc
:6 0 702 701 :3 0
704 :2 0 a52 6fe
705 :2 0 59 :3 0
d5 :a 0 15 71e
:4 0 2e3 :2 0 2e1
9 :3 0 81 :7 0
70b 70a :3 0 708
70f 0 70d :3 0
a3 :3 0 23 :3 0
26 :3 0 27 :3 0
28 :3 0 29 :3 0
2a :3 0 2b :3 0
37 :3 0 a4 :3 0
2f :3 0 81 :3 0
a3 :4 0 dd 1
:8 0 71f 708 70f
720 0 a50 2e5
720 722 71f 721
:6 0 71e 1 :6 0
720 59 :3 0 de
:a 0 16 774 :4 0
2e9 20cb 0 2e7
9 :3 0 82 :7 0
727 726 :3 0 2ed
:2 0 2eb 9 :3 0
df :7 0 72b 72a
:3 0 9 :3 0 20
:7 0 72f 72e :3 0
724 733 0 731
:3 0 e0 :3 0 46
:3 0 e0 :3 0 21
:3 0 e0 :3 0 48
:3 0 e1 :3 0 48
:3 0 e0 :3 0 48
:3 0 e1 :3 0 48
:3 0 e0 :3 0 48
:3 0 e0 :3 0 49
:3 0 e1 :3 0 49
:3 0 e0 :3 0 49
:3 0 e1 :3 0 49
:3 0 e0 :3 0 49
:3 0 e2 :3 0 49
:3 0 e2 :3 0 48
:3 0 37 :3 0 a2
:3 0 a3 :3 0 82
:3 0 e0 :3 0 37
:3 0 a2 :3 0 a3
:3 0 df :3 0 e1
:3 0 37 :3 0 8d
:3 0 e2 :3 0 e0
:3 0 46 :3 0 e1
:3 0 46 :3 0 e0
:3 0 46 :3 0 e2
:3 0 46 :3 0 e2
:3 0 2f :3 0 20
:3 0 e0 :3 0 49
:3 0 e1 :3 0 49
:3 0 e0 :3 0 49
:3 0 e1 :3 0 49
:3 0 e0 :3 0 49
:3 0 10 :4 0 e3
1 :8 0 775 724
733 776 0 a50
2f1 776 778 775
777 :6 0 774 1
:6 0 776 2f5 2235
0 2f3 9 :3 0
77a :7 0 77d 77b
0 a50 0 e4
:6 0 22 :2 0 2f7
9 :3 0 77f :7 0
782 780 0 a50
0 e5 :6 0 4a
:3 0 46 :2 0 4
784 785 0 56
:3 0 56 :2 0 1
786 788 :3 0 789
:7 0 78c 78a 0
a50 0 e6 :6 0
2fd 229a 0 2fb
5 :3 0 2f9 78e
790 :6 0 793 791
0 a50 0 21
:6 0 301 22ce 0
2ff 9 :3 0 795
:7 0 798 796 0
a50 0 d7 :6 0
9 :3 0 79a :7 0
79d 79b 0 a50
0 d8 :6 0 305
2302 0 303 24
:3 0 79f :7 0 7a2
7a0 0 a50 0
e7 :6 0 24 :3 0
7a4 :7 0 7a7 7a5
0 a50 0 e8
:6 0 309 2336 0
307 9 :3 0 7a9
:7 0 7ac 7aa 0
a50 0 25 :6 0
9 :3 0 7ae :7 0
7b1 7af 0 a50
0 e9 :6 0 30d
236a 0 30b 9
:3 0 7b3 :7 0 7b6
7b4 0 a50 0
ea :6 0 9 :3 0
7b8 :7 0 7bb 7b9
0 a50 0 eb
:6 0 311 239e 0
30f 9 :3 0 7bd
:7 0 7c0 7be 0
a50 0 ec :6 0
9 :3 0 7c2 :7 0
7c5 7c3 0 a50
0 ed :6 0 315
23d2 0 313 9
:3 0 7c7 :7 0 7ca
7c8 0 a50 0
ee :6 0 9 :3 0
7cc :7 0 7cf 7cd
0 a50 0 ef
:6 0 319 2406 0
317 9 :3 0 7d1
:7 0 7d4 7d2 0
a50 0 f0 :6 0
9 :3 0 7d6 :7 0
7d9 7d7 0 a50
0 f1 :6 0 31d
243a 0 31b 9
:3 0 7db :7 0 7de
7dc 0 a50 0
f2 :6 0 9 :3 0
7e0 :7 0 7e3 7e1
0 a50 0 f3
:6 0 321 246e 0
31f 9 :3 0 7e5
:7 0 7e8 7e6 0
a50 0 f4 :6 0
9 :3 0 7ea :7 0
7ed 7eb 0 a50
0 7a :6 0 325
24a2 0 323 9
:3 0 7ef :7 0 7f2
7f0 0 a50 0
7b :6 0 9 :3 0
7f4 :7 0 7f7 7f5
0 a50 0 7c
:6 0 329 24d6 0
327 9 :3 0 7f9
:7 0 7fc 7fa 0
a50 0 7d :6 0
9 :3 0 7fe :7 0
801 7ff 0 a50
0 7e :6 0 32d
250a 0 32b 9
:3 0 803 :7 0 806
804 0 a50 0
7f :6 0 9 :3 0
808 :7 0 80b 809
0 a50 0 f5
:6 0 331 253e 0
32f 9 :3 0 80d
:7 0 810 80e 0
a50 0 f6 :6 0
9 :3 0 812 :7 0
815 813 0 a50
0 86 :6 0 335
2572 0 333 9
:3 0 817 :7 0 81a
818 0 a50 0
f7 :6 0 9 :3 0
81c :7 0 81f 81d
0 a50 0 f8
:6 0 339 25a6 0
337 9 :3 0 821
:7 0 824 822 0
a50 0 f9 :6 0
9 :3 0 826 :7 0
829 827 0 a50
0 fa :6 0 33d
:2 0 33b 9 :3 0
82b :7 0 82e 82c
0 a50 0 fb
:6 0 9 :3 0 830
:7 0 833 831 0
a50 0 20 :6 0
dc :3 0 12 :3 0
834 835 0 a4e
2f :3 0 30 :3 0
31 :3 0 32 :3 0
33 :3 0 34 :3 0
20 :3 0 25 :3 0
35 :4 0 36 1
:8 0 a4e 60 :3 0
d5 :3 0 20 :3 0
842 844 0 845
:2 0 a4e 842 844
:2 0 d5 :3 0 d7
:3 0 e7 :3 0 e9
:3 0 ea :3 0 eb
:3 0 ec :3 0 ed
:3 0 ee :4 0 851
:2 0 a4e 847 852
:3 0 33f :3 0 d5
:3 0 66 :3 0 853
854 :3 0 855 :2 0
6b :3 0 d5 :4 0
85a :2 0 85d 858
0 1d :6 0 85d
348 85e 856 85d
0 85f 34b 0
a4e d5 :3 0 d8
:3 0 e8 :3 0 ef
:3 0 f0 :3 0 f1
:3 0 f2 :3 0 f3
:3 0 f4 :4 0 86a
:2 0 a4e 860 86b
:3 0 34d :3 0 d5
:3 0 66 :3 0 86c
86d :3 0 86e :2 0
6b :3 0 d5 :4 0
873 :2 0 876 871
0 1d :6 0 876
356 877 86f 876
0 878 359 0
a4e 6b :3 0 d5
:4 0 87c :2 0 a4e
87a 0 2c :3 0
2d :3 0 87d 87e
0 fc :4 0 8b
:2 0 d7 :3 0 35b
881 883 :3 0 8b
:2 0 fd :4 0 35e
885 887 :3 0 8b
:2 0 d8 :3 0 361
889 88b :3 0 364
87f 88d :2 0 a4e
37 :3 0 a6 :3 0
a3 :3 0 a3 :3 0
37 :3 0 a7 :3 0
2f :3 0 20 :4 0
fe 1 :8 0 a4e
37 :3 0 a7 :3 0
2f :3 0 20 :4 0
ff 1 :8 0 a4e
f7 :3 0 100 :2 0
89d 89e 0 a4e
f9 :3 0 100 :2 0
8a0 8a1 0 a4e
fa :3 0 100 :2 0
8a3 8a4 0 a4e
fb :3 0 100 :2 0
8a6 8a7 0 a4e
60 :3 0 de :3 0
d7 :3 0 d8 :3 0
20 :3 0 366 8aa
8ae 0 8af :2 0
a4e 8aa 8ae :2 0
64 :3 0 de :3 0
e6 :3 0 21 :3 0
f5 :3 0 f6 :3 0
e4 :3 0 e5 :4 0
8ba :2 0 934 8b2
8bb :3 0 36a :3 0
65 :3 0 de :3 0
66 :3 0 8bd 8be
:4 0 8bf :3 0 934
2c :3 0 2d :3 0
8c1 8c2 0 101
:4 0 8b :2 0 e6
:3 0 371 8c5 8c7
:3 0 8b :2 0 102
:4 0 374 8c9 8cb
:3 0 8b :2 0 f5
:3 0 377 8cd 8cf
:3 0 8b :2 0 103
:4 0 37a 8d1 8d3
:3 0 8b :2 0 f6
:3 0 37d 8d5 8d7
:3 0 8b :2 0 104
:4 0 380 8d9 8db
:3 0 8b :2 0 e4
:3 0 383 8dd 8df
:3 0 8b :2 0 105
:4 0 386 8e1 8e3
:3 0 8b :2 0 e5
:3 0 389 8e5 8e7
:3 0 38c 8c3 8e9
:2 0 934 86 :3 0
f5 :3 0 92 :2 0
f6 :3 0 38e 8ed
8ef :3 0 8eb 8f0
0 934 f8 :3 0
86 :3 0 106 :2 0
e4 :3 0 391 8f4
8f6 :3 0 8f2 8f7
0 934 2c :3 0
2d :3 0 8f9 8fa
0 107 :4 0 8b
:2 0 108 :3 0 f8
:3 0 62 :2 0 394
8fe 901 397 8fd
903 :3 0 39a 8fb
905 :2 0 934 f9
:3 0 f9 :3 0 109
:2 0 f8 :3 0 39c
909 90b :3 0 907
90c 0 934 fa
:3 0 fa :3 0 109
:2 0 e5 :3 0 39f
910 912 :3 0 90e
913 0 934 fb
:3 0 fb :3 0 109
:2 0 f5 :3 0 3a2
917 919 :3 0 915
91a 0 934 f7
:3 0 f7 :3 0 109
:2 0 62 :2 0 3a5
91e 920 :3 0 91c
921 0 934 37
:3 0 a6 :3 0 a3
:3 0 21 :3 0 46
:3 0 49 :3 0 48
:3 0 86 :3 0 10a
:3 0 d7 :3 0 21
:3 0 e6 :3 0 f6
:3 0 f5 :3 0 86
:3 0 f8 :4 0 10b
1 :8 0 934 3a8
936 64 :4 0 934
:4 0 a4e 6b :3 0
de :4 0 93a :2 0
a4e 938 0 f7
:3 0 10c :2 0 100
:2 0 3b6 93c 93e
:3 0 dc :3 0 f9
:3 0 92 :2 0 fa
:3 0 3b9 942 944
:3 0 106 :2 0 93
:2 0 3bc 946 948
:3 0 940 949 0
94b 3bf 94c 93f
94b 0 94d 3c1
0 a4e 25 :3 0
3e :2 0 3f :2 0
3c5 94f 951 :3 0
e9 :3 0 ef :3 0
3e :2 0 3ca 955
956 :3 0 957 :2 0
7a :3 0 e9 :3 0
13 :2 0 ef :3 0
3cd 95b 95d :3 0
95e :2 0 92 :2 0
93 :2 0 3d0 960
962 :3 0 959 963
0 965 3d3 96e
7a :3 0 e9 :3 0
92 :2 0 93 :2 0
3d5 968 96a :3 0
966 96b 0 96d
3d8 96f 958 965
0 970 0 96d
0 970 3da 0
a26 ea :3 0 f0
:3 0 3e :2 0 3df
973 974 :3 0 975
:2 0 7b :3 0 ea
:3 0 13 :2 0 f0
:3 0 3e2 979 97b
:3 0 97c :2 0 92
:2 0 93 :2 0 3e5
97e 980 :3 0 977
981 0 983 3e8
98c 7b :3 0 ea
:3 0 92 :2 0 93
:2 0 3ea 986 988
:3 0 984 989 0
98b 3ed 98d 976
983 0 98e 0
98b 0 98e 3ef
0 a26 eb :3 0
f1 :3 0 3e :2 0
3f4 991 992 :3 0
993 :2 0 7c :3 0
eb :3 0 13 :2 0
f1 :3 0 3f7 997
999 :3 0 99a :2 0
92 :2 0 93 :2 0
3fa 99c 99e :3 0
995 99f 0 9a1
3fd 9aa 7c :3 0
eb :3 0 92 :2 0
93 :2 0 3ff 9a4
9a6 :3 0 9a2 9a7
0 9a9 402 9ab
994 9a1 0 9ac
0 9a9 0 9ac
404 0 a26 ec
:3 0 f2 :3 0 3e
:2 0 409 9af 9b0
:3 0 9b1 :2 0 7d
:3 0 ec :3 0 13
:2 0 f2 :3 0 40c
9b5 9b7 :3 0 9b8
:2 0 92 :2 0 93
:2 0 40f 9ba 9bc
:3 0 9b3 9bd 0
9bf 412 9c8 7d
:3 0 ec :3 0 92
:2 0 93 :2 0 414
9c2 9c4 :3 0 9c0
9c5 0 9c7 417
9c9 9b2 9bf 0
9ca 0 9c7 0
9ca 419 0 a26
ed :3 0 f3 :3 0
3e :2 0 41e 9cd
9ce :3 0 9cf :2 0
7e :3 0 ed :3 0
13 :2 0 f3 :3 0
421 9d3 9d5 :3 0
9d6 :2 0 92 :2 0
93 :2 0 424 9d8
9da :3 0 9d1 9db
0 9dd 427 9e6
7e :3 0 ed :3 0
92 :2 0 93 :2 0
429 9e0 9e2 :3 0
9de 9e3 0 9e5
42c 9e7 9d0 9dd
0 9e8 0 9e5
0 9e8 42e 0
a26 ee :3 0 f4
:3 0 3e :2 0 433
9eb 9ec :3 0 9ed
:2 0 7f :3 0 ee
:3 0 13 :2 0 f4
:3 0 436 9f1 9f3
:3 0 9f4 :2 0 92
:2 0 93 :2 0 439
9f6 9f8 :3 0 9ef
9f9 0 9fb 43c
a04 7f :3 0 ee
:3 0 92 :2 0 93
:2 0 43e 9fe a00
:3 0 9fc a01 0
a03 441 a05 9ee
9fb 0 a06 0
a03 0 a06 443
0 a26 37 :3 0
a7 :3 0 2f :3 0
a3 :3 0 23 :3 0
73 :3 0 10d :3 0
10e :3 0 10f :3 0
110 :3 0 26 :3 0
27 :3 0 28 :3 0
29 :3 0 2a :3 0
2b :3 0 20 :3 0
d7 :3 0 e7 :3 0
e8 :3 0 fb :3 0
f9 :3 0 fa :3 0
dc :3 0 7a :3 0
7b :3 0 7c :3 0
7d :3 0 7e :3 0
7f :4 0 111 1
:8 0 a26 446 a3b
37 :3 0 a7 :3 0
2f :3 0 a3 :3 0
23 :3 0 73 :3 0
10d :3 0 10e :3 0
10f :3 0 110 :3 0
20 :3 0 d7 :3 0
e7 :3 0 e8 :3 0
fb :3 0 f9 :3 0
fa :3 0 dc :4 0
112 1 :8 0 a3a
44e a3c 952 a26
0 a3d 0 a3a
0 a3d 450 0
a4e 54 :4 0 54
1 :8 0 a4e dc
:3 0 e :3 0 10c
:2 0 455 a42 a43
:3 0 113 :3 0 a45
a47 :2 0 a4b 0
114 :3 0 a48 a4a
:2 0 a4b 0 458
a4c a44 a4b 0
a4d 45b 0 a4e
45d a51 :3 0 a51
474 a51 a50 a4e
a4f :6 0 a52 1
0 6fe 705 a51
da2 :2 0 113 :a 0
a91 18 :8 0 a55
:2 0 a91 a54 a56
:2 0 49d 2de8 0
49b 9 :3 0 a59
:7 0 a5c a5a 0
a8f 0 20 :6 0
2c :3 0 9 :3 0
a5e :7 0 a61 a5f
0 a8f 0 115
:6 0 2d :3 0 a62
a63 0 116 :4 0
49f a64 a66 :2 0
a8d 2f :3 0 20
:3 0 35 :4 0 9f
1 :8 0 a8d a3
:3 0 115 :3 0 37
:3 0 a7 :3 0 2f
:3 0 20 :4 0 117
1 :8 0 a8d 37
:3 0 aa :3 0 a3
:3 0 115 :4 0 118
1 :8 0 a8d 37
:3 0 a9 :3 0 a3
:3 0 115 :4 0 119
1 :8 0 a8d 37
:3 0 aa :3 0 37
:3 0 a7 :3 0 a3
:3 0 115 :4 0 11a
1 :8 0 a8d 37
:3 0 a9 :3 0 37
:3 0 a6 :3 0 a3
:3 0 115 :4 0 11b
1 :8 0 a8d 54
:4 0 54 1 :8 0
a8d 4a1 a90 :3 0
a90 4aa a90 a8f
a8d a8e :6 0 a91
1 0 a54 a56
a90 da2 :2 0 114
:a 0 b32 19 :8 0
a94 :2 0 b32 a93
a95 :2 0 59 :3 0
d5 :a 0 1a aa7
:4 0 4af :2 0 4ad
9 :3 0 81 :7 0
a9b a9a :3 0 a98
a9f 0 a9d :3 0
a3 :3 0 37 :3 0
aa :3 0 2f :3 0
81 :3 0 a3 :4 0
11c 1 :8 0 aa8
a98 a9f aa9 0
b30 4b1 aa9 aab
aa8 aaa :6 0 aa7
1 :6 0 aa9 4b5
2fa3 0 4b3 9
:3 0 aad :7 0 ab0
aae 0 b30 0
115 :6 0 4b9 2fd7
0 4b7 9 :3 0
ab2 :7 0 ab5 ab3
0 b30 0 11d
:6 0 9 :3 0 ab7
:7 0 aba ab8 0
b30 0 20 :6 0
2c :3 0 9 :3 0
abc :7 0 abf abd
0 b30 0 f7
:6 0 2d :3 0 ac0
ac1 0 11e :4 0
4bb ac2 ac4 :2 0
b2e 2f :3 0 20
:3 0 35 :4 0 9f
1 :8 0 b2e f7
:3 0 100 :2 0 aca
acb 0 b2e 115
:3 0 13 :2 0 62
:2 0 4bd ace ad0
:3 0 acd ad1 0
b2e 60 :3 0 d5
:3 0 20 :3 0 4bf
ad4 ad6 0 ad7
:2 0 b2e ad4 ad6
:2 0 64 :3 0 2c
:3 0 2d :3 0 ada
adb 0 11f :4 0
8b :2 0 f7 :3 0
4c1 ade ae0 :3 0
4c4 adc ae2 :2 0
b01 d5 :3 0 11d
:4 0 ae7 :2 0 b01
ae4 ae5 :3 0 65
:3 0 d5 :3 0 66
:3 0 ae9 aea :4 0
aeb :3 0 b01 f7
:3 0 d :3 0 120
:2 0 4c8 aef af0
:3 0 115 :3 0 11d
:3 0 af2 af3 0
af7 65 :8 0 af7
4cb af8 af1 af7
0 af9 4ce 0
b01 f7 :3 0 f7
:3 0 109 :2 0 62
:2 0 4d0 afc afe
:3 0 afa aff 0
b01 4d3 b03 64
:4 0 b01 :4 0 b2e
6b :3 0 d5 :4 0
b07 :2 0 b2e b05
0 2c :3 0 2d
:3 0 b08 b09 0
121 :4 0 8b :2 0
115 :3 0 4d9 b0c
b0e :3 0 4dc b0a
b10 :2 0 b2e 115
:3 0 3e :2 0 100
:2 0 4e0 b13 b15
:3 0 37 :3 0 a9
:3 0 a3 :3 0 a3
:3 0 37 :3 0 aa
:3 0 2f :3 0 20
:3 0 a3 :3 0 115
:4 0 122 1 :8 0
b2b 37 :3 0 aa
:3 0 2f :3 0 20
:3 0 a3 :3 0 115
:4 0 123 1 :8 0
b2b 54 :4 0 54
1 :8 0 b2b 4e3
b2c b16 b2b 0
b2d 4e7 0 b2e
4e9 b31 :3 0 b31
4f3 b31 b30 b2e
b2f :6 0 b32 1
0 a93 a95 b31
da2 :2 0 124 :a 0
c92 1c :7 0 4fb
:2 0 4f9 55 :3 0
126 :3 0 125 :6 0
b38 b37 :3 0 b3a
:2 0 c92 b34 b3b
:2 0 4ff 3216 0
4fd 9 :3 0 b3e
:7 0 b41 b3f 0
c90 0 20 :6 0
503 324a 0 501
9 :3 0 b43 :7 0
b46 b44 0 c90
0 dc :6 0 9
:3 0 b48 :7 0 b4b
b49 0 c90 0
127 :6 0 507 327e
0 505 9 :3 0
b4d :7 0 b50 b4e
0 c90 0 128
:6 0 9 :3 0 b52
:7 0 b55 b53 0
c90 0 6d :6 0
50b 32b2 0 509
9 :3 0 b57 :7 0
b5a b58 0 c90
0 129 :6 0 24
:3 0 b5c :7 0 b5f
b5d 0 c90 0
12a :6 0 50f 32fb
0 50d 24 :3 0
b61 :7 0 b64 b62
0 c90 0 12b
:6 0 9 :3 0 b66
:7 0 b69 b67 0
c90 0 12c :6 0
59 :3 0 d5 :a 0
1d b7a :4 0 b6b
b72 0 511 9
:3 0 81 :7 0 b6e
b6d :3 0 b70 :3 0
42 :3 0 37 :3 0
41 :3 0 2f :3 0
81 :3 0 42 :4 0
12d 1 :8 0 b7b
b6b b72 b7c 0
c90 513 b7c b7e
b7b b7d :6 0 b7a
1 :6 0 b7c 2f
:3 0 20 :3 0 35
:4 0 9f 1 :8 0
c8e 127 :3 0 100
:2 0 b83 b84 0
c8e 127 :3 0 3a
:3 0 37 :3 0 8f
:3 0 2f :3 0 20
:4 0 12e 1 :8 0
b8d 515 b97 12f
:3 0 127 :3 0 100
:2 0 b8f b90 0
b92 517 b94 519
b93 b92 :2 0 b95
51b :2 0 b97 0
b97 b96 b8d b95
:6 0 c8e 1c :3 0
127 :3 0 120 :2 0
100 :2 0 51f b9a
b9c :3 0 2c :3 0
2d :3 0 b9e b9f
0 130 :4 0 522
ba0 ba2 :2 0 c1c
60 :3 0 d5 :3 0
20 :3 0 524 ba5
ba7 0 ba8 :2 0
c1c ba5 ba7 :2 0
d5 :3 0 6d :4 0
bad :2 0 c1c baa
bab :3 0 d5 :3 0
66 :3 0 bae baf
:3 0 bb0 :2 0 2c
:3 0 2d :3 0 bb2
bb3 0 131 :4 0
526 bb4 bb6 :2 0
bc3 1e :3 0 b
:3 0 528 bb8 bba
:2 0 bc3 6b :3 0
d5 :4 0 bbf :2 0
bc3 bbd 0 dc
:3 0 19 :3 0 bc0
bc1 0 bc3 52a
c19 6b :3 0 d5
:4 0 bc7 :2 0 c18
bc5 0 3c :3 0
12a :3 0 3a :4 0
132 1 :8 0 c18
23 :3 0 12b :3 0
37 :3 0 41 :3 0
2f :3 0 20 :4 0
133 1 :8 0 c18
12c :3 0 134 :3 0
12b :3 0 12a :3 0
52f bd4 bd7 bd3
bd8 0 c18 12c
:3 0 3e :2 0 135
:2 0 534 bdb bdd
:3 0 2c :3 0 2d
:3 0 bdf be0 0
136 :4 0 537 be1
be3 :2 0 c10 6c
:3 0 6d :3 0 8
:3 0 539 be5 be8
:2 0 c10 137 :3 0
128 :3 0 37 :3 0
8d :3 0 2f :3 0
20 :4 0 138 1
:8 0 c10 128 :3 0
10c :2 0 100 :2 0
53e bf2 bf4 :3 0
2c :3 0 2d :3 0
bf6 bf7 0 139
:4 0 541 bf8 bfa
:2 0 c08 c0 :3 0
bfc bfe :2 0 c08
0 2c :3 0 2d
:3 0 bff c00 0
13a :4 0 543 c01
c03 :2 0 c08 dc
:3 0 15 :3 0 c05
c06 0 c08 545
c0d dc :3 0 17
:3 0 c09 c0a 0
c0c 54a c0e bf5
c08 0 c0f 0
c0c 0 c0f 54c
0 c10 54f c15
dc :3 0 19 :3 0
c11 c12 0 c14
554 c16 bde c10
0 c17 0 c14
0 c17 556 0
c18 559 c1a bb1
bc3 0 c1b 0
c18 0 c1b 55f
0 c1c 562 c84
137 :3 0 128 :3 0
37 :3 0 8d :3 0
2f :3 0 20 :4 0
13b 1 :8 0 c83
128 :3 0 10c :2 0
100 :2 0 569 c25
c27 :3 0 3c :3 0
12a :3 0 3a :4 0
132 1 :8 0 c7b
13c :3 0 23 :3 0
12b :3 0 37 :3 0
a4 :3 0 2f :3 0
20 :4 0 13d 1
:8 0 c7b 12c :3 0
134 :3 0 12b :3 0
12a :3 0 56c c36
c39 c35 c3a 0
c7b 12c :3 0 3e
:2 0 135 :2 0 571
c3d c3f :3 0 2c
:3 0 2d :3 0 c41
c42 0 139 :4 0
574 c43 c45 :2 0
c63 37 :3 0 3
:3 0 c47 c48 0
c0 :3 0 c49 c4a
0 c4b c4d :2 0
c63 0 37 :3 0
3 :3 0 c4e c4f
0 d4 :3 0 c50
c51 0 c52 c54
:2 0 c63 0 2c
:3 0 2d :3 0 c55
c56 0 13e :4 0
576 c57 c59 :2 0
c63 37 :3 0 3
:3 0 c5b c5c 0
db :3 0 c5d c5e
0 dc :3 0 578
c5f c61 :2 0 c63
57a c78 110 :3 0
dc :3 0 37 :3 0
a7 :3 0 2f :3 0
20 :4 0 13f 1
:8 0 c6b 580 c75
12f :3 0 dc :3 0
15 :3 0 c6d c6e
0 c70 582 c72
584 c71 c70 :2 0
c73 586 :2 0 c75
0 c75 c74 c6b
c73 :6 0 c77 1c
:3 0 588 c79 c40
c63 0 c7a 0
c77 0 c7a 58a
0 c7b 58d c80
dc :3 0 17 :3 0
c7c c7d 0 c7f
592 c81 c28 c7b
0 c82 0 c7f
0 c82 594 0
c83 597 c85 b9d
c1c 0 c86 0
c83 0 c86 59a
0 c8e 60 :3 0
125 :3 0 dc :3 0
3a :4 0 140 1
:a 0 c8d :2 0 c8e
c88 c8b :2 0 59d
c91 :3 0 c91 5a3
c91 c90 c8e c8f
:6 0 c92 1 0
b34 b3b c91 da2
:2 0 141 :a 0 d41
20 :7 0 5b0 :2 0
5ae 55 :3 0 126
:3 0 125 :6 0 c98
c97 :3 0 c9a :2 0
d41 c94 c9b :2 0
5b4 37e6 0 5b2
9 :3 0 c9e :7 0
ca1 c9f 0 d3f
0 20 :6 0 5b8
381a 0 5b6 9
:3 0 ca3 :7 0 ca6
ca4 0 d3f 0
142 :6 0 9 :3 0
ca8 :7 0 cab ca9
0 d3f 0 143
:6 0 5bc 384e 0
5ba 9 :3 0 cad
:7 0 cb0 cae 0
d3f 0 144 :6 0
24 :3 0 cb2 :7 0
cb5 cb3 0 d3f
0 23 :6 0 5c0
3882 0 5be 9
:3 0 cb7 :7 0 cba
cb8 0 d3f 0
145 :6 0 9 :3 0
cbc :7 0 cbf cbd
0 d3f 0 128
:6 0 2f :3 0 9
:3 0 cc1 :7 0 cc4
cc2 0 d3f 0
146 :6 0 20 :3 0
35 :4 0 9f 1
:8 0 d3d 143 :3 0
100 :2 0 cc9 cca
0 d3d 144 :3 0
100 :2 0 ccc ccd
0 d3d 137 :3 0
145 :3 0 37 :3 0
8f :3 0 2f :3 0
20 :4 0 147 1
:8 0 d3d 145 :3 0
10c :2 0 100 :2 0
5c4 cd7 cd9 :3 0
137 :3 0 128 :3 0
37 :3 0 8d :3 0
2f :3 0 20 :4 0
13b 1 :8 0 d0e
128 :3 0 10c :2 0
100 :2 0 5c9 ce3
ce5 :3 0 137 :3 0
146 :3 0 37 :3 0
a4 :3 0 2f :3 0
20 :4 0 148 1
:8 0 d06 146 :3 0
3e :2 0 67 :2 0
5ce cef cf1 :3 0
110 :3 0 10f :3 0
10e :3 0 142 :3 0
143 :3 0 144 :3 0
37 :3 0 a7 :3 0
2f :3 0 20 :4 0
149 1 :8 0 cfe
5d1 d03 142 :3 0
15 :3 0 cff d00
0 d02 5d3 d04
cf2 cfe 0 d05
0 d02 0 d05
5d5 0 d06 5d8
d0b 142 :3 0 17
:3 0 d07 d08 0
d0a 5db d0c ce6
d06 0 d0d 0
d0a 0 d0d 5dd
0 d0e 5e0 d13
142 :3 0 19 :3 0
d0f d10 0 d12
5e3 d14 cda d0e
0 d15 0 d12
0 d15 5e5 0
d3d 2c :3 0 2d
:3 0 d16 d17 0
14a :4 0 8b :2 0
142 :3 0 5e8 d1a
d1c :3 0 5eb d18
d1e :2 0 d3d 2c
:3 0 2d :3 0 d20
d21 0 14b :4 0
8b :2 0 143 :3 0
5ed d24 d26 :3 0
5f0 d22 d28 :2 0
d3d 2c :3 0 2d
:3 0 d2a d2b 0
14c :4 0 8b :2 0
144 :3 0 5f2 d2e
d30 :3 0 5f5 d2c
d32 :2 0 d3d 60
:3 0 125 :3 0 142
:3 0 143 :3 0 144
:3 0 3a :4 0 14d
1 :a 0 d3c :2 0
d3d d35 d3a :2 0
5f7 d40 :3 0 d40
601 d40 d3f d3d
d3e :6 0 d41 1
0 c94 c9b d40
da2 :2 0 1b :3 0
134 :a 0 d9b 21
:7 0 60c 3ac6 0
60a 14f :3 0 14e
:7 0 d47 d46 :3 0
611 3aeb 0 60e
14f :3 0 150 :7 0
d4b d4a :3 0 1d
:3 0 9 :3 0 d4d
d4f 0 d9b d44
d50 :2 0 151 :3 0
151 :3 0 152 :3 0
153 :3 0 154 :7 0
d56 d57 :6 0 d5a
d58 0 d99 0
151 :6 0 150 :3 0
13 :2 0 14e :3 0
613 d5d d5f :3 0
d5b d60 0 d97
1d :3 0 155 :3 0
152 :3 0 152 :4 0
151 :3 0 156 :3 0
616 d67 d68 :3 0
106 :2 0 157 :2 0
619 d6a d6c :3 0
109 :2 0 155 :3 0
158 :3 0 158 :4 0
151 :3 0 156 :3 0
61c d73 d74 :3 0
61f d6e d76 :3 0
d77 :2 0 106 :2 0
159 :2 0 622 d79
d7b :3 0 109 :2 0
155 :3 0 15a :3 0
15a :4 0 151 :3 0
156 :3 0 625 d82
d83 :3 0 628 d7d
d85 :3 0 d86 :2 0
106 :2 0 159 :2 0
62b d88 d8a :3 0
109 :2 0 155 :3 0
153 :3 0 153 :4 0
151 :3 0 156 :3 0
62e d91 d92 :3 0
631 d8c d94 :3 0
d95 :2 0 d97 634
d9a :3 0 d9a 637
d9a d99 d97 d98
:6 0 d9b 1 0
d44 d50 d9a da2
:3 0 da0 0 da0
:3 0 da0 da2 d9e
d9f :6 0 da3 :2 0
3 :3 0 639 0
3 da0 da6 :3 0
da5 da3 da7 :8 0

654
4
:3 0 1 6 1
4 1 c 1
12 1 18 1
1e 1 24 1
2e 1 2a 1
37 1 33 1
40 1 3c 1
49 1 45 1
57 1 60 1
63 1 67 1
6c 1 73 1
71 1 78 1
7d 1 82 1
87 1 8c 1
91 1 96 1
9b 1 a3 1
bc 2 ba bc
6 c0 c1 c2
c3 c4 c5 2
c7 dc 1 e6
2 e8 e9 9
a5 af b5 b9
ea ff 104 10c
10e b 6a 6f
76 7b 80 85
8a 8f 94 99
9e 1 116 1
11b 1 120 1
125 1 12a 1
12f 6 11a 11f
124 129 12e 133
1 13d 1 144
1 142 1 14a
1 16c 1 169
1 173 1 170
1 17a 1 177
1 181 1 17e
1 188 1 185
1 18f 1 18c
1 194 7 16f
176 17d 184 18b
192 196 3 161
167 198 3 158
19b 19e 4 13c
140 148 152 1
1a7 1 1ab 2
1aa 1ae 1 1b2
1 1b7 1 1bc
1 1c6 1 1cb
1 1d0 1 1d7
1 1d5 1 1dc
1 1e1 1 1e6
1 1eb 1 1f0
1 1f5 1 1fa
1 1ff 1 204
1 209 1 20e
1 213 1 218
1 21d 1 222
1 227 1 22c
1 231 1 236
1 23b 1 240
1 247 1 24b
1 24f 3 24a
24e 252 1 28d
2 296 298 1
29a 1 29e 1
2bb 2 2b9 2bb
6 2bf 2c0 2c1
2c2 2c3 2c4 1
2db 2 2da 2db
2 2e1 2e3 2
2e6 2e8 1 2eb
2 2ee 2f0 1
2f3 2 2f5 2f6
1 2f9 2 2f8
2f9 2 2ff 301
2 304 306 1
309 2 30c 30e
1 311 2 313
314 1 317 2
316 317 2 31d
31f 2 322 324
1 327 2 32a
32c 1 32f 2
331 332 1 335
2 334 335 2
33b 33d 2 340
342 1 345 2
348 34a 1 34d
2 34f 350 1
353 2 352 353
2 359 35b 2
35e 360 1 363
2 366 368 1
36b 2 36d 36e
1 371 2 370
371 2 377 379
2 37c 37e 1
381 2 384 386
1 389 2 38b
38c 9 2c6 2d9
2f7 315 333 351
36f 38d 3a2 2
3aa 3b3 2 3b5
3b6 3 3ba 3bb
3bc 5 3c7 3c8
3c9 3ca 3cb 3
3cc 3d3 3e2 e
29c 2a0 2aa 2af
2b4 2b8 3b7 3bf
3e5 3e8 3fa 3ff
404 406 1d 1b5
1ba 1c4 1c9 1ce
1d3 1da 1df 1e4
1e9 1ee 1f3 1f8
1fd 202 207 20c
211 216 21b 220
225 22a 22f 234
239 23e 243 28e
1 40e 1 411
1 415 7 41d
428 42f 43a 445
450 452 1 418
1 45d c 465
46e 473 478 47d
486 48b 494 499
4a2 4a7 4a9 1
460 1 4b1 1
4b4 1 4b8 1
4bd 1 4c2 1
4c7 1 4e0 2
4df 4e0 1 4ed
2 4eb 4ed 1
4fa 1 4fc 4
4ea 4fd 502 509
1 50b 4 4d7
4de 50c 50e 4
4bb 4c0 4c5 4ca
1 519 1 51e
1 523 1 528
1 541 2 540
541 1 552 2
550 552 1 55f
1 561 4 54f
562 567 56e 1
570 4 538 53f
571 573 4 51c
521 526 52b 1
580 1 583 1
591 1 599 1
59c 1 5a5 1
5ab 1 5b2 1
5b7 1 5bc 1
5c1 1 5c6 1
5cb 1 5d0 1
5d5 1 5da 1
5df 1 5e6 1
5e4 1 5ee 1
60a 1 611 1
619 2 627 636
1 639 1 63b
1 63c 1 642
2 640 642 6
646 647 648 649
64a 64b 2 64d
662 1 66c 2
66e 66f 9 5f0
5f3 5fd 603 607
63f 670 683 685
e 592 5a6 5b0
5b5 5ba 5bf 5c4
5c9 5ce 5d3 5d8
5dd 5e2 5e9 1
692 1 695 1
69f 1 6a5 1
6aa 1 6af 1
6ba 2 6c8 6cb
1 6cd 2 6d9
6dc 1 6de a
6b7 6bd 6c0 6ce
6d1 6df 6e2 6ee
6f5 6f7 4 6a0
6a8 6ad 6b2 1
6ff 1 703 1
709 1 70c 1
71d 1 725 1
729 1 72d 3
728 72c 730 1
773 1 779 1
77e 1 783 1
78f 1 78d 1
794 1 799 1
79e 1 7a3 1
7a8 1 7ad 1
7b2 1 7b7 1
7bc 1 7c1 1
7c6 1 7cb 1
7d0 1 7d5 1
7da 1 7df 1
7e4 1 7e9 1
7ee 1 7f3 1
7f8 1 7fd 1
802 1 807 1
80c 1 811 1
816 1 81b 1
820 1 825 1
82a 1 82f 1
843 8 848 849
84a 84b 84c 84d
84e 84f 2 859
85c 1 85e 8
861 862 863 864
865 866 867 868
2 872 875 1
877 2 880 882
2 884 886 2
888 88a 1 88c
3 8ab 8ac 8ad
6 8b3 8b4 8b5
8b6 8b7 8b8 2
8c4 8c6 2 8c8
8ca 2 8cc 8ce
2 8d0 8d2 2
8d4 8d6 2 8d8
8da 2 8dc 8de
2 8e0 8e2 2
8e4 8e6 1 8e8
2 8ec 8ee 2
8f3 8f5 2 8ff
900 2 8fc 902
1 904 2 908
90a 2 90f 911
2 916 918 2
91d 91f b 8b9
8c0 8ea 8f1 8f8
906 90d 914 91b
922 933 1 93d
2 93b 93d 2
941 943 2 945
947 1 94a 1
94c 1 950 2
94e 950 1 954
2 953 954 2
95a 95c 2 95f
961 1 964 2
967 969 1 96c
2 96e 96f 1
972 2 971 972
2 978 97a 2
97d 97f 1 982
2 985 987 1
98a 2 98c 98d
1 990 2 98f
990 2 996 998
2 99b 99d 1
9a0 2 9a3 9a5
1 9a8 2 9aa
9ab 1 9ae 2
9ad 9ae 2 9b4
9b6 2 9b9 9bb
1 9be 2 9c1
9c3 1 9c6 2
9c8 9c9 1 9cc
2 9cb 9cc 2
9d2 9d4 2 9d7
9d9 1 9dc 2
9df 9e1 1 9e4
2 9e6 9e7 1
9ea 2 9e9 9ea
2 9f0 9f2 2
9f5 9f7 1 9fa
2 9fd 9ff 1
a02 2 a04 a05
7 970 98e 9ac
9ca 9e8 a06 a25
1 a39 2 a3b
a3c 1 a41 2
a40 a41 2 a46
a49 1 a4c 16
836 840 846 850
85f 869 878 87b
88e 897 89c 89f
8a2 8a5 8a8 8b0
936 939 94d a3d
a3f a4d 26 71e
774 77c 781 78b
792 797 79c 7a1
7a6 7ab 7b0 7b5
7ba 7bf 7c4 7c9
7ce 7d3 7d8 7dd
7e2 7e7 7ec 7f1
7f6 7fb 800 805
80a 80f 814 819
81e 823 828 82d
832 1 a58 1
a5d 1 a65 8
a67 a6b a72 a77
a7c a83 a8a a8c
2 a5b a60 1
a99 1 a9c 1
aa6 1 aac 1
ab1 1 ab6 1
abb 1 ac3 1
acf 1 ad5 2
add adf 1 ae1
1 aee 2 aed
aee 2 af4 af6
1 af8 2 afb
afd 5 ae3 ae6
aec af9 b00 2
b0b b0d 1 b0f
1 b14 2 b12
b14 3 b21 b28
b2a 1 b2c 9
ac5 ac9 acc ad2
ad8 b03 b06 b11
b2d 5 aa7 aaf
ab4 ab9 abe 1
b35 1 b39 1
b3d 1 b42 1
b47 1 b4c 1
b51 1 b56 1
b5b 1 b60 1
b65 1 b6c 1
b6f 1 b79 1
b8c 1 b91 1
b8e 1 b94 1
b9b 2 b99 b9b
1 ba1 1 ba6
1 bb5 1 bb9
4 bb7 bbb bbe
bc2 2 bd5 bd6
1 bdc 2 bda
bdc 1 be2 2
be6 be7 1 bf3
2 bf1 bf3 1
bf9 1 c02 4
bfb bfd c04 c07
1 c0b 2 c0d
c0e 4 be4 be9
bf0 c0f 1 c13
2 c15 c16 5
bc6 bcb bd2 bd9
c17 2 c19 c1a
4 ba3 ba9 bac
c1b 1 c26 2
c24 c26 2 c37
c38 1 c3e 2
c3c c3e 1 c44
1 c58 1 c60
5 c46 c4c c53
c5a c62 1 c6a
1 c6f 1 c6c
1 c72 1 c75
2 c78 c79 4
c2c c34 c3b c7a
1 c7e 2 c80
c81 2 c23 c82
2 c84 c85 5
b82 b85 b97 c86
c8c a b40 b45
b4a b4f b54 b59
b5e b63 b68 b7a
1 c95 1 c99
1 c9d 1 ca2
1 ca7 1 cac
1 cb1 1 cb6
1 cbb 1 cc0
1 cd8 2 cd6
cd8 1 ce4 2
ce2 ce4 1 cf0
2 cee cf0 1
cfd 1 d01 2
d03 d04 2 ced
d05 1 d09 2
d0b d0c 2 ce1
d0d 1 d11 2
d13 d14 2 d19
d1b 1 d1d 2
d23 d25 1 d27
2 d2d d2f 1
d31 9 cc8 ccb
cce cd5 d15 d1f
d29 d33 d3b 8
ca0 ca5 caa caf
cb4 cb9 cbe cc3
1 d45 1 d49
2 d48 d4c 1
d52 2 d5c d5e
2 d66 d65 2
d69 d6b 2 d72
d71 2 d6d d75
2 d78 d7a 2
d81 d80 2 d7c
d84 2 d87 d89
2 d90 d8f 2
d8b d93 2 d61
d96 1 d59 1a
a 10 16 1c
22 28 31 3a
43 4c 5d 113
1a4 40b 457 4ae
513 578 68a 6fc
a52 a91 b32 c92
d41 d9b 
1
4
0 
da6
0
1
28
21
c3
0 1 1 1 4 1 6 6
1 1 1 1 1 d d d
10 1 12 1 14 14 14 1
1 19 19 1 1c 1c 1c 1
1 0 0 0 0 0 0 0

7d5 14 0
24 1 0
57a 1 d
3 0 1
779 14 0
1c6 6 0
a54 1 18
724 14 16
b65 1c 0
b5b 1c 0
794 14 0
6a5 12 0
5df d 0
204 6 0
12f 4 0
9b 3 0
799 14 0
6aa 12 0
ab1 19 0
5d0 d 0
1f5 6 0
120 4 0
8c 3 0
d45 21 0
aac 19 0
a5d 18 0
222 6 0
c94 1 20
1cb 6 0
ca7 20 0
7ad 14 0
5cb d 0
213 6 0
1f0 6 0
11b 4 0
87 3 0
7cb 14 0
459 1 a
7f8 14 0
236 6 0
45 1 0
7b2 14 0
7d0 14 0
c 1 0
d52 21 0
c95 20 0
b35 1c 0
7bc 14 0
4b0 1 b
7da 14 0
811 14 0
2a 1 0
5b7 d 0
81b 14 0
14a 4 0
82a 14 0
7e9 14 0
227 6 0
d44 1 21
5f 1 3
20e 6 0
57f d e
138 4 0
b60 1c 0
79e 14 0
21d 6 0
7a3 14 0
24f 7 0
a93 1 19
78d 14 0
5e4 d 0
1d5 6 0
71 3 0
67 3 0
c9d 20 0
b6c 1d 0
b3d 1c 0
a99 1a 0
ab6 19 0
a58 18 0
72d 16 0
709 15 0
82f 14 0
692 13 0
6af 12 0
580 e 0
5b2 d 0
523 c 0
4c2 b 0
45d a 0
415 9 0
247 7 0
1b2 6 0
6c 3 0
515 1 c
598 d f
13d 4 0
802 14 0
240 6 0
d49 21 0
608 10 0
820 14 0
7f3 14 0
5da d 0
519 c 0
4b8 b 0
231 6 0
1ff 6 0
12a 4 0
96 3 0
33 1 0
b51 1c 0
b47 1c 0
7ee 14 0
4b1 b 0
22c 6 0
1e1 6 0
1a7 6 0
b56 1c 0
80c 14 0
615 11 0
5ab d 0
cac 20 0
807 14 0
5c6 d 0
1eb 6 0
116 4 0
82 3 0
b34 1 1c
68c 1 12
3c0 8 0
1b7 6 0
159 5 0
1a6 1 6
40e 9 0
1bc 6 0
cc0 20 0
cbb 20 0
cb6 20 0
b4c 1c 0
b42 1c 0
6ff 14 0
6fe 1 14
cb1 20 0
7c6 14 0
5bc d 0
246 6 7
209 6 0
1dc 6 0
78 3 0
7e4 14 0
1d0 6 0
4f 1 2
4 1 0
ca2 20 0
783 14 0
218 6 0
115 1 4
7a8 14 0
5c1 d 0
528 c 0
4c7 b 0
1e6 6 0
7d 3 0
60 3 0
3c 1 0
1e 1 0
18 1 0
12 1 0
b6b 1c 1d
a98 19 1a
7fd 14 0
7c1 14 0
77e 14 0
708 14 15
691 12 13
51e c 0
4bd b 0
23b 6 0
1ab 6 0
7df 14 0
40d 1 9
725 16 0
825 14 0
24b 7 0
abb 19 0
729 16 0
816 14 0
599 f 0
142 4 0
7b7 14 0
5d5 d 0
1fa 6 0
125 4 0
91 3 0
0

/
/*
set timing on
variable rc refcursor;
begin
end;
/
print rc
prompt @@  content of mgmt_snapshot
SELECT snap_id FROM dbsnmp.mgmt_snapshot;
prompt @@  content of mgmt_baseline_sql
SELECT sql_id, hash_value, executions, elapsed_time FROM dbsnmp.mgmt_baseline_sql
order by hash_value;
prompt @@  content of mgmt_capture_sql
select capture_id, sql_id, hash_value, executions, elapsed_time from dbsnmp.mgmt_capture_sql
order by hash_value, capture_id;
prompt @@  content of mgmt_history_sql
select capture_id, sql_id, hash_value, executions, elapsed_time, t_per_exec, adjusted_elapsed_time
from dbsnmp.mgmt_history_sql
order by capture_id, sql_id, hash_value DESC;
prompt @@ unique capture_id in mgmt_history_sql
select unique capture_id 
from dbsnmp.mgmt_history_sql;
*/
show errors
