#!/usr/local/bin/perl
# 
# dgutil.pl
# 
# Copyright (c) 2002, 2005, Oracle. All rights reserved.  
#
#    NAME
#      dgutil.pl
#
#    DESCRIPTION
#      Data Guard utility routines. Used only for 10.1 DG remote ops/jobs. 
#      Used for 10.1 OMS -> 10.2 Agent compatibility.
#
#    NOTES
#      <other useful comments, qualifications, etc.>
#
#    MODIFIED   (MM/DD/YY)
#    sjconnol    06/24/05 - Move get_dg_dbres_status here from dgDatabases.pl
#    sjconnol    06/17/05 - Bug 4439779: replace content from dgutil_core.pl
#    sjconnol    06/02/05 - Bug 4407481: Move content to dgutil_core.pl
#    sjconnol    05/12/05 - Bug 4366174: Put back executeSQLPlus methods
#    sjconnol    10/12/04 - Remove startObserver
#    pbantis     05/17/04 - Remove executeSQLPlus methods. 
#    gallison    05/14/04 - Support for non-broker metrics
#    sjconnol    05/07/04 - Use observer logfile
#    sjconnol    04/22/04 - Add startObserver
#    sjconnol    03/16/04 - Remove getLDA
#    sjconnol    11/18/03 - Return val from executeSQLPlus
#    sjconnol    11/10/03 - Bug 3193231
#    gallison    10/13/03 - Fix 10i to 10g 
#    gallison    06/04/03 - Add dgError sub
#    sjconnol    02/26/03 - Add SIGCHLD handler; redo SQPLUS error handling
#    sjconnol    02/10/03 - Null-out username & password if not passed-in
#    pbantis     01/31/03 - Change size of buffer in get_dg_property().
#    sjconnol    01/09/03 - Add getLDA, waitForDG
#    sjconnol    12/18/02 - Make use of connect desc and db creds optional
#    pbantis     12/04/02 - Add handleError/Warning subroutines
#    pbantis     11/20/02 - Add flush
#    sjconnol    11/16/02 - misc changes
#    pbantis     11/12/02 - pbantis_update021101
#    pbantis     11/11/02 - Added more methods.
#    sjconnol    10/29/02 - Change SQLPLUS failure error message
#    sjconnol    10/17/02 - Add additional routines
#    sjconnol    08/01/02 - Creation
# 
use File::Spec;
use DBI;
use DBI qw(:sql_types);
use POSIX "sys_wait_h";
require "emd_common.pl";
require "flush.pl";
require "$ENV{EMDROOT}/sysman/admin/scripts/db/db_common.pl";
use vars qw/ $TEST_101 /;

## Only include dbstate if &executeSQLPlus is not defined, which
##  means this is likely a 10.1.0.X OMS -> 10.2 Agent
##  (executeSQLPlus is pulled in from the OMS side when running from 
##   10.2 OMS, but
##   must be obtained from from the Agent side when running from 10.1.0.X OMS.)
## 10.2 OMS -> 10.2 Agent: executeSQLPlus pulled from ha_dbstate on OMS side
## 10.2 OMS -> 10.1.0.X Agent: same
## 10.1.0.X OMS -> 10.2 Agent: executeSQLPlus pulled from dbstate.pl on Agent side
## 10.1.0.X OMS -> 10.1.0.X Agent: executeSQLPlus pulled from dgutil.pl on Agent side
if($TEST_101 && !defined(&executeSQLPlus)){
  debug("dgutil: including dbstate.pl");  
  require "$ENV{EMDROOT}/sysman/admin/scripts/db/dbstate.pl";
}
else{
  debug("dgutil: not including dbstate.pl");  
}

## Handle DBI sql errors
sub handleError{
  dgError("dgutil.handleError()");
  if(defined($DBI::errstr)){
    dgError("SQL Error: $DBI::errstr");
    die("SQL Error: $DBI::errstr\n");
  }
  else{
    dgError("$_[0]");
    die("$_[0]\n");
  }
}

## Handle DBI sql warnings
sub handleWarning{
  debug("dgutil.handleWarning()");
  if(defined($DBI::errstr)){
    debug("SQL Warning: $DBI::errstr");
  }
  else{
    debug("$_[0]");
  }
}

# Execute a Broker command.
# Parameters
# lda - The login identifier (Login Data Area).
# input_document - The Broker XML input document.
# Returns the Broker XML output document.
sub get_dg_document
{
  my ($lda, $input_document) = @_;

  my $do_control_raw = q{
    begin
      sys.dbms_drs.do_control_raw(:indoc, :outdoc, :rid, :piece, :context);
    end;
  };

  my $request_id = 0;
  my $output_document = "";
  my $request_id = "";
  my $piece_id = 1;
  my $context = "";

  my $dbcur = $lda->prepare($do_control_raw);

  $dbcur->bind_param(":indoc", $input_document, SQL_BINARY);
  $dbcur->bind_param_inout(":outdoc", \$output_document, 4096, SQL_BINARY);
  $dbcur->bind_param_inout(":rid", \$request_id, 16);
  $dbcur->bind_param(":piece", $piece_id);
  $dbcur->bind_param(":context", $context);

  $dbcur->execute;

  return $output_document;
}

# Retrieves a Data Guard property value.
# Parameters
# lda - The login identifier (Login Data Area).
# object_id - The id of the object.
# property_name - The property name.
# Returns the property value.
sub get_dg_property
{
  my ($lda, $object_id, $property_name) = @_;

  my $get_property_obj = q{
    begin
      :prop_value := sys.dbms_drs.get_property_obj(:id, :prop_name);
    end;
  };

  my $property_value = "";

  my $dbcur = $lda->prepare($get_property_obj);

  $dbcur->bind_param(":id", $object_id);
  $dbcur->bind_param(":prop_name", $property_name);
  $dbcur->bind_param_inout(":prop_value", \$property_value, 1024);

  $dbcur->execute;

  return $property_value;
}

# Given a name, get it's value from a Data Guard XML output document.
# Parameters
# document - The Data Guard output document.
# token_name - The token name.
# start_pos - IN/OUT - The starting search position.
# Returns the token value.
sub get_dg_token
{
  my ($document, $token_name, $start_pos) = @_;

  my $token_value = "";
  if (($start_pos = index($document, $token_name, $start_pos)) > -1)
  {
    # Skip over token_name + '="'
    $start_pos = $start_pos + length($token_name) + 2;

    my $end_pos = -1;
    my $length = 0;
    if (($end_pos = index($document, "\"", $start_pos)) > -1)
    {
      $length = $end_pos - $start_pos;
      $token_value = substr($document, $start_pos, $length);
      $start_pos = $end_pos + 1;
    }
  }
  $_[2] = $start_pos;
  return $token_value;
}

# Map the Data Guard Broker status to a status used by the UI.
# Parameters
# status - IN/OUT - The status of the object.
# Returns the token value.
sub map_dg_status
{
  my ($status) = @_;

  if ($status =~ /SUCCESS/i)
  {
    $_[0] = "Normal";
  }
  elsif ($status =~ /WARNING/i)
  {
    $_[0] = "Warning";
  }
  elsif ($status =~ /FAILURE/i)
  {
    $_[0] = "Error";
  }

}

# Get a database initialization parameter.
# Parameters
# lda - The login identifier (Login Data Area).
# param_name - The parameter name.
# Returns the parameter value.
sub get_init_param
{
  my ($lda, $param_name) = @_;

  my $sql = "select value from v\$parameter where name='$param_name'";
  my $dbcur = $lda->prepare($sql);
  $dbcur->execute;

  my @row = $dbcur->fetchrow_array();
  my $param_value = $row[0];
  return $param_value;
}

# Verify the database role. If not primary, exit.
sub verify_role
{
  my ($db_role) = @_;  

  EMD_PERL_DEBUG("Database role=$db_role");
  if (!($db_role =~ /Primary/i))
  {
    EMD_PERL_INFO("In order to run a Data Guard metric, the database must be in the primary role with at least one standby.");
    exit 0;
  }
}

# Get the database version from v$version.
# Parameters
# lda - The login identifier (Login Data Area).
# Returns the db version such as "92" for 9iR2, "100" for 10iR1.
# Obsolete method (for now - 10/01/02).
sub get_dbversion
{
  my ($lda) = @_;

  # Get the database banner.
  my $sql = "select banner from v\$version";
  my $dbcur = $lda->prepare($sql);
  $dbcur->execute;

  my @row = $dbcur->fetchrow_array();
  my $db_version_string = $row[0];
  #EMD_PERL_DEBUG("Database version string=$db_version_string");

  # Parse out the version number.
  my(@db_version_tokens, @text_tokens, $major_version);
  @db_version_tokens = split(/\./, $db_version_string);
  @text_tokens = split(/\ /, $db_version_tokens[0]);
  $major_version = pop(@text_tokens);

  my $db_version = $major_version . $db_version_tokens[1];
  return $db_version;
}

# Verify that the database version is 9iR2 or 10gR1 or 10gBeta.
# If it's not the correct version, then exit the program.
# Parameters
# db_version - The database version (9iR2, 10gR1, 10gBeta).
sub verify_dbversion
{
  my ($db_version) = @_;  

  EMD_PERL_DEBUG("Database version=$db_version");
  if (!($db_version =~ /9iR2/i) && !($db_version =~ /^10g/i))
  {
    EMD_PERL_INFO("In order to run a Data Guard metric, the database version must be 9.2 or higher.");
    exit 0;
  }
}

# Return the initialization parameter 'dg_broker_start'.
# Caller now decides how to handle it.
# Return Parameter
#  TRUE/FALSE
sub verify_broker
{
  my ($lda) = @_;
  my $dg_broker_start = get_init_param($lda, "dg_broker_start");
  EMD_PERL_DEBUG("dg_broker_start=$dg_broker_start");
  return $dg_broker_start;
}

# Verify that the Data Guard Configuration is enabled.
# If it's not, then exit the program.
# Parameters
# lda - The login identifier (Login Data Area).
sub verify_config_enabled
{
  my ($lda) = @_;

  my $drc_enabled = get_dg_property($lda, "0", "ENABLED");
  EMD_PERL_DEBUG("drc_enabled=$drc_enabled");
  # The configuration must be enabled.
  if (!($drc_enabled =~ /YES/i))
  {
    EMD_PERL_ERROR("The Data Guard configuration must be enabled.");
    $lda->disconnect;
    die "The Data Guard configuration must be enabled.\n";
  }
}

sub waitForDG{
  my($lda) = $_[0];
  debug("dgutil.waitForDG: *** START ***");
  my($WAIT_FOR_DATAGUARD_MASK) = 1 | 8 | 16 | 256 | 1024 | 2048;

  my($MAX_TRIES) = 180;
  my($sql)= "SELECT FLAGS FROM X\$RFMP";

  my($count) = 0;
						
  while ($count < $MAX_TRIES){
    my $dbcur = $lda->prepare($sql);
    $dbcur->execute;
    my @row = $dbcur->fetchrow_array();
    if(!defined(@row) || !defined($row[0])){
      debug("dgutil.waitForDG: can't determine Data Guard state");
      return 0;
    }

    my($value) = $row[0];
    if (($value & $WAIT_FOR_DATAGUARD_MASK) == 1){
      debug("dgutil.waitForDG: Waited $count seconds for Data Guard to start");
      return 1;
    }
    ## Wait 1 second between retries
    sleep(1);
    $count++;
  }

  debug("dgutil.waitForDG: Timed out after $count seconds waiting for Data Guard to start");
  return 0;
}

# Get the db resource status.
# Parameters
# lda - The login identifier (Login Data Area). 
# dbres_id - The database resource id.
# dbres_status - OUT - The database resource status.
# dbres_status_text - OUT - The database resource status text.
sub get_dg_dbres_status
{
  my ($lda, $dbres_id, $dbres_status, $dbres_status_text) = @_;

  # Retrieve the db resource STATUS property.
  my $indoc = "<DO_CONTROL><DO_COMMAND type=\"GetStatus\" object_id=\"$dbres_id\"/></DO_CONTROL>";
  my $dbres_big_status = get_dg_document($lda, $indoc);
  # EMD_PERL_DEBUG("dbres_big_status=$dbres_big_status");

  my @dbres_status_tokens = split(/[><]+/, $dbres_big_status);
  # Token 0 is "", token 1 is "RESULT ", token 2 is "MESSAGE "
  $dbres_status = $dbres_status_tokens[3];
  my @temp_tokens = split(/\s+/, $dbres_status);
  $dbres_status = $temp_tokens[0];
  EMD_PERL_DEBUG("dbres_status=$dbres_status.");

  if ($dbres_status eq "SUCCESS")
  {
    $dbres_status_text = "ORA-00000: normal, successful completion";
  }
  else
  {
    # Parse out the error from the big status.
    my $dbres_status_token;
    my $token_count = 0;
    foreach $dbres_status_token (@dbres_status_tokens)
    {
      # EMD_PERL_DEBUG("dbres_status_token=$dbres_status_token.");
      if ($dbres_status_token eq "ERROR_TEXT ")
      {
        $dbres_status_text = $dbres_status_tokens[$token_count + 1];
        last;
      }
      $token_count++;
    }
  }
  EMD_PERL_DEBUG("dbres_status_text=$dbres_status_text");
  map_dg_status($dbres_status);

  # Pass the values back.
  $_[2] = $dbres_status;
  $_[3] = $dbres_status_text;
}

sub printDebug{
  print "$_[0]\n";
	flush(STDOUT);
  debug($_[0]);
}

sub debug{
  EMD_PERL_DEBUG($_[0]);
}

sub dgError{
  EMD_PERL_ERROR($_[0]);
}

1;
