# $Header: hanganalyze.pl 18-may-2005.16:51:40 jsoule Exp $
#
# hanganalyze.pl
# 
# Copyright (c) 2004, 2005, Oracle. All rights reserved.  
#
#    NAME
#      hanganalyze.pl - Script for hang analysis
#
#    DESCRIPTION
#      This script invokes ORADEBUG HANGANALYZE and returns
#      the <blocking-session, blocked-session> pairs
#
#    NOTES
#      -
#
#    MODIFIED   (MM/DD/YY)
#    jsoule      05/31/05 - get trace file properly 
#    zsyed       04/19/05 - Fixing GC new issue for GRID AGENT 
#    zsyed       02/02/05 - Fixing new break 
#    zsyed       11/22/04 - Removing references to emdw 
#    zsyed       10/27/04 - Retrieving OS pid 
#    zsyed       06/22/04 - Creation
#

use strict;

require "emd_common.pl";

my $oracle_home = trim($ARGV[0]);
my $oracle_sid  = trim($ARGV[1]);
my $sqlcommfile = trim($ARGV[2]);

################################
# Produce an ORADEBUG HANGANALYZE trace file and get its name.
# Also record ORA- errors.
################################

my $trace_file = '';
my @errors;

{
  # Locally instantiate the relevant pieces of the ENV array so that SQL*Plus
  #  is spawned in the correct context.
  # Override the ORACLE_HOME, ORACLE_SID, LD_LIBRARY_PATH env variables.
  local $ENV{'ORACLE_HOME'}     = $oracle_home;
  local $ENV{'ORACLE_SID'}      = $oracle_sid;
  local $ENV{'LD_LIBRARY_PATH'} = $oracle_home."/lib:".$ENV{'LD_LIBRARY_PATH'};

  # Open a pipe from SQL*Plus and accept its output.
  open(SQLPLUS_OUTPUT, "$oracle_home/bin/sqlplus -s \"/ as sysdba\" < $sqlcommfile |");
}
while (<SQLPLUS_OUTPUT>)
{
  # Look for the line matching "Hang Analysis in <my trace file>"
  if (/Hang Analysis in (.*)/)
  {
    $trace_file = $1;
    last;
  }
  elsif (/(ORA-\d+:.*)/)
  {
    # Gather up errors in case no trace file can be found.
    $errors[++$#errors] = $1;
  }
}
close(SQLPLUS_OUTPUT);

if (!$trace_file)
{
  my $err_msg = "ORADEBUG HANGANALYZE could not be run";
  if ($#errors)
  {
    $err_msg .= " [" . join(', ',@errors) . "]";
  }
  die $err_msg;
}
elsif (EMAGENT_isPerlDebugEnabled())
{
  EMAGENT_PERL_DEBUG("ORADEBUG HANGANALYZE trace file: $trace_file");
}

################################
# Open file and read content lines into an array
################################

open(INFO, $trace_file) or die "Cannot open file $trace_file";
while (<INFO>)
{
  if (/State of nodes/)
  {
    # skip all lines upto and including this one
    last;
  }
}
my @waitfor_lines = ();
while (<INFO>)
{
  if (/END OF HANG ANALYSIS/)
  {
    # back out the one before this match...
    $#waitfor_lines--;

    # ...and break
    last;
  }
  else
  {
    # this is a content line; take off the trailing '\n'...
    chop;

    # ...and add it to the list
    $waitfor_lines[++$#waitfor_lines] = $_;
  }
}
close(INFO);

################################
# Find positions of columns of interest
################################

my $i = 0;
my $nodenumcol = 0;
my $sidcol = 0;
my $sesssrncol = 0;
my $sesscol = 0;
my $adjlistcol = 0;
my $predlistcol = 0;
my $pidcol = 0;

my $colname;
my @colnames = split("/", $waitfor_lines[0]);

foreach $colname (@colnames)
{
    if($colname =~ /nodenum/)
    {
	$nodenumcol = $i;
    }
    elsif($colname =~ /ospid/)
    {
        $pidcol = $i;
    }
    elsif($colname =~ /sid/)
    {
	$sidcol = $i;
    }
    elsif($colname =~ /adjlist/)
    {
	$adjlistcol = $i;
    }
    elsif($colname =~ /sess_srno/)
    {
	$sesssrncol = $i;
    }
    elsif($colname =~ /session/)
    {
	$sesscol = $i;
    }
    elsif($colname =~ /predecessor/)
    {
	$predlistcol = $i;
    }
    
    $i++;
}

################################
# Return data
################################

# First write number of total sessions found
print "em_result=$#waitfor_lines|";

my $count;
my $adj_entry;
my @colvalues;

# Then only return blocked/blocking sessions with columns of interest being
# communicated.
for($count = 1; $count < @waitfor_lines; $count++)
{
  @colvalues = split("/", $waitfor_lines[$count]);
    
  # Return only blocked or stuck sessions.
  # Blocked sessions have a non-"none" predecessor list.
  # Root blocking sessions have no adjacency list.
  if ($colvalues[$adjlistcol] || ($colvalues[$predlistcol] ne "none"))
  {
    # If session is a root blocker, return "none" as adjacency list element
    if (!$colvalues[$adjlistcol])
    {
      $adj_entry = "none";
    }
    else
    {
      $adj_entry = $colvalues[$adjlistcol];
    }
    
    # Return nodenum, sid, serial#, address, blockers
    print "$colvalues[$nodenumcol],";
    print "$colvalues[$sidcol],";
    print "$colvalues[$sesssrncol],";
    print "$colvalues[$sesscol],";
    print "$adj_entry,";
    print "$colvalues[$pidcol]|";
  }
}
print "\n";

################################
# Remove ORADEBUG HANGANALYZE trace file
################################

unlink($trace_file);

sub trim {
 my $string = shift(@_);
 $string =~ s/\s//g;
 return $string;
}

exit 0;

