#!/usr/local/bin/perl
#
# $Header: config_files.pl 26-jun-2006.11:25:59 jsutton Exp $
#
# Copyright (c) 2005, 2006, Oracle. All rights reserved.  
#
#    NAME
#      config_files.pl
#
#    DESCRIPTION
#      Script to parse the config files.
#
#    NOTES
#      N/A
#
#    MODIFIED   (MM/DD/YY)
#    jsutton     06/16/06 - Skip getpwuid call on Win32 
#    fle         11/02/05 - Remove all *_tnsnames_*, *_sqlnet_*, *_listener_* metrics.
#			  - Parse name/value instead of line by line.
#    fle         10/31/05 - [Bug# 4696815] Strip whitespace (except newline) characters at the end of the line.
#    fle         10/09/05 - Fixed localtime
#    fle         09/22/05 - Add error handling
#    fle         09/21/05 - Changed forms_machine_sid_dbc metric to dbc_file
#    fle         09/20/05 - Added file mode and uid file's owner info
#    fle         09/20/05 - Added wfmail_cfg and forms_machine_sid_dbc metrics
#    fle         09/16/05 - Added appsora_env and applsys_env metrics
#    fle         09/11/05 - Created
#

use strict;

use File::stat;
use File::Spec;
use Time::localtime;
use TXK::XML();
use TXK::Common();
use ias::asecm;
use Config;

require "$ENV{CCR_HOME}/sysman/admin/scripts/emd_common.pl";

my $contextFile = $ARGV[0];
my $metric = $ARGV[1];

if (!(-r $contextFile)) {
    EMD_PERL_ERROR("Cannot access context file: $contextFile");
    exit(1);
}
my $xmlObj = TXK::XML->new();
$xmlObj->loadDocument( { 'file' => $contextFile  } );

if ($metric eq 'appsweb_cfg') {
    my $file = $xmlObj->getOAVar('s_f60webcfg');
    printPropertiesWithSections($file, ';');
} elsif ($metric eq 'appsweb_cfg_meta') {
    my $file = File::Spec->catfile($xmlObj->getOAVar('s_oahtmlbin'), 'appsweb.cfg');
    printFileInfo($file);
} elsif ($metric eq 'wfmail_cfg') {
    my $file = File::Spec->catfile($xmlObj->getOAVar('s_fndtop'), 'resource', 'wfmail.cfg');
    printProperties($file, '#', ('CONNECT'));
} elsif ($metric eq 'wfmail_cfg_meta') {
    my $file = File::Spec->catfile($xmlObj->getOAVar('s_fndtop'), 'resource', 'wfmail.cfg');
    printFileInfo($file);
} elsif ($metric eq 'dbc_file') {
    my $file = File::Spec->catfile($xmlObj->getOAVar('s_fndtop'), 'secure', $xmlObj->getOAVar('s_hostname').'_'.$xmlObj->getOAVar('s_dbSid').'.dbc');
    printProperties($file, '#', ('GUEST_USER_PWD', 'APPL_SERVER_ID', 'GWYUID'));
} elsif ($metric eq 'dbc_file_meta') {
    my $file = File::Spec->catfile($xmlObj->getOAVar('s_fndtop'), 'secure', $xmlObj->getOAVar('s_hostname').'_'.$xmlObj->getOAVar('s_dbSid').'.dbc');
    printFileInfo($file);
} elsif ($metric eq 'appsora_env_meta') {
    my $file = $xmlObj->getOAVar('s_appsora_file');
    printFileInfo($file);
} elsif ($metric eq 'applsys_env_meta') {
    my $file = $xmlObj->getOAVar('s_applsys_file');
    printFileInfo($file);
} else {
    print "em_error=Unsupported metric: $metric";
    EMD_PERL_ERROR("Unsupported metric: $metric");
    exit(1);
}

sub getOraFilePath {
    my $fileName = shift(@_);
    my $filePath = File::Spec->catfile($ENV{'ORACLE_HOME'}, 'network', 'admin', $ENV{'TWO_TASK'}, $fileName);
    if (-r $filePath) {
        return $filePath;
    } elsif (-r File::Spec->catfile($ENV{'TNS_ADMIN'}, $fileName)) {
        $filePath = File::Spec->catfile($ENV{'TNS_ADMIN'}, $fileName);
    } else {
        $filePath = File::Spec->catfile($ENV{'ORACLE_HOME'}, 'network', 'admin', $fileName);
    }
    return $filePath;
}

sub printFileInfo($) {
    my $file = shift(@_);
    if (-r $file) {
        open(PROPFILE, $file);
        my $sf = stat($file);
        my $mode = sprintf("%04o", $sf->mode & 07777);
        my $owner;
        if ($Config{'osname'} eq "MSWin32")
        {
          $owner = " ";
        }
        else
        {
          $owner = sprintf("%s", getpwuid($sf->uid));
        }
        my $t = localtime($sf->mtime);
        my $mtime = sprintf("%4d-%02d-%02d %02d:%02d:%02d", $t->year()+1900, $t->mon()+1, $t->mday(), $t->hour(), $t->min(), $t->sec());
        printResult(
            $file, 'raw-string', # Can't use 'path' since it will prepend $OH.
            $file, 'raw-string',
            $sf->size, 'number',
            $mtime, 'string',
            $mode, 'string',
            $owner, 'string'
        );
        close(PROPFILE);
    } else {
        EMD_PERL_WARN("(Metric: $metric) Cannot access file: $file");
    }
}

sub printProperties($$;@) {
    my ($file, $ctag, @exprops) = @_;
    if (-r $file) {
        open(PROPFILE, $file);
        LINE: while (<PROPFILE>) {
            chomp;                  # no newline
            s/$ctag.*//;            # no comments
            s/^\s+//;               # no leading whitespace
            s/\s+$//;               # no trailing whitespace
            next unless length;     # anything left?
            foreach my $exprop (@exprops) {
                next LINE if (/^$exprop=/);
            }
            my ($name, $value) = split(/\s*=\s*/, $_, 2);
            print "em_result=$file|$name|$value\n";
        }
        close(PROPFILE);
    } else {
        EMD_PERL_WARN("(Metric: $metric) Cannot access file: $file");
    }
}

sub printPropertiesWithSections($$;@) {
    my ($file, $ctag, @exprops) = @_;
    if (-r $file) {
        open(PROPFILE, $file);
        my $currSection = 'GLOBAL';
        LINE: while (<PROPFILE>) {
            chomp;                  # no newline
            s/$ctag.*//;            # no comments
            s/^\s+//;               # no leading whitespace
            s/\s+$//;               # no trailing whitespace
            next unless length;     # anything left?
            foreach my $exprop (@exprops) {
                next LINE if (/^$exprop=/);
            }
            if (/^\[.+\]$/) {
                $currSection = $_;
                $currSection =~ s/^\[|\]$//g;
                next;
            }
            my ($name, $value) = split(/\s*=\s*/, $_, 2);
            print "em_result=$file|$currSection|$name|$value\n";
        }
        close(PROPFILE);
    } else {
        EMD_PERL_WARN("(Metric: $metric) Cannot access file: $file");
    }
}

