# 
# ecmRpasDomain.pm
# 
# Copyright (c) 2007, 2008, Oracle. All rights reserved.  
#
#    NAME
#      ecmRpasDomain.pm - OCM collection subroutines for Rpas Domains.
#
#    DESCRIPTION
#    This file contains subroutines to collect Rpas domain information for OCM.
#    The subroutines included require either RPAS_HOME or LL_ORACLE_HOME
#     to be set.
#

use strict;
use POSIX;
use RpasDomainInfo;
use ecmAssoc;
package ecmRpasDomain;

my $domainType = "";
my $targetType = "";

sub new
{
   my $class = shift;
   my $self = {};
   bless($self, $class);
   return $self;
}

# Collects domain size and prints it in the following format:
# em_result=<number of directories>|<number of files>|<size of files>|<number of databases>|<number of arrays>|<size of arrays>
sub printDomainSize {
   my $domain = shift;
   die "Domain not provided.\n" unless $domain;
   
   # getDomainInfo returns a hash table containing the values.  These values need to
   # be printed in the following order:
   # Directories | Files | File Size | Databases | Arrays | Array Size.
   my %info = &RpasDomainInfo::getDomainSize($domain);
   my $directories = %info->{'Directories'};
   my $files       = %info->{'Files'};
   my $fileSize    = %info->{'File Size'};
   my $databases   = %info->{'Databases'};
   my $arrays      = %info->{'Arrays'};
   my $arraySize   = %info->{'Array Size'};
   print "em_result=${directories}|${files}|${fileSize}|${databases}|${arrays}|${arraySize}\n";
}

# Collect the domain history and prints it in the following format:
# em_result=<version>|<action>|<date>
sub printHistory {
   my $domain = shift;
   die "Domain not provided.\n" unless $domain;
  
   # getHistory returns an array of hashes containing the 'version', 'action', and 'date' for each
   # item in the history.
   my @history = &RpasDomainInfo::getHistory($domain);
   foreach my $line (@history) {
      my %item = %$line;
      my $version = %item->{ 'version' };
      my $action  = %item->{ 'action' };
      my $date    = %item->{ 'date' };
      print "em_result=${version}|${action}|${date}\n";
   }
}

# Prints the association to the RpasServerApp
sub printRpasServerAppAssoc {
   my $rpasHome = shift;
   die "RPAS_HOME not provided\n" unless $rpasHome;
   print "em_result=${ecmAssoc::CONNECTS_TO}|${rpasHome}|${ecmAssoc::ORACLE_RETAIL_RPAS_SERVER}|\n";
}

# Prints all the simple domains associated with RPAS_HOME
sub printRpasSimpleDomainAssoc {
   my $rpasHome = shift;
   die "RPAS_HOME not provided\n" unless $rpasHome;
   my @domains = &RpasDomainInfo::getDomains($rpasHome, 'Simple');
   foreach my $domain (@domains) {
      print "em_result=${ecmAssoc::CONNECTS_TO}|${domain}|${ecmAssoc::ORACLE_RETAIL_RPAS_SIMPLE_DOMAIN}|\n";
   }
}

# Prints all the global and sub domains associated with RPAS_HOME
sub printRpasGlobalDomainAssoc {
   my $rpasHome = shift;
   die "RPAS_HOME not provided\n" unless $rpasHome;
   my @domains = &RpasDomainInfo::getDomains($rpasHome, 'Global');
   foreach my $domain (@domains) {
      print "em_result=${ecmAssoc::CONNECTS_TO}|${domain}|${ecmAssoc::ORACLE_RETAIL_RPAS_GLOBAL_DOMAIN}|\n";
      &printRpasSubDomainAssoc($domain);
   }
}

# Prints all the subdomains associated with a global domain
sub printRpasSubDomainAssoc {
   my $global = shift;
   die "Domain name not provided\n" unless $global;
   die "Domain is not a global domain\n" unless (&RpasDomainInfo::getType($global) eq 'Global');

   #  getSubdomainsFull returns a hash containing the following information:
   # { 'dim' } - Partition dimension
   # { 'subdomains' } - An array of subdomains in the following format:
   #                              { 'path' } - Path to the subdomain
   #                              { 'pos'  } - Partition position
   my %info = &RpasDomainInfo::getSubdomainsFull($global);
   my $subdomainsref = %info->{ 'subdomains' };
   my @subdomains = @$subdomainsref;
   foreach my $line (@subdomains)   
   {
      my %domain = %$line;
      my $path = %domain->{ 'path' };
      my $pos  = %domain->{ 'pos'  };
      print "em_result=${ecmAssoc::CONTAINS}|${path}|${ecmAssoc::ORACLE_RETAIL_RPAS_SUB_DOMAIN}|${pos}\n";
   }
}

# Prints a subdomain's association with its master domain
sub printRpasMasterDomainAssoc {
   my $subdomain = shift;
   die "Domain name not provided\n" unless $subdomain;
   die "Domain is not a subdomain\n" unless (&RpasDomainInfo::getType($subdomain) eq 'Subdomain');
   
   my %masterInfo = &RpasDomainInfo::getMasterInfo($subdomain);
   my $master = $masterInfo{ 'Master' };
   my $pos    = $masterInfo{ 'Partition Pos' };
   print "em_result=${ecmAssoc::MEMBER_OF}|${master}|$ecmAssoc::ORACLE_RETAIL_RPAS_GLOBAL_DOMAIN}|${pos}\n";
}

1;