# $Header: ecmSysData1.pl 30-may-2005.19:49:30 mgoodric Exp $
#
# ecmSysData1.pl
#
# Copyright (c) 2001, 2005, Oracle. All rights reserved.  
#
#   NAME
#       ecmSysData1.pl
#
#   DESCRIPTION
#       Collects Hardware and Software configuration data
#
#   NOTES
#
#   MODIFIED   (MM/DD/YY)
#    mgoodric   05/27/05 - add -help to print usage 
#    mgoodric   09/27/04 - code cleanup 
#    mgoodric   03/27/04 - Add all currently supported OSNAME platforms
#    mgoodric   03/17/04 - Added LC_ALL=C
#    mgoodric   06/16/03 - fix converting Solaris timezone to JAVA timezones
#    mgoodric   05/28/03 - files are now OSD-specific
#    mgoodric   05/11/03 - added -debug tracing
#    mgoodric   04/07/03 - fix finding MAC address
#    rlal       03/16/03 - Adding OS Registerd software
#    rlal       02/25/03 - EM 4.0 changes for Linux
#    goramamo   01/21/03 - Minor error msg changes
#    goramamo   09/11/02 - Remove unused code
#    xxu        06/25/02 - remove /usr/local/bin/perl
#    goramamo   06/01/02 - GIT5 Changes
#    goramamo   05/01/02 - Added code to detect all vendor products
#    goramamo   04/14/02 - Added code to check pkginfo files
#    goramamo   03/28/02 - Added code to generte XML from files
#    jmansur    02/18/02 - allow for no OS kernel parameters
#    mgoodric   02/14/02 - Make scripts more portable
#    jmansur    01/23/02 - collect OS kernel parameters
#    goramamo   11/28/01 - Added perl module directory name as second argument
#    goramamo   10/15/01 - Creation
#
##*********************************************************
##  Arguments
##*********************************************************
##  ARGV[0] - A keyword one among HARDWARE, OS_SOFTWARE, VENDOR_SOFTWARE, ALL
##          - (Must be supplied)
##  ARGV[1] - Directory where the user written Ptdpm*.pm modules are located.
##          - (Optional)
##  ARGV[2] - Directory where the saved data files for testing are saved.
##          - (Optional)
##  ARGV[3]..ARGV[n] - Additional options (e.g. -debug)
##          - (Optional)
##*********************************************************
##  PROGRAM/SUBROUTINES FLOW
##*********************************************************
## BEGIN
##     getParam
## initialize
##     setOutputAutoFlush
##     getParam
##     setDataDir
##     importModules
##     setupOSD
##     getOSname
## checkOSSupport
## checkArgs
## callModules
##     GetHardwareDetails
##         getSysSummary
##         appendsysXML
##         getHWDetails
##         appendhwXML
##     GetOSDetails
##         getOSSummary
##         appendossumXML
##         getOSConfiguration
##         appendOSConfHashXML
##         getMountedFileSystemDetails
##         appendfsXML
##         getSoftwareComponents
##         appendOSHashXML
##     GetVendorComponents
##         getSoftwareComponents
##         appendVendorHashXML
## finalize
## AUTOLOAD
## END
##*********************************************************
##   MODULE CALLS
##*********************************************************
use strict;

use Carp;
require 5.005;

#require 5.6.1;

use Cwd;
use File::Basename;

#******************************************
#    Global Variables
#******************************************
use constant NIL => '';

$main::COPYRIGHT       = 'Copyright (c) 2001, 2005, Oracle. All rights reserved.';
$main::VERSION         = '10.2.0.0';
$main::OSNAME          = NIL;
$main::DATADIR         = NIL;
$main::PROGMAIN        = File::Basename::basename($0);
$main::SCRIPTSDIR      = File::Basename::dirname($0);
$main::LIBDIR          = $main::SCRIPTSDIR;
$main::DEBUG           = 0;
$main::HELP            = 0;
$main::INDENT          = 0;
$main::ALL             = 0;
$main::HARDWARE        = 0;
$main::OS_SOFTWARE     = 0;
$main::VENDOR_SOFTWARE = 0;
$main::OS              = $^O;
$main::PERL            = $^X;

my $noofargs = scalar(@ARGV);
my @recdargs = @ARGV;
my $osflag   = NIL;

initialize();

if ($noofargs == 0)
{
    usage();
}
else
{
    $osflag = checkOSSupport();
    if ($osflag eq 'true')
    {
        checkArgs();
        callModules();
    }
}

finalize();

##================================================
##   SUBROUTINE DEFINITIONS
##================================================

sub usage
{
    print <<_EOF_

$main::PROGMAIN $main::VERSION
$main::COPYRIGHT

Usage: $main::PROGMAIN  [<option>] <component> [<path>]

 option
   -debug            : add debugging trace output
   -indent           : indent XML output

 component
   ALL               : output all component details
   HARDWARE          : output Hardware details
   OS_SOFTWARE       : output Operating System details
   VENDOR_SOFTWARE   : output OS-Registered Software details

 path                : absolute path to package modules

_EOF_
}

#================================================
#   get the n-th parameter argument
#================================================
sub getParam($)
{
    my ($index) = @_;
    my $i = 0;
    my $j = 0;
    my $value = NIL;
    
    $noofargs = scalar(@ARGV);

    if ($noofargs >= $index)
    {
        for ($i = 0; $i < $noofargs; $i++)
        {
            if ('-' ne substr($ARGV[$i],0,1))
            {
                $j++;
                if ($j == $index)
                {
                   $value = $ARGV[$i];
                   last;
                }
            }
        }
    }

    return $value;
}

#================================================
#   Do any pre/post setups
#================================================
BEGIN
{
    $main::SCRIPTSDIR = File::Basename::dirname($0);
    $main::LIBDIR     = $main::SCRIPTSDIR;

    $noofargs = scalar(@ARGV);
    chomp(my $libdir = main::getParam(2));
    if (($libdir ne NIL) && -d "$libdir")
    {
        $main::LIBDIR = $libdir;
    }
}

END
{
}

#================================================
#   Setup our program environment
#================================================
sub initialize
{
    my $perl = $main::PERL;    # makes perl -w happy!
    setOutputAutoflush();

    $ENV{'LC_ALL'} = 'C';      # makes cmds NLS-independent

    chomp(my $libdir = getParam(2));
    if (($libdir ne NIL) && -d "$libdir")
    {
        $main::LIBDIR = $libdir;
    }

    setDataDir();    # For Simulation Testing
    importModules();
    setupOSD($main::DATADIR);

    ## $main::OSNAME is a global variable
    $main::OSNAME = getOSname();
}

#================================================
#   Bring in our modules
#================================================
sub importModules
{

    ## List all the packages you want to import with their directories
    use lib($main::LIBDIR) || die "Not able to import";
    use ecmErrorMsgs;
    use Ptdpm0;
    use Ptdpm1;
    use Ptdpm2;
    use Ptdpm3;
    use Ptdpm4;
    use Ptdpm7;
    use Ptdpm8;
    use Ptdpm9;
    use Ptdpm11;
    use Ptdpm12;
    use Ptdpm14;
    use Ptdpm15;
    use Ptdpm16;
}

#================================================
#   Generate the requested output
#================================================
sub callModules
{

    # Purpose : To call the corresponding subroutines
    #           based on the argument passed by user.

    if ($main::HELP)
    {
        usage();

        return;
    }

    if ($main::ALL)
    {
        print typeBegin($Ptdpm2::INVENTORY);
    }

    if ($main::HARDWARE)
    {
        GetHardwareDetails();
    }

    if ($main::OS_SOFTWARE)
    {
        GetOSDetails();
    }

    if ($main::VENDOR_SOFTWARE)
    {
        GetVendorComponents();
    }

    if ($main::ALL)
    {
        print typeEnd($Ptdpm2::INVENTORY);
    }
}

#================================================
#   Output the HARDWARE xml
#================================================
sub GetHardwareDetails
{
    debugTrace('Calling... GetHardwareDetails()');

    ##---------SYSTEM SUMMARY--------------##
    my (@syssum)    = getSysSummary();
    my (@syssumXML) = appendsysXML(@syssum);
    print @syssumXML;

    ##---------HW DETAILS------------------##
    my (@hwdata)    = getHWDetails();
    my (@hwdataXML) = appendhwXML(@hwdata);
    print @hwdataXML;
}

#================================================
#   Output the OS_SOFTWARE xml
#================================================
sub GetOSDetails
{
    debugTrace('Calling... GetOSDetails()');

    my $n     = 0;
    my $k     = 0;
    my $size1 = 0;
    my (@ospatXML) = ();

    ##-------- OS SUMMARY------------------##
    my (@ossum)    = getOSSummary();
    my (@ossumXML) = appendossumXML(@ossum);
    print @ossumXML;

    ##----- OS Config and Parameters ------##
    my (%parms)    = getOSConfiguration();
    my (@parmsXML) = appendOSConfHashXML(%parms);
    print @parmsXML;

    ##----- Mounted File Systems -----------##
    my (@fs) = getMountedFileSystemDetails();
    $size1 = scalar(@fs);
    if ($size1 > 0)
    {
        my (@fsXML) = appendfsXML(@fs);
        print @fsXML;
    }

    ##----- OS Packages and Patches --------##
    my (%prods)    = getSoftwareComponents('OS');
    my (@prodsXML) = appendOSHashXML(%prods);
    print @prodsXML;
}

#================================================
#   Output the VENDOR_SOFTWARE xml
#================================================
sub GetVendorComponents
{
    debugTrace('Calling... GetVendorComponents()');

    ##----- Vendor Component --------------##
    my (%prods)    = getSoftwareComponents('VENDOR');
    my (@prodsXML) = appendVendorHashXML(%prods);
    print @prodsXML;
}

#================================================
#   Check for valid options and optional params
#================================================
sub checkArgs
{

    # Function :  Check the validity of all ARGV arguments
    # Input    :  @ARGV
    # Output   :  Valid Arg(s)

    #11/28/2001 Changes
    if ($noofargs < 1)
    {
        #throwECMError('NoArgs', $main::PROGMAIN);
        usage();

        return;
    }

    my $k        = 0;

    for ($k = 0 ; $k < $noofargs ; $k++)
    {
        if ($recdargs[$k] =~ m/^ALL$/i)
        {
            $main::ALL             = 1;
            $main::HARDWARE        = 1;
            $main::OS_SOFTWARE     = 1;
            $main::VENDOR_SOFTWARE = 1;
            $main::INDENT          = 1;
        }
        elsif ($recdargs[$k] =~ m/^HARDWARE$/i)
        {
            $main::HARDWARE = 1;
        }
        elsif ($recdargs[$k] =~ m/^OS_SOFTWARE$/i)
        {
            $main::OS_SOFTWARE = 1;
        }
        elsif ($recdargs[$k] =~ m/^VENDOR_SOFTWARE$/i)
        {
            $main::VENDOR_SOFTWARE = 1;
        }
        elsif ($recdargs[$k] =~ m/^-debug$/i || $recdargs[$k] =~ m/^-d$/i)
        {
            $main::DEBUG = 1;
        }
        elsif ($recdargs[$k] =~ m/^-indent$/i || $recdargs[$k] =~ m/^-i$/i)
        {
            $main::INDENT = 1;
        }
        elsif ($recdargs[$k] =~ m/^-help$/i || $recdargs[$k] =~ m/^-h$/i)
        {
            $main::HELP = 1;
        }
        else
        {
            ##print "Wrong argument $recdargs[$k] sent from Capture Module ";
            ##print "to ecmSysData1.pl\n Please check the calling program.\n";
        }
    }

    return;
}

#================================================
#   Check for the optional debugging datadir
#================================================
sub setDataDir
{
    my $dir  = NIL;
    my $base = NIL;

    chomp(my $datadir = getParam(3));
    if (($datadir ne NIL) && -d "$datadir")
    {
        if ($datadir eq "\.")
        {
            chomp($main::DATADIR = cwd());
            $main::DATADIR = $main::DATADIR . '/';
        }
        else
        {
            $dir = File::Basename::dirname($datadir);
            if ($dir eq '/') { $dir = NIL; }
            $base = File::Basename::basename($datadir);
            if ($base eq NIL)
            {
                $main::DATADIR = $datadir;
            }
            else
            {
                $main::DATADIR = $dir . '/' . $base . '/';
            }
        }
    }
}

#================================================
#   Check the platform we support
#================================================
sub checkOSSupport
{
    my $osflag = 'false';

    #$main::OSNAME = 'NewOS'; ## For testing the behaviour of non-supported OS

    # This file is generic.  Add any supported platforms to the following list:
    if (
        (index($main::OSNAME, 'SunOS') > -1)     # Solaris
        || (index($main::OSNAME, 'Linux') > -1)  # RedHat Linux, SuSE Linux, UnitedLinux
        || (index($main::OSNAME, 'HP-UX') > -1)  # HP
        || (index($main::OSNAME, 'AIX') > -1)    # IBM
        || (index($main::OSNAME, 'OSF1') > -1)   # HP Tru64 UNIX
        || (index($main::OSNAME, 'Darwin') > -1) # MAC OS X
       )
    {
        $osflag = 'true';
    }

    return $osflag;
}

#================================================
#   Do any cleanup
#================================================
sub finalize
{
}

#================================================
#  Test AUTOLOAD subroutines
#================================================
sub AUTOLOAD
{
    my (@arguments) = @_;
    my $args = join (',', @arguments);
    throwECMError('Failed',              $main::PROGMAIN);
    throwECMError('UndefinedSubroutine', $main::AUTOLOAD);
    throwECMError('PassedArguments',     $args);
}

# End of the Program
