#!/usr/local/bin/perl
# 
# $Header: em_br_ifc.pl 18-jan-2006.20:59:16 rvijjapu Exp $
#
# em_br_ifc.pl
# 
# Copyright (c) 2004, 2006, Oracle. All rights reserved.  
#
#    NAME
#      em_br_ifc.pl - <one-line expansion of the name>
#
#    DESCRIPTION
#      <short description of component this file declares/defines>
#
#    NOTES
#      <other useful comments, qualifications, etc.>
#
#    MODIFIED   (MM/DD/YY)
#    rvijjapu    01/18/06 - Backport rvijjapu_bug-4938521 from main 
#    akmahesh    12/21/05 - Backport akmahesh_bug-4757089 from main 
#    rvijjapu    01/12/06 - Fix for 4938521 
#    akmahesh    12/13/05 - Fix for 4757089 
#    rvijjapu    12/05/05 - Fix for 4859924 
#    rvijjapu    11/18/05 - Fix for 4741530 
#    rvijjapu    08/03/05 - Fix for 4452600 
#    rvijjapu    07/27/05 - Changes for interface invoke 
#    rvijjapu    07/15/05 - 
#    rvijjapu    05/18/05 - bkp_res_interface.pl
#    rvijjapu    05/05/05 - fix_4333484
#    rvijjapu    03/28/05 - Addding validate option 
#    rkuriako    04/04/05 - fix_4226960
#    rvijjapu    03/07/05 - fix 4173987 
#    rvijjapu    11/24/04 - 
#    rvijjapu    11/04/04 - rvijjapu_em_ias_br_002
#    rvijjapu    10/18/04 - Creation
#
# Include Libraries
use Getopt::Std;
use File::Basename;
use File::Copy;
use File::Path;

use strict;                # Disable automatic variables

#------------------------------------------------------------------------
# Global variables
#------------------------------------------------------------------------
my $debug                     = 0;
my $silent                    = 0;
my $verbose                   = 0;
my $version                   = "9.0.4";
my $backup_level              = 987654;
my $em_timestamp                 = "";
my $group_guid                = "";
my $target_guid               = "";
my $job_execution_id          = "";
my $force                     = 1;
my $param_str                 = "";
my $script_mode               = "";
my $env_file                  = "";
my $no_prompt                 = 0;
my $cur_date_time             = GetCurrentDateTime();
my $recovery_name             = "";
my $restore_controlfile       = 0;
my $interface_exists          = 0;
my $cluster                   = 0;
my $oracle_home               = "";
my $orig_oracle_home          = "";
my $oracle_sid                = "";
my $orig_oracle_sid           = "";
my $br_install_dir            = "";
my $install_type              = "";
my $log_path                  = "";
my $database_backup_path      = "";
my $config_backup_path        = "";
my $ias_instance              = "";
my $opts                      = "";

my $br_option                 = "backup_online";
my $backup_name               = "";
my $status                    = 0;

my @arr                       = ();

# This will hold the output of all external commands
my $output                    = "";

my $command_separator         = ";";
my $restore_command           = "restore_repos";
my $backup_success_msg        = "";
my $repos_backup_failure_msg  = "";
my $repos_restore_failure_msg = "";
my $perlexec                  = "perl";
my $br_interface_script       = "bkp_res_interface.pl";
my $br_new_shell              = "bkp_restore.sh";

#------------------------------------------------------------------------
# Main Logic
#------------------------------------------------------------------------

ParseCommandLine();
Initialize();
PerformOperation();
$ENV{ORACLE_HOME}=$orig_oracle_home;
$ENV{ORACLE_SID}=$orig_oracle_sid;
#------------------------------------------------------------------------
sub Initialize
{

    # Set Operating System level commands
    my $OSNAME = $^O;

    if(  ($OSNAME eq "MSWin32") or ($OSNAME eq "Windows_NT") )
    {
        $command_separator = " & ";
        $br_new_shell = "bkp_restore.bat";

        # For Windows change the path  from \ to /
        $br_install_dir =~ s/\//\\/;

    }
    else
    {
        $command_separator = ";";
        $br_new_shell = "bkp_restore.sh";
    }

    # Set Environment Variables
    $orig_oracle_home = $ENV{ORACLE_HOME};
    $ENV{ORACLE_HOME} = "$oracle_home";

    # If oracle_sid is not provided as the input argument, get it from
    # ias.properties file and set it in the environment in order to
    # query dbid.
    $orig_oracle_sid = $ENV{ORACLE_SID};
    if ($oracle_sid eq "")
    {
	$oracle_sid = GetOracleSID();
    }
    # ExtendedPrint("oracle_sid: $oracle_sid \n");
    $ENV{ORACLE_SID} = $oracle_sid;


    # Retrieve PERL_BIN from env variable to exec br tool script.
    $perlexec = "$ENV{PERL_BIN}/perl";

    # Set AS Version specific variables
    if ($version =~ m/^9.0.4/)
    {
        $restore_command = "restore_db";
        $backup_success_msg = "Backup directory is ";
    }
    else 
    {
        $restore_command = "restore_repos";
        $backup_success_msg = "Configuration backup archive is ";
        $repos_backup_failure_msg = "Unable to export file based repository using dcmctl";
        $repos_restore_failure_msg = "Unable to import file based repository using dcmctl";
    }
    if (-e "$br_install_dir/$br_interface_script")
    {
        $interface_exists = 1;
    }
    
}

#------------------------------------------------------------------------
sub Fatal
{
   ExtendedPrint ("Failure : $script_mode failed\n");
   my @print_text = (@_, "");
   print @print_text unless $silent;
   map { $output .= $_; } @print_text;

   exit (-1);
}

#------------------------------------------------------------------------
sub ParseCommandLine
{
   getopts('ncvsdfm:o:e:l:t:u:i:p:b:h:z');

   use vars qw( $opt_n $opt_c $opt_v $opt_s $opt_d $opt_f $opt_e $opt_m $opt_l $opt_t $opt_u $opt_i $opt_p $opt_b $opt_h $opt_z);

   $oracle_home     = $opt_h ? $opt_h : "";
   if ($oracle_home eq "") 
   {
      Fatal("Oracle Home not specified !!! \n");
   }
   $oracle_home     = trim($oracle_home);
   # For 9.0.4, -b <BR install directory> will be passed.
   $br_install_dir  = $opt_b ? $opt_b : "";

   # For 10.1.2, B/R tool will be installed in $oracle_home/backup_restore
   if ($br_install_dir eq "")
   {
      $br_install_dir = "$oracle_home/backup_restore";
   }

   $br_install_dir  = trim($br_install_dir);
   $env_file        = $opt_e ? $opt_e : "$br_install_dir/config/config.inp";
   $env_file        = trim($env_file);
   $script_mode     = $opt_m ? $opt_m : "unspecified";
   $script_mode     = trim($script_mode);
   $backup_level    = defined($opt_l) ? $opt_l : 987654;

   $restore_controlfile = $opt_c ? 1 : 0;
   $no_prompt       = $opt_n ? 1 : 0;
   if ($opt_d)
   {
      $debug = 1;
      $opts .= " -d";
   }
   if ($opt_s)
   {
      $silent = 1;
      $opts .= " -s";
   }
   if ($opt_v)
   {
      $verbose = 1;
      $opts .= " -v";
   }
   if ($opt_f)
   {
      $force = 1;
      $opts .= " -f";
   }
   if ($opt_z)
   {
      $opts .= " -z";
   }

   if ($interface_exists)
   {
      $opts .= " -h $oracle_home";
   }
   $opts .= " -e $env_file";

   $param_str       = $opt_p ? $opt_p : "";
   $param_str       = trim($param_str);
   GetOptions();
}

#------------------------------------------------------------------------
sub GetOptions
{
   # This method is used to parse the parameters from -p<param_str>
   # param_str will be in the format -> param1=value1,param2=value2
   @arr = split(/::/, $param_str);
   my $element = "";
   my  @flds;
   my $key="";
   my $value="";
   foreach $element (@arr)
   {
      @flds = trim(split(/=/, $element));
      if (defined($flds[0]) && defined($flds[1]))
      {
         if ($flds[0] eq "option")
         {
            $br_option = $flds[1];
         }
         elsif ($flds[0] eq "level")
         {
            $backup_level = $flds[1];
         }
         elsif ($flds[0] eq "version")
         {
            $version = $flds[1];
         }
         elsif ($flds[0] eq "restore_instance")
         {
            $recovery_name = $flds[1];
         }
         elsif ($flds[0] eq "config_backup_path")
         {
            $config_backup_path = $flds[1];
         }
         elsif ($flds[0] eq "log_path") 
         {
            $log_path = $flds[1];
         }
         elsif ($flds[0] eq "database_backup_path") 
         {
            $database_backup_path = $flds[1];
         }
         elsif ($flds[0] eq "oracle_home")
         {
         }
         elsif ($flds[0] eq "oracle_sid")
         {
            $oracle_sid = $flds[1];
         }
         elsif ($flds[0] eq "ias-instance") 
         {
             $ias_instance = $flds[1];
         }
         elsif ($flds[0] eq "backup_name")
         {
             $backup_name = $flds[1];
         }    
         elsif ($flds[0] eq "recovercontrolfiles") 
         {
             if ($flds[1] eq "Yes") 
             {
                 $restore_controlfile=1;
             }
         }
         elsif ($flds[0] eq "cluster") 
         {
             if ($flds[1] eq "Yes") 
             {
                 $cluster=1;
             }
         }
         else
         {
            Fatal("Invalid Option Specified!!!\n");
         }
      }  
   }
}

#------------------------------------------------------------------------
sub CheckBRTool
{
    # This function is used to check if the BR Tool is present in the 
    # specified location.
    my $OSNAME = $^O;
    ExtendedPrint($OSNAME);
    if( ! ($OSNAME eq "MSWin32") and !($OSNAME eq "Windows_NT") )
    {
        ExtendedPrint($br_install_dir);       
        open (BRDIR, $br_install_dir) or exit(100);
        close(BRDIR);
    }

    open (CONFIGFILE, $env_file) or exit(101);
    close(CONFIGFILE);
    return 0;
}

#------------------------------------------------------------------------
sub GetValueForConfigParameter
{
   # This method is used to get the value of a given parameter from config.inp
   my $param_name = shift;
   my $param_value;
   my @flds;

   open (CONFIGFILE, "<$env_file")
          or Fatal "Problem opening file $env_file for reading: $!\n";

   while (<CONFIGFILE>)
   {
      next if /^\s*#/; # Skip comments
      next if /^\s*$/; # Skip blank lines

      chomp ($_);
      @flds = split /=/ ;
      if (defined($flds[0]) && defined($flds[1]))
      {
         if ($flds[0] eq $param_name)
         {
            $param_value = $flds[1];
            # Once we found the required parameter, break from the loop.
            last;
         }
      }
   }
   close (CONFIGFILE);
   VerbosePrint("$param_name=$param_value\n");
   return $param_value;
}

#------------------------------------------------------------------------
sub GetOracleSID
{
   my $ias_properties_file = "${oracle_home}/config/ias.properties";
   my $infra_db_common_name;
   my $sid;
   my @flds;
   if (-e $ias_properties_file)
   {
      open (IASPROPERTIES, "<$ias_properties_file") or
         Fatal "Problem opening file $ias_properties_file for reading: $!\n";
      while (<IASPROPERTIES>)
      {
         next if /^\s*#/; # Skip comments
         next if /^\s*$/; # Skip blank lines

         chomp ($_);
         @flds = split /=/ ;

         if (defined($flds[0]) && defined($flds[1]))
         {
            if ($flds[0] eq "InfrastructureDBCommonName")
            {
               $infra_db_common_name = $flds[1];
               last;
            }
         }
      }
      close(IASPROPERTIES);
   }
   # The value before "." in infra_db_common_name should be oracle_sid.
   if (index ($infra_db_common_name, ".") != -1)
   {
      $sid = substr ($infra_db_common_name, 0, index ($infra_db_common_name, "."));
   }
   else
   {
      $sid = $infra_db_common_name;
   }
   
   return $sid;
}

#------------------------------------------------------------------------
sub GetDBID
{
   my $dbid;
   my $spaces;

   my $OSNAME = $^O;
   my $minusloption = " -L ";

    if(  ($OSNAME eq "MSWin32") or ($OSNAME eq "Windows_NT") )
    {
        $minusloption = "";
    }

   my $cmd = "${oracle_home}/bin/sqlplus $minusloption";
   $cmd .= " -s \"/ as sysdba\" \@${br_install_dir}/query_dbid.sql" ;

   my $rc = ExecSystemCall($cmd);

   # ExecSystemCall creates this output file.
   my $dbid_log = "$log_path/${cur_date_time}_output.log"; 

   open (LOGFILE, "<$dbid_log") or
     Fatal "Problem opening file $dbid_log for reading: $!\n";
   my $line;
   while (<LOGFILE>)
   {
      next if /^\s*$/;                # Skip blank lines
      $line = $_;
      # Look for any ORA errors
      if (substr($line, 0, 4) eq "ORA-")
      {
         close(LOGFILE);
         Fatal("Could not obtain the dbid..\n");
      }
      # Search for pattern with just digits
      ($spaces, $dbid) = m/(\s*)(\d*)/ ;
      if (defined($dbid) && $dbid ne "")
      {
         last;
      }
   }
   close(LOGFILE);
   # Confirm that we got a valid dbid
   if (!defined($dbid) || $dbid eq "")
   {
      Fatal("Invalid DBID !!!\n");
   }
   else
   {
      ExtendedPrint("DBID: $dbid\n");
   }
   unlink $dbid_log;
   return $dbid;
}
#------------------------------------------------------------------------
sub UpdateConfigParameter
{
   my $searchStr = shift;
   my $replaceVal = shift;
   my $msg;
   my @flds;
   my $newconfigfile = $env_file.$cur_date_time;

   open (NEWCONFIGFILE, ">$newconfigfile") or
         Fatal "Problem opening file $newconfigfile for writing: $!\n";

   open (CONFIGFILE, "$env_file") or
         Fatal "Problem opening file $env_file: $!\n";
   while (<CONFIGFILE>)
   {
      # Skip comments and blank lines
      if ((/^\s*#/) || (/^\s*$/))
      {
         print (NEWCONFIGFILE $_);
         next;
      }
      @flds = split /=/;
      if (defined($flds[0]) && defined($flds[1]))
      {
         if ($flds[0] eq $searchStr)
         {
            print (NEWCONFIGFILE "$searchStr=$replaceVal\n");
         }
         else
         {
            print (NEWCONFIGFILE $_);
         }
      }
   }
   close(NEWCONFIGFILE);
   close(CONFIGFILE);

   # Unlink the old file and rename new one
   unlink($env_file);
   rename("$newconfigfile", $env_file) or
          Fatal "can't rename $newconfigfile to $env_file: $!";
}

#------------------------------------------------------------------------
sub PerformOperation
{
   if ($script_mode eq "get_install_type")
   {
      # Get the install_type value from config.inp file.
      $install_type = GetValueForConfigParameter("install_type");

      # Check if this oracle home hosts a file-based repository.
      # Not sure, if we really need to expose FBR to the end-user - TBD
      my $isfbr = CheckFileBasedRepository();

      my $response = "";
      $response = "install_type=$install_type";
      $response .= ",";
      $response .= "isfbr=$isfbr";
      ExtendedPrint("$response \n");
   }
   elsif ($script_mode eq "list_backups")
   {
      $config_backup_path = GetValueForConfigParameter("config_backup_path");
      my $ts_options = GetBackupList();
      ExtendedPrint("$ts_options\n");
   }
   elsif ($script_mode eq "execute_backup")
   {
       if ($interface_exists)
       {
            $status = PerformBackupOperationWithInterface();
            if ($status == 0)
            {
                 GenerateSQLCommand();
             }
       }
       else
       {
          $install_type = GetValueForConfigParameter("install_type");
          $status = 0;
          if ($br_option =~ m/cold/) {
             $status = StopIASInstance();
          }

          if ($status == 0 ) {
              $status = PerformBackupOperation();
          }
          # If it is a group backup, we need to construct a SQL and return it to
          # the caller.
          if ($status == 0)
          {
              GenerateSQLCommand();
          }
          if ($br_option =~ m/cold/) {
             my $temp_status = StartIASInstance();
             if ($status == 0) {
                 $status = $temp_status;
             }
          }
       }
   }
   elsif ($script_mode eq "execute_recovery")
   {
       if ($interface_exists)
       {
           $status = 0;
           if ($cluster && $version =~ m/10.1.2.0.2/ ) 
           {
               $status = StopOC4JAcrossCluster();
           }

           if ($status == 0) {
               $status = PerformRecoveryOperationWithInterface();
           }
           if ($cluster && $version =~ m/10.1.2.0.2/ ) 
           {
               my $temp_status = StartOC4JAcrossCluster();
               $status = $status ? $status : $temp_status;
           }
           
       }
       else
       {

          $install_type = GetValueForConfigParameter("install_type");
          $status = 0;
          $status = StopIASInstance();

          if ($status == 0 )
          {
              $status = PerformRecoveryOperation();
          }
          my $temp_status = StartIASInstance();
          if ($status == 0)
          {
              $status = $temp_status;
          }
      }
   }
   elsif ($script_mode eq "get_config_params")
   {
       if ($interface_exists)
       {       
          GetConfigParametersWithInterface();
       } 
       else 
       {
             GetConfigParameters();
       }
   }
   elsif ($script_mode eq "set_config_params")
   {
       if ($interface_exists)
       {       
          $status = SetConfigParametersWithInterface();
       } 
       else 
       {
          $status = SetConfigParameters();
       }
   }
   elsif ($script_mode eq "check_brtool")
   {
       $status = CheckBRTool();
   }
   elsif ($script_mode eq "get_oracle_sid")
   {
      $oracle_sid = GetOracleSID();
      ExtendedPrint("$oracle_sid\n");
   }
   elsif ($script_mode eq "validate_backup")
   {
      ValidateBackup();
   }
   else
   {
      Fatal("Unknown Operation Attempted !!!\n");
   }
   exit($status);
}

#------------------------------------------------------------------------

sub GetConfigParametersWithInterface() {
    my $cmd  = "$br_install_dir/$br_new_shell ";
    $cmd .= "-m get_config_params $opts";
    VerbosePrint ("Running command:  \n$cmd\n")   unless $debug;
    my $rc = system("$cmd");
    $rc = ($? >> 8);
    return $rc;
}


#------------------------------------------------------------------------

sub GetConfigParameters() {
  my $response = "";
  my $commaflag = 0;
  foreach my $param (@arr) 
  {
     my $param_value = GetValueForConfigParameter($param);
     if($commaflag) 
     {
        $response .= ",";
     } 
     else 
     {
        $commaflag++;
     }
     $response .= $param; 
     $response .= "=";
     $response .= $param_value;

  }

  ExtendedPrint("$response \n");
}

#------------------------------------------------------------------------
sub CreateDirectory
{
   my $directory = (shift or "");
   return if -d $directory;

   ExtendedPrint("   Creating dir $directory\n");
   eval { mkpath ($directory) };
   if ($@)
   {
       Fatal("Couldn't create $directory: $@\n");
   }
}

#------------------------------------------------------------------------

sub ValidateBackup
{
   my $found_backup = "false";

   $config_backup_path = GetValueForConfigParameter("config_backup_path");

   if ($version =~ m/9.0.4/)
   {
       $backup_name =~ s/(config_bkp_)(.*)/$2/g;
       $backup_name =~ s/(.*)(_incr)/$1/g;
       my $backup_directory_1 = "config_bkp_$backup_name";
       my $backup_directory_2 = "config_bkp_$backup_name" . "_incr";

       if ((-d "$config_backup_path/$backup_directory_1") ||
           (-d "$config_backup_path/$backup_directory_2")) {
           $found_backup = "true";
       }
   }
   elsif ( $version =~ m/10.1.3/ && -e "$config_backup_path/$backup_name") {
   
                $found_backup = "true";
   }
   else 
   {
       my $dcm_name = $backup_name;
       $dcm_name =~ s/(config_bkp_)(.*)(\.jar)/$2/g;
       $dcm_name =~ s/(.*)(_incr)/$1/g;
       $dcm_name = "dcm_archive_$dcm_name" . ".jar";

       print "\ndcm_name : $dcm_name";
       if ((-e "$config_backup_path/$backup_name") &&
           (-e "$config_backup_path/$dcm_name"))
       {
           $found_backup = "true";
       }
   }
   if ($found_backup eq "false") {
       Fatal ("Backup Not Found");
   }

}

#------------------------------------------------------------------------

sub SetConfigParametersWithInterface()
{
    my $cmd  = "$br_install_dir/$br_new_shell ";
    $cmd .= "-m set_config_params $opts -f -p $param_str";
    VerbosePrint ("Running command:  \n$cmd\n")   unless $debug;
    my $rc = system("$cmd");
    $rc = ($? >> 8);
    return $rc;
}

#------------------------------------------------------------------------

sub SetConfigParameters() {

    my $rc;

    $rc = CheckBRTool();
    if ($rc != 0) 
    {
        return $rc;
    }
    
    my $cmd;
    my $old = $env_file;
    my $time = GetCurrentDateTime();
    my $new = "$env_file.$time";

    open(OLD, "< $old")         or die "can't open $old: $!";
    open(NEW, "> $new")         or die "can't open $new: $!";
    while (<OLD>) 
    {

        foreach my $myparam(@arr) 
        {
            (my $key, my $value) = split("=", $myparam);
            s/^[ ]*$key=[\w\/\\\:]*/$key=$value/;
        }
        print NEW $_            or die "can't write $new: $!";
    }
    close(OLD)                  or die "can't close $old: $!";
    close(NEW)                  or die "can't close $new: $!";

    rename($old, "$old.orig")   or die "can't rename $old to $new: $!";
    rename("$new", $old)   or die "can't rename $new to $old: $!";


    $install_type = GetValueForConfigParameter("install_type"); 
    $log_path = GetValueForConfigParameter("log_path"); 

    # Create directories for 9.0.4 here. For 10.1.2, BR tool creates them
    # during configure.
    if ($version eq "9.0.4")
    {
       $config_backup_path = GetValueForConfigParameter("config_backup_path"); 
       if (!-d $config_backup_path)
       {
          CreateDirectory ($config_backup_path);
       }

       if (!-d $log_path)
       {
          CreateDirectory ($log_path);
       }
       
    }

    if ($install_type eq "Infrastructure")
    {

       $database_backup_path = GetValueForConfigParameter("database_backup_path");
       if (!-d $database_backup_path)
       {
           CreateDirectory ($database_backup_path);
       }
        
       my $dbid = GetDBID();
       # Now, update config.inp file with this dbid.
       UpdateConfigParameter("dbid", $dbid);
    }

    # Execute -m config on backup and recovery tool.
    $cmd = "cd $br_install_dir";
    $cmd .= "$command_separator";
    $cmd .= "$perlexec bkp_restore.pl -m configure -f";
    VerbosePrint ("Running command:  \n$cmd\n")   unless $debug;
    $rc = system("$cmd");
    $rc = ($? >> 8);
    return $rc;

}

#------------------------------------------------------------------------

sub GenerateSQLCommand
{
   # Generate the sql command for successful backup (both group and single node)
   # This is executed on OMS by Job system.
	my $sql = "\$\$\$--*\$\$\n";
        $sql .= "<commandBlock>";
        $sql .= "<executeProc name=\"mgmt_target_as_backuprecovery.insert_backup_name\">";
        $sql .= "<scalar>%job_step_id%</scalar>";
        $sql .= "<scalar>%job_execution_id%</scalar>";
        $sql .= "<scalar>$backup_name</scalar>";
        $sql .= "</executeProc>";
        $sql .= "</commandBlock>";
        $sql .= "\$\$\$*--\$\$";

        ExtendedPrint($sql);
}

#------------------------------------------------------------------------

sub GetDatabaseBackupMode()
{
   my $repos_backup_mode;

   if (($br_option eq "backup_instance_online") ||
       ($br_option eq "backup_group_online"))
   {
      $repos_backup_mode = "backup_online";
   }
   elsif (($br_option eq "backup_instance_cold") ||
          ($br_option eq "backup_group_cold"))
   {
      $repos_backup_mode = "backup_cold";
   }
   elsif (($br_option eq "backup_instance_online_incr") ||
          ($br_option eq "backup_group_online_incr"))
   {
      $repos_backup_mode = "backup_online_incr -l $backup_level";
   }
   elsif (($br_option eq "backup_instance_cold_incr") ||
          ($br_option eq "backup_group_cold_incr"))
   {
      $repos_backup_mode = "backup_cold_incr -l $backup_level";
   }
   else
   {
      # Setting the default backup mode.
      $repos_backup_mode = "backup_online";
   }
   return $repos_backup_mode;
}

#------------------------------------------------------------------------
sub GetInterfaceBackupMode()
{
   my $config_backup_mode;

   if (($br_option eq "backup_instance_online") ||
       ($br_option eq "backup_group_online"))
   {
      $config_backup_mode = "backup_instance_online";
   }
   elsif (($br_option eq "backup_instance_cold") ||
          ($br_option eq "backup_group_cold"))
   {
      $config_backup_mode = "backup_instance_cold";
   }
   elsif (($br_option eq "backup_instance_online_incr") ||
          ($br_option eq "backup_group_online_incr"))
   {
      $config_backup_mode = "backup_instance_online_incr";
   }
   elsif (($br_option eq "backup_instance_cold_incr") ||
          ($br_option eq "backup_group_cold_incr"))
   {
      $config_backup_mode = "backup_instance_cold_incr";
   }
   else
   {
      # Setting the default backup mode.
      $config_backup_mode = "backup_instance_online";
   }
   return $config_backup_mode;
}


#------------------------------------------------------------------------
sub GetConfigBackupMode()
{
   my $config_backup_mode;

   if (($br_option eq "backup_instance_online") ||
       ($br_option eq "backup_group_online"))
   {
      $config_backup_mode = "backup_config";
   }
   elsif (($br_option eq "backup_instance_cold") ||
          ($br_option eq "backup_group_cold"))
   {
      $config_backup_mode = "backup_config";
   }
   elsif (($br_option eq "backup_instance_online_incr") ||
          ($br_option eq "backup_group_online_incr"))
   {
      $config_backup_mode = "backup_config_incr";
   }
   elsif (($br_option eq "backup_instance_cold_incr") ||
          ($br_option eq "backup_group_cold_incr"))
   {
      $config_backup_mode = "backup_config_incr";
   }
   else
   {
      # Setting the default backup mode.
      $config_backup_mode = "backup_config";
   }
   return $config_backup_mode;
}
#------------------------------------------------------------------------
sub PerformBackupOperationWithInterface
{
    my $backup_mode = GetInterfaceBackupMode();
    # Run the command in verbose mode to get the backup name
    my $cmd  = "$br_install_dir/$br_new_shell -v -m" ;
    $cmd .= " $backup_mode$opts -f ";
    my $rc  = ExecSystemCall($cmd);
    ExtendedPrint("Return Code: $rc\n");
    return $rc;
}

#------------------------------------------------------------------------
sub GetFBRArchiveName
{
   my $fbr_arch_name;
   # ExecSystemCall creates this output file.
   my $fbr_arch_log = "$log_path/${cur_date_time}_output.log"; 

   open (LOGFILE, "<$fbr_arch_log") or
     Fatal "Problem opening file $fbr_arch_log for reading: $!\n";
   my $line;
   while (<LOGFILE>)
   {
      next if /^\s*$/;                # Skip blank lines
      $line = $_;
      if ($line =~ /exportrepository/)
      {
         # Line example: ORACLE_HOME/bin/dcmctl exportrepository -file 
         # CONFIG_BACKUP_PATH/fbr_archive_2005-07-11_11-24-50.jar ....

         my $idx = index ($line, "fbr_archive_");
         if ($idx != -1) 
         {
            # FBR archive name example:  fbr_archive_2005-07-11_12-17-05.jar
            $fbr_arch_name = substr ($line, $idx, 35);
         }
      }
   }
   return $fbr_arch_name;
}

#------------------------------------------------------------------------
sub PerformBackupOperation
{
   my $cmd;
   my $rc = -1;
   if ($install_type eq "Infrastructure")
   {
      # This backup should include config and db backup
      my $repos_backup_mode = GetDatabaseBackupMode();

      my $config_backup_mode = GetConfigBackupMode();

      $cmd = "cd $br_install_dir";
      $cmd .= "$command_separator";
      $cmd .= "$perlexec bkp_restore.pl -m $repos_backup_mode$opts -f ";
      if ($no_prompt)
      {
         $cmd .= " -n";
      }
      $rc = ExecSystemCall($cmd);

      if ( ($rc == 0 ) && ($br_option =~ m/cold/)) 
      {
         $rc = StartIASComponent("OID");
         $rc = StartIASComponent("dcm-daemon");
      }

      # If the repository backup was completed successfully, 
      # proceed with the config backup.
      if ($rc == 0)
      {
         $cmd = "cd $br_install_dir";
         $cmd .= "$command_separator";
         $cmd .= "$perlexec bkp_restore.pl -m $config_backup_mode$opts -f ";
         if ($no_prompt)
         {
            $cmd .= " -n";
         }
         $rc = ExecSystemCall($cmd);
      }
   }
   elsif ($install_type eq "Infrastructure_ID")
   {

      if ( ($rc == 0 ) && ($br_option =~ m/cold/))
      {
         $rc = StartIASComponent("OID");
         $rc = StartIASComponent("dcm-daemon");
      }

      # This backup should include config backup
      my $config_backup_mode = GetConfigBackupMode();

      $cmd = "cd $br_install_dir";
      $cmd .= "$command_separator";
      $cmd .= "$perlexec bkp_restore.pl -m $config_backup_mode$opts -f ";
      if ($no_prompt)
      {
         $cmd .= " -n";
      }
      VerbosePrint("Backup command: $cmd \n");
      $rc = ExecSystemCall($cmd);
   }
   elsif ($install_type eq "Midtier")
   {
      # Backup the FBR, if it exists
      my $isfbr = CheckFileBasedRepository();
      if ($isfbr eq "yes")
      {
         $cmd = "cd $br_install_dir";
         $cmd .= "$command_separator";
         # Executing this command in verbose mode because we need to
         # get the fbr archive name from log file.
         $cmd .= "$perlexec bkp_restore.pl -m backup_online -v $opts -f ";
         if ($no_prompt)
         {
            $cmd .= " -n";
         }
         $rc = ExecSystemCall($cmd);
           # This backup should include config backup
         if ($rc == 0)
         {
             my $old_fbr_arch_name = GetFBRArchiveName();
             my $config_backup_mode = GetConfigBackupMode();
 
             $cmd = "cd $br_install_dir";
             $cmd .= "$command_separator";
             $cmd .= "$perlexec bkp_restore.pl -m $config_backup_mode$opts -f ";
             if ($no_prompt)
             {
                 $cmd .= " -n";
             }
             VerbosePrint("Backup command: $cmd \n");
            $rc = ExecSystemCall($cmd);
            my $cfg_bkp_timestamp;
            # backup_name is like config_bkp_2005-07-11_12-23-06.jar
            # We need to parse it to get the timestamp value.
            if (index ($backup_name, ".jar") != -1)
            {
               $cfg_bkp_timestamp = substr ($backup_name, 0, -4);
               if (index ($cfg_bkp_timestamp, "config_bkp_") == 0)
               {
                  $cfg_bkp_timestamp = substr ($cfg_bkp_timestamp, 11);
               }
               # We need to group both the FBR and config backups.
               # So, the FBR archive name will be renamed to match the config
               # backup timestamp.
               my $new_fbr_arch_name = "fbr_archive_" ;
               $new_fbr_arch_name .= $cfg_bkp_timestamp;
               $new_fbr_arch_name .= ".jar";
               $config_backup_path = GetValueForConfigParameter("config_backup_path");
               VerbosePrint("Old : $old_fbr_arch_name : New : $new_fbr_arch_name\n");
		move("$config_backup_path/$old_fbr_arch_name",
                    "$config_backup_path/$new_fbr_arch_name");
            }
         }
      
      }
      else 
      {
          # This backup should include config backup
          my $config_backup_mode = GetConfigBackupMode();

          $cmd = "cd $br_install_dir";
          $cmd .= "$command_separator";
          $cmd .= "$perlexec bkp_restore.pl -m $config_backup_mode -f ";
          if ($no_prompt)
          {
              $cmd .= " -n";
          }
          VerbosePrint("Backup command: $cmd \n");
          $rc = ExecSystemCall($cmd);
      }
   }
   else
   {
      Fatal("Unknown Install Type !!! \n");
   }

   return $rc;
}

sub StartIASComponent 
{
    my $ias_component = shift;

    my $cmd = "$perlexec $ENV{EMDROOT}/sysman/admin/scripts/em_optic.pl $oracle_home en_US start ias-instance=$ias_instance\#ias-component=$ias_component";

    return ExecSystemCall($cmd);
}


sub StartIASInstance() {
    
    my $cmd = "$perlexec $ENV{EMDROOT}/sysman/admin/scripts/em_optic.pl $oracle_home en_US start ias-instance=$ias_instance";

    return ExecSystemCall($cmd);
}

sub StopIASInstance() {

    my $cmd = "$perlexec $ENV{EMDROOT}/sysman/admin/scripts/em_optic.pl $oracle_home en_US stop ias-instance=$ias_instance";

    return ExecSystemCall($cmd);
}


sub StopOC4JAcrossCluster() {

    my $cmd = "${oracle_home}/opmn/bin/opmnctl \@cluster stopproc ias-component=OC4J";
    
    my $rc = ExecSystemCall($cmd);
    ($rc == 150) ? return 0: return $rc;
}

sub StartOC4JAcrossCluster() {

    my $cmd = "${oracle_home}/opmn/bin/opmnctl \@cluster startproc ias-component=OC4J";

    my $rc = ExecSystemCall($cmd);
    ($rc == 150) ? return 0: return $rc;
}


#------------------------------------------------------------------------
sub ConvertDateTimeStampForDBRecovery
{
   my $orig_timestamp_param = shift;
   VerbosePrint("Original Timestamp: $orig_timestamp_param \n");

   # Remove "config_bkp_" and "_incr" from the backup string
   $orig_timestamp_param =~ s/(config_bkp_)(.*)/$2/g;
   $orig_timestamp_param =~ s/(.*)(_incr)/$1/g;

   # First split the timestamp to date and time [2004-07-21 and 12-03-00]
   my  @flds;
   @flds = split(/_/, $orig_timestamp_param); 
   
   my $date_str;
   my $time_str;
   if (defined($flds[0]) && defined($flds[1]))
   {
      $date_str = $flds[0];
      $time_str = $flds[1];
   }
   else
   {
      Fatal("Invalid Timestamp !!! \n");
   }

   # Now, get the year,month, and day components from $date_str
   my  @flds1;
   my $year_str;
   my $month_str;
   my $day_str;
   @flds1 = split(/-/, $date_str);
   if (defined($flds1[0]) && defined($flds1[1]) && defined($flds1[2]))
   {
      $year_str  = $flds1[0];
      $month_str = $flds1[1];
      $day_str   = $flds1[2];
   }
   else
   {
      Fatal("Invalid Timestamp !!! \n");
   }

   # Now, get the hour, minute, and second components from $time_str
   my  @flds2;
   my $hour_str;
   my $min_str;
   my $sec_str; 
   @flds2 =  split(/-/, $time_str);
   if (defined($flds2[0]) && defined($flds2[1])) 
   {
      $hour_str   = $flds2[0];
      $min_str    = $flds2[1];
      if (defined($flds2[2]))
      {
         $sec_str = $flds2[2];
      }
      else
      {
         $sec_str = "00"; 
      }
   }
   else
   {
      Fatal("Invalid Timestamp !!! \n");
   }
   my $modified_timestamp;
   $modified_timestamp = $month_str;
   $modified_timestamp .= "/";
   $modified_timestamp .= $day_str;
   $modified_timestamp .= "/";
   $modified_timestamp .= $year_str;
   $modified_timestamp .= "_";
   $modified_timestamp .= $hour_str;
   $modified_timestamp .= ":";
   $modified_timestamp .= $min_str;
   $modified_timestamp .= ":";
   $modified_timestamp .= $sec_str;

   VerbosePrint("Modified Timestamp: $modified_timestamp \n");
   return $modified_timestamp; 
}

#------------------------------------------------------------------------
sub PerformRecoveryOperationWithInterface
{

    $recovery_name =~ s/(config_bkp_)(.*)(.jar)/$2/g;
    $recovery_name =~ s/(.*)(_incr)/$1/g;

    my $cmd  = "$br_install_dir/$br_new_shell ";
    $cmd .= "-m restore_instance$opts -t $recovery_name -n";
    my $rc   = ExecSystemCall($cmd);
    return $rc;
}

#------------------------------------------------------------------------
sub PerformRecoveryOperation
{
   my $rc = 0;
   my $cmd;

   if ($install_type eq "Infrastructure")
   {

      if ( ($rc = StartIASComponent("OID")) != 0) 
      {
	return $rc;
      }
      if ( ($rc = StartIASComponent("dcm-daemon")) != 0) 
      {
	return $rc;
      }

      # This recovery should include config and db restore
      $cmd = "cd $br_install_dir";
      $cmd .= "$command_separator";
      $cmd .= "$perlexec bkp_restore.pl -m restore_config$opts -n ";
      if ($recovery_name eq "")
      {
         Fatal("No backup name has been specified for restore !!!\n");
      }
      else
      {
          if ($version =~ m/^10.1.2/)
          {
             $recovery_name =~ s/(config_bkp_)(.*)(.jar)/$2/g;
             $recovery_name =~ s/(.*)(_incr)/$1/g;
          }
          
         $cmd .= "-t $recovery_name"; 
      }
      if ($no_prompt)
      {
         $cmd .= " -n";
      }
      VerbosePrint("Restore command: $cmd \n");
      $rc = ExecSystemCall($cmd);

      # If the restore_config was successful, try to restore the Repository.
      if ($rc == 0)
      {
         $cmd = "cd $br_install_dir";
         $cmd .= "$command_separator";
         if ($recovery_name eq "")
         {
            $cmd .= "$perlexec bkp_restore.pl -m $restore_command$opts -n ";
         }
         else
         {
            # We need to convert this timestamp from 2004-07-21_12-03 to 
            # 07/21/2004_12:03:00 format for database recovery
            my $new_time = ConvertDateTimeStampForDBRecovery($recovery_name);
            $cmd .= "$perlexec bkp_restore.pl -m $restore_command$opts -n -u $new_time ";
         }
         # Check if the control file needs to be restored
         if ($restore_controlfile)
         {
            $cmd .= "-c";
         }
         if ($no_prompt)
         {
            $cmd .= " -n";
         }
         # Now, we can execute this restore command.
         VerbosePrint("Restore command: $cmd \n");
         $rc = ExecSystemCall($cmd);
      }
   }
   elsif ($install_type eq "Infrastructure_ID")
   {

      if ( ($rc = StartIASComponent("OID")) != 0)
      {
        # return $rc;
      }
      if ( ($rc = StartIASComponent("dcm-daemon")) != 0)
      {
        return $rc;
      }

      # Config Restore
      $cmd = "cd $br_install_dir";
      $cmd .= "$command_separator";
      $cmd .= "$perlexec bkp_restore.pl -m restore_config -n ";
      if ($recovery_name eq "")
      {
         Fatal("No time has been specified for restore !!!\n");
      }
      else
      {
          if ($version =~ m/^10.1.2/)
          {
             $recovery_name =~ s/(config_bkp_)(.*)(.jar)/$2/g;
             $recovery_name =~ s/(.*)(_incr)/$1/g;
          }
          
         $cmd .= "-t $recovery_name"; 
      }
      if ($no_prompt)
      {
         $cmd .= " -n";
      }
      $cmd .= $opts; 
      VerbosePrint("Restore command: $cmd \n");
      $rc = ExecSystemCall($cmd);
   }
   elsif ($install_type eq "Midtier")
   {
      # Restore file-based repository, if it exists
      my $dcm_repos = GetValueForConfigParameter("dcm_repository_type");
      if ($dcm_repos eq "file")
      {
          $cmd = "cd $br_install_dir";
          $cmd .= "$command_separator";
          $cmd .= "$perlexec bkp_restore.pl -m $restore_command$opts -n ";
          if ($recovery_name eq "")
          {
             Fatal("No backup name has been specified for restore !!! \n");
          }
          else
          {
              # Not Required. But just in case
               if ($version =~ m/^10.1.2/)
               {
                  $recovery_name =~ s/(config_bkp_)(.*)(.jar)/$2/g;
                  $recovery_name =~ s/(.*)(_incr)/$1/g;
               }

              $cmd .= "-t $recovery_name";
          }
         if ($no_prompt)
         {
            $cmd .= " -n";
         }
         VerbosePrint("Restore command: $cmd \n");
         $rc = ExecSystemCall($cmd);
      }
      # Config Restore
      if ($rc == 0)
      {
         $cmd = "cd $br_install_dir";
         $cmd .= "$command_separator";
         $cmd .= "$perlexec bkp_restore.pl -m restore_config -n ";
         if ($recovery_name eq "")
         {
            Fatal("No time has been specified for restore !!!\n");
         }
         else
         {
             if ($version =~ m/^10.1.2/)
             {
                $recovery_name =~ s/(config_bkp_)(.*)(.jar)/$2/g;
                $recovery_name =~ s/(.*)(_incr)/$1/g;
             }
   
             $cmd .= "-t $recovery_name"; 
         }
         if ($no_prompt)
         {
            $cmd .= " -n";
         }
         $cmd .= $opts;
         VerbosePrint("Restore command: $cmd \n");
         $rc = ExecSystemCall($cmd);
      }
   }
   else
   {
      Fatal("Unknown Install Type !!!\n");
   }
    
   return $rc;
}


#------------------------------------------------------------------------
sub VerbosePrint
{
    if ($verbose) 
    {
       my @print_text = (@_, "");
       print @print_text unless $silent;
       map { $output .= $_; } @print_text;
    }
}

#------------------------------------------------------------------------
sub ExtendedPrint
{
   my @print_text = (@_, "");
   print @print_text unless $silent;
   map { $output .= $_; } @print_text;
}

#------------------------------------------------------------------------
sub CheckFileBasedRepository
{
    if ($version =~ m/^9.0.4/ || $version =~ m/^10.1.3/ )
    {
        return "no";
    }

    $log_path = GetValueForConfigParameter("log_path");
    my $tmplogfile = "$log_path/${cur_date_time}_whichfarm.log";

    my $cmd = "${oracle_home}/dcm/bin/dcmctl whichfarm -v > $tmplogfile";
   
    VerbosePrint("Command : $cmd");
    system(" $cmd 2>&1");
    my $rc = ($? >> 8);
   
    if ($rc != 0)
    {
        Fatal("Unable to check file based repository with dcmctl !!! \n");
    }

    my $repos_str = "";
    my  @flds;
    my $isfbr = "no";
    open (LOGFILE, "<$tmplogfile") or
         Fatal "Problem opening file $tmplogfile for reading: $!\n";
    while (<LOGFILE>)
    {
      next if /^\s*#/; # Skip comments
      next if /^\s*$/; # Skip blank lines

      chomp ($_);
      @flds = split /:/ ;
      if (defined($flds[0]) && defined($flds[1]))
      {
         if ($flds[0] eq "Repository Type")
         {
            ($repos_str = $flds[1]) =~ s/\/$//;

            #trim leading and trailing whitespaces
            $repos_str =~ s/^\s*(.*?)\s*$/$1/;

            if ($repos_str eq "Distributed File Based (host)")
            {
               $isfbr = "yes";
            }
         }
      }
    }
    # Now, remove the temporary log file
    unlink $tmplogfile;

    return $isfbr;
}

#------------------------------------------------------------------------
sub ExecSystemCall
{
   my $cmd = shift;
   return 0 unless $cmd;         # Abort unless there's a command string

   VerbosePrint ("Running command:  \n$cmd\n")   unless $debug;
   VerbosePrint ("Would run command:  \n$cmd\n") if     $debug;
   
   my $rc = 0;

   unless ($debug)
   {
      my $match = "execute_";
      my $tmplogfile = "";

      $log_path = GetValueForConfigParameter("log_path");
      $tmplogfile = "$log_path/${cur_date_time}_output.log";
      $cmd .= " > ";
      $cmd .= $tmplogfile;
      
      VerbosePrint("Command : $cmd");
      if ($interface_exists)
      {
         $rc = system($cmd);
         $rc = $rc >> 8;
      }
      else
      {
         $cmd .= " > ";
         $cmd .= $tmplogfile;
         $rc = system(" $cmd 2>&1");
         $rc = $rc >> 8;
      }

      VerbosePrint("Return : $rc\n" );
     
      if ($cmd =~ m/em_optic.pl/) {
        $rc = ParseStartStop($tmplogfile);
      } 
      else 
      {
          # Print the content of the log file
          open (LOGFILE, "<$tmplogfile") or
                Fatal "Problem opening file $tmplogfile for reading: $!\n";
          my @log_lines = <LOGFILE>;
          my $element;
          foreach $element (@log_lines)
          {
             ExtendedPrint($element);
          }
          if ($rc == 0 && $script_mode =~ /$match/ && !$interface_exists)
          {
              $rc = ParseBackupAndRecovery($tmplogfile, $cmd);

          }
          elsif ($rc == 0 && $script_mode =~ /execute_backup/
                            && $interface_exists)
          {
              # Identify the backup name alone. 
              $rc = IdentifyBackupName(@log_lines);
          }
      }
   }
   return $rc;
}

#------------------------------------------------------------------------
sub GetCurrentDateTime
{
   my ($day,$month,$year) = (localtime)[3,4,5];
   my ($min,$hour) = (localtime)[1,2];
   return sprintf("%04d-%02d-%02d_%02d-%02d", $year+1900,$month+1,$day,
                  $hour, $min);
}

#------------------------------------------------------------------------
sub GetBackupList()
{
   # This method parses the contents in config_backup_path.
   my  $bkp_dir;
   my  @all_files;
   my  @sorted_files;
   my  $f;
   my  $ts_options = "";
   my  $cnt = 1;

   opendir bkp_dir, $config_backup_path or Fatal "Could not open $bkp_dir: $!";
   @all_files = grep /^config_bkp.*$/, readdir bkp_dir;
   @sorted_files = sort @all_files;
   closedir bkp_dir;
  
   foreach  $f (@sorted_files)
   {
       #$f =~ s/(config_bkp_)(.*)/$2/g;
       #$f =~ s/(.*)(_incr)/$1/g;
       #
      # For 10.1.2, if both the DCM archive and config file backup
      # exist for this timestamp. Then only, we should add it in the
      # ts_options, otherwise not.
      if ($version =~ m/^9.0.4/)
      {

         if($cnt gt 1) { $ts_options .= ","; }
         $ts_options .= "$f";
         $cnt++;
      }
      else
      {
         if (-e "$config_backup_path/dcm_archive_${f}.jar")
         {
            if($cnt gt 1) { $ts_options .= ","; }
            $ts_options .= "$f";
            $cnt++;
         }
      }
   }
   return $ts_options;
}

#------------------------------------------------------------------------

sub trim {
    my @out = @_;
    for(@out) 
    {
        s/^\s+//;
        s/\s+$//;
    }
    return wantarray ? @out : $out[0];
}

#------------------------------------------------------------------------

sub ParseStartStop
{
    my $tmplogfile = shift;
    # open log file
    open (LOGFILE, "<$tmplogfile") or
         Fatal "Problem opening file $tmplogfile for reading: $!\n";

    my @lines = <LOGFILE>;
    my $line = "@lines"; 
    
    $line  =~ /<ResponseMessage>(.+)<\/ResponseMessage>/;
    ExtendedPrint("$1\n");
    
    $line  =~ /<ExitCode>(\d+)<\/ExitCode>/;
    return ($1);

}

#------------------------------------------------------------------------

sub GetExitCode
{
    my ($success_msg, $error_msg, @lines) = @_;
    my $return_code = -1;

    if ($success_msg eq "") 
    {
        if ($error_msg eq "") 
        {
            $return_code = 0;
        } else
        {
            $return_code  = grep /$error_msg/i, @lines;
        }
     } 
     else 
     {
        my $num_success = grep /$success_msg/i, @lines;
        if ($num_success > 0) 
        {
            if ($error_msg eq "") 
            {
                 $return_code = 0;
            } 
            else 
            {
                $return_code  = grep /$error_msg/i, @lines;
            }
         }
         else
         {
            $return_code = -1;
         }
     }
     return $return_code;
}

#------------------------------------------------------------------------

sub ParseBackupAndRecovery
{
    my $tmplogfile = shift;
    my $cmd = shift;
    my $success_msg = "";
    my $error_msg   = "";
    my @strgs;

    # open log file
    open (LOGFILE, "<$tmplogfile") or
         Fatal "Problem opening file $tmplogfile for reading: $!\n";

    # Read log file content to an array
    my @lines = <LOGFILE>;

   if ($version =~ m/^9.0.4/)
   {
        if (($script_mode =~ /execute_backup/)  && ($cmd =~ /backup_config/))
        {
            $success_msg = "Backup directory is ";
            $error_msg   = "Error|Failure";
        }

        if (($script_mode =~ /execute_recovery/) && ($cmd =~ /restore_config/))
        {
            $success_msg = "Restoring configuration files from";
            $error_msg   = "Error|Failure";
        }

    }
    else
    {
        if (($script_mode =~ /execute_backup/)  && ($cmd =~ /backup_config/))
        {
            $success_msg = "Configuration backup archive is ";
            $error_msg   = "Error|Failure";
        }

        if (($script_mode =~ /execute_backup/) &&
                (($cmd =~ /backup_online/) || ($cmd =~ /backup_cold/)))
        {
            $error_msg = "Unable to export file based repository using dcmctl";
        }

        if (($script_mode =~ /execute_recovery/) && ($cmd =~ /restore_repos/))
        {
            $error_msg = "Unable to import file based repository using dcmctl";
        }

    }
    my $rc = GetExitCode($success_msg, $error_msg, @lines);

    if (($rc == 0) && ($script_mode =~ /execute_backup/) &&
                                ($cmd =~/backup_config/))
    {
        IdentifyBackupName(@lines);
    }
    return $rc;
}

#------------------------------------------------------------------------

sub IdentifyBackupName 
{

    my @lines = @_;
    my @strgs;

    #we need to parse $tmplogfile to get the backup name (config_bkp...)
    chomp(@strgs= grep /$backup_success_msg/i, @lines);
    my $backup_str="@strgs";
    if ($backup_str ne "")
    {
        $backup_str = substr($backup_str, length($backup_success_msg));
        #trim leading and trailing whitespaces
        $backup_str =~ s/^\s*(.*?)\s*$/$1/;
            # backup_str contains the complete backup path ->
            # /private/sunishar/orahome1/infra/backups/
            # config_files/config_bkp_2004-08-12_17-53-39
            # from "/a/b/c/d" extract "d"
        ($backup_name = $backup_str) =~ s#.*/([^/]+)$#$1#;
            # backup_name is like config_bkp_2004-08-12_17-53-39
    }
    if ($backup_name eq "")
    {
        return 200;
    }
    
}
