#!/usr/bin/perl
#
# Copyright (c) 2003, Oracle Corporation.  All rights reserved.  
#
#    NAME
#      fileMonitoring.pl 
#
#    DESCRIPTION
#      Metric script that checks whether certain  Files exist or not and prints out
#      different stastics about the file
#       
#
#    OUTPUT:
#         if file is found
#         em_result=<filename>|<found>|<permsission of files>|< file size change rate> |
#           <file size> | <last modified Date>
#         if file is not found
#         em_result=<filename>|<not found>
#
#    NOTES
#      Supported platforms: Solaris, HP-UX, Windows and Linux. 
#      
#      Usage: fileMonitoring.pl <input> 
#         where: 
#            <input>   :  files to be monitored. The value is a string 
#                           in the form:
#                           <filename>|||<filetype>[|||..]
#                           filetype is 0 if filename is a file
#                           filetype is 1 if filename is a directory
#                           filetype is 2 if fileanme is a link
#                           
#                         
#
#    MODIFIED    (MM/DD/YY)
#       kchander  09/13/03 - kchander_host_mgmt_enh 
#    kchander    (08/01/03)
use strict;
require "emd_common.pl";

my $fileName=$ARGV[0];

# check the number of parameters 
if ( $#ARGV < 0 ) {
   print "Insufficient parameters. Note: This metric cannot be collected through the Metric Browser.\n";
   EMD_PERL_DEBUG("ERROR: Insufficient parameters.\nUsage: <Input Parameters>");  
   EMD_PERL_DEBUG("ERROR: Usage fileMonitoring.pl  '/etc/passwd|||0'");
   exit -1;
}

my @tokens = split(/\|\|\|/, $fileName);
my @fileNames = ();
my @fileTypes = ();
my @results = ();
my %savedState = ();


EMD_PERL_DEBUG("fileMonitoring.pl Arguments $fileName");

for (my $i=0, my $j = 0; $i<@tokens; $j++)
{
    $fileNames[$j] = $tokens[$i++];
    $fileTypes[$j] = $tokens[$i++];
}
#Load the previous state of the files
getPreviousState();

for (my $i=0; $i<@fileNames; $i++)
{
    my $result = "em_result=";
    if ( ($fileTypes[$i] == 0) and (-e  $fileNames[$i]) and !(-d $fileNames[$i]))
    {
        $result = $result.$fileNames[$i].'|found|'.getValues($i);
    }
    elsif (($fileTypes[$i] == 1) and (-d $fileNames[$i]))
    {
        $result = $result.$fileNames[$i].'|found|'.getValues($i);
    }
    elsif (($fileTypes[$i] == 2 )and ( -l $fileNames[$i]))
    {
        $result = $result.$fileNames[$i].'|found|'.getValues($i);
    }
    else
    {
        $result=$result.$fileNames[$i].'|not found||||';
    }

    $results[$i] = $result;
}

#print output
my $line;
foreach $line (@results)
{
    print "$line\n";
    EMD_PERL_DEBUG("fileMonitoring.pl $line\n");
}
#Save the State file
saveStateFile();


sub getValues
{
    my($index) = @_;
    my @attr = stat($fileNames[$index]);
    my $mode_s = mode_to_sum($attr[2]);
    my $result = "";
    my $osize = $attr[7];
    my $size = $osize/1024;
    my $date = scalar(localtime($attr[9]));
    my $prevSize = $savedState{$fileNames[$index]}{LastKnownSize};
    my $change=0;
    if ($prevSize)
    {
        $change = $osize - $prevSize;
    }
    $savedState{$fileNames[$index]}{LastKnownSize}=$osize;
    my $changeRate=0;
    if ($change != 0)
    {
        $changeRate = ($change/1024)*12;
    }
    $result=$mode_s.'|'.$changeRate.'|'.$size.'|'.$date;
    return $result;
}

sub getPreviousState
{
    my $stateFileName = getStateFileName();

    open(STATE, "< $stateFileName");
    my $myLogFile;
    while ($myLogFile = <STATE>)
    {
        chomp($myLogFile);
        if ($myLogFile)
        {
            $savedState{$myLogFile}{LastKnownSize} =<STATE>;
            chomp($savedState{$myLogFile}{LastKnownSize});
        }
    }
}


sub saveStateFile
{
    my $stateFileName = getStateFileName();

    open(STATE, "> $stateFileName");

    my $logFile;
    foreach $logFile (keys %savedState)
    {
        print STATE "$logFile\n";
        print STATE "$savedState{$logFile}{LastKnownSize}\n";
    }
    close(STATE);
}





sub mode_to_sum 
{
    my $mode = shift;
    my $myNum;

    my $or = $mode & 0400 ? '4' : '0'; 
    my $ow = $mode & 0200 ? '2' : '0';
    my $ox = $mode & 0100 ? '1' : '0';

    my $gr = $mode &  040 ? '4' : '0'; 
    my $gw = $mode &  020 ? '2' : '0'; 
    my $gx = $mode &  010 ? '1' : '0';

    my  $otr = $mode &    4 ? '4' : '0';
    my  $otw = $mode &    2 ? '2' : '0'; 
    my  $otx = $mode &    1 ? '1' : '0';

    my  $op = $or + $ow + $ox;
    my  $gp = $gr + $gw + $gx;
    my  $otp = $otr + $otw + $otx;


    return $op.$gp.$otp;
}

sub getStateFileName
{
    #State File where I store previous values of file thats need to monitored
    #The content in this file will be something like this, firstline contains
    #filename, second line contains Last known size of the file, third line contains

    my $agentStateDir = $ENV{AGENTSTATEDIR};
    EMD_PERL_DEBUG("fileMonitoring.pl AGENTSTATE DIR $agentStateDir\n");
    my $targetGuid = $ENV{EM_TARGET_GUID};
    EMD_PERL_DEBUG("fileMonitoring.pl EM_TARGET_GUID $targetGuid\n");

    my $separator = $^O =~ m/MSWin32/ ? "\\" : "\/";
    my $stateFile =  $agentStateDir.$separator."sysman".$separator."emd".$separator."state".$separator.$targetGuid."fileMon.log";
    return $stateFile;
}
