#!/usr/local/bin/perl
# 
# $Header: hostGenFunctions.pm 28-jul-2005.18:27:14 ajere Exp $
#
# hostGenFunctions.pm
# 
# Copyright (c) 2004, 2005, Oracle. All rights reserved.  
#
#    NAME
#      hostGenFunctions.pm - <one-line expansion of the name>
#
#    DESCRIPTION
#      <short description of component this file declares/defines>
#
#    NOTES
#      <other useful comments, qualifications, etc.>
#
#    MODIFIED   (MM/DD/YY)
#    ajere       07/28/05 - Fix bug# 4503778 
#    sacgoyal    01/25/05 - add getCpuCount() method 
#    sreddy      10/06/04 - 
#    sreddy      10/05/04 - 
#    sacgoyal    10/04/04 - Adding getDirSize function 
#    mkiran      09/18/04 - 3770903: Use -P option with df
#    rzkrishn    06/11/04 - rzkrishn_add_host_gen_func_pl
#    rzkrishn    06/11/04 - Creation
# 

use strict;

package hostGenFunctions;

require Exporter;

#*********************************
# Export Subroutines
#*********************************

@hostGenFunctions::ISA =  ('Exporter');
@hostGenFunctions::EXPORT = qw(
  &trim
  &getDiskInfo
  &getDirSize
  &getCpuCount
  );




#*********************************
#  Global Variables
#*********************************

$hostGenFunctions::DF        =  '/bin/df';
$hostGenFunctions::DATALOC   =  '';
$hostGenFunctions::TAIL      =  '/usr/bin/tail';
$hostGenFunctions::SORT      =  '/bin/sort';
$hostGenFunctions::CAT       =  '/bin/cat';
$hostGenFunctions::DU       =  '/usr/bin/du';
$hostGenFunctions::FILE       =  '/usr/bin/file';




#*********************************
# Exported Subroutines
#*********************************

sub trim
{

  #Arguments:  string or array
  #Outputs  :  string or array
  #Function : trim

  my (@out) = @_;
  for (@out)
  {
     s/^\s+//;
     s/\s+$//;
  }

  return wantarray ? @out : $out[0];
}

sub getDiskInfo
{
    # Purpose: To get disk space/filesystem space

    my $datadir = $hostGenFunctions::DATALOC;
    my $cmd     = '';
    my $value   = '';
    my @diskinfo = ();
    my $DFLocalFileSystemsParams = '-lkP -x tmpfs -x none';
    my $DFAllFileSystemsParams = '-kP -x tmpfs -x none';
    my ($DFtype) = @_;
    my $DFparams = $DFLocalFileSystemsParams;

    if ($datadir eq '')
    {
        my $rc = checkCmd($hostGenFunctions::DF) & 
                 checkCmd($hostGenFunctions::TAIL) & 
                 checkCmd($hostGenFunctions::SORT);
        if ($rc == 1)
        {
            $DFparams = $DFAllFileSystemsParams if $DFtype eq 'ALL';
            $cmd = "$hostGenFunctions::DF $DFparams | 
                    $hostGenFunctions::TAIL +2 | 
                    $hostGenFunctions::SORT -u";
        }
        else
        {
            return @diskinfo;
        }
    }
    else
    {
        $cmd = catCmd('dfFufsk.out');
    }
    $value = execCmd($cmd);
    @diskinfo = split ("\n", $value);
    chomp(@diskinfo);

    return @diskinfo;
}

#*****************************************************************
# getDirSize() routine 
# 
# It takes the name of the directory as input parameter.
# Returns list of (allocated size in bytes, exitStatus).
# exitStatus will be 0 upon success.
#*****************************************************************

sub getDirSize
{
  my ($dirname)   = @_ ;
  my $cmd         = '';
  my $exitStatus  = 1;
  my $size        = -1;

  my $rc = checkCmd($hostGenFunctions::DU);
  return ($size, $exitStatus) if ($rc != 1);

  my $du_result = `$hostGenFunctions::DU -xsb $dirname`;
  $exitStatus = $?;
  chomp $du_result;
  if ($du_result ne "")
  {
    my @du_result = split( /\s+/, $du_result);
    $size = $du_result[0];
    chomp($size);
  }
  return ($size, $exitStatus);
}

#*****************************************************************
# getCpuCount() routine 
#
# Returns list of (number of processors in the Linux system, exitStatus).
# exitStatus will be 0 upon success.
#*****************************************************************

sub getCpuCount()
{
  my @processorLines = `grep '^processor[	]*:' /proc/cpuinfo`;

  if ($? != 0)
  {
    my $exitStatus  = $?;
    return (1, $exitStatus);
  }
  elsif ( @processorLines > 1 )
  {
    my $cpuCount = @processorLines;
    return ($cpuCount, 0);
  }
  else
  {
    return(1, 0)
  }
}

#******************************************
#   Non-Exported Subroutines
#******************************************

sub catCmd($)
{
    return join ($hostGenFunctions::DATALOC, 
                 $hostGenFunctions::CAT . ' ', $_[0]);
}

sub checkCmd($)
{
    # Purpose: To check whether the given command is present and executable

    my ($cmd) = @_;
    my $rc  = 1;
    if (!(-x "$cmd"))
    {
        $rc = 0;
    }

    return $rc;
}

sub execCmd($)
{
    # Purpose: To execute the given command checking for errors

    my ($cmd) = @_;
    my $value = '';
    chomp($value = `$cmd`);

    return $value;
}

#*****************************************************************
# isTextFile() routine 
# 
# It takes the fully qualified name of the file as input parameter.
# Returns 1 if text file, 0 otherwise
#*****************************************************************

sub isTextFile($)
{
  my ($fileName) = @_ ;
  my $retVal     = 0;

  #For a valid file, check if it's a text file
  if(-f $fileName)
  {
    if((checkCmd($hostGenFunctions::FILE))&&
       (`sh -c \"LC_ALL=C $hostGenFunctions::FILE $fileName\"` =~ / text/))
    {
      $retVal = 1;
    }
  }

  return $retVal;
}

1;

# End of Program
