#!/usr/local/bin/perl
# 
# $Header: emll/sysman/admin/scripts/hostOSD.pm /solaris/37 2011/03/31 12:35:09 jshoup Exp $
#
# hostOSD.pm
# 
# Copyright (c) 2004, 2011, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      hostOSD - <one-line expansion of the name>
#
#    DESCRIPTION
#      <short description of component this file declares/defines>
#
#    NOTES
#      <other useful comments, qualifications, etc.>
#
#    MODIFIED   (MM/DD/YY)
#    jshoup      03/30/11  - Fix for missing prtconf parameter on system-model
#                            data collection
#    jshoup      03/22/11  - Fix for 11850412 to provide Solaris patch
#                            collection.
#    jshoup      03/10/11  - bug 11658613: update system_config metric to
#                            contain system model
#                          - bug 11658699: remove type3 SN collection
#    prasawas    09/21/10  - Bug 8510103: Inserting not null and unique constraints
#                            on primary key columns of ECM_OS_COMPONENT table.
#    prasawas    08/11/10  - Fixing Bug 9381953: Incorrect command mentioned
#                            as source in OCM collected XML files.
#    prasawas    08/02/10  - Bug 9843322: Modifying getDNSNameServerResults by
#                            using already existing ocmprop module for retrieving the
#                            property value and introducing ignore case comparison of it.
#    prasawas    07/07/10  - Changing OS Vendor Name for all Solaris ports
#    ckalivar    06/22/10  - Fix bug 9830928: add description column to
#                            ECM_OS_REGISTERED_SW
#    asunar      06/10/10  - Adding filter for Ip Adress
#    ndutko      03/02/10  - Pipe the STDERR in pkgparam ops to dev null
#    ndutko      02/22/10  - Inclusion of functions to return the hostID and
#                            SystemSerialNumber
#    ckalivar    01/11/10  - Fix Bug8922174:Collect all the OS parameters set
#                            in /etc/system
#    ckalivar    10/30/09  - Fix Bug-8948168;fix local diskspace;added method
#                            getSortedDiskSizes
#    aghanti     10/09/09  - Removal of dependency on xpg4 utilities; removing
#                            unused sub getDirSize()
#    ckalivar    09/10/09  - Bug 8287160: Fixed host alises collection in
#                            getNICResults
#    ckalivar    06/29/09  - Fix-Bug:8572708- getCpuCount now supports multi
#                            core CPU's, discards virtual CPU's. getPerCPUResults
#                            will look for alternate source for cpu information
#                            if prtdiag fails 
#    ckalivar    02/17/09  - #Bug7689756, remove duplicate entries occuring in
#                            getModulesResults, #Bug7687452, corrected variable
#                            misspelt in getCurrSysDefnResults and fixed
#                            formatCPUResults giving NIL values for Virtual
#                            CPUs
#    aghanti     12/11/08  - Bugs 7623669 & 7623698 - Ignore null names for
#                            getIOCardResults & getOSRegisteredSoftwareResults
#    aghanti     10/04/07  - Bug 6454676 - Use isainfo (whenever avbl) to get
#                            ADDRESS_LENGTH_IN_BITS
#    aghanti     09/17/07  - Fix for bug 6334427
#    cvaishna    02/19/07  - Bug Fix : 5872070
#    cvaishna    03/21/07  - Bug Fix : 5942024
#    cvaishna    03/02/07  - fixing ifconfig output for ZONE issue
#    cvaishna    03/02/07  - Adding getMACAddress()
#    cvaishna    02/19/07  - Bug Fix : 5872070
#    ckannusa    03/07/07  - add method getHostNameForIP
#    ndutko      03/14/07  - Do not use '==' to compare strings
#    pparida     01/03/07  - Re-enable MAC in getNICResults(). Collection of MAC can be
#                            disabled by specifying so in collector.properties
#    glavash     01/03/07  - remove dups on os parameters
#    glavash     12/14/06  - pass through disk type on getMountedFileSystems
#    glavash     09/18/06  - bug 5548270 null os_name
#    cvaishna    12/26/06 - Fixing getOSPlatformId & getAddressLength for
#                           Solx86
#    cvaishna    06/08/06 - Bug Fix : 5293717 
#    cvaishna    05/28/06 - Impl. getInitProcesses for SunOS 
#    cvaishna    05/08/06 - Bug Fix : 5205041 
#    cvaishna    03/24/06 - Adding getModulesResults for SunOS 
#    cvaishna    03/23/06 - Bug Fix : 5093101 
#    ndutko      04/05/06  - Addition of getInitProcesses() - ER 5097554 
#    glavash     12/02/05  - ensure version and name are not null 4778267 
#    glavash     12/02/05  - fix getCpuCount and add siblings to cpu metric 
#    glavash     12/01/05  - remove emcp modules bug 4755461 
#    glavash     11/14/05  - remove MAC 4727761 
#    glavash     10/28/05  - upper case linux test 4707939 
#    glavash     10/10/05  - remove ulimit 
#    glavash     10/10/05  - change ulimit to remove units 4662538 
#    glavash     10/06/05  - bug 4652659 remove changing os_prop metric values 
#    glavash     09/28/05  - add ulimit and modules
#    glavash     09/28/05  - put NFS back in 
#    glavash     09/27/05  - default platform_id to -999 
#    glavash     08/30/05  - add platform id 
#    glavash     08/16/05  - fix bug 4554157 in NIC area 
#    glavash     08/10/05 -  fix 4520473 remove fsaio-pinned 
#    ndutko      08/03/05 - ndutko_code_checkin
#    glavash     07/14/05 - fix getCpuCount to ignore hyprethreaded CPU's
#    glavash     07/06/05 - add reiserfs to support fs, and fix MTU lookup
#    glavash     06/03/05 - test for local disks, and GB calc for disk space
#    glavash     04/06/05 - creation, routines taken from PtdpN.pm modules and centralized
# 

use strict;

package hostOSD;

require Exporter;

use constant NIL            => '';

use ecmErrorMsgs;
use hostDate;
use hostUtil;
use ecmAssoc;
use Config;
use ocmprop;

use File::Spec;
#*********************************
# Export Subroutines
#*********************************

@hostOSD::ISA =  ('Exporter');
@hostOSD::EXPORT = qw(
  &trim
  &getDiskInfo
  &getCpuCount
  &getHostName
  &getAdminEntries
  &left
  &right
  &catCmd
  &checkCmd
  &execCmd
  &evalCmd
  &getMaxSwapSpace
  &getCurrSysDefnResults
  &getCurrSysDef
  &getSysConfigResults
  &getPerCPUResults
  &getIOCardCount
  &getHardwareVendor
  &getHardwareName
  &getHardwarePlatform
  &getGetConfig
  &getInstructionSetList
  &getProcessorType
  &getArchType
  &getDomainName
  &getTotalMemory
  &getTotalDiskSpace
  &getClockSpeed
  &getIOCardResults
  &getNICResults
  &getNICDetails
  &getAddressLength
  &getOSNameVendorDist
  &getOSRelease
  &getDNSNameServersResults
  &getMountedFileSystemResults
  &getOSSoftwareComponentsResults 
  &getOSRegisteredSoftwareResults
  &getOSPlatformId
  &getModulesResults
  &getUlimitResults
  $getInitProcesses
  $getHostNameForIP
  &getHostId
  &getSystemSerialNumber
  &getSolarisPatches
  );


#*********************************
#  Global Variables
#*********************************

#should em_results be sorted?

$hostOSD::SORT_RESULTS = 0;

$hostOSD::ARCH      =  '/bin/arch';
$hostOSD::CAT       =  '/bin/cat';
$hostOSD::CHKCONFIG =  '/sbin/chkconfig';
$hostOSD::CUT       =  '/usr/bin/cut';
$hostOSD::DATALOC   =  '';
$hostOSD::DEVNULL   =  '/dev/null';
$hostOSD::DF        =  '/bin/df';
$hostOSD::EGREP     =  '/bin/egrep';
$hostOSD::ULIMIT    =  '/bin/ulimit';
$hostOSD::ETCSYSTEM =  '/etc/system';
$hostOSD::ETCMNTTABFILE =  '/etc/mnttab';
$hostOSD::ETCRESOLVCONFFILE = '/etc/resolv.conf';
$hostOSD::GETCONF   =  '/usr/bin/getconf';
$hostOSD::GETENT    =  '/usr/bin/getent';
$hostOSD::HEAD      = '/usr/bin/head';
$hostOSD::HOSTID    = '/usr/bin/hostid';
$hostOSD::SMBIOS    =  '/usr/sbin/smbios';
$hostOSD::PKGPARAM  =  '/usr/bin/pkgparam';
$hostOSD::HOSTNAME  =  '/bin/hostname';
$hostOSD::IFCONFIG  = '/sbin/ifconfig';
$hostOSD::KSTAT      = '/usr/bin/kstat';
$hostOSD::LSPCI     =  '/sbin/lspci';  # NOT ON SOL. -- lspci
$hostOSD::NSLOOKUP   = '/usr/sbin/nslookup';
$hostOSD::PROCMODULESFILE =  '/proc/modules';
$hostOSD::RPM       = '/usr/local/bin/rpm';
$hostOSD::RUNLEVEL  = '/sbin/runlevel';
$hostOSD::SORT      =  '/bin/sort';
$hostOSD::SYSCTL   = '/sbin/sysctl'; # NOT ON SOL. -- sysctl
$hostOSD::TAIL      =  '/usr/bin/tail';
$hostOSD::TIMEZONEFILE = '/etc/TIMEZONE';#used in hostDate.pm
$hostOSD::UNAME      = '/bin/uname';
$hostOSD::ASSOCIATION_RESULT = 1;
$hostOSD::METRIC_RESULT = 0;
$hostOSD::SWAP       = '/usr/sbin/swap';
$hostOSD::MACH       = '/bin/mach';
$hostOSD::SYSDEF     = '/usr/sbin/sysdef';
$hostOSD::MODINFO    = '/usr/sbin/modinfo';
$hostOSD::AWK        = '/usr/bin/awk';
$hostOSD::WHO        = '/usr/bin/who';
$hostOSD::ISAINFO    = '/usr/bin/isainfo';
$hostOSD::ARP        = '/usr/sbin/arp';
$hostOSD::PATCHADD   = '/usr/sbin/patchadd';

$hostOSD::CPU_VENDOR_TAG = NIL;
$hostOSD::CPU_SPEED_IN_MHZ_TAG = NIL;
$hostOSD::ECACHE_IN_MB_TAG = NIL;
$hostOSD::CPU_IMPLEMENTATION_TAG = NIL;
$hostOSD::REVISION_TAG = NIL;
$hostOSD::MASK_TAG = NIL;
$hostOSD::PLATFORMLOC = '/usr/platform/';
$hostOSD::PRTDIAG    = '/sbin/prtdiag';
$hostOSD::PRTPICL    = '/usr/sbin/prtpicl';
$hostOSD::PRTCONF    = '/usr/sbin/prtconf';
$hostOSD::KVMPRTDIAG = '/usr/kvm/prtdiag';
$hostOSD::ISALIST = '/usr/bin/isalist';

use constant NIL            => '';
use constant SEP            => ':';
use constant DOT_CHR        => '.';
use constant NOT_FOUND      => -1;
use constant COLON_CHR      => ':';
use constant NOT_APPLICABLE => 'N/A';
use constant UNKNOWN        => 'UNKNOWN';
use constant CPU_SIZE       => 6;
use constant IOCARD_SIZE    => 5;
use constant DETAIL_SP      => '  ';

#*********************************
# Exported Subroutines
#*********************************

sub trim
{

  #Arguments:  string or array
  #Outputs  :  string or array
  #Function : trim

  my (@out) = @_;
  for (@out)
  {
     s/^\s+//;
     s/\s+$//;
  }

  return wantarray ? @out : $out[0];
}

sub getDiskInfo
{
    # Purpose: To get disk space/filesystem space

    my $cmd     = '';
    my $value   = '';
    my @diskinfo = ();
    my $DFLocalFileSystemsParams = '-lkF ufs';
    my $DFAllFileSystemsParams = '-kF ufs';
    my ($DFtype) = @_;
    my $DFparams = $DFLocalFileSystemsParams;

    my $rc = checkCmd($hostOSD::DF) & 
                 checkCmd($hostOSD::TAIL);
    if ($rc == 1)
    {
      $DFparams = $DFAllFileSystemsParams if $DFtype eq 'ALL';
      $cmd = "$hostOSD::DF $DFparams | 
                    $hostOSD::TAIL +2";
    }
    else
    {
      return @diskinfo;
    }
    $value = execCmd($cmd);
    @diskinfo = split ("\n", $value);
    chomp(@diskinfo);

    return @diskinfo;
}

# returns hash of  "volume" and its "total size" as their key and  value pairs.
sub getSortedDiskSizes()
{
#
# get diskinfo for total space calculation
#
    my @diskinfo     = getDiskInfo();
    my %diskCaps = ();
    my @fields = ();
    my $totalkb = 0;
    my $dev;
# Sample getDiskInfo() output
# /dev/vg01/test_filesystem_longnamelong
#                      21121420    22607 21098813     1%   /test_longname
# /dev/vg00/lvol4       617872    16640   601232     3%   /home_local
# /dev/vg00/lvol5       7542816  4161904  3380912    56%   /opt
# /dev/vg01/lvol2       96025850   412570 95613280     1%   /scratch
    my $i = 0;
    my $noOfEs = 0;
    $noOfEs = scalar(@diskinfo);
    for ($i = 0; $i < $noOfEs; $i++)
    {
      my @tokens = split(m/[\t\s]+/, $diskinfo[$i]);
      $dev = $tokens[0];
      my $tokenSize=scalar(@tokens);
      if ($tokenSize > 1)
      {
        $totalkb = $tokens[1];
      }
      elsif ($tokenSize == 1) #next if disk space is given in next line
      {
        @tokens = split(m/[\t\s]+/, $diskinfo[$i+1]);
        $totalkb= ($tokens[0] =~ m/^\d+$/) ? $tokens[0] : $tokens[1];
              $i++;
      }
      next if defined $diskCaps{$dev}; # skip duplicate entries
      $diskCaps{$dev} = $totalkb;
    }
    return %diskCaps;
}

sub getTotalDiskSpace
{
    my $totalDiskCap = 0;
    my %diskCaps= getSortedDiskSizes();
    while ( my ($key, $value) = each(%diskCaps) ) {
        $totalDiskCap+=$value;
    }

    # truncate to 2 decimal points
    $totalDiskCap = sprintf("%.2f", $totalDiskCap / (1024 * 1024));

    return $totalDiskCap;

}

sub getSolarisPatches
{
    # returns an array of patch ids

    my $PATCH_PARAMS = "-p";
    my $cmd = NIL;
    my $id = NIL;
    my $rest = NIL;
    my @patchIds;

    my $rc = checkCmd($hostOSD::PATCHADD);
    if ($rc == 1)
    {
        $cmd = "$hostOSD::PATCHADD $PATCH_PARAMS";
    }
    else
    {
        return @patchIds;
    }

    my $output = execCmd($cmd);
    my @patches = split(/\n/, $output);
    for(my $i = 0; @patches; $i++)
    {
        my $patchLine = shift @patches;
        ($rest, $id) = split (': ', $patchLine, 2);
        ($id, $rest) = split (' Obsoletes', $id, 2);
        if ($id ne NIL)
        {
            push(@patchIds, $id);
        }
    }
    return @patchIds;
}


#*****************************************************************
# getCpuCount() routine 
#
# Returns list of (number of processors in the Linux system, exitStatus).
# exitStatus will be 0 upon success.
#*****************************************************************


sub getCpuCount()
{
      my @cpudata = ();
      my $noOfCPUs = 0;
      my $failed  = 1;
      my %chipIds;
      my $rc = checkCmd($hostOSD::KSTAT);
      # return if kstat command not successful with a failure status
      return ($noOfCPUs , $failed ) if ($rc != 1);

      chomp(my @data = `$hostOSD::KSTAT cpu_info`);
      # Build CPUs components tree
      for(my $i=0;$i<=$#data;$i++) {
         if($data[$i] =~ m/^module:/ ) {
            $i++;
            my $chip_id = NIL;
            my $core_id = NIL;

            while($data[$i]  !~ m/^module:/ && $data[$i]  !~ m/^\s*$/ && $i <=$#data ) {
               if($data[$i] =~ m/\s+chip_id\s+/i) {
                   $chip_id = (split('\s+',$data[$i]))[2];
               }
               elsif($data[$i] =~ m/\s+core_id\s+/i) {
                   $core_id = (split('\s+',$data[$i]))[2];
               }
               $i++;
            }
            if($chip_id ne NIL || $core_id ne NIL) {
              $chipIds{$chip_id}{$core_id}=1;
            }
            $i-- if $data[$i] =~ m/^module:/;

         }
      }

     #traverse the tree to get the Number of CPUs
     foreach my $cid (keys %chipIds) {
         my $count = scalar(keys %{$chipIds{$cid}}); # get the number of core ids associated with that chip id
         $noOfCPUs += $count;
     }

    return ($noOfCPUs, 0);
}

#gets perCPU results if prtdiag fails
sub getAlternateCPUresults
{
      my ($vendor) = @_;
      my @cpudata = ();
      my @result = ();
      my $failed  = 1;
      my %chipIds;
      my $cache = NIL;
      my $revision = NIL;
      my $mask = NIL;
      my $rc = checkCmd($hostOSD::KSTAT);
      # return if kstat command not successful with a failure status
      return @result if ($rc != 1);

      chomp(my @data = `$hostOSD::KSTAT cpu_info`);
      # Build CPUs components tree
      for(my $i=0;$i<=$#data;$i++) {
         if($data[$i] =~ m/^module:/ ) {
            $i++;
            my $chip_id = NIL;
            my $core_id = NIL;
            my @implementation= ();
            my $vendor_id = NIL;
            my $freq_MHz = NIL;
            my @brand_info =NIL;

            while($data[$i]  !~ m/^module:/ && $data[$i]  !~ m/^\s*$/ && $i <=$#data )
            {
               if($data[$i] =~ m/\s+chip_id\s+/i) {
                   $chip_id = (split('\s+',$data[$i]))[2];
               }
               elsif($data[$i] =~ m/\s+core_id\s+/i) {
                   $core_id = (split('\s+',$data[$i]))[2];
               }
               elsif($data[$i] =~ m/\s+implementation\s+/i) {
                   @implementation = (split('\s+',$data[$i]));
               }
               elsif($data[$i] =~ m/\s+clock_MHz\s+/i) {
                   $freq_MHz = (split('\s+',$data[$i]))[2];
               }
               elsif($data[$i] =~ m/\s+vendor_id\s+/i) {
                   $vendor_id = (split('\s+',$data[$i]))[2];
               }

               $i++;
            }

            my $brandString =  join(' ', splice(@brand_info,2));
            my $implString = join(' ', splice(@implementation,2));
            $implString = "$brandString" if ( $implString eq NIL);

            $vendor_id = 'GenuineIntel' if ( $implString =~ m/GenuineIntel/i && $vendor_id eq NIL);
            $vendor_id = 'AuthenticAMD' if ($implString =~ m/AuthenticAMD/i && $vendor_id eq NIL);
            $vendor_id=$vendor if($vendor_id eq NIL);

            my $rString="$vendor_id|$freq_MHz|$cache|$implString|$revision|$mask";

            if($chip_id ne NIL || $core_id ne NIL) {
              $chipIds{"$chip_id|$core_id"}=$rString;
            }
            $i-- if $data[$i] =~ m/^module:/;

         }
      }
     my $index=0;
     #traverse the tree to get the Number of CPUs
     foreach my $cid (keys %chipIds)
     {
         $result[$index]="em_result=$chipIds{$cid}\n";
         $index++;
     }

    return @result;
}

sub getHostId
{
    # Purpose: To get hostid

    my $cmd     = NIL;
    my $value   = NIL;
    my $rc = checkCmd($hostOSD::HOSTID);
    if ($rc == 1)
    {
      $cmd = "$hostOSD::HOSTID";
    }
    else
    {
      return $value;
    }
    $value            = execCmd($cmd);

    return $value;
}

sub getSystemSerialNumber
{
    my $sn = NIL;

    # try to obtain SN via sneep
    $sn = getSneepSN();
    if ($sn ne NIL) {
      return $sn;
    }

    # how we collect the SN depends on the arch
    # these are the sparc collection cases
    my $hwPlat = getHardwarePlatform();
    if ($hwPlat ne NIL)
    {
      if ($hwPlat =~ /sun4u/)
      {
        $sn = getPrtpiclSN();
      }
      elsif ($hwPlat =~ /sun4v/)
      {
        $sn = getPrtdiagSN();
      }
    }

    # if we're here, assume x86/64 and attempt smbios collection
    if ($sn eq NIL)
    {
      $sn = getSmbiosSN();
    }

    return $sn;
}

sub getSneepSN
{
    my $sneepCmd = "/bin/sneep";
    my $pkgparamCmd = $hostOSD::PKGPARAM;
    my $rc = checkCmd($pkgparamCmd);
    if ($rc == 1)
    {
        my $basedir = execCmd($pkgparamCmd . " SUNWsneep BASEDIR 2>/dev/null");
        $sneepCmd = $basedir . $sneepCmd;
    }

    my $rc = checkCmd($sneepCmd);
    if ($rc != 1)
    {
        return NIL;
    }

    my $sneepSN = execCmd($sneepCmd);

    if ($sneepSN =~ m/unknown/i) {
        return NIL;
    } else {
        return $sneepSN;
    }
}

sub getPrtpiclSN
{
    my $cmd = $hostOSD::PRTPICL;
    my $rc = checkCmd($cmd);
    if ($rc != 1)
    {
      return NIL;
    }

    my $target = ":chassis-sn";
    my $output = execCmd($cmd . " -v 2>/dev/null");
    my @lines = split(/\n/, $output);
    for(my $i = 0; @lines; $i++)
    {
      my $line = shift @lines;
      if ($line =~ m/$target/)
      {
        my $indx = index($line, $target) + length($target);
        if ($indx < length($line))
        {
          return trim(substr $line, $indx);
        }
      }
    }

    return NIL;
}

sub getPrtdiagSN
{
    my $cmd = "/usr" . $hostOSD::PRTDIAG;
    my $rc = checkCmd($cmd);
    if ($rc != 1)
    {
      return NIL;
    }

    # the output to grab a serial number via prtdiag looks like:

    # Chassis Serial Number
    # ---------------------
    # 0822MTE06Z

    # so, we'll want to check for the line containing "Chassis Serial Number"
    # and then take the 2nd line after it

    my $csnLineCnt = 0;
    my $target = "Chassis Serial Number";
    my $output = execCmd($cmd . " -v 2>/dev/null");
    my @lines = split(/\n/, $output);
    for(my $i = 0; @lines; $i++)
    {
      my $line = shift @lines;

      if ($csnLineCnt == 2)
      {
        return $line;
      }
      if ($csnLineCnt == 1)
      {
        $csnLineCnt = 2;
      }
      if ($line =~ m/$target/)
      {
        $csnLineCnt = 1;
      }
    }
    return NIL;
}

sub getSmbiosData
{
    my $smbiosCmd = $hostOSD::SMBIOS;
    my $rc = checkCmd($smbiosCmd);
    if ($rc != 1)
    {
        return NIL;
    }

    my $type = $_[0];
    my $target = $_[1];

    my $output = execCmd($smbiosCmd . " -t $type 2>/dev/null");
    my @lines = split(/\n/, $output);
    for(my $i = 0; @lines; $i++) {
        my $line = shift @lines;
        if ($line =~ m/$target/) {
            my $indx = index($line, $target) + length($target);
            if ($indx < length($line)) {
                my $tmp = trim(substr $line, $indx);
                if ($tmp =~ m/not available/i) {
                    # do nothing
                } elsif ($tmp =~ m/to be filled by o.e.m/i) {
                    # do nothing
                } else {
                    return $tmp;
                }
            }
        }
    }

    return NIL;
}

sub getSmbiosSN
{
    # try type 1 - the system serial number
    return getSmbiosData("1","Serial Number: ");
}

sub getHostName
{
    # Purpose: To get hostname

    my $cmd     = NIL;
    my $value   = NIL;
    my $rc = checkCmd($hostOSD::HOSTNAME);
    if ($rc == 1)
    {
      $cmd = "$hostOSD::HOSTNAME";
    }
    else
    {
      return $value;
    }
    $value            = execCmd($cmd);

    return $value;
}

sub getAdminEntries($)
{
    # Purpose: To get the entries from administrative database for given database and value

    my ($parm)  = @_;
    my $cmd     = NIL;
    my $value   = NIL;
    if (-x "$hostOSD::GETENT")
    {
      $cmd = "$hostOSD::GETENT $parm";
    }
    else
    {
      return $value;
    }
    $value = evalCmd($cmd);

    return $value;
}

#******************************************
#   glavash they are now ... Non-Exported Subroutines
#******************************************

sub catCmd($)
{
    return join ($hostOSD::DATALOC, 
                 $hostOSD::CAT . ' ', $_[0]);
}

sub checkCmd($)
{
    # Purpose: To check whether the given command is present and executable

    my ($cmd) = @_;
    my $rc  = 1;
    if (!(-x "$cmd"))
    {
        $rc = 0;
    }

    return $rc;
}

sub execCmd($)
{
    # Purpose: To execute the given command checking for errors

    my ($cmd) = @_;
    my $value = '';
    chomp($value = `$cmd`);

    return $value;
}

sub evalCmd($)
{
    # Purpose: To execute the given command w/o checking for errors

    my ($cmd) = @_;
    my $value = NIL;
    chomp($value = `$cmd`);

    return $value;
}

sub left($$;$)
{
    my ($tag, $text, $ignorecase) = @_;
    $ignorecase = 0 if (!defined $ignorecase);

    my $value = $text;
    if ($ignorecase)
    {
        $tag = lc($tag);
        $text = lc($text);
    }
    my $loc = index($text, $tag);
    if ($loc != -1)
    {
        $value = substr($value, 0, $loc);
        $value = trim($value);
        $value =~ s/\s+/ /g;
    }

    return $value;
}

sub right($$;$)
{
    my ($tag, $text, $ignorecase) = @_;
    $ignorecase = 0 if (!defined $ignorecase);

    my $value = $text;
    if ($ignorecase)
    {
        $tag = lc($tag);
        $text = lc($text);
    }
    my $loc = index($text, $tag);
    if ($loc != -1)
    {
        $value = substr($value, $loc + length($tag));
        $value = trim($value);
        $value =~ s/\s+/ /g;
    }

    return $value;
}

sub getMaxSwapSpace
{
    debugTrace('Calling... getMaxSwapSpace()');

    #Function: Calculate maximum swap space

    my $i         = 0;
    my $swapspace = 0;
    my @swap      = getSwapSpaceInfo();

    my @swap1 = ();
    my @swap2 = ();
    my $posn  = 0;
    my $size = 0;

    @swap1 = split (' ', $swap[0]);

    $size = scalar(@swap1);
    for ($i = 0 ; $i < $size ; $i++)
    {
        if ($swap1[$i] eq "blocks") { $posn = $i; last; }
    }

    $size = scalar(@swap);
    for ($i = 1 ; $i < $size ; $i++)
    {
        @swap2 = split (' ', $swap[$i]);
        $swapspace = $swapspace + trim($swap2[$posn]);
    }
    $swapspace = 512 * $swapspace;

    return $swapspace;
}

sub getSwapSpaceInfo
{
    debugTrace('Calling... getSwapSpaceInfo()');

    # Purpose: To get swap space

    my $datadir = $hostOSD::DATALOC;
    my $cmd     = '';
    my $value   = '';
    my @swap    = ();
    if ($datadir eq '')
    {
        my $rc = checkCmd($hostOSD::SWAP);
        if ($rc == 1)
        {
            $cmd = "$hostOSD::SWAP -l";
        }
        else
        {
            return @swap;
        }
    }
    else
    {
        $cmd = catCmd('swapl.out');
    }
    $value = execCmd($cmd);
    @swap = split ("\n", $value);
    chomp(@swap);

    return @swap;
}


#
# This sub routine is used to help construct the OS Component result set
# the data returned by this routine is only used by this one metric so we are 
# building the acutal em_result strings here and returning an array of results
# This will prevent us from having to walk the arrays twice
#
# the em_result is of the format  em_result=SOURCE|NAME|VALUES
#
sub getCurrSysDefnResults
{
    my @results = ();

#    my $ipcs = getIPCSStatus();

    my @parm = getCurrSysDef();
    my $size     = scalar(@parm);
    my @value  = ();
    my @parms  = ();
    my $i        = 0;
    my $j        = 0;
    my $procflag = 0;
    my $exclude  = NIL;

    for ($i = 0 ; $i < $size ; $i++)
    {
        if ($procflag == 0)
        {
            if (index($parm[$i], "* Tunable Parameters") > -1)
            {
                $procflag = 1;
            }
        }
        elsif (substr($parm[$i], 0, 1) eq '*') { next; }

        # To be rewritten later in a better way.
        elsif (index($parm[$i], "mapped memory") > -1)
        {
            ($value[$j], $parms[$j]) = split ("mapped memory", $parm[$i], 2);
            $parms[$j] = "mapped memory";
            $results[$j] = "em_result=$hostOSD::SYSDEF|$parms[$j]|$value[$j]\n";
            $j++;
        }
        else
        {
            ($value[$j], $parms[$j]) = split (' ', $parm[$i], 2);
            $results[$j] = "em_result=$hostOSD::SYSDEF|$parms[$j]|$value[$j]\n";
            $j++;
        }
    }

    # add maximum swap space

    my $max_swap = getMaxSwapSpace();
    $max_swap = sprintf("%.2f", (trim($max_swap)) / (1024 * 1024));

    $results[$j++] = "em_result=$hostOSD::SWAP|Maximum Swap Space (MB)|$max_swap\n";

    # Add Shared Memory/Semaphore related information
    # sample lines of /etc/system content:
    ###############################
    #set mvfs:mvfs_largeinit = 1
    #
    #set shmsys:shminfo_shmmax=4294967295
    #set shmsys:shminfo_shmmin=1
    #set shmsys:shminfo_shmmni=512
    #
    # * Message Queue
    # set msgsys:msginfo_msgmap=1024
    # set msgsys:msginfo_msgmax=8192
    # set msgsys:msginfo_msgtql=1024
    ################################
    if (open(ETCSYSTEM,"$hostOSD::ETCSYSTEM"))
    {
      my $line;
      while($line = <ETCSYSTEM>)
      {
          chomp($line);
          my $key=NIL;
          my $val=NIL;
          #skip comment lines
          next if ($line =~ /\s*(#|\*)/ );
          # get key and values from any assignment statement
          if ($line =~ /\s*set\s*(.*?)\s*=\s*(.*?)\s*$/)
          {
             $key = $1;
             $val = $2;
             # This step is to keep compatibility of HC metrics
             if( $key =~ m/^(semsys|shmsys)/ )
             {
                $key = uc((split(/_/ , $key))[1]); #split based on underscore and get laterpart and convert it to uppercase
             }
             #collect any OS parameter thats set in /etc/system
             if ( $key ne NIL )
             {
                 $results[$j++] = "em_result=$hostOSD::ETCSYSTEM|$key|$val\n";
             }
          }#if
      }#while
   }#if

    # Bug : 5872070 (Part-2)
    my $ulimitVal;

    chomp($ulimitVal = `$hostOSD::ULIMIT -c`);
    $results[$j++] = "em_result=$hostOSD::ULIMIT|Maximum Core File Size|$ulimitVal\n";

    chomp($ulimitVal = `$hostOSD::ULIMIT -d`);
    $results[$j++] = "em_result=$hostOSD::ULIMIT|Maximum Size of Data Segment or Heap (kbytes)|$ulimitVal\n";

    chomp($ulimitVal = `$hostOSD::ULIMIT -f`);
    $results[$j++] = "em_result=$hostOSD::ULIMIT|Maximum File Size|$ulimitVal\n";

    chomp($ulimitVal = `$hostOSD::ULIMIT -s`);
    $results[$j++] = "em_result=$hostOSD::ULIMIT|Maximum Size of Stack Segment (kbytes)|$ulimitVal\n";

    chomp($ulimitVal = `$hostOSD::ULIMIT -t`);
    $results[$j++] = "em_result=$hostOSD::ULIMIT|Maximum CPU Time (Seconds)|$ulimitVal\n";
    
    return @results;
}
sub getCurrSysDef
{
    # Purpose: To get current system definition

    my $cmd     = NIL;
    my $value   = NIL;
    my @sdef    = ();
    my $rc = checkCmd($hostOSD::SYSDEF);
    if ($rc == 1)
    {
      $cmd = "$hostOSD::SYSDEF -i 2>$hostOSD::DEVNULL";
    }
    else
    {
      return @sdef;
    }
    $value = execCmd($cmd);
    @sdef = split ("\n", $value);
    chomp(@sdef);

    return @sdef;
}

#
# This sub routine is used to help construct the OS Component result set
# the data returned by this routine is only used by this one metric so we are 
# building the acutal em_result strings here and returning an array of results
# This will prevent us from having to walk the arrays twice
#
# em_result=SOURCE|NAME|VALUE
#

sub getSysConfigResults
{

    my @parmlist = ('OPEN_MAX', 'CHILD_MAX');

    #  If you want to collect all the params under getconf,
    #  uncommnet next line;  comment previous line.
    #  my @parmlist  = ('-a');
    #  glavash ignore above comment, 
    #  getSysConf only returns the two params you'll need to change it. 
    my $i     = 0;
    my $j     = 0;
    my $k     = 0;
    my $rest1 = NIL;
    my $rest2 = NIL;
    my @sdef  = getSysConf();

    my $size2 = scalar(@sdef);
    my $size1 = scalar(@parmlist);

    my @results = ();
    my $param = NIL;
    my $value = NIL;

    if (($size1 == 1) && ($parmlist[0] eq '-a'))
    {
        for ($i = 0 ; $i < $size2 ; $i++)
        {
            ($param, $value) = split (SEP, $sdef[$i], 2);
            $results[$i] = "em_result=$hostOSD::GETCONF|$param|$value\n";
        }
    }
    else
    {
        for ($j = 0 ; $j < $size1 ; $j++)
        {
            for ($i = 0 ; $i < $size2 ; $i++)
            {
                ($rest1, $rest2) = split (SEP, $sdef[$i], 2);
                if (defined $rest1)
                {
                    $rest1 = trim($rest1);
                    if ($rest1 eq $parmlist[$j])
                    {
                        $param = $rest1;
                        $value = $rest2;
                        $results[$k++] = "em_result=$hostOSD::GETCONF|$param|$value\n";
                        last;
                    }
                }
            }
        }
    }
    return @results;
}
sub getSysConf
{   

    # Purpose: To get system configuration values

    my $cmd     = NIL;
    my $value   = NIL;
    my @sdef    = ();
    my $rc = checkCmd($hostOSD::GETCONF);
    if ($rc == 1)
    {
      $cmd = "$hostOSD::GETCONF";
    }
    else
    {
      return @sdef;
    }

    $cmd   = "$hostOSD::GETCONF OPEN_MAX";
    $value = execCmd($cmd);
    $sdef[0] = "OPEN_MAX:$value";
    $cmd   = "$hostOSD::GETCONF CHILD_MAX";
    $value = execCmd($cmd);
    $sdef[1] = "CHILD_MAX:$value";
    chomp(@sdef);

    return @sdef;
}
sub getHardwareDetails
{
    my $cmd      = '';
    my $value    = '';
    my @md       = ();
    my @hw       = ();
    my $i        = 0;
    my $platform = getHardwarePlatform();
    chomp(my @plat = split (' ', $platform));	
    my $prtdiag = '';


    for ($i = 1 ; $i > -1 ; $i--)
    {
       $cmd = '';
       $prtdiag = join ($plat[$i], $hostOSD::PLATFORMLOC, $hostOSD::PRTDIAG);
       if (-x "$prtdiag")
       {
                $cmd = "$prtdiag -v";
                last;
	   }
       if ($cmd eq '' && -x $hostOSD::KVMPRTDIAG)
       {
            $cmd = $hostOSD::KVMPRTDIAG;
       }   
	}
    $value = execCmd($cmd);
    @hw = split ("\n", $value);
    chomp(@hw);

    return @hw;
}


sub findIdx($$)
{
    my ($header, $column) = @_;

    my $key = NIL;
    my $j   = 0;

    $header =~ s/^FRU Name/FRU_Name/;
    my @detail = split (' ', $header);

    foreach $key (@detail)
    {
        if ($key eq $column)
        {
            return $j;
        }
        $j++;
    }

    return NOT_FOUND;
}

sub joinValue($;$)
{
    my ($tag, $value) = @_;
    $value = NIL if (!defined $value);

    return join (':', trim($tag), trim($value));
}


sub formatIOdetails
{

    my ($i, @hwdetails) = @_;

    my $hwdetailsize = scalar(@hwdetails);
    my @ioddata      = ();
    my $brd          = NIL;
    my $busType      = NIL;
    my $freqMHz      = NIL;
    my $name         = NIL;
    my $model        = NIL;
    my $slot         = NIL;
    my $brdIdx       = NOT_FOUND;
    my $busIdx       = NOT_FOUND;
    my $freqIdx      = NOT_FOUND;
    my $nameIdx      = NOT_FOUND;
    my $modelIdx     = NOT_FOUND;
    my $slotIdx      = NOT_FOUND;
    my $lastIdx      = NOT_FOUND;
    my $locationIdx  = NOT_FOUND;
    my $nameplus     = 0;

    my $vendor = "Unknown";
    my $line   = NIL;
    my @detail = ();
    my $j      = 0;
    my $k      = 0;

    for (; $i < $hwdetailsize ; $i++)
    {
        $line = trim($hwdetails[$i]);
	# fix for 4955550
	$line =~ s/Slot\sID/Slot-ID/g;
	
        if ($line eq NIL)
        {
            next;
        }
        if ((!defined $line) || ($line eq NIL) || ($line =~ m/^-/)
            || ($line =~ m/^ERROR:/))
        {
            last if ($j > 0);
            next;
        }
        if ($nameIdx == NOT_FOUND)
        {
            if ($line =~ m/Name\s\+/)
            {
                $nameplus = 1;
                next;
            }
            if ($nameplus eq 1)
            {
                $nameIdx = findIdx($line,'Path');
                if ($nameIdx != NOT_FOUND)
                {
                        $busIdx   = findIdx($line, 'Type');
                        $freqIdx  = findIdx($line, 'MHz');
                        $modelIdx = findIdx($line, 'Model');
                        $slotIdx  = findIdx($line, 'Status');
                }
                next;
            }
            else
            {
                $nameIdx = findIdx($line, 'Name');
                if ($nameIdx != NOT_FOUND)
                {
                        $brdIdx   = findIdx($line, 'Brd');
                        $busIdx   = findIdx($line, 'Type');
                        $freqIdx  = findIdx($line, 'MHz');
                        $modelIdx = findIdx($line, 'Model');
                        $slotIdx  = findIdx($line, 'Slot');
                        $locationIdx = findIdx($line, 'Location');
                        if (($busIdx == NOT_FOUND) && ($freqIdx > $modelIdx))
                        {
                                $slotIdx  -= 1;
                                $nameIdx  -= 1;
                                $modelIdx -= 1;
                                $freqIdx  -= 1;
                        }
                }
                next;
             }
        }
        if (($line =~ m/^=/) || ($line =~ m/^No failure/i)
            || ($line =~ m/^Failed/i)
            || ($line =~ m/^No Recent System/i))
        {
            last;
        }

        $line =~ s/(  )\s+/$1/g;
        if ($line =~ m/okay/)
        {
             next;
        }
        @detail = split /\s+/, $line;
        $lastIdx = scalar(@detail) - 1;
        $name = ($detail[$nameIdx] ? $detail[$nameIdx] : NIL);
        if ($brdIdx != NOT_FOUND)
        {
            $brd = ($detail[$brdIdx] ? $detail[$brdIdx] : NIL);
        }
        if ($busIdx != NOT_FOUND)
        {
            $busType = ($detail[$busIdx] ? $detail[$busIdx] : NIL);
        }
        if ($freqIdx != NOT_FOUND)
        {
            $k = ($freqIdx > $lastIdx ? $lastIdx : $freqIdx);
            $freqMHz = ($detail[$k] ? $detail[$k] : NIL);
            $freqMHz =~ s/\s*MHz//i;
        }
        if ($slotIdx != NOT_FOUND)
        {
            $slot = ($detail[$slotIdx] ? $detail[$slotIdx] : NIL);
        }
        if ($modelIdx != NOT_FOUND)
        {
            if (($modelIdx > $freqIdx) || ($lastIdx == $freqIdx))
            {
                $model = ($detail[$modelIdx] ? $detail[$modelIdx] : NIL);
            }
            else
            {
                $model = NIL;
            }
            $name .= ' ' . $model if ($model ne NIL);
        }
        # If the name is null, ignore the record
        if ($name eq NIL)
        {
            next;
        }
        if ($name =~ m/SUN/)
        {
            $vendor = 'Sun Microsystems';
        }
        elsif ($name =~ m/Symbios/)
        {
            $vendor = 'Symbios';
        }
        elsif ($name =~ m/QLGC/)
        {
            $vendor = 'Qlogic';
        }
        elsif ($name =~ m/LP8000/)
        {
            $vendor = 'Emulex';
        }
        else
        {
            $vendor = 'Unknown';
        }
        $ioddata[$j++] = joinValue('',           $vendor);
        $ioddata[$j++] = joinValue('',             $name);
        $ioddata[$j++] = joinValue('', $freqMHz);
        $ioddata[$j++] = joinValue('',         $busType);
        $ioddata[$j++] = joinValue('',         'N/A');
    }

    return @ioddata;
}

sub formatCPUdetails
{
    debugTrace('Calling... formatCPUdetails()');

    my ($i, $vendor, @hwdetails) = @_;

    my $hwdetailsize = scalar(@hwdetails);
    my @cpudata      = ();
    my $brd          = NIL;
    my $freqMHz      = NIL;
    my $cacheMB      = NIL;
    my $impl         = NIL;
    my $mask         = NIL;
    my $brdIdx       = NOT_FOUND;
    my $freqIdx      = NOT_FOUND;
    my $cacheIdx     = NOT_FOUND;
    my $implIdx      = NOT_FOUND;
    my $maskIdx      = NOT_FOUND;
    my $noIdx        = NOT_FOUND;
    my $lastIdx      = NOT_FOUND;

    my $line   = NIL;
    my @detail = ();
    my %boards = ();
    my $j      = 0;
    my $k      = 0;
    my $l      = -1;
    my $kb     = 0;
    my $checkComma = 0;

    for (; $i < $hwdetailsize ; $i++)
    {
        $line = trim($hwdetails[$i]);
        if ((!defined $line) || ($line eq NIL) || ($line =~ m/^-/))
        {
            last if ($j > 0);
            next;
        }
        if ($implIdx == NOT_FOUND)
        {
	    $line =~ s/Slot\sID/Slot-ID/g;
            $line =~ s/CPU\sID/CPU-ID/g;
            $implIdx  = findIdx($line, 'Impl.');
            $implIdx  = findIdx($line, 'Implementation') if ($implIdx == NOT_FOUND);
            if ($implIdx != NOT_FOUND)
            {
                $brdIdx   = findIdx($line, 'Brd');
                $brdIdx   = findIdx($line, 'CPU') if ($brdIdx == NOT_FOUND);
                $brdIdx   = findIdx($line, 'Slot-ID') if ($brdIdx == NOT_FOUND);
                $brdIdx   = findIdx($line, 'CPU-ID') if ($brdIdx == NOT_FOUND);
                $noIdx    = findIdx($line, 'No.');
                $brdIdx   = $noIdx if ($brdIdx == NOT_FOUND);
                $freqIdx  = findIdx($line, 'MHz');
                $freqIdx  = findIdx($line, 'Freq') if ($freqIdx == NOT_FOUND);
                $freqIdx  = findIdx($line, 'Frequency') if ($freqIdx == NOT_FOUND);
                $cacheIdx = findIdx($line, 'MB');
                $cacheIdx = findIdx($line, 'Size') if ($cacheIdx == NOT_FOUND);
                $maskIdx = findIdx($line, 'Mask');
            }
            next;
        }
        if ($line =~ m/^=/ || $line =~ m/^Used Memory:/i)
        {
            last;
        }

        if ($line =~ m/\,/)
	{
	    $line =~ s/\,\s+/\,/g;
	}

        $line =~ s/MB//g;
        $line =~ s/MHz//g;
        $line =~ s/(  )\s+/$1/g;

        if (($line =~ m/\,/) && ($checkComma == 0) && ($maskIdx !=NOT_FOUND))
        {
	    $checkComma = 1;
        }

        #@detail = split (DETAIL_SP, $line);
	@detail = split /\s+/,$line;
        $lastIdx = scalar(@detail) - 1;
        if ($maskIdx !=NOT_FOUND)
        {
            $mask = ($detail[$maskIdx] ? $detail[$maskIdx] : NIL);
        }
        if ($brdIdx != NOT_FOUND)
        {
            $brd = ($detail[$brdIdx] ? $detail[$brdIdx] : NIL);
            if ($brd ne NIL)
            {
                $boards{$brd} = 1;
            }
        }
        if ($freqIdx != NOT_FOUND)
        {
            $freqMHz = ($detail[$freqIdx] ? $detail[$freqIdx] : NIL);
            $freqMHz =~ s/\s*MHz//i;
        }
        if ($implIdx != NOT_FOUND)
        {
            $impl = ($detail[$implIdx] ? $detail[$implIdx] : NIL);
        }
        if ($cacheIdx != NOT_FOUND)
        {
            $cacheMB = ($detail[$cacheIdx] ? $detail[$cacheIdx] : NIL);
            if ($cacheMB =~ m/KB/i)
            {
                $cacheMB =~ s/\s*KB//i;
                $kb      = 1;
            }
            else
            {
                $cacheMB =~ s/\s*MB//i;
            }
            $l = index($cacheMB, ' ');
            if($l != NOT_FOUND)
            {
               $mask =  $impl;
               $impl = substr($cacheMB,$l+1,length($cacheMB));
               $cacheMB = substr($cacheMB,0,$l);
            }
            if($kb == 1)
            {
               $cacheMB = ($cacheMB / 1024);
               $kb = 0;
            }
        }

        $cpudata[$j++] = joinValue('', $vendor);
        $cpudata[$j++] = joinValue(NIL, $freqMHz);
        $cpudata[$j++] = joinValue('', $cacheMB);
        $cpudata[$j++] = joinValue('', $impl);
        $cpudata[$j++] = joinValue('',  NIL);
        $cpudata[$j++] = joinValue('', $mask);


	if ((($noIdx != NOT_FOUND) && ($maskIdx < $lastIdx)) || ($checkComma == 1))
	{
            $cpudata[$j++] = joinValue('',         $vendor);
            $cpudata[$j++] = joinValue('',   $freqMHz);
            $cpudata[$j++] = joinValue('',       $cacheMB);
            $cpudata[$j++] = joinValue('', $impl);
            $cpudata[$j++] = joinValue('',           NIL);
            $cpudata[$j++] = joinValue('',           $mask);
        }
    }

    return @cpudata;
}



#
# getCPUResults - returns an array of em_result records 
# 1 for each CPU found
#
sub getPerCPUResults
{
	my $i;
   	my $j;
   	my $k;
   	my $line;
   	my $vendor = NIL;
	my @result;
	my @cpudata;
	my $NoOfCPU=0;
        my $cmd     = NIL;
        my $arch     = NIL;
        my $rc = checkCmd($hostOSD::UNAME);

        if ($rc == 1)
        {
          $cmd = "$hostOSD::UNAME -p";
          $arch = execCmd($cmd);
        }
 

	my @hwdetails  = getHardwareDetails();
   	if ($arch eq "sparc")
   	{ 
		$vendor = 'Sun Microsystems';
   	}
	my $hwdetailsize = scalar(@hwdetails);

        
        #if @hwdetails size is zero (can be because prtdiag command failed)
        if ($hwdetailsize == 0)
        {
            @result = getAlternateCPUresults($vendor);
            return @result;
        }

 
	for ($i = 0 ; $i < $hwdetailsize ; $i++)
   	{
		$line = trim($hwdetails[$i]);
		if (($line =~ m/\s*CPUs\s*=/i) || ($line =~ m/^CPU Units:/i))
		{
			@cpudata = formatCPUdetails($i, $vendor, @hwdetails);
			$NoOfCPU = int(scalar(@cpudata) / CPU_SIZE);
        		$i += ($NoOfCPU + 1);
			next;
		}
	}

        #if @cpudata size is zero (failed to extract cpu info from prtdiag)
        if (scalar(@cpudata) == 0)
        {
            @result = getAlternateCPUresults($vendor);
            return @result;
        }

        # get actual number of CPUS(cores, not including hyperthreads)
        # which were found out from `kstat cpu_info`
        my @cpu_cnt = getCpuCount();
        $NoOfCPU = $cpu_cnt[0];

        #run the loop for total number of cores (truncating virtuals)
	for ($i = 0 ; $i < $NoOfCPU ; $i++)
	{
		$result[$i]="em_result=";
		for ($j=0; $j<5;$j++)
		{
			my $p;
			$k = $j + ($i * 6); 
			($p,$cpudata[$k])= split(':',$cpudata[$k],2);
                        $result[$i]="$result[$i]$cpudata[$k]|";
		}
		my $p;
		$k = $j + ($i * 6);
		($p,$cpudata[$k])= split(':',$cpudata[$k],2);
		$result[$i]="$result[$i]$cpudata[$k]";		
		$result[$i]="$result[$i]\n";
	}
    return @result;	
}
#
# getNumIOCards
# returns the number of IO cards present on a box
#
sub getIOCardCount
{
    my @iobuf  = ();
    my $n = 0;

    my $NoOfCards = 0;

	my @res = getIOCardResults();
	$NoOfCards = scalar(@res);
    return $NoOfCards;
}
sub getHardwareVendor
{

    # Purpose: To get hardware vendor

    my $vendor = NIL;
    my $cmd     = NIL;
    my $value   = NIL;
    my @info    = ();
    my $rc = checkCmd($hostOSD::UNAME);
    if ($rc == 1)
    {
      $cmd = "$hostOSD::UNAME";
    }
    else
    {
      return $vendor;
    }

    $value = execCmd($cmd);

    if (index($value, 'SunOS') > -1)
    {
        $vendor = "Sun Microsystems";
    }

    return $vendor;
}

sub getHardwarePlatform
{
    # Purpose: To get hardware platform

    my $cmd     = NIL;
    my $value   = NIL;
    my $rc = checkCmd($hostOSD::UNAME);
    if ($rc == 1)
    {
      $cmd = "$hostOSD::UNAME -m";
    }
    else
    {
      return $value;
    }
    $value = execCmd($cmd);

    return $value;
}

sub getHardwareName
{
    # Purpose: get the hardware model

    my $cmd     = NIL;
    my $value   = NIL;
    my $rc      = NIL;
    my $hwPlat  = NIL;

    $hwPlat = getHardwarePlatform();
    if ($hwPlat ne NIL)
    {
      if ($hwPlat =~ /sun4[uv]/)
      {
        $cmd = $hostOSD::PRTCONF;
        $rc = checkCmd($cmd);
        if ($rc == 1)
        {
          my $target = "banner-name:";
          my $result = execCmd($cmd . " -b 2>/dev/null");
          my @lines = split ("\n", $result);
          for(my $i = 0; @lines; $i++)
          {
            my $line = shift @lines;
            if ($line =~ m/$target/)
            {
              my $indx = index($line, $target) + length($target);
              if ($indx < length($line))
              {
                $value = trim(substr $line, $indx);
                last;
              }
            }
          }
        }
      }
    }
    if ($value eq NIL)
    {
      $value = getSmbiosData("1","Product: ");
    }

    if ($value ne NIL)
    {
      return $value;
    }

    $rc = checkCmd($hostOSD::UNAME);
    if ($rc == 1)
    {
      $cmd = "$hostOSD::UNAME -i";
    }
    else
    {
      return $value;
    }
    $value = execCmd($cmd);

    return $value;
}

sub getGetConfig($)
{

    # Purpose: To get value for given system configuration setting

    my ($parm)  = @_;
    my $cmd     = NIL;
    my $value   = NIL;

    my $rc = checkCmd($hostOSD::GETCONF);
    if ($rc == 1)
    {
      $cmd = "$hostOSD::GETCONF $parm";
    }
    else
    {
      return $value;
    }

    $value = execCmd($cmd);

    return $value;
}
sub getInstructionSetList
{

    # Purpose: To get the set of instruction lists that can be run Sun SPARC machines

    my $cmd     = NIL;
    my $value   = NIL;
    my $rc = checkCmd($hostOSD::UNAME);

    if ($rc == 1)
    {
      $cmd = "$hostOSD::UNAME -m";
    }
    else
    {
      return $value;
    }
    $value = execCmd($cmd);

    return $value;
}

sub getProcessorType
{
    # Purpose: To get processor type

    my $datadir = $hostOSD::DATALOC;
    my $cmd     = '';
    my $value   = '';
    if ($datadir eq '')
    {
        my $rc = checkCmd($hostOSD::MACH);
        if ($rc == 1)
        {
            $cmd = "$hostOSD::MACH";
        }
        else
        {
            return $value;
        }
    }
    else
    {
        $cmd = catCmd('mach.out');
    }
    $value = execCmd($cmd);

    return $value;

}
#
# getArchType
# get archeticture type
# on Linux we append machine/processor and arch.
#

sub getArchType
{
    # Purpose: To get  architecture type

    my $mach = getInstructionSetList();    # Added on 01/07/2002

    if (index($mach, 'sparcv9') > NOT_FOUND)
    {
        $mach = '64-bit sparcv9';
    }
    else
    {
        $mach = getProcessorType();
    }

    $mach = trim($mach);

    my $cmd     = NIL;
    my $value  = NIL;
    my $arch   = NIL;
    my $rc = checkCmd($hostOSD::ARCH);
    if ($rc == 1)
    {
      $cmd = "$hostOSD::ARCH -k";
      $value = execCmd($cmd);
    }

    $arch =  trim($value);

    $arch = join (' ', $mach, $arch);

    return $arch;
}

sub getDomainName($)
{
    # Purpose: To get domain name for host passed in

    my $hostname = $_[0];
    my $host = left(DOT_CHR,$hostname);
    my $value = NIL;
    my $domain = NIL;

    my @admnt = split (' ', getAdminEntries("hosts $host"));
    foreach $value (@admnt)
    {
        if ($value =~ m/^$host[.]/)
        {
            $domain = right(DOT_CHR, $value);
            last;
        }
    }

    if ($domain ne NIL)
    {
      return $domain;
    }
    else
    {
      my $cmd     = NIL;
      my $value   = NIL;
      my @domarr  = ();

      #Changed to fix the bug 2226505

      my $rc = checkCmd($hostOSD::NSLOOKUP);
      if ($rc == 1)
      {
        $cmd = "$hostOSD::NSLOOKUP $hostname 2>&1";
      }
      else
      {
        return $domain;
      }

      #my $dom =`$cmd 2>$hostOSD::DEVNULL`;
      $value = evalCmd($cmd);
      my @dom = split ("\n", $value);
      @domarr = grep /$hostname/, @dom;
      chomp(@domarr);

      foreach $value (@admnt)
        {
            if ($value =~ m/^Name:\s*$host[.]/)
            {
                $domain = right(DOT_CHR, $value);
                last;
            }
        }
    }
    return $domain;
}
#
# CPU Clock speed
#
sub getClockSpeed
{
   my $clock = NIL;

   if (!(($clock =~ m/^\d+$/) || ($clock =~ m/^\d+[.]\d*$/)))
    {
        $clock = getGetConfig("CLK_TCK");
    }

   return $clock;
}
#
# Total Memory on machine
#
sub getTotalMemory
{
    my $memory = NIL;
    my $extra  = NIL;
    my @hwdetails  = getHardwareDetails();
    my $hwdetailsize = scalar(@hwdetails);
    my $i;
    my $line;
	for ($i = 0 ; $i < $hwdetailsize ; $i++)
	{
		$line = trim($hwdetails[$i]);
		if ($line =~ /Memory\ssize/)
		{
			my @col = split /\s+/,$line;
                        ( $memory ) = ( $col[2] =~ /(\d+)/ );
                        if ( $col[2] =~ m/GB/ )
                        {
                            $memory = $memory * 1024;
                        }
			return $memory;
		}
	}	
    return $memory;
}
#
# getIOCardResults
# this routine returns the IO Card em_results
# returns potentially multiple em_result=VENDOR_NAME|NAME|FREQ_IN_MHZ|REVISION 
#
sub getIOCardResults
{
    my $id        = NIL;
    my $value     = NIL;
    my $numbr     = 0;
    my $size      = 0;

    my $result;
    my $NoOfCards = 0;

    my $vendor_id    = NIL;
    my $freq_MHz     = 0;
    my $name         = NIL;
    my $revision = NIL;
    my $n=0;

    my @iobuf  = ();
    my @iodat1 = ();

    my @ioc_result = ();

	my @ioddata = ();
	my $NoOfCards = 0;
	my $IOCardSection ;
    my $line;
    my $i;
	my $j;
	my $k;
    my $GenericSection;
    my @hwdetails  = getHardwareDetails();

    my $hwdetailsize = scalar(@hwdetails);
    for ($i = 0 ; $i < $hwdetailsize ; $i++)
    {
        $line = trim($hwdetails[$i]);
        if (($line =~ m/=\s*IO Cards\s*=/) || ($line =~ m/=\s*IO Devices\s*=/))
        {
	    	$IOCardSection  = 1;
            $GenericSection = 0;
            $i += 1;
            @ioddata = formatIOdetails($i, @hwdetails);
            $NoOfCards = int(scalar(@ioddata) / IOCARD_SIZE);
            $i += ($NoOfCards + 1);
            next;
        }
    }
    for ($i = 0 ; $i < $NoOfCards; $i++)
    {
		$ioc_result[$i]="em_result=";
		for ($j=0; $j<4;$j++)
		{
			my $p;
			$k = $j + ($i * 5);
			($p, $ioddata[$k]) = split(':', $ioddata[$k],2);
			$ioc_result[$i]="$ioc_result[$i]$ioddata[$k]|";
		}
		my $p;
		$k = $j + ($i * 5);
		($p, $ioddata[$k]) = split(':', $ioddata[$k],2);
		$ioc_result[$i]="$ioc_result[$i]$ioddata[$k]\n";
    }		
    return @ioc_result;
}
#
# getNICResults
# this routine returns an array of em_results for the 
# NIC CARD Metric
#
#   returns potentially multiple 
# em_result=NAME|FLAGS|MAX_TRANSFER_UNIT|INET_ADDRESS|MASK|BROADCAST_ADDRESS|MAC_ADDRESS|HOSTNAME_ALIASES|DESCRIPTION|DEFAULT_GATEWAY|DHCP_ENABLED
#
#
sub getNICResults
{

    my $bcad  = NIL;
    my $inet  = NIL;
    my $mtu   = NIL;
    my $index = NIL;
    my $mac   = NIL;
    my $flags = NIL;
    my $mask  = NIL;
    my @nicdetail = ();
    my @results = ();
    my $nic1  = NIL;
    my $i     = 0;
    my $j     = 0;
    my $extra1 = NIL;
    my $extra2 = NIL;
    my $extra3 = NIL;

    my @nic = getNICDetails();
    my $nic1 = NIL;

    my $NoOfNICs   = 0;
    $NoOfNICs = scalar(@nic)/2;

    for ($i = 0 ; $i < $NoOfNICs*2 ; $i+=2)
    {

	my $patt = ": ";

	my @col = split $patt, $nic[$i];

# In this loop, we assume that each interface has *exactly* 2 lines of output
# But there some cases, like interfaces configured in groups, whose
# information would be displayed on a separate line, and we need to ignore it
# So, if the pattern doesn't match, then skip the line and restart the loop
# Since we have to skip exactly *one* line, the combination of '$i--' and 'next'
# (If there's no interface name, col[0] begins with space)
        if ($col[0] =~ m/^\s/) 
        {
                $i--;
                next;
        }

	$nic1 = $col[0];
	($extra1,$flags,$extra2) = split ('[<>]', $col[1],3);	

        ($extra1,$mtu,$extra3,$index) = split (' ', $extra2,4);

        my @flagsArray = split (',',$flags);
        @flagsArray = sort(@flagsArray);

        my $Numflag = scalar(@flagsArray);
        for ($j=1,$flags=''; $j<$Numflag;$j++)
        {
            $flags="$flags$flagsArray[$j-1],";
        }
        $flags="$flags$flagsArray[$j-1]";

        chomp(@nicdetail = `/sbin/ifconfig $nic1`);
        $nicdetail[1]=trim($nicdetail[1]);
        my @detail = split (' ', $nicdetail[1]);

        my $elements = scalar(@detail);

        for ($j=0; $j < $elements; $j+=2)
        {
            if($detail[$j] eq "inet")
            {
                $inet = $detail[$j+1];
            }
            elsif($detail[$j] eq "netmask")
            {
                $mask = $detail[$j+1];
            }
            elsif($detail[$j] eq "broadcast")
            {
                $bcad =  $detail[$j+1];
            }
        }

        my $aliases = NIL;
        $aliases = getAdminEntries("hosts $inet") if ($inet =~ m/\./);
        my @hosts     = split (' ', $aliases);
        my @hostnames = ();
        my $m = scalar(@hosts);
        my $n = 0;
        my $l = 0;
        for ($l = 0 ; $l < $m ; $l++)
        {
            if ($hosts[$l] ne $inet) 
            {
                $hostnames[$n] = $hosts[$l];
                $n++;
            }
        }
        $aliases = join (',', @hostnames);


        # Calculate MAC-Address
        $mac = getMACAddress($inet) if ($inet =~ m/\./);
        if (index($mac, "no entry") > -1 || index($mac, "unknown host") > -1 || $mac eq '')
        {
            $mac = NIL;
        }
        else
        {
            my @mac1 = ();
            my @mac2 = ();
            @mac1 = split (" at ", $mac);
            @mac2 = split (' ',    $mac1[1]);
            $mac = $mac2[0];
        }

        $results[$i/2] = "em_result=$nic1|$flags|$mtu|$inet|$mask|$bcad|$mac|$aliases|||\n";
    }
    return (@results);

}

sub getNICDetails
{

    # Purpose: To get network interface details

    my $cmd     = NIL;
    my $value   = NIL;
    my @nic     = ();
    my $rc = checkCmd($hostOSD::IFCONFIG) & checkCmd($hostOSD::EGREP);


    if ($rc == 1)
    {
      $cmd = "$hostOSD::IFCONFIG -a | $hostOSD::EGREP -v zone";
    }
    else
    {
      return @nic;
    }

    $value = execCmd($cmd);
    @nic = split ("\n", $value);
    chomp(@nic);

    return @nic;
}
sub getlinuxOsName($)
{
  my $o_name = NIL;
  my ($file_name)  = @_;

  open(RS, "$file_name");
        while (<RS>)
        {
            if ($_ =~ m/^\s*\n/) { next; }
            else
            {
              $o_name = $_;
              last;
            }
        }
  close(RS);

  return $o_name;
}
#
# getOSName - returns OS system name
#
sub getOSNameVendorDist
{
   my $os_name = NIL;
#  Hardcoding Vendor Name for all Solaris ports
   my $vendor = 'Oracle USA';
   my $dist = NOT_APPLICABLE;
   my $value   = NIL;
   my $rc = checkCmd($hostOSD::UNAME);

   if ($rc == 1)
   {
     $value = "$hostOSD::UNAME";
     $os_name = execCmd($value);
   }
   return ($os_name,$vendor,$dist);

}
#
# getOSRelease
# returns baseversion-updatelevel
#
sub getOSRelease
{
    # Purpose: To get OS Release

    my $cmd     = NIL;
    my $value   = NIL;
    my $base    = NIL;
	my $update_level = NIL;
    my $rc = checkCmd($hostOSD::UNAME);
    if ($rc == 1)
    {
      $cmd = "$hostOSD::UNAME -r";
    }
    else
    {
      return $value;
    }
    $base = execCmd($cmd);
	$update_level = execCmd("$hostOSD::UNAME -v");
	$value = "$base-$update_level";
    return $value;
}

sub getOSPlatformId
{
  # Purpose: To return internal Oracle Platform ID for this OS.
  # Parameters: OS NAME, OS VENDOR, ADDRESS_LENGHT (32-bit,64-bit)

  # Here is the mapping table for OS and address size to platform ID
  # each port needs to return the proper value
  #
  #PLATFORM_NAME                                BUGDB_PLATFORM_ID
  #--------------------------------------------------------------------
  #AIX 4.3 Based Systems (64-bit)               38
  #AIX Based Systems (32-bit)                   319
  #AIX5L Based Systems (64-bit)                 212
  #Apple Mac OS                                 421
  #Bull DPX/20 AIX                              727
  #Data General AViiON ix86 Unix                612
  #Data General AViiON M88K DG                  162
  #Fujitsu UXP/DS                               221
  #Generic Platform                             2000
  #Hitachi 3050/R Risc UNIX                     93
  #HP 3000 Series MPE/iX                        70
  #HP Alpha SEVMS (Secure)                      913
  #HP Alpha Windows NT                          601
  #HP OpenVMS Alpha                             89
  #HP Tru64 UNIX                                87
  #HP VAX OpenVMS                               1
  #HP-UX Itanium                                197
  #HP-UX PA-RISC (32-bit)                       2
  #HP-UX PA-RISC (64-bit)                       59
  #HP-UX PA-RISC 10.20                          2
  #IBM iSeries OS/400                           43
  #IBM NUMA-Q DYNIX/ptx                         198
  #IBM S/390 Based Linux                        211
  #IBM VM/CMS                                   3
  #IBM z/OS (OS/390)                            30
  #IBM zSeries Based Linux                      209
  #Linux Intel (64-bit)    110
  #Linux Itanium                                214
  #Linux x86                                    46
  #Linux x86-64 (AMD64/EM64T)                   226
  #Microsoft Windows (32-bit)                   912
  #Microsoft Windows (32-bit) Client            176
  #Microsoft Windows (64-bit)                   208
  #Microsoft Windows 2000                       100
  #Microsoft Windows 98                         50
  #Microsoft Windows Server 2003                215
  #Microsoft Windows XP                         207
  #Microsoft Windows XP (64-bit)                206
  #NCR Unix SVR4                                358
  #NEC EWS4800/UP4800 Unix                      476
  #NLS Generic Platform                         1234
  #NLS Merged Translations                      2234
  #NLS Pseudo Translation                       3234
  #Novell NetWare                               21
  #SCO UNIX                                     28
  #SGI IRIX (32-bit)                            168
  #SGI IRIX (64-bit)                            24
  #Siemens BS2000/OSD                           361
  #Siemens MPP RM1000 Unix                      611
  #Siemens RM200-RM600                          724
  #Siemens/Nile/MIServer DC/OSX                 116
  #Solaris Operating System (SPARC 32-bit)      453
  #Solaris Operating System (SPARC 64-bit)      23
  #Solaris Operating System (x86)               173
  #Solaris Operating System (x86_64)            267
  #Stratus i860 VOS                             477
  #Stratus PA-RISC VOS                          619
  #UnitedLinux (32-bit)                         217
  #UnixWare                                     501

  my $os_name = $_[0];
  my $os_vendor = $_[1];
  my $address_length = $_[2];
  my $id = NIL;
  my $cmd     = '';
  my $value     = '';
  my $hw_vendor = getHardwareVendor();

  my $hardware_platform = getHardwarePlatform();

  # glavash uppercase check for linux, SuSE has it as LINUX, redhat Linux...
  # if the vendor is Red Hat this is linux, could be Centos, Fedora, but
  # they all fall into the linux bucket
  if (uc($os_name) =~ m/LINUX/ || $os_vendor =~ /Red Hat/)
  {
    if ($hardware_platform =~ m/86/)
      {
        if ($address_length =~ '64-bit')
          {
            if ($hw_vendor =~ m/Intel/)
            {
              #Linux Intel (64-bit)    110
              $id = 110;
            }
            else #x86 platform and 64 bit.  Not sure about Itanium
            {
              #Linux x86-64 (AMD64/EM64T)  226
              $id = 226;
            }
          }
       else # not 64 bit and x86
         {
           #Linux x86     46
           $id = 46;
         }
      }
   else # Linux and not 86 (Itanium?)
   {
     #Linux Itanium  214
     $id = 214;
   }
  }
  elsif ($os_name eq "SunOS")
  {
      # Default Value Solaris 64-bit
      $id = 23;

      my $rc = checkCmd($hostOSD::UNAME);
      if ($rc == 1)
      {
          $cmd = "$hostOSD::UNAME -m";
          $value = execCmd($cmd);
          if ($value =~ m/i86pc/)
          {
	    if ($address_length =~ '64-bit')
            {
                #Solaris Operating System (x86_64)
                $id = 267;
            }
            else
            { 
                #Solaris Operating System (x86)
                $id = 173;
            }
          }
          else
          {
             my $rc = checkCmd($hostOSD::ISALIST);
             if ($rc == 1)
             {
               $cmd = "$hostOSD::ISALIST";
               $value = execCmd($cmd);
               if ($value =~ m/sparcv9/)
               {
                 #Solaris 64-bit
                 $id = 23;
               }
               else
               {
                 #Solaris 32-bit
                 $id = 453;
               }
             }
          }
      }
  }
  else
  {
   $id = -999;
  }
  #add other platform checks here
  return $id;
}

sub getAddressLength
{
   my $value;
   my $mach;

   my $hardware = getHardwarePlatform();
   my $rc = checkCmd($hostOSD::ISAINFO);
   if ($rc == 1)
   {
       chomp($mach = `$hostOSD::ISAINFO -kv | $hostOSD::CUT -d'-' -f1`);
   }
   else
   {
       chomp($mach = `$hostOSD::GETCONF LONG_BIT`);
   }

   $value = "${mach}-bit";

   return $value;

}
#
# getDNSNameServersResults
# returns array of em_results in format em_result=SOURCE|NAME|VALUE
# for information on nameservers
#

sub getDNSNameServersResults($)
{
    my $option = $_[0];
    my $file  =  $hostOSD::ETCRESOLVCONFFILE;
    my $param;
    my $value;
    my @val   = ();
    my @results = ();
    my $filterIP = 1;
    my $i = 0;
    # Specifying the propertyname to search in propertyfile.
    my $propName = "ccr.metric.host.ecm_hw_nic.inet_address";
    # Specifying the propertyfilename for search.
    my $fileName = "collector.properties";
    # Getting the propertyvalue for specified propertyname here using getConfigPropertyValue in ocmprop.pm.
    my $propVal = ocmprop::getConfigPropertyValue($fileName , $propName);
    # Checking propertyvalue and accordingly setting filterIP.
    # Ignore case comparison of collector property value.

    if ( ($propVal ne NIL) && (lc($propVal) eq "false") )
       {
         $filterIP = 0;
       }

    if (($filterIP == 1) && ($file ne NIL) && (-e $file))
    {
        open(ES, "$file");
        while (<ES>)
        {
            if ($_ =~ m/^\s*#/) { next; }
            if ($_ =~ m/^\s*nameserver\s/)
            {
                @val = split (' ', $_, 2);
                $value = $val[1];
                $value = trim($value);
                $param = "nameserver $i";

                # return metric result or ASSOC result
                if ($option == $hostOSD::METRIC_RESULT)
                {
                  $results[$i++] = "em_result=$hostOSD::ETCRESOLVCONFFILE|$param|$value\n";
                }
                else
                {
                   $results[$i++] = "em_result=$ecmAssoc::CONNECTS_TO||$ecmAssoc::NAMESERVER_TYPE|$value\n"
                }
            }
        }
        close(ES);
    }
   return @results;
}

#
# getMountedFileSystemResults
# returns array of em_results with following format
# em_result=RESOURCE_NAME|MOUNT_LOCATION|TYPE|MOUNT_OPTIONS|DISK_SPACE_IN_GB|LOCAL_DRIVE|AVAIL_DISK_SPACE_IN_GB
#
sub getMountedFileSystemResults
{

    my @results = ();
    my $file   = getEtcMnttabLocation();

    if ($file eq NIL)
    {
        return @results;
    }

    my @att       = ();
    my @autofs    = ();
    my $automount = NIL;
    my $flag      = 'false';
    my $rest      = NIL;
    my $i         = 0;
    my $j         = 0;
    my $r         = 0;

    my $mount_opt;
    my $avail_disk_space = NIL; # never return a value 
    my $tot_disk_space = NIL;   # not supported on linux 
    my $local_drive = NIL;

    #get disk info
    my %diskCaps=getSortedDiskSizes();

    open(MNT, $file);
    LOOP: while (<MNT>)
    {
        # /dev/hda1 / ext3 rw 0 0
        # automount(pid1633) /home autofs rw,fd=5,pgrp=1633,minproto=2,maxproto=4 0 0
        # host1:/vol/home1/user /home/user nfs rw,tcp,intr,timeo=600,rsize=8192,wsize=8192,addr=0.0.0.0 0 0

        @att = split (' ', $_);
        $flag = doesContain($att[2]);

        if ($flag eq 'true')
        {
            foreach $automount (@autofs)
            {
                next LOOP if ($att[1] =~ m~^$automount/~);
            }

            ($att[3], $rest) = split (",dev=", $att[3], 2);
              if (!(index($att[3], "dev=") > -1))
              {
                $mount_opt =  $att[3];
              }
              else
              {
                $mount_opt = NIL;
              }
          
            # gnl only return disks that showed up as local disks
            # from the getDiskInfo command


            if ( exists  $diskCaps{ $att[0]} )
            { 
 
              # get disk space from diskCap hash
              $tot_disk_space = $diskCaps{$att[0]};
              $tot_disk_space  = sprintf("%.2f", $tot_disk_space / (1024 * 1024));

              # last 3 fields are null
             
              $results[$r++] = "em_result=$att[0]|$att[1]|$att[2]|$mount_opt|$tot_disk_space|$local_drive|$avail_disk_space\n";
            }
            $i++;
        }
        elsif ($att[2] eq 'autofs')
        {
            $autofs[$j++] = $att[1];
        }
    }
    close(MNT);

   return @results;
}
#******************************************
#   Non-Exported Subroutine
#******************************************
sub getProcModulesLocation
{
    # Purpose: To confirm the presence of /etc/modules

    my $value   = $hostOSD::PROCMODULESFILE;

    if (!(-e "$value"))
    {
        throwECMError('FileNotFound', $value);
        $value = NIL;
    }

    return $value;
}

sub getEtcMnttabLocation
{
    # Purpose: To confirm the presence of /etc/mnttab
    
    my $value   = $hostOSD::ETCMNTTABFILE;

    if (!(-e "$value"))
    {
        throwECMError('FileNotFound', $value);
        $value = NIL;
    }

    return $value;
}   


sub doesContain($)
{

    my @SUPPORTED_FS = ('nfs', 'ufs', 'vxfs', 'jfs', 'ext3', 'ext2','reiserfs');

    my ($str)  = @_;
    my $i      = 0;
    my $flag   = 'false';
    my $NoOfFS = scalar(@SUPPORTED_FS);

    for ($i = 0 ; $i < $NoOfFS ; $i++)
    {
        if ($str eq $SUPPORTED_FS[$i])
        {
            $flag = 'true';
            last;
        }

    }

    return $flag;
}
#
# getOSSoftwareComponentsResults
#
# Originally this was one routine in Ptdpm8, but to ease
# porting efforts I am going to split it into two routines.
#
# 
#
sub getOSSoftwareComponentsResults
{
	my @results = ();
	my $type= 'Package';
	my $pkgloc = "/var/sadm/pkg/";	
	opendir(DIR, $pkgloc)
		|| throwECMError('OpenFailure', $pkgloc);
	my @dlist;
	@dlist = readdir(DIR);
	closedir(DIR)
		|| throwECMError('CloseFailure', $pkgloc);
	my $size;
	$size = scalar(@dlist);
        my %hashset = ();
	my $i=0;
	my $pkginfofile;
	my $tz       = NIL;
	my $desc;
	my $instdate;
	my $version;
	my $vendor;
	my $pkg;
	my $rest=NIL;
	my $r=0;
	for ($i = 0 ; $i < $size ; $i++)
	{
		$tz = NIL;
		$desc = NIL;
		$instdate= NIL;
		$version= NIL;
		$vendor= NIL;
		$pkg = NIL;
		chomp($dlist[$i]);
		next if (($dlist[$i] eq '.') || ($dlist[$i] eq '..'));
	 	if (!(-d "$pkgloc/$dlist[$i]"))
		{
			next;
		}	
		$pkginfofile = "$pkgloc/$dlist[$i]/pkginfo";
		if (-e $pkginfofile)
		{
			open(IN1, "< $pkginfofile")
				|| throwECMError('OpenFailure', $pkginfofile);
			while (<IN1>)
			{
				chomp($_);
				if (m/PKG=/)
				{
					($rest, $pkg) = split ('=', $_, 2);
                                        $pkg =~ s/\|/\-/;
				}
				elsif (m/VERSION=/)
				{
					($rest,$version) = split ('=', $_,       2);
					($version, $rest)    = split (',', $version, 2);
				}
				elsif (m/DESC=/)
				{
					($rest, $desc) = split ('=', $_, 2);
				}
				elsif (m/TZ=/) 	
				{
					($rest, $tz) = split ('=', $_, 2);
				}
				elsif (m/INSTDATE=/)
				{
					($rest, $instdate) = split ('=', $_, 2);
				}
			}
			close(IN1)
			|| throwECMError('CloseFailure', $pkginfofile);
                        if ($version eq NIL)
                        {
                            $version = 'n/a';
                        } 
                        
                        if ($pkg eq NIL)
                        {
                            $pkg = 'unknown package';
                        }

			$instdate = trim($instdate);
			$instdate = convMon_DD_YYYY_To_YYYY_MM_DD2($instdate, $tz);

                        # Checking that all three values; pkgname,type and version are not null.
                        # Applying uniqueness condition on key values pkg,version and type.
                        if (exists $hashset{"$pkg|$type|$version"})
                        {
                        	next;
                        }
                        $hashset{"$pkg|$type|$version"} = $instdate;
                        $results[$r++] = "em_result=$pkg|$type|$version|$instdate\n";
 		}
	}
	return @results;
}

#
# getOSRegisteredSoftwareResults
#
# Originally this was one routine in Ptdpm8, but to ease
# porting efforts I am going to split it into two routines.
#
# 
#
sub getOSRegisteredSoftwareResults
{
    my @results = ();
    my @dlist      = ();

#                $results[$r++] = "em_result=$id|$prodname|$vendor|$version|$instdate|$installed_location|description|$vendor_sw_info\n";

    my $type= 'Package';
    my $pkgloc = "/var/sadm/pkg/";
    opendir(DIR, $pkgloc)
             || throwECMError('OpenFailure', $pkgloc);
    my @dlist;
    @dlist = readdir(DIR);
    closedir(DIR)
            || throwECMError('CloseFailure', $pkgloc);
    my $size;
    $size = scalar(@dlist);
    my $i=0;
    my $pkginfofile;
    my $tz       = NIL;
    my $desc;
    my $instdate;
    my $version;
    my $vendor;
    my $pkg;
    my $id= NIL;
    my $rest=NIL;
    my $location=NIL;
    my $r=0;
    my $vendor_sw_info=NIL;
    for ($i = 0 ; $i < $size ; $i++)
    {
            $tz = NIL;
            $desc = NIL;
            $instdate= NIL;
            $version= NIL;
            $vendor= NIL;
            $pkg = NIL;
	    $location = NIL;
            chomp($dlist[$i]);
            next if (($dlist[$i] eq '.') || ($dlist[$i] eq '..'));
            if (!(-d "$pkgloc/$dlist[$i]"))
            {
                    next;
            }
            $pkginfofile = "$pkgloc/$dlist[$i]/pkginfo";
            if (-e $pkginfofile)
            {
                    open(IN1, "< $pkginfofile")
                            || throwECMError('OpenFailure', $pkginfofile);
                    while (<IN1>)
                    {
                          chomp($_);
	                  if (m/VENDOR=/)
			  {
			    	($rest, $vendor) = split ('=', $_, 2);
                                $vendor =~ s/\|/\-/;
			  }
                          if (m/PKG=/)
                          {
				($rest, $pkg) = split ('=', $_, 2);
                                $pkg =~ s/\|/\-/;
                          }
                          elsif (m/VERSION=/)
                          {
                                ($rest,$version) = split ('=', $_,       2);
                                ($version, $rest)    = split (',', $version, 2);
                          }
                          elsif (m/DESC=/)
                          {
                                ($rest, $desc) = split ('=', $_, 2);
                          }
                          elsif (m/TZ=/)
                          {
                                ($rest, $tz) = split ('=', $_, 2);
                          }
	                  elsif (m/BASEDIR=/)
                	  {
                    		($rest, $location) = split ('=', $_, 2);
                	  }
                          elsif (m/INSTDATE=/)
                          {
                                ($rest, $instdate) = split ('=', $_, 2);
                          }
                     } 
                     close(IN1)
                     || throwECMError('CloseFailure', $pkginfofile);
                     if ($version eq NIL)
                     {
                         $version = 'n/a';
                     }
                     $instdate = trim($instdate);
                     $instdate = convMon_DD_YYYY_To_YYYY_MM_DD2($instdate, $tz);
                     # If the name is null, ignore the record
                     # Pass null for desc column it saves lot of space
                     if ($pkg ne NIL)
                     {
                         $results[$r++] = "em_result=$id|$pkg|$vendor|$version|$instdate|$location||$vendor_sw_info\n";
                     }
                }
        }
        return @results;

}

#
# getModulesResults
# This subroutine opens the /proc/modules files parses it and returns
# a result set in the format of
# em_result=name|size_in_bytes|referring_modules
#
# This is a LINUX specific routine I believe
#  returning NULL is acceptable for other platforms
#  if there is no suitable modules info
#
# On SunOS this subroutine use system command MODINFO and returns
# a result set in the format similar to Linux
sub getModulesResults
{

    my @results = ();
    my %resultshash;

    my $name = NIL;
    my $size = NIL;
    my $used = NIL;
    my $ref_mods = NIL;
    my $value = NIL;
    my $cmd   = NIL;
    my $i     = NIL;

    my $extra  = NIL;

    my @modinfo = ();
    my @modrow  = ();

    my $rc = checkCmd($hostOSD::MODINFO) & checkCmd($hostOSD::TAIL);

    if ($rc == 1)
    {
      $cmd = "$hostOSD::MODINFO | $hostOSD::TAIL +2";
    }
    else
    {
      return @results;
    }

    $value = execCmd($cmd);

    @modinfo = split ("\n", $value);
    chomp(@modinfo);

    for ($i = 0 ; $i < @modinfo ; $i++)
    {
       $modinfo[$i] = trim($modinfo[$i]);
       @modrow =  split ( /\s+/, $modinfo[$i]);
       $name = $modrow[5];
       $size = hex($modrow[2]);
       #Bug 7689756, to skip duplicate primary keys
       next if defined $resultshash{"$name"};  
       $resultshash{"$name"}="$size|$ref_mods";
    }
    while ( my ($key, $value) = each(%resultshash) ) {
        push(@results,"em_result=$key|$value\n");
    }
   return @results;
}

sub getRunlevel
{
    # Purpose: To return the current running level at which BOOT occured. 
    # This is an integer value that corresponds to single user, multi-user, 
    # etc run mode of the system.
                                                                                                            
    my $cmd     = NIL;
    my $value   = NIL;
    my @runlevel = ();

    my $rc = checkCmd($hostOSD::WHO);
    if ($rc == 1)
    {
      $cmd = "$hostOSD::WHO -r";
    }
    else
    {
      return $value;
    }
    $value            = execCmd($cmd);
    $value            = trim($value);
    @runlevel         = split( /\s+/, $value);
    $value            = $runlevel[2];
    # on Linux, RUNLEVEL returns two values, the previous run level and the
    # current. The value for previous can be 'N' indicating unknown. This is 
    # not of concern as we only want the current runLevel.
    #$value =~ s/^(N|\d+)\s*?(\d+)$/$2/;                                                 
    return $value;
}

sub getInitProcesses
{
    # Purpose: To return an array of objects that are the name of the init.d
    # entry and the run state. The run state can be 'on, off'. They are platform
    # specific, in other words, on Windows, this can be Manual, Started, Stopped.
    my @results = ();

    my $runLevel = getRunlevel();

    my @initList = `ls -r /etc/rc$runLevel.d/*`;

    my $row = NIL;
    my $count = 0;
    my @value = ();

    # Process the rowsets returned from the /etc/rc[0-6].d/ directory.

    foreach $row (@initList)
    {
        chomp $row;
        my @object = split( /\//,$row);
        my $key;
        my $dup = 0;
	if( $object[3] =~ m/^S/)
	{
		@value = split( /\d+/, $object[3]);
		push (@results, $value[1], "on");
	}
	elsif ( $object[3] =~ m/^K/ )
	{
                @value = split( /\d+/, $object[3]);
		foreach $key (@results)
		{
			if( $key == $value[1] )
			{
		       	    $dup = 1;
			}
		}
		if( $dup != 1 )
		{ 
		    push (@results, $value[1], "off");
		}
	}
    }
    return @results;
}

sub getMACAddress($)
{
    # Purpose: To get MAC address for a given inet address

    my ($inet)  = @_;
    my $cmd     = '';
    my $value   = '';

    my $rc = checkCmd($hostOSD::ARP);
    if ($rc == 1)
    {
        $cmd = "$hostOSD::ARP $inet 2>&1";
    }
    else
    {
        return $value;
    }

    $value = evalCmd($cmd);

    return $value;
}
sub getHostNameForIP
{
        my $server = shift(@_);
        my $temp=$server;
        $temp=~ s/\.//g;
        if ($temp =~ /^\d+$/)
        {
                my @numbers = split(/\./, $server);
                my $ip_number = pack("C4", @numbers);
                my ($name) = (gethostbyaddr($ip_number, 2))[0];
                if ($name)
                {
                        #print "The host is: $name";
                        return $name;
                }
                else
                {
                        #print "This IP has no name";
                }

                return $server;
        }
        else
        {
                #print "$string does not contain only digits.<BR>\n";
                return $server;
        }
}
1;
# End of Program
