#!/usr/local/bin/perl
# 
# $Header: emll/sysman/admin/scripts/ias/Apache_getMountPoints.pl /main/2 2010/10/07 08:33:20 asunar Exp $
#
# Apache_getMountPoints.pl
# 
# Copyright (c) 2004, 2010, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      Apache_getMountPoints.pl - Apache topology information: get 
#      mod_oc4j mount point information
#
#    DESCRIPTION
#      Command-line utility called by the Agent to retrieve a list of 
#      mod_oc4j mount points.  It outputs the following information about 
#      the mount points:
#
#      em_result={mount-point}|{dest}|{dest-type}|{OC4J-name}|{hostname}|{AJP-port}|{ias-name}|
#      {cluster-name}
#
#      destination-type can be one of the following:
#      LOCAL    - local cluster
#      AJP      - AJP 1.3
#      INSTANCE - an specific OC4J instance
#      CLUSTER  - an OC4J that is a member of the specified cluster
#
#      Command-line arguments: <oracle home>
#
#    NOTES
#
#    MODIFIED   (MM/DD/YY)
#    asunar       10/04/10 - bug 9859640:Setting CCR_OHS_ID in the environment
#    jsmoler      06/09/09 - grab fixes from 10.2.0.5 GC
#    jsmoler      07/14/06 - use parseApacheConf (bug 5224028)
#    jsmoler      05/19/05 - dynamically determine root conf file 
#    echolank     03/23/05 - ignore duplicate mountpoint/destination pairs 
#    echolank     02/01/05 - allow for multiple local destinations 
#    jsmoler      01/28/05 - apache -> Apache
#    klmichae     01/26/05 - allow for multiple AJP destinations 
#    klmichae     01/05/05 - fix OC4J name 
#    klmichae     12/27/04 - add destination to metric 
#    klmichae     12/22/04 - klmichae_oc4j_mount_info
#    klmichae     12/13/04 - Creation
# 

use strict;
use ias::Apache_confFileParser;
use ias::asecm;
require "emd_common.pl";

my $oracleHome = $ARGV[0];
setOracleHome($oracleHome);
my $ohsId = "";
if (-e "$oracleHome/config/OPMN/opmn/opmn.xml")
{
  # AS11: OHS instance ID has been passed as an argument
  $ohsId = $ARGV[1];
  # Set ohsId in the ENV. It will be used by asecm::parseApacheConf() later
  $ENV{'CCR_OHS_ID'} = $ohsId;
}
my $hash = parseApacheConf();

my $ref = $$hash{'Oc4jMount'};
if( !$ref )
{
    exit;
}
my @mountPoints = @{$ref};

# global variable to keep track of the (mountpoint, subdestination)
# pairs that have already been processed
my %addedMpointDests = ();

for my $mount ( @mountPoints )
{
    processMountDirective( $mount );
}

#
# Processes the specified mount point and outputs the following for 
# each destination:
#  em_result={mount-point}|{dest}|{dest-type}|{OC4J-name}|{hostname}|{AJP-port}|{ias-name}|{cluster-name}
#
# Arguments
#  0: the mount point directive
sub processMountDirective
{
    my $mountDirective = $_[0];

    # split apart the directive into the mount point and the destination
    my ( $mount, $dest );
    ( $mount, $dest ) = split( /[\s+]/, $mountDirective, 2 );

    # Removing leading a trailing whitespace from the mount point
    $mount =~ s/^\s*(.*?)\s*$/$1/;

    # No destination, home oc4j in local cluster
    if( !defined( $dest ) )
    {
        if (!isDuplicateMpointDest($mount, " ")) {
            print( "em_result=$mount| |LOCAL|home| || |local_ias_cluster\n");
        }
    } else {
        # remove embedded whitespace from the destination
        $dest =~ s/\s+//g;

        # Destination is one or more oc4js in local cluster:
        # {oc4j}[,{oc4j}]...
        if( index( $dest, ":" ) == -1 ) {
            processLocalDest( $mount, $dest );

        } else {
            # See what type of destination we have:
            # ajp13://{host}:{port}
            # cluster://{cluster}:{oc4j}[,{cluster}:{oc4j}]...
            # instance://[host]:{ias-instance}:{oc4j}[,[host]:{ias-instance}:{oc4j}]...
            # {cluster}:{oc4j}[,{cluster}:{oc4j}]...
            my ( $destType, $rest ) = split( "://", $dest );

            # Is this the default cluster destination?
            if( !defined( $rest ) )
            {
                processClusterDest( $mount, $dest );  
            } elsif( $destType eq "cluster" ) {
                processClusterDest( $mount, $rest );  
            } elsif( $destType eq "instance" ) {
                processInstanceDest( $mount, $rest );  
            } elsif( $destType eq "ajp13" ) {
                processAjpDest( $mount, $rest );  
            } else {
                # unknown destination
                EMD_PERL_ERROR( 
                    "Unrecognized mount point directive $mountDirective\n" );
            } 
        } # end if ( index( $dest, ":" ) ... 
    } # end if( !defined( $dest ...

}

# 
# process a local cluster destination
# The destination here is in the form:
#  {oc4j}[,{oc4j}]...
#
# Arguments
#  0: the name of the mount point
#  1: the destination value
#
sub processLocalDest
{
    my $mount = $_[0];
    my $dest = $_[1];

    # pull apart the list 
    my @destArray = split( ',', $dest );
    for my $eachDest ( @destArray )
    {
        if( defined( $eachDest ) )
        {
            if (!isDuplicateMpointDest($mount,$eachDest)) {
                print( "em_result=$mount|$eachDest|LOCAL|$eachDest| || |local_ias_cluster\n");
            }
        } else {
            EMD_PERL_ERROR( "Unrecognized local destination $eachDest\n" );
        }
    }
}

# 
# process a cluster mount point and output information about it
# The destination here is in the form:
#  {cluster}:{oc4j}[,{cluster}:{oc4j}]...
#
# Arguments
#  0: the name of the mount point
#  1: the destination value (without cluster://)
sub processClusterDest
{
    my $mount = $_[0];
    my $dest = $_[1];

    # pull apart the list 
    my @destArray = split( ',', $dest );
    for my $eachDest ( @destArray )
    {
        # take apart the cluster and oc4j name
        my ( $cluster, $oc4j ) = split( ":", $eachDest ); 
        if( defined( $cluster ) && defined( $oc4j ) )
        {
            if (!isDuplicateMpointDest($mount,"cluster://$eachDest")) {
                print( "em_result=$mount|cluster://$eachDest|CLUSTER|$oc4j| || |$cluster\n");
            }
        } else {
            EMD_PERL_ERROR( "Unrecognized cluster destination $eachDest\n" );
        }
    }
}

# 
# process an instance mount point and output information about it
# 
# The destination here is in the form:
#  [host]:{ias-instance}:{oc4j}[,[host]:{ias-instance}:{oc4j}]...
#
# Arguments
#  0: the name of the mount point
#  1: the destination value (without instance://)
sub processInstanceDest
{
    my $mount = $_[0];
    my $dest = $_[1];

    # pull apart the list 
    my @destArray = split( ',', $dest );
    for my $eachDest ( @destArray )
    {
        # take apart the host, ias instance and oc4j name
        my @pieces = split( ":", $eachDest ); 
        
        # if we have two pieces they are ias-instance and oc4j
        if( $#pieces == 1 ) 
        {
            if (!isDuplicateMpointDest($mount,"instance://$eachDest")) {
                print( "em_result=$mount|instance://$eachDest|INSTANCE|$pieces[1]| ||$pieces[0]| \n");
            }
            # if we have three pieces they are host, ias-instance and oc4j
        } elsif( $#pieces == 2 ) {
            if (!isDuplicateMpointDest($mount,"instance://$eachDest")) {
                print( "em_result=$mount|instance://$eachDest|INSTANCE|$pieces[2]|$pieces[0]||$pieces[1]| \n");
            }
        } else {
            # something is wrong with this directive
            EMD_PERL_ERROR( 
                "Unrecognized instance destination $eachDest\n" );
        } # end if( $#pieces == 1 ) 
    }
}

# 
# process an ajp13 mount point and output information about it
# 
# The destination here is in the form:
#  {host}:{port}[,{host}:{port}]...
#
# Arguments
#  0: the name of the mount point
#  1: the destination value (without ajp13://)
sub processAjpDest
{
    my $mount = $_[0];
    my $dest = $_[1];

    # pull apart the list 
    my @destArray = split( ',', $dest );
    for my $eachDest ( @destArray )
    {
        # parse into host:port
        my ($host, $port) = split( ":", $eachDest );
        if( defined( $host ) && defined( $port ) )
        {
            if (!isDuplicateMpointDest($mount,"ajp13://$eachDest")) {
                print( "em_result=$mount|ajp13://$eachDest|AJP| |$host|$port| | | \n" );
            }
        } else {
            EMD_PERL_ERROR( "Unrecognized ajp destination $eachDest\n" );
        }
    }
}

#
# Checks if a (mount point, subdestination) pair is a duplicate entry
# of one already processed.  (If so, it should be ignored, because it
# provides no new information, and because this metric is keyed by
# mount point and destination.)  If the pair does not represent
# a duplicate entry, adds the pair to addedMpointDests.
#
# Arguments
#  0: the name of the mount point 
#  1: a subdestination in a particular Oc4jMount directive.  For
#       example, if the directive is "Oc4jMount /foo instance://inst1,inst2",
#       then the subdestinations are "instance://inst1" and "instance://inst2"
#
# Returns:
#   true if the inputs represent a duplicate entry, false otherwise.
#
# Modifies:
#   addedMpointDests
#
sub isDuplicateMpointDest
{
    my $mount = $_[0];
    my $subdest = $_[1];

    # note that it is safe to use ' ' as delimiter,
    # because embedded whitespace has already been eliminated
    # from subdest.  subdest only contains space in the 
    # case that the destination is blank (local cluster 'home' case)
    my $currentEntry = $addedMpointDests{"$mount $subdest"};
    if (!defined $currentEntry) {
        $addedMpointDests{ "$mount $subdest" } = '1';
    }

    return (defined $currentEntry);
}
