#!/usr/local/bin/perl
# 
# $Header: IAS_getFarmInfo.pl 08-feb-2007.11:06:56 jsutton Exp $
#
# IAS_getFarmInfo.pl
# 
# Copyright (c) 2006, 2007, Oracle. All rights reserved.  
#
#    NAME
#      IAS_getFarmInfo.pl - Utility script used to get Farms and CLuster info
#
#    DESCRIPTION
#      Takes oracleHome as argument and returns a list (iasVersion, farmName, clusterName)
#
#    NOTES
#      <other useful comments, qualifications, etc.>
#
#    MODIFIED   (MM/DD/YY)
#    jsutton     02/08/07 - XbranchMerge aime_pre904_farms from st_emll_10.2.5
#    pparida     02/06/07 - Fix bug 5858010: Use backticks instead of system to
#                           prevent sending undesirable output to targets.xml
#    jsutton     01/12/07 - Check if opmn running before discovery, stop it
#                           (when done) if not
#    pparida     09/28/06 - Fix bug 5572327: For 10.1.2 file-based, use Hosting Instance as farmName
#                           instead of the 'Farm Name' returned by dcmctl
#    pparida     06/23/06 - Creation (Bug 5293161: Add support for iAS farms 
#                           and clusters) 
# 

use ias::simpleXPath;

sub getIASFarmInfo
{
  my @farmInfo;
  my ($oracleHome) = @_;
  chomp($oracleHome);
  my $ver;
  my $farmName = "";
  my $clusterName = "";
  my $iasVersion = getPropertyFromFile($oracleHome, "/config/ias.properties", "Version");

  my ($majorVer,$firstSub,$secondSub) =  split(/\./ , $iasVersion);
  
  if(($majorVer > 10) ||
    (($majorVer == 10) && ($firstSub > 1)) ||
    (($majorVer == 10) && ($firstSub == 1) && ($secondSub > 3))) 
  {
    $ver = "post1013";
  }

  if(($majorVer == 10) && ($firstSub == 1) && ($secondSub == 3))
  {
    $ver = "1013";
  }

  if(($majorVer < 10) ||
    (($majorVer == 10) && ($firstSub < 1)) ||
    (($majorVer == 10) && ($firstSub == 1) && ($secondSub < 3))) 
  {
    $ver = "pre1013";
  }

  # Now depending on the version, do the appropriate discovery
 
  if($ver eq "pre1013")
  {
    my $isDB_based = getPropertyFromFile($oracleHome, "/config/ias.properties", "InfrastructureUse");

    # Pre 10.1.3 iAS with DB-based repository
    if($isDB_based eq "true")
    {
      # DB-based farm repository; hence use the repository name as the farmName and then find cluster if present
      $farmName = getPropertyFromFile($oracleHome, "/config/ias.properties", "InfrastructureDBCommonName");
      $clusterName = getClusterName($oracleHome, $farmName);
    }

    # Pre 10.1.3 iAS with File-based repository OR standAlone instance
    # post-903 only (opmnctl status, dcmdtl whichfarm not supported)
    if(($isDB_based eq "false") && 
       (($majorVer == 10) || 
        (($majorVer == 9) && ($firstSub == 0) && ($secondSub > 3))))
    {
      my @opmnVal = `$oracleHome/opmn/bin/opmnctl ping 1`;
      my $opmnUp = $?; 
      # File-based repository; hence OK to use dcmctl to get the farmName and then find cluster if present
      my @retVal = `$oracleHome/dcm/bin/dcmctl whichfarm`;
      if ($opmnUp != 0)
      {
        `$oracleHome/opmn/bin/opmnctl stopall`;
      }
      foreach $line (@retVal)
      {
        # We do not use the 'Farm Name' returned by 'dcmctl whichfarm' beacause it is contructed out of 
        # a path eg, ".scratch.pparida.product.10.1.2.OraHome_1.dcm.repository". Such a name is not
        # guaranteed to be unique (two different farms on two different machines could possibly have the
        # same path structure resulting in the same 'Farm Name'). Moreover, this name is also not guaranteed
        # to be within 256 chars in length (limit of agent_target_name is 256).
        #
        # But we can use the 'Host Instance' filed returned by 'dcmctl whichfarm as the farmName because 
        # of the following reasons taken together:
        #
        # 1. Each member of the farm returns the same Host Instance (the member instance where the
        #    file-based repository for that farm is hosted).
        # 2. An iAS instance name is unique in structure.
        # 3. No two farms can host their file repositories on the same instance (since the host has to 
        #    be a member of that farm and two farms cannot have a common member).
        #
        if($line =~ /^Host Instance: (.*)$/)
        {
          $farmName = $1;
        }
      }
      $clusterName = getClusterName($oracleHome, $farmName);
    }
  }

  if($ver eq "1013")
  {
    my $opmnFile = $oracleHome . "/opmn/conf/opmn.xml";
    # Find the value of the multicast or hub address, which is the the value of the attribute 
    # "list" of the element opmn/notification-server/topology/discover in opmn.xml
    # After getting rid of the initial *, if any, this value will act as the farmName
    if(-e $opmnFile)
    {
      my @subTargets = simpleXPathQuery($opmnFile, 'opmn/notification-server/topology/discover');
      my $hashRef;
      foreach $hashRef (@subTargets)
      {
        $farmName = $$hashRef{'list'};
      }
      # Get rid of the * in the beginning in case it is a multicast address
      $farmName =~ s/^\*//;
    }
  }

  push(@farmInfo, $iasVersion);
  push(@farmInfo, $farmName);
  push(@farmInfo, $clusterName);
  return @farmInfo;
}

sub getClusterName
{
   my ($oh, $fName) = @_;
   my $clusterName;
   my $instanceID;
   my $dcmconfFile = "$oh/dcm/config/dcm.conf";
   # We dont have to invoke OPTIC; Instead, clusterName can be obtained from dcm.conf file
   # We can get the clusterName from the dcm.conf and decide if it is an individual instance or a real cluster
   if(!-e $dcmconfFile || $fName eq "")
   {
     return "";
   }
   open (DCMCONF, $dcmconfFile) || die "Could not open dcm.conf file!\n";
   while ($line = <DCMCONF>)
   {
     if($line =~ /^cluster_name=(.*)$/)
     {
       $clusterName = $1;
     }
     if($line =~ /^instance_id=(.*)$/)
     {
       $instanceID = $1;
     }
   }
   close (DCMCONF);

   # Print cluster target only if it is not an individual instance
   if($clusterName ne "C_".$instanceID)
   {
     return $fName."_".$clusterName;
   }
   else
   {
     return "";
   }
}

sub getPropertyFromFile
{
  my ($emdHomeRef, $fileSpec, $property, $delim) = @_;

  my $propertyValue = "";
  my $propertiesFile = $emdHomeRef . $fileSpec;
  my ($propertiesLine, @propInfo);

  if (!defined($delim))
  {
    $delim = "=";
  }
  if (-e $propertiesFile)
  {
    open (PROPERTIES_FILE_READER, $propertiesFile);
    while($propertiesLine = <PROPERTIES_FILE_READER>)
    {
      chomp($propertiesLine);
      if (index($propertiesLine, "$property") eq 0)
      {
        @propInfo = split(/$delim/ ,$propertiesLine);
        if ($propInfo[0] eq $property)
        {
          $propertyValue = $propInfo[1];
          last;
        }
      }
    }
    close PROPERTIES_FILE_READER;
  }
  return $propertyValue;
}

1;
