# $Header: emll/sysman/admin/scripts/ias/apache_assoc.pl /main/7 2012/10/03 15:57:41 fmorshed Exp $
#
# apache_assoc.pl
#
# Copyright (c) 2005, 2012, Oracle and/or its affiliates. All rights reserved. 
#
#   NAME
#       apache_assoc.pl
#
#   DESCRIPTION
#       Returns information on targets associated with this apache instance
#
#       Command-line arguments: <oracle home> <ohs instance id (AS11 onwards)>
#
#   List of standard assocation types are in
#   ecmAssoc.pm
#
#   returns potentially multiple 
# em_result=association_type|target_name|target_type|target_properties
#
# association_type: one of the types listed in ecmAssoc.pm
# target_name: the target_name of the associated target
# target_type: the target_type of the associated target
# target_properties: related information that may help determine true target name given the target_type
#
#   NOTES
#
#   MODIFIED     (MM/DD/YY)
#      fmorshed   10/02/12 - Remove / as first character to query targets.xml
#      fmorshed   01/19/12 - In advent of central collector, change query of
#                 targets.xml to account for the new hierarchy of tags.
#      aghanti    06/17/10 - Consider OHS instance id for AS11
#      jsmoler    06/03/09 - use parseApacheConf (bug 5224028)
#      jsutton    07/26/07 - Pick up existing shared OH work
#      jsutton    07/25/07 - Shared home work
#      pparida    05/05/06 - Fix bug 5206272 
#      jsutton    09/16/05 - jsutton_ias_assoc
#      jsutton    09/15/05 - initial
##*************************************************************

use strict;
use ias::Apache_confFileParser;
use ias::asecm;
use ecmAssoc;
use ias::simpleXPath;
require "emd_common.pl";

my $oracleHome = $ARGV[0];

# Use ORACLE_CONFIG_HOME, if defined
if (defined ($ENV{ORACLE_CONFIG_HOME}))
{
  $oracleHome = $ENV{ORACLE_CONFIG_HOME};
}
setOracleHome($oracleHome);

my $ohsId = "";
if (-e "$oracleHome/config/OPMN/opmn/opmn.xml")
{
  # AS11: OHS instance ID has been passed as an argument
  $ohsId = $ARGV[1];
  # Set ohsId in the ENV. It will be used by asecm::parseApacheConf() later
  $ENV{'CCR_OHS_ID'} = $ohsId;
}

my $hash = parseApacheConf();

my $ref = $$hash{'Oc4jMount'};
if( !$ref )
{
  exit;
}
my @mountPoints = @{$ref};

# global variable to keep track of the (mountpoint, subdestination)
# pairs that have already been processed
my %addedMpointDests = ();

my @apacheAssoc;

for my $mount ( @mountPoints )
{
  processMountDirective( $mount );
}

foreach my $result (@apacheAssoc)
{
  print $result;
}


# Given the oc4j display name, gets the iasName from the ccr/config/default/targets.xml
sub get_iasName
{
  my $oc4j_display_name = $_[0];
  my $discoveredTargetsFile = $ENV{CCR_CONFIG_HOME} . "/config/default/targets.xml";
  # Since ias::simpleXpath does not implememt boolean expressions for <@attr-name=attr-value>, 
  # we first extract the nodes of target-type "oc4j" and then find the vale of NAME for which the
  # DISPLAY_NAME matches the argument to this function. The ias_name can be obtained by
  # stripping out the _oc4jdisplayname from the end of the NAME.
  my $ohQueryPrefix = 'TargetHomes/TargetHome[@LOCATION="' . $ENV{ORACLE_HOME} . '"]';
  my @subTargets = simpleXPathQuery($discoveredTargetsFile, $ohQueryPrefix . '/Targets/Target[@TYPE="oc4j"]');
  my $hashRef;
  my $ias_name = "";
  foreach $hashRef (@subTargets)
  {
    my $display_name = $$hashRef{'DISPLAY_NAME'};
    if($oc4j_display_name eq $display_name)
    {
      $ias_name = $$hashRef{'NAME'};
      last;
    }
  }
  my @ias_names = split(/_$oc4j_display_name$/, $ias_name);
  $ias_name = shift(@ias_names);
  return $ias_name;
}


#
# Processes the specified mount point and outputs the following for 
# each destination:
#  em_result={mount-point}|{dest}|{dest-type}|{OC4J-name}|{hostname}|{AJP-port}|{ias-name}|{cluster-name}
#
# Arguments
#  0: the mount point directive
sub processMountDirective
{
  my $mountDirective = $_[0];

  my ( $mount,$dest,$destType,$oc4jName,$hostname,$ajpPort,$iasName,$clusterName );

  # split apart the directive into the mount point and the destination
  ( $mount, $dest ) = split( /[\s+]/, $mountDirective, 2 );

  # Removing leading a trailing whitespace from the mount point
  $mount =~ s/^\s*(.*?)\s*$/$1/;

  $destType = "LOCAL";
  $clusterName = "local_ias_cluster";
  if( !defined( $dest ) )
  {
    $oc4jName = "home";
    $iasName = get_iasName($oc4jName);
    my $oc4jMountInfo = "$destType;$hostname;$clusterName;$iasName;$oc4jName;$ajpPort";
    if (!defined $addedMpointDests{"$oc4jMountInfo"})
    {
      $addedMpointDests{"$oc4jMountInfo"} = 1;
      push(@apacheAssoc, "em_result=$ecmAssoc::CONNECTS_TO||$ecmAssoc::OC4J_TYPE|$oc4jMountInfo\n");
    }
  } 
  else
  {
    # remove embedded whitespace from the destination
    $dest =~ s/\s+//g;

    # Destination is one or more oc4js in local cluster:
    # {oc4j}[,{oc4j}]...
    if( index( $dest, ":" ) == -1 ) 
    {
      my @destArray = split( ',', $dest );
      for my $eachDest ( @destArray )
      {
        if( defined( $eachDest ) )
        {
          $dest = $oc4jName = $eachDest;
          $iasName = get_iasName($oc4jName);
          my $oc4jMountInfo = "$destType;$hostname;$clusterName;$iasName;$oc4jName;$ajpPort";
          if (!defined $addedMpointDests{"$oc4jMountInfo"})
          {
            $addedMpointDests{"$oc4jMountInfo"} = 1;
            push(@apacheAssoc, "em_result=$ecmAssoc::CONNECTS_TO||$ecmAssoc::OC4J_TYPE|$oc4jMountInfo\n");
          }
        }
      }
    }
    else
    {
      # See what type of destination we have:
      # ajp13://{host}:{port}
      # cluster://{cluster}:{oc4j}[,{cluster}:{oc4j}]...
      # instance://[host]:{ias-instance}:{oc4j}[,[host]:{ias-instance}:{oc4j}]...
      # {cluster}:{oc4j}[,{cluster}:{oc4j}]...
      my ( $destType, $rest ) = split( "://", $dest );
      # Is this the default cluster destination?
      if( !defined( $rest ) )
      {
        processClusterDest($mount, $dest);
      }
      elsif( $destType eq "cluster" ) 
      {
        processClusterDest($mount, $rest);
      }
      elsif( $destType eq "instance" )
      {
        processInstanceDest( $mount, $rest );  
      }
      elsif( $destType eq "ajp13" ) 
      {
        processAjpDest( $mount, $rest );  
      } 
    }
  }
}

# 
# process a cluster mount point and output information about it
# The destination here is in the form:
#  {cluster}:{oc4j}[,{cluster}:{oc4j}]...
#
# Arguments
#  0: the name of the mount point
#  1: the destination value (without cluster://)
sub processClusterDest
{
  my $mount = $_[0];
  my $dest = $_[1];

  # pull apart the list 
  my @destArray = split( ',', $dest );
  for my $eachDest ( @destArray )
  {
    # take apart the cluster and oc4j name
    my ( $cluster, $oc4j ) = split( ":", $eachDest ); 
    if( defined( $cluster ) && defined( $oc4j ) )
    {
      $dest = "cluster://".$eachDest;
      my $destType = "CLUSTER";
      my $oc4jName = $oc4j;
      my $clusterName = $cluster;
      my $oc4jMountInfo = "$destType;;$clusterName;;$oc4jName;";
      if (!defined $addedMpointDests{"$oc4jMountInfo"})
      {
        $addedMpointDests{"$oc4jMountInfo"} = 1;
        push(@apacheAssoc, "em_result=$ecmAssoc::CONNECTS_TO||$ecmAssoc::OC4J_TYPE|$oc4jMountInfo\n");
      }
    }
  }
}

# 
# process an instance mount point and output information about it
# 
# The destination here is in the form:
#  [host]:{ias-instance}:{oc4j}[,[host]:{ias-instance}:{oc4j}]...
#
# Arguments
#  0: the name of the mount point
#  1: the destination value (without instance://)
sub processInstanceDest
{
  my $mount = $_[0];
  my $dest = $_[1];

  # pull apart the list 
  my @destArray = split( ',', $dest );
  my $oc4jMountInfo;
  for my $eachDest ( @destArray )
  {
    # take apart the host, ias instance and oc4j name
    my @pieces = split( ":", $eachDest ); 
        
    # if we have two pieces they are ias-instance and oc4j
    if( $#pieces == 1 ) 
    {
      $oc4jMountInfo = "INSTANCE;;;$pieces[0];$pieces[1];";
    } 
    # if we have three pieces they are host, ias-instance and oc4j
    elsif( $#pieces == 2 ) 
    {
      $oc4jMountInfo = "INSTANCE;$pieces[0];;$pieces[1];$pieces[2];";
    }
    if (!defined $addedMpointDests{"$oc4jMountInfo"})
    {
      $addedMpointDests{"$oc4jMountInfo"} = 1;
      push(@apacheAssoc, "em_result=$ecmAssoc::CONNECTS_TO||$ecmAssoc::OC4J_TYPE|$oc4jMountInfo\n");
    }
  }
}

# 
# process an ajp13 mount point and output information about it
# 
# The destination here is in the form:
#  {host}:{port}[,{host}:{port}]...
#
# Arguments
#  0: the name of the mount point
#  1: the destination value (without ajp13://)
sub processAjpDest
{
  my $mount = $_[0];
  my $dest = $_[1];

  # pull apart the list 
  my @destArray = split( ',', $dest );
  for my $eachDest ( @destArray )
  {
    # parse into host:port
    my ($host, $port) = split( ":", $eachDest );
    if( defined( $host ) && defined( $port ) )
    {
      my $oc4jMountInfo = "AJP;$host;;;;$port";
      if (!defined $addedMpointDests{"$oc4jMountInfo"})
      {
        $addedMpointDests{"$oc4jMountInfo"} = 1;
        push(@apacheAssoc, "em_result=$ecmAssoc::CONNECTS_TO||$ecmAssoc::OC4J_TYPE|$oc4jMountInfo\n");
      }
    }
  }
}

1;
# End of the Program

