# 
# $Header: emll/sysman/admin/scripts/ias/asecm.pm /main/11 2010/10/07 08:33:20 asunar Exp $
#
# asecm.pm
# 
# Copyright (c) 2004, 2010, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      asecm.pm - App Server ECM integration utilities
#
#    DESCRIPTION
#       Contains utility methods for App Server ECM integration.
#       You should call setOracleHome before calling any other functions.
#
#       Entry points:
#
#         General:
#
#           setOracleHome
#           getPlatform
#           statFile
#           statFiles
#           parseCommandLine
#           printResult
#           formatResult
#           setMetricFormat
#           printMetricResult
#           clearVisitedResultKeys
#           ensureSlash
#           contains
#
#         OC4J:
#
#           listOC4JInstances
#           listOC4JModules
#           listOC4JDeployedApps
#           listOC4JAppInfo
#           listOC4JWebModuleMappings
#           listOC4JDataSourceFiles
#           parseWebAppInfo
#
#         Web Cache:
#
#           getMyCache
#           getListenIP
#           getListenProtocol
#
#         OHS:
#
#           getApacheRootConfFile
#           getApacheDefines
#           getApacheModules
#           parseApacheConf
#           getApacheVHosts
#
#
#    NOTES
#      <other useful comments, qualifications, etc.>
#
#    MODIFIED   (MM/DD/YY)
#    asunar      09/24/10 - Setting the right paths for AS11
#    aghanti     06/18/10 - Consider OHS/WebCache instance ids for AS11
#    jsutton     03/26/10 - AS11 requirements
#    jsmoler     09/10/09 - check if webcache.xml exists
#    jsmoler     06/09/09 - grab fixes from 10.2.0.5 GC
#    jsmoler     12/28/06 - handle relative and absolute paths (bug 5736476),
#                           support OPMN variables
#    jsmoler     08/14/06 - fix comment typo
#    jsmoler     07/07/06 - call httpd for Apache compiled modules 
#                           (bug 5224028)
#    jsutton     03/29/07 - Pick up changes from EMAS
#    pparida     07/18/06 - Sync with 10.2.0.2 EMGC 
#    jsmoler     12/23/05 - support system app for 10.1.3 
#    echolank    01/03/06 - XbranchMerge echolank_1013_topo_support from main 
#    jsmoler     12/09/05 - Backport jsmoler_bug-4769271 from main 
#    jsmoler     11/01/05 - XbranchMerge jsmoler_bug-4613189 from main 
#    echolank    12/08/05 - add listOC4JWebModuleMappings, listOC4JAppInfo
#    jsmoler     12/05/05 - add listOC4JDataSourceFiles 
#    jsmoler     09/30/05 - handle \'s in paths for NT 
#    jsmoler     08/22/05 - change listOC4JDeployedApps to rollup from listOC4JModules
#    jsutton     11/15/05 - resync with 051008 GC MAIN 
#    jsutton     09/23/05 - Make sure opmn.xml exists before attempting parse
#    jsutton     09/13/05 - Align with changes in main line 
#    ndutko      08/03/05 - ndutko_code_checkin
#    jsmoler     08/03/05 - get rid of restatFiles 
#    jsmoler     05/19/05 - add function to determine root Apache conf file 
#                           from opmn.xml 
#    jsmoler     03/03/05 - varibale length strings, Apache vhost utility
#    jjshah      02/17/05 - Bug Fix 
#    jsmoler     02/11/05 - fix restatfiles 
#    jsmoler     02/04/05 - changing file path order
#    jsmoler     01/10/05 - handle undefined value for SSLENABLED attribute
#    jsmoler     12/16/04 - make protocol lowercase, add restatFiles
#    jsmoler     12/13/04 - add display path column for files 
#    jsmoler     11/12/04 - add raw-string datatype
#    jsmoler     10/22/04 - don't stat files that aren't found 
#    jsmoler     10/20/04 - update
#    jsmoler     10/15/04 - add 'path' type for results; fix command-line parser
#    jsmoler     10/04/04 - jsmoler_ecm01
#    jsmoler     08/18/04 - Creation
# 

BEGIN
{
    require "emd_common.pl";
}

package ias::asecm;

use strict;
use Exporter 'import';

our @EXPORT = qw(
    setOracleHome
    getPlatform
    statFile
    statFiles
    parseCommandLine
    printResult
    formatResult
    setMetricFormat
    printMetricResult
    clearVisitedResultKeys
    ensureSlash
    contains
    
    listOC4JInstances
    listOC4JModules
    listOC4JDeployedApps
    listOC4JAppInfo
    listOC4JWebModuleMappings
    listOC4JDataSourceFiles
    parseWebAppInfo
    
    getWebCacheXml
    getMyCache
    getListenIP
    getListenProtocol
    
    getApacheRootConfFile
    getApacheDefines
    getApacheVHosts
    getApacheModules
    parseApacheConf
);

# Default maximum length for strings in metric results
my $strLengthLimit = 200;

# Default maximum length for file paths in metric results
my $pathLengthLimit = 2000;

# Default maximum length for raw strings in metric results
my $rawLengthLimit = 2000;

# Oracle Home path to use, set by setOracleHome
my $oracleHome;

# as11 flag
my $isAS11 = 0;

#
# Sets the Oracle Home path.
#
# Arguments
#   0: The oracle home.
#
sub setOracleHome
{
    $oracleHome = $_[0];
}

#
# Prints a result formatted by formatResult, below.
# Requires that setOracleHome has already been called.
#
# Arguments: Same as formatResult.
#
sub printResult
{
    print formatResult(@_);
}

#
# Formats a metric-collection result so that the agent can use it.
# Requires that setOracleHome has already been called.
#
# Arguments: An even-length list of result values and types. Each result
#            value in the list must be followed by a type name, either 'string',
#            'path', or 'number'. String-type values are shortened to $strLengthLimit,
#            defined above; also, empty stings are replaced with a single space.
#            Path-type variables are shortened to $pathLengthLimit, defined above;
#            also, the any occurrences of the Oracle Home path
#            are replaced with '$ORACLE_HOME'. Number-type values have no special
#            processing.
#
# Returns: A formatted result. For example, on the input:
#               ( 'a', 'string', 3, 'number')
#          this function will return:
#               "em_result=a|3\n";
#
sub formatResult
{
    my $str;

    for (my $i = 0; $i < @_; $i += 2)
    {
        my $val = $_[$i];
        my $type = $_[$i + 1];
        if (!defined($val))
        {
            $val = '';
        }
        if (!defined($type))
        {
            die "No type for result formatting";
        }
        if ($type eq 'number')
        {
            # do nothing
        }
        elsif ($type eq 'string')
        {
            $val = formatStr($val, $strLengthLimit);
        }
        elsif ($type =~ /^string\(([0-9]+)\)$/)
        {
            my $lengthLimit = $1;
            $val = formatStr($val, $lengthLimit);
        }
        elsif ($type eq 'path')
        {
            $val = formatPath($val, $pathLengthLimit);
        }
        elsif ($type =~ /^path\(([0-9]+)\)$/)
        {
            my $lengthLimit = $1;
            $val = formatPath($val, $lengthLimit);
        }
        elsif ($type eq 'raw-string')
        {
            $val = formatRawStr($val, $rawLengthLimit);
        }
        elsif ($type =~ /^raw-string\(([0-9]+)\)$/)
        {
            my $lengthLimit = $1;
            $val = formatRawStr($val, $lengthLimit);
        }
        else
        {
            die "Unknown type for result formatting: '$type'";
        }
        $str = $str . '|' . $val;
    }
    
    return "em_result=" . substr($str, 1) . "\n";
}

#
# Formats a string for a metric-collection result. Utility method for
# formatResult. The string is shortened to $lengthLimit, defined above. Also,
# any occurrences of the Oracle Home path are replaced with $ORACLE_HOME.
# Requires that setOracleHome has already been called.
#
# Arguments
#   0: The string to format.
#
# Returns: The string, formatted.
#
sub formatStr
{
    my $str = $_[0];
    my $lengthLimit = $_[1];
    if (length($str) == 0)
    {
        # An empty string will be stored as a null in the repository.
        # Keys cannot be null, so output a space instead.
        return ' ';
    }
    return shortenStr($str, $lengthLimit);
}

sub formatPath
{
    my $str = $_[0];
    my $lengthLimit = $_[1];
    
    # Replace the Oracle Home path with '$ORACLE_HOME'
    my $oracleHome1 = $oracleHome;
    $oracleHome1 =~ s/\\/\\\\/g; # Escape \'s before matching
    if ($str =~ /^$oracleHome1(.*)/)
    {
        $str = '$ORACLE_HOME' . $1;
    }

    return shortenStr($str, $lengthLimit);
}

sub formatRawStr
{
    my $str = $_[0];
    my $lengthLimit = $_[1];
    if (length($str) > $lengthLimit)
    {
        die "String is too long: '$str'";
    }
    return $str;
}

#
# Formats a string for a metric-collection result. Utility method for
# formatResult. The string is shortened to $lengthLimit, defined above. Also,
# any occurrences of the Oracle Home path are replaced with $ORACLE_HOME.
# Requires that setOracleHome has already been called.
#
# Arguments
#   0: The string to format.
#
# Returns: The string, formatted.
#
sub shortenStr
{
    my $str = $_[0];
    my $lengthLimit = $_[1];

    if (length($str) > $lengthLimit)
    {
        # The string is too long. Shorten, and place '...' in the middle.
        my $a = int(($lengthLimit - 3) / 2);
        my $b = $lengthLimit - $a;
        $str = substr($str, 0, $a) . '...' . substr($str, length($str) - $b + 3);
    }
    
    return $str;
}

#
# Lists all OC4J instances in the current oracle home.
# Requires that setOracleHome has already been called.
#
# Returns: A list of names of OC4J instances in the current oracle home.
#
sub listOC4JInstances
{
    my $oracleHomeDir;
    my @results;
    
    opendir($oracleHomeDir, "$oracleHome/j2ee");
    for my $entry (readdir($oracleHomeDir))
    {
        if ($entry eq '.' || $entry eq '..')
        {
            next;
        }
        my $fullName = "$oracleHome/j2ee/$entry";
        if (-d $fullName &&
            -e "$fullName/config" &&
            -e "$fullName/application-deployments")
        {
            push(@results, $entry);
        }
    }
    closedir($oracleHomeDir);

    return @results;
}

#
# Lists all deployed applications for the specified OC4J instance.
# Requires that setOracleHome has already been called.
#
# Arguments
#   0: The OC4J instance name.
#   1: OC4J version category.
#
# Returns: A list of names of deployed applications.
#
sub listOC4JDeployedApps
{
    my $instanceName = $_[0];
    my $versionCategory = $_[1];
    
    my @results;
    
    my %names;
    my @modules = listOC4JModules($instanceName, $versionCategory);
    for my $module (@modules)
    {
        my $name = $$module{'appName'};
        if ($names{$name})
        {
            next;
        }
        $names{$name} = 1;
        push(@results, $name);
    }
    
    return @results;
}

#
# Lists all modules for the specified OC4J instance.
# Requires that setOracleHome has already been called.
#
# Arguments
#   0: The OC4J instance name.
#   1: OC4J version category.
#
# Returns: A list of hashes. Each hash represents one module, and has three keys:
#              'appName' => The name of the parent application of the module
#              'name' => The name of the module
#              'type' => The type of the module, either 'web', 'ejb',
#                        'application-client', or 'ra'
#
sub listOC4JModules
{
    my $instanceName = $_[0];
    my $versionCategory = $_[1];
    
    my @results;
    
    my $appsDirName = "$oracleHome/j2ee/$instanceName/application-deployments";
    my $appsDir;
    opendir($appsDir, $appsDirName);
    for my $appEntry (readdir($appsDir))
    {
        if ($appEntry eq '.' || $appEntry eq '..')
        {
            next;
        }
        my $appDirName = "$appsDirName/$appEntry";
        push(@results, listOC4JModulesForApp($appEntry, $appDirName));
    }
    closedir($appsDir);
    
    if ($versionCategory eq '1013plus')
    {
        # Add the system application for 10.1.3.
        # Modules of the system application are stored directly in the
        # OC4J instance's application-deployments directory.
        # Each module has its own directory. These directories are siblings
        # of the directories representing other applications. listOC4JModulesForApp
        # distinguishes between the module and application directories by only
        # selecting directories that contain particular configuration files.
        
        push(@results, listOC4JModulesForApp('system', $appsDirName));
    }
    
    return @results;
}

#
# Lists all modules for application deployed in the specified directory.
#
# Arguments
#   0: Application name.
#   1: Application deployment directory path.
#
# Returns: A list of hashes. Each hash represents one module, and has three keys:
#              'appName' => The name of the parent application of the module
#              'name' => The name of the module
#              'type' => The type of the module, either 'web', 'ejb',
#                        'application-client', or 'ra'
#
sub listOC4JModulesForApp
{
    my $appName = $_[0];
    my $appDirName = $_[1];

    my @results;

    my $appDir;
    opendir($appDir, $appDirName);
    for my $moduleEntry (readdir($appDir))
    {
        if ($moduleEntry eq '.' || $moduleEntry eq '..')
        {
            next;
        }
        my $moduleDirName = "$appDirName/$moduleEntry";
        if (-d $moduleDirName)
        {
            my $moduleType;
            if (-e "$moduleDirName/orion-web.xml")
            {
                $moduleType = 'web';
            }
            elsif (-e "$moduleDirName/orion-ejb-jar.xml")
            {
                $moduleType = 'ejb';
            }
            elsif (-e "$moduleDirName/orion-application-client.xml")
            {
                $moduleType = 'application-client';
            }
            elsif (-e "$moduleDirName/oc4j-ra.xml")
            {
                $moduleType = 'ra';
            }
            
            if (defined($moduleType))
            {
                my $resultRef = { 'appName' => $appName,
                                  'name' => $moduleEntry,
                                  'type' => $moduleType };
                push(@results, $resultRef);
            }
        }
    }
    closedir($appDir);
    
    return @results;
}


#
# Returns a list of hashes containing information about applications
# deployed to an OC4J instance.
#
# Requires that setOracleHome has already been called.
#
# Arguments
#   0: The OC4J instance name.
#   1: OC4J version category.
#
# Returns: A list of hashes. Each hash represents a
# <global-application> or <application> tag defined in server.xml.
# The key/value pairs in the hash correspond to
# the attributes of the <global-application> or <application> tag.
#
sub listOC4JAppInfo
{
    use ias::simpleXPath;

    my $instanceName = $_[0];
    my $versionCategory = $_[1];

    my $configFile = getOC4JConfigFile($instanceName);
    my @results = simpleXPathQuery(
        $configFile,
        'application-server/global-application');
    my @results2 = simpleXPathQuery(
        $configFile,
        'application-server/application');
    push(@results, @results2);
    
    if ($versionCategory eq '1013plus')
    {
        # Add the system application for 10.1.3.
        # The system application has no parent.
        push(@results, { 'name' => 'system', 'parent' => ' ' });
    }
    
    return @results;
}

#
# Determines path of the configuration file (server.xml) for an OC4J instance.
#
# Arguments
#  0: OC4J instance name
#
# Returns: The absolute path of the configuration file.
#
sub getOC4JConfigFile
{
    my ( $instanceName ) = @_;

    my $configFile;
    if (-e "$oracleHome/opmn/conf/opmn.xml")
    {
        my %moduleData = getOpmnModuleData({
            'module' => 'OC4J',
            'process-type' => $instanceName
        });
        $configFile = $moduleData{'start-parameters'}{'config-file'};
        # Relative paths do not appear to be supported; assume the path is absolute
    }
    if (!defined($configFile))
    {
        $configFile = "$oracleHome/j2ee/$instanceName/config/server.xml";
    }

    return $configFile;
}

#
# Returns a list of hashes containing information about web module
# mappings for an OC4J instance, for applications that are deployed to
# that instance.
#
# Requires that setOracleHome has already been called.
#
# Arguments
#   0: The OC4J instance name.
#   1: OC4J version category.
#
# Returns: A list of hashes. Each hash represents a
# <web-app> or <default-web-app> tag defined in one
# of the OC4J's *-web-site.xml files.
# The key/value pairs in the hash correspond to
# the attributes of the <web-app> or <default-web-app> tag.
#
sub listOC4JWebModuleMappings
{
    use ias::simpleXPath;
    use File::Spec;

    my $instanceName = $_[0];
    my $versionCategory = $_[1];

    my @webModuleMappings;

    my %isDeployedApp;
    my @deployedApps = listOC4JDeployedApps($instanceName, $versionCategory);
    for my $appName (@deployedApps)
    {
        $isDeployedApp{$appName} = 1;
    }
    
    my $configFile = getOC4JConfigFile($instanceName);
    my ( $configVolume, $configDir ) = File::Spec->splitpath($configFile);
    $configDir = File::Spec->catpath($configVolume, $configDir);

    my @webSites = simpleXPathQuery(
        $configFile,
        'application-server/web-site');
    
    # loop through all website files.
    # for each web site file, get web-app and default-web-app tags.
    for my $webSite (@webSites)
    {
        main::EMD_PERL_DEBUG("found website: " . $webSite);

        my $path = $$webSite{'path'};
        
        my $fileName = File::Spec->rel2abs($path, $configDir);
        
        my @results = simpleXPathQuery(
            $fileName,
            'web-site/default-web-app');
        my @results2 = simpleXPathQuery(
            $fileName,
            'web-site/web-app');
        push(@results, @results2);
        for my $resultRef (@results)
        {
            my $appName = $$resultRef{'application'};
            main::EMD_PERL_DEBUG("found app: " . $appName);
            if ($isDeployedApp{$appName})
            {
                push(@webModuleMappings, $resultRef);
            }
        }
    }

    return @webModuleMappings;
}

#
# Prints information for each file in a list, using statFile.
# Requires that setOracleHome has already been called.
#
# Arguments
#   0: Flag specifying whether to output columns for file upload.
#   1: Target name.
#   2... List of files.
#
sub statFiles
{
    my $forUpload = shift(@_);
    my $targetName = shift(@_);
    my @files = @_;
    for my $file (@files)
    {
        if (-e "$oracleHome/$file")
        {
            statFile("$oracleHome/$file", $forUpload, $targetName);
        }
    }
}

#
# Prints information about a file, formatted as an em_result for
# agent metric collection. Outputs the full file path, size (in bytes),
# and modification time (the raw number from the file spec).
# Requires that setOracleHome has already been called.
#
# Arguments
#   0: Full file path.
#   1: Flag specifying whether to output columns for file upload.
#   2: Target name.
#   3: OC4J instance name, if applicable.
#
sub statFile
{
    use File::stat ();

    my $file = $_[0];
    my $forUpload = $_[1];
    my $targetName = $_[2];
    my $oc4jName = $_[3];

    my $display = $file;
    if (-e $file)
    {
        my $st = File::stat::stat($file);
        my $size = $st->size;
        my $modTime = formatTimestamp($st->mtime);
        my $oracleHome1 = $oracleHome;
        $oracleHome1 =~ s/\\/\\\\/g; # Escape \'s before matching
        if (defined($oc4jName) && $file =~ /^$oracleHome1\/j2ee\/$oc4jName\/(.*)/)
        {
            $display = $oracleHome . '/j2ee/$OC4J_NAME/' . $1;
        }
        if ($file =~ /^(.*)\.(.+)$/)
        {
            my $extension = $2;
            if (contains($extension, 'rar', 'ear', 'war'))
            {
                $file = '';
            }
        }
        if ($forUpload)
        {
            if ($file ne '')
            {
                printResult(
                    $file,          'raw-string',
                    $file,          'raw-string',
                    $modTime,       'string',
                    $targetName,    'raw-string',
                );
            }
        }
        else
        {
            printResult(
                $display,   'path',
                $file,      'raw-string',
                $size,      'number',
                $modTime,   'string',
            );
        }
    }
}

#
# Formats a raw timestamp from a file spec into the canonical form
# for dates used by the repository.
#
# Arguments
#   0: Raw timestamp
#
# Returns: A date/time sting of the form yyyy-mm-dd hh:mm:ss
#
sub formatTimestamp
{
    use Time::localtime ();

    my $timestamp = $_[0];

    my $time = Time::localtime::localtime($timestamp);
    return sprintf('%04u-%02u-%02u %02u:%02u:%02u',
                   ($time->year + 1900),
                   ($time->mon + 1),
                   $time->mday,
                   $time->hour,
                   $time->min,
                   $time->sec);
}

#
# Get the environment defined in opmn.xml for the specified process set.
# This includes the values of all variables defined for any environment
# in the process set element, and any environments of parent elements.
# See the OPMN documentation for more details.
#
# Arguments
#   0: A reference to a hash of identifiers for the process set. This may
#      include the following keys:
#          'ias-component' => value for the id attribute of an ias-component tag
#          'process-type' => value of the id attribute of a process-type tag
#          'module' => value of the module-id attribute of a process-type tag
#          'process-set' => value of the id attribute of a process-set tag
#
# Returns: Reference to a hash mapping from variable IDs to values.
#
sub getOpmnEnvironment
{
    my ( $processSetParams ) = @_;

    use ias::simpleXPath;

    # Find all the variables defined at the specified process set element.
    # This includes all the variable tags listed in all environment
    # tags in the given process-set and its ancestors.

    my $query = getOpmnProcessSetQuery($processSetParams);
    $query .= '/ancestor-or-self::*/environment/variable';

    my @variables;
    my $opmnXml = "$oracleHome/opmn/conf/opmn.xml";
    if (! -e $opmnXml)
    {
        $opmnXml = "$oracleHome/config/OPMN/opmn/opmn.xml";
        if (-e $opmnXml) {$isAS11 = 1; }
    }
    if (-e $opmnXml)
    {
        @variables = simpleXPathQuery($opmnXml, $query);
    }
    # Interpret the listed variables to create the environment, according to
    # the rules documented for opmn.xml.

    my %env;

    # By default the environment will contain ORACLE_HOME and ORACLE_CONFIG_HOME
    # variables, both with the value passed to setOracleHome.
    $env{'ORACLE_HOME'} = $oracleHome;
    $env{'ORACLE_CONFIG_HOME'} = $oracleHome;

    # The platform-specific path delimiter is needed to interpret variables
    # that are specified with append="true"
    my $delim;
    if (getPlatform() eq 'windows')
    {
        $delim = ';';
    }
    else
    {
        $delim = ':';
    }

    for my $variable (@variables)
    {
        if (!defined($$variable{'id'}) || !defined($$variable{'value'}))
        {
            # This should be an error in opmn.xml
            next;
        }

        my $varName = $$variable{'id'};
        my $value = substituteOpmnVariables($$variable{'value'}, \%env);

        if (defined($$variable{'append'}) &&
            $$variable{'append'} eq 'true' &&
            defined($env{$varName}))
        {
            $env{$varName} = $env{$varName} . $delim . $value;
        }
        else
        {
            $env{$varName} = $value;
        }
    }
    
    return \%env;
}

#
# Utility to substitute variables from the given environment into a string value,
# as documented for opmn.xml. Variables may appear embedded in the string like:
#      ...$VAR_NAME...
#      ...${VAR_NAME}...
#      ...%VAR_NAME%...
# Variables that are not defined in the given environment are not substituted.
# See the documentation for opmn.xml.
#
# Arguments
#   0: The string value to do substitution in.
#   1: Reference to a hash of variables that may be substituted.
#
# Returns: The input value with all defined variables substituted.
#
sub substituteOpmnVariables
{
    my ( $value, $env ) = @_;
    
    if (!defined($value))
    {
        return undef;
    }
    
    if ($value =~ /^(.*?)(\${)([^}]+)(})(.*)$/ ||
        $value =~ /^(.*?)(\$)(.+?)()(\W.*)$/ ||
        $value =~ /^(.*?)(\$)(.+?)()()$/ ||
        $value =~ /^(.*?)(%)([^%]+)(%)(.*)$/)
    {
        my $before = $1;
        my $varBegin = $2;
        my $varName = $3;
        my $varEnd = $4;
        my $after = $5;

        my $varValue = $$env{$varName};
        if (!defined($varValue))
        {
            $varValue = $varBegin . $varName . $varEnd;
        }
        $value = substituteOpmnVariables($before, $env) .
                 $varValue .
                 substituteOpmnVariables($after, $env);
    }

    return $value;
}

#
# Utility to remove ^ escape characters used by OPMN.
#
# Arguments
#   0: The string to remove escape characters from.
#
# Returns: The argument with escape characters removed.
#
sub removeOpmnEscapeCarets
{
    my ( $value ) = @_;

    if (defined($value))
    {
        $value =~ s/\^(.)/$1/g;
    }
    return $value;
}

#
# Gets the aggregated module data for the specified process set. This includes all
# module data defined within the process-set element, any parent elements, and any
# associated module element.
# See the OPMN documentation for more details.
#
# Arguments
#   0: A reference to a hash of identifiers for the process set. This may
#      include the following keys:
#          'ias-component' => value for the id attribute of an ias-component tag
#          'process-type' => value of the id attribute of a process-type tag
#          'module' => value of the module-id attribute of a process-type tag
#          'process-set' => value of the id attribute of a process-set tag
#
# Returns: A hash mapping from module data category IDs to (a reference to a hash
#          mapping from module data IDs to module data values).
#
sub getOpmnModuleData
{
    my ( $processSetParams ) = @_;

    my $opmnXml = "$oracleHome/opmn/conf/opmn.xml";
    if (! -e $opmnXml)
    {
        $opmnXml = "$oracleHome/config/OPMN/opmn/opmn.xml"; 
        if (! -e $opmnXml)
        {
            main::EMD_PERL_DEBUG("Could not find opmn.xml");
            return ();
        }
        $isAS11 = 1;
    }

    # Get the XPath query for the specified process-set element
    my $processSetQuery = getOpmnProcessSetQuery($processSetParams);

    # Construct queries to the module-data elements of the various ancestors of the process-set
    my $processTypeQuery = $processSetQuery . '/ancestor::process-type/module-data';
    my $iasComponentQuery = $processSetQuery . '/ancestor::ias-component/module-data';
    my $iasInstanceQuery = $processSetQuery . '/ancestor::ias-instance/module-data';

    # Qualify the process-set query to refer to its module data
    $processSetQuery .= '/module-data';

    # If a module is specified, construct a query for its module data.
    # The module element will not be an ancestor of the process-set element,
    # it is associated by the module-id attribute on the process-type element.
    my $moduleQuery;
    if (defined($$processSetParams{'module'}))
    {
        $moduleQuery =
            'opmn/process-manager/process-modules/module/module-id[@id="' .
            $$processSetParams{'module'} .
            '"]/ancestor::module/module-data';
    }

    # Order queries so that overrides follow the correct pattern. For example,
    # data values defined in a process-set should override values with the
    # same IDs defined in a process-type.
    my @queries = (
        $iasInstanceQuery,
        $iasComponentQuery,
        $moduleQuery,
        $processTypeQuery,
        $processSetQuery
    );

    # Execute queries and build an aggregate result hash
    my %results;
    for my $query (@queries)
    {
        if (!defined($query))
        {
            next;
        }
        my @moduleDataNodes = simpleXPathQueryForNodes($opmnXml, $query);
        for my $moduleDataNode (@moduleDataNodes)
        {
            addModuleData(\%results, $moduleDataNode);
        }
    }

    # Perform variable substitutions on data values
    my $env = getOpmnEnvironment($processSetParams);
    for my $category (keys %results)
    {
        my $data = $results{$category};
        for my $dataId (keys %{$data})
        {
            my $value = $$data{$dataId};
            $value = substituteOpmnVariables($value, $env);
            $$data{$dataId} = $value;
        }
    }

    return %results;
}

#
# Utility to add module data under the specified element node to a result map.
#
# Arguments
#   0: Reference to a result hash as returned by getOpmnModuleData
#   1: Reference to a module-data element node as returned by
#      ias::simpleXPath::simpleXPathQueryForNodes
#
sub addModuleData
{
    my ( $results, $moduleData ) = @_;

    # Visit each category element in the given module-data element
    my @categories = ias::simpleXPath::getChildren($moduleData, 'category');
    for my $category (@categories)
    {
        my $categoryId = $$category{'attributes'}{'id'};
        if (!defined($categoryId))
        {
            # This should be an error in opmn.xml
            next;
        }
        # See if the result map already has an entry for this category ID.
        # If not, create a new entry.
        my $categoryData = $$results{$categoryId};
        if (!defined($categoryData))
        {
            my %categoryDataHash = ();
            $categoryData = \%categoryDataHash;
            $$results{$categoryId} = $categoryData;
        }
        # Visit all the data elements defined in the category element
        # and store their values
        my @dataElements = ias::simpleXPath::getChildren($category, 'data');
        for my $dataElement (@dataElements)
        {
            my $dataId = $$dataElement{'attributes'}{'id'};
            my $dataValue = $$dataElement{'attributes'}{'value'};
            if (!defined($dataId) || !defined($dataValue))
            {
                # This should be an error in opmn.xml
                next;
            }

            # Unless process-conversion is false, remove ^ escape characters.
            # This is done before variable substitution to match observed OPMN behavior.
            if (!defined($$dataElement{'attributes'}{'process-conversion'}) ||
                $$dataElement{'attributes'}{'process-conversion'} eq 'true')
            {
                $dataValue = removeOpmnEscapeCarets($dataValue);
            }

            $$categoryData{$dataId} = $dataValue;
        }
    }
}

#
# Utility to contruct an XPath query for a process-set element in opmn.xml.
# The path to a process-set element should be like:
#     opmn/process-manager/ias-instance/ias-component/process-type/process-set
# There should be only one ias-instance element in a given opmn.xml file.
# The descendent elements ias-component, process-type, and process-set
# may occur multiple times and can be selected by their id attributes.
# See the OPMN documentation for more details.
#
# Arguments
#   0: A reference to a hash of identifiers for the process set. This may
#      include the following keys:
#          'ias-component' => value for the id attribute of an ias-component tag
#          'process-type' => value of the id attribute of a process-type tag
#          'module' => value of the module-id attribute of a process-type tag
#          'process-set' => value of the id attribute of a process-set tag
#
# Returns: An XPath query string.
#
sub getOpmnProcessSetQuery
{
    my %params = %{$_[0]};

    my $iasComponent = $params{'ias-component'};
    my $processType = $params{'process-type'};
    my $module = $params{'module'};
    my $processSet = $params{'process-set'};

    use ias::simpleXPath;

    # Start with a query for the single ias-instance element
    my $query = 'opmn/process-manager/ias-instance';

    # Add child ias-component elements
    $query .= '/ias-component';
    if (defined($iasComponent))
    {
        $query .= '[@id="' . $iasComponent . '"]';
    }

    # Add child process-type elements
    $query .= '/process-type';
    if (defined($processType))
    {
        $query .= '[@id="' . $processType . '"]';
    }
    if (defined($module))
    {
        $query .= '[@module-id="' . $module . '"]';
    }

    # Add child process-set elements
    $query .= '/process-set';
    if (defined($processSet))
    {
        $query .= '[@id="' . $processSet . '"]';
    }

    return $query;
}

# 
# Gets the working directory attribute for a specified process set. The working
# directory is the base path used to qualify relative paths configured for the
# component processes. It may be defined by the working-dir attribute on the
# process-set element, or its parent process-type element. The value may contain
# OPMN variables.
# See the OPMN documentation for more details.
#
# Arguments
#   0: A reference to a hash of identifiers for the process set. This may
#      include the following keys:
#          'ias-component' => value for the id attribute of an ias-component tag
#          'process-type' => value of the id attribute of a process-type tag
#          'module' => value of the module-id attribute of a process-type tag
#          'process-set' => value of the id attribute of a process-set tag
#
# Returns: The working directory specified for the process set.
#
sub getOpmnWorkingDir
{
    use ias::simpleXPath;

    my ( $processSetParams ) = @_;

    # Retrieve the specified process-set element
    my $query = getOpmnProcessSetQuery($processSetParams);

    my $opmnXml = "$oracleHome/opmn/conf/opmn.xml";
    if (! -e $opmnXml)
    {
        # handle AS11 
        $opmnXml = "$oracleHome/config/OPMN/opmn/opmn.xml";
        if (-e $opmnXml) { $isAS11 = 1; }
    }

    my $processSet = (simpleXPathQueryForNodes($opmnXml, $query))[0];
    if (!defined($processSet))
    {
        # This probably means the query is incorrect
        return $oracleHome;
    }
    # See if the process-set element has a working-dir attribute
    my $workingDir = $$processSet{'attributes'}{'working-dir'};
    if (!defined($workingDir))
    {
        # There is no working-dir attribute on the process-set element,
        # look at the parent process-type element
        my $processType = $$processSet{'parent'};
        if ($$processType{'tagName'} ne 'process-type')
        {
            # This should be an error in opmn.xml
            return $oracleHome;
        }
        $workingDir = $$processType{'attributes'}{'working-dir'};
    }
    if (!defined($workingDir))
    {
        # No working-dir atttribute found, default to the Oracle Home
        return $oracleHome;
    }

    # Perform variable substitutions on the working-dir value
    my $env = getOpmnEnvironment($processSetParams);
    $workingDir = substituteOpmnVariables($workingDir, $env);

    # Remove ^ escape characters. This is done after variable substitution
    # to match observed OPMN behavior.
    $workingDir = removeOpmnEscapeCarets($workingDir);

    return $workingDir;
}

#
# Gets the name of the root Apache conf file, as specified in 
# $ORACLE_HOME/opmn/conf/opmn.xml. The default is
# $ORACLE_HOME/Apache/Apache/conf/httpd.conf.
# Requires that setOracleHome has already been called.
#
# AS11 puts httpd.conf in $ORACLE_INSTANCE/config/OHS/<ohs-instance>
# where OHS instance id is passed through $ENV{'CCR_OHS_ID'}
sub getApacheRootConfFile
{
    use ias::simpleXPath;
    use File::Spec;

    my $ohsID = "";
    if (! -e "$oracleHome/opmn/conf/opmn.xml")
    {
        if (-e "$oracleHome/config/OPMN/opmn/opmn.xml")
        {
            $isAS11 = 1;
            if (defined $ENV{'CCR_OHS_ID'})
            {
                $ohsID = $ENV{'CCR_OHS_ID'};
            }
        }
        if ($ohsID eq "")
        {
            main::EMD_PERL_DEBUG("returning default");
            return "$oracleHome/Apache/Apache/conf/httpd.conf";
        }
    }

    my %moduleData = 
        ($isAS11 ? 
            getOpmnModuleData({ 'process-type' => 'OHS', }) :
            getOpmnModuleData({ 'ias-component' => 'HTTP_Server', 'module' => 'OHS', }));

    my $fileName = $moduleData{'start-parameters'}{'config-file'};

    if (!defined($fileName))
    {
        # we must get the OHS instance name (assume ohs1??)
        $fileName = ($isAS11 ? "$oracleHome/config/OHS/$ohsID/httpd.conf" : "$oracleHome/Apache/Apache/conf/httpd.conf");
    }
    elsif (!File::Spec->file_name_is_absolute($fileName))
    {
        my $workingDir = 
            ($isAS11 ? 
                getOpmnWorkingDir({ 'process-type' => 'OHS', }) :
                getOpmnWorkingDir({ 'ias-component' => 'HTTP_Server','module' => 'OHS', }));
        $fileName = File::Spec->rel2abs($fileName, $workingDir);
    }
    return $fileName;
}

#
# Gets a list of the defined symbols Apache is configured to run with, as defined in
# $ORACLE_HOME/opmn/conf/opmn.xml.
# Requires that setOracleHome has already been called.
#
# Returns: A list of defined symbol names.
#
sub getApacheDefines
{
    use ias::simpleXPath;

    if (! -e "$oracleHome/opmn/conf/opmn.xml")
    {
        if (! -e "$oracleHome/config/OPMN/opmn/opmn.xml")
        {
            return ();
        }
        $isAS11 = 1;
    }

    my %moduleData = 
        ($isAS11 ?
        getOpmnModuleData({ 'process-type' => 'OHS', }) :
        getOpmnModuleData({ 'ias-component' => 'HTTP_Server', 'module' => 'OHS', }));
    
    my $startMode = $moduleData{'start-parameters'}{'start-mode'};
    my $ssl = (defined($startMode) && $startMode eq 'ssl-enabled') ? 1 : 0;

    my $params = $moduleData{'start-parameters'}{'command-line'};
    
    my @defines;
    
    if ($ssl)
    {
        push(@defines, 'SSL');
    }

    if (defined($params))
    {
        my @parsed = parseCommandLine($params);
        for (my $i = 0; $i < @parsed; $i += 2)
        {
            my $param = $parsed[$i];
            if ($param =~ /^D(.*)$/)
            {
                push(@defines, $1);
            }
        }
    }
    
    return @defines;
}

#
# Calls httpd for a list of compiled-in modules.
#
# Returns: A list of module names.
#
sub getApacheModules
{
    my @modules;
    my $os;

    my $httpd = "$oracleHome/Apache/Apache/bin/httpd";
    my $platform = getPlatform();
   
    # Check for AS11
    my $opmnXml = "$oracleHome/opmn/conf/opmn.xml";
    if (! -e $opmnXml)
    {
        $opmnXml = "$oracleHome/config/OPMN/opmn/opmn.xml";
        if (-e $opmnXml) {$isAS11 = 1; }
    }

    my $swInstallRoot = "";

    if ($isAS11)
      {
         my $file = '$oracleHome/config/OPMN/opmn/instance.properties';
         my @OH;
         open(RS, "$file");
         while (<RS>)
         {
           if ($_ =~ m/oracleHome=/)
          {
             @OH = split ('=', $_);
             $swInstallRoot=$OH[1];
             last;
          }
          else
          {
            next;
          }
         }
         close(RS);
      }

    my $oracleHome1 = $oracleHome;
    if ($platform eq 'windows')
    {
      #Make all directory separators forward slashes
      #else perl -e check will fail on windows
      $oracleHome1 =~ s/\\/\//g;
      $swInstallRoot =~ s/\\/\//g;
      $httpd=
            ($isAS11 ?
            "$swInstallRoot/ohs/bin/Apache.exe":
            "$oracleHome1/Apache/Apache/Apache.exe");
    }
    else
    {
      $httpd=
             ($isAS11 ?
              "$swInstallRoot/ohs/bin/httpd.worker":
              "$oracleHome/Apache/Apache/bin/httpd");
      chomp($os = `uname -s`);
      if ($os eq "AIX")
      {
         $ENV{LD_LIBRARY_PATH} = "$oracleHome/lib32:" . $ENV{LD_LIBRARY_PATH};
         #main::EMD_PERL_DEBUG ("setting LD_LIBRARY_PATH = $ENV{LD_LIBRARY_PATH}");
         $ENV{LIBPATH} = "$oracleHome/lib32:" . $ENV{LIBPATH};
         #main::EMD_PERL_DEBUG ("setting LIBPATH = $ENV{LIBPATH}");
      }
    }

    if (-e $httpd)
    {
        if ($isAS11 && ($swInstallRoot ne ''))
        {
           $ENV{LD_LIBRARY_PATH} ="$swInstallRoot/lib:" . $ENV{LD_LIBRARY_PATH};
        }
        my $httpdOut = `$httpd -l`;
        
        for my $line (split("\n", $httpdOut))
        {
            if ($line =~ /^\s*(\S+)\s*$/)
            {
                push(@modules, $1);
            }
        }
    }
    return @modules;
}

#
# Utility to parse the full OHS configuration, starting at the root configuration
# file specified in opmn.xml.
#
# Arguments
#   0: Optional reference to a list that will be filled in with the names of
#      parsed files.
#
# Returns: The parse results, as specified for ias::Apache_confFileParser::parseConfFile.
#
sub parseApacheConf
{
    my $filesRef = $_[0];

    use ias::Apache_confFileParser;

    my @definedParams = getApacheDefines();
    my @modules = getApacheModules();
    my $rootConfFile = getApacheRootConfFile();
    my $defaultServerRoot = "$oracleHome/Apache/Apache";
    my $ohsID = "";
    if (! -e "$oracleHome/opmn/conf/opmn.xml")
    {
        if (-e "$oracleHome/config/OPMN/opmn/opmn.xml")
        {
            if (defined $ENV{'CCR_OHS_ID'})
            {
                $ohsID = $ENV{'CCR_OHS_ID'};
                $defaultServerRoot ="$oracleHome/config/OHS/$ohsID";
            }
        }
    }
    if (! -e $rootConfFile)
    {
        main::EMD_PERL_DEBUG("Apache configuration file not found: $rootConfFile");
        return ();
    }

    ias::Apache_confFileParser::setApacheOracleHome($oracleHome);
    return ias::Apache_confFileParser::parseConfFile(
        $rootConfFile,
        \@definedParams,
        \@modules,
        $filesRef,
        $defaultServerRoot
    );
}

#
# Gets a list of virtual hosts configured for Apache.
#
# Arguments
#   0: Hash resulting from a call to ias::Apache_confFileParser::parseConfFile
#
# Returns: A list of hash references, each describing a virtual host.
#
sub getApacheVHosts
{
    my $hash = $_[0];

    my @results;

    my $vhosts = $$hash{'VirtualHost'};
    if (!$vhosts)
    {
        return ();
    }
    
    for my $vhostExpr (keys %{$vhosts})
    {
        my $vhostHash = $$vhosts{$vhostExpr};
        my @addrs = split(/\s+/, $vhostExpr);
        
        for my $addr (@addrs)
        {
            my $ip = $addr;
            my $port;
            if ($addr =~ /^(.*):([^\]]*)$/)
            {
                # IPv6 addresses are in brackets and contain colons
                # If the virtual host expression specifies a port,
                # it will be after a colon but not inside brackets
                $ip = $1;
                $port = $2;
            }
            if ($ip =~ /^\[(.*)\]$/)
            {
                # remove brackets around IPv6 addresses
                $ip = $1;
            }
            
            my $vhost = {
                'address' => $addr,
                'ip' => $ip,
                'port' => $port,
                'properties' => $vhostHash,
            };
            push(@results, $vhost);
        }
    }
    
    return @results;
}

#
# Parses a list of command-line parameters from a string. Parameters
# are separated by whitespace. Switches are tokens starting with "-".
# Arguments are all other tokens. Currently, quotes are not handled.
#
# Arguments
#   0: A string containing command-line parameters.
#
# Returns: A list of switch / argument pairs. Even-numbered list items are
#          switches, odd-numbered items are the corresponding arguments. If a
#          switch does not have an argument, the following entry is the empty
#          string; similarly if an argument does not have a switch, the
#          preceding entry is the empty string.
#
sub parseCommandLine
{
    my $str = $_[0];
    my @options;
    
    while (1)
    {
        if ($str =~ /^\s*$/)
        {
            # String is just whitespace
            last;
        }
        my $switch = '';
        my $arg = '';
        if ($str =~ /^\s*-(\S*)\s*(-.*)$/)
        {
            # String starts with two switches in a row
            $switch = $1;
            $str = $2;
        }
        elsif ($str =~ /^\s*-(\S*)\s*(\S*)(.*)$/)
        {
            # String starts with a switch and an argument
            $switch = $1;
            $arg = $2;
            $str = $3;
        }
        elsif ($str =~ /^\s*(\S*)(.*)$/)
        {
            # String starts with an argument
            $arg = $1;
            $str = $2;
        }
        push(@options, $switch, $arg);
    }
    
    return @options;
}

#
# locates webcache.xml in the file system
# requires that setOracleHome has been called
# Arguments
#   0: Webcache instance id ("" for pre-AS11).
#
sub getWebCacheXml
{
    my $wcId = $_[0];
    my $webcacheXml = "$oracleHome/webcache/webcache.xml";
    my $cacheID;
    # need to account for $instanceHome construct (AS11)
    if (! -e $webcacheXml)
    {
        if (-e "$oracleHome/config/OPMN/opmn/opmn.xml")
        {
            $webcacheXml = "$oracleHome/config/WebCache/$wcId/webcache.xml";
        }
        if (! -e $webcacheXml)
        {
          main::EMD_PERL_DEBUG("Could not find webcache.xml");
          return undef;
        }
    }
    return $webcacheXml;
}

#
# Computes the name of the Web Cache for the machine.
# Requires that setOracleHome has already been called.
#
# Arguments
#   0: Host name.
#   1: Webcache instance id ("" for pre-AS11).
#
# Returns: The name of the cache for the specified host and oracle home.
#
sub getMyCache
{
    use ias::simpleXPath;
    use Socket ();
    use Net::hostent ();

    my $hostName = $_[0];
    my $wcId     = $_[1];
    
    my $platform = getPlatform();
    my $cacheID;
    my $wcOH = $oracleHome;

    my $webCacheXml = getWebCacheXml($wcId);
    if (! defined ($webCacheXml))
    {
        main::EMD_PERL_DEBUG("Could not find webcache.xml");
        return undef;
    }

    if (! -e "$oracleHome/opmn/conf/opmn.xml")
    {
        if (! -e "$oracleHome/config/OPMN/opmn/opmn.xml")
        {
            return ();
        }
        $isAS11 = 1;
        $wcOH = $ENV{ORACLE_HOME};
    }

    my @caches = simpleXPathQuery($webCacheXml, 'CALYPSO/CACHE');

    main::EMD_PERL_DEBUG("no caches found") if (@caches < 1);

    for my $cache (@caches)
    {
        my $cacheName = ensuredef($$cache{'NAME'});
        my $cacheHome = ensuredef($$cache{'ORACLEHOME'});
        my $cacheHostName = ensuredef($$cache{'HOSTNAME'});
        
        # Make all directory separators forward slashes
        my $oracleHome1 = $wcOH;
        if ($platform eq 'windows')
        {
            $oracleHome1 =~ s/\\/\//g;
            $cacheHome =~ s/\\/\//g;
        }
        # Remove trailing slashes from paths
        if ($oracleHome1 =~ /^(.*)\/$/)
        {
            $oracleHome1 = $1;
        }
        if ($cacheHome =~ /^(.*)\/$/)
        {
            $cacheHome = $1;
        }
        
        my $eq;
        if ($platform eq 'windows')
        {
            $eq = lc($oracleHome1) eq lc($cacheHome);
        }
        else
        {
            $eq = $oracleHome1 eq $cacheHome;
        }
        if (!$eq)
        {
            next;
        }
        
        if ($hostName ne $cacheHostName)
        {
            my $host = Net::hostent::gethostbyname($hostName);
            my $cacheHost = Net::hostent::gethostbyname($cacheHostName);
            if (!defined($host) || !defined($cacheHost))
            {
                next;
            }
            $hostName = $host->name;
            $cacheHostName = $cacheHost->name;
            
            if ($hostName ne $cacheHostName)
            {
                my @addrList1 = map(Socket::inet_ntoa($_), @{$host->addr_list});
                my @addrList2 = map(Socket::inet_ntoa($_), @{$cacheHost->addr_list});
                my $found = 0;
                for my $addr1 (@addrList1)
                {
                    for my $addr2 (@addrList2)
                    {
                        if ($addr1 eq $addr2)
                        {
                            $found = 1;
                            last;
                        }
                    }
                    if ($found)
                    {
                        last;
                    }
                }
                if (!$found)
                {
                    next;
                }
            }
        }
        return $cacheName;
    }
    return undef;
}

#
# Determines the OS type, either 'windows' or 'unix'.
#
# Returns: 'windows' or 'unix'
#
sub getPlatform
{
    # On non-windows systems, the OS environment variable is not defined
    #my $os = $ENV{'OS'};
    #if ($os && lc($os) =~ /win/)

    if (( $^O eq "Windows_NT") ||
        ( $^O eq "MSWin32"))
    {
        return 'windows';
    }
    return 'unix';
}

#
# Converts the SSLENABLED attribute of a WebCache listen port into
# a protocol.
#
# Arguments
#   0: SSLENABLED attribute of a WebCache listen port
#
# Returns: Either 'http' or 'https'.
#
sub getListenProtocol
{
    my $ssl = $_[0];
    if (!defined($ssl) || ($ssl eq 'NONE'))
    {
        return 'http';
    }
    return 'https';
}

#
# If the IPADDR attribute of a WebCache listen port is
# ANY, replaces it with '*'. Otherwise, passes through
# the address.
#
# Arguments
#   0: IPADDR attribute of a WebCache listen port
#
# Returns: Either '*' or an address.
#
sub getListenIP
{
    my $addr = $_[0];
    if ($addr eq 'ANY')
    {
        return '*';
    }
    return $addr;
}

#
# Makes sure URLs start with '/'
#
# Arguments
#   0: A URL, or an undefined value, or the empty string
#
# Returns: Either a URL starting with '/', or the empty string
#
sub ensureSlash
{
    use ias::simpleXPath;

    my $url = ensuredef($_[0]);
    if (!defined($url) || $url eq '')
    {
        return '';
    }
    if ($url =~ /^\//)
    {
        return $url;
    }
    return "/$url";
}

#
# Parses information the web module defined in the given XML file.
#
# Arguments
#   0: XML file name
#
# Returns: A hash with keys 'jspMode', 'sessionTimeout', and 'distributable'.
#
sub parseWebAppInfo
{
    use ias::simpleXPath;

    my $fileName = $_[0];
    
    if (! -e $fileName)
    {
        return ();
    }
    
    my @results;
    
    @results = simpleXPathQueryForText(
        $fileName,
        '//init-param[param-name="main_mode"]/param-value');
    my $jspMode = '';
    for my $str (@results)
    {
        if ($str)
        {
            $jspMode = $str;
        }
    }
    
    @results = simpleXPathQueryForText(
        $fileName,
        '//session-timeout');
    my $sessionTimeout = $results[0];
    
    @results = simpleXPathQueryForText(
        $fileName,
        '//distributable');
    my $distributable;
    if (@results > 0)
    {
        $distributable = 1;
    }
    
    return ( 'jspMode' => $jspMode,
             'sessionTimeout' => $sessionTimeout,
             'distributable' => $distributable );
}

#
# Tests whether an array of strings contains a particular string.
#
# Arguments
#  0: The string to test for.
#  1... The list to search in
#
# Returns: Whether the string was found in the list.
#
sub contains
{
    my $item = shift();
    for my $x (@_)
    {
        if ($x eq $item)
        {
            return 1;
        }
    }
    return 0;
}

#
# Lists the OC4J data sources XML files for the default application and all
# applications deployed to a particular OC4J instance.
# Requires that setOracleHome has been called.
#
# Arguments
#   0: OC4J instance name.
#
# Returns
#   A list of hash references. Each hash contains:
#       'application' => application name
#       'file' => absolute path to the filename of the data sources XML file
#                 for that application
#
sub listOC4JDataSourceFiles
{
    use ias::simpleXPath;
    use File::Spec;

    my $instanceName = $_[0];
    my @results;
    
    # Find the data sources XML file for the default application
    my $configPath = "$oracleHome/j2ee/$instanceName/config";
    my $dataSourcesPath = (simpleXPathQuery(
        "$configPath/application.xml",
        'orion-application/data-sources'
    ))[-1];
    if (defined($dataSourcesPath))
    {
        my $fileName = $$dataSourcesPath{'path'};
        my $dataSourcesFile = File::Spec->rel2abs($fileName, $configPath);
        my $result = { 'application' => 'default', 'file' => $dataSourcesFile };
        push(@results, $result);
    }
    
    # List the data sources for other applications
    
    # Don't pass the version category to listOC4JDeployedApps
    # This means the special 'system' application won't be listed for 10.1.3 OC4Js.
    # However, a 'system' application may be listed for prior OC4J versions if
    # the user has actually deployed an application called 'system'.
    # The 10.1.3 system application cannot have data sources, but an application
    # called 'system' deployed by the user may have data sources.
    my @deployedApps = listOC4JDeployedApps($instanceName);
    for my $app (@deployedApps)
    {
        if ($app eq 'default')
        {
            next;
        }
        # Find the data sources XML file
        my $appPath = "$oracleHome/j2ee/$instanceName/application-deployments/$app";
        if (-e "$appPath/orion-application.xml")
        {
            my $dataSourcesPath = (simpleXPathQuery(
                "$appPath/orion-application.xml",
                'orion-application/data-sources'
            ))[-1];
            if (defined($dataSourcesPath))
            {
                my $fileName = $$dataSourcesPath{'path'};
                my $dataSourcesFile = File::Spec->rel2abs($fileName, $appPath);
                my $result = { 'application' => $app, 'file' => $dataSourcesFile };
                push(@results, $result);
            }
        }
    }
    
    return @results;
}

return 1;
