# $Header: emll/sysman/admin/scripts/ias/oracle_ias_assoc.pl /main/18 2012/10/03 15:57:41 fmorshed Exp $
#
# Package : oracle_ias_assoc.pl
#
# Copyright (c) 2005, 2012, Oracle and/or its affiliates. All rights reserved. 
#
#   NAME
#       oracle_ias_assoc.pl
#
#   DESCRIPTION
#       Returns information on targets associated with this iAS instance
#
#   List of standard assocation types are in
#   ecmAssoc.pm
#
#   returns potentially multiple 
# em_result=association_type|target_name|target_type|target_properties
#
# association_type: one of the types listed in ecmAssoc.pm
# target_name: the target_name of the associated target
# target_type: the target_type of the associated target
# target_properties: related information that may help determine true target name given the target_type
#
#   NOTES
#
#   MODIFIED     (MM/DD/YY)
#      fmorshed   10/02/12 - Remove / as first character to query targets.xml
#      fmorshed   01/19/12 - In advent of central collector, change query of
#                 targets.xml to account for the new hierarchy of tags.
#      jsutton    07/26/07 - Pick up existing shared OH work
#      jsutton    07/25/07 - Shared home work
#      jsutton    11/21/06 - Need more detail for resolution
#      pparida    09/25/06 - Fix bug 5561858: Do not collect infra db assoc if it
#                            begins with a %
#      pparida    06/23/06 - Add associations of IasInstance to Farms and 
#                            Clusters (Bug 5293161: Add support for iAS farms and 
#                            clusters) 
#      jsutton    03/10/06 - Upcase all aliases from tns for lookup
#      pparida    03/03/06 - Fix bug 5029956 
#      jsutton    12/07/05 - Check for empty infra db 
#      jsutton    11/17/05 - handle TNS-like DB connect info better
#      jsutton    11/07/05 - Fix up requires and directory paths 
#      jsutton    10/17/05 - refine db assoc info 
#      jsutton    10/13/05 - Fix hardcoded refs to liveLink 
#      jsutton    09/19/05 - trim leading and trailing whitespace from DB
#      ndutko     08/03/05 - ndutko_code_checkin
#      jsutton    06/08/05 - lifted shamelessly from hostAssoc
##*************************************************************

use strict;
#
# relationship types are defined in ecmassoc.pm
#
use ecmAssoc;
use ias::simpleXPath;
use Config;

$ENV{EMDROOT} = $ENV{CCR_HOME};
require "$ENV{CCR_HOME}/sysman/admin/scripts/db/net/listenerUtil.pl";
require "$ENV{CCR_HOME}/sysman/admin/scripts/ias/IAS_getFarmInfo.pl";

#******************************************
#     Global Variables
#*****************************************
my $result = ();

#
# as new relationships are exposed, add them here
#
my @ias_assoc_results;

my $oracleHome = shift;
my $iasConfigFile = $oracleHome . "/config/ias.properties";
my $discoveredTargetsFile = $ENV{CCR_CONFIG_HOME} . "/config/default/targets.xml";
my $iasDadsFile = $oracleHome . "/Apache/modplsql/conf/dads.conf";
my $ias1022DadsFile = $oracleHome . "/Apache/modplsql/cfg/wdbsvr.app";

my $tnsParams;
my $tnsHash;
my %tnsHashAllCaps = ();

my $targetsHome = $oracleHome;
if (defined $ENV{USER_ORACLE_HOME} && $ENV{USER_ORACLE_HOME} ne $targetsHome)
{
  $targetsHome = $ENV{USER_ORACLE_HOME};
}
getIasSubtargets();
getIasDbAssoc();
getIasFarmAssoc();

foreach $result (@ias_assoc_results)
{
  print $result;
}

sub getIasFarmAssoc
{
  my ($iasVersion, $farmName, $clusterName) = getIASFarmInfo($oracleHome);
  if($clusterName ne "")
  {
    push(@ias_assoc_results, "em_result=".$ecmAssoc::MEMBER_OF."|".$clusterName."|".$ecmAssoc::ORACLE_IAS_CLUSTER_TYPE."|\n");
  }
  else
  {
    if($farmName ne "")
    {
      push(@ias_assoc_results, "em_result=".$ecmAssoc::MEMBER_OF."|".$farmName."|".$ecmAssoc::ORACLE_IAS_FARM_TYPE."|\n");
    }
  }
}

sub getIasSubtargets
{
  # read thru discovered targets file and pull out associations (MemberOf)
  if (-e $discoveredTargetsFile)
  {
    my $ohQueryPrefix = 'TargetHomes/TargetHome[@LOCATION="' . $ENV{ORACLE_HOME} . '"]';
    my @allTargets = simpleXPathQuery($discoveredTargetsFile, $ohQueryPrefix . "/Targets/Target");
    for my $target (@allTargets)
    {
      my $targetName = $$target{'NAME'};
      my $targetType = $$target{'TYPE'};

      my @subTargets = simpleXPathQuery($discoveredTargetsFile, $ohQueryPrefix . '/Targets/Target[@NAME="' . $targetName . '"]/CompositeMembership/MemberOf');
      my ($key, $value,$hashRef);
      foreach $hashRef (@subTargets)
      {
        my $memberType = $$hashRef{'TYPE'};
        if ($memberType eq "oracle_ias")
        {
          push(@ias_assoc_results, "em_result=".$ecmAssoc::CONTAINS."||".$targetType."|".$targetsHome.":".$targetName."\n");
        }
      }
    }
  }
}

sub getIasDbAssoc
{
  my $infraDb;
  if (-e $iasConfigFile)
  {
    # read ias.properties file and retrieve the Infra DB name (9.0.2+)
    if (open(IASPROPS, "<$iasConfigFile"))
    {
      while (<IASPROPS>)
      {
        if(/InfrastructureDBCommonName/)
        {
          (undef,$infraDb) = split(/=/, $_);
          if (defined($infraDb))
          {
            $infraDb =~ s/^\s+//;
            $infraDb =~ s/\s+$//;
            if (length($infraDb) > 0 && !($infraDb eq "INVALID") && ($infraDb !~ m/^%/))
            {
              # This is supposed to be a global DB name (per iAS docs)
              push(@ias_assoc_results, "em_result=$ecmAssoc::CONNECTS_TO||$ecmAssoc::ORACLE_DATABASE_TYPE|$infraDb\n");
            }
          }
          last;
        }
      }
      close(IASPROPS);
    }
  }
  # check DADs file (9.0.2+) for connect strings
  if (-e $iasDadsFile)
  {
    if (open(IASDADS, "<$iasDadsFile"))
    {
      while (<IASDADS>)
      {
        if(/PlsqlDatabaseConnectString/)
        {
          my (@info) = split;
          shift(@info);
          if (scalar(@info) == 1)
          {
            # no format designator, so it should be a TNS alias, host:port:sid, or TNS connect string (?)
            resolveTnsDesignator($info[0]);
          }
          else
          {
            for ($info[1])
            {
              if (/ ServiceNameFormat/)
              {
                # we get host:servicename, and use the same resolution as for oc4j
                # ServiceNameFormat should have host:port:servicename -- split, give host:servicename
                my ($snHost,$snPort,$snSvc) = split(/:/,$info[0]);
                push(@ias_assoc_results, "em_result=$ecmAssoc::CONNECTS_TO||$ecmAssoc::ORACLE_DATABASE_TYPE|".$snHost.":".$snSvc."\n");
              }
              elsif (/SIDFormat/)
              {
                # host:port:sid, return as is
                push(@ias_assoc_results, "em_result=$ecmAssoc::CONNECTS_TO||$ecmAssoc::ORACLE_DATABASE_TYPE|$info[0]\n");
              }
              elsif (/TNSFormat/)
              {
                # TNS Alias *or* connect descriptor
                resolveTnsDesignator($info[0]);
              }
              elsif (/NetServiceNameFormat/)
              {
                # LDAP-like, e.g. cn=oracle,cn=iasDB
                push(@ias_assoc_results, "em_result=$ecmAssoc::CONNECTS_TO||$ecmAssoc::ORACLE_DATABASE_TYPE|NSNF=\"$info[0]\"\n");
              }
            }
          }
        }
      }
      close(IASDADS);
    }
  }
  # check DADs file (1.0.2.2) for connect strings
  if (-e $ias1022DadsFile)
  {
    if (open(IAS1022DADS, "<$ias1022DadsFile"))
    {
      while (<IAS1022DADS>)
      {
        if(/connect_string/)
        {
          (undef,$infraDb) = split(/=/, $_);
          $infraDb =~ s/^\s+//;
          $infraDb =~ s/\s+$//;
          if (index($infraDb,"sample-tcp") < 0)
          {
            resolveTnsDesignator($infraDb);
          }
        }
      }
      close(IAS1022DADS);
    }
  }
}

sub resolveTnsDesignator
{
  my $tnsDes = shift;
  # in case this is an alias, go look it up in TNS config
  if (index($tnsDes,"(") < 0) # no open paren
  {
    if (index($tnsDes,":") < 0) # no colon separators
    {
      # check for a TNS alias
      my $connDesc = tnsLookup($tnsDes);
      if (length($connDesc) > 0)
      {
        getAssocFromConnDesc($connDesc);
      }
      # not resolved to an alias, assume it's some other form.. return as is
      else
      {
        push(@ias_assoc_results, "em_result=$ecmAssoc::CONNECTS_TO||$ecmAssoc::ORACLE_DATABASE_TYPE|$tnsDes\n");
      }
    }
    # there's a colon separator, assume host:port:sid and return as is
    else
    {
      push(@ias_assoc_results, "em_result=$ecmAssoc::CONNECTS_TO||$ecmAssoc::ORACLE_DATABASE_TYPE|$tnsDes\n");
    }
  }
  # we got parentheses, so parse & get associations from the descriptor
  else
  {
    getAssocFromConnDesc($tnsDes);
  }
}

sub getAssocFromConnDesc
{
  my $connDesc = shift;
  # pick off host:port:sid or host:service_name sets in TNS descriptor
  # Host/Port info is in ADDRESS, Service Name/SID is in CONNECT_DATA
  my $addresses = getParamValueListFor($connDesc,"ADDRESS");
  my $connectData = getParamValueListFor($connDesc,"CONNECT_DATA");
  my $sidOrService = "";
  my ($addrInfo,$connInfo);
  foreach $addrInfo(@$addresses)
  {
    foreach $connInfo(@$connectData)
    {
      if (defined($connInfo->{SERVICE_NAME}))
      {
        $sidOrService = $connInfo->{SERVICE_NAME};
      }
      elsif (defined($connInfo->{SID}))
      {
        $sidOrService = $addrInfo->{PORT}.":".$connInfo->{SID};
      }
      push(@ias_assoc_results, "em_result=$ecmAssoc::CONNECTS_TO||$ecmAssoc::ORACLE_DATABASE_TYPE|$addrInfo->{HOST}:$sidOrService\n");
    }
  }
}

sub tnsLookup
{
  # look up the input parameter to see if it's in the list of tns aliases
  my $tnsAlias = uc(shift);
  my ($resultParams,$resultHash) = getTnsAliases();
  if (defined(${$resultHash}{$tnsAlias}))
  {
    return ${$resultHash}{$tnsAlias};
  }
}

sub getTnsAliases
{
  # make sure we only get tnsnames.ora parsed once
  if (!defined($tnsParams))
  {
    my $tnsFile;

    if ($Config{'osname'} ne "MSWin32")
    {
      if (-e "$ENV{HOME}/.tnsnames.ora")
      {
        $tnsFile = "$ENV{HOME}/.tnsnames.ora";
      }
      elsif (defined($ENV{TNS_ADMIN}) && (-e "$ENV{TNS_ADMIN}/tnsnames.ora"))
      {
        $tnsFile = "$ENV{TNS_ADMIN}/tnsnames.ora";
      }
      elsif (lc($Config{'osname'}) eq "solaris")
      {
        if (-e "/var/opt/oracle/tnsnames.ora")
        {
          $tnsFile = "/var/opt/oracle/tnsnames.ora";
        }
        else
        {
          $tnsFile = "$oracleHome/network/admin/tnsnames.ora";
        }
      }
      else
      {
        if (-e "/etc/tnsnames.ora")
        {
          $tnsFile = "/etc/tnsnames.ora";
        }
        else
        {
          $tnsFile = "$oracleHome/network/admin/tnsnames.ora";
        }
      }
    }
    else
    {
      if (defined($ENV{TNS_ADMIN}) && (-e "$ENV{TNS_ADMIN}/tnsnames.ora"))
      {
        $tnsFile = "$ENV{TNS_ADMIN}/tnsnames.ora";
      }
      else
      {
        $tnsFile = "$oracleHome/network/admin/tnsnames.ora";
      }
    }  
    ($tnsParams,$tnsHash) = parseOracleConfigFileWithOrder($tnsFile);
  }
  my ($tnsAlias, $tnsInfo);
  while (($tnsAlias, $tnsInfo) = each %{$tnsHash})
  {
    $tnsHashAllCaps{uc($tnsAlias)} = uc($tnsInfo);
  }
  return ($tnsParams,\%tnsHashAllCaps);
}
1;
# End of the Program
