# Copyright (c) 2003, 2005, Oracle. All rights reserved.  
#
# NAME
#   ifsresourceusage.pl
#     Gets the OS resource usage for all the iFS processes of the
#     given iFS target, grouped by four process types:
#       - Domain Controller
#       - Node Guardian
#       - Node Manager
#       - OC4J instances
#          * OC4J_iFS_cmsdk and OC4J_iFS_webstarterapp for CM SDK
#          * OC4J_iFS_files for Files
#          * OC4J_iFS_websites for Websites
#
#                       
# DESCRIPTION
#   ifsresourceusage.pl <emd_root> <ifs_root_home> <ifs_domain_name>
#                       <ifs_domain_type>
#
#    where:
#      <emd_root> is the path of the Oracle Home of the current EM Agent
#      <ifs_root_home> is the path of the iFS Root Home of the
#        iFS target; it is in the format of "<oracle_home>/ifs"
#      <ifs_domain_name> is the domain name of the iFS target
#      <ifs_domain_type> is the domain type of the iFS target
#
#    note:
#      iFS's Oracle Home is different than the EM Agent's Oracle Home
#
#    returns:
#      em_result=[process_name]|[cpu_util]|[cpu_other]|[cpu_idle]|
#      [memory_usage]|[memory_usge_percent]|[memory_total]
#      one line per process group, plus the last line for sum total
#
#      where:
#        [process_name] is one of the following:
#          - Domain Controller
#          - Node Guardian
#          - Node Manager
#          - OC4J_iFS_xxx
#          - Total
#        [cpu_util] is the CPU usage percentage of the given process group
#        [cpu_other] is the CPU usage percentage of all other processes
#        [cpu_idle] is the CPU idle percentage
#        [memory_usage] is the physical memory usage (in MB) of the given
#          process group
#        [memory_usage_percent] is the physical memory usage percentage
#          of the given process group
#        [memory_total] is the total physical memory (in MB) of the host
#
#
# HISTORY
#   02/16/2004    mhong - support windows platforms
#   12/22/2003    mhong - use getPhysicalMemoryMetrics()
#                         in iasresourceusage.pl to get total physical memory
#                       - remove divide by 100 from the result of
#                         getTotalCpuTime() because we should use percentage
#                         values in our calculation
#   11/14/2003    mhong - created
#

use strict;
require "emd_common.pl";
require "iasresourceusage.pl";
require "iasntresourceusage.pl";

my $emd_root;
my $ifs_root_home;
my $ifs_oracle_home;
my $ifs_domain_name;
my $ifs_domain_type;

my %pids_table;

$emd_root = $ARGV[0];
$ifs_root_home = $ARGV[1];
$ifs_domain_name = $ARGV[2];
$ifs_domain_type = $ARGV[3];

$ifs_oracle_home = getOracleHome($ifs_root_home);

executePSCommand(); # required by getCPUUsageForPids()/getMemoryUsageForPids()

if (($^O =~ "Windows") || ($^O =~ "MSWin32"))  # windows platforms
{
    getResourceUsageForNT($emd_root, $ifs_oracle_home, $ifs_domain_name,
			  $ifs_domain_type);
}
else  # non-windows platforms
{
    getResourceUsage($emd_root, $ifs_oracle_home, $ifs_domain_name,
		     $ifs_domain_type);
}

# end of main

###########################################################################

# Get OS resource usage for the given iFS processes on UNIX platforms.
# Parameters
#   emd_root - the root of the emd tree
#   ifs_oracle_home - the oracle home of the iFS target
#   ifs_domain_name - the domain name of the iFS target
#   ifs_domain_type - the domain type of the iFS target
# Return
#   For each process type, return
#    "em_result=[process_name]|[cpu_util]|[cpu_other]|[cpu_idle]|
#    [memory_usage]|[memory_usge_percent]|[memory_total]"
#
sub getResourceUsage
{
    my ($emd_root, $ifs_oracle_home, $ifs_domain_name, $ifs_domain_type) = @_;
    my $ps_command = getPsCommand();
    my $processors = getNumCpus();
    my $process_info;
    my $cpu_usage;
    my $cpu_total;
    my $memory_usage;
    my $cpu_usage_sum = 0;
    my $memory_usage_sum = 0;
    my $cpu_total_sum = 0;
    my $cpu_total_count = 0;
    my $oc4j_key = "|";
    my $oc4j_instance1;
    my $oc4j_instance2;

    # get total physical memory on the system
    my $memory_metrics = getPhysicalMemoryMetrics($emd_root);
    my ($memory_total, $memory_free_raw, $page_size) =
	split('\|', $memory_metrics);

    # check Domain Controller processes
    $process_info =
	getProcessInfo("$ps_command | grep java | grep \"$ifs_oracle_home\" | grep oracle.ifs.management.domain.DomainController | grep $ifs_domain_name | grep -v grep", 
		       $emd_root, $processors);
    ($cpu_usage, $cpu_total, $memory_usage) = split('\|', $process_info);
    $cpu_usage_sum += $cpu_usage;
    if ($cpu_total > 0)
    {
	$cpu_total_sum += $cpu_total;
	$cpu_total_count += 1;
    }
    $memory_usage_sum += $memory_usage;
    formatOutput("Domain Controller", $cpu_usage, $cpu_total,
		 $memory_usage, $memory_total, $processors);

    # check Node Guardian processes
    $process_info =
	getProcessInfo("$ps_command | grep java | grep \"$ifs_oracle_home\" | grep oracle.ifs.management.domain.NodeGuardian | grep $ifs_domain_name | grep -v grep", 
		       $emd_root, $processors);
    ($cpu_usage, $cpu_total, $memory_usage) = split('\|', $process_info);
    $cpu_usage_sum += $cpu_usage;
    if ($cpu_total > 0)
    {
	$cpu_total_sum += $cpu_total;
	$cpu_total_count += 1;
    }
    $memory_usage_sum += $memory_usage;
    formatOutput("Node Guardian", $cpu_usage, $cpu_total,
		 $memory_usage, $memory_total, $processors);
    
    # check Node Manager processes
    $process_info =
	getProcessInfo("$ps_command | grep java | grep \"$ifs_oracle_home\" | grep oracle.ifs.management.domain.NodeManager | grep $ifs_domain_name | grep -v grep", 
		       $emd_root, $processors);
    ($cpu_usage, $cpu_total, $memory_usage) = split('\|', $process_info);
    $cpu_usage_sum += $cpu_usage;
    if ($cpu_total > 0)
    {
	$cpu_total_sum += $cpu_total;
	$cpu_total_count += 1;
    }
    $memory_usage_sum += $memory_usage;
    formatOutput("Node Manager", $cpu_usage, $cpu_total,
		 $memory_usage, $memory_total, $processors);

    # check OC4J processes
    if ($ifs_domain_type eq "cmsdk")
    {
	$oc4j_key="OC4J_iFS_cmsdk|OC4J_iFS_webstarterapp";
    }
    elsif ($ifs_domain_type eq "files")
    {
	$oc4j_key="OC4J_iFS_files|";
    }
    elsif ($ifs_domain_type eq "websites")
    {
	$oc4j_key="OC4J_iFS_websites|";
    }

    ($oc4j_instance1, $oc4j_instance2) = split('\|', $oc4j_key);

    if ($oc4j_instance1 ne "")
    {
	# to be compatible with iAS902* OC4J command line, we cannot use
	# iAS904 specific patterns such as oracle.home and
	# oracle.oc4j.instancename
	$process_info =
	    getProcessInfo("$ps_command | grep java | grep \"oracle.ons.oraclehome=$ifs_oracle_home\" | grep \"oracle.ons.indexid=$oc4j_instance1\" | grep -v grep", 
			   $emd_root, $processors);
	($cpu_usage, $cpu_total, $memory_usage) = split('\|', $process_info);
	$cpu_usage_sum += $cpu_usage;
	if ($cpu_total > 0)
	{
	    $cpu_total_sum += $cpu_total;
	    $cpu_total_count += 1;
	}
	$memory_usage_sum += $memory_usage;
	formatOutput("$oc4j_instance1", $cpu_usage, $cpu_total,
		     $memory_usage, $memory_total, $processors);
    }
    if ($oc4j_instance2 ne "")
    {
	$process_info =
	    getProcessInfo("$ps_command | grep java | grep \"oracle.ons.oraclehome=$ifs_oracle_home\" | grep \"oracle.ons.indexid=$oc4j_instance2\" | grep -v grep", 
			   $emd_root, $processors);
	($cpu_usage, $cpu_total, $memory_usage) = split('\|', $process_info);
	$cpu_usage_sum += $cpu_usage;
	if ($cpu_total > 0)
	{
	    $cpu_total_sum += $cpu_total;
	    $cpu_total_count += 1;
	}
	$memory_usage_sum += $memory_usage;
	formatOutput("$oc4j_instance2", $cpu_usage, $cpu_total,
		     $memory_usage, $memory_total, $processors);
    }

    # finally, a sum total of all the iFS processes
    if ($cpu_total_count > 0)
    {	
        $cpu_total = $cpu_total_sum/$cpu_total_count;
    }
    else
    {
        $cpu_total = $cpu_total_sum;
    }
    formatOutput("Total", $cpu_usage_sum, $cpu_total,
		 $memory_usage_sum, $memory_total, $processors);
}


# Get CPU usage and memory usage for the given processes.
# Parameters
#   ps_command - ps command to get the full command line
#   emd_root - the root of the emd tree
#   processors - the number of processors on the system
# Return
#   <cpuUsage>|<cpuTotal>|<memoryUsage>
#
sub getProcessInfo
{
    my ($ps_command, $emd_root, $processors) = @_;

    # get total CPU time on the system
    my $cpu_total = getTotalCpuTime();

    # Execute the ps command to find the processes
    my $ps_output = `$ps_command`;
    my @lines = split("\n", $ps_output);
    if ($#lines > -1) 
    {
	# Get the first line
	my @tokens = split(" ", $lines[0]);
	my $pid_list = $tokens[1];

	# check for many pids
	if ($#lines > 0) 
	{
	    for (my $i = 1; $i <= $#lines; $i ++) 
	    {
		@tokens = split(" ", $lines[$i]);
		$pid_list = "$pid_list,$tokens[1]";
	    }
        }        

	# Get the resource information for the pids
	my ($cpu_usage, $memory_usage) =
	    split('\|',
		  getProcessInfoFromPids($pid_list, $emd_root, $processors));

	return "$cpu_usage|$cpu_total|$memory_usage";
    }
    else # no process returned by the ps command
    {
	return "0|$cpu_total|0";
    }
}


# Get CPU usage and memory usage from pids.
# Parameters
#   pid_list - comma separated pid list
#   emd_root - the root of the emd tree
#   processors - the number of processors on the system
# Return
#   <cpuUsage>|<memoryUsage>
#
sub getProcessInfoFromPids
{
    my ($pid_list, $emd_root, $processors) = @_;

    # If we have some pids, get the cpu and memory
    my $cpu = 0; 
    my $memory = 0;
    if ($pid_list ne "") 
    {
      $cpu = getCPUUsageForPids($pid_list, $processors); 
      $memory = getMemoryUsageForPids($pid_list, $emd_root);
      if ($memory == -1) 
      {
        $memory = 0;
      }
    } 
    
    return "$cpu|$memory";
}


# Get the Oracle Home from IfsRootHome by stripping the trailing "/ifs"
# Parameters:
# ifs_root_home - the IfsRootHome that is in the format of "<oracle_home>/ifs"
sub getOracleHome
{
    my $ifs_root_home = shift(@_);

    # replace any '\' to '/'
    $ifs_root_home =~ s/\\/\//g;

    my $ifs_oracle_home = $ifs_root_home;

    my $pos = rindex($ifs_root_home, "/ifs");
    if ($pos > 0)
    {
	$ifs_oracle_home = substr($ifs_root_home, 0, $pos);
    }

    return $ifs_oracle_home;
}


# Format the output of OS resource usage data.
# Parameters
#   process_name - the name that identifies a particular type of iFS process
#   cpu_usage - the CPU utilization for the given processes
#   cpu_total - the total CPU utilization on the system
#   memory_usage - the memory utilization for the given processes
#   memory_total - the total memory on the system
#   processors - the number of CPUs on the system
# Return
#   "em_result=<process_name>|<cpu_usage>|<cpu_other>|<cpu_idle>|
#    <memory_usage>|<memory_usage_percent>|<memory_total>"
#
sub formatOutput
{
    my ($process_name, $cpu_usage, $cpu_total, $memory_usage, $memory_total, $processors) = @_;

    my $cpu_component = $cpu_usage * $processors;
    if ($cpu_total < $cpu_component)
    {
	$cpu_total = $cpu_component;
    }

    my $cpu_other = ($cpu_total/$processors) - $cpu_usage;
    if (($cpu_usage + $cpu_other) > 100) 
    {
	$cpu_other = 100 - $cpu_usage;
    }
    my $cpu_idle = 100 - $cpu_usage - $cpu_other;

    if ($memory_usage < 0)
    {
	$memory_usage = 0;
    }

    my $memory_usage_percent = ($memory_usage*100)/$memory_total;
	
    printf("em_result=%s|%.2f|%.2f|%.2f|%.2f|%.2f|%.2f\n",
	   $process_name, $cpu_usage, $cpu_other, $cpu_idle,
	   $memory_usage, $memory_usage_percent, $memory_total);
}

###########################################################################

# Get OS resource usage for the given iFS processes on Windows platforms.
# Parameters
#   emd_root - the root of the emd tree
#   ifs_oracle_home - the oracle home of the iFS target
#   ifs_domain_name - the domain name of the iFS target
#   ifs_domain_type - the domain type of the iFS target
# Return
#   For each process type, return
#    "em_result=[process_name]|[cpu_util]|[cpu_other]|[cpu_idle]|
#    [memory_usage]|[memory_usge_percent]|[memory_total]"
#
sub getResourceUsageForNT
{
    my ($emd_root, $ifs_oracle_home, $ifs_domain_name, $ifs_domain_type) = @_;
    my $processors = getNumCpus();
    my $transformed_ifs_domain_name = transformDomainName($ifs_domain_name);
    my $process_info;
    my $cpu_usage;
    my $cpu_total;
    my $memory_usage;
    my $cpu_usage_sum = 0;
    my $memory_usage_sum = 0;
    my $cpu_total_sum = 0;
    my $cpu_total_count = 0;
    my $oc4j_key = "|";
    my $oc4j_instance1;
    my $oc4j_instance2;

    # get the process ids for the domain instance
    getGroupedPidsForDomain($ifs_oracle_home, $transformed_ifs_domain_name,
			    $ifs_domain_type);

    # get total physical memory on the system
    my $memory_metrics = getPhysicalMemoryMetrics($emd_root);
    my ($memory_total, $memory_free_raw, $page_size) =
	split('\|', $memory_metrics);

    # check Domain Controller processes
    $process_info = getNTProcessInfo($pids_table{"Domain Controller"},
				     $emd_root, $processors);
    ($cpu_usage, $cpu_total, $memory_usage) = split('\|', $process_info);
    $cpu_usage_sum += $cpu_usage;
    if ($cpu_total > 0)
    {
	$cpu_total_sum += $cpu_total;
	$cpu_total_count += 1;
    }
    $memory_usage_sum += $memory_usage;
    formatOutput("Domain Controller", $cpu_usage, $cpu_total,
		 $memory_usage, $memory_total, $processors);

    # check Node Guardian processes
    $process_info = getNTProcessInfo($pids_table{"Node Guardian"},
				     $emd_root, $processors);
    ($cpu_usage, $cpu_total, $memory_usage) = split('\|', $process_info);
    $cpu_usage_sum += $cpu_usage;
    if ($cpu_total > 0)
    {
	$cpu_total_sum += $cpu_total;
	$cpu_total_count += 1;
    }
    $memory_usage_sum += $memory_usage;
    formatOutput("Node Guardian", $cpu_usage, $cpu_total,
		 $memory_usage, $memory_total, $processors);
    
    # check Node Manager processes
    $process_info = getNTProcessInfo($pids_table{"Node Manager"},
				     $emd_root, $processors);
    ($cpu_usage, $cpu_total, $memory_usage) = split('\|', $process_info);
    $cpu_usage_sum += $cpu_usage;
    if ($cpu_total > 0)
    {
	$cpu_total_sum += $cpu_total;
	$cpu_total_count += 1;
    }
    $memory_usage_sum += $memory_usage;
    formatOutput("Node Manager", $cpu_usage, $cpu_total,
		 $memory_usage, $memory_total, $processors);

    # check OC4J processes
    if ($ifs_domain_type eq "cmsdk")
    {
	$oc4j_key="OC4J_iFS_cmsdk|OC4J_iFS_webstarterapp";
    }
    elsif ($ifs_domain_type eq "files")
    {
	$oc4j_key="OC4J_iFS_files|";
    }
    elsif ($ifs_domain_type eq "websites")
    {
	$oc4j_key="OC4J_iFS_websites|";
    }

    ($oc4j_instance1, $oc4j_instance2) = split('\|', $oc4j_key);

    if ($oc4j_instance1 ne "")
    {
	$process_info = getNTProcessInfo($pids_table{$oc4j_instance1},
					 $emd_root, $processors);
	($cpu_usage, $cpu_total, $memory_usage) = split('\|', $process_info);
	$cpu_usage_sum += $cpu_usage;
	if ($cpu_total > 0)
	{
	    $cpu_total_sum += $cpu_total;
	    $cpu_total_count += 1;
	}
	$memory_usage_sum += $memory_usage;
	formatOutput("$oc4j_instance1", $cpu_usage, $cpu_total,
		     $memory_usage, $memory_total, $processors);
    }
    if ($oc4j_instance2 ne "")
    {
	$process_info = getNTProcessInfo($pids_table{$oc4j_instance2},
					 $emd_root, $processors);
	($cpu_usage, $cpu_total, $memory_usage) = split('\|', $process_info);
	$cpu_usage_sum += $cpu_usage;
	if ($cpu_total > 0)
	{
	    $cpu_total_sum += $cpu_total;
	    $cpu_total_count += 1;
	}
	$memory_usage_sum += $memory_usage;
	formatOutput("$oc4j_instance2", $cpu_usage, $cpu_total,
		     $memory_usage, $memory_total, $processors);
    }

    # finally, a sum total of all the iFS processes
    if ($cpu_total_count > 0)
    {	
        $cpu_total = $cpu_total_sum/$cpu_total_count;
    }
    else
    {
        $cpu_total = $cpu_total_sum;
    }
    formatOutput("Total", $cpu_usage_sum, $cpu_total,
		 $memory_usage_sum, $memory_total, $processors);
}

# Gather all the process ids stored in .pid files for this domain instance
# Parameters
#   oracle_home: the oracle home
#   transformed_domain_name: the tranformed domain name - no prefix ifs:// and
#                            ' ' and ':' and '.' being replaced with '_'
#   domain_type: the type of the domain: cmsdk or files
# Return
#   no explicit return, it populates the pids_table hash with process ids
#   grouped by process types
sub getGroupedPidsForDomain
{
    my ($oracle_home, $transformed_domain_name, $domain_type) = @_;
    my $dir =
	$oracle_home . "/ifs/" . $domain_type . "/log/" . $transformed_domain_name;
    my $basename;
    my $filename;
    my $oc4j_instance_name;
    my $pid;

    # reset the pids_table first
    %pids_table =
    {
	"Domain Controller", "",
	"Node Guardian", "",
	"Node Manager", "",
	"OC4J_iFS_cmsdk", "",
	"OC4J_iFS_files", "",
	"OC4J_iFS_websites", "",
	"OC4J_iFS_webstarterapp", ""
    };

    # replace any '\' to '/'
    $dir =~ s/\\/\//g;

    if (!opendir(PIDDIR, $dir))
    {
	EMD_PERL_ERROR("Cannot list directory $dir: $!");
	return;
    }

    while ($basename = readdir(PIDDIR))
    {
	$filename = $dir . '/' . $basename;

	if ((-f $filename) && ($basename =~ /\.pid$/))
	{
	    $pid = getProcessIdFromPidFile($filename);
	    if ($pid ne "")
	    {
		if ($basename eq "DomainController\.pid")
		{
		    if ($pids_table{"Domain Controller"} eq "")
		    {
			$pids_table{"Domain Controller"} = $pid;
		    }
		    else
		    {
			$pids_table{"Domain Controller"} .= ',' . $pid;
		    }
		}
		elsif ($basename =~ /OC4J_iFS_(.)*_HTTPNodeGuardian\.pid$/)
		{
		    $oc4j_instance_name = $&;
		    $oc4j_instance_name =~ s/_HTTPNodeGuardian\.pid$//;
		    if ($pids_table{$oc4j_instance_name} eq "")
		    {
			$pids_table{$oc4j_instance_name} = $pid;
		    }
		    else
		    {
			$pids_table{$oc4j_instance_name} .= ',' . $pid;
		    }
		}
		elsif ($basename =~ /_NodeGuardian\.pid$/)
		{
		    if ($pids_table{"Node Guardian"} eq "")
		    {
			$pids_table{"Node Guardian"} = $pid;
		    }
		    else
		    {
			$pids_table{"Node Guardian"} .= ',' . $pid;
		    }
		}
		elsif ($basename =~ /_NodeManager\.pid$/)
		{
		    if ($pids_table{"Node Manager"} eq "")
		    {
			$pids_table{"Node Manager"} = $pid;
		    }
		    else
		    {
			$pids_table{"Node Manager"} .= ',' . $pid;
		    }
		}
	    }
	}
    }

    closedir(PIDDIR);
}


# Get CPU usage and memory usage for the given NT processes.
# Parameters
#   pid_list - the process ids, separated by ','
#   emd_root - the root of the emd tree
#   processors - the number of processors on the system
# Return
#   <cpuUsage>|<cpuTotal>|<memoryUsage>
#
sub getNTProcessInfo
{
    my ($pid_list, $emd_root, $processors) = @_;

    # get total CPU time on the system
    my $cpu_total = getTotalCpuTime();

    if ($pid_list ne "") 
    {
	# Get the resource information for the pids
	my ($cpu_usage, $memory_usage) =
	    split('\|',
		  getProcessInfoFromPids($pid_list, $emd_root, $processors));

	return "$cpu_usage|$cpu_total|$memory_usage";
    }
    else # no process 
    {
	return "0|$cpu_total|0";
    }
}


##### FIXME: move the following utility subroutines to their own file #####

# Retrieve the process id from the given .pid file. If the .pid file contains
# more than one process id, we only care for the first one.
# Parameters
#   pid_filename: the absolute path of the .pid file
# Return
#   the pid
sub getProcessIdFromPidFile
{
    my $pid_filename = shift(@_);

    # checkPIDFile is defined in iasntresourceusage.pl
    my @pids = checkPIDFile($pid_filename);

    if (@pids && $#pids >= 0)
    {
	return $pids[0];
    }
    else
    {
	return "";
    }
}


# Transform the domain name by removing the prefix "ifs://" and replacing any
# ' ', ':', or '.' with '_'.
# Parameters
#   domain_name: the name of the domain, in the format of
#                ifs://<db-host>:<db-port>:<db-service>:<schema>
# Return
#   the transformed domain name, which could be used 
sub transformDomainName
{
    my $domain_name = shift(@_);

    # remove leading ifs://
    $domain_name =~ s/^ifs:\/\///;

    # replace any ' ', ':', or '.' with '_'
    $domain_name =~ s/[ :\.]/_/g;

    return $domain_name;
}

# EOF
