# $Header: ocal_config.pl 03-aug-2005.20:39:25 tjaiswal Exp $
#
# ocal_config.pl
# 
# Copyright (c) 2003, 2005, Oracle. All rights reserved.  
#
# NAME
# ocal_config.pl - the perl script collects the OCAL config data
#
# DESCRIPTION
# This script collects the OCAL config data
#
# NOTES
#
# MODIFIED   (MM/DD/YY)
#    rayleung    02/22/06 - Add case handling file not found
#    tjaiswal    01/10/06 - Creation
# 

use Getopt::Long; # set up to accept user input

my $OSNAME = $^O;
my $l_oracle_home;
my $l_file_name;
my $l_full_file_name;
my $l_scope;

# get command line parameters
GetOptions(
            "OracleHome=s" => \$l_oracle_home,
            "fileName=s" => \$l_file_name,
            "scope=s" => \$l_scope );

# NOTE: l_file_name is the file path relative to the oracle home. So, for example, this will
#       will be something like: ocal/misc/unison.ini. If this perl script is invoked from
#       a metadata file, then l_file_name will always have "/" as that is how the relative
#       path is sent from the metadata file. However, this perl file could also be 
#       invoked from other sources. Hence, process for "/" anyways i.e for NT convert
#       any occurrences of "/" in l_file_name to "\".
#       Also, note that scope value will be available only if the file name is ocst.conf
#       The value for scope will be either "general" OR "device"

$l_full_file_name = getCompleteFileName();

parseConfFile();

sub getCompleteFileName
{
    # create a local variable as the file path will be changed based on the platform.
    # $fname will hold the modified file path name    
    my( $fname ) = $l_file_name;
    
    # variable to store the full file path name 
    my( $fullname );
    
    if(($OSNAME eq "MSWin32") or ($OSNAME eq "Windows_NT")) 
    {
        $fname =~ s/\//\\/g;
        $fullname = $l_oracle_home . "\\" . $fname;
    } 
    else 
    {
        $fname =~ s/\\/\//g;
        $fullname = $l_oracle_home . "/" . $fname;
    }
    
    return $fullname;
}

sub parseConfFile
{
    my( $category_name );
    
    # first check to see if the file exists
    if( ! -e $l_full_file_name )
    {
        print "em_error=Cannot open file \"".$l_full_file_name."\"\n";
        exit 0;
    }

    # read thru the file and print out the contents of the file in the form:
    # em_result=<category name>|<property name>|<property value>
    
    open( CONFFILE, $l_full_file_name );
    
    while( <CONFFILE> )
    {
        my( $current_line ) = $_;
        
        # chop the current line
        chomp( $current_line );
        
        # trim the leading and trailing spaces
        $current_line =~ s/^\s+|\s+$//g;
                
        # ignore comments - lines starting with "#"
        if( $current_line =~ /^#/ )
        {
            next;
        }
        
        # ignore empty line
        if( length( $current_line ) == 0 )
        {
            next;
        }
        
        # if the line begins with a "[" and ends with a "]", then this is a category
        if( $current_line =~ /^\[/ && $current_line =~ /\]$/ )
        {
            # remove the leading "["
            $current_line =~ s/^\[//;
            
            # remove the trailing "]"
            $current_line =~ s/\]$//;

            # remove the leading the trailing spaces
            $current_line =~ s/^\s+|\s+$//g;                     
            
            $category_name = $current_line;
            #print "CATEGORY FOUND: $category_name\n";
            next;
        }
        
        # if we have reached this point then this is not a line with a comment, not an
        # empty line, not a line that has the category. This line should therefore have
        # the property name and value
        
        # check if this line has a property name and value. This is true if the line has a
        # equals sign. 
        # NOTE: the pattern being search is: line with 1 or more characters(^.+), 
        # followed by 0 or more spaces(\s*), followed by "="(=), followed by 0 or more
        # spaces(\s*), followed by 0 or more characters(.*). Note that we are using 0 or 
        # more characters at the end to also include lines like: swv=
        # Such a line has 1 or more characters, 0 or more spaces, "=", 0 or more spaces,
        # 0 or more characters
        next unless $current_line =~ /^.+\s*=\s*.*/;
        
        # split the current line based on "="
        ( $pname, $pvalue ) = split( /=/, $current_line );
        
        # trim the leading and trailing spaces
        $pname =~ s/^\s+|\s+$//g;
        
        # trim the leading and trailing spaces
        $pvalue =~ s/^\s+|\s+$//g;        
        
        # now print the value in the form: em_result=<category name>|<property name>|<property value>
        # if file name is ocst.conf and scope is general, print only if category name is ACE
        # or ocst
        # if file name is ocst.conf and scope is device, print only if category name is not 
        # ACE and not ocst
        # for other file names, print all lines.
        # this is being done because ocst.conf file has a large number of parameters -
        # only ACE and ocst sections have non-device parameters. Rest all have device 
        # parameters. Hence, this seperation is being made.
        if( $l_full_file_name =~ /ocst.conf/ && $l_scope =~ /^general$/ )
        {
            # print only if category name is ACE or ocst
            if( $category_name =~ /^ACE$/ || $category_name =~ /^ocst$/ )
            {
                print "em_result=$category_name|$pname|$pvalue"."\n";
            }            
        }
        elsif( $l_full_file_name =~ /ocst.conf/ && $l_scope =~ /^device$/ )
        {
            # print only if category name is "not" ACE or ocst
            if( !($category_name =~ /^ACE$/) && !($category_name =~ /^ocst$/) )
            {
                print "em_result=$category_name|$pname|$pvalue"."\n";
            }            
        }
        else
        {
            # print always
            print "em_result=$category_name|$pname|$pvalue"."\n";
        }        
    }
    
    close( CONFFILE );
}
