# $Header: iasresourceusage.pl 15-may-2003.17:35:03 klmichae Exp $
#
# iasresourceusage.pl
# 
# Copyright (c) 2002, 2003, Oracle Corporation.  All rights reserved.  
#
#    NAME
#      iasresourceusage.pl - gets cpu and memory usage for a list of pids
#
#    DESCRIPTION
#      iasresourceusage.pl 
#
#    returns: 
#      em_result=<cpuUsage>|<cpuOther>|<cpuIdle>|<physicalMemoryUsage>|
#      <totalPhysicalMemoryUsage>|<freePhysicalMemoryRaw>|<pageSize>|
#      <physicalMemoryPercentage>
#
#      where:
#         <cpuUsage> percentage of CPU time being used by the specified pid list
#         <cpuOther> percentage of CPU time being used by other system processes
#         <cpuIdle> percentage of idle CPU time 
#         <physicalMemoryUsage> amount of physical memory (MB) being used by  
#          specified pid list
#         <totalPhysicalMemoryUsage> amount of physical memory (MB) on the system. 
#         <freePhysicalMemoryRaw> Raw value of free memory 
#         <pageSize> Memory page size
#         <physicalMemoryPercentage> percentage of physical memory (MB) being  
#          used by specified pid list
#
#    NOTES
#
#
#    MODIFIED   (MM/DD/YY)
#    sawadhwa   08/29/03 - Port cpu usage changes to HPUX
#    klmichae   05/15/03 - remove calls to opmnadmin
#    vsekuboy   04/29/03 - Changes for HP and linux
#    klmichae   03/31/03 - account for multiple cpus
#    klmichae   03/21/03 - handle non-existent processes
#    jmcclung   12/12/02 - rename nmp to nmb
#    klmichae   12/02/02 - call pmap program
#    klmichae   11/18/02 - report an error for ias installed as different user
#    klmichae   10/23/02 - set oracle_home before calling webcachectl
#    klmichae   10/15/02 - remove use of meminfo
#    klmichae   10/14/02 - set oracle home environment variable
#    klmichae   08/23/02 - klmichae_bug-2519611
#    klmichae   08/19/02 - Initial revision

use strict;
use Fcntl ':mode';

my $osType=" ";
    
# Determine the OSType
sub computeOSType
{
  # See what OS we have so that we can determine the algorithm to get info
  my $os;
  my $ver;
  chomp ($os = `uname -s`) or die "Failed to run the uname command";
  SWITCH: {
    $os eq "SunOS" && do
    {
      chomp ($ver = `uname -r`);
      if ( $ver !~ /^4./ ) {
         $osType = "Sol";
         last SWITCH;
      }
    };
    $os eq "HP-UX" && do
    {
      $osType = "HP";
      $ENV{UNIX95} = "XPG4";
      last SWITCH;
    };
    $os eq "Linux" && do
    {
      $osType = "LNX";
      last SWITCH;
    };
    die "Unsupported Operating System\n";
  }
}

# Get physical memory metrics for the system.  Return them in the form: 
# {TotalMemoryInMB}|{FreeMemoryRaw}|{PageSize}
# Parameters:
# emdRoot the root directory for the emd
sub getPhysicalMemoryMetrics
{
  my $emdroot = shift(@_);

  my $mem_output=" "; 

  # get the osLoad info from nmupm.  The result will be
  # em_result={cpuLoad}|{PageScanRaw}|{noOfProcs}|{noOfUsers}|{transfers}|
  # {idleTimeRaw}|{userTimeRaw}|{systemTimeRaw}|{waitTimeRaw}|{pageSize}|
  # {realMem}|{freeMemRaw}|{usedSwapRaw}|{freeSwapRaw}

  if ( $osType eq "LNX" ) {
	$mem_output = `$emdroot/perl/bin/perl $emdroot/sysman/admin/scripts/osLoad.pl`;
  }
  else
  {
	$mem_output = `$emdroot/bin/nmupm osLoad`;
  }
  my @mem_lines = split( '\|', $mem_output );

  # pull out pageSize, realMem and freeMemRaw
  my $totalMem = $mem_lines[10]/1024;
  return "$totalMem|$mem_lines[11]|$mem_lines[9]";
}

# Get a ps command that returns the the username followed by the pid
# plus the full command line.  Note that this command is platform
# specific.
sub getPsCommand
{
  my $os;
  chomp ($os = `uname -s`) or die "Failed to run the uname command";
  SWITCH: {
    $os eq "SunOS" && do
    {
      return "/usr/ucb/ps auxww";
    };
    $os eq "HP-UX" && do
    {
      $ENV{UNIX95} = "XPG4";
      return "/bin/ps -efxo user,pid,pcpu,comm";
    };
    $os eq "Linux" && do
    {
      return "/bin/ps auxww";
    };
    die "Unsupported Operating System\n";
  }
}

# Get the pid list for the Http Server 
# parameters:
#   oracle_home - the oracle home in which the iAS instance is running
sub getHttpPids 
{
  my $oracle_home = shift(@_);
  my $pidList = "";

  # The ps command is platform specific
  my $ps_command = getPsCommand();

  # Get the apache pids from ps
  # The output will be in the following form:
  # USER       PID %CPU %MEM   SZ  RSS TT       S    START  TIME COMMAND
  my $ps_output = `$ps_command | grep "[h]ttpd" | grep $oracle_home`;

  # Get the ps output lines and walk through them Looking for the pids
  my $line;
  my @lines = split( "\n", $ps_output );
  my @tokens;
  foreach $line (@lines)
  {
    @tokens = split( " ", $line );
    # Get the pid; it is the second token
    if( $pidList eq "" )
    {
      $pidList = "$tokens[1]";
    } else {
      $pidList = "$pidList,$tokens[1]";
    }
  } # end foreach line

  return $pidList;

}

# Compute the resource usage of the OC4J instance. This information includes
# cpu and physical memory.
# parameters:
#   emdroot - the emd root directory
#   oracle_home - the oracle home in which the iAS instance is running
#   oc4j_name - the oc4j instance name (if empty, return all instances)
sub computeOc4jResourceUsage
{
  # Get the arguments
  my $emdroot = shift(@_);
  my $oracle_home = shift(@_);
  my $oc4j_name = shift(@_);
 
  # To get a single oc4j look for "".  Get get all of them look for
  # "oracle.ons.instanceid"
  my $oc4j_key;
  if( $oc4j_name eq "" )
  {
    $oc4j_key = "oracle.ons.instanceid";
  } else {
    $oc4j_key = "#$oc4j_name#";
  }

  # Look for the specified oc4j Instance 
  getProcessInfoFromPsOutput( $emdroot, 
    "grep \"[j]ava\" | grep \"#$oc4j_name#\" | grep $oracle_home" );

}

# Compute the resource usage of the HTTP Server. This information includes
# cpu and physical memory.
# parameters:
#   emdroot - the emd root directory
#   oracle_home - the oracle home in which the iAS instance is running
sub computeHttpResourceUsage
{
  # Get the arguments
  my $emdroot = shift(@_);
  my $oracle_home = shift(@_);

  # Look for all apache processes
  getProcessInfoFromPsOutput( $emdroot, "grep \"$oracle_home/[A]pache/Apache\"" );
}

# Compute the resource usage of the Web Cache Server. This information includes
# cpu and physical memory.
# parameters:
#   emdroot - the emd root directory
#   oracle_home - the oracle home in which the iAS instance is running
sub computeWebcacheResourceUsage
{
  # Get the arguments
  my $emdroot = shift(@_);
  my $oracle_home = shift(@_);

  # Look for all webcache processes
  getProcessInfoFromPsOutput( $emdroot, "grep \"$oracle_home/[w]ebcache/bin/webcache\"" );
}

# Compute the resource usage of the Application Server. This information includes
# cpu and physical memory.
# parameters:
#   emdroot - the emd root directory
#   oracle_home - the oracle home in which the iAS instance is running
sub computeIasResourceUsage
{
  # Get the arguments
  my $emdroot = shift(@_);
  my $oracle_home = shift(@_);

  # Look for all oc4j, http server and webcache processes
  my $grep_command = "egrep \"$oracle_home.+[j]ava.+oracle\.ons\.instanceid|$oracle_home/[A]pache/Apache|$oracle_home/[w]ebcache/bin/webcache\"";

  # Look for the specified oc4j Instance
  getProcessInfoFromPsOutput( $emdroot, $grep_command );
}


# Get resource usage information from process info
# parameters
#   emdroot - the emd root directory
#   grep_command - the command to identify the desired processes from ps 
sub getProcessInfoFromPsOutput
{
  my $emdroot = shift(@_);
  my $grep_command = shift(@_);
  my $process_name = shift(@_);
  
  # The ps command is platform specific
  computeOSType();
  my $ps_command = getPsCommand();

  # Look for all the required processes
  # The output will be in the following form:
  # USER       PID %CPU %MEM   SZ  RSS TT       S    START  TIME COMMAND
  my $ps_output = `$ps_command | $grep_command`; 

  # Get the ps output lines 
  my $line;
  my @lines = split( "\n", $ps_output );
 
  # compute the cpu usage for the processes
  my $cpu_component = getCpuFromPsOutput( \@lines );
#print "KLM: cpu = $cpu_component\n";

  # get the total cpu usage on the system
  my $cpu_util_total = getTotalCpuTime();
#print "KLM: cpu = $cpu_util_total\n";
  if( $cpu_util_total < $cpu_component ) {
    $cpu_util_total = $cpu_component;
  }

  # Get the number of the processors on the
  # machine so we can compute the true cpu time
  my $num_cpus = getNumCpus($emdroot);
  my $cpu_util = $cpu_component/$num_cpus;
  my $cpu_other = ($cpu_util_total/$num_cpus) - $cpu_util;
  if ( ($cpu_util + $cpu_other) > 100 ) 
  {
    $cpu_other = 100 - $cpu_util;
  }
  my $cpu_idle = 100 - $cpu_util - $cpu_other;

  # compute the memory usage for the processes
  my $physical_mem_kb = getMemoryFromPsOutput( $emdroot, \@lines );
#print "KLM: memory = $physical_mem_kb\n";

  # Get total memory usage for the system
  my $memory_metrics = getPhysicalMemoryMetrics( $emdroot );
  my ($physical_mem_total, $physical_mem_free_raw, $pagesize) =
    split( '\|', $memory_metrics );

  # Convert the memory number from KB to MB
  my $physical_mem = $physical_mem_kb/1024;
  my $physical_mem_percent = ($physical_mem*100)/$physical_mem_total;
  $physical_mem_free_raw = $physical_mem_free_raw/(1024*1024);
  # Create the results:
  #print "em_result=$cpu_util|$cpu_other|$cpu_idle|$physical_mem|$physical_mem_total|$physical_mem_free_raw|$pagesize|$physical_mem_percent\n";
  printf ("em_result=%s|%.2f|%.2f|%.2f|%.2f|%.2f|%.2f|%d|%.2f\n",$process_name,$cpu_util,$cpu_other,$cpu_idle,$physical_mem,$physical_mem_total,$physical_mem_free_raw,$pagesize,$physical_mem_percent);


}

# Get the cpu usage for a list of processes
# parameters
#  ps_lines - an array of lines with ps output
#
sub getCpuFromPsOutput
{
  my $ps_lines = shift(@_);

  my $cpu = 0;

  my $i;
  my @tokens;
#print "KLM: number pids = $#$ps_lines\n";
  for $i (0 .. $#$ps_lines)
  {
    @tokens = split( " ", $ps_lines->[$i] );
    $cpu = $cpu + $tokens[2];
  } # end $i

  return $cpu;
}

# Get the total cpu time being used on the system
# This will return the average amount of cpu being
# used during the last minute
sub getTotalCpuTime
{
  # Execute the uptime command.  The output could look like:
  #   6:23am  up 13 days, 21:32,  3 users,  load average: 1.31, 1.30, 1.32
  #  11:50am  up 23:49,  5 users,  load average: 0.06, 0.09, 0.11

  my $uptime_output = `uptime`;

  # pull out the cpu column
  my @tokens = split( " ", $uptime_output );

#print "KLM: uptime output = $uptime_output\n";

  # Count backward two tokens from the end of the list.
  # remove the trailing comma and change it from a fraction to a percentage
  my $cpu = substr( $tokens[$#tokens - 2], 0, -1 );
  return $cpu*100;
}

# Get the memory usage for a list of processes
# parameters
#   emdroot - the emd root directory
#   ps_lines - an array of lines with ps output
#
sub getMemoryFromPsOutput
{
  my $emdroot = shift(@_);
  my $ps_lines = shift(@_);

  my $memory;
  my $i;
  my @tokens;
  my %processHash = ();
  my $pidList = "";
  my $key;

  # walk through all the pids. We only want count processes, not
  # threads, so assume that any process with the same vsz, rss and
  # command all represent the same process
  for $i (0 .. $#$ps_lines)
  { 
    @tokens = split( " ", $ps_lines->[$i] );

    # On Linux the ps command returns process and threads.  But, we only want 
    # count processes, so assume that any process with the same vsz, rss and
    # command all represent the same process
    if ($osType eq "LNX") {

      # use vsz**rss**command as the key
      $key = "$tokens[4]**$tokens[5]**$tokens[10]";
      if (!defined $processHash{$key})
      {
         # This is a process that we should count
         # $tokens[1] is pid
         $pidList = "$pidList$tokens[1] ";

         # add it to the hash table, so we don't found it again
         $processHash{$key} = '1';
#print "KLM: pid=$tokens[1], key = $key\n";
      }
    } else {
      # just add the pid
      $pidList = "$pidList$tokens[1] ";
    } # end if Linux
 } # end $i

  # Create an array of pids and get the number usage
  my @pid_array = split( " ",$pidList );
#print "KLM: number of pids for memory = $#pid_array\n";
  return getPhysicalMemoryIncludingShared( \@pid_array, $emdroot );

}

# Get the pid list for the specified OC4J Instance
# parameters
#   oracle_home - the oracle home in which the iAS instance is running
#   oc4j_name - the oc4j instance name (if empty, return all instances)
sub getOc4jPids 
{
  # Get the arguments
  my $oracle_home = shift(@_);
  my $oc4j_name = shift(@_);
  
  my $pidList = "";

  # The ps command is platform specific
  my $ps_command = getPsCommand();

  # To get a single oc4j look for "".  Get get all of them look for
  # "oracle.ons.instanceid"
  my $oc4j_key;
  if( $oc4j_name eq "" )
  {
    $oc4j_key = "oracle.ons.instanceid";
  } else {
    $oc4j_key = "#$oc4j_name#";
  }

  # Get the oc4j pids from ps
  # The output will be in the following form:
  # USER       PID %CPU %MEM   SZ  RSS TT       S    START  TIME COMMAND
  my $ps_output = `$ps_command | grep "[j]ava" | grep "$oc4j_key" | grep $oracle_home`;

  # Get the ps output lines and walk through them Looking for the pids
  my $line;
  my @lines = split( "\n", $ps_output );
#print "KLM: num pids =  $#lines\n";
  my @tokens;
  foreach $line (@lines)
  {
    @tokens = split( " ", $line );
    # Get the pid; it is the second token
    if( $pidList eq "" )
    {
      $pidList = "$tokens[1]";
    } else {
      $pidList = "$pidList,$tokens[1]";
    }
  } # end foreach line

  return $pidList;
}

# Get the physical memory usage (KB) for a set of pids.  Make sure to account  
# for shared memory
#
# Parameters:
#  pid_list - an array of pids to account for
#  emdroot  - the root of the emd tree
# 
sub getPhysicalMemoryIncludingShared
{
  my $pid_list = shift(@_);
  my $emdroot = shift(@_);
  
  my $totalPrivate = 0;
  my $maxShared = 0;
  my $private = 0;
  my $shared;
  my $checkedNmb = 0;

  if ($osType eq "LNX") {
     return getLinuxPhysicalMemoryIncludingShared( $pid_list, $emdroot );
  }

  for my $i (0 .. $#$pid_list)
  {
    # Call nmb and get the shared and private memory usage
    my $pmap_output = `$emdroot/bin/nmb $pid_list->[$i] 2>&1`;

    # Make sure we don't get an error
    if( index( $pmap_output, "Error" ) == -1 ) 
    {

      # the output will be: {private-memory-in-kb} {shared-memory-in-kb}
      ($private, $shared) = split(  " ",  $pmap_output );
    
      # Add this to our totals
      $totalPrivate += $private;
      if( $shared > $maxShared ) {
        $maxShared = $shared;
      }      
    } else {
      # we could not parse the output, so make sure nmb is protected properly
      if( $checkedNmb == 0 ) 
      {
        validateNmbProtection($emdroot);
        $checkedNmb = 1;
      }
    } # end if( index( ...
   
  } # end foreach

  # The total memory usage is SUM(private) + MAX(shared)
  return $totalPrivate + $maxShared;

}

# Get the physical memory usage (KB) for a set of pids.  Make sure to account
# for shared memory.  This is for Linux only.
#
# Parameters:
#  pid_list - an array of pids to account for
#  emdroot            - the root of the emd tree
#
sub getLinuxPhysicalMemoryIncludingShared
{
  my $pid_list = shift(@_);

  my $totalPrivate = 0;
  my $maxShared = 0;
  my $private = 0;
  my $shared;

  for my $i (0 .. $#$pid_list)
  {
    $private = 0;
    $shared = 0;
    my $file = "/proc/$pid_list->[$i]/statm";
    if ( -e $file )
    {
      my $line = `cat $file`;
      my @fields=split(/\s+/,$line); #field separator is space
      # The output of this command is
      # {program-pages} {memory-pages} {shared-pages} {code-pages}
      # {data/stack-pages} {library-pages} {dirty-pages}
      $shared = $fields[2];
      $private = $fields[1] - $shared;
    }

    $totalPrivate += $private;
    if( $shared > $maxShared ) {
      $maxShared = $shared;
    }

  } # end foreach

  # compute the page size
  my $pagesize = `getconf PAGE_SIZE` or die "Failed to run getconf PAGE_SIZE";
  chomp($pagesize);

  # The total memory usage is SUM(private) + MAX(shared)
  # since this is in pages, divide by page size and then 
  # divide by 1024 to change from bytes to KB
  my $totalMemory = ($totalPrivate + $maxShared)*$pagesize/1024;
  return $totalMemory;

}

#
# Check is the protection on nmb is correct.  It should be owned by root and 
# have the sticky bit set.  If the protect is not correct, print an error and
# exit.
#
# Parameters:
#  emdroot            - the root of the emd tree
#
sub validateNmbProtection
{
  my $emdroot = shift(@_);

  # Get the stats for the file
  my @stat_array = stat("$emdroot/bin/nmb");

  # Make sure that it is owned by root (uid=0)
  if( $stat_array[4] == 0 ) 
  {
    # check that the sticky bit is set
    if( $stat_array[2] & S_ISUID ) {
      # The protection is correct
      return;
    }
  }

  # The protection is incorrect
  print "nmb does not have root access. Please see the release notes for information on setting the uid bit.\n";
  exit 1;
}

# Get the number of CPUs for the system 
#
# Parameters:
#  emdroot            - the root of the emd tree
#
sub getNumCpus()
{

  # The linux computation is different
  if( $osType eq "LNX" ) {
    my $cpu_stat = `cat /proc/stat | grep cpu | wc -l`  or die "Failed to get processor information";
    return $cpu_stat;
  }

  # Execute nmupm asking for CPU information.  It will return one row per CPU
  my $emdroot = shift(@_);
  my $cpu_info = `$emdroot/bin/nmupm osCpuUsage`;
  my @lines = split("\n", $cpu_info);
  if( $#lines > 0 )
  {
    return ($#lines+1);
  } else {
    return 1;
  }
  
}

1;

