# $Header: iasresourceusageforimeeting.pl 02-aug-2005.13:53:30 rayleung Exp $
#
# iasresourceusage.pl
# 
# Copyright (c) 2002, 2005, Oracle. All rights reserved.  
#
#    NAME
#      iasresourceusage.pl - gets cpu and memory usage for a list of pids
#
#    DESCRIPTION
#      iasresourceusage.pl 
#
#    returns: 
#      em_result=<cpuUsage>|<cpuOther>|<cpuIdle>|<physicalMemoryUsage>|
#      <totalPhysicalMemoryUsage>|<freePhysicalMemoryRaw>|<pageSize>|
#      <physicalMemoryPercentage>
#
#      where:
#         <cpuUsage> percentage of CPU time being used by the specified pid list
#         <cpuOther> percentage of CPU time being used by other system processes
#         <cpuIdle> percentage of idle CPU time 
#         <physicalMemoryUsage> amount of physical memory (MB) being used by  
#          specified pid list
#         <totalPhysicalMemoryUsage> amount of physical memory (MB) on the system. 
#         <freePhysicalMemoryRaw> Raw value of free memory 
#         <pageSize> Memory page size
#         <physicalMemoryPercentage> percentage of physical memory (MB) being  
#          used by specified pid list
#
#    NOTES
#
#
#    MODIFIED   (MM/DD/YY)
#    rayleung   08/02/05 - add codes the handle the division case when 
#                          $num_Processors is equal to zero 
#    rayleung   01/21/05 - fix zero divide 
#    prbhoj     12/21/04 - Fix the return values from osLoad script 
#    gmulchan   10/03/03 - gmulchan_cal_main_merge 
#    klmichae   05/15/03 - remove calls to opmnadmin
#    vsekuboy   04/29/03 - Changes for HP and linux
#    klmichae   03/31/03 - account for multiple cpus
#    klmichae   03/21/03 - handle non-existent processes
#    jmcclung   12/12/02 - rename nmp to nmb
#    klmichae   12/02/02 - call pmap program
#    klmichae   11/18/02 - report an error for ias installed as different user
#    klmichae   10/23/02 - set oracle_home before calling webcachectl
#    klmichae   10/15/02 - remove use of meminfo
#    klmichae   10/14/02 - set oracle home environment variable
#    klmichae   08/23/02 - klmichae_bug-2519611
#    klmichae   08/19/02 - Initial revision

use strict;
use Fcntl ':mode';

require "emd_common.pl";

# This is the output from the ps command: ps -A -o "ppid pid pcpu rss"
my @process_output_lines;
my $PPID_COL;
my $PID_COL;
my $CPU_COL;
my $MEM_COL;
my $osType=" ";
    
# Get the resource usage for the specified list of pids
# parameters
#  emdroot            - the root directory of the emd
#  all_pids           - the list of pids to check
#  component_pid_list - an array of pid lists broken down by component type
sub getResourceUsage
{
  #  Pull off the parameters
  my $emdroot = shift(@_);
  my $perldir = shift(@_);
  my $all_pids = shift(@_);
  my $component_pid_list = shift(@_);
  my $process_name = shift(@_);

  # See what OS we have so that we can determine the algorithm to get info
  my $os;
  my $ver;
  chomp ($os = `uname -s`) or die "Failed to run the uname command";
  SWITCH: {
    $os eq "SunOS" && do 
    {
      chomp ($ver = `uname -r`);
      if ( $ver !~ /^4./ ) {
         $osType = "Sol";
         last SWITCH;
      }
    };
    $os eq "HP-UX" && do 
    {
      $osType = "HP";
      $ENV{UNIX95} = "XPG4";
      last SWITCH;
    };
    $os eq "Linux" && do
    {
      $osType = "LNX";
      last SWITCH;
    };
    $os eq "OSF1" && do
    {
      $osType = "OSF1";
      last SWITCH;
    };
    $os eq "AIX" && do
    {
      $osType = "AIX";
      last SWITCH;
    };
    die "Unsupported Operating System\n";
  }

  # Get process information
  executePSCommand();
  
  # Get a tree of pids to check.
  $all_pids = getPidTree( $all_pids, $os );

  # Walk through all the pids to get the process information
  my @pid_list = split(",",$all_pids);
  my $pid;
  my $physical_mem_kb = 0;
  my $physical_mem_total = 0;
  my $physical_mem_free_raw = 0;
  my $pagesize = 0;
  my $cpu_util = 0;
  my $cpu_util_total = 0;

  # Get the metrics on Solaris
  if ($os eq "SunOS" || $os eq "Linux" || $os eq "HP-UX") {

    # see if we can get the real memory usage including shared memory
    $physical_mem_kb = getMemoryUsageForAllPids( $component_pid_list, \@pid_list, $os, $emdroot );
    my $have_memory_info = ( $physical_mem_kb != 0 );
    foreach $pid (@pid_list)
    {
      if( !$have_memory_info ) {
        $physical_mem_kb += getMemoryUtilization($pid);
      }
      $cpu_util += getCPUUtilization($pid);
    }

    # Get total cpu usage for the system
    $cpu_util_total = getTotalCPUUtilization();    

    # Get total memory usage for the system
    my $memory_metrics = getPhysicalMemoryMetrics( $emdroot, $perldir );
    ($physical_mem_total, $physical_mem_free_raw, $pagesize) = 
      split( '\|', $memory_metrics );
  }

  # Convert the memory number from KB to MB
  my $physical_mem =$physical_mem_kb/1024;

  # The CPU usage must account for multiple processors, so divide the 
  # results by the number of cpus on the system
  my $num_Processors = getNumCpus($emdroot);
  if ( $num_Processors != 0 ) {
    $cpu_util = $cpu_util/$num_Processors;
    $cpu_util_total = $cpu_util_total/$num_Processors;
  }
  else {
    $cpu_util = 0;
    $cpu_util_total = 0;
  }

  # Create the results:
  # $cpu_util = cpu - component (all pids)
  # $cpu_other = cpu - other
  # $cpu_idle = cpu - idle
  # $physical_mem = memory - component (all pids)
  # $physical_mem_total = memory MB - total 
  # $physical_mem_percent = memory % - component (all pids)
  my $cpu_other = $cpu_util_total - $cpu_util;
  my $cpu_idle = 100 - $cpu_util_total;
  my $physical_mem_percent = 0;
  if ($physical_mem_total != 0)
  {
    $physical_mem_percent = ($physical_mem*100)/$physical_mem_total;
  }
  $physical_mem_free_raw = $physical_mem_free_raw/(1024*1024);
  printf ("em_result=%s|%.2f|%.2f|%.2f|%.2f|%.2f|%.2f|%d|%.2f\n",$process_name,$cpu_util,$cpu_other,$cpu_idle,$physical_mem,$physical_mem_total,$physical_mem_free_raw,$pagesize,$physical_mem_percent);
 # print "em_result=$cpu_util|$cpu_other|$cpu_idle|$physical_mem|$physical_mem_total|$physical_mem_free_raw|$pagesize|$physical_mem_percent\n";
}

#
# Execute the ps command that will be used to get process information
#
sub executePSCommand
{
  # execute the ps command
  my $process_output=" ";
  if ( $osType eq "HP" ) {
	$ENV{UNIX95}="XPG4";
	$process_output = `ps -A -o \"ppid pid pcpu\"`;
  }
  else {
	$process_output = `ps -A -o \"ppid pid pcpu rss\"`;
  }
  @main::process_output_lines = split("\n", $process_output);	 

  # Define variables for each column
  $main::PPID_COL = 0;
  $main::PID_COL = 1;
  $main::CPU_COL = 2;
  $main::MEM_COL = 3;  
}

# Get the process tree for the specified list of processes.  Be sure to 
# remove duplicate processes
#
# parameters
# parent_pids - the list of parent pids
# os          - the os platform
sub getPidTree
{

  # Pull off the parameters
  my $parent_pids = shift(@_);
  my $os = shift(@_);

  # resulting pid list - start this out with a leading comma.  That way we 
  # can search for ",pid,"
  my $all_pids = ",";

  if ($os eq "SunOS" || $os eq "HP-UX" || $os eq "Linux") 
  {
    my @pid_list = split(",",$parent_pids);
    my $pid;
    foreach $pid (@pid_list)
    {
      $all_pids = getChildPids( $pid, $all_pids );
    }

    # remove the leading and trailing comma
    my $len = length( $all_pids );
    if( $len > 2 ) 
    {
      $all_pids = substr( $all_pids, 1, $len -2 );
    } else {
      # there are no pids
      $all_pids = "";
    }
  } else {
    $all_pids = $parent_pids;
  }

  return $all_pids;

}

# Get the process tree pid of the specified pid. Return  list of all pids to the 
# root for the specified pid.
# Parameters:
# pid                  - the pid to check
# pid_list             - the list of parent pids (must have leading and 
#                        trailing commas)
sub getChildPids
{
  # Pull off the parameters
  my $pid = shift(@_);
  my $pid_list = shift(@_);

  # Add the pid to the pid list (if it is not already there)
  if( index( $pid_list, ",$pid," ) == -1 ) 
  {
    # Add the pid to the list
    $pid_list = "$pid_list$pid,";

    # Find the parent pid in the process output.  Note the first line is the 
    # header so skip it.
    my @process_info;
    my $parentPid;
    my $childPid;
    for( my $i=1; $i <= $#main::process_output_lines; $i++ )
    {	
      @process_info = split(" ", @main::process_output_lines->[$i]);
      $parentPid = $process_info[$main::PPID_COL];
      $childPid = $process_info[$main::PID_COL];
      if( $parentPid == $pid )
      {
        $pid_list = getChildPids( $childPid, $pid_list );
      }
    }				
  }
  return $pid_list;    

}

# Get physical memory metrics for the system.  Return them in the form: 
# {TotalMemoryInMB}|{FreeMemoryRaw}|{PageSize}
# Parameters:
# emdRoot the root directory for the emd
sub getPhysicalMemoryMetrics
{
  my $emdroot = shift(@_);
  my $perldir = shift(@_);

  my $mem_output; 

  # get the osLoad info from nmupm.  The result will be
  # em_result={cpuLoad_1min}|{cpuLoad_5min}|{cpuLoad_15min}|{PageScanRaw}|
  # {noOfProcs}|{noOfUsers}|{transfers}|{idleTimeRaw}|{userTimeRaw}|
  # {systemTimeRaw}|{waitTimeRaw}|{pageSize}|{realMem}|{freeMemRaw}|
  # {usedSwapRaw}|{freeSwapRaw}

  my $PAGE_SIZE_COL;
  my $REAL_MEM_COL;
  my $FREE_MEM_RAW_COL;

  if ( $osType eq "LNX" ) {
    $PAGE_SIZE_COL = 12;
    $REAL_MEM_COL = 13;
    $FREE_MEM_RAW_COL = 14;

    $mem_output = `$perldir/perl $emdroot/sysman/admin/scripts/osLoad.pl`;
  }
  else
  {
	$mem_output = `$emdroot/bin/nmupm osLoad`;
  }
  my @mem_lines = split( '\|', $mem_output );

  # pull out pageSize, realMem and freeMemRaw
  my $totalMem = $mem_lines[$REAL_MEM_COL]/1024;
  EMD_PERL_DEBUG("Total Mem: $totalMem| Free Mem: $mem_lines[$FREE_MEM_RAW_COL]| Real Mem: $mem_lines[$PAGE_SIZE_COL]");
  return "$totalMem|$mem_lines[$FREE_MEM_RAW_COL]|$mem_lines[$PAGE_SIZE_COL]";
}

# Get the current CPU Utilization on the system
sub getTotalCPUUtilization 
{	 
  # sum up the results from the ps command
  my @process_info;
  my $sum = 0;
  for( my $i=1; $i <= $#main::process_output_lines; $i++ )
    {
      @process_info = split(" ", @main::process_output_lines->[$i]);
      $sum = $sum + $process_info[$main::CPU_COL];
    }
  return $sum;			       
}

# Get the CPU usage for the specified pid
#
# Parameters:
# pid the process to get cpu info for
sub getCPUUtilization
{
  my $pid = shift(@_);
  return getUtilization($pid, $main::CPU_COL );
}

# Get the Memory usage for the specified pid
#
# Parameters:
# pid the process to get cpu info for
sub getMemoryUtilization
{
  my $pid = shift(@_);
  if ($osType eq "HP-UX") {
	return getHpMemUtilization($pid, $main::MEM_COL );
  }
  else {
	return getUtilization($pid, $main::MEM_COL );
  }
}

# Get the Resource usage for the specified pid
#
# Parameters:
# pid the process to get cpu info for
# $process_column the column of process information to read
sub getUtilization
{
  my $pid = shift(@_);
  my $process_column = shift(@_);

  # sum up the results from the ps command
  my @process_info;
  my $sum = 0;
  for( my $i=1; $i <= $#main::process_output_lines; $i++ )
    {
      @process_info = split(" ", @main::process_output_lines->[$i]);
      if( $process_info[$main::PID_COL] == $pid )
      {
        $sum = $sum + $process_info[$process_column];
      }
    }
  return $sum;
}

sub getHpMemUtilization
{
  my $pid = shift(@_);
  my $process_column = shift(@_);
  my $rss = 0;

  # sum up the results from the ps command
  my @process_info;
  my $sum = 0;
  for( my $i=1; $i <= $#main::process_output_lines; $i++ )
    {
      @process_info = split(" ", @main::process_output_lines->[$i]);
      if( $process_info[$main::PID_COL] == $pid )
      {
	$rss = `$ENV{EMDROOT}/bin/nmupm rss $pid`; chomp($rss);
        $sum = $sum + $rss;
      }
    }
  return $sum;
}

# Get the pid list for all the webcache processes
#
# Parameters
# oracle_home - the oracle_home in which the webcache is running 
sub getWebcachePids
{
  my $oracle_home = shift(@_);
  $ENV{ORACLE_HOME} = "$oracle_home";
  
  # Execute $ORACLE_HOME/webcache/bin/webcachectl status
  # [ade-klmichae_emdw] klmichae-sun> $ORACLE_HOME/webcache/bin/webcachectl status
  # 
  # The following is a sample of webcache output
  # Web Cache admin server is running as process 6134.
  # Web Cache auto-restart monitor is not running.
  # Web Cache cache server is running as process 6136.
  #
  # Note that we need to redirect the webcache output since webcache insists
  # on writing the results to stderr
  my $pidList = "";
  my $wc_command =  "$oracle_home/webcache/bin/webcachectl status 2>&1";
  my $wc_output = `$wc_command`;

  # Split the lines and walk through them 
  my $line;
  my @lines = split( ".\n", $wc_output );
  foreach $line (@lines)
  {
    # pull off the pid
    my $pid;
    my $front;
    ($front,$pid) = split( " as process ", $line );
    
    # if the process if down, we won't have a pid
    if( defined($pid) ) 
    {	
      # add the pid to the pid list
      if( $pidList eq "" ) 
      {
        $pidList = "$pid";
      } else {
        $pidList = "$pidList,$pid";
      }
    }
	
  }

  return $pidList;
}

# Get a ps command that returns the the username followed by the pid
# plus the full command line.  Note that this command is platform
# specific.
sub getPsCommand
{
  my $os;
  chomp ($os = `uname -s`) or die "Failed to run the uname command";
  SWITCH: {
    $os eq "SunOS" && do
    {
      return "/usr/ucb/ps auxww";
    };
    $os eq "HP-UX" && do
    {
      return "/bin/ps -efx";
    };
    $os eq "Linux" && do
    {
      return "/bin/ps auxww";
    };
    die "Unsupported Operating System\n";
  }
}

# Get the pid list for the Http Server 
# parameters:
#   oracle_home - the oracle home in which the iAS instance is running
sub getHttpPids 
{
  my $oracle_home = shift(@_);
  my $pidList = "";

  # The ps command is platform specific
  my $ps_command = getPsCommand();

  # Get the apache pids from ps
  # The output will be in the following form:
  # USER       PID %CPU %MEM   SZ  RSS TT       S    START  TIME COMMAND
  my $ps_output = `$ps_command | grep "[h]ttpd" | grep $oracle_home`;

  # Get the ps output lines and walk through them Looking for the pids
  my $line;
  my @lines = split( "\n", $ps_output );
  my @tokens;
  foreach $line (@lines)
  {
    @tokens = split( " ", $line );
    # Get the pid; it is the second token
    if( $pidList eq "" )
    {
      $pidList = "$tokens[1]";
    } else {
      $pidList = "$pidList,$tokens[1]";
    }
  } # end foreach line

  return $pidList;

}

# Get the pid list for the specified OC4J Instance
# parameters
#   oracle_home - the oracle home in which the iAS instance is running
#   oc4j_name - the oc4j instance name (if empty, return all instances)
sub getOc4jPids 
{
  # Get the arguments
  my $oracle_home = shift(@_);
  my $oc4j_name = shift(@_);
  
  my $pidList = "";

  # The ps command is platform specific
  my $ps_command = getPsCommand();

  # To get a single oc4j look for "".  Get get all of them look for
  # "oracle.ons.instanceid"
  my $oc4j_key;
  if( $oc4j_name eq "" )
  {
    $oc4j_key = "oracle.ons.instanceid";
  } else {
    $oc4j_key = "#$oc4j_name#";
  }

  # Get the oc4j pids from ps
  # The output will be in the following form:
  # USER       PID %CPU %MEM   SZ  RSS TT       S    START  TIME COMMAND
  my $ps_output = `$ps_command | grep "[j]ava" | grep "$oc4j_key" | grep $oracle_home`;

  # Get the ps output lines and walk through them Looking for the pids
  my $line;
  my @lines = split( "\n", $ps_output );
  my @tokens;
  foreach $line (@lines)
  {
    @tokens = split( " ", $line );
    # Get the pid; it is the second token
    if( $pidList eq "" )
    {
      $pidList = "$tokens[1]";
    } else {
      $pidList = "$pidList,$tokens[1]";
    }
  } # end foreach line

  return $pidList;
}

# Get the physical memory usage (KB) for a set of pids.  Make sure to account  
# for shared memory.
#
# Parameters:
#  component_pid_list - an array of pids per component type (does not 
#                       include child processes)
#  pid_list           - an array of pids to account for (includes child processes)
#  os                 - the os platform
#  emdroot            - the root of the emd tree
#
sub getMemoryUsageForAllPids
{
  # Pull off the paramters
  my $component_pid_list = shift(@_);
  my $pid_list = shift(@_);
  my $os = shift(@_);
  my $emdroot = shift(@_);
  my $total = 0;

  # if there is only one component, just use the pid list
  if( $#$component_pid_list == 0 ) 
  {
    # We have just one component type, so just use the pid list
    $total = getPhysicalMemoryIncludingShared( $pid_list, $emdroot ); 
  } else {
    # we need to get a pid list for each component type and then add the results
    # together
    my $pids_including_child;
    my @pid_array;
    my $i;
    my %pid_hash=();
    for $i (0 .. $#$component_pid_list)
    {
       # Get all the child pids and then make sure they are unique
       if( $component_pid_list->[$i] ne "" )
       {
         $pids_including_child = getPidTree( $component_pid_list->[$i], $os );
         $pids_including_child = removeDuplicates( $pids_including_child, \%pid_hash );
         @pid_array = split( ",", $pids_including_child );
         $total += getPhysicalMemoryIncludingShared( \@pid_array, $emdroot );
       }
       
    } # end foreach
  } # end if( @$component_pid_list == 1 )

  return $total;
}

# Look at the pid list and remove any duplicate pids from it and return a new 
# comma separated list.
# 
# Parameters:
#  current_pid_list - a comma separated list of pids for the current component
#  pid_hash - a hash table containing all pids seen so far
sub removeDuplicates
{
  # Pull off the paramters
  my $current_pid_list = shift(@_);
  my $pid_hash = shift(@_);

  # pull apart the current pid list and make sure that none of the 
  # pids are found in the pid hash
  my $unique_pids = "";
  my @pid_array = split( ",", $current_pid_list );
  my $pid;
  foreach $pid (@pid_array)
  {
    if( !defined( $pid_hash->{$pid} ) ) 
    {
      # This is a unique pid. Add it to the pid hash and unique_pids lists
      $$pid_hash{$pid} = 1;
      if( $unique_pids eq "" ) 
      {
        $unique_pids = $pid;
      } else {
        $unique_pids = "$unique_pids,$pid";
      }
    } # end if( !defined(  ...
  } # end foreach

  return $unique_pids;
}

# Get the physical memory usage (KB) for a set of pids.  Make sure to account  
# for shared memory
#
# Parameters:
#  pid_list - an array of pids to account for
#  emdroot            - the root of the emd tree
# 
sub getPhysicalMemoryIncludingShared
{
  my $pid_list = shift(@_);
  my $emdroot = shift(@_);
  
  my $totalPrivate = 0;
  my $maxShared = 0;
  my $private = 0;
  my $shared;
  my $checkedNmb = 0;

  if ($osType eq "LNX") {
     return getLinuxPhysicalMemoryIncludingShared( $pid_list, $emdroot );
  }

  for my $i (0 .. $#$pid_list)
  {
    # Call nmb and get the shared and private memory usage
    my $pmap_output = `$emdroot/bin/nmb $pid_list->[$i] 2>&1`;

    # Make sure we don't get an error
    if( index( $pmap_output, "Error" ) == -1 ) 
    {

      # the output will be: {private-memory-in-kb} {shared-memory-in-kb}
      ($private, $shared) = split(  " ",  $pmap_output );
    
      # Add this to our totals
      $totalPrivate += $private;
      if( $shared > $maxShared ) {
        $maxShared = $shared;
      }      
    } else {
      # we could not parse the output, so make sure nmb is protected properly
      if( $checkedNmb == 0 ) 
      {
        validateNmbProtection($emdroot);
        $checkedNmb = 1;
      }
    } # end if( index( ...
   
  } # end foreach

  # The total memory usage is SUM(private) + MAX(shared)
  return $totalPrivate + $maxShared;

}

# Get the physical memory usage (KB) for a set of pids.  Make sure to account
# for shared memory.  This is for Linux only.
#
# Parameters:
#  pid_list - an array of pids to account for
#  emdroot            - the root of the emd tree
#
sub getLinuxPhysicalMemoryIncludingShared
{
  my $pid_list = shift(@_);
  my $emdroot = shift(@_);

  my $totalPrivate = 0;
  my $maxShared = 0;
  my $private = 0;
  my $shared;

  for my $i (0 .. $#$pid_list)
  {
        $private = 0;
        $shared = 0;
        my $line="";
        if ( open(MAPS,"/proc/$pid_list->[$i]/maps") )
        {
                while ( $line = <MAPS> )
                {
                        my @fields=split(/\s+/,$line); #field separator is space
                        my @fields1 = split(/-/,$fields[0]); # field separator is -
                        my @types = split(//,$fields[1]); # get the chars
                        if($types[3] eq "s") {
                                $shared = $shared + hex($fields1[1])-hex($fields1[0]);
                        }
                        elsif($types[3] eq "p") {
                                $private = $private + hex($fields1[1])-hex($fields1[0]);
                        }
                }
                close(MAPS);
        }
        $shared = $shared / 1024;
        $private = $private / 1024;

        if ( $totalPrivate < $private ) {
                $totalPrivate = $private;
        }
        if( $shared > $maxShared ) {
                $maxShared = $shared;
        }

  } # end foreach

  # The total memory usage is SUM(private) + MAX(shared)
  return $totalPrivate + $maxShared;

}

#
# Check is the protection on nmb is correct.  It should be owned by root and 
# have the sticky bit set.  If the protect is not correct, print an error and
# exit.
#
# Parameters:
#  emdroot            - the root of the emd tree
#
sub validateNmbProtection
{
  my $emdroot = shift(@_);

  # Get the stats for the file
  my @stat_array = stat("$emdroot/bin/nmb");

  # Make sure that it is owned by root (uid=0)
  if( $stat_array[4] == 0 ) 
  {
    # check that the sticky bit is set
    if( $stat_array[2] & S_ISUID ) {
      # The protection is correct
      return;
    }
  }

  # The protection is incorrect
  print "nmb does not have root access. Please see the release notes for information on setting the uid bit.\n";
  exit 1;
}

# Get the number of CPUs for the system 
#
# Parameters:
#  emdroot            - the root of the emd tree
#
sub getNumCpus()
{

  # The linux computation is different
  if( $osType eq "LNX" ) {
    my $cpu_stat = `cat /proc/stat | grep cpu | wc -l`  or die "Failed to get processor information";
    return $cpu_stat;
  }

  # Execute nmupm asking for CPU information.  It will return one row per CPU
  my $emdroot = shift(@_);
  my $cpu_info = `$emdroot/bin/nmupm osCpuUsage`;
  my @lines = split("\n", $cpu_info);
  if( $#lines > 0 )
  {
    return ($#lines+1);
  } else {
    return 1;
  }
  
}

1;

