#!/usr/bin/perl
#
# Copyright (c) 2003, 2004, Oracle Corporation.  All rights reserved.  
#
#    NAME
#      procExist.pl 
#
#    DESCRIPTION
#      Metric script that checks whether certain processes exist or not 
#       
#
#    OUTPUT:
#      Depending on the metric code, output lines that are in the form:   
#         em_result=<[owner]:<process>|<instance_count>
#      or:
#         em_result=<[owner]:<process>|<pids>|<instance_count>
#
#    NOTES
#      Supported platforms: Solaris, HP-UX and Linux. 
#      
#      Usage: procExist.pl <metricCode> <criteria> 
#         where: 
#            <metricCode> : [0|1]
#                           0 : for metric Critical Processes 
#                           1 : for metric Unwanted Processes     
#
#            <criteria>   : monitoring rules. The value is a string 
#                           in the form:
#                           <process_name>|||<owner>[|||..]
#                         
#
#    MODIFIED    (MM/DD/YY)
#      rmenon     02/23/04 - rmenon_host_mgmt_checkin 
#      lnhan      08/05/03 - Creation 

use strict;
use File::Basename;
require "emd_common.pl";

my $monProcStr;                  # string containing monitoring criteria  

# check the number of parameters 
if ( $#ARGV < 1 ) {
   print "Insufficient parameters. Note: This metric does not support real-time collection.\n";
   EMD_PERL_DEBUG("ERROR: Insufficient parameters.\nUsage: $0 <metric_code> <criteria>");  
   exit -1;
}

my $metricCode;                   # code to determine the metric  
if ( ( $metricCode != 0 ) and ( $metricCode != 1 ) ) {
   EMD_PERL_DEBUG("ERROR: Invalid metric code: [$metricCode]. Abort"); 
   exit -1;
}
($metricCode, $monProcStr) = @ARGV[0..1];


# ------------------------------------------------------------------------
# validate rules for monitoring processes and combine the rules (fields) 
# for each process into an element of array monProcRules. 
# ------------------------------------------------------------------------

my @monProcs = split(/\|\|\|/,$monProcStr);
my @monProcRules;
my $numFields = 2;               # predefined number of fields 
 
for (my $i=0; $i <= $#monProcs; $i += $numFields) {
   my $soughtProc = $monProcs[$i];
   my $soughtUser = $monProcs[$i + 1];

   if ( $soughtProc eq ""  ) { 
      EMD_PERL_DEBUG("ERROR: Process field must not be empty. Abort");
      exit -1;
   }
   
   my @monProcRec;
   @monProcRec = ($soughtProc,$soughtUser);
   @monProcRec = join '|||', @monProcRec;
   push @monProcRules, @monProcRec;

}

EMD_PERL_DEBUG("Criteria: @monProcRules");


# ------------------------------------------------------------------------
# get the process list, store it in array psLines and check for errors  
# ------------------------------------------------------------------------

my @psLines = &getProcList;

my $returnMsg = shift @psLines;

if ( $returnMsg ne ""  ) {
   EMD_PERL_DEBUG ("ERROR: $returnMsg");
   exit -1;
}


# ------------------------------------------------------------------------
# go through each of the processes  listed in array monProcRules to look   
# for matching processes in array psLines  
# ------------------------------------------------------------------------


foreach my $rule ( @monProcRules ) {
   my ($soughtProc, $soughtUser) = split(/\|\|\|/, $rule);
   my $command;
   my $instanceCount = 0;                  # instance counter
   my $pidStr        = "";                 # string that holds PIDs

   foreach my $psLine ( @psLines ) {

      # remove any leading space
      $psLine =~ s/^\s+//;

      my ($pid, $user, @cmdArgs) = split(/\s+/,$psLine);
      $command = $cmdArgs[0];
      $command = basename($command," ");

      if ( $command =~ /^$soughtProc$/ ) {
         # found a matching command 

         if ( ( $soughtUser eq "" ) or ( $soughtUser eq $user ) ) {  
            # username not specified OR username matches user in ps output line

            $instanceCount++;

            if ( $metricCode == 1 ) {
               # this is an alert-if-found case
               $pidStr .= "$pid\,";
            } 
            next;
         }
      }
   } # end of foreach
 
   if ( $metricCode == 0 ) {

      # output results
      print          "em_result=[$soughtUser]:$soughtProc|$instanceCount\n";
      EMD_PERL_DEBUG("em_result=[$soughtUser]:$soughtProc|$instanceCount");
   }
   elsif ( $metricCode == 1 ) {

      if ( $pidStr ne "" ) {
 
          # remove trailing comma (,)     
          $pidStr =~ s/,$//;

      }

      # output results
      print          "em_result=[$soughtUser]:$soughtProc|$pidStr|$instanceCount\n";
      EMD_PERL_DEBUG("em_result=[$soughtUser]:$soughtProc|$pidStr|$instanceCount");
   }

} # end of foreach


# ======================================================================== 
# subroutines 
# ======================================================================== 

# -------------
sub getProcList {
# -------------
# determine OS type and run ps get the list of processes and their
# characteristics.
#
# parameter: none
#
# return:
#   - an array containing processes information.
#     Note:  first element of the array stores error message.
#            blank if there is no error.

   $ENV{PATH} = "/usr/bin:/bin:/usr/local/bin";
   my $os;                         # operating system
   my $errTxt = "";
   chomp ($os = `uname -s`) or $errTxt = "Failed to run the uname command";

   my @psList = ();                # list of processes

   SWITCH: {
      $os eq "SunOS" && do {
             chomp (my $ver = `uname -r`);
             if ( $ver !~ /^4./ ) {
                # OS is Solaris
                @psList = `ps -eo "pid user args"`;
                last SWITCH;
             }
          };
      $os eq "HP-UX" && do {
             # OS is HP-UX
             # Enable XPG4 environment for ps
             $ENV{UNIX95} = "XPG4";
             @psList = `ps -eo "pid user args"`;
             last SWITCH;
          };
      $os eq "Linux" && do {
             # OS is Linux
             @psList = `ps -eo "pid user args"`;
             last SWITCH;
          };
      $errTxt = "Unsupported Operating System\n";
   }

   # remove ps header line
   shift @psList;

   unshift @psList,($errTxt);
   return @psList;

} # end of sub getProcList
# ========================
