# $Header: image-stage-directive.pl 23-dec-2005.21:12:57 ORACLE Exp $
#
# image-stage-directive.pl
#
# Copyright (c) 2005, Oracle. All rights reserved.
#
#    NAME
#      image-stage-directive.pl - Sample directive to stage Image
#
#    DESCRIPTION
#      Sample directive to stage image
#
#    PRE_REQUISITES
#      User must have associated this directive as a stage directive while creating an image
#    pshroff     03/10/06 - Backport rattipal_lastest_directives from main 
#    rattipal    12/12/05 - Backport pshroff_bootfile_auto from main 

use File::Find;
use strict;

require "$ENV{EMDROOT}/sysman/admin/scripts/provisioning/provisionCommon.pl";


# GLOBAL CONSTANTS
my $SAMPLE_BOOT_FILE = "$ENV{EMDROOT}/sysman/admin/scripts/provisioning/SAMPLE-PXE-BOOTFILE";
my $KICKSTART_URL = "KICKSTART_URL"; 
my $PXECONFIGDIR  = "pxelinux.cfg";
my $PWD_COMMAND = "pwd";
my $CHMOD_COMMAND = "chmod -f";
my $CAT_COMMAND = "cat";
my $NAME_MAC_ADDRESS_PROP = "NAME_MacAddress";
my $BOOT_SERVER_PATH_PROP = "NAME_BootServerPath";
my $STAGE_BASE_URL_PROP = "NAME_StageBaseUrl";
my $g_pathSeparator = "/";
my $g_image_kickstart_path;
my $g_rel_image_kickstart_path;

main();

sub main()
{
    # Constants
    my $KICKSTART_SUFFIX = "-ks.cfg";
    my $REBOOT_COMMAND = "reboot";
    my $INSTALL_COMMAND = "install";

    # Local
    my $m_imageFolder;
    my $m_mac_address;
    my $m_kickstart;
    my $m_ref_kickstart_name;
    my $m_image_kickstart_path;
    my $m_ref_kickstart_file;

    my $m_imageFolder=`$PWD_COMMAND`;
    chomp $m_imageFolder;

    print "Current image folder is: $m_imageFolder\n"; 

    my $m_rel_imageFolder=$m_imageFolder;
	my $stage_top_dir=get_oraProperty("NAME_StageSharedMountPoint");
	$m_rel_imageFolder=~ s/$stage_top_dir//;

	
	#image kickstart name
    $m_mac_address = get_oraProperty($NAME_MAC_ADDRESS_PROP); 
    $m_kickstart = $KICKSTART_SUFFIX;

    #reference kickstart file name
    $m_ref_kickstart_name="ref-anaconda-ks.cfg";
	$g_rel_image_kickstart_path=$m_rel_imageFolder.$g_pathSeparator.$m_mac_address.$m_kickstart; 
    $g_image_kickstart_path = $m_imageFolder.$g_pathSeparator.$m_mac_address.$m_kickstart; 
    $m_ref_kickstart_file = $m_imageFolder.$g_pathSeparator.$m_ref_kickstart_name;

    #Getting entries from the reference kickstart file
    #following entries are added here:
    #mouse, keybord, lang, langsupport, bootloader, xconfig
    createAndAppendToKickstart("#Following entries from reference kickstart file\n");
    print "Reference kickstart file location: $m_ref_kickstart_file\n";
    createDefaultEntries($m_ref_kickstart_file);

    #Adding default entries
    #Following entries are added here:
    #reboot, install, nfs
    appendToKickstart("#Adding default hardcoded values\n");
    appendToKickstart("$REBOOT_COMMAND\n");
    appendToKickstart("$INSTALL_COMMAND\n");

    #Combine smaller kickstart files
    appendToKickstart("\n#Combining entries from smaller kickstart files\n");
    find(\&create_kickstart, $m_imageFolder);

    #Add package section to the image kickstart file
    #Package section file "package-ks.cfg" is generated by linux os component 
    find(\&addPackageSection, $m_imageFolder);

    #Add post section to the image kickstart file
    #post section file "post-ks.cfg" is generated by linux os component 
    find(\&addPostSection, $m_imageFolder); 

    #now at this point, kickstart file is ready, modify boot server , update bootfile 
    # and use the kickstart file in ks-url
    modifyBootServer(); 
} #main end


# Purpose
# Modify boot server, copy sample-boot file and then 
# modify the URL to point to proper kickstart file
# available on stage server
# @throws: will exit if any error is found during execution
sub modifyBootServer()
{ 
    # Local
    my $m_chmod_command;
    my $m_bootFilename;
    my $m_bootServerPath;
    my $m_bootFilePath;
    my $m_exitStatus;

    #chmod 755 to final kickstart on stage server
    my $m_chmod_command = "$CHMOD_COMMAND 755 $g_image_kickstart_path";
    system($m_chmod_command);
    $m_exitStatus = $? >> 8;

    if ($m_exitStatus != 0)
    {
        print "\nFailed to Change te Permission of the kickstart File \n";
        exit $m_exitStatus;
    }

    my $m_bootFilename = getBootFilename(); 

    #copying final kickstart file to boot server
    my $m_bootServerPath=get_oraProperty($BOOT_SERVER_PATH_PROP);  

    my $m_bootFilePath = $m_bootServerPath.$g_pathSeparator.$PXECONFIGDIR.$g_pathSeparator.$m_bootFilename;

    print "boot path is $m_bootFilePath \n";

    # change ks url  in boot file
    changeKickstartUrl($m_bootFilePath);
}

# Purpose
# Reads the sample boot file, modifies kickstart URL Info and 
# adds each line into boot file, specified by input param.
# @params: boot filepath
# @throws: will exit if any error is found during execution
sub changeKickstartUrl()
{
    # Local
    my $m_bootUrl;
    my $m_chmod_command;
    my $m_exitStatus;

    my ($m_bootFilePath) = @_;

    my $m_bootUrl = getBootUrl(); 

    open IN_BOOT_FILEHANDLE, "$SAMPLE_BOOT_FILE" or die "Can't open sample boot file $SAMPLE_BOOT_FILE $!";
    open OUT_BOOT_FILEHANDLE, "> $m_bootFilePath" or die "Can't create boot file $m_bootFilePath $!";

    while (my $m_entry = <IN_BOOT_FILEHANDLE> )
    {
        chomp $m_entry; 

        if ($m_entry =~ $KICKSTART_URL)
        {
            $m_entry = findAndReplace($m_entry, $KICKSTART_URL, $m_bootUrl); 
        } 

        print OUT_BOOT_FILEHANDLE "$m_entry\n";
    }
    close IN_BOOT_FILEHANDLE;
    close OUT_BOOT_FILEHANDLE;

    my $m_chmod_command = "$CHMOD_COMMAND 777 $m_bootFilePath";
    system($m_chmod_command);
    $m_exitStatus = $? >> 8;

    if ($m_exitStatus != 0)
    {
        print "\nFailed to Change the Permission of the boot File: Please check \n";
    }

    return;
}
                     
															   
# Purpose
# Returns boot(kickstart) file name
# @returns: boot filename
sub getBootFilename()
{
    # Local Variable
    my $m_mac_address;
    my $m_bootFilename;

    $m_mac_address=get_oraProperty($NAME_MAC_ADDRESS_PROP);  

    # replace ":" with "-" as per pxe boot standard
    $m_mac_address = findAndReplace($m_mac_address, ":", "-"); 

    # convert all hex uppercase letter to lowercase
    $m_mac_address = lc($m_mac_address);

    # append "01-" as prefix as per pxe format
    my $m_bootFilename = "01-".$m_mac_address;

    return $m_bootFilename;
} 


# Purpose
# Returns Boot Url for a given assignment
# @returns: bool url

sub getBootUrl()
{
    # Constants;
    my $COLON = ":";
    my $DOUBLE_SLASH = "//";

    # Local Variables
    my $m_baseUrl;
    my $m_host_name;
    my @m_urlInfo;
    my $m_len;
    my $m_count;
    my $m_urlEntry;
    my $m_url;


    $m_baseUrl = get_oraProperty($STAGE_BASE_URL_PROP); 
	if($m_baseUrl=~ /^nfs:/ || $m_baseUrl=~/^file:/)
    {
		my $m_host_dir;
		$m_host_dir="";
		@m_urlInfo = split /\/+/ ,$m_baseUrl ;
		$m_len = $#m_urlInfo;
		for($m_count=1;$m_count<=$m_len;$m_count++)
		{
			$m_urlEntry=$m_urlInfo[$m_count];
			if($m_count ==1)
			{
				$m_host_name = $m_urlInfo[$m_count];
			}
			if($m_count >1)
			{
				$m_host_dir = $m_host_dir."/".$m_urlInfo[$m_count];
			}
		}

   
        $m_url = "nfs".$COLON.$m_host_name.$COLON.$m_host_dir.$g_rel_image_kickstart_path;
    }
    else
    {
        $m_url = $m_baseUrl.$g_rel_image_kickstart_path;
    }

    print "kickstart url is $m_url\n";

    return $m_url;
}



# Purpose
# Adds package file 'package-ks.cfg' to image's kickstart file
# @throws: will exit if any error is found during execution
sub addPackageSection()
{
    # Constant
    my $PACKAGE_SUFFIX = "package-ks.cfg";

    # Local Variables
    my $m_packageFilePath;
    my $m_appendCommand;
    my $m_exitStatus;

    my ($package_file_path, $append_cmd);
    if($_=~$PACKAGE_SUFFIX)
    {
        $m_packageFilePath="$File::Find::dir/$_";

        print "Appending package section file $m_packageFilePath to final kicktstart file\n";
        $m_appendCommand="$CAT_COMMAND $m_packageFilePath >> $g_image_kickstart_path";
        system($m_appendCommand);
        my $m_exitStatus = $? >> 8;

        if ($m_exitStatus != 0)
        {
            print "\nFailed to Copy package section into the kickstart File \n";
            exit $m_exitStatus;
        }
    }
}

# Purpose
# Adds post file 'post-ks.cfg' to image's kickstart file
# @throws: will exit if any error is found during execution
sub addPostSection()
{
    # Constants;
    my $POST_SUFFIX = "post-ks.cfg";

    # Local Variables
    my $m_packageFilePath;
    my $m_appendCommand;
    my $m_exitStatus;

    if($_=~$POST_SUFFIX)
    {
        $m_packageFilePath = "$File::Find::dir/$_";

        print "Appending post section file $m_packageFilePath to final kicktstart file\n";
        $m_appendCommand = "$CAT_COMMAND $m_packageFilePath >> $g_image_kickstart_path";
        system($m_appendCommand);
        $m_exitStatus = $? >> 8;
        if ($m_exitStatus != 0)
        {
            print "\nFailed to Copy post section into the kickstart File \n";
            exit $m_exitStatus;
        }	
    }
}

sub create_kickstart()
{
    # Local Variables
    my $m_smallKickstartPath;

    if(/kickstart$/)
    {
        $m_smallKickstartPath = "$File::Find::dir/$_" ;
        print "Appending smaller kickstart $m_smallKickstartPath to final kickstart file \n";
        open READ,"$m_smallKickstartPath" or die "Can not read $m_smallKickstartPath : $!";
        while(<READ>)		
        {
            appendToKickstart("$_");
        }
        close READ;
    }
}

# Purpose
# Read entries from reference kickstart file and add to the image kickstart file.
# Following entries are added here:
# mouse, keybord, lang, langsupport, bootloader, xconfig
# @params: path to the image kickstart file
# @throws: will exit if any erro is found during execution.
sub createDefaultEntries()
{
    # Local Variables;
    my @m_referenceEntries;
    my @m_mouse;
    my @m_keyboard;
    my @m_lang;
    my @m_langSupport;
    my @m_bootLoader;
    my @m_firewall;
    my @m_xconfig;

    my ($m_image_kickstart_path)=@_;

    #Open reference kickstart file and add entries to the image kickstart file
    open REF_KICKSTART, "$m_image_kickstart_path" or die "\nCan not read reference kickstart file: $m_image_kickstart_path $!";
    while(<REF_KICKSTART>)
    {
        chomp;
        push(@m_referenceEntries, $_);
    }
    close REF_KICKSTART;

    @m_mouse=grep /^mouse /, @m_referenceEntries;
    appendToKickstart("$m_mouse[0]\n");

    @m_keyboard=grep /^keyboard /, @m_referenceEntries;
    appendToKickstart("$m_keyboard[0]\n");

    @m_lang=grep /lang /, @m_referenceEntries;
    appendToKickstart("$m_lang[0]\n");

    @m_langSupport=grep /langsupport /, @m_referenceEntries;
    appendToKickstart("$m_langSupport[0]\n");

    @m_bootLoader=grep /^bootloader /, @m_referenceEntries;
    appendToKickstart("$m_bootLoader[0]\n");

    #firewall info
    @m_firewall=grep /^firewall /, @m_referenceEntries;
    if ($m_firewall[0] eq '')
    {
        $m_firewall[0]="firewall --disabled";
    }
    appendToKickstart("$m_firewall[0]\n");

    #xconfig info
    @m_xconfig=grep /^xconfig /, @m_referenceEntries;
    if ($m_xconfig[0] eq '')
    {
        $m_xconfig[0] = "skipx";
    }
    appendToKickstart("$m_xconfig[0]\n");

    close OUT;
}

# Purpose
# Appends a string to a file
# @params: string to be appended to the file
# @throws: will exit if any error is found during execution
sub appendToKickstart()
{
    my ($m_entry)=@_;
    open OUT, ">> $g_image_kickstart_path" or die "Can't open kickstart file $g_image_kickstart_path $!";
    print OUT $m_entry;
    close OUT;
}

# Purpose
# Writes a string to a file
# @params: string to be appended to the file
# @throws: will exit if any error is found during execution
sub createAndAppendToKickstart()
{
    my ($m_entry)=@_;
    open OUT, "> $g_image_kickstart_path" or die "Can't open kickstart file $g_image_kickstart_path $!";
    print OUT $m_entry;
    close OUT;
}
