#!/bin/sh
#
# $Header: undoHostPatching.sh 26-may-2005.08:49:32 tasingh Exp $
#
# undoHostPatching.sh
#
# Copyright (c) 2005, Oracle. All rights reserved.  
#
#    NAME
#      undoHostPatching.sh - Uninstalls or rolls back packages on Linux hosts.
#
#    DESCRIPTION
#      Uninstalls the specified packages or rolls them back to the specified
#      version or reverts back the changes done during the last update session.
#
#    NOTES
#      Only Linux Hosts are supported.  sudo and rpm are pre-requisites.
#      The AIME run_as_root utility can optionally be used in case sudo
#      access is not available.  This is controlled by setting the
#      T_USE_RUNASROOT variable.
#
#    MODIFIED   (MM/DD/YY)
#    tasingh     05/26/05 - tasingh_bug-4327813
#    tasingh     05/23/05 - Creation
#

#
# Uninstalls or rolls back the RPMs on the Host.
#
# Usage:
#
# undoHostPatching.sh <action> <confFile> [pkg1 pkg2...]
#
# NOTE:
#   The action can be "uninstallPkgs", "rollbackPkgs" or "rollbackLastSession".
#
#   In case of "uninstallPkgs", the pkg1 pkg2... are the names of the RPMs to be
#   uninstalled e.g.
#
#       undoHostPatching.sh uninstallPkgs foo bar
#
#   In case of "rollbackPkgs", the pkg1, pkg2... are the names of the RPMs along
#   with the versions they should be rolled back to. The names and versions are
#   separated by "@" e.g
#
#       undoHostPatching.sh rollbackPkgs foo@1.2-15 bar@2.11-5
#
#   The "rollbackLastSession" does not require pkg1 pkg2... arguments.
#
#   The <confFile> parameter gives the location of hostpatch.conf file.
#

# Global variables.

# The Linux variant in use.
# Currently supported distribution is"RHEL"
DISTRO="RHEL"

# The rpm command
RPM_CMD=`which rpm`

# Configuration file used to get the list of repositories, in case of rollback.
# Also the location of last changes file, log file etc. are derived from it.
# It will be overwritten by the value of the <confFile> parameter.
CONF_FILE="../../config/hostpatch.conf"

# The log file that contains the changes done during the last update cycle.
# It is used to rollback the last update session.
# It will be overwritten once the value of <confFile> parameter is fetched.
LAST_CHANGES_FILE=`dirname "$CONF_FILE"`/../log/patching/lastChanges.log

# The undo patching log file. It will be overwritten once the value of
# <confFile> parameter is fetched.
LOG_FILE=`dirname "$CONF_FILE"`/../log/patching/undoHostPatching.log

# Whether to use an alternative to sudo if it fails.
USE_SUDO_ALT=no

# The AIME run_as_root utility for regression testing, in case sudo fails.
RUNASROOT=/usr/local/packages/aime/emdw/run_as_root

# Whether to refresh host configuration or not
REFRESH_HOST_CFG=no


# Checks the pre-requisites for running this script.
#
checkPreReqs( )
{
  # We should be running on Linux.
  #
  HOST_OS=`uname -s`
  if test "$HOST_OS" != "Linux"; then
    echo ERROR: Only Hosts running Linux are supported! 1>&2
    exit 1
  fi

  # We should be running a supported Linux variant.
  #
  if test -e /etc/redhat-release; then
    DISTRO="RHEL"
  else
    echo ERROR: Sorry, this Linux variant is not supported yet! 1>&2
    exit 1
  fi

  # We need sudo to Undo Host Patching.
  #
  which sudo >/dev/null 2>&1
  if test "$?" != "0"; then
    if test -x /usr/local/bin/sudo; then
      SUDO=/usr/local/bin/sudo
    else
      echo ERROR: sudo is required to Undo Host Patching! 1>&2
      exit 1
    fi
  else
    SUDO=`which sudo`
  fi
}


# Runs the given command with the given arguments as a privileged user.
#
runPrivileged( )
{
  if test "$USE_SUDO_ALT" != "yes"; then
    # Kill sudo password. This ensures that the password is not
    # inadvertently passed to the executing command in stdin.
    $SUDO -k
    echo "$PASSWORD" | $SUDO -S -p "" $*
  else
    $RUNASROOT "$*"
  fi
}


# Installs the given package from the repositories mentioned in
# CONF_FILE
#
# $1 is the name of the package (along with version, release & arch) that 
# is to be installed e.g. foo-2.0-1.i386
#
installPkg( )
{
  for URL in `grep -i ^baseurl "$CONF_FILE" | cut -d'=' -f2`
  do
    # Now install the rpm. The rpm command expects the complete
    # name of the rpm file e.g. foo-1.9-3.i386.rpm.
    # But, we have only the name-version & arch of the rpm so we will
    # append the .rpm extension and try to install the rpm from the 
    # available repositories one by one
    RPM_NAME=$URL"/"$1".rpm"
    runPrivileged $RPM_CMD --install --force --nodeps $RPM_NAME > /dev/null 2>&1
    RET_VAL=$?

    if test "$RET_VAL" = "0"; then
      break
    fi
  done

  if test "$RET_VAL" != "0"; then
    echo "ERROR: Could not locate package \"$PKG_NAME\""
  fi
}

# Refreshes host configuration information in the EM Repository. It is called
# if some package was uninstalled or rolled back.
#
refreshHostCfg( )
{
  echo
  echo Updating host configuration information...

  # We need to find the current host's target name as known to EM.
  # We use "emctl config agent listtargets" and look for a line of
  # the form:
  #   [foo.bar.com, host]
  #
  for myhost in `emctl config agent listtargets | grep "host]$" | sed 's/, host]$//' |cut -c2-`
  do
    emctl control agent runCollection $myhost:host Inventory
  done
}


# Logs the packages installed or uninstalled during the undo patching session.
# The format is mm/dd/yy hh:mm:ss <action>: <pkgName-ver-rel>
#
# $1 is the action performed like "Installed" or "Uninstalled"
#
# $2 is the name of the package
logMsg( )
{
  echo "`date +%D\ %T` $1: $2" >>"$LOG_FILE"
}


# The main processing logic.
#

# Check pre-requisites.
checkPreReqs

# Password must be on the standard input.
read PASSWORD

# We need sudo to work for this user.
# Alternatively, we will use $RUNASROOT if T_USE_RUNASROOT is defined.
# (NOTE: $USER does not give the effective user name at all times.)
#
IDEXEC=`which id`
echo Checking sudo access for user \"`$IDEXEC -un`\"...
$SUDO -k
echo "$PASSWORD" | $SUDO -S -p "" -v
CAN_SUDO="$?"

# Check fine-grained sudo root access to the rpm command.
# If we run rpm without any option it will fail, so we
# run it with --version option.
#
if test "$CAN_SUDO" = "0"; then
  echo "  Checking if user can run rpm as root..."
  $SUDO -k
  echo "$PASSWORD" | $SUDO -S -p "" $RPM_CMD --version 1>&2>/dev/null
  CAN_SUDO="$?"
fi

if test "$CAN_SUDO" != "0"; then
  if test -z "$T_USE_RUNASROOT"; then
    echo ERROR: Undo Host Patching needs sudo to work for this user! 1>&2
    exit 1
  else
    if test -x $RUNASROOT; then
      USE_SUDO_ALT=yes
      echo Using $RUNASROOT instead of sudo...
    else
      echo ERROR: T_USE_RUNASROOT defined but $RUNASROOT unavailable! 1>&2
      exit 1
    fi
  fi
fi


# OK, now comes the meat of the processing...

ACTION="$1"
shift
CONF_FILE="$1"

if ! test -r "$CONF_FILE"; then
  echo "ERROR: Could not read configuration file \"$CONF_FILE\"" 1>&2
  exit 1
fi

LOG_FILE=`dirname "$CONF_FILE"`/../log/patching/undoHostPatching.log
shift

if test "$ACTION" = "uninstallPkgs"; then

  RET_VAL=0

  while test -n "$1"; do
    PACKAGE="$1"

    # First see if the package is indeed installed on the machine
    # Save the result of the rpm -q, which is later used to log
    # which version was uninstalled.
    TMP_PKG_NAME=`$RPM_CMD -q --queryformat "%{NAME}-%{VERSION}-%{RELEASE}.%{ARCH}" $PACKAGE` 

    if test "$?" != "0"; then
      echo "ERROR: Package \"$PACKAGE\" is not installed" 1>&2
      RET_VAL=1
    else
      # If there are more than one versions of the package installed,
      # then raise error.
      NO_OF_PKGS=`$RPM_CMD -q $PACKAGE | wc -l`
      if test $NO_OF_PKGS -gt 1; then
        echo "ERROR: Package \"$PACKAGE\" specifies multiple pacakges" 1>&2
        RET_VAL=1
      else
        runPrivileged $RPM_CMD --erase --nodeps $PACKAGE

        if test "$?" != "0"; then
          echo "ERROR: Could not uninstall package \"$PACKAGE\"" 1>&2
          RET_VAL=1
        else
          echo "Package \"$PACKAGE\" successfully uninstalled"
          logMsg "Uninstalled" "$TMP_PKG_NAME"
          REFRESH_HOST_CFG=yes
        fi
      fi
    fi
    shift
  done

  if test "$REFRESH_HOST_CFG" = "yes"; then
    refreshHostCfg
  fi

  exit $RET_VAL

elif test "$ACTION" = "rollbackPkgs"; then

  RET_VAL=0

  while test -n "$1"; do
    PACKAGE="$1"

    # Separate the package name and the version, here the version
    # is the one to which the package is to be rolled back
    NAME=`echo $PACKAGE | cut -d'@' -f1`
    VERSION=`echo $PACKAGE | cut -d'@' -f2`

    # First see if the package is indeed installed on the machine
    # Save the result of the rpm -q, which is later used to log
    # which version was uninstalled.
    TMP_PKG_NAME=`$RPM_CMD -q --queryformat "%{NAME}-%{VERSION}-%{RELEASE}.%{ARCH}" $NAME` 

    if test "$?" != "0"; then
      echo "ERROR: Package \"$NAME\" is not installed" 1>&2
      RET_VAL=1
    else
      # If there are more than one versions of the package installed,
      # then raise error.
      NO_OF_PKGS=`$RPM_CMD -q $NAME | wc -l`
      if test $NO_OF_PKGS -gt 1; then
        echo "ERROR: Package \"$NAME\" specifies multiple pacakges" 1>&2
        RET_VAL=1
      else
        # Uninstall the current version of the package installed.
        # But before that fetch its arch, it will be used during
        # installation of lower version.
        ARCH=`$RPM_CMD -q --queryformat "%{ARCH}" $NAME`
        runPrivileged $RPM_CMD --erase --nodeps $NAME

        if test "$?" != "0"; then
          echo "ERROR: Could not rollback package \"$NAME\"" 1>&2
          RET_VAL=1
        else
          REFRESH_HOST_CFG=yes
          logMsg "Uninstalled" $TMP_PKG_NAME
          
          # Now install the lower version of the package
          PKG_NAME=$NAME"-"$VERSION
          installPkg "$PKG_NAME.$ARCH"

          if test "$?" != "0"; then
            echo "ERROR: Could not rollback package \"$NAME\" to version \"$VERSION\"" 1>&2
            RET_VAL=1
          else
            echo "Package \"$NAME\" successfully rolled back to version \"$VERSION\""
            logMsg "Installed" "$NAME-$VERSION.$ARCH"
          fi
        fi
      fi
    fi
    shift
  done

  if test "$REFRESH_HOST_CFG" = "yes"; then
    refreshHostCfg
  fi

  exit $RET_VAL

elif test "$ACTION" = "rollbackLastSession"; then

  LAST_CHANGES_FILE=`dirname "$CONF_FILE"`/../log/patching/lastChanges.log

  if ! test -r $LAST_CHANGES_FILE; then
    echo "ERROR: Could not read file containing changes done during last session- $LAST_CHANGES_FILE"
    exit 1
  fi

  RET_VAL=0

  # First uninstall the packages that were installed or updated
  # during the last update cycle.
  for PACKAGE in `grep ^">" $LAST_CHANGES_FILE | tr -d ' ' | cut -d'>' -f2`
  do
    runPrivileged $RPM_CMD --erase --nodeps $PACKAGE

    if test "$?" != "0"; then
      echo "ERROR: Could not rollback the package \"$PACKAGE\"" 1>&2
      RET_VAL=1
    else
      REFRESH_HOST_CFG=yes
      logMsg "Uninstalled" $PACKAGE
    fi
  done

  # Now install the lower-versioned packages that were upgraded to
  # higher-versioned packages
  for PACKAGE in `grep ^"<" $LAST_CHANGES_FILE | tr -d ' ' | cut -d'<' -f2`
  do
    installPkg $PACKAGE

    if test "$?" != "0"; then
      echo "ERROR: Could not rollback the pacakge \"$PACKAGE\"" 1>&2
      RET_VAL=1
    else
      REFRESH_HOST_CFG=yes
      logMsg "Installed" $PACKAGE
    fi
  done

  # Remove the file containing the changes of the last update session
  /bin/rm $LAST_CHANGES_FILE

  if test "$RET_VAL" != "0"; then
    echo "ERROR: Could not rollback the last update session properly." 1>&2
  else
    echo "Successfully rolled back the last session"
  fi

  if test "$REFRESH_HOST_CFG" = "yes"; then
    refreshHostCfg
  fi

  exit $RET_VAL

else
  echo ERROR: Invalid command line argument \"$1\"! 1>&2
  exit 1
fi
