#!/bin/sh
#
# $Header: updHostPackages.sh 06-dec-2005.02:30:37 ppalleti Exp $
#
# updHostPackages.sh
#
# Copyright (c) 2004, 2005, Oracle. All rights reserved.  
#
#    NAME
#      updHostPackages.sh - Update Host Packages (RPMs) for Host Patching.
#
#    DESCRIPTION
#      Updates Linux Host RPMs participating in Host Patching using YUM.
#
#    NOTES
#      Only Linux Hosts are supported. sudo and yum are pre-requisites.
#      The AIME run_as_root utility can optionally be used in case sudo
#      access is not available.  This is controlled by setting the 
#      T_USE_RUNASROOT variable. If the T_PATCH_IMMEDIATELY variable is
#      set, RPMs will be applied immediately instead of at random intervals.
#
#    MODIFIED   (MM/DD/YY)
#    ppalleti    12/06/05 - Backport ppalleti_hp_bugs from main 
#    ppalleti    11/22/05 - bug-4593333 Need Reboot Packages fails to update 
#                           if sudo doesn't work 
#    ranmath     09/12/05 - bug-4601879: Wait five minutes for hostpatch.conf 
#                           to be created if it is not present. 
#    tasingh     05/20/05 - er-4327813 Save the changes done during update
#                           session, so as to support Rollback feature 
#    ranmath     04/27/05 - er-4327827: Support SuSE. Reorganise slightly.
#    ranmath     04/19/05 - bug-4306086: Fix update packages on reboot. 
#    tasingh     04/13/05 - Ensure that sudo password is cleared before 
#                           execution. 
#    nbhandar    04/06/05 - Do not ignore the output of yum check-update. 
#    ranmath     04/01/05 - bug-4272881: Fix thinko. 
#    ranmath     03/30/05 - er-4272881: Update host inventory information only 
#                           if packages were actually updated. 
#    ranmath     03/24/05 - bug-4234607: Log patching activity.
#    ranmath     03/08/05 - bug-4227423: Update even need-reboot packages in
#                           emergency patching. 
#    ranmath     03/03/05 - bug-4216058: Check fine-grained sudo access. 
#    ranmath     01/17/05 - bug-4046545: Do not succeed if intermediate 
#                           commands fail. 
#    ranmath     12/09/04 - er-4043645: Use T_PATCH_IMMEDIATELY to indicate 
#                           whether to apply patches immediately or not. 
#    ranmath     12/07/04 - er-4043645: Eliminate waiting interval for YUM 
#                           in case regressions are being run. 
#    ranmath     11/25/04 - Check for T_USE_RUNASROOT in environment and use 
#                           run_as_root in case sudo fails and this variable
#                           is defined. 
#    ranmath     10/06/04 - Refactor script. 
#    ranmath     09/13/04 - Creation
#


#
# Updates the RPMs on the Host configured for Host Patching.
#
# Usage:
#
# updHostPackages.sh [-rebooting] -confFile file [-packages pkg1 pkg2...]
#


# Global variables.

# The Linux variant in use.
# Currently supported values are "RHEL" and "SLES".
DISTRO="RHEL"

# List of packages to update explicitly.
PACKAGES=""

# Whether we are being invoked from within the Oracle Enterprise Manager
# update service upon rebooting.
REBOOTING="no"

# Configuration file used for Host Patching.
CONF_FILE="../../config/hostpatch.conf"

# The file containing a list of packages that can only be applied on
# a reboot.
NRP_LIST_FILE=`dirname "$CONF_FILE"`/needRebootPkgs.lst

# Whether to use an alternative to sudo if it fails.
USE_SUDO_ALT=no

# The AIME run_as_root utility for regression testing, in case sudo fails.
RUNASROOT=/usr/local/packages/aime/emdw/run_as_root

# The option to specify the waiting interval in minutes within which YUM
# tries to randomly download a package in order to not overwhelm the server
# and the network.
WAIT_OPT="-R 10"

# The debug level (0-10) for YUM to control the amount of messages it spews.
DBG_LVL=1

# The rpm command used to get the snapshot of the system before and
# after the update cycle.
RPM_CMD=`which rpm`

# The tmp file used to store the snapshot of the packages installed on the
# system before the update cycle starts
SNAPSHOT_BEFORE_FILE=/tmp/`mktemp befXXXXXX`

# The tmp file used to store the snapshot of the packages installed on the
# system after the update cycle has finished.
SNAPSHOT_AFTER_FILE=/tmp/`mktemp aftXXXXXX`

# The file used to store the changes that were done durnig the update cycle.
# It is a diff of the two files - SNAPSHOT_BEFORE_FILE and
# SNAPSHOT_AFTER_FILE
LAST_CHANGES_FILE=`dirname $CONF_FILE`/../log/patching/lastChanges.log

# Checks the pre-requisites for running this script.
#
checkPreReqs( )
{
  # We should be running on Linux.
  #
  HOST_OS=`uname -s`
  if test "$HOST_OS" != "Linux"; then
    echo ERROR: Only Hosts running Linux are supported! 1>&2
    exit 1
  fi

  # We should be running a supported Linux variant.
  #
  if test -e /etc/redhat-release; then
    DISTRO="RHEL"
  elif test -e /etc/SuSE-release; then
    DISTRO="SLES"
  else
    echo ERROR: Sorry, this Linux variant is not supported yet! 1>&2
    exit 1
  fi

  # We need sudo for Host Patching.
  #
  which sudo >/dev/null 2>&1
  if test "$?" != "0"; then
    if test -x /usr/local/bin/sudo; then
      SUDO=/usr/local/bin/sudo
    elif test "$REBOOTING" = "no"; then

      echo ERROR: sudo is required for Linux Host Patching! 1>&2
      exit 1
    fi
  else
    SUDO=`which sudo`
  fi

  # Find out the distribution-specific updater client.
  #
  if test "$DISTRO" = "RHEL"; then
    # We need YUM on Red Hat.
    #
    which yum >/dev/null 2>&1
    if test "$?" != "0"; then
      if test -x /usr/local/bin/yum; then
        YUM=/usr/local/bin/yum
      else
        echo ERROR: YUM is required for Linux Host Patching! 1>&2
        exit 1
      fi
    else
      YUM=`which yum`
    fi
  elif test "$DISTRO" = "SLES"; then
    # We need YaST Online Update (YOU) client on SuSE.
    #
    which online_update >/dev/null 2>&1
    if test "$?" != "0"; then
      echo ERROR: YaST Online Update is required for Linux Host Patching! 1>&2
      exit 1
    else
      YOU=`which online_update`
    fi
  fi
}


# Runs the given command with the given arguments as a privileged user.
#
runPrivileged( )
{
  if test "$USE_SUDO_ALT" != "yes"; then
    # Kill sudo password. This ensures that the password is not
    # inadvertently passed to the executing command in stdin.
    $SUDO -k
    echo "$PASSWORD" | $SUDO -S -p "" $*
  else
    $RUNASROOT "$*"
  fi
}


#
# The main processing logic.
#

# Process command line arguments.
while test -n "$1"; do

  if test "$1" = "-rebooting"; then
    # We are being invoked from within the initialisation script
    # in /etc/init.d.
    REBOOTING="yes"
  elif test "$1" = "-confFile"; then
    CONF_FILE="$2"
    NRP_LIST_FILE=`dirname "$CONF_FILE"`/needRebootPkgs.lst
    LAST_CHANGES_FILE=`dirname $CONF_FILE`/../log/patching/lastChanges.log
    if test ! -r "$CONF_FILE"; then
      # bug-4601879: It could be that we are clashing with the configuration
      # file creation script. Wait for five minutes before really giving up.
      echo WARNING: Could not read configuration file \"$CONF_FILE\"!
      echo -n WARNING: Waiting for five minutes...
      sleep 300
      if test ! -r "$CONF_FILE"; then
        echo FAILED!
	echo ERROR: Could not read configuration file \"$CONF_FILE\"! 1>&2
	exit 1
      else
        echo SUCCEEDED!
      fi
    fi
    shift
  elif test "$1" = "-packages"; then
    # This must be the LAST command line option.
    shift
    while test -n "$1"; do
      PACKAGES="$PACKAGES $1"
      shift
    done
  else
    echo ERROR: Invalid command line argument \"$1\"! 1>&2
    exit 1
  fi
  
  # Move on to the next argument, if any.
  if test -n "$1"; then
    shift
  fi
done

# Check pre-requisites.
checkPreReqs

# If we are not rebooting, password must be on the standard input.
# (While rebooting from the initialisation script, we are already 
# running as root.)
if test "$REBOOTING" = "no"; then
  read PASSWORD

  # We *must* have a password to update packages.
  #
  if test -z "$PASSWORD"; then
    echo ERROR: Password must be specified! 1>&2
    exit 1
  fi

  # We need sudo to work for this user.
  # Alternatively, we will use $RUNASROOT if T_USE_RUNASROOT is defined.
  # (NOTE: $USER does not give the effective user name at all times.)
  #
  IDEXEC=`which id`
  echo Checking sudo access for user \"`$IDEXEC -un`\"...
  $SUDO -k
  echo "$PASSWORD" | $SUDO -S -p "" -v
  CAN_SUDO="$?"

  # Check fine-grained sudo root access to the commands we need.
  if test "$CAN_SUDO" = "0"; then
    if test "$DISTRO" = "RHEL"; then
      SUDO_CMDS="$YUM"
    elif test "$DISTRO" = "SLES"; then
      SUDO_CMDS="$YOU"
    fi

    for cmd in $SUDO_CMDS
    do
      # Note that all of our commands gladly accept the --version option,
      # *except* for the YaST Online Update (YOU) client on SLES.
      echo "  Checking if user can run \"$cmd\" as root..."
      $SUDO -k
      if test "$cmd" = "$YOU"; then
        echo "$PASSWORD" | $SUDO -S -p "" $cmd -c >/dev/null
      else
        echo "$PASSWORD" | $SUDO -S -p "" $cmd --version >/dev/null
      fi
      CAN_SUDO="$?"
      if test "$CAN_SUDO" != "0"; then
        break
      fi
    done
  fi

  if test "$CAN_SUDO" != "0"; then
    if test -z "$T_USE_RUNASROOT"; then
      echo ERROR: Linux Host Patching needs sudo to work for this user! 1>&2
      exit 1
    else
      if test -x $RUNASROOT; then
	USE_SUDO_ALT=yes
        echo Using $RUNASROOT instead of sudo...
      else
        echo ERROR: T_USE_RUNASROOT defined but $RUNASROOT unavailable! 1>&2
	exit 1
      fi
    fi
  fi

  # er-4043645: If T_PATCH_IMMEDIATELY is set, we do not need to throttle
  # YUM.
  if test -n "$T_PATCH_IMMEDIATELY"; then
    WAIT_OPT=""
  fi
fi


# OK, now comes the meat of the processing...

if test "$REBOOTING" = "no"; then
  # Exclude packages that need a reboot to be applied.
  # (bug-4227423: Not valid when doing emergency patching.)
  # FIXME: On SuSE, how do you tell this to YOU?
  #
  EXCL_PKGS=""
  if test -f "$NRP_LIST_FILE" -a -z "$PACKAGES"; then
    for i in `cat "$NRP_LIST_FILE"`
    do
      EXCL_PKGS="--exclude $i $EXCL_PKGS"
    done
  fi

  # Update host packages.
  #
  if test "$DISTRO" = "RHEL"; then
    # Run YUM to update the Host packages if needed.
    #

    echo
    echo "Checking if any package needs to be updated..."
    runPrivileged $YUM $WAIT_OPT -d $DBG_LVL -t -c "$CONF_FILE" $EXCL_PKGS check-update

    # NOTE: "yum check-update" exits with code 100 if a package needs to be
    # updated, 0 if none needs to be updated.
    CHECK_STATUS="$?"
    if test "$CHECK_STATUS" = "100"; then
      echo
      echo Updating packages...

      #
      # ER-4327813 Save the snapshot of packages installed on the system
      # before any package is updated.
      #
      $RPM_CMD -qa --queryformat "%{NAME}-%{VERSION}-%{RELEASE}.%{ARCH}\n" | \
        sort >"$SNAPSHOT_BEFORE_FILE"

      runPrivileged $YUM $WAIT_OPT -d $DBG_LVL -t -c "$CONF_FILE" $EXCL_PKGS update $PACKAGES
      if test "$?" != "0"; then
	echo ERROR: Could not update packages! 1>&2
	exit 1
      else
	# Some packages were updated. It would be better to refresh host
	# configuration information in the EM Repository.
	echo
	echo Updating host configuration information...

  #
  # ER-4327813 Take the snapshot of the packages installed on the system and
  # compare it with the snapshot taken earlier.
  #
  $RPM_CMD -qa --queryformat "%{NAME}-%{VERSION}-%{RELEASE}.%{ARCH}\n" | \
    sort >"$SNAPSHOT_AFTER_FILE"

  diff "$SNAPSHOT_BEFORE_FILE" "$SNAPSHOT_AFTER_FILE" > "$LAST_CHANGES_FILE"

  # Remove the tmp files
  /bin/rm "$SNAPSHOT_BEFORE_FILE" "$SNAPSHOT_AFTER_FILE"

	# We need to find the current host's target name as known to EM.
	# We use "emctl config agent listtargets" and look for a line of
	# the form:
	#   [foo.bar.com, host]
	#
	for myhost in `emctl config agent listtargets | grep "host]$" | sed 's/, host]$//' |cut -c2-`
	do
	  emctl control agent runCollection $myhost:host Inventory
	done
      fi
    else
      if test "$CHECK_STATUS" = "0"; then
	echo "All packages are up-to-date."
      else
	echo "ERROR: Could not check for updates!" 1>&2
	exit 1
      fi
    fi
  elif test "$DISTRO" = "SLES"; then
    echo

    for rep_url in `cat "$CONF_FILE" | grep ^baseurl | cut -f2 -d=`
    do
      echo Updating packages from \"$rep_url\"...

      runPrivileged $YOU -u "$rep_url" security recommended patchlevel document optional

      if test "$?" = "255"; then
        echo "ERROR: Could not update packages!" 1>&2
	exit 1
      fi
    done

    echo
    echo Updating host configuration information...

    # We need to find the current host's target name as known to EM.
    # We use "emctl config agent listtargets" and look for a line of
    # the form:
    #   [foo.bar.com, host]
    #
    for myhost in `emctl config agent listtargets | grep "host]$" | sed 's/, host]$//' |cut -c2-`
    do
      emctl control agent runCollection $myhost:host Inventory
    done
  fi

  # Done updating packages.
else
  # We are executing upon a reboot...
  # (...and already have root privileges)
  #
  NEED_REBOOT=0
  # Note: At debug-level 0, "yum check-update" just prints the list of
  # packages that will be updated. "grep" returns a non-zero exit code
  # if nothing matches.
  $YUM -d 0 -c "$CONF_FILE" check-update | grep -if "$NRP_LIST_FILE"
  if test "$?" = "0"; then
    $YUM -d $DBG_LVL -t -c "$CONF_FILE" update `cat "$NRP_LIST_FILE"`
    if test "$?" = "0"; then
      NEED_REBOOT=1
    fi
  fi

  # Did any package get updated that needs a reboot again?
  if test "$NEED_REBOOT" = "1"; then
    # Yes.  Reboot the machine then.
    /sbin/shutdown -r now
  fi
fi
