#!/usr/local/bin/perl
# 
# $Header: emagent/sysman/admin/scripts/LL_OH_Utilities.pm /main/2 2011/11/30 12:01:32 hmodawel Exp $
#
# LL_OH_Utilities.pm
# 
# Copyright (c) 2010, 2011, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      LL_OH_Utilities.pm - A utility container, will be maintained as SDK
#
#    DESCRIPTION
#      This is a utility container for inventory collection related 
#      requirements, will be maintained as SDK and it's routines will
#      be exposed to other modules.
#
#    NOTES
#      <other useful comments, qualifications, etc.>
#
#    MODIFIED   (MM/DD/YY)
#    hmodawel      11/12/11 - add API to get WLS home for a given MW_HOME
#    sanjkuma      01/17/11 - Creation
#
use OH_Discovery;
use OH_Utilities;
use OH_OUI;
use emdcommon;
use strict;

package LL_OH_Utilities; 

@LL_OH_Utilities::ISA    = ('Exporter');
@LL_OH_Utilities::EXPORT = qw(
  &getCompVersion
  &getOracleHomes
  &getWLSOracleHome
  );

#################### sub getCompVersion ###################################
# If base version is not specified - it returns based on the              #
# components name.                                     
# Return value = string or undef                                          #
###########################################################################

sub getCompVersion
{
  my ($oracleHome, $compName, $baseVer) = @_;

  if (!(defined $oracleHome) || !(defined $compName))
  {
     emdcommon::EMD_PERL_ERROR("Either ORACLE_HOME or Component Name is passed as NULL, must specify BOTH");
     return undef;
  }

  # First of all comps.xml to be read and the Metric Collection Object
  # is to be filled

  my $error      = new Error();         # Error Object
  my $warnings   = new Warn();          # Warnings Object
  my $metricInfo = new OHConfigMetric();# Metric data object
  my $inventory  = undef;               # Required only to get the OH name

  # logging the input to the API
  if ($baseVer)
  {
      emdcommon::EMD_PERL_DEBUG("getCompVersion called with ORACLE_HOME = $oracleHome Component = $compName Version = $baseVer");
  } 
  else {
      emdcommon::EMD_PERL_DEBUG("getCompVersion called with ORACLE_HOME = $oracleHome Component = $compName");
  }

  emdcommon::EMD_PERL_DEBUG("Creating an OUI Object");
  # Create an OUI object to collect the required data
  my $collectionObj = OH_OUI->new(  home        => $oracleHome,
                                    inventory   => $inventory,   # Passing Inventory location as NULL as we don't have inv loc
                                    error_obj   => $error );

  # Check the status of creation and exit on failure
  if (!defined($collectionObj))
  {
      # Read the info about the error from error object
      emdcommon::EMD_PERL_ERROR("OUI object creation failed for ORACLE_HOME = $oracleHome Component = $compName");
      exit($error->{CODE});
  }

  # This will read comps.xml (via collectCompsXml) and fill in the OUI object
  my $retVal = $collectionObj->getComponents (metric_info => $metricInfo,
                                              warnings    => $warnings );

  # Report the collection Errors
  if($retVal)
  {
     emdcommon::EMD_PERL_ERROR("Collection failed for ORACLE_HOME = $oracleHome Component = $compName");
     exit($error->{CODE});
  }

  emdcommon::EMD_PERL_DEBUG("Succesfully collected details on $oracleHome");

 # Now traverse the component data to return the version details for the
 # component being enquired for ...
 my $complist_ref = $collectionObj->{COMPS_DATA}->getComponents();
 foreach my $idx (sort keys(%{$complist_ref})) {
   if($complist_ref->{$idx}->{_NAME} eq $compName) {
     if((defined($baseVer) && ($complist_ref->{$idx}->{_VER} eq $baseVer))||(!defined($baseVer))) { #component found
        return $complist_ref->{$idx}->{_CURR_VER};
     }
   }
 }
 emdcommon::EMD_PERL_WARN("$compName DOES NOT EXIST or is the different base version, in Oracle Home $oracleHome");
 return undef ;# Could not find the component
}

####################### sub getOracleHomes #################################
# Returns all the Oracle Homes (OH) under a given MiddleWare home          #
# Output : Return all Oracle Homes with MH in an array                     #
#          return empty list in case of errors
############################################################################

sub getOracleHomes 
{
    my ($MWHome) = @_;
    my @OHomes = ();
    if (!(defined $MWHome))
    {
       emdcommon::EMD_PERL_ERROR("getOracleHomes in MWH : must specify middleware home as parameter");
       return @OHomes;
    }
    $MWHome = OH_Utilities::removeTrailingSlash($MWHome);
    my $emState = undef;
    my $me_type = 'mwh';
    my $OS = $^O;

    # warnings object to collect warnings/errors during
    # during installation related file operations
    my $warnings = new Warn();
    #my $error    = new Error(); Discovery package treats all errors as warnings
    
    # logging the input
    emdcommon::EMD_PERL_DEBUG("getOracleHomes is called with MIDDLEWARE HOME = $MWHome");

    # Create the Discovery object to collect the list of Oracle Homes
    my $discovery = OH_Discovery->new(EMDROOT =>$MWHome,
                                      EMSTATE =>$emState,
                                      WARNINGS=>$warnings,
				      ME_LOC  =>$MWHome,
 				      ME_TYPE =>$me_type);

    # Check the status of creation and exit on failure
    if (!defined($discovery))
    {
        # Read the info about the error from error object
        emdcommon::EMD_PERL_ERROR("getOracleHomes: Discovery object creation failed for MIDDLEWARE HOME = $MWHome");
        # exit($error->{CODE}); no error object return empty
        return @OHomes;
    }

    my $OHTargets_ref = $discovery->getHomeTargets();

    foreach my $idx (sort keys(%{$OHTargets_ref})) 
    {
       # Traverse all Oracle Homes and return the OH if it's 
       # the child dir of the MW Home being enquired for ... 
       if (($OS eq "Windows_NT") || ($OS  eq "MSWin32"))
       {
           if ($OHTargets_ref->{$idx}->{INSTALL_LOCATION} =~ /^$MWHome/i)
           {
               push (@OHomes, $OHTargets_ref->{$idx}->{INSTALL_LOCATION});
           } 
       } else
       { # Linux & Unixes
           if ($OHTargets_ref->{$idx}->{INSTALL_LOCATION} =~ /^$MWHome/)
           {
               push (@OHomes, $OHTargets_ref->{$idx}->{INSTALL_LOCATION});
           } 
       }
    }
    return @OHomes;
}


####################### sub getWLSOracleHome #################################
# Returns WLS Oracle Home (OH) under a given MiddleWare home
# Output : Return WLS Oracle Home location
#          return empty list in case of errors
############################################################################

sub getWLSOracleHome
{
    my ($MWHome) = @_;
    if (!(defined $MWHome))
    {
       emdcommon::EMD_PERL_ERROR("getWLSOracleHome in MWH : must specify middleware home as parameter");
       return undef;
    }
    $MWHome = OH_Utilities::removeTrailingSlash($MWHome);
    my $emState = undef;
    my $me_type = 'mwh';
    my $OS = $^O;

    # warnings object to collect warnings/errors during
    # during installation related file operations
    my $warnings = new Warn();
    
    # logging the input
    emdcommon::EMD_PERL_DEBUG("getWLSOracleHome is called with MIDDLEWARE HOME = $MWHome");

    # Create the Discovery object to collect the list of Oracle Homes
    my $discovery = OH_Discovery->new(EMDROOT =>$MWHome,
                                      EMSTATE =>$emState,
                                      WARNINGS=>$warnings,
				      ME_LOC  =>$MWHome,
 				      ME_TYPE =>$me_type);

    # Check the status of creation and exit on failure
    if (!defined($discovery))
    {
        # Read the info about the error from error object
        emdcommon::EMD_PERL_ERROR("getWLSOracleHome: Discovery object creation failed for MIDDLEWARE HOME = $MWHome");
        # exit($error->{CODE}); no error object return empty
        return undef;
    }

    my $OHTargets_ref = $discovery->getHomeTargets();

    foreach my $idx (sort keys(%{$OHTargets_ref}))
    {
       # Traverse all Oracle Homes and return the OH if it's 
       # of type W (WLS Oracle Home)
       return $OHTargets_ref->{$idx}->{INSTALL_LOCATION} if ($OHTargets_ref->{$idx}->{HOME_TYPE} eq 'W');
    }
    #WLS Home not found
    return undef;

}

1;
