#
#  $Header: emagent/sysman/admin/scripts/MultiTableOut.pm /st_emagent_10.2.0.1.0/1 2009/03/10 05:29:23 pnayak Exp $
#
#
# Copyright (c) 2001, Oracle Corporation.  All rights reserved.  
#
#    NAME
#      MultiTableOut.pm - Multiple Table Out
#
#    DESCRIPTION
#      The class provides a generic way to output multiple data tables to 
#    one or more files. Multiple temporary files with names like 
#    file_name.ext.N, where N varies from 1 to the number of files 
#    are opened to allow for parallel output of large amounts of data. 
#    The class uses the File::Copy package in order to achieve the "merge" 
#    of files in the end. This class is an extension (though not a derived 
#    class) of the TableOut class.
#      The user provides some file elements and then appropriate table elements.
#    The file elements are as bellow, for the table elements, please refer to 
#    TableOut.pm.
#    - File Header, File Footer, Table Separator, Merged Tables, Buffer Size
#
#    NOTES
#      No extra punctuation or formating is added, so don't expect to see
#      newlines added for you (use a \n in the string where you want a
#      new line).
#
#    MODIFIED   (MM/DD/YY)
#     njagathe   11/07/01 - Separating the tmp space from the final destination.
#     asawant    09/10/01 - Allowing for composite file names.
#     asawant    06/18/01 - Creation
#
#

package MultiTableOut;
use TableOut;
use strict;
use IO::File;
use File::Copy;

# Trace is Off by default
my $TRACE_ON = 0;
sub trace
{
  if($TRACE_ON)
  {
    print "DEBUG: $_[0]"."\n";
  };
}

sub ToggleTrace
{
  $TRACE_ON = 1 - $TRACE_ON;
}

# The properties of an object of this class are:
# (
#   file_config_hash
#   tableout_array
# )

# Usage : new(class, err_ref, file_config_hash)
# A file configurations hash looks like this:
# {
#   file_name => <file_name>
#   file_header => <file_header>
#   file_footer => <file_footer>
#   table_separator => <table_separator>
#   merged_tables => <merged_tables>
#   tmp_file_name => <temporary_base_file_name>, OPTIONAL, FILE_NAME WILL BE 
#     USED IF MISSING. MUST BE A FULL PATH AND FILE NAME
#   buf_size => <buffer_size> OPTIONAL, WILL BE ADDED BY CODE IF NOT PRESENT
# }
# Return a new object of this class on success and undef on failure
sub new 
{
  my ($class, $err_ref, $file_config_hash) = @_;
  my %object_vars = 
  (
    file_config_hash => undef,
    tableout_array => []
  );
  my $self = {%object_vars};
  $self->{file_config_hash} = $file_config_hash; # Get the file configuration
  unless(defined($self->{file_config_hash}{merged_tables}) &&
    ((! $self->{file_config_hash}{merged_tables}) ||
    (defined($self->{file_config_hash}{file_name}) &&
     defined($self->{file_config_hash}{file_header}) &&
     defined($self->{file_config_hash}{file_footer}) &&
     defined($self->{file_config_hash}{table_separator}))))
  {
    $$err_ref = "Missing parameters.\n";
    return undef;
  };
  bless $self, $class;
  return $self;
}

# Usage : add_table(self, table_config_hash)
# Input: table_config_hash
# Output: undef on failure or the table number on success.
# A table configuration hash looks like this (this is only a ref copy, please
# look into TableOut.pm for the latest information):
# {
#   table_header => <table_header>
#   table_footer => <table_footer>
#   line_prefix => <line_prefix>
#   line_sufix => <line_sufix>
#   line_separator => <line_separator>
#   column_desc_array_ref => 
#     [[<col1_prefix>, <col1_sufix>] , ... , [<coln_prefix>, <coln_sufix>]]
#   column_separator => <column_separator>
# }
#
# Notes:
#   The column_desc_array_ref, is kept as a reference, and thus SHOULD NOT BE
# MODIFIED outside elsewhere.
sub add_table 
{
  my ($self, $err_ref, $table_config_hash, $file_name) = @_;

  if($self->{file_config_hash}{merged_tables})		# auto create name
  {
    if(exists($self->{file_config_hash}{tmp_file_name}))
    {
      $file_name = $self->{file_config_hash}{tmp_file_name}.".".
        @{$self->{tableout_array}}
    }
    else
    {
      $file_name = $self->{file_config_hash}{file_name}.".".
        @{$self->{tableout_array}}
    }
  }
  else
  {
    unless(defined($file_name))
    {
      $$err_ref = "Tables are 'no merge' and no file name was passed in.\n";
      return undef;
    };
  };

  # Create a new TableOut object for this table
  unless($self->{tableout_array}[@{$self->{tableout_array}}] = 
    new TableOut $err_ref,$file_name,$table_config_hash)
  {
    $$err_ref = "$${err_ref}Failed creating new TableOut.\n";
    return undef;
  };

  #return table number
  return @{$self->{tableout_array}} - 1;
}

# Usage: add_line(self, err_ref, table_number, table_row)
# Output: 1 on success and 0 on failure
sub add_line {
  my ($self, $err_ref, $table_number, $table_row) = @_;
  trace("Table Number: $table_number");
  unless($self->{tableout_array}[$table_number]->add_line($err_ref, $table_row))
  {
    $$err_ref = "$${err_ref}Failed adding line.\n";
    return 0;
  };
  return 1;
}

# Usage : merge_tables(err_ref, ref_table_out_array, config_hash_ref)
# This is a static function that takes a reference to an array of TableOut
# objects and merges them all in one file. 
# The file config hash looks like this:
# {
#   file_name => <file_name>
#   file_header => <file_header>
#   file_footer => <file_footer>
#   table_separator => <table_separator>
#   buf_size => size of a page on your system, default is 8192
# }
# Returns 1 on success and 0 on failure. Nothing is done to the TableOut objects
# on failure.
sub merge_tables
{
  my ($err_ref, $ref_table_out_array, $config_hash_ref) = @_;

  unless(exists $$config_hash_ref{buf_size})
  {
    $$config_hash_ref{buf_size} = 8192;
  };

  trace("Create/Open the output file");
  my $output_file = new IO::File;
  unless($output_file->open(">".$$config_hash_ref{file_name}))
  {
    $$err_ref = "$!\nFailed opening output file.\n";
    return 0;
  };

  # force the output to be flushed every print
  autoflush $output_file 1;

  trace("Print header to file.");
  unless(print $output_file $$config_hash_ref{file_header})
  {
    $$err_ref = "$!\nFailed printing file header.\n";
    return 0;
  };
  # force the output to be flushed every block (back to default)
  autoflush $output_file 0;

  while(my $next_table = shift(@$ref_table_out_array))
  {
    # Print table separator unless this is the last table or the table separator
    # is an empty string
    if((length $$config_hash_ref{table_separator}) && (@$ref_table_out_array))
    {
      # force the output to be flushed every print
      autoflush $output_file 1;

      unless(print $output_file "$$config_hash_ref{table_separator}")
      {
        $$err_ref = "$!\nFailed printing to file.\n";
        $output_file->close();
        return 0;
      };

      # force the output to be flushed every block (back to default)
      autoflush $output_file 0;
    };
    # Copy file to the output file
    unless(copy("$next_table->{file_name}",
      $output_file, $$config_hash_ref{buf_size}))
    {
      $$err_ref = "$!\nFailed copying file.\n";
      $output_file->close();
      return 0;
    };
    # Remove the table file
    unless(unlink($next_table->{file_name}))
    {
      $$err_ref = "$!\nFailed removing $next_table->{file_name}.\n";
      $output_file->close();
      return 0;
    };
  };

  trace("Print footer to file.");
  unless(print $output_file $$config_hash_ref{file_footer})
  {
    $$err_ref = "$!\nFailed printing file footer.\n";
    $output_file->close();
    return 0;
  };

  trace("Close output file.");
  unless($output_file->close())
  {
    $$err_ref = "$!\nFailed closing file.\n";
    return 0;
  };

  return 1;
}

# Usage: end_data(self, err_ref)
# Input: none
# Output: 1 on success 0 on failure
sub end_data 
{
  my ($self, $err_ref) = @_;

  trace("call end_data for all TableOut's");
  for(my $counter = 0; $counter < @{$self->{tableout_array}}; $counter++)
  {
    unless($self->{tableout_array}[$counter]->end_data($err_ref))
    {
      $$err_ref = "$${err_ref}Failed closing one of the tmp files.\n";
      $self->end_force();
      return 0;
    }
  }
  if($self->{file_config_hash}{merged_tables})	#merged tables
  {
    trace("Merge all tables.");
    unless(merge_tables($err_ref, $self->{tableout_array}, 
      $self->{file_config_hash}))
    {
       $$err_ref = "$${err_ref}Failed merging tables.\n";
       $self->end_force();
       return 0;
    };
  }
  return 1;
}

# Usage: end_force(self)
# Always returns 1. Abruptly closes all open files. The state of the output
# is not guaranteed to make sense.
sub end_force 
{
  my ($self) = @_;
  trace("Call end_force for all TableOut's");
  for(my $counter = 0; $counter < @{$self->{tableout_array}}; $counter++) 
  {
    $self->{tableout_array}[$counter]->end_force();
  };
  return 1;
}

# * Just printing a line to the file handle seems not to work, for some reason
# the lines end up after all the merges. Thus I have decided to append the
# original files with the table_separators.

# So the require or use succeeds
1;
