# $Header: emagent/sysman/admin/scripts/OH_Date.pm /main/2 2011/04/20 03:41:24 hmodawel Exp $
#
# Package : OH_Date.pm
#
# Copyright (c) 2002, 2011, Oracle and/or its affiliates. All rights reserved. 
#
#   NAME
#       OH_Date.pm
#
#   DESCRIPTION
#       Common date conversion utilities
#
#   NOTES
#
#   MODIFIED     (MM/DD/YY)
#      hmodawel   03/20/11 - handle different date formats from OUI/fusion
#                            App/OPatch
##*************************************************************

use strict;

package OH_Date;

require 5.005;
require Exporter;

#******************************************
#   Export Subroutines
#******************************************

@OH_Date::ISA    = ('Exporter');
@OH_Date::EXPORT = qw(
  &conDateToYYYMMDD
  );

use constant DASH_CHR   => '-';
use constant SEP => ':';
use constant NIL => '';


sub GMT_Conversion {
    my ($mon, $day, $year, $hhmmss, $indt, $tz) = @_;
    
  my  %Timezones = (

    # Timezone regions with +00:00 standard offset from GMT
    "Africa/Casablanca"  => 'GMT',
    "Africa/Freetown"    => 'GMT',
    "Africa/Nouakchott"  => 'GMT',
    "Atlantic/Canary"    => 'GMT',
    "Atlantic/Faeroe"    => 'GMT',
    "Atlantic/Madeira"   => 'GMT',
    "Atlantic/Reykjavik" => 'GMT',
    "Atlantic/St_Helena" => 'GMT',
    "Eire"               => 'GMT',
    "Etc/GMT"            => 'GMT',
    "Etc/GMT+0"          => 'GMT',
    "Etc/GMT-0"          => 'GMT',
    "Etc/GMT0"           => 'GMT',
    "Etc/Greenwich"      => 'GMT',
    "Europe/Belfast"     => 'GMT',
    "Europe/Dublin"      => 'GMT',
    "Europe/Lisbon"      => 'GMT',
    "Europe/London"      => 'GMT',
    "GB"                 => 'GMT',
    "GB-Eire"            => 'GMT',
    "GMT"                => 'GMT',
    "GMT+0"              => 'GMT',
    "GMT-0"              => 'GMT',
    "GMT0"               => 'GMT',
    "Greenwich"          => 'GMT',
    "Iceland"            => 'GMT',
    "Portugal"           => 'GMT',
    "UTC"                => 'GMT',
    "WET"                => 'GMT',

    # Timezone regions with +01:00 standard offset from GMT
    "Africa/Algiers"      => 'GMT+01:00',
    "Africa/Ceuta"        => 'GMT+01:00',
    "Africa/Tunis"        => 'GMT+01:00',
    "Arctic/Longyearbyen" => 'GMT+01:00',
    "CET"                 => 'GMT+01:00',
    "Etc/GMT-1"           => 'GMT+01:00',
    "Europe/Amsterdam"    => 'GMT+01:00',
    "Europe/Belgrade"     => 'GMT+01:00',
    "Europe/Berlin"       => 'GMT+01:00',
    "Europe/Bratislava"   => 'GMT+01:00',
    "Europe/Brussels"     => 'GMT+01:00',
    "Europe/Budapest"     => 'GMT+01:00',
    "Europe/Copenhagen"   => 'GMT+01:00',
    "Europe/Gibraltar"    => 'GMT+01:00',
    "Europe/Ljubljana"    => 'GMT+01:00',
    "Europe/Luxembourg"   => 'GMT+01:00',
    "Europe/Madrid"       => 'GMT+01:00',
    "Europe/Monaco"       => 'GMT+01:00',
    "Europe/Oslo"         => 'GMT+01:00',
    "Europe/Paris"        => 'GMT+01:00',
    "Europe/Prague"       => 'GMT+01:00',
    "Europe/Rome"         => 'GMT+01:00',
    "Europe/San_Marino"   => 'GMT+01:00',
    "Europe/Sarajevo"     => 'GMT+01:00',
    "Europe/Skopje"       => 'GMT+01:00',
    "Europe/Stockholm"    => 'GMT+01:00',
    "Europe/Tirane"       => 'GMT+01:00',
    "Europe/Vatican"      => 'GMT+01:00',
    "Europe/Vienna"       => 'GMT+01:00',
    "Europe/Warsaw"       => 'GMT+01:00',
    "Europe/Zagreb"       => 'GMT+01:00',
    "Europe/Zurich"       => 'GMT+01:00',
    "MET"                 => 'GMT+01:00',
    "Poland"              => 'GMT+01:00',

    # Timezone regions with +02:00 standard offset from GMT
    "Africa/Cairo"        => 'GMT+02:00',
    "Africa/Johannesburg" => 'GMT+02:00',
    "Africa/Khartoum"     => 'GMT+02:00',
    "Africa/Tripoli"      => 'GMT+02:00',
    "Africa/Windhoek"     => 'GMT+02:00',
    "Asia/Beirut"         => 'GMT+02:00',
    "Asia/Damascus"       => 'GMT+02:00',
    "Asia/Gaza"           => 'GMT+02:00',
    "Asia/Istanbul"       => 'GMT+02:00',
    "Asia/Jerusalem"      => 'GMT+02:00',
    "Asia/Nicosia"        => 'GMT+02:00',
    "Asia/Tel_Aviv"       => 'GMT+02:00',
    "EET"                 => 'GMT+02:00',
    "Egypt"               => 'GMT+02:00',
    "Etc/GMT-2"           => 'GMT+02:00',
    "Europe/Athens"       => 'GMT+02:00',
    "Europe/Bucharest"    => 'GMT+02:00',
    "Europe/Helsinki"     => 'GMT+02:00',
    "Europe/Istanbul"     => 'GMT+02:00',
    "Europe/Kaliningrad"  => 'GMT+02:00',
    "Europe/Kiev"         => 'GMT+02:00',
    "Europe/Minsk"        => 'GMT+02:00',
    "Europe/Riga"         => 'GMT+02:00',
    "Europe/Simferopol"   => 'GMT+02:00',
    "Europe/Sofia"        => 'GMT+02:00',
    "Europe/Tallinn"      => 'GMT+02:00',
    "Europe/Vilnius"      => 'GMT+02:00',
    "Israel"              => 'GMT+02:00',
    "Libya"               => 'GMT+02:00',
    "Turkey"              => 'GMT+02:00',

    # Timezone regions with +03:00 standard offset from GMT
    "Africa/Djibouti"  => 'GMT+03:00',
    "Africa/Mogadishu" => 'GMT+03:00',
    "Africa/Nairobi"   => 'GMT+03:00',
    "Asia/Aden"        => 'GMT+03:00',
    "Asia/Amman"       => 'GMT+03:00',
    "Asia/Baghdad"     => 'GMT+03:00',
    "Asia/Bahrain"     => 'GMT+03:00',
    "Asia/Kuwait"      => 'GMT+03:00',
    "Asia/Qatar"       => 'GMT+03:00',
    "Asia/Riyadh"      => 'GMT+03:00',
    "Etc/GMT-3"        => 'GMT+03:00',
    "Europe/Moscow"    => 'GMT+03:00',
    "Indian/Mayotte"   => 'GMT+03:00',
    "W-SU"             => 'GMT+03:00',

    # Timezone regions with +03:30 standard offset from GMT
    "Asia/Tehran" => 'GMT+03:30',
    "Iran"        => 'GMT+03:30',

    # Timezone regions with +04:00 standard offset from GMT
    "Asia/Aqtau"     => 'GMT+04:00',
    "Asia/Baku"      => 'GMT+04:00',
    "Asia/Dubai"     => 'GMT+04:00',
    "Asia/Muscat"    => 'GMT+04:00',
    "Asia/Tbilisi"   => 'GMT+04:00',
    "Asia/Yerevan"   => 'GMT+04:00',
    "Etc/GMT-4"      => 'GMT+04:00',
    "Europe/Samara"  => 'GMT+04:00',
    "Indian/Reunion" => 'GMT+04:00',

    # Timezone regions with +04:30 standard offset from GMT
    "Asia/Kabul" => 'GMT+04:30',

    # Timezone regions with +05:00 standard offset from GMT
    "Asia/Aqtobe"        => 'GMT+05:00',
    "Asia/Bishkek"       => 'GMT+05:00',
    "Asia/Karachi"       => 'GMT+05:00',
    "Asia/Tashkent"      => 'GMT+05:00',
    "Asia/Yekaterinburg" => 'GMT+05:00',
    "Etc/GMT-5"          => 'GMT+05:00',
    "Indian/Chagos"      => 'GMT+05:00',

    # Timezone regions with +05:30 standard offset from GMT
    "Asia/Calcutta" => 'GMT+05:30',

    # Timezone regions with +06:00 standard offset from GMT
    "Asia/Almaty"      => 'GMT+06:00',
    "Asia/Dacca"       => 'GMT+06:00',
    "Asia/Novosibirsk" => 'GMT+06:00',
    "Asia/Omsk"        => 'GMT+06:00',
    "Etc/GMT-6"        => 'GMT+06:00',

    # Timezone regions with +06:30 standard offset from GMT
    "Asia/Rangoon" => 'GMT+06:30',
    "Indian/Cocos" => 'GMT+06:30',

    # Timezone regions with +07:00 standard offset from GMT
    "Asia/Bangkok"     => 'GMT+07:00',
    "Asia/Jakarta"     => 'GMT+07:00',
    "Asia/Krasnoyarsk" => 'GMT+07:00',
    "Asia/Saigon"      => 'GMT+07:00',
    "Etc/GMT-7"        => 'GMT+07:00',
    "Indian/Christmas" => 'GMT+07:00',

    # Timezone regions with +08:00 standard offset from GMT
    "Asia/Chungking"     => 'GMT+08:00',
    "Asia/Harbin"        => 'GMT+08:00',
    "Asia/Hong_Kong"     => 'GMT+08:00',
    "Asia/Irkutsk"       => 'GMT+08:00',
    "Asia/Kashgar"       => 'GMT+08:00',
    "Asia/Kuala_Lumpur"  => 'GMT+08:00',
    "Asia/Kuching"       => 'GMT+08:00',
    "Asia/Macao"         => 'GMT+08:00',
    "Asia/Manila"        => 'GMT+08:00',
    "Asia/Shanghai"      => 'GMT+08:00',
    "Asia/Singapore"     => 'GMT+08:00',
    "Asia/Taipei"        => 'GMT+08:00',
    "Asia/Ujung_Pandang" => 'GMT+08:00',
    "Asia/Urumqi"        => 'GMT+08:00',
    "Australia/Perth"    => 'GMT+08:00',
    "Australia/West"     => 'GMT+08:00',
    "Etc/GMT-8"          => 'GMT+08:00',
    "Hongkong"           => 'GMT+08:00',
    "PRC"                => 'GMT+08:00',
    "ROC"                => 'GMT+08:00',
    "Singapore"          => 'GMT+08:00',

    # Timezone regions with +09:00 standard offset from GMT
    "Asia/Jayapura" => 'GMT+09:00',
    "Asia/Seoul"    => 'GMT+09:00',
    "Asia/Tokyo"    => 'GMT+09:00',
    "Asia/Yakutsk"  => 'GMT+09:00',
    "Etc/GMT-9"     => 'GMT+09:00',
    "Japan"         => 'GMT+09:00',
    "JST"           => 'GMT+09:00',
    "ROK"           => 'GMT+09:00',

    # Timezone regions with +09:30 standard offset from GMT
    "Australia/Darwin" => 'GMT+09:30',
    "Australia/North"  => 'GMT+09:30',

    # Timezone regions with +10:00 standard offset from GMT
    "Asia/Vladivostok"     => 'GMT+10:00',
    "Australia/Brisbane"   => 'GMT+10:00',
    "Australia/Lindeman"   => 'GMT+10:00',
    "Australia/Queensland" => 'GMT+10:00',
    "Etc/GMT-10"           => 'GMT+10:00',
    "Pacific/Guam"         => 'GMT+10:00',
    "Pacific/Saipan"       => 'GMT+10:00',

    # Timezone regions with +10:30 standard offset from GMT
    "Australia/Adelaide"    => 'GMT+10:30',
    "Australia/Broken_Hill" => 'GMT+10:30',
    "Australia/South"       => 'GMT+10:30',
    "Australia/Yancowinna"  => 'GMT+10:30',

    # Timezone regions with +11:00 standard offset from GMT
    "Asia/Magadan"        => 'GMT+11:00',
    "Australia/ACT"       => 'GMT+11:00',
    "Australia/Canberra"  => 'GMT+11:00',
    "Australia/Hobart"    => 'GMT+11:00',
    "Australia/LHI"       => 'GMT+11:00',
    "Australia/Lord_Howe" => 'GMT+11:00',
    "Australia/Melbourne" => 'GMT+11:00',
    "Australia/NSW"       => 'GMT+11:00',
    "Australia/Sydney"    => 'GMT+11:00',
    "Australia/Tasmania"  => 'GMT+11:00',
    "Australia/Victoria"  => 'GMT+11:00',
    "Etc/GMT-11"          => 'GMT+11:00',
    "Pacific/Noumea"      => 'GMT+11:00',

    # Timezone regions with +11:30 standard offset from GMT
    "Pacific/Norfolk" => 'GMT+11:30',

    # Timezone regions with +12:00 standard offset from GMT
    "Asia/Kamchatka"    => 'GMT+12:00',
    "Etc/GMT-12"        => 'GMT+12:00',
    "Kwajalein"         => 'GMT+12:00',
    "Pacific/Kwajalein" => 'GMT+12:00',
    "Pacific/Wake"      => 'GMT+12:00',
    "Pacific/Wallis"    => 'GMT+12:00',

    # Timezone regions with +13:00 standard offset from GMT
    "Asia/Anadyr"      => 'GMT+13:00',
    "Etc/GMT-13"       => 'GMT+13:00',
    "NZ"               => 'GMT+13:00',
    "Pacific/Auckland" => 'GMT+13:00',
    "Pacific/Fiji"     => 'GMT+13:00',

    # Timezone regions with +13:45 standard offset from GMT
    "NZ-CHAT"         => 'GMT+13:45',
    "Pacific/Chatham" => 'GMT+13:45',

    # Timezone regions with -01:00 standard offset from GMT
    "America/Scoresbysund" => 'GMT-01:00',
    "Atlantic/Azores"      => 'GMT-01:00',
    "Etc/GMT+1"            => 'GMT-01:00',

    # Timezone regions with -02:00 standard offset from GMT
    "America/Araguaina" => 'GMT-02:00',
    "America/Fortaleza" => 'GMT-02:00',
    "America/Maceio"    => 'GMT-02:00',
    "America/Noronha"   => 'GMT-02:00',
    "America/Sao_Paulo" => 'GMT-02:00',
    "Brazil/DeNoronha"  => 'GMT-02:00',
    "Brazil/East"       => 'GMT-02:00',
    "Etc/GMT+2"         => 'GMT-02:00',

    # Timezone regions with -03:00 standard offset from GMT
    "America/Asuncion"     => 'GMT-03:00',
    "America/Belem"        => 'GMT-03:00',
    "America/Boa_Vista"    => 'GMT-03:00',
    "America/Buenos_Aires" => 'GMT-03:00',
    "America/Cayenne"      => 'GMT-03:00',
    "America/Cuiaba"       => 'GMT-03:00',
    "America/Godthab"      => 'GMT-03:00',
    "America/Miquelon"     => 'GMT-03:00',
    "America/Montevideo"   => 'GMT-03:00',
    "America/Santiago"     => 'GMT-03:00',
    "Atlantic/Stanley"     => 'GMT-03:00',
    "Chile/Continental"    => 'GMT-03:00',
    "Etc/GMT+3"            => 'GMT-03:00',

    # Timezone regions with -03:30 standard offset from GMT
    "America/St_Johns"    => 'GMT-03:30',
    "Canada/Newfoundland" => 'GMT-03:30',

    # Timezone regions with -04:00 standard offset from GMT
    "America/Anguilla"    => 'GMT-04:00',
    "America/Caracas"     => 'GMT-04:00',
    "America/Curacao"     => 'GMT-04:00',
    "America/Goose_Bay"   => 'GMT-04:00',
    "America/Guadeloupe"  => 'GMT-04:00',
    "America/Halifax"     => 'GMT-04:00',
    "America/La_Paz"      => 'GMT-04:00',
    "America/Manaus"      => 'GMT-04:00',
    "America/Martinique"  => 'GMT-04:00',
    "America/Montserrat"  => 'GMT-04:00',
    "America/Porto_Velho" => 'GMT-04:00',
    "America/Puerto_Rico" => 'GMT-04:00',
    "America/St_Thomas"   => 'GMT-04:00',
    "America/Thule"       => 'GMT-04:00',
    "America/Tortola"     => 'GMT-04:00',
    "America/Virgin"      => 'GMT-04:00',
    "Atlantic/Bermuda"    => 'GMT-04:00',
    "Brazil/West"         => 'GMT-04:00',
    "Canada/Atlantic"     => 'GMT-04:00',
    "Etc/GMT+4"           => 'GMT-04:00',

    # Timezone regions with -05:00 standard offset from GMT
    "America/Bogota"               => 'GMT-05:00',
    "America/Cayman"               => 'GMT-05:00',
    "America/Detroit"              => 'GMT-05:00',
    "America/Fort_Wayne"           => 'GMT-05:00',
    "America/Grand_Turk"           => 'GMT-05:00',
    "America/Guayaquil"            => 'GMT-05:00',
    "America/Havana"               => 'GMT-05:00',
    "America/Indianapolis"         => 'GMT-05:00',
    "America/Indiana/Indianapolis" => 'GMT-05:00',
    "America/Indiana/Knox"         => 'GMT-05:00',
    "America/Indiana/Marengo"      => 'GMT-05:00',
    "America/Indiana/Vevay"        => 'GMT-05:00',
    "America/Jamaica"              => 'GMT-05:00',
    "America/Knox_IN"              => 'GMT-05:00',
    "America/Lima"                 => 'GMT-05:00',
    "America/Louisville"           => 'GMT-05:00',
    "America/Montreal"             => 'GMT-05:00',
    "America/New_York"             => 'GMT-05:00',
    "America/Panama"               => 'GMT-05:00',
    "America/Thunder_Bay"          => 'GMT-05:00',
    "America/Porto_Acre"           => 'GMT-05:00',
    "Brazil/Acre"                  => 'GMT-05:00',
    "Canada/Eastern"               => 'GMT-05:00',
    "Chile/EasterIsland"           => 'GMT-05:00',
    "Cuba"                         => 'GMT-05:00',
    "EST"                          => 'GMT-05:00',
    "EST5EDT"                      => 'GMT-05:00',
    "Etc/GMT+5"                    => 'GMT-05:00',
    "Jamaica"                      => 'GMT-05:00',
    "Pacific/Easter"               => 'GMT-05:00',
    "US/East-Indiana"              => 'GMT-05:00',
    "US/Eastern"                   => 'GMT-05:00',
    "US/Indiana-Starke"            => 'GMT-05:00',
    "US/Michigan"                  => 'GMT-05:00',

    # Timezone regions with -06:00 standard offset from GMT
    "America/Cambridge_Bay"    => 'GMT-06:00',
    "America/Cancun"           => 'GMT-06:00',
    "America/Chicago"          => 'GMT-06:00',
    "America/Costa_Rica"       => 'GMT-06:00',
    "America/El_Salvador"      => 'GMT-06:00',
    "America/Guatemala"        => 'GMT-06:00',
    "America/Iqaluit"          => 'GMT-06:00',
    "America/Managua"          => 'GMT-06:00',
    "America/Mexico_City"      => 'GMT-06:00',
    "America/Rankin_Inlet"     => 'GMT-06:00',
    "America/Regina"           => 'GMT-06:00',
    "America/Swift_Current"    => 'GMT-06:00',
    "America/Tegucigalpa"      => 'GMT-06:00',
    "America/Winnipeg"         => 'GMT-06:00',
    "CST"                      => 'GMT-06:00',
    "CST6CDT"                  => 'GMT-06:00',
    "Canada/Central"           => 'GMT-06:00',
    "Canada/East-Saskatchewan" => 'GMT-06:00',
    "Canada/Saskatchewan"      => 'GMT-06:00',
    "Etc/GMT+6"                => 'GMT-06:00',
    "Mexico/General"           => 'GMT-06:00',
    "US/Central"               => 'GMT-06:00',

    # Timezone regions with -07:00 standard offset from GMT
    "America/Boise"        => 'GMT-07:00',
    "America/Chihuahua"    => 'GMT-07:00',
    "America/Dawson_Creek" => 'GMT-07:00',
    "America/Denver"       => 'GMT-07:00',
    "America/Edmonton"     => 'GMT-07:00',
    "America/Inuvik"       => 'GMT-07:00',
    "America/Mazatlan"     => 'GMT-07:00',
    "America/Phoenix"      => 'GMT-07:00',
    "America/Shiprock"     => 'GMT-07:00',
    "America/Yellowknife"  => 'GMT-07:00',
    "Canada/Mountain"      => 'GMT-07:00',
    "Etc/GMT+7"            => 'GMT-07:00',
    "MST"                  => 'GMT-07:00',
    "MST7MDT"              => 'GMT-07:00',
    "Mexico/BajaSur"       => 'GMT-07:00',
    "Navajo"               => 'GMT-07:00',
    "US/Arizona"           => 'GMT-07:00',
    "US/Mountain"          => 'GMT-07:00',

    # Timezone regions with -08:00 standard offset from GMT
    "America/Dawson"      => 'GMT-08:00',
    "America/Ensenada"    => 'GMT-08:00',
    "America/Los_Angeles" => 'GMT-08:00',
    "America/Tijuana"     => 'GMT-08:00',
    "America/Vancouver"   => 'GMT-08:00',
    "America/Whitehorse"  => 'GMT-08:00',
    "Canada/Pacific"      => 'GMT-08:00',
    "Canada/Yukon"        => 'GMT-08:00',
    "Etc/GMT+8"           => 'GMT-08:00',
    "Mexico/BajaNorte"    => 'GMT-08:00',
    "PST"                 => 'GMT-08:00',
    "PST8PDT"             => 'GMT-08:00',
    "US/Pacific"          => 'GMT-08:00',
    "US/Pacific-New"      => 'GMT-08:00',

    # Timezone regions with -08:30 standard offset from GMT
    "Pacific/Pitcairn" => 'GMT-08:30',

    # Timezone regions with -09:00 standard offset from GMT
    "America/Anchorage" => 'GMT-09:00',
    "America/Juneau"    => 'GMT-09:00',
    "America/Nome"      => 'GMT-09:00',
    "Etc/GMT+9"         => 'GMT-09:00',
    "Pacific/Gambier"   => 'GMT-09:00',
    "US/Alaska"         => 'GMT-09:00',

    # Timezone regions with -09:30 standard offset from GMT
    "Pacific/Marquesas" => 'GMT-09:30',

    # Timezone regions with -10:00 standard offset from GMT
    "America/Adak"      => 'GMT-10:00',
    "America/Atka"      => 'GMT-10:00',
    "Etc/GMT+10"        => 'GMT-10:00',
    "HST"               => 'GMT-10:00',
    "Pacific/Fakaofo"   => 'GMT-10:00',
    "Pacific/Honolulu"  => 'GMT-10:00',
    "Pacific/Johnston"  => 'GMT-10:00',
    "Pacific/Rarotonga" => 'GMT-10:00',
    "Pacific/Tahiti"    => 'GMT-10:00',
    "US/Aleutian"       => 'GMT-10:00',
    "US/Hawaii"         => 'GMT-10:00',

    # Timezone regions with -11:00 standard offset from GMT
    "Etc/GMT+11"        => 'GMT-11:00',
    "Pacific/Midway"    => 'GMT-11:00',
    "Pacific/Niue"      => 'GMT-11:00',
    "Pacific/Pago_Pago" => 'GMT-11:00',
    "Pacific/Samoa"     => 'GMT-11:00',
    "US/Samoa"          => 'GMT-11:00',

    # Timezone regions with -12:00 standard offset from GMT
    "Etc/GMT+12" => 'GMT-12:00',

    # Timezone regions with mappings
    "Pacific/Apia"              => 'GMT-11:00',
    "America/Yakutat"           => 'GMT-09:00',
    "America/Menominee"         => 'GMT-06:00',
    "America/Rainy_River"       => 'GMT-06:00',
    "Pacific/Galapagos"         => 'GMT-06:00',
    "America/Nipigon"           => 'GMT-05:00',
    "America/Pangnirtung"       => 'GMT-06:00',
    "America/Antigua"           => 'GMT-04:00',
    "America/Aruba"             => 'GMT-04:00',
    "America/Dominica"          => 'GMT-04:00',
    "America/Glace_Bay"         => 'GMT-04:00',
    "America/Grenada"           => 'GMT-04:00',
    "America/Guyana"            => 'GMT-04:00',
    "America/Port_of_Spain"     => 'GMT-04:00',
    "America/Santo_Domingo"     => 'GMT-04:00',
    "America/St_Kitts"          => 'GMT-04:00',
    "America/St_Lucia"          => 'GMT-04:00',
    "America/St_Vincent"        => 'GMT-04:00',
    "America/Catamarca"         => 'GMT-03:00',
    "America/Cordoba"           => 'GMT-03:00',
    "America/Jujuy"             => 'GMT-03:00',
    "America/Mendoza"           => 'GMT-03:00',
    "America/Paramaribo"        => 'GMT-03:00',
    "America/Rosario"           => 'GMT-03:00',
    "Atlantic/South_Georgia"    => 'GMT-02:00',
    "Africa/Abidjan"            => 'GMT',
    "Africa/Bamako"             => 'GMT',
    "Africa/Banjul"             => 'GMT',
    "Africa/Bissau"             => 'GMT',
    "Africa/Conakry"            => 'GMT',
    "Africa/Dakar"              => 'GMT',
    "Africa/El_Aaiun"           => 'GMT',
    "Africa/Lome"               => 'GMT',
    "Africa/Monrovia"           => 'GMT',
    "Africa/Ouagadougou"        => 'GMT',
    "Africa/Sao_Tome"           => 'GMT',
    "Africa/Timbuktu"           => 'GMT',
    "Africa/Bangui"             => 'GMT+01:00',
    "Africa/Brazzaville"        => 'GMT+01:00',
    "Africa/Douala"             => 'GMT+01:00',
    "Africa/Kinshasa"           => 'GMT+01:00',
    "Africa/Lagos"              => 'GMT+01:00',
    "Africa/Libreville"         => 'GMT+01:00',
    "Africa/Luanda"             => 'GMT+01:00',
    "Africa/Malabo"             => 'GMT+01:00',
    "Africa/Ndjamena"           => 'GMT+01:00',
    "Africa/Niamey"             => 'GMT+01:00',
    "Africa/Porto-Novo"         => 'GMT+01:00',
    "Europe/Andorra"            => 'GMT+01:00',
    "Europe/Malta"              => 'GMT+01:00',
    "Europe/Vaduz"              => 'GMT+01:00',
    "Africa/Blantyre"           => 'GMT+02:00',
    "Africa/Bujumbura"          => 'GMT+02:00',
    "Africa/Gaborone"           => 'GMT+02:00',
    "Africa/Harare"             => 'GMT+02:00',
    "Africa/Kigali"             => 'GMT+02:00',
    "Africa/Lubumbashi"         => 'GMT+02:00',
    "Africa/Lusaka"             => 'GMT+02:00',
    "Africa/Maputo"             => 'GMT+02:00',
    "Africa/Maseru"             => 'GMT+02:00',
    "Africa/Mbabane"            => 'GMT+02:00',
    "Europe/Chisinau"           => 'GMT+02:00',
    "Africa/Addis_Ababa"        => 'GMT+03:00',
    "Africa/Asmera"             => 'GMT+03:00',
    "Africa/Dar_es_Salaam"      => 'GMT+03:00',
    "Africa/Kampala"            => 'GMT+03:00',
    "Antarctica/Syowa"          => 'GMT+03:00',
    "Indian/Antananarivo"       => 'GMT+03:00',
    "Indian/Comoro"             => 'GMT+03:00',
    "Indian/Mahe"               => 'GMT+04:00',
    "Indian/Mauritius"          => 'GMT+04:00',
    "Asia/Ashkhabad"            => 'GMT+05:00',
    "Asia/Dushanbe"             => 'GMT+05:00',
    "Asia/Samarkand"            => 'GMT+05:00',
    "Indian/Kerguelen"          => 'GMT+05:00',
    "Indian/Maldives"           => 'GMT+05:00',
    "Antarctica/Mawson"         => 'GMT+06:00',
    "Asia/Colombo"              => 'GMT+06:00',
    "Asia/Thimbu"               => 'GMT+06:00',
    "Antarctica/Davis"          => 'GMT+07:00',
    "Asia/Phnom_Penh"           => 'GMT+07:00',
    "Asia/Vientiane"            => 'GMT+07:00',
    "Antarctica/Casey"          => 'GMT+08:00',
    "Asia/Brunei"               => 'GMT+08:00',
    "Asia/Dili"                 => 'GMT+08:00',
    "Asia/Pyongyang"            => 'GMT+09:00',
    "Pacific/Palau"             => 'GMT+09:00',
    "Antarctica/DumontDUrville" => 'GMT+10:00',
    "Pacific/Port_Moresby"      => 'GMT+10:00',
    "Pacific/Truk"              => 'GMT+10:00',
    "Pacific/Yap"               => 'GMT+10:00',
    "Pacific/Funafuti"          => 'GMT+12:00',
    "Pacific/Majuro"            => 'GMT+12:00',
    "Pacific/Nauru"             => 'GMT+12:00',
    "Pacific/Tarawa"            => 'GMT+12:00',

    # Atlantic
    "ADT" => 'GMT-03:00',

    # Eastern
    "EDT" => 'GMT-04:00',

    # Central
    "CDT" => 'GMT-05:00',

    # Mountain
    "MDT" => 'GMT-06:00',

    # Pacific
    "PDT" => 'GMT-07:00',

    # Hawaii
    "HDT" => 'GMT-09:30',

    # Mideast/Riyadh
    "Mideast/Riyadh87" => 'GMT+03:07',
    "Mideast/Riyadh88" => 'GMT+03:07',
    "Mideast/Riyadh89" => 'GMT+03:07',

    # GMT
    "GMT+0"     => 'GMT',
    "GMT+1"     => 'GMT+01:00',
    "GMT+2"     => 'GMT+02:00',
    "GMT+3"     => 'GMT+03:00',
    "GMT+4"     => 'GMT+04:00',
    "GMT+5"     => 'GMT+05:00',
    "GMT+6"     => 'GMT+06:00',
    "GMT+7"     => 'GMT+07:00',
    "GMT+8"     => 'GMT+08:00',
    "GMT+9"     => 'GMT+09:00',
    "GMT+10"    => 'GMT+10:00',
    "GMT+11"    => 'GMT+11:00',
    "GMT+12"    => 'GMT+12:00',
    "GMT+13"    => 'GMT+13:00',
    "GMT-0"     => 'GMT',
    "GMT-1"     => 'GMT-01:00',
    "GMT-2"     => 'GMT-02:00',
    "GMT-3"     => 'GMT-03:00',
    "GMT-4"     => 'GMT-04:00',
    "GMT-5"     => 'GMT-05:00',
    "GMT-6"     => 'GMT-06:00',
    "GMT-7"     => 'GMT-07:00',
    "GMT-8"     => 'GMT-08:00',
    "GMT-9"     => 'GMT-09:00',
    "GMT-10"    => 'GMT-10:00',
    "GMT-11"    => 'GMT-11:00',
    "GMT-12"    => 'GMT-12:00',
    "UCT"       => 'GMT',
    "Universal" => 'GMT',
    "UTC"       => 'GMT',
    "Zulu"      => 'GMT'

    );

    my $instdt;

    if (isDate($day, $mon, $year) && isTime($hhmmss))
    {
        # Doing GMT conversion for the time
        $tz = getTimeZone() if ((!defined $tz) || (length($tz) == 0));
        my $tzID = $Timezones{$tz};

        my ($operator, $time_offst, $hr_offst, $min_offst);

        if (defined $tzID) {
           $operator = substr($tzID,3,1);
           $time_offst = substr($tzID,4,5);
           ($hr_offst, $min_offst) = split (/:/,$time_offst);
           my ($hr, $min, $sec) = split (/:/,$hhmmss);

           # Address the 12 hour clock
           if ($indt =~ /pm/i) { $hr_offst += 12; $min_offst += 0; }

           #Adjust the min and hr offset value based on timezone offset from GMT
           if ($operator eq '-') { $min += $min_offst; } else { $min -= $min_offst; }
           if($min >= 60) { $hr_offst +=1; $min -=60; } elsif ($min < 0) { $min +=60; $hr_offst -=1; }

           #Add the hr offset to the time and accomodate the impact on the date change
           if ($operator eq '-') { $hr += $hr_offst; } else { $hr -= $hr_offst; }

           if($hr >= 24) { $day +=1; $hr -=24; } elsif ($hr < 0) { $day -=1; $hr +=24; }

           if ($day <= 0) {
              if ($mon == 2) { $day += 31;}
              if (($mon == 3)&&(!($year%4))) { $day += 29;}
              if (($mon == 3)&&($year%4)) { $day += 28;}
              if (($mon == 1)||($mon == 5)||($mon == 7)||($mon == 8)||($mon == 10)||($mon == 12)) { $day += 30;}
              if (($mon == 4)||($mon == 6)||($mon == 9)||($mon == 11)) { $day += 31;}
              $mon -=1;
           } else {
              if (($mon == 2)&&(!($year%4))&&($day > 29)) { $day -= 29; $mon += 1;}
              if (($mon == 2)&&($year%4)&&($day > 28)) { $day -= 28; $mon += 1;}
              if ((($mon == 1)||($mon == 3)||($mon == 5)||($mon == 7)||($mon == 8)||($mon == 10)||($mon == 12))&&($day > 31)) { $day -= 31; $mon += 1;}
              if ((($mon == 4)||($mon == 6)||($mon == 9)||($mon == 11))&&($day > 30)) { $day -= 30; $mon += 1;}
           }

           if ($mon <= 0) { $mon +=12; $year -=1; } elsif ($mon > 12) { $mon -=12; $year +=1; }

           $hhmmss = zeroPad($hr) . ':' . zeroPad($min) . ':' . zeroPad($sec);
           $hhmmss .= ':00' if (length($hhmmss) <= 5);
   
           $instdt =
              $year . DASH_CHR
           . zeroPad($mon) . DASH_CHR
           . zeroPad($day) . ' ' . $hhmmss;
       }
     }
     $instdt  =~ s/[,]//g;

     return $instdt;
}

sub zeroPad($;$)
{
    my ($value, $n) = @_;
    $value = NIL if (!defined $value);
    $n     = 2  if (!defined $n);

    $value = trim($value);
    while (length($value) < $n)
    {
        $value = '0' . $value;
    }

    return $value;
}

sub getTimeZone
{

    # Method 1 to get time zone
    use POSIX;
    chomp(my $tz = strftime "%Z\n", localtime);

    return $tz;
}


sub isNumeric($)
{
    my ($value)  = @_;
    my $result = 0;
    if (defined $value)
    {
        $result = $value =~ m/^[0-9]+$/;
    }

    return $result;
}


sub isDay($)
{
    return isInRange($_[0], 1, 31);
}

sub isMonth($)
{
    return isInRange($_[0], 1, 12);
}

sub isWeekday($)
{
    return isInRange($_[0], 1, 7);
}

sub isYear($)
{
    return isInRange($_[0], 0, 2099);
}

sub isHour($)
{
    return isInRange($_[0], 0, 23);
}

sub isMins($)
{
    return isInRange($_[0], 0, 59);
}

sub isSecs($)
{
    return isInRange($_[0], 0, 59);
}

sub isTime($)
{
    my ($value) = @_;
    my @in     = ();
    my $result = 0;

    $value = NIL if (!defined $value);
    if (length($value) > 0)
    {
        @in = split (SEP, $value);
        if (scalar(@in) == 2)
        {
            $result = (isHour($in[0]) & isMins($in[1]));
        }
        elsif (scalar(@in) == 3)
        {
            $result = (isHour($in[0]) & isMins($in[1]) & isSecs($in[2]));
        }
    }

    return $result;
}

sub isDate {
    my ($day, $mon, $year) = @_;
    my ($isYear, $isMonth, $isDay, $result) = (undef, undef, undef, undef);

    # Address the case if the input timestamp has ',' in it
    $day  =~ s/[. ,]//g;
    $year =~ s/[. ,]//g;

    $isYear = 1 if (isInRange($year, 0, 2038));
    $isMonth = 1 if (isInRange($mon, 1, 12));
    $isDay = 1 if (isInRange($day, 1, 31));

    if ($isYear && $isMonth && $isDay) {
        $result = 1;
        if (!($year%4)) {
           $result = 0 if (($day > 29) && ($mon == 2));
        } else {
           $result = 0 if (($day > 28) && ($mon == 2));
        }
        $result = 0 if ((($mon == 4)||($mon == 6)||($mon == 9)||($mon == 11))&&($day > 30));
    }
    return $result;
}


sub isInRange($$;$)
{
    my ($value, $lowRange, $highRange) = @_;
    my $result = 0;
    if (isNumeric($value))
    {
        if (!defined $highRange)
        {
            $result = ($value >= $lowRange);
        }
        else
        {
            $result = ($value >= $lowRange) & ($value <= $highRange);
        }
    }

    return $result;
}

sub trim
{
    
    #Arguments:  string or array
    #Outputs  :  string or array
    #Function :  trim
    
    my (@out) = @_;
    for (@out)
    {
        s/^\s+//;
        s/\s+$//;
    }
    
    return wantarray ? @out : $out[0];
}
#---------------------------------------#

sub conDateToYYYMMDD {

  my %Months = (
    # en (English) "Thu Jun 19 18:11:11 EDT 2003"
    "January"   => '01',
    "February"  => '02',
    "March"     => '03',
    "April"     => '04',
    "May"       => '05',
    "June"      => '06',
    "July"      => '07',
    "August"    => '08',
    "September" => '09',
    "October"   => '10',
    "November"  => '11',
    "December"  => '12',
    "Jan"       => '01',
    "Feb"       => '02',
    "Mar"       => '03',
    "Apr"       => '04',
    "May"       => '05',
    "Jun"       => '06',
    "Jul"       => '07',
    "Aug"       => '08',
    "Sep"       => '09',
    "Oct"       => '10',
    "Nov"       => '11',
    "Dec"       => '12'
  );
  my %Weekdays = (
    "Sunday"    => '01',
    "Monday"    => '02',
    "Tuesday"   => '03',
    "Wednesday" => '04',
    "Thursday"  => '05',
    "Friday"    => '06',
    "Saturday"  => '07',
    "Sun"       => '01',
    "Mon"       => '02',
    "Tue"       => '03',
    "Wed"       => '04',
    "Thu"       => '05',
    "Fri"       => '06',
    "Sat"       => '07'
  );


  my ($indt) = @_;
  my ($year,$mon,$day,$wday,$hhmmss,$outdt,$tz, $date, $time);
  my $blnk_chr = ' ';
  my $srch_pat = '';
  my @in = split (' ', $indt);
  if ((scalar(@in) == 2) or (scalar(@in) == 1) ) # with or without hh:mm:ss
  #                                 then date is in format YYYY-MM-DD HH:MM:SS
  #                                 or in format MM-DD-YYYY HH:MM:SS
  #                                 or in format DD-MM-YYYY HH:MM:SS
  #                                 or in format DD-MMM-YYYY MMM=name of month
  #                                 or 2010-03-03-08:00
  {
      $year=substr($in[0], 0,  4);
      if (isYear($year)) # YYYY-MM-DD
      {
        $mon=substr($in[0], 5, 2);
        $day=substr($in[0], 8, 2);
      } else # MM-DD-YYYY or DD-MM-YYYY or DD-MMM-YYYY
      {
        $mon=substr($in[0], 0, 2);
        if (isMonth($mon)) #MM-DD-YYYY (preference)
        {
          $day=substr($in[0], 3, 2);
          if(isDay($day)) 
          {
            $year=substr($in[0], 6, 4);
          } else # DD-MMM-YYY with DD <= 12
          {
            $day=substr($in[0], 0, 2);
            $mon=substr($in[0], 3, 3);
            $mon=$Months{$mon};
            $year=substr($in[0], 7, 4);
          }
        } else #DD-MM-YYYY with DD > 12 or DD-MMM-YYYY
        {
           $day=substr($in[0], 0, 2);
           $mon=substr($in[0], 3, 2);
           $year=substr($in[0], 6, 4);
           if (!isMonth($mon)) 
           {
             $mon=substr($in[0], 3, 3);
             $mon=$Months{$mon};
             $year=substr($in[0], 7, 4);  
           }
        }
      }
      $outdt = $year . '-' . $mon . '-' . $day ;
      if (scalar(@in) == 2) # with hhmmss
      {
        # output in YYYY-MM-DD HH:MM:SS
        $outdt = $outdt . ' ' . $in[1];
      }
      elsif (length($in[0]) > 11) # long pole in case month is a letter
      {
        $hhmmss = substr($in[0],11,8);
        if (!isTime($hhmmss)) # mon is in 3 letters
        {
          $hhmmss = substr($in[0],12,8);
        }
        $outdt = $outdt . ' ' . $hhmmss;
      } 
  }
  elsif (scalar(@in) == 3)  #2011.Jan.03 04:30:57 PST
                              #2011-Jan-03 04:30:57 PST
                              #2011/Jan/03 04.30.57 PST
                              #04:30:57 2011.Jan.03 PST
  {
    # Find which one is time and which one is date
    if ($indt =~ /\./) { $srch_pat = '\.';
    } elsif ($indt =~ /-/) { $srch_pat = '-';
    } elsif ($indt =~ /\//) { $srch_pat = '\/';
    }

    ($date,$time) = split (' ', $indt);

    # change the date to have ' ' and time to have ':'
    if (($date =~ /\./)&&($time =~ /\./)) {
        $date =~ s/$srch_pat/$blnk_chr/g;
        $time =~ s/$srch_pat/:/g;
        $indt = $date.' '.$time;
    } elsif (($date =~ /-/)&&($time =~ /\./)) {
        $date =~ s/-/$blnk_chr/g;
        $time =~ s/\./:/g;
        $indt = $date.' '.$time;
    } else {
        $indt =~ s/$srch_pat/$blnk_chr/g;
    } # no need to handle case where time already has ':', no substitution

    $indt =~ s/\./ /g;

    my $timeoffset;

    @in   = split (' ', $indt);
    $year = $in[0];
    if (isYear($year))
    {
      $hhmmss = $in[3];
      $timeoffset = 0;
    } else {
      $year = $in[1];
      $hhmmss = $in[0];
      $timeoffset = 1;
    }
    $mon = $in[$timeoffset + 1];
    if ($mon =~ m/\D/) # alphabets
    {
       $mon = $Months{$mon};
    }
    if (isMonth($mon))
    {
      $day = $in[$timeoffset + 2];
      if (isDay($day))
      {
          $tz = $in[4];
      }
    }
    $outdt = GMT_Conversion($mon, $day, $year, $hhmmss, $indt, $tz);
  }
  elsif ((scalar(@in) == 5) or (scalar(@in) == 6)) #Dec 09 22:14:15 PST 2010 
                                                   #Dec 09 2010 22:14:15 GMT
                                                   #Thu Dec 09 22:14:15 PST 2010 or its korean (TODO)
                                                   #Mar 3, 2010 4:56:16 AM
  {
    my $isWday = 0;
    $wday = $in[0];
    # this may be actually month
    $mon=$Months{$wday};
    if (!isMonth($mon))
    {
      $isWday = 1; #if not a month, then its weekday
      $wday = $Weekdays{$wday};
    }

    #ignore the wday, thats just to check the format
    $mon = $in[$isWday + 0];
    $mon=$Months{$mon};
    if (isMonth($mon))
    {
        $day = $in[$isWday + 1];
        $day =~ s/[,]//g;
        if (isDay($day))
        {
          $hhmmss = $in[$isWday + 2];
          if (!isTime($hhmmss)) # its year then
          {
            $year = $in[$isWday + 2];
            $hhmmss = $in[$isWday + 3];
            $tz = $in[$isWday + 4];
            # this could be AM/PM, no pattern check as it may match some tz
            if ( ($tz eq 'am') || ($tz eq 'Am') || ($tz eq 'AM')
               ||($tz eq 'pm') || ($tz eq 'Pm') || ($tz eq 'PM') )
            {
              $tz = '';
            }
          } else # its time 
          {
            $tz = $in[$isWday + 3];
            $year = $in[$isWday + 4];
          }
          if (isYear($year))
          {
            $outdt = GMT_Conversion($mon, $day, $year, $hhmmss, $indt, $tz);
            #$outdt = $year . '-' . $mon . '-' . $day . ' ' . $hhmmss;
          }
        }
      }
    }
  return $outdt;
}
