# $Header: emagent/sysman/admin/scripts/OH_Discovery.pm /main/13 2012/03/09 17:36:31 jashukla Exp $
#
# OH_Discovery.pm
# 
# Copyright (c) 2010, 2012, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      OH_Discovery.pm - <one-line expansion of the name>
#
#    DESCRIPTION
#      <short description of component this file declares/defines>
#
#    NOTES
#      <other useful comments, qualifications, etc.>
#
#    MODIFIED   (MM/DD/YY)
#    irraju      02/23/12 - XbranchMerge irraju_bug-13645834 from main
#    irraju      01/05/12 - unique WLS target name
#    hmodawel    12/26/11 - remove fix for 12907385
#    hmodawel    07/06/11 - Hack - do not discover Plugin homes
#    hmodawel    06/09/11 - fix OH Target Name
#    ravpatel    06/01/11 - Removing redundant propeties
#    hmodawel    05/22/11 - read oraInst.loc from OracleHome
#    ravpatel    05/18/11 - Handling Short path on windows
#    hmodawel    04/29/11 - Check OUI homes in MW home
#    hmodawel    03/20/11 - Fix issues
#    chkaushi    03/11/11 - Changing HOME_BASE to MW_HOME
#    ravpatel    01/07/11 - Adding lookup for weblogic home
#    ravpatel    08/08/10 - Oracle Home Discovery Package
#    ravpatel    08/08/10 - Creation
#

package OH_Discovery;

require Exporter;

use strict;
use warnings;
use File::Basename;
use File::Spec();
use File::Spec::Functions;
use File::Path;
use Net::Domain qw(hostname hostfqdn hostdomain);
use Fcntl qw(:DEFAULT :flock);
use Digest::MD5;

use OH_WLS;
use OH_OUI;
use OH_InventoryOrCompositeHome;
use OH_Error;
use OH_Utilities;
use OH_ConfigClasses;
use OH_ParseOHPropXml;

use emdcommon_ocm;

use fields qw ( EMDROOT
                EMSTATE
                ME_LOC
                ME_TYPE
                WARNINGS
                HOMES
                TARGETS
                INVENTORIES
                COMPOSITE_HOMES
                MW_HOMES
                STATUS
            );

# Global Constants
use constant {
        MIDDLEWARE_HOME_TAG => 'middleware home',
        HOSTNAME_SHORT => hostname(),
        HOSTNAME_LONG => hostfqdn(),
        SUCCEEDED     => 'SUCCEEDED',
        FAILED        => 'FAILED',
        ERROR => 0,
        WARN  => 1,
        INFO  => 2,
        DEBUG => 3
        };

my $OS = $^O;
my $windows = (($OS eq "Windows_NT") || ($OS  eq "MSWin32")); 

sub new
{
    my ($class, %args) = @_;
    my $self           = fields::new(ref($class)||$class);

    $self->{EMDROOT}   = $args{EMDROOT};
    $self->{EMSTATE}   = $args{EMSTATE};
    $self->{WARNINGS}  = $args{WARNINGS};
    $self->{ME_LOC}    = OH_Utilities::removeTrailingSlash($args{ME_LOC}) if $args{ME_LOC};
    $self->{ME_TYPE}   = $args{ME_TYPE} if $args{ME_TYPE};
    $self->{STATUS}    = "NOT_RUN";

    return $self;
}

sub setME
{
    my ($self, %args) = @_;
    my $me_loc = $args{ME_LOC} if $args{ME_LOC};
    my $me_type = $args{ME_TYPE} if $args{ME_TYPE};

    if($me_loc && $me_type) 
    {
      $self->{ME_LOC} = $me_loc;
      $self->{ME_TYPE} = $me_type;
    }
    $self->{STATUS}    = "NOT_RUN";
}

sub reset
{
    my $self = shift;

    $self->{HOMES}   = undef;
    $self->{TARGETS} = undef;
    $self->{INVENTORIES} = undef;
    $self->{COMPOSITE_HOMES} = undef;
    $self->{MW_HOMES} = undef;
#    Commenting out as we are losing warnings in FixInv case.
#    $self->{WARNINGS}->reset();

    $self->{STATUS} = "NUT_RUN";
}


sub getAllInventories
{
    my $self = shift;
    my $DirToSearch = shift;
    my $agentHome = $self->{EMDROOT};
    my $emState   = $self->{EMSTATE};
    my $warnings  = $self->{WARNINGS};
    my @InvList;

    # first read oraInst.loc from Oracle Home or Homes within MW home
    if (defined($DirToSearch) && !($DirToSearch eq ''))
    {
      my $localInv;
      log_it("Looking for Inventory in $DirToSearch", DEBUG);

      my $oraInstloc = File::Spec->catfile($DirToSearch, 'oraInst.loc'); 
      if(-e "$oraInstloc")
      {
        log_it("$oraInstloc found, reading it for inventory", DEBUG); 
        $localInv = OH_Utilities::read_oraInst_loc($oraInstloc, $warnings);
        push(@InvList, $localInv);
        log_it("added $localInv to the inventory list", DEBUG);
      }
      else # it may be a middleware home, look in subdirectories
      {
        log_it("No oraInst.loc file found in $DirToSearch",DEBUG);
        if (opendir(MW_HOME, $DirToSearch))
        {
          while (my $dirEntry = readdir(MW_HOME))
          {
            # skip "." and ".." directories
            next if ($dirEntry eq "." || $dirEntry eq "..");
            # check in directories only
            my $OHwithinMW = File::Spec->catfile($DirToSearch, $dirEntry);
            next unless (-d "$OHwithinMW");
            # look for the oraInst.loc  file; read for inventory information, if it exists
            $oraInstloc = File::Spec->catfile($OHwithinMW, 'oraInst.loc');
            if (-e "$oraInstloc")
            {
              log_it("$oraInstloc found, reading it for inventory", DEBUG); 
              $localInv = OH_Utilities::read_oraInst_loc($oraInstloc, $warnings);
              push(@InvList, $localInv);
              log_it("added $localInv to the inventory list", DEBUG);
            }
          }
          closedir(MW_HOME);
        }
        else 
        {
          log_it("Unable to read directory $DirToSearch", DEBUG);
        }
      }
    }
  
    log_it("Looking for Central Inventory on the host",DEBUG);

    my $CInv = $self->getCentralInventory($warnings);

    if($CInv)
    {
      log_it("Found Central Inventory $CInv",DEBUG);
      push(@InvList,$CInv);
    }
    else
    {
      log_it("No Central Inventory Found on the host",DEBUG);
    }
 
    my $AgentInv = $self->getAgentInventory();

    if($AgentInv)
    {
      log_it("Found Agent Inventory $AgentInv",DEBUG);
      push(@InvList,$AgentInv);
    }
    else
    {
      log_it("No Agent Inventory Found on the host",DEBUG);
    }

    log_it("Looking for Additional Inventories on the host",DEBUG);

    my @addInvList = $self->getAdditionalInventories();

    log_it("Found Additional Inventories @addInvList",DEBUG);

    foreach my $addInv (@addInvList){
        if(OH_Utilities::verify_Entity_Not_in_List($addInv, \@InvList)){
            push(@InvList,$addInv);
        }
    }
    return @InvList;
}

sub getCentralInventory
{
    my $self = shift;
    my $warnings  = $self->{WARNINGS};

    my $instLoc;

    log_it("$OS Operating System detected",DEBUG);

    if($windows) {
        log_it("Reading Windows Registry",DEBUG);

        no strict;

        eval 'use Win32::TieRegistry;';
        $Registry->Delimiter("/");
        # LMachine is just a sorthand for HKEY_LOCAL_MACHINE
        $instLoc = $Registry->{"LMachine/Software/Oracle/inst_loc"};

        use strict; # Continue with strict

        return $instLoc;
    }
    else
    {
        if($OS eq 'solaris')  {
            $instLoc = "/var/opt/oracle/oraInst.loc";
        }
        else    {
            $instLoc = "/etc/oraInst.loc";
        }
    }
    return OH_Utilities::read_oraInst_loc($instLoc, $warnings);
}

sub getAgentInventory
{
    my $self = shift;

    my $agentHome = $self->{EMDROOT};
    my $warnings  = $self->{WARNINGS};

    my $agentInstLoc = File::Spec->catfile($agentHome,'oraInst.loc');

    if(-e "$agentInstLoc")
    {
      return OH_Utilities::read_oraInst_loc($agentInstLoc, $warnings);
    }
    else
    {
      log_it("No oraInst.loc file found in agent home $agentHome",WARN); #Should we display it in UI ?
    }
}

sub getAdditionalInventories
{
    my $self = shift;
    my $emState   = $self->{EMSTATE};
    my $warnings  = $self->{WARNINGS};

    # Handle the case of Discovery Object creation for API (of LL_OH_Utilities)
    # where we set EMSTATE as NULL during Discovery object creation
    if(!(defined $emState)) {
      return;
    }

    my $invAddFile = File::Spec->catfile($emState,'sysman','config','OUIinventories.add'); 
    my @InstList = ();
    my @InvList = ();

    log_it("Trying to open file $invAddFile",DEBUG);

    if ( open(INVADD, "<$invAddFile") )
    {
      # Try to aquire a shared read lock
      if(!(flock(INVADD, LOCK_SH)))
      {
        log_it("Couldn't acquire read-lock on $invAddFile, Please check if the file is in consistent state.",WARN);
      }

      my @lns = <INVADD>;
      close(INVADD);
      # No need to unlock. flock ensures that lock is released before closing a file. 

      foreach my $ln (@lns)
      {
        $ln =~ s/\n$//;
        if ($ln =~ /^\s*inventory:\s*([^\s]+)\s*$/i)
        {
          my $inst = $1;
          log_it("Found inventory entry for $inst",DEBUG);
          #add it to array
          push(@InstList, $inst);
        }
      }
      foreach my $instLoc (@InstList)
      {
        my $Invloc = OH_Utilities::read_oraInst_loc($instLoc, $warnings);
        if ($Invloc) { push(@InvList, $Invloc); }
      }
    }
    else
    {
      log_and_publish(OH_Error::getErrorMessage(OH_Utilities::checkPermissions($invAddFile), $invAddFile),$warnings,ERROR);
    }
    return @InvList;
}

sub getAllMWHomes
{
    my $self = shift;

    my $emState   = $self->{EMSTATE};
    my $warnings  = $self->{WARNINGS};
    my $beaHomeFile = undef;

    my $homeDir = $windows ? $ENV{'SYSTEMDRIVE'} : $ENV{'HOME'};
    $beaHomeFile = File::Spec->catfile($homeDir, "bea", "beahomelist");
   
    my @MWHomes = ();

    my $ouiHomesRef = $self->{HOMES};
    foreach my $homeLoc (keys %$ouiHomesRef)
    {
      my($fname, $parent) = fileparse($homeLoc);
      if ((OH_Utilities::isValidMWHome($parent)) && (OH_Utilities::verify_Entity_Not_in_List($parent, \@MWHomes)))
      {
        push(@MWHomes, $parent);
      }
    }

    if(open(BEAFILE,"<$beaHomeFile"))
    {
      log_it("Trying to aquire read lock on file $beaHomeFile",DEBUG);

      #Try to acquire a shared read lock
      if(!(flock(BEAFILE,LOCK_SH)))
      {
        log_and_publish("Couldn't acquire read-lock on $beaHomeFile, Please check if the file is in consistent state.",$warnings, ERROR);
      }

      my @beaHomesListing = <BEAFILE>;
      close(BEAFILE);
      my $beaHomesList = join ('',@beaHomesListing);
      $beaHomesList =~ s/\n//gs;
      $beaHomesList =~ s/(\s)*;(\s)*/;/gs ;

      my @beaHomes = split(/;/,$beaHomesList);

      foreach my $MWHome (@beaHomes)
      {
        log_it("Found Middleware Home $MWHome in $beaHomeFile", DEBUG);
        if(OH_Utilities::verify_Entity_Not_in_List($MWHome, \@MWHomes))
        {
          push(@MWHomes, $MWHome);
        }
      }
    }
    else
    {
      #Show warning only if the beahomelist file exists.
      log_and_publish(OH_Error::getErrorMessage(OH_Utilities::checkPermissions($beaHomeFile), $beaHomeFile),$warnings,ERROR) if -f $beaHomeFile;
    }

    # Handle the case of Discovery Object creation for API (of LL_OH_Utilities)
    # where we set EMSTATE as NULL during Discovery object creation
    # TODO : Why can't we edit OUIinventories.add for MWHome on Windows ?
    if(!(defined $emState) || ($windows)) {
      return @MWHomes;
    }
    my $invAddFile = File::Spec->catfile($emState,'sysman','config','OUIinventories.add');

    if ( open(INVADD, "<$invAddFile") )
    {
        log_it("Trying to aquire read lock on file $invAddFile",DEBUG);
        # Try to aquire a shared read lock
        if(!(flock(INVADD, LOCK_SH)))
        {
          log_and_publish("Couldn't acquire read-lock on $invAddFile, Please check if the file is in consistent state.",$warnings,WARN);
        }

        my @lns = <INVADD>;
        # close the file, it will also release the lock
        close(INVADD);

        foreach my $ln (@lns)
        {
          my $MWHOME_TAG = MIDDLEWARE_HOME_TAG;
          $ln =~ s/\n$//;
          if ($ln =~ /^\s*$MWHOME_TAG\s*:\s*([^\s]+)\s*$/i)
          {
            my $MWHome = $1;
            log_it("Found Middleware Home $MWHome in $invAddFile",DEBUG);
            #add it to Middleware Home List
            if(OH_Utilities::verify_Entity_Not_in_List($MWHome, \@MWHomes))
            {
              push(@MWHomes, $MWHome);
            }
          }
        }
    }
    else
    {
        log_and_publish(OH_Error::getErrorMessage(OH_Utilities::checkPermissions($invAddFile), $invAddFile),$warnings,ERROR);
    }
    return @MWHomes;
}



sub getHomes 
{
    my $self = shift;

    if ($self->{STATUS} eq 'NOT_RUN')
    {
        $self->run();
    }
    return $self->{HOMES};
}

sub getHomeTargets
{
    my $self = shift;

    if ($self->{STATUS} eq 'NOT_RUN')
    {
        $self->run();
    }
    return $self->{TARGETS};
}

sub getInventories
{
    my $self = shift;

    if ($self->{STATUS} eq 'NOT_RUN')
    {
       $self->run();
    }
    return $self->{INVENTORIES};
}

sub getMWHomes
{
    my $self = shift;

    if ($self->{STATUS} eq 'NOT_RUN')
    {
       $self->run();
    }
    return $self->{MW_HOMES};
}

sub getCompositeHomes
{
    my $self = shift;

    if ($self->{STATUS} eq 'NOT_RUN')
    {
       $self->run();
    }
    return $self->{COMPOSITE_HOMES};
}

sub run
{
    my $self = shift;

    my $agentHome = $self->{EMDROOT};
    my $emState   = $self->{EMSTATE};
    my $warnings  = $self->{WARNINGS};
    #TODO : Error ?

    my $me_loc = $self->{ME_LOC} ? $self->{ME_LOC} : undef;
    my $me_type = $self->{ME_TYPE} ? $self->{ME_TYPE} : undef;

    $self->{HOMES}     = {};
    $self->{TARGETS}   = {};

    if (defined $emState) {
       log_it("Started Oracle Home Discovery on host ".HOSTNAME_LONG.". Agent Home = $agentHome and State Dir = $emState", DEBUG);
    } else {
       log_it("Collecting details on Oracle Homes on host (API Call)".HOSTNAME_LONG, DEBUG);
    }

    if($me_loc && $me_type)
    {

      if($me_type eq 'oh')
      {
        # Check if its a weblogic home.
        # This code will work only if WLS Home is an immediate subdirectory of middleware home.
        my($fname, $parent) = fileparse($me_loc);

        if((($fname =~ /^weblogic/) || ($fname =~ /^wlserver/)) && OH_Utilities::isValidMWHome($parent))
        {
          $self->discoverMWHome($parent);
        }
        else
        {
          log_it("Searching for home $me_loc on the host.",DEBUG);
          my $ohInstLoc = File::Spec->catfile($me_loc,'oraInst.loc');
          my $inv = OH_Utilities::read_oraInst_loc($ohInstLoc, $warnings) if (-e "$ohInstLoc");
          if($inv)
          {
            log_it("Found inventory $inv in $ohInstLoc",DEBUG);
            my $invObj = new OH_InventoryOrCompositeHome( LOC=>$inv, WARNINGS=>$warnings);
            my $homeList_ref = $invObj->getHomes();
            my $home_ref = $homeList_ref->{$me_loc};
            if ($home_ref)
            {
              log_it("Found home $me_loc in inventory $inv",DEBUG);
              $self->addHome($me_loc, $home_ref);
              $self->{STATUS} = 'COMPLETE';
              return;
            }
           else
            {
              if($windows)
              {
                foreach my $home_loc (keys %$homeList_ref)
                {
                  if(lc(Win32::GetShortPathName($me_loc)) eq lc(Win32::GetShortPathName($home_loc)))
                  {
                    log_it("Found home $home_loc in inventory $inv",DEBUG);
                    $self->addHome($home_loc, $homeList_ref->{$home_loc});
                    $self->{STATUS} = 'COMPLETE';
                    return;
                  }
                }
              }
              log_it("Home $me_loc not found in inventory $inv",DEBUG);
            }
          }
          # We are here means the home is not found in the inventory specified in home/oraInst.loc
          if(OH_Utilities::isValidOUIHome($me_loc)) #Assuming its a stray Oracle Home
          {
            my @InvList = $self->getAllInventories($me_loc);

            foreach my $inv (@InvList)
            {
              my $invObj = new OH_InventoryOrCompositeHome( LOC=>$inv, WARNINGS=>$self->{WARNINGS});
              log_it("Searching Homes in Inventory $inv",DEBUG);

              my $newHomes_ref = $invObj->getHomes();
              if($newHomes_ref->{$me_loc})
              {
                log_it("Found Home $me_loc in Inventory $inv",DEBUG);
                $self->addHome($me_loc, $newHomes_ref->{$me_loc});
                $self->{STATUS} = 'COMPLETE';
                return;
              }
            }

            # We are here means the home is not found in any inventory
            if ($windows)
            {
              log_it("Searching Home $me_loc in Registry",DEBUG);

              no strict;
              eval 'use Win32::TieRegistry';
              $Registry->Delimiter("/");
              my $oracleRegBase = $Registry->{"HKEY_LOCAL_MACHINE/Software/Oracle/"};

              use strict; # Continue with strict

              foreach (keys %$oracleRegBase)
              {
                if (($_ =~ m|^HOME|) || ($_ =~ m|^KEY_|))
                {
                  my $homeRegInfo = $oracleRegBase->{$_};
                  if (lc(Win32::GetShortPathName($me_loc)) eq lc(Win32::GetShortPathName($homeRegInfo->{ORACLE_HOME})))
                  {
                    log_it("Found home $me_loc in registry", DEBUG);
                    my $homeRef = { _NAME => $homeRegInfo->{ORACLE_HOME_NAME},
                                    _HOME_LOC => $me_loc,
                                    _TYPE => 'O'};
                    $self->addHome($me_loc, $homeRef);
                    $self->{STATUS} = 'COMPLETE';
                    return;
                  }
                }
              }
            }
            # The home not found in registry either (Windows Only)
            my $homeRef = { _HOME_LOC => $me_loc,
                            _TYPE => 'O'};
            $self->addHome($me_loc, $homeRef);
          }
        }
        $self->{STATUS} = 'COMPLETE';
        return;
      }
      elsif(($me_type eq 'inv') || ($me_type eq 'ch'))
      {
        $self->discoverInventory($me_loc);
        $self->{STATUS} = 'COMPLETE';
        return;
      }
      elsif($me_type eq 'mwh')
      {
        $self->discoverMWHome($me_loc);

        my @ouiFAHomes = OH_Utilities::getOUIHomesInMWHome($me_loc);

        my @InvList = $self->getAllInventories($me_loc);

        foreach my $inv (@InvList)
        {
          my $invObj = new OH_InventoryOrCompositeHome( LOC=>$inv, WARNINGS=>$self->{WARNINGS});
          log_it("Searching Homes in Inventory $inv",DEBUG);

          my $newHomes_ref = $invObj->getHomes();
          foreach my $home_loc (keys %$newHomes_ref)
          {
            if(!(verify_Entity_Not_in_List($home_loc, \@ouiFAHomes)))
            {
              log_it("Found Home $home_loc in Inventory $inv",DEBUG);
              $self->addHome($home_loc, $newHomes_ref->{$home_loc});
            }
          }
        }

        foreach my $home_loc (@ouiFAHomes)
        {
          if (!($self->{HOMES}->{$home_loc}))
          {
            my $homeRef = {_HOME_LOC => $home_loc,
                           _TYPE => 'O',
                           _MW_HOME => $me_loc};
            $self->addHome($home_loc, $homeRef);
          }
        }

        $self->{STATUS} = 'COMPLETE';
        return;
      }
      else
      {
        my $msg = "Invalid Entity Type specified"; 
        $warnings->addWarning($msg);
        log_it($msg,ERROR);
        return;
      }
    }

    log_it("Searching all the Inventories on the host",DEBUG);
    
    my @InvList = $self->getAllInventories(); # dont pass home/mwhome here, all inventorieson host
    $self->{INVENTORIES} = \@InvList;
    log_it("Inventory List prepared. Inventories found are ".join(',',@InvList),DEBUG);

    foreach my $inv (@InvList)
    {
      $self->discoverInventory($inv);
    }

    log_it("Searching for Middleware Homes on the host.",DEBUG);
    my @MWHomeList = $self->getAllMWHomes();
    $self->{MW_HOMES} = \@MWHomeList;

    log_it("Middleware Homes found are ".join(',',@MWHomeList),DEBUG);

    foreach my $MWHome (@MWHomeList)
    {
      $self->discoverMWHome($MWHome);
    }
    $self->{STATUS} = "COMPLETE";
}

sub discoverInventory
{
    my $self = shift;
    my $inv = shift;

    my $invObj = new OH_InventoryOrCompositeHome( LOC=>$inv, WARNINGS=>$self->{WARNINGS});

    log_it("Searching Homes in Inventory $inv",DEBUG);

    my $newHomes_ref = $invObj->getHomes();

    foreach my $home_loc (keys %$newHomes_ref)
    {
      log_it("Found Home $home_loc in Inventory $inv",DEBUG);
      $self->addHome($home_loc, $newHomes_ref->{$home_loc});
    }
    my $compositeHomesRef = $invObj->getCompositeHomes();
    my @CHList = keys (%$compositeHomesRef);

    if ($self->{COMPOSITE_HOMES})
    {
      my $oldCHList = $self->{COMPOSITE_HOMES};
      push (@$oldCHList, @CHList)
    }
    else
    {
      $self->{COMPOSITE_HOMES} = \@CHList;
    }
}

sub discoverMWHome
{
    my $self = shift;
    my $MWHome = shift;

    my $warnings = $self->{WARNINGS};
    #create a error object to capture the error
    my $error = new Error();
    #create a result object
    my $result = new OHConfigMetric();

    log_it("Looking for Weblogic Homes in Middleware Home $MWHome",DEBUG);

    my $status = OH_WLS->getAllHomesInInventory(INVENTORY=>$MWHome,ERROR=>$error,WARNINGS=>$warnings,RESULT_OBJ=>$result);

    if($status)
    {
      my $errMsg = $error->getErrorString();
      log_and_publish("Error $errMsg encountered while reading Middleware Home $MWHome.",$warnings,ERROR);
    }
    else
    {
      my $newHomes_ref = $result->getRows();
      foreach my $home_loc (keys %$newHomes_ref)
      {
        log_it("Found Welblogic home $home_loc in Middleware Home $MWHome",DEBUG, $warnings);
        $self->addHome($home_loc, $newHomes_ref->{$home_loc});
      }
    }
}

sub addHome
{
  my $self = shift;
  my $home_loc = shift;
  my $homeInfo_ref = shift;

  my $homeList_ref  = $self->{HOMES};
  my $OHTargets_ref = $self->{TARGETS};
  my $warnings      = $self->{WARNINGS};
  $home_loc = OH_Utilities::removeTrailingSlash($home_loc);

  my $exHomeInfo = $homeList_ref->{$home_loc};

  if ($exHomeInfo && ($exHomeInfo->{_TYPE} eq 'O'))
  {
    if($homeInfo_ref->{_TYPE} eq 'O')
    {
      my $oldInv = $exHomeInfo->{_INVENTORY};
      log_it("Home $home_loc is already discovered thru Inventory $oldInv",DEBUG);

      my $ohInstLoc = File::Spec->catfile($home_loc,'oraInst.loc');
      my $primInv = OH_Utilities::read_oraInst_loc($ohInstLoc, $warnings);

      if($primInv && ($primInv ne $oldInv))
      {
        $exHomeInfo->{_INVENTORY} = $primInv;
        $OHTargets_ref->{$exHomeInfo->{_TARGET_NAME}}->{INVENTORY} = $primInv;
        log_it("Updated Inventory Value for home $home_loc. old = $oldInv, new = $primInv",DEBUG);
      }
      return;
    }
    else
    {
      #May happen in some cases when Weblogic Home is also registered with some inventory
      # We'll let new target overwrite the old one
      delete($OHTargets_ref->{$exHomeInfo->{_TARGET_NAME}});
    }
  }

  # Creating target propeties hash
  my ($fname, $path) = fileparse($home_loc);
  my $homeName = $homeInfo_ref->{_NAME} ? $homeInfo_ref->{_NAME} : $fname;
  my $home_idx = $homeInfo_ref->{_IDX}  ? $homeInfo_ref->{_IDX} : '0';

  if ($homeInfo_ref->{_TYPE} eq 'O')
  {
    $path = OH_Utilities::removeTrailingSlash($path);
    $homeInfo_ref->{_MW_HOME} = $path if (OH_Utilities::isValidMWHome($path));
    # hack start
    # dont discover plugin OHs, this needs to be removed later : TODO
    collectOHProps($homeInfo_ref, $warnings);
    if(defined($homeInfo_ref->{_CONTENT_TYPE})) 
    {
      #do not add this target and return;
      return if ($homeInfo_ref->{_CONTENT_TYPE} =~ /Plugin/);
    }# hack end
  }
 
  # Adding it to the global home list
  $homeList_ref->{$home_loc} = $homeInfo_ref; 

  #my $ctx = Digest::MD5->new;
  #$ctx->add($homeName.$home_loc);
  #my $digest = $ctx->hexdigest;
  # Add underscore after homeName and after digest
  #my $target_name = $homeName."_".$digest."_".HOSTNAME_SHORT;

  # Use home name, IDX, host name to reduce the probability of conflicting names
  # We can consider using fname or part of home loc to further reduce the conflicts
  my $target_name = "";
  if($homeInfo_ref->{_TYPE} eq 'O')
  {
   #OUI home ? add index in inv.xml - combination of homename +hostname +index reduces the conflicts
   $target_name = $homeName."_".$home_idx."_".HOSTNAME_SHORT;
  }
  else
  {
   #Generate unique 4 digit number taking home_path as input and use it. Same path always produces same four digits
   #Having same WLS version installed multiple times is a rare condition.  getUniqueHash reduces the conflicts of 
   # of the target name in that case.
   $target_name = $homeName."_".HOSTNAME_SHORT."_".getUniqueHash($home_loc);
  }

  $target_name =~ s/ //g;
  $target_name =~ s/\./_/g;

  my $OHTargetProp_ref = {};
  
  $OHTargetProp_ref->{INSTALL_LOCATION} = $home_loc; 
#  #req from MOS ... all targets should have OracleHome property, bug 12907385 
#  $OHTargetProp_ref->{OracleHome} = $home_loc; 
  $OHTargetProp_ref->{HOME_TYPE} = $homeInfo_ref->{_TYPE}; 
  $OHTargetProp_ref->{INVENTORY} = $homeInfo_ref->{_INVENTORY} if ($homeInfo_ref->{_INVENTORY});
  $OHTargetProp_ref->{MW_HOME} = $homeInfo_ref->{_MW_HOME} if ($homeInfo_ref->{_MW_HOME});
  $OHTargetProp_ref->{HOME_NAME} = $homeInfo_ref->{_NAME} if ($homeInfo_ref->{_NAME});

  $OHTargets_ref->{$target_name} = $OHTargetProp_ref;
  $homeInfo_ref->{_TARGET_NAME} = $target_name;
  log_it("Home $home_loc added to Home List",DEBUG);
}

#---
sub log_it
{
    my ($message, $code ) = @_;

    if ( DEBUG eq $code )
    {
      emdcommon_ocm::EMD_PERL_DEBUG($message);
    }
    elsif ( INFO eq $code)
    {
      emdcommon_ocm::EMD_PERL_INFO($message);
    }
    elsif (WARN eq $code)
    {
      emdcommon_ocm::EMD_PERL_WARN($message);
    }
    elsif (ERROR eq $code)
    {
      emdcommon_ocm::EMD_PERL_ERROR($message);
    }
}

sub log_and_publish
{
    my($message, $warnings, $code) = @_;
    $warnings->addWarning($message);
    log_it($message, $code);
}

#used in HACK -- remove this method when hack is removed
sub collectOHProps
{
  my $homeInfoRef = shift;
  my $warnings = shift;
  my $home_loc = $homeInfoRef->{_HOME_LOC};

  my $OHPropXml = File::Spec->catfile($home_loc,"inventory","ContentsXML","oraclehomeproperties.xml");

  my $failed = OH_Utilities::checkPermissions($OHPropXml);

  if ($failed)
  {
    $warnings->addWarning(OH_Error::getErrorMessage($failed, $OHPropXml));
  }
  else
  {
    my $OHProps = new OH_ParseOHPropXml(XML_FILE=>$OHPropXml);
    my $error = new Error();
    if (my $retVal =  $OHProps->getOHProperties($warnings,$error))
    {
      # something wrong while collecting OH Properties
      $warnings->addWarning($error->getErrorString());
    }
    else
    {
     # $homeInfoRef->{_HOME_GUID}   = $OHProps->{HOME_GUID};
     # $homeInfoRef->{_ARU_ID}      = $OHProps->{ARU_ID};
     # $homeInfoRef->{_ORACLE_BASE} = OH_OUI::getOracleHomeProperty("ORACLE_BASE", $OHProps);
     $homeInfoRef->{_CONTENT_TYPE}  = OH_OUI::getOracleHomeProperty("ContentType", $OHProps);
    }
  }
  return undef;
}

###
# Generate unique four digit number for given file path
#    give equal weight for each character in the path and 
#    same path always should return the same number.  
# INPUT : Path of the Oracle Home
# OUTPUT: four digit number
####
sub getUniqueHash
{ my $path = shift;
  use integer;
  my $hash = 0;
  foreach(split //, $path)
  {
     $hash = (31*$hash+ord($_))%10000;
  }
  return $hash;
}

1;

