# $Header: emagent/sysman/admin/scripts/OH_OUI.pm /main/17 2011/12/27 00:12:29 hmodawel Exp $
# Copyright (c) 2010, 2011, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      OH_OUI.pm - <Package to collect all the metrics related to OUI homes>
#
#    DESCRIPTION
#      <short description of component this file declares/defines>
#
#    NOTES
#      A new OUI object can be created by calling new with home location and inventory
#      as arguments. If the arguments passes the validation tests reference to the 
#      created OUI object is returned otherwise reference to the ERROR object is 
#      returned.
#      STRUCTURE OF THE ERROR OBJECT
#      class ERROR
#           {
#            CODE   => error code
#            ARGS   => on which entity above error occured
#           }
#      When a metric is called on the give OUI object. It returns reference to the
#      OHConfigMetric object. It has the following structure
#      class OHConfiMetric
#      {      
#        SUCCESS => Y/N based on whether metric collection has succeeded or not
#        ROWS    =>
#                [index(1)] => hash of metric name class
#                  {
#                    variable1 => value1,
#                    variable2 => value2,
#                    variable3 => value3,
#                    ......
#                    variableList=>{
#                                    [0] => FirstInTheList,
#                                    [1] => SecondInTheList
#                                    .......
#                                  }                
#                  }
#                [index(2)] => ......
#     }          ....
#
#     If one needs to print the metric result on their own, they need to be aware of this. 
#     or else they can use pretttyPrint method in Utilities.pm
#
#    MODIFIED   (MM/DD/YY)
#    hmodawel    12/23/11 - 12c Agent Compatible
#    hmodawel    12/10/11 - get comps/patches/bugfixed from Plugin Homes as well
#    hmodawel    07/06/11 - generalize getting properties method
#    ravpatel    06/02/11 - read <home>/oraInst.loc if inventory is empty
#    hmodawel    05/23/11 - check of empty inventory
#    hmodawel    05/04/11 - handle mounted locations
#    hmodawel    04/29/11 - handle no ARU_ID tags
#    hmodawel    03/20/11 - Fix issues
#    chkaushi    03/11/11 - Changing HOME_BASE to MW_HOME
#    sanjkuma    01/10/11 - Update to collect FusionApps patch details
#    schaluva    07/22/10 - Adding method to get user information
#    irraju      06/16/10 - Fixing Is Cloanable bug
#    irraju      05/25/10 - Adding error and warning objects
#    irraju      05/19/10 - incorporating the comments in orareview
#    irraju      05/17/10 - Perl Module to define all the methods
#                           to be called on OUI homes.
#    irraju      05/17/10 - Creation
#

package OH_OUI;

## USE DECLARATIONS
use strict;
use warnings;
use Carp;
use File::Spec;
use ias::simpleXPath;
use List::Util qw(max);
use OH_ParseCompsXml;
use OH_ConfigClasses;
use OH_ParseOHPropXml;
use OH_ParseCloneXml;
use OH_Utilities;
use OH_Error;
use OH_InventoryOrCompositeHome;
use OH_Date;

use emdcommon_ocm;

## CONSTANTS
use constant CONTENTS_XML => File::Spec->catfile("inventory","ContentsXML");
use constant COMPS_XML    => File::Spec->catfile(CONTENTS_XML,"comps.xml");
use constant OH_PROP_XML  => File::Spec->catfile(CONTENTS_XML,"oraclehomeproperties.xml");
use constant CLONE_XML    => File::Spec->catfile("inventory","Clone","clone.xml");
#use constant INV_XML      => File::Spec->catfile("ContentsXML","inventory.xml");

use constant FUSION_APP_HISTORY_XML => File::Spec->catfile("faPatchInventory","fusionAppsPatchHistory.xml");
use constant INSTALL_PLAT => 'install.platform'; 
use constant ERROR        => "E";
use constant COLLECTED    => "C";

## FIELDS
# This is an OUI home object. Home is charecterized by its location, inventory it belongs to, 
# components object, oracle_home properties object, inventory object and clone properties.
# In future if it requires any more files to be parsed, those objects should be attached to 
# the OUI object.
use fields qw (  HOME_LOC 
                 INVENTORY
                 COMPS_DATA 
                 INVENTORY_DATA 
                 OH_PROPS 
                 OUI_PLATFORM_ID 
                 CLONE_PROPS 
                 ARU_ID 
                 ARU_ID_DESC
                 USER_INFO
                 HOME_RW_STATUS
                 ERROR
                 WARNINGS
             );

# GLOBAL VARIABLES
# Not recommended

###############################################################################
#                   EXPOSED SUBROUTINES                                       #
###############################################################################

#********** new****************************************************************
# This creates a new OUI object and validates the arguments. Later various
# metric subroutines can be called on this object
# INPUT ARGS
#       1. Obsolute home location
#       2. Central Inventory
# RETURNS
#       1. OUI object on success


sub new
{
    my ($class, %args) 	= @_;
    #parse the arguments passed
    my $homeLoc 	= $args{home};
    my $inventory 	= $args{inventory};
    my $error           = $args{error_obj};
    
    #create the object with desired fields   
    my $self 		= fields::new(ref($class)||$class);
    
    # -- set the inventory location to the object
    $self->{INVENTORY} = $inventory if $inventory;
    $self->{ERROR}     = $error;

    # -- check existence and readability of home location
    #     return with appropriate exit status
    if(my $retVal = OH_Utilities::checkPermissions($homeLoc)) {
      # something wrong with the permissions
      $self->{ERROR}->setError( CODE  => $retVal,
                        ARG   => $homeLoc);
      return undef; #FAILED
    }
    $self->{HOME_LOC} = $homeLoc;

    # -- check existence and readability of inventory location, if passed
    #     return with appropriate exit status
    if ((defined $inventory) && !($inventory eq '')) {
      # -- check the inventory location now
      if(my $retVal = OH_Utilities::checkPermissions($inventory)) 
      {
        # something wrong with the permissions
        $self->{ERROR}->setError( CODE  => $retVal,
                        ARG   => $inventory);
         return undef; #FAILED
      }
    }

    return $self; #SUCCESS
}


#****************SUB getHomeInfo()*********************************************
# returns the home properties hash
#
# INPUT ARGS
#       1. OUI object reference
#
# RETURNS
#       OHConifigMetric Object
#       In case of failure, exit code corresponding to the exception occured will
#       be set in ERROR field of the Object. Check OH_Error for the list of
#       EXIT CODES  and their descriptions.
#
#******************************************************************************
sub getHomeInfo()
{
  # If homeproperties.xml and inventory.xml files are already parsed, use those objects directly
  # othewise start parsing them now. 
  # We need to parse the comps.xml to know the isClobable status of the home.
  #
  my ($self,%args) = (shift,@_);
  my $metricInfo   = $args{metric_info}; # to hold the metric data
  $self->{WARNINGS}= $args{warnings};    # to hold warnings. Made it part of OUI object.
                                         # It contains the collection warnings obtained
                                         # while collecting the recent metric

  my $homeInfoRef = new HomeInfoClass;
  $homeInfoRef->{_HOME_LOC}  = ($self->{HOME_LOC});
  #setting default values
  $homeInfoRef->{_TYPE}      = "O";
  $homeInfoRef->{_CRS}       = "N";
  $homeInfoRef->{_CLONABLE}  = "N";
  $homeInfoRef->{_RW_STATUS} = 'NRNW'; # not readable, not writable
  $homeInfoRef->{_ARU_ID} = -1;
  $homeInfoRef->{_OUI_PLATFORM_ID} = -1;

  if(!($self->{INVENTORY_DATA}))#if not collected already
  {
    my $failed = collectInvXml($self); #may fail in case of permission issue etc.

    if (!$failed)
    {
      $homeInfoRef->{_NAME} = $self->{INVENTORY_DATA}->{_NAME} if $self->{INVENTORY_DATA}->{_NAME};
      $homeInfoRef->{_CRS}  = $self->{INVENTORY_DATA}->{_CRS} if $self->{INVENTORY_DATA}->{_CRS};
    }
  }

  # It should be below  Collection of inventory data as the above may set INVENTORY
  $homeInfoRef->{_INVENTORY} = ($self->{INVENTORY}); 

  #now we get some info from the OracleHomeProperties.xml also. If not parsed already do that now...
  if(!defined($self->{OH_PROPS})) # if not already prased oh properties.xml
  {
    my $failed = collectOHPropXml($self);
    if (!$failed)
    {
      $homeInfoRef->{_HOME_GUID}       = $self->{OH_PROPS}->{HOME_GUID} if $self->{OH_PROPS}->{HOME_GUID};
      $homeInfoRef->{_ARU_ID}          = $self->{OH_PROPS}->{ARU_ID} if $self->{OH_PROPS}->{ARU_ID};
      $homeInfoRef->{_ORACLE_BASE}     = getOracleHomeProperty("ORACLE_BASE",$self->{OH_PROPS});
    }
    $homeInfoRef->{_ORACLE_BASE} = '' if(!defined($homeInfoRef->{_ORACLE_BASE})); #give it a storage in mem
  }

  # now process comps.xml to check if home is clonable or not
  if(!defined($self->{COMPS_DATA}))
  {
    #collect comps.xml now 
    my $failed = collectCompsXml($self);
    $homeInfoRef->{_CLONABLE} = $self->{COMPS_DATA}->{IS_CLONABLE} if (!$failed);
  }

  # Now read OUI Platform ID from <inventory>/install.platform.
  if(!defined($self->{OUI_PLATFORM_ID}))
  {
    if (defined $self->{INVENTORY}) #In Composite Home OUI install.platform does not exist
    {
      my $ouiPlatId = collectOUIPlatformID($self->{INVENTORY}, $self->{WARNINGS});
      $homeInfoRef->{_OUI_PLATFORM_ID} = $self->{OUI_PLATFORM_ID} = $ouiPlatId if $ouiPlatId;
    }
  }
  # this was supposed to be collected from comps.xml... never made through
  #$homeInfoRef->{_HOME_SIZE}=0; #give it a storage in mem
  # Now collect user info
    
  if(!defined($self->{USER_INFO}))
  {
    my $userInfo = OH_Utilities::getUserInfo($self->{HOME_LOC}, $self->{WARNINGS});
    if ($userInfo)
    {
      $self->{USER_INFO} = $userInfo;
      $homeInfoRef->{_OH_OWNER_ID}        = $userInfo->{OH_OWNER_ID};
      $homeInfoRef->{_OH_OWNER}           = $userInfo->{OH_OWNER};
      $homeInfoRef->{_OH_GROUP_ID}        = $userInfo->{OH_GROUP_ID};
      $homeInfoRef->{_OH_GROUP}           = $userInfo->{OH_GROUP};
      $homeInfoRef->{_OH_OWNER_GROUPS_ID} = $userInfo->{OH_OWNER_GROUPS_ID};
      $homeInfoRef->{_OH_OWNER_GROUPS}    = $userInfo->{OH_OWNER_GROUPS};
    }
  }

  # Collect Home RW status
  if (-e  $self->{HOME_LOC})
  {
     if (-r $self->{HOME_LOC})
     {
       if (-w  $self->{HOME_LOC})
       {
         # My be its mounted, try touching a file
         my $checkFile = File::Spec->catfile($self->{HOME_LOC}, "checkforRW");
         if ( open(CHKFORRW, ">$checkFile"))
         {
           $homeInfoRef->{_RW_STATUS} = 'RW';
           close(CHKFORRW);
           unlink $checkFile;
         }
         else {
           $homeInfoRef->{_RW_STATUS} = 'RO';
         }
       }
       else {
         $homeInfoRef->{_RW_STATUS} = 'RO';
       }
     } else {
       if (-w  $self->{HOME_LOC})
       {
         $homeInfoRef->{_RW_STATUS} = 'WO';
       }
       else {
         $homeInfoRef->{_RW_STATUS} = 'NRNW';
       }
     }
  }
  else # home does not exist, this will never be executed
  {
     $self->{_RW_STATUS} = 'NRNW';
  }


  # If we reached here, means all the xml files are collected successfully. 
  # Now organize data in the metric result object
  
  $metricInfo->{STATUS} = COLLECTED;
  $metricInfo->{ROWS}->{0}  = $homeInfoRef; # only one row 

  return OH_Error::SUCCESS;
}

sub getOracleHomeProperty
{
  my $propName = shift;
  my $ohProps = shift;

  if (defined($ohProps) && defined($propName))
  {
    my $propList = $ohProps->{PROP_LIST};

    if(defined($propList))
    {
      foreach my $index( sort keys %$propList )
      {
        my $prop = $propList->{$index};
        if($prop->{_NAME} eq "$propName")
        {
          return $prop->{_VAL};
        }
      }
    }
  }
  return undef;
}

#**********SUB getCRSNodes()***************************************************
# Returns a hash of all CRS nodes. CRS Nodes inforamtin is present in the inventory.xml
# in case of older homes and in oraclehomeproperties.xml in newer homes.
#
# INPUT ARGS
#     1. OUI Object reference
#
# RETURNS
#     1. hash reference.  Ecah entry holds a CRS node
#
#******************************************************************************

sub getCRSNodes
{
  my ($self,%args)  = (shift,@_);
  my $metricInfo    = $args{metric_info};
  $self->{WARNINGS} = $args{warnings};

  # Parse the oraclehomeproperties first, if this is a CRS home then return the list
  if(!defined($self->{OH_PROPS}))
  {
    my $failed = collectOHPropXml($self);
    if($failed)
    {
      #collection of oraclehomeproperties.xml failed
      return $failed;
    }
  }
  if(defined $self->{OH_PROPS}->{NODE_LIST})
  {
     #get the CRS nodes from oraclehomeproperties
     $metricInfo->{STATUS} = COLLECTED;
     $metricInfo->{ROWS}   = $self->{OH_PROPS}->{NODE_LIST};
     return OH_Error::SUCCESS;
  }
  if(!defined($self->{INVENTORY_DATA}))#if not collected already
  {
    my $failed = collectInvXml($self);
    if($failed)
    {
     #collection of inventory.xml failed
     return $failed;
    }
  }
  if(defined $self->{INVENTORY_DATA}->{NODE_LIST})
  {
   # we can get the info from inventory.xml
   $metricInfo->{STATUS} = COLLECTED;
   $metricInfo->{ROWS}   = $self->{INVENTORY_DATA}->{NODE_LIST};
   return OH_Error::SUCCESS;
  }

 # else return the empty object .. no CRS nodes are found
  $metricInfo->{STATUS} = COLLECTED;
  return OH_Error::SUCCESS;
  
}

#******************** getDepHomes()********************************************
# Returns a hash of all the dependent homes
#
# INPUT ARGS
#       1. OUI Object Reference
#
# RETURNS
#       1. hash  reference. Each entry holds a dependent home location
#
#*****************************************************************************
sub getDepHomes
{
  my ($self,%args)  = @_;
  my $metricInfo    = $args{metric_info};
  $self->{WARNINGS} = $args{warnings};
  if(!defined($self->{INVENTORY_DATA}))#if not collected already
  {
    my $failed = collectInvXml($self);
    if($failed)
    {
      #collection of inventory.xml failed
      return $failed;
    }
  }
  $metricInfo->{STATUS} = COLLECTED;
  $metricInfo->{ROWS}   = $self->{INVENTORY_DATA}->{DEP_ON_HOMES};
  return OH_Error::SUCCESS;
}


#*******SUB getComponents()****************************************************
# Returns a hash containing all the coponents in the current OUI home
#
# INPUT ARGS
#       1. OUI Object Reference
#
# RETURNS
#       1. Hash reference. Each entry holds a component information
#******************************************************************************
sub getComponents()
{
   my ($self,%args)  = (shift,@_);
   my $metricInfo    = $args{metric_info};
   $self->{WARNINGS} = $args{warnings};

  if(!defined($self->{COMPS_DATA})) #if not parsed already
  {   
      if(my $failed = collectCompsXml($self)) {
       #collection of comps.xml failed
       return $failed;
      }
  }
  $metricInfo->{STATUS} = COLLECTED;
  $metricInfo->{ROWS}   = $self->{COMPS_DATA}->getComponents();
  return OH_Error::SUCCESS;

}


#*********SUB getTopLevelComponents()******************************************
# Returns a hash containing all the top level coponents in the current OUI home
#
# INPUT ARGS
#       1. OUI Object Reference
#
# RETURNS
#       1. hash. Each hash entry contains a TL component
#
#******************************************************************************
sub getTopLevelComponents()
{
    my ($self,%args)  = (shift,@_);
    my $metricInfo    = $args{metric_info};
    $self->{WARNINGS} = $args{warnings};
    if(!defined($self->{COMPS_DATA}))
    {
      my $failed = collectCompsXml($self);
      if($failed)
       {
        #collection of comps.xml failed
        return $failed;
      }
    }
    $metricInfo->{STATUS} = COLLECTED;
    $metricInfo->{ROWS}   = $self->{COMPS_DATA}->getTLComponents();
    return OH_Error::SUCCESS;
}


#*********SUB getComponentDependencies()***************************************
# Returns a hash containing all the coponent dependencies in the current OUI home
#
# INPUT ARGS
#       1. OUI Object Reference
#
# RETURNS
#       1. a hash reference , each hash entry contains one dependency
#
#******************************************************************************
sub getComponentDependencies()
{
    my ($self,%args)  = (shift,@_);
    my $metricInfo    = $args{metric_info};
    $self->{WARNINGS} = $args{warnings};
    if(!defined($self->{COMPS_DATA}))
    {
      my $failed = collectCompsXml($self);
      if($failed)
      {
       #collection of comps.xml failed
        return $failed;
       }
     }
     $metricInfo->{STATUS} = COLLECTED;
     $metricInfo->{ROWS}   = $self->{COMPS_DATA}->getCompDeps();
     return OH_Error::SUCCESS; 
}

#**************SUB getPatchSets()**********************************************
# Returns a hash containing all the patchsets in the current OUI home
#
# INPUT ARGS
#       1. OUI Object Reference
#
# RETURNS
#       1. hash reference, each hash entry holds information about a patchset
#
#******************************************************************************
sub getPatchSets()
{
    my ($self,%args)  = (shift,@_);
    my $metricInfo    = $args{metric_info};
    $self->{WARNINGS} = $args{warnings};
    if(!defined($self->{COMPS_DATA}))
    {
      my $failed = collectCompsXml($self);
      if($failed)
      {
       #collection of comps.xml failed
       return $failed;
     }
    }
    $metricInfo->{STATUS} = COLLECTED;
    $metricInfo->{ROWS}   = $self->{COMPS_DATA}->getPatchSets();
    return OH_Error::SUCCESS;
}

#************SUB getVersionPatches()*******************************************
# Returns a hash containing all the versioned patches in the current OUI home
#
# INPUT ARGS
#       1. OUI Object Reference
#
# RETURNS
#       1. hash reference, each hash entry holding a pspatch
#
sub getVersionPatches()
{
    my ($self,%args)  = (shift,@_);
    my $metricInfo    = $args{metric_info};
    $self->{WARNINGS} = $args{warnings};
    if(!defined($self->{COMPS_DATA}))
    {
      my $failed = collectCompsXml($self);
      if($failed)
      {
       #collection of comps.xml failed
       return $failed;
      }
    }
    $metricInfo->{STATUS} = COLLECTED;
    $metricInfo->{ROWS}   = $self->{COMPS_DATA}->getVersionPatches();
    return OH_Error::SUCCESS;
}



#*************SUB getPatches()*************************************************
# Returns a hash containing all the patches in the current OUI home
#
# INPUT ARGS
#       1. OUI Object Reference
#
# RETURNS
#       1. hash reference, each hash entry holding a patch
#
#******************************************************************************
sub getPatches()
{
    my ($self,%args)  = (shift,@_);
    my $metricInfo    = $args{metric_info};
    $self->{WARNINGS} = $args{warnings};
    if(!defined($self->{COMPS_DATA}))
    {
      my $failed = collectCompsXml($self);
      if($failed)
      {
       #collection of comps.xml failed
        return $failed;
     }
    }
    # FUSION APP PATCHES
    # we do not require pfOUIMapping.xml
    getFusionAppPatches($self);

    $metricInfo->{STATUS} = COLLECTED;
    $metricInfo->{ROWS}   = $self->{COMPS_DATA}->getPatches();
    return OH_Error::SUCCESS;
}

#***********SUB getPatchedComps()**********************************************
# Returns a hash containing all the patches in the current OUI home
#
# INPUT ARGS
#       1. OUI Object Reference
#
# RETURNS
#       1. hash reference, each hash entry holding info about the component 
#          affected by the patch. In case of PSU , info include from_version 
#          and to_version
#
#******************************************************************************
sub getPatchedComps
{
    my ($self,%args)  = (shift,@_);
    my $metricInfo    = $args{metric_info};
    $self->{WARNINGS} = $args{warnings};
    if(!defined($self->{COMPS_DATA}))
    {
      my $failed = collectCompsXml($self);
      if($failed)
      {
       #collection of comps.xml failed
       return $failed;
      }
    }
    $metricInfo->{STATUS} = COLLECTED;
    $metricInfo->{ROWS}   = $self->{COMPS_DATA}->getPatchComps();
    return OH_Error::SUCCESS;
}

#******SUB getPatchedBugs******************************************************
# Returns a hash containing all the bugs fixed by patches in the current OUI home
#
# INPUT ARGS
#       1. OUI Object Reference
#
# RETURNS
#       1. hash reference, each hash entry holding info about the bugs fixed by 
#          the patch
#******************************************************************************
sub getPatchedBugs()
{
    my ($self,%args) = (shift,@_);
    my $metricInfo   = $args{metric_info};
    $self->{WARNINGS}= $args{warnings};
    if(!defined($self->{COMPS_DATA}))
    {
      my $failed= collectCompsXml($self);
      if($failed)
      {
       #collection of comps.xml failed
        return $failed;

      }
    }
    $metricInfo->{STATUS} = COLLECTED;
    $metricInfo->{ROWS}   = $self->{COMPS_DATA}->getPatchBugs();
    return OH_Error::SUCCESS;

}

#*********SUB PatchedFiles*****************************************************
# Returns a hash containing all the files affected by patches in the current OUI
# home
#
# INPUT ARGS
#       1. OUI Object Reference
#
# RETURNS
#       1. hash reference, each hash entry consisting of information about the
#          file patched
#******************************************************************************
#
sub getPatchedFiles()

{
    my ($self,%args) = (shift,@_);
    my $metricInfo   = $args{metric_info};
    $self->{WARNINGS}= $args{warnings};

    
    if(!defined($self->{COMPS_DATA}))
    {
      my $failed = collectCompsXml($self);
      if($failed)
      {
       #collection of comps.xml failed
        return $failed;
      }
    }
    $metricInfo->{STATUS} = COLLECTED;
    $metricInfo->{ROWS}   = $self->{COMPS_DATA}->getPatchFiles();
    return OH_Error::SUCCESS;
}


#********** SUB getInstalTypes*************************************************
# Returns a hash containing all the install types of components in the current
# OUI home isntall type  of a component is replaced with the isntall type of 
# patch if present;
#
# INPUT ARGS
#       1. OUI Object Reference
# RETURNS
#       1. hash reference, Each entry holding one components install type.
#******************************************************************************
 sub getInstallTypes
 {
    my ($self,%args) = (shift,@_);
    my $metricInfo   = $args{metric_info};
    $self->{WARNINGS}= $args{warnings};
    if(!defined($self->{COMPS_DATA}))
    {
      my $failed= collectCompsXml($self);
      if($failed) #return value on failure
      {
        #collection of comps.xml failed
        return $failed;
      }
    }
    $metricInfo->{STATUS} = COLLECTED;
    $metricInfo->{ROWS}   = $self->{COMPS_DATA}->getCompInstTypes();
    return OH_Error::SUCCESS;
 }


#******* SUB cloneProperties***************************************************
# Returns a hash containing all the files affected by patches in the current 
# OUI home
#
# INPUT ARGS
#       1. OUI Object Reference
# RETURNS
#       1. Hash reference, Each hash entry containing all the clone properties
#*****************************************************************************
 sub getCloneProperties
 {
  my ($self,%args) = (shift,@_);
  my $metricInfo   = $args{metric_info};
  $self->{WARNINGS}= $args{warnings};
  
  if(!defined($self->{CLONE_PROPS}))
  {
    my $failed = collectCloneXml($self);
    if($failed)
    {
    # Collection of clone properties failed but we would not terminate the collection here
    # We'd return an empty result set instead and display appropriate warning.
      if ($failed != OH_Error::FILE_NOT_FOUND)
      {
        # We wont diplay an UI warning if the file doesn't exist
        $self->{WARNINGS}->addWarning($self->{ERROR}->getErrorString());
      }
      $self->{ERROR} = undef;
      $self->{CLONE_PROPS} = undef;

      return OH_Error::SUCCESS;
    }
  }
  $metricInfo->{STATUS} = COLLECTED;
  $metricInfo->{ROWS}   = $self->{CLONE_PROPS}->getCloneProperties();
  return OH_Error::SUCCESS;
 }

#*********SUB getAruId*********************************************************
# returns ARU ID
# Not being used by collection code but can be used by consumers
# just like the Java APIS exposed by OUI
sub getAruId()
{
    my ($self,%args) = (shift,@_);
    my $metricInfo   = $args{metric_info};
    $self->{WARNINGS}= $args{warnings};
    if(!defined($self->{OH_PROPS}))
    {
      my $failed= collectOHPropXml($self);
      if($failed)
      {
       #collection of comps.xml failed
       return $metricInfo;
      }
    }
    my $aruId = ($self->{OH_PROPS}->{ARU_ID}?$self->{OH_PROPS}->{ARU_ID}:OHUtilities::getARUId());
    $metricInfo->{STATUS}  = COLLECTED;
    $metricInfo->{ROWS}->{0}  = $aruId; #just one value- so just one row 
    return OH_Error::SUCCESS;
}

#***************SUB getAruIdDesc()*********************************************
# gives the ARU ID descriptions as provided in the oracle home properties
# Not being used by collection code but can be used by consumers
# just like the Java APIS exposed by OUI
sub getAruIdDesc()
{
    my ($self,%args) = (shift,@_);
    my $metricInfo   = $args{metric_info};
    $self->{WARNINGS}= $args{warnings};
    if(!defined($self->{OH_PROPS}))
    {
      my $failed= collectOHPropXml($self);
      if($failed)
      {
       return $metricInfo;
      }
    }
    $metricInfo->{STATUS}    = COLLECTED;
    $metricInfo->{ROWS}->{0} = $self->{OH_PROPS}->{ARU_ID_DESC};
    return OH_Error::SUCCESS; 
}


#***********SUB getHomeGuid()**************************************************
# Not being used by collection code but can be used by consumers
# just like the Java APIS exposed by OUI
# returns the homeguid as found in oraclehomeproperties file
#
sub getHomeGuid()
{
    # -- return $HOME_GUID
    my ($self,%args) = (shift,@_);
    my $metricInfo   = $args{metric_info};
    $self->{WARNINGS}= $args{warnings};
    if(!defined($self->{OH_PROPS}))
    {
      my $failed= collectOHPropXml($self);
      if($failed)
      {
       return $failed;
      }
    }
    $metricInfo->{STATUS}    = COLLECTED;
    $metricInfo->{ROWS}->{0} = $self->{OH_PROPS}->{HOME_GUID};
    return OH_Error::SUCCESS;
}


###############################################################################
#      INTERNAL SUBROUTINES                                                   #
###############################################################################

#*******collectCompsXML********************************************************
# sub routine to parse comps.xml
# This sub-routine parses comps.xml and puts the resulting comps object in the
# OUI object with the name COMPS_DATA
#
sub collectCompsXml
{
    my ($self, $loc) = @_;

    if(!defined($loc))
    {
      $loc = $self->{HOME_LOC}
    }
     
    # -- now parse the comps.xml file
    my $compsXmlFile = File::Spec->catfile($loc,COMPS_XML);
    emdcommon_ocm::EMD_PERL_DEBUG("Started collection of comps.xml:$compsXmlFile now ...");

    # -- check for the permissions
    my $failed;
    if($failed = OH_Utilities::checkPermissions($compsXmlFile))
    {
      $self->{WARNINGS}->addWarning(OH_Error::getErrorMessage($failed, $compsXmlFile));
      return $failed;
    }

    $self->{COMPS_DATA}= my $compsObject = new OH_ParseCompsXml(
                                                   WARNINGS => $self->{WARNINGS},
                                                   XML_FILE  => $compsXmlFile, 
                                                   HOME_LOC  => $loc);
    if(defined($compsObject)) 
    {
      my $retVal = $compsObject->collectInventory($self->{WARNINGS},$self->{ERROR});
       emdcommon_ocm::EMD_PERL_DEBUG("Started collection of comps.xml:$compsXmlFile");
      return $retVal;
    }

    $self->{ERROR}->setError( CODE => OH_Error::FAILED_TO_LOAD_MOD,
                              ARG  => "OH_ParseCompsXML");
    return OH_Error::FAILED_TO_LOAD_MOD;
}

#*******collectOHPropXML********************************************************
# sub routine to parse oracleproperties.xml
# This subroutine parses oraclehomeproperties.xml file and resulting object is placed in 
# the oui object with the name OH_PROPS 
#
sub collectOHPropXml
{
    my $self = shift;
    
    # -- get Oracle HOme properties from oraclehomeproperties.xml
    my $OHPXmlFile = File::Spec->catfile($self->{HOME_LOC},OH_PROP_XML);
    emdcommon_ocm::EMD_PERL_DEBUG("Started collection of oraclehomeproperties.xml: $OHPXmlFile");
 
    # -- check for the permissions
    if(my $failed = OH_Utilities::checkPermissions($OHPXmlFile))
    {
      $self->{WARNINGS}->addWarning(OH_Error::getErrorMessage($failed, $OHPXmlFile));
      return $failed;
    }

    $self->{OH_PROPS} = my $OHPObject = OH_ParseOHPropXml->new(XML_FILE => $OHPXmlFile);
    if(defined($OHPObject))
    {
      my $retVal = $OHPObject->getOHProperties($self->{WARNINGS},$self->{ERROR});
      emdcommon_ocm::EMD_PERL_DEBUG("Finished collection of oraclehomeproperties.xml: $OHPXmlFile");
      return $retVal;
    }
    $self->{ERROR}->setError( CODE  => OH_Error::FAILED_TO_LOAD_MOD,
                              ARG   => "OH_ParseOHPropXml");
    return OH_Error::FAILED_TO_LOAD_MOD;
}

#*******collectOUIPlatformID******************************************************
# sub routine to read OUI Platform ID from <Inventory>/install.platform
# This subroutine returns OUI Platform ID of an Oracle Home.
#
sub collectOUIPlatformID
{
  my $invLoc = shift;
  my $warnings = shift;
  my $platformFile = File::Spec->catfile($invLoc,INSTALL_PLAT) ;

  emdcommon_ocm::EMD_PERL_DEBUG("Opening file $platformFile for reading");

  if ( open(INSTPLAT, "<$platformFile") )
  {
    my @arr = (<INSTPLAT>);
    close INSTPLAT;
    emdcommon_ocm::EMD_PERL_DEBUG("Finished reading file $platformFile");
    my $line = join('',@arr);

    $line =~ s/^(.*?\s*\[Platform\]\s*ID=.*?)(.*)$/$2/s;
    if ($line =~ /\s*([^\s]+)\s*$/i)
    {
      return $1;
    }
  }
  else
  {
    my $failed = OH_Utilities::checkPermissions($platformFile);
    $warnings->addWarning(OH_Error::getErrorMessage($failed, $platformFile));
  }
  return -1; #failure reading Platform ID
}

#*******collectInvXML********************************************************
# subroutine to parse Inventory.xml
# This subroutine parses inventory xml and resulting object is placed in the 
# with INVENTORY_DATA in the OUI object
#
#
sub collectInvXml
{
    my $self = shift;

    my $invLoc = $self->{INVENTORY};

    if (!($invLoc))
    {
      # This code path is useful when an Oracle Home was stray home when added as a target but
      # later on it was attached to an inventory.
      my $ohInstLoc = File::Spec->catfile($self->{HOME_LOC},'oraInst.loc');
      $invLoc= OH_Utilities::read_oraInst_loc($ohInstLoc, $self->{WARNINGS}) if (-e "$ohInstLoc");
      $self->{INVENTORY} = $invLoc if $invLoc;
    }

    if ($invLoc)
    {
      my $invObject = new OH_InventoryOrCompositeHome(LOC => $invLoc,
                                                      WARNINGS => $self->{WARNINGS});

      my $error = $invObject->getError();

      if($error)
      {
        $self->{WARNINGS}->addWarning($error->getErrorString());
        return $error->{CODE};
      }
      $self->{INVENTORY_DATA}= $invObject->getHomeInvInfo($self->{HOME_LOC});
      return OH_Error::SUCCESS;
    }
    else
    {
      printStrayHomeWarning($self->{HOME_LOC}, $self->{WARNINGS});
    }
}

#*********collectCloneXML******************************************************
# subroutine to parse Clone.xml
# This subroutine parses the clone.xml and resulting object is palced in the OUI
# object with the name CLONE_PROPS
sub collectCloneXml
{
  my ($self) = shift;
  my $cloneFile = File::Spec->catfile($self->{HOME_LOC},CLONE_XML);
  emdcommon_ocm::EMD_PERL_DEBUG("Starting collection of clone.xml:$cloneFile");

  # -- check for the permissions
  if(my $failed = OH_Utilities::checkPermissions($cloneFile))
   {
     $self->{ERROR}->setError( CODE  => $failed,
                       ARG   => $cloneFile);
     return $failed;
   }

  # -- create the clone properties object
  $self->{CLONE_PROPS} = my $clonePropsObject = OH_ParseCloneXml->new(XML_FILE => $cloneFile );
  if($clonePropsObject)
  {
    my $retVal = $clonePropsObject->collectCloneProperties(WARNINGS => $self->{WARNINGS},
                                                           ERROR    => $self->{ERROR});
    emdcommon_ocm::EMD_PERL_DEBUG("Finishded collection of clone.xml:$cloneFile");
    return $retVal; #SUCCESS or FAILURE no matter what
  }

  $self->{ERROR}->setError( CODE  => OH_Error::FAILED_TO_LOAD_MOD,
                    ARG   => "OH_ParseCloneXml");
  return OH_Error::FAILED_TO_LOAD_MOD;
}

sub getFusionAppPatches
{
  # Get the patches from the patch history files
  # for each patch
  # Is it already in the oneoffs hash?
    # NO  -put it as a new oneoff
    # YES - update the lang/patch type info to the patch object

  my $collection_obj = shift;
  my $home_loc = $collection_obj->{HOME_LOC};

  # patch history xml will be inside faPatchInventory directory
  # There is no way to determine that this is APPL_TOP Oracle Home
  # if fusionAppsPatchHistory.xml exists in OH/faPatchInventory dir, assume OH=APPL_TOP
  # and parse the xml file
  my $appsPatchHistoryFileName = File::Spec->catfile($home_loc,FUSION_APP_HISTORY_XML);

  #Check if file exists and we have permissions to read it
  # Log warning only if it exists and is not readable
  if(my $failed = OH_Utilities::checkPermissions($appsPatchHistoryFileName))
  {
      if ($failed == OH_Error::FILE_PERM_DENIED )
      {
          $collection_obj->{WARNINGS}->addWarning(OH_Error::getErrorMessage($failed, $appsPatchHistoryFileName));
          return $failed; # no return catch outside, continue with other patches
      }
      else { # its not fusion app home, file does not exist
         return OH_Error::SUCCESS; 
      }
  }
  my $appsPatchHistoryFile = ias::simpleXPath::parseFile($appsPatchHistoryFileName);
   # Get a list of patches
  my @patches = ias::simpleXPath::queryForNodes($appsPatchHistoryFile, 'FAPatchInventory/PatchList/Patch');

  #Iterate through each patch and get the desired information
  foreach my $patch (@patches)
  {
    my @node_ouis = (ias::simpleXPath::queryForNodes($patch, 'Patch/CompList/Comp'));
    my $attributes = (ias::simpleXPath::queryForAttributes($patch, 'Patch'))[0];

    if (scalar(@node_ouis) > 0)
    {
      # Get the attributes
      my $patch_id = $attributes->{'ID'};
      my $patchRef = undef;
      unless($patchRef = $collection_obj->doesThisPatchExist($patch_id))
      {
        #create a new patch object
        my $patchlist_ref = $collection_obj->{COMPS_DATA}->getPatches();
        my $patchCount = List::Util::max(keys %{$patchlist_ref});
        $patchCount += 1;
        $patchlist_ref->{$patchCount} = $patchRef = new PatchClass;
        $patchRef->{_PATCH_ID} = $patch_id;
        $patchRef->{_IS_PSU} = 'N'; # these are not PSUs
        $patchRef->{_UPI} = 'N/A'; # default as it didn't exist in comps.xml
        $patchRef->{_LANG} = 'en'; # default
      }
      # Handle the variance in the value for IsRollbackable attribute
      $patchRef->{_ROLLBACK} = ($attributes->{'IsRollbackable'} =~ /^y/i)?"T":"F";
      $patchRef->{_LANG} = $attributes->{'Languages'};
      # convert the date into YYMMDD format...
      $patchRef->{_TIMESTAMP} = OH_Date::conDateToYYYMMDD($attributes->{'Date'});
      # _PATCH_TYPE has been added as an additinal field in PatchClass
      $patchRef->{_PATCH_TYPE} = $attributes->{'Type'};

   
      # Get the Bugs Fixed
      my @bugs_fixed = ias::simpleXPath::queryForAttributes($patch, 'Patch[@ID="' . $patch_id . '"]/BugfixList/Bugfix');
      my @bugList = ();
      my $bugCount = 0;
      # Iterate through the list of bugs fixed and add it to the bugs string
      foreach my $bug_fixed (@bugs_fixed)
      {
        my $bug = $bug_fixed->{'Number'};
        $bugList[$bugCount++] = $bug;
      }
      $patchRef->{_BUG_LIST} = [@bugList];
      #Get the components affected
      # Component names will be fully qualified,
      my @comps_affected = ias::simpleXPath::queryForAttributes($patch, 'Patch[@ID="' . $patch_id . '"]/CompList/Comp');
      my $compCount = 0;
      foreach my $comp_affected(@comps_affected)
      {
        my $comp = $comp_affected->{'Name'};
        my $ver  = $collection_obj->getCompBaseVer($comp);
        $patchRef->{_REF_LIST}->{$compCount++} = my $refComp = {};
        $refComp->{_NAME} = $comp;
        $refComp->{_VER}  = $ver;
      }

  }
 }
}

sub doesThisPatchExist
{
 my $self = shift;
 my $patchId = shift;
 my $oneoff_ref = $self->{COMPS_DATA}->getPatches();

 foreach my $idx (keys %{$oneoff_ref}) {
  if($oneoff_ref->{$idx}->{_PATCH_ID} eq $patchId) {
    return $oneoff_ref->{$idx};
  }
 }
 return 0; #patch does not exist
}

sub getCompBaseVer
{
  # given the component name, returns its base version.
  #
  my ($self, $name, $currVer) = @_ ;

  if (!(defined $name)) {
     emdcommon::EMD_PERL_DEBUG("getCompBaseVer: Component Name has to be specified to get it's version");
     return undef;
  }
  my $complist_ref = $self->{COMPS_DATA}->getComponents();

  foreach my $idx (sort keys(%{$complist_ref})) 
  {
    if($complist_ref->{$idx}->{_NAME} eq $name) 
    { #component found
      if(defined($currVer) && defined $complist_ref->{$idx}->{_CURR_VER}) 
      { # is current version specified ?
         if($complist_ref->{$idx}->{_CURR_VER} eq $currVer) 
         {
            return $complist_ref->{$idx}->{_VER};
         } else 
         {
            return 0; # there is no component with the given component name 
                   # and current version
         }
      }
      return $complist_ref->{$idx}->{_VER};# This is the special case for fusion app patches where
                                          # we just know the component name but not it's version
                                          # Its guaranteed that there will be only one entry for
                                          # any component.
    }
  }
  return 0; #Component doesn't exist
}

sub printStrayHomeWarning
{
  my $homeLoc = shift;
  my $warnings = shift;
  my $msg1 = "Oracle Home $homeLoc is not attached with any inventory. Some config data may be incomplete/incorrect";
  my $oui = File::Spec->catfile($homeLoc, "oui", "bin", "runInstaller" );
  my $command = "$oui -silent -attachHome ORACLE_HOME=\"$homeLoc\" ORACLE_HOME_NAME=\"<Home Name>\"";
  my $msg2 = "Please run $command to attach the home.";
  $warnings->addWarning($msg1);
  $warnings->addWarning($msg2);
  emdcommon_ocm::EMD_PERL_WARN($msg1);
  emdcommon_ocm::EMD_PERL_WARN($msg2);
}

1;

