#
# $Header: emagent/sysman/admin/scripts/OH_ParseInvXml.pm /main/8 2011/12/16 04:06:52 hmodawel Exp $
#
# OH_ParseInvXml.pm
#
# Copyright (c) 2010, 2011, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      OH_ParseInvXml.pm - <Package to collect all the metrics related to OUI homes>
#
#    DESCRIPTION
#      <short description of component this file declares/defines>
#
#    NOTES
#        This module parses inventory.xml and provides with various methods to 
#        access various properties of the oracle homes present in it.
#        ******** STRUCTURE OF %inventory hash object which will be returned***
#        inventory is a hash. It has the following structure
#        {HOME_LOC_1}->{NAME}
#                    ->{TYPE}
#                    ->{CRS}
#                    ->{CENTRAL_INV}            
#                    ->{NODE_LIST}
#
#    MODIFIED   (MM/DD/YY)
#    hmodawel    12/10/11 - collect Plugin Homes as part of core home
#    hmodawel    06/16/11 - handle PLUGIN attribute
#    hmodawel    06/09/11 - fix IDX
#    pguruswa    11/30/10 - check for composite inventory file
#    irraju      05/17/10 - Perl Module 
#                           to extarct info from inventory.xml.
#    irraju      05/17/10 - Creation
#

package OH_ParseInvXml;

# USE DECLARATIONS
use strict;
use warnings;
no warnings qw(redefine);
use XML::Parser;
use OH_Error;
use File::Spec;
use OH_InventoryOrCompositeHome;
use OH_Utilities ;#qw (removeTrailingSlash @encTable);

use emdcommon_ocm;

# GLOBAL VARIABLES
use fields qw(XML_FILE INV_LOC STATUS WARNINGS HOMES COMPOSITE_HOMES ERROR);

#********* SUB NEW ************************************************************
# CONSTRUCTOR
# Constructor requires absolute loaction of the inventory.xml file. Details 
# about the HOME_LOC are searched int he collected data object.
#
#******************************************************************************
sub new
{
  my ($class, %vals) = @_;
  my $self = fields::new(ref($class)||$class);
  $self->{WARNINGS} = $vals{WARNINGS};

  if(exists($vals{XML_FILE}) && exists($vals{INV_LOC}))
  {
    $self->{XML_FILE} = $vals{XML_FILE}; #absolute location of the inventory.xml file
    $self->{INV_LOC}  = $vals{INV_LOC}; # Absolute path to Inventory or Composite Home

    $self->{STATUS} = "NOT_COLLECTED";
  }
  else
  {
    my $error = new Error();
    $error->setError(CODE => OH_Error::MAND_ARG_MISSING, ARG =>"OH_ParseInvXml::new");
    $self->{ERROR} = $error;
    emdcommon_ocm::EMD_PERL_ERROR("Insufficient Arguments in call to $class"."::new");
  }
  return $self; #success
}

sub getData
{
  my $self = shift;
  my $key = shift;

  if($self->{STATUS} eq "NOT_COLLECTED")
  {
    $self->collect();
  }

  if($self->{STATUS} eq "FAILED")
  {
    return undef;
  }

  return (defined $key) ? $self->{$key} : undef;
}

sub getError
{
  my $self = shift;

  if($self->{STATUS} eq "NOT_COLLECTED")
  {
    $self->collect();
  }

  return (exists $self->{ERROR}) ? $self->{ERROR} : undef;
}

sub newParseObject
{
  my $invLoc = shift;

  my $parseObj = { INV_LOC          => $invLoc,
                   CURR_HOME        => undef,
                   IN_HOME_LIST     => "F",
                   IN_COMPHOME_LIST => "F",
                   IN_DEPHOME_LIST  => "F",
                   IN_REFHOME_LIST  => "F",
                   IN_NODE_LIST     => "F",
                   DEPHOMES_FOUND   => 0,
                   REFHOMES_FOUND   => 0,
                   NODES_FOUND      => 0,
                   HOMES            => {},
                   COMPOSITE_HOMES  => {}
               };

  return $parseObj;
}

#***** collect*********************************************************
# given the location of the inventory.xml file, it collects homes present in that inventory.xml .
sub collect
{
  my $self = shift;
  my $invXMLFile = $self->{XML_FILE}; 
  my $warnings = $self->{WARNINGS};

  emdcommon_ocm::EMD_PERL_INFO("Reading $invXMLFile file....");

  my $failed = OH_Utilities::checkPermissions($invXMLFile);

  if ($failed)
  {
    $warnings->addWarning(OH_Error::getErrorMessage($failed,$invXMLFile));
    $self->{STATUS} = "FAILED";
    my $error = new Error();
    $error->setError(CODE => $failed, ARG => $invXMLFile);
    $self->{ERROR} = $error;

    return;
  }

  my $parseObj = newParseObject($self->{INV_LOC});

  my $p2 = new XML::Parser(ErrorContext => 2, ref_obj=>$parseObj, warnings=>$warnings);

  emdcommon_ocm::EMD_PERL_INFO("Starting parsing of $invXMLFile");

  $p2->setHandlers(Start => \&startTagHandler,
                   End => \&endTagHandler );

  eval {$p2->parsefile($invXMLFile)};

  if ($@)
  {
    emdcommon_ocm::EMD_PERL_INFO("Failed to parse using default encoding:$@");
    foreach $_ (@encTable)
    {
      $p2->{ref_obj} = newParseObject($self->{INV_LOC});

      emdcommon_ocm::EMD_PERL_INFO("Trying with $_ eincoding..");
      eval '$p2->parsefile($invXMLFile, ProtocolEncoding => $_)';

      if ($@)
      {
        emdcommon_ocm::EMD_PERL_INFO("Failed again :$@");
      }
      else {
        last;
      }
    }
  }
  if ($@)
  {
    my $msg = "Unknown encoding. Could not parse $invXMLFile";
    $warnings->addWarning($msg);
    emdcommon_ocm::EMD_PERL_ERROR($msg);

    $self->{STATUS} = "FAILED";
    my $error = new Error();
    $error->setError(CODE => OH_Error::UNKNOWN_ENCODING_XML, ARG => $invXMLFile);
    $self->{ERROR} = $error;

    return; 
  }

  emdcommon_ocm::EMD_PERL_INFO("Finished parsing $invXMLFile ...");

  # Store collected data in self object and mark the status as collected
  $self->{STATUS} = "COLLECTED";
  $self->{HOMES} = $parseObj->{HOMES};
  $self->{COMPOSITE_HOMES} = $parseObj->{COMPOSITE_HOMES};
}

sub startTagHandler
{
  my $p = shift;
  my $el = shift;
  my $refObj = $p->{ref_obj};
  my $warnings = $p->{warnings};

  if ($el eq "HOME_LIST")
  {
    $refObj->{IN_HOME_LIST} = "T";
  }
  elsif($el eq "COMPOSITEHOME_LIST")
  {
    $refObj->{IN_COMPHOME_LIST} = "T";
  } 
  elsif ($el eq "HOME")
  {
    if($refObj->{IN_HOME_LIST} eq "T")
    {
      my %atts = @_;
      if( (defined($atts{REMOVED}) && ( $atts{REMOVED} eq "T")) 
       || (defined($atts{PLUGIN}) &&  ( $atts{PLUGIN} eq "T")) )
      {
        #do nothing
      }
      else
      {
        # if REMOVED is "F" or it doesn't exist proceed with collection of the home;
        if(exists ($atts{LOC}))
	{
	  my $loc = $atts{LOC};
	  $loc = removeTrailingSlash($loc);
	  ## -- check if home with this location already exists?
	  if(exists $refObj->{HOMES}->{$loc})
	  {
  	    # that means this is a duplicate entry
            emdcommon_ocm::EMD_PERL_WARN("DUPLICATE entry found for home with location $atts{LOC}");
#	   $refObj->{HOMES} = $homes{$atts{LOC}};#overwrite
            next;
	  }
	  else
	  {
	    ## -- create an entry for this home in homes hash
	    $refObj->{HOMES}->{$loc} = $refObj->{CURR_HOME} = {};
	  }
          $refObj->{CURR_HOME}->{_HOME_LOC}  = $loc;
	  $refObj->{CURR_HOME}->{_NAME} = $atts{NAME};
	  $refObj->{CURR_HOME}->{_TYPE} = $atts{TYPE};
	  $refObj->{CURR_HOME}->{_IDX}  = $atts{IDX};
          $refObj->{CURR_HOME}->{_CRS}  = (defined($atts{CRS}) && ($atts{CRS} eq "T"))? 'Y':'N';
          $refObj->{CURR_HOME}->{_INVENTORY}  = $refObj->{INV_LOC};
        }
      }
    }
  }
  elsif($el eq "COMPOSITEHOME")
  {
    if ( $refObj->{IN_COMPHOME_LIST} eq "T")
    {
      ## -- get the home list from the compsoite home also
      my %atts = @_;
      if (exists ($atts{LOC}))
      {
        my $compHomeLoc = removeTrailingSlash($atts{LOC});
        my $compositeHome = $refObj->{COMPOSITE_HOMES}->{$compHomeLoc};
        if (!defined($compositeHome))
        {
          $refObj->{COMPOSITE_HOMES}->{$compHomeLoc} = new OH_InventoryOrCompositeHome(
                                                                          LOC => $compHomeLoc,
                                                                          WARNINGS => $warnings); 
        }
      }
    }
  }
  elsif($el eq "DEPHOMELIST")
  {
    $refObj->{IN_DEPHOME_LIST} = "T";
    $refObj->{DEPHOMES_FOUND} = 0;
  }
  elsif($el eq "REFHOMELIST")
  {
    $refObj->{IN_REFHOME_LIST} = "T";
    $refObj->{REFHOMES_FOUND} = 0;
  }
  elsif($el eq "DEPHOME")
  {
    if($refObj->{IN_DEPHOME_LIST} eq "T")
    {
      my %atts = @_;
      $refObj->{CURR_HOME}->{DEP_ON_HOMES}->{$refObj->{DEPHOMES_FOUND}++}  =  removeTrailingSlash($atts{LOC});
    }
  }
  elsif($el eq "REFHOME")
  {
    if($refObj->{IN_REFHOME_LIST} eq "T")
    {
      my %atts = @_;
      $refObj->{CURR_HOME}->{REF_HOMES}->{$refObj->{REFHOMES_FOUND}++}  =  removeTrailingSlash($atts{LOC});
    }
  }
  elsif($el eq "NODE_LIST")
  {
    $refObj->{IN_NODE_LIST} = "T";
    $refObj->{NODES_FOUND} = 0;
  }
  elsif( $el eq "NODE")
  {
    my %atts = @_;
    if(exists($atts{NAME}) and($refObj->{IN_NODE_LIST} eq "T"))
    {
      $refObj->{CURR_HOME}->{NODE_LIST}->{$refObj->{NODES_FOUND}++} = $atts{NAME};
    } 
  }
}

sub endTagHandler
{
  my $p = shift;
  my $el = shift;
  my $refObj = $p->{ref_obj};

  if( $el eq "HOME_LIST")
  {
    $refObj->{IN_HOME_LIST}= "F";
  }
  elsif( $el eq "COMPOSITEHOME_LIST")
  {
    $refObj->{IN_COMPHOME_LIST} = "F"; 
  }
  elsif( $el eq "DEPHOMELIST")
  {
    $refObj->{IN_DEPHOME_LIST} = "F";
  }
  elsif( $el eq "NODE_LIST")
  {
    $refObj->{IN_NODE_LIST} = "F";
  }
}

1;
