#
# $Header: emagent/sysman/admin/scripts/OH_Utilities.pm /main/9 2011/12/27 00:12:29 hmodawel Exp $
#
# OH_Utilities.pm
#
# Copyright (c) 2010, 2011, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      OH_Utilities.pm - <Package to collect all the metrics related to OUI homes>
#
#    DESCRIPTION
#      <short description of component this file declares/defines>
#
#    NOTES
#        <>
#
#    MODIFIED   (MM/DD/YY)
#    hmodawel    12/23/11 - 12c Agent Compatible
#    hmodawel    11/12/11 - move printMetricResult and printError
#    ravpatel    10/19/11 - move methods from discovery
#    ravpatel    06/02/11 - Moving read_oraInst_loc here
#    ravpatel    01/21/11 - CCR Convergence
#    schaluva    07/23/10 - Adding the method getUserInfo
#    irraju      06/17/10 - adding version compare
#    irraju      05/17/10 - Package to define commonly useful methods.
#    irraju      05/17/10 - Creation
#
# This perl file contains various utility subroutines necessary for inventory collectioni
package OH_Utilities;

# USE DIRECTIVES
use strict;
use Fcntl qw(:DEFAULT :flock);

use OH_Error;
use emdcommon_ocm;

use Exporter;
use base 'Exporter';
our @EXPORT = qw(
    &removeTrailingSlash
    %months
    @encTable
    &verify_Entity_Not_in_List
    &isValidMWHome
    &isValidOUIHome
    &isValidOUICompositeHome
    &getOUIHomesInMWHome
    &read_oraInst_loc
    &checkPermissions
    &convertArrayToHash
    &appendHash
    &getUserInfo
    );

## VARIABLES
  # -- for date conversion
  our  %months = ( Jan => '01',
                 Feb => '02',
                 Mar => '03',
                 Apr => '04',
                 May => '05',
                 Jun => '06',
                 Jul => '07',
                 Aug => '08',
                 Sep => '09',
                 Oct => '10',
                 Nov => '11',
                 Dec => '12',
             );

  # -- encodings
  our @encTable = (
    'US-ASCII',
    'iso-8859-1',        # [west european, a.k.a. Latin1]
    'windows-1252',      # [win Latin1 (8859-1 superset)]
    'iso-8859-2',        # [east european]
    'windows-1250',      # [win Latin2 (8859-2 superset)]
    'iso-8859-3',        # [south european]
    'iso-8859-4',        # [north european]
    'iso-8859-5',        # [cyrillic]
    'iso-8859-7',        # [greek]
    'iso-8859-8',        # [hebrew]
    'iso-8859-9',        # [turkish]
    'UTF-8',
    'UTF-16',
    'big5',              # [traditional chinese]
    'euc-kr',            # [extended unix code for korean]
    'x-euc-jp-jisx0221', # [extended unix code for japanese, JIS X0221]
    'x-euc-jp-unicode',  # [extended unix code for japanese, unicode]
    'x-sjis-cp932',      # [shift-jis win codepage 932]
    'x-sjis-jdk117',     # [shift-jis JDK117]
    'x-sjis-jisx0221',   # [shift-jis JIS X0221]
    'x-sjis-unicode',    # [shift-jis unicode]
  );




## SUBROUTINES


#*****removeTrailingSlash()****************************************************
# This subroutine removes forward or backward trailing slashes in the given path 
#
#
sub removeTrailingSlash
{
  my $path = shift;
  if(defined($path))
  {
   chomp($path); #if new line at the end
   if(($path=~/(.*?)(\/+)*$/) || ($path=~/(.*?)(\\+)$/))
   {
    return $1;
   }
  }
  return $path;
}

#**************** SUB verify_Entity_Not_in_List()***********************************
# to verify that an entity is not a member of given list
sub verify_Entity_Not_in_List
{
    my($Entity, $listRef) = @_;
    if ((defined $Entity) && (defined $listRef))
    {
      foreach my $Item (@$listRef)
      {
       if($Item eq $Entity)
        {
            return 0;
        }
      }
    }
    return 1;
}

#**************** SUB isValidMWHome()***************************************
# to check whether a given path is valid middleware home location
sub isValidMWHome
{
    my $MWHome = shift;
    my $registry_xml = File::Spec->catfile($MWHome, "registry.xml");
    my $registry_dat = File::Spec->catfile($MWHome, "registry.dat");

    return ((-e $registry_xml) || (-e $registry_dat));
}

#**************** SUB isValidOUIHome()***************************************
# to check whether a given path is valid OUI Home
sub isValidOUIHome
{
    my $homeLoc = shift;
    my $compXml = File::Spec->catfile($homeLoc, "inventory", "ContentsXML", "comps.xml");

    return (-e $compXml);
}

#**************** SUB isValidOUICompositeHome() ***************************************
# to check whether a given path is valid OUI Composite Home
sub isValidOUICompositeHome
{
    my $homeLoc = shift;
    my $invXml = File::Spec->catfile($homeLoc, "inventory", "ContentsXML", "inventory.xml");
    return (-e $invXml);
}

#**************** SUB getOUIHomesInMWHome ***************************************
# returns list of OUI Homes that are immediate subdirectory of given Middleware Home.
sub getOUIHomesInMWHome
{
  my $mwHome = shift;
  my @homeList;

  if (opendir(MW_HOME, $mwHome))
  {
    while (my $dirEntry = readdir(MW_HOME))
    {
      # skip "." and ".." directories
      next if ($dirEntry eq "." || $dirEntry eq "..");

      my $child = File::Spec->catfile($mwHome, $dirEntry);

      # check in directories only
      next unless (-d $child);

      # look for the comps.xml file; add to OH list if it exists
      my $compsXml = File::Spec->catfile($child,'inventory','ContentsXML','comps.xml');
      push(@homeList, $child) if (-f $compsXml);
    }
    closedir(MW_HOME);
  }
  return @homeList;
}

#**************** SUB read_oraInst_loc ***************************************
# reads a given oraInst.loc file and returns inventory entry in the file. 

sub read_oraInst_loc
{
    my $instLoc = shift;
    my $warnings = shift;

    emdcommon_ocm::EMD_PERL_INFO("Trying to open file $instLoc");

    if(open(ORAINST, "<$instLoc"))
    {
      # Try to aquire a shared read lock
      if(!(flock(ORAINST, LOCK_SH)))
      {
        my $msg = "Couldn't acquire read-lock on $instLoc, Please check if the file is in consistent state.";
        emdcommon_ocm::EMD_PERL_ERROR($msg);
        $warnings->addWarning($msg);
      }

      my @lns = <ORAINST>;
      close(ORAINST);
      foreach my $ln (@lns)
      {
        $ln =~ s/\r$//;
        $ln =~ s/\n$//;
        if ($ln =~ /^\s*inventory_loc\s*=\s*/i)
        {
          my ($label, $inv) = split('=', $ln);
          $inv =~ s/^\s*//g;
          $inv =~ s/\s*$//g;
          emdcommon_ocm::EMD_PERL_INFO("Found inventory $inv in file $instLoc");
          return OH_Utilities::removeTrailingSlash($inv);
        }
      }

      emdcommon_ocm::EMD_PERL_WARN("No inventory entry found in file $instLoc");
      return undef;
    }
    else
    {
      my $msg = OH_Error::getErrorMessage(OH_Utilities::checkPermissions($instLoc), $instLoc);
      emdcommon_ocm::EMD_PERL_WARN($msg);

      $warnings->addWarning($msg) if (-f $instLoc); #Display only permission errors in the UI
      return undef;
    }
}

#**************** SUB checkPermissions()***************************************
# check the read permissions
sub checkPermissions
{
  my $file = shift;
  my $msg;
  if(-e $file)
  {
    if (-r $file)
    {
      emdcommon_ocm::EMD_PERL_INFO("$file exists and is readable");
      return OH_Error::SUCCESS;
    }
    else
    {
      $msg = $OH_Error::errorMessage{OH_Error::FILE_PERM_DENIED} . "$file";
      emdcommon_ocm::EMD_PERL_ERROR($msg);
      return OH_Error::FILE_PERM_DENIED;
    }
  }
  else
  {
    $msg = $OH_Error::errorMessage{OH_Error::FILE_NOT_FOUND} . "$file";
    emdcommon_ocm::EMD_PERL_ERROR($msg);
    return OH_Error::FILE_NOT_FOUND;
  }
}

#**********convertArrayToHash*******************************************
#
#
sub convertArrayToHash()
{
 my @array = @_;
 my %hash;
 my $index = 0;
 foreach my $element (@array)
  {
   $hash{$index++}= $element;
  }
 return \%hash;
}

#*****************appendHash**********************************************
# This Method joins two hashes (old and new) into one (old)
# Duplicate keys are omitted.
# NOTE:- only exact matching keys are considered same. Its caller's
# responsibily to perform any formatting on keys if required

sub appendHash 
{
  my $oldHashRef = shift;
  my $newHashRef = shift;

  if((defined $oldHashRef) && (defined $newHashRef))
  {
    foreach my $key (keys(%$newHashRef))
    {
      my $val = $oldHashRef->{$key};
      if (!defined($val))
      {
        $oldHashRef->{$key} = $newHashRef->{$key};
      }
    }
  }
}

#***************versionCompare********************************************
#
#
sub versionCompare
{
 # returns 1 if first version is later than secomd
 # returns 0 if frist version is earlier than second
 my $first = shift;
 my $second = shift;
 my @firstArray  = split(/\./,$first);
 my @secondArray = split(/\./,$second);
 for my $i (0 .. $#firstArray)
 {
  my $a1 = 0+$firstArray[$i]+0;
  my $a2 = 0+$secondArray[$i]+0;
  if($a1> $a2)
  {
   return 1;
  }
  elsif($a1 < $a2)
  {
   return 0;
  }
  #equal then continue with the next digit
 }
return 0;
}

#*********************getUserInfo()************************************
#Returns a hash of UserInformation usernames and groups
# INPUT ARGS
#       1.OUI Object Reference
#
# Returns
#       1.has reference. Each entrly holds the UserInformation
#*********************************************************************
  
sub getUserInfo
{ 
  my $HomeLocation = shift;
  my $warnings = shift;

  my %groupInfo;#A hash containing group information of the user

  #Declaring the Variables
  my ($dev,$ino,$mode,$nlink,$uid,$gid,$ownerOfFile,$pass,$princGid,$ownerUid,$OHGrpId);
  my ($name,$members,$passwd,$princGrpName,$groupList,$groupIdList);

  #Checking for non windows operating system
  my $os = $^O;
  if($os !~ /win/i)
  {
    #Getting the owner information
    if(($dev,$ino,$mode,$nlink,$uid,$gid)=stat($HomeLocation))
    {
      #Getting the OwnerName  by using user id
      if(($ownerOfFile,$pass,$ownerUid,$princGid)=getpwuid ($uid))
      {
        #Getting the priciple group name of the user and the group name of the OracleHome
        my $OHGrpName;
        if( ($OHGrpName = getgrgid($gid)) && ($princGrpName = getgrgid($princGid)) )
        { 
          $groupInfo{$princGid}=$princGrpName;
          $OHGrpId = $gid;

          # Searching the group entris one by one and finding the groups the user belongs to
          while(($name,$passwd,$gid,$members)=getgrent())
          {
            if($members =~ /\s*$ownerOfFile\s*/)
            {
              if(!exists($groupInfo{$gid}))
              {
                $groupInfo{$gid}=$name;
              }
            }
          }#end ofwhile loop

          my $userInfo = { OH_OWNER           => $ownerOfFile,
                           OH_GROUP           => $OHGrpName,
                           OH_OWNER_GROUPS    => join (";",values  %groupInfo),
                           OH_OWNER_ID        => $ownerUid,
                           OH_GROUP_ID        => $OHGrpId,
                           OH_OWNER_GROUPS_ID => join (";",keys %groupInfo)
                        };
          return $userInfo;

        }#End of Checking the getgrgid function  
        else
        {
          my $msg = OH_Error::getErrorMessage(OH_Error::RETURNED_NULL, "getgrgid() Function ");
          $warnings->addWarning($msg);
          emdcommon_ocm::EMD_PERL_WARN($msg);
        }
      }#End of getpwuid function checking if.
      else
      {
        my $msg= OH_Error::getErrorMessage(OH_Error::RETURNED_NULL, "getpwid() Function ");
        $warnings->addWarning($msg);
        emdcommon_ocm::EMD_PERL_WARN($msg);
      }
    }#end of stat checking if
    else
    {
      my $msg= OH_Error::getErrorMessage(OH_Error::RETURNED_NULL, "stat() Function ");
      $warnings->addWarning($msg);
      emdcommon_ocm::EMD_PERL_WARN($msg);
    }
  }#End of Checking Operating System
  return undef;
}#End of Subroutine getUserInfo()

1;
