#
# $Header: emagent/sysman/admin/scripts/OH_WLS.pm /main/11 2012/06/26 02:29:48 irraju Exp $
#
# OH_WLS.pm
#
# Copyright (c) 2010, 2012, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      OH_WLS.pm - <Package to collect all the metrics related to OUI homes>
#
#    DESCRIPTION
#      <short description of component this file declares/defines>
#
#    NOTES
#        <>
#
#    MODIFIED   (MM/DD/YY)
#    hmodawel    05/12/11 - add initVariables method
#    hmodawel    05/04/11 - fix RW status for mounts
#    hmodawel    04/21/11 - add RW status
#    hmodawel    04/14/11 - use new Date function
#    chkaushi    03/11/11 - Changing HOME_BASE to MW_HOME
#    ravpatel    01/21/11 - CCR Convergence
#    sanjkuma    12/09/10 - Update to remove the usage of CPAN modules
#                           DateFormat, DateParse and TimeZone
#    schaluva    07/23/10 - Adding metrics to get UserInformation
#    irraju      05/17/10 - Package to define all the methods to extract 
#                             information about WLS homes.
#    irraju      05/17/10 - Creation
# 

## USE DECLARATIONS
package OH_WLS;
use strict;
use warnings;
use Carp;
use XML::Parser;
use OH_Error;
use OH_Utilities;
use fields qw(MW_HOME HOME_LOC ERROR COLLECTED);
use File::Basename;
use OH_Date;

use emdcommon_ocm;

## VARIABLES
use vars qw($warnings $error);# $mwHome $homeLoc $inventoryCollected);
my %products;
my %TLComps;
my @compsList;
my %comps;
my @wlsHomes;
my %instDir2Prod;   # Install dir to product map
#my $inventoryCollected = "F";
## CONSTANTS
use constant REG_DAT_FILE  => "registry.dat";
use constant REG_XML_FILE  => "registry.xml";
use constant PATCH_REG_XML => File::Spec->catfile("registry","patch-registry.xml");
use constant COLLECTED     => 'C';
use constant ERROR         => 'E';

## GLOBAL VARIABLES
my $inProductAliases;
my $inProductAlias;
my $productsFound;
my $inProduct;
my $inComponent;
my $componentsFound;
my $inReqGroup;
my $compDepFound;
my $depFound;
my $currProduct;
my $currComp;

## -- initialize the hashes and variables
sub initVariables() {

    $inProductAliases    =   "F";
    $inProductAlias      =   "F";
    $productsFound       =   0;
    $inProduct           =   "F";
    $inComponent         =   "F";
    $componentsFound     =   0;
    $inReqGroup          =   "F";
    $compDepFound        =   0;
    $depFound            =   0;
    %products            =   ();
    %TLComps             =   ();
    @compsList           =   ();
    %comps               =   ();
    @wlsHomes            =   ();
    %instDir2Prod        =   ();   # Install dir to product map

}

################################################################################
#                 EXPOSED SUBROUTINES                                          #
################################################################################

#*********SUB new **************************************************************
# Collects the specified WLS home
# INPUT ARGS
#       1. Obsolute Location of the WLS home
#       2. middleware home
#      
# RETURNS
#       1. Reference to the WLS object created
#
#******************************************************************************
sub new
{
    my ($class, %args) = @_;
    my $self = fields::new(ref($class)||$class);
    my $mwHome  = $args{mw_home};
    my $homeLoc = $args{home};
    $error   = $args{error_obj};

    if (!$mwHome)
    {
      $error->setError( CODE  => OH_Error::MAND_ARG_MISSING,
                        ARG => "OH_WLS");
      return undef;
    }
    # We can proceed with the collection even when the home
    # doesn't exist as the registry.dat/xml reside in middleware home
    # However, it does not make sense to collect inventory data for 
    # the Weblogic home which does not exist.
    
    if (my $retVal = OH_Utilities::checkPermissions($homeLoc))
    {
      # something wrong with the permissions
      $error->setError( CODE  => $retVal,
                        ARG   => $homeLoc);
      return undef; #Failed

    }
     $self->{HOME_LOC} = $homeLoc;
  
    # -- check the inventory location now
    if(my $retVal = OH_Utilities::checkPermissions($mwHome))
    {
      # something wrong with the permissions
      $error->setError( CODE  => $retVal,
                        ARG   => $mwHome);
      return undef; #Failed
    }
    # -- set the inventory location to the object
    $self->{MW_HOME} = $mwHome;
    $self->{ERROR}   = $error;
    $self->{COLLECTED} = "F";
    return $self; #success
  
}



#********* SUB getHomeInfo******************************************************
# Returns the properties of WLS home
# INPUT ARGS
#       1. WLS object reference
# RETURNS
#       1. Hash reference of WLS home properties
#
#*******************************************************************************
sub getHomeInfo()
{
  my ($self,%args) = (shift,@_);
  my $metricInfo   = $args{metric_info};
  my $warnings     = $args{warnings};
  my $error        = $self->{ERROR};
  my $isClonable = 'N';
  my $mwhDir = OH_Utilities::removeTrailingSlash($self->{MW_HOME});
  my $homeParentDir = dirname(OH_Utilities::removeTrailingSlash($self->{HOME_LOC}));

  my $userInfo = OH_Utilities::getUserInfo($self->{HOME_LOC}, $warnings);

  if($mwhDir eq $homeParentDir)
  {
    emdcommon_ocm::EMD_PERL_DEBUG("HOME $self->{HOME_LOC} is immediately in the MW HOME $mwhDir so this home is clonable");
    $isClonable ='Y';
  }
  
   my $rwstatus =  'NRNW';
   # Collect Home RW status
   if (-e  $self->{HOME_LOC})
   {
      if (-r $self->{HOME_LOC})
      {
        if (-w  $self->{HOME_LOC})
        {
          # My be its mounted, try touching a file
          my $checkFile = File::Spec->catfile($self->{HOME_LOC}, "checkforRW");
          if ( open(CHKFORRW, ">$checkFile"))
          {
            $rwstatus = 'RW';
            close(CHKFORRW);
            unlink $checkFile;
          }
          else {
            $rwstatus = 'RO';
          }
        }
        else {
          $rwstatus = 'RO';
        }
      } else {
        if (-w  $self->{HOME_LOC})
        {
          $rwstatus = 'WO';
        }
        else {
          $rwstatus = 'NRNW';
        }
      }
   }
   else # home does not exist, this will never be executed
   {
      $rwstatus = 'NRNW';
   }

   $metricInfo->{STATUS} = COLLECTED;
   $metricInfo->{ROWS}->{0}= bless({
                  _HOME_LOC  => $self->{HOME_LOC},
                  _TYPE      => "W",
                  _INVENTORY => $mwhDir,
                  _CLONABLE  => $isClonable,
                  _ARU_ID    => "-1",
                  _OUI_PLATFORM_ID => "-1",
                  _CRS       => 'N',
                  _RW_STATUS => $rwstatus,
                  _OH_OWNER_ID => ($userInfo && $userInfo->{OH_OWNER_ID})?$userInfo->{OH_OWNER_ID}:-1,
		  _OH_OWNER  => ($userInfo && $userInfo->{OH_OWNER})?$userInfo->{OH_OWNER}:-1,
                  _OH_GROUP_ID => ($userInfo && $userInfo->{OH_GROUP_ID})?$userInfo->{OH_GROUP_ID}:-1,
		  _OH_GROUP  => ($userInfo && $userInfo->{OH_GROUP})?$userInfo->{OH_GROUP}:-1,
                  _OH_OWNER_GROUPS_ID => ($userInfo && $userInfo->{OH_OWNER_GROUPS_ID})?$userInfo->{OH_OWNER_GROUPS_ID}:-1, 
		  _OH_OWNER_GROUPS => ($userInfo && $userInfo->{OH_OWNER_GROUPS})?$userInfo->{OH_OWNER_GROUPS}:-1,
               },"HomeInfoClass");

    return OH_Error::SUCCESS
}


#**********SUB getPatches******************************************************
# Returns all the patches installed in the current WLS home
# INPUT ARGS
#       1. WLS object ref
# RETURNS
#       1. hash of hashes, each hash entry holding info about a patch
#
#*****************************************************************************
sub getPatches()
{
  my ($self,%args) = (shift,@_);
  my $metricInfo   = $args{metric_info};
  my $warnings     = $args{warnings};
  my $error        = $self->{ERROR};

  if(!($self->{COLLECTED} eq "T"))
  {
    # Inventory collection is yet to happen
    emdcommon_ocm::EMD_PERL_DEBUG("Started collection of WLS inventory");
    if(my $failed= $self->collectInfo(WARNINGS => $warnings, HOME_LOC => $self->{HOME_LOC}))
    {
       #prasing  failed
       return $failed;
    }
    emdcommon_ocm::EMD_PERL_DEBUG("Finished collection of WLS inventory");
  }
  
  ## read the patch directories and update the patches
  if(opendir(MW_HOME,$self->{MW_HOME}))
  {
    # Find all the directories which start with patch
    my @patchDirs = grep {/^patch_.*/} readdir(MW_HOME);
    closedir(MW_HOME);
    foreach my $patchDir (@patchDirs)
    {
      my $patchRegistryFile = File::Spec->catfile($self->{MW_HOME},$patchDir,PATCH_REG_XML);
      if( my $retVal = OH_Utilities::checkPermissions($patchRegistryFile))
      {
       #something wrong with the permissions
       my $msg = "";
       $warnings->addWarning($msg);
      }
      else
      {
        #patch-registry fiel exists and readable
        emdcommon_ocm::EMD_PERL_DEBUG("Started parsing of $patchRegistryFile");
        my $retVal = parsePatchRegistryXml($patchRegistryFile);
        if($retVal)
        {
          my $msg = "Failed to parse patch registry file $patchRegistryFile";
          emdcommon_ocm::EMD_PERL_WARN($msg);
          $warnings->addWarning($msg);
        }
        emdcommon_ocm::EMD_PERL_DEBUG("Finished parsing of $patchRegistryFile");
      }#if
    }#foreach  
  }#if
  else
  {
    my $msg = "Failed to scan patch directories in MWHOME $self->{MW_HOME}";
    emdcommon_ocm::EMD_PERL_WARN($msg);
    $warnings->addWarning($msg);
  }
  
  ## -- put the data in PatchClass format

  my $numRows = 0;
  foreach my $compKey (@compsList)
  {
    my $prod = $products{$compKey};
    if(exists ($prod->{PATCH_LIST}))
    {
      foreach my $id (keys %{$prod->{PATCH_LIST}})
      {
        $metricInfo->{ROWS}->{$numRows++}= bless ({
                       _PATCH_ID  =>  $id,
                       _UPI       =>  "N/A",
                       _LANG      =>  "en",
                       _ROLLBACK  =>  'Y',   # Weblogic Patches are always rollbackable 
                       _IS_PSU    =>  'N',
                       _PROFILE   =>  $prod->{PATCH_LIST}->{$id}->{PROFILE},
                       _TIMESTAMP =>  OH_Date::conDateToYYYMMDD($prod->{PATCH_LIST}->{$id}->{TIMESTAMP})},
                      "PatchClass");
      }
    }
  }
  $metricInfo->{STATUS} = COLLECTED;
  return OH_Error::SUCCESS;
}


#*********SUB getComponents****************************************************
# Returns all the components present in the cuarrent WLS home
# INPUT ARGS
#       1. WLS home Objecat reference
# RETURNS
#       1. hash of hashes, each hash entry consisting of one component info
#
#******************************************************************************
sub getComponents()
{
  my ($self,%args) = (shift,@_);
  my $metricInfo   = $args{metric_info};
  my $warnings     = $args{warnings};
  my $error        = $self->{ERROR};
  if(!($self->{COLLECTED} eq "T"))
  {
    # Inventory collection is yet to happen
    emdcommon_ocm::EMD_PERL_DEBUG("Started collection of WLS inventory");
    my $failed = $self->collectInfo(WARNINGS => $warnings,HOME_LOC=>$self->{HOME_LOC});
    if($failed)
    {
       #collection  failed
       return $failed;
    }
    emdcommon_ocm::EMD_PERL_DEBUG("Finished collection of WLS inventory");
  }
  # put the result in component format
  my $numRows = 0;
  my %duplicateCheckHash = ();
  foreach my $compKey (@compsList)
  {
    my $prod = $products{$compKey};
    # Get the versioned components first
    # component name and external name are same
    my $pn = $prod->{NAME};#product name
    my $pv = $prod->{VER};#product version
    if(!exists($duplicateCheckHash{$pn.$pv}))
    {
     $duplicateCheckHash{$pn.$pv} = 1;
     $metricInfo->{ROWS}->{$numRows++}= bless({
                  _NAME        => $pn,
                  _VER         => $pv,
                  _CURR_VER    => $prod->{VER},
                  _TIMESTAMP   => OH_Date::conDateToYYYMMDD($prod->{TIMESTAMP}), 
                  _IS_TOP_LEVEL=> ($prod->{IS_TOP_LEVEL} && ($prod->{IS_TOP_LEVEL} eq 'Y')) ?"Y":"N",
                  _EXT_NAME    => $prod->{NAME},
                  _INST_LOC    => $prod->{INST_LOC}
                     },"ComponentClass");
    }
    # get the non versioned components now
   foreach my $index (keys %{$prod->{DEP_ON_LIST}})
    {
      my $compRef = $prod->{DEP_ON_LIST}->{$index}->{COMP};
      my $dn = $compRef->{NAME};#depname
      my $dv = ($compRef->{VER})?($compRef->{VER}):("0.0.0.0");#depver
      if(!exists($duplicateCheckHash{$dn.$dv}))
      {
       $duplicateCheckHash{$dn.$dv} = 1;
       $metricInfo->{ROWS}->{$numRows++} = bless(
                   {
                    _NAME        => $dn,
                    _VER         => $dv,
                    _CURR_VER    => $dv,
                    _TIMESTAMP   => OH_Date::conDateToYYYMMDD($prod->{TIMESTAMP}),
                    _IS_TOP_LEVEL=> 'N',
                    _EXT_NAME    => $dn,
                    _INST_LOC    => $prod->{INST_LOC}
                   },"ComponentClass");
      }
     #in case of .dat file there can be componenent dependencies among these dependent
     # components also
     my $depComp = $prod->{DEP_ON_LIST}->{$index}->{COMP};
     foreach my $idx (keys %{$depComp->{DEP_ON_LIST}})
     {
      my $ddn = $depComp->{DEP_ON_LIST}->{$idx}->{NAME};
      my $ddv = ($depComp->{DEP_ON_LIST}->{$idx}->{VER})?($depComp->{DEP_ON_LIST}->{$idx}->{VER}):("0.0.0.0");
      if(!exists($duplicateCheckHash{$ddn.$ddv}))
      {
       $duplicateCheckHash{$ddn.$ddv} = 1;
       $metricInfo->{ROWS}->{$numRows++} = bless(
                   {
                    _NAME        => $ddn, 
                    _VER         => $ddv,
                    _CURR_VER    => $ddv,
                    _TIMESTAMP   => OH_Date::conDateToYYYMMDD($prod->{TIMESTAMP}),
                    _IS_TOP_LEVEL=> 'N',
                    _EXT_NAME    => $ddn, 
                    _INST_LOC    => $prod->{INST_LOC}
                   },"ComponentClass");
      }
     }
    }


  }
  $metricInfo->{STATUS}   = COLLECTED;
  return OH_Error::SUCCESS ;

}

#***** SUB getTLComponents ****************************************************
# Returns all  the products in the given registry.dat/xml. This another way of
# getting all the weblogic servers in the given middleware home
#
#******************************************************************************
sub getTopLevelComponents
{
  my ($self,%args) = (shift,@_);
  my $metricInfo   = $args{metric_info};
  my $warnings     = $args{warnings};
  my $error        = $self->{ERROR};
  if(!($self->{COLLECTED} eq "T"))
  {
    # Inventory collection is yet to happen
    emdcommon_ocm::EMD_PERL_DEBUG("Started collection of WLS inventory");
    my $failed = $self->collectInfo(WARNINGS => $warnings,HOME_LOC=>$self->{HOME_LOC});
    if($failed)
    {
       #collection  failed
       return $failed;
    }
    emdcommon_ocm::EMD_PERL_DEBUG("Finished collection of WLS inventory");
  }
  # put the result in component format
  my $numRows = 0;
  foreach my $idx (keys (%TLComps))
  {
    $metricInfo->{ROWS}->{$numRows++} = bless({ 
                  HOME_LOC  => $TLComps{$idx}->{HOME_LOC},
                  INVENTORY => $self->{MW_HOME},
                  NAME      => $TLComps{$idx}->{NAME}." ".$TLComps{$idx}->{VER}
                 },"HomeInfoClass");
  }
  $metricInfo->{STATUS} = COLLECTED;
  return OH_Error::SUCCESS;
}

#*******SUB getAllHomesInInventory*********************************************
# This sub routine retuns all the weblogic server homes listed in the registry.xml
# of the given middleware home
#
#******************************************************************************
sub getAllHomesInInventory
{
  # parse the registry.dat first if not already parsed
  emdcommon_ocm::EMD_PERL_DEBUG("Started collection of WLS inventory");
  my $self = shift;
  my %args = @_;
  my $mwHome = OH_Utilities::removeTrailingSlash($args{INVENTORY});
  my $result = $args{RESULT_OBJ};
  $error = $args{ERROR};
  my $warnings = $args{WARNINGS}; 
  if(my $retVal = OH_Utilities::checkPermissions($mwHome))
  {
    $error->setError(CODE => $retVal,
                     ARG  => $mwHome);
    return $retVal;
  }

  my $failed = $self->collectInfo(WARNINGS => $warnings, MW_HOME => $mwHome);
  if($failed)
  {
     #collection  failed
     return $failed;

  } 
  # put the result in component format
  my %returnHash;
  my $numRows = 0;
  foreach my $productRef (@wlsHomes)
  {
    if(($productRef->{INST_LOC}) && exists($result->{ROWS}->{$productRef->{INST_LOC}}))
    {
     $warnings->addWarning("Duplicate WLS home $productRef->{INST_LOC} found in MW home $mwHome ");
    }
    else
    {
     my $productName = $productRef->{NAME}.$productRef->{VER};
     $productName =~ s/ //g;
     $result->{ROWS}->{$productRef->{INST_LOC}} = { 
                  _HOME_LOC        => $productRef->{INST_LOC},
                  _MW_HOME         => $mwHome,
                  _TYPE            => "W",
                  _NAME            => $productName 
               };
    }
  }
  $result->{STATUS} = COLLECTED;
  return OH_Error::SUCCESS;
}



#*********SUB getDependees####*************************************************
#
#
#******************************************************************************
sub getComponentDependencies()
{
  my ($self,%args) = (shift,@_);
  my $metricInfo   = $args{metric_info};
  my $warnings     = $args{warnings};
  my $error        = $self->{ERROR};
  if(!($self->{COLLECTED} eq "T"))
  {
    # Inventory collection is yet to happen
    emdcommon_ocm::EMD_PERL_DEBUG("Started collection of WLS inventory");
    my $failed = $self->collectInfo(WARNINGS => $warnings,HOME_LOC=>$self->{HOME_LOC});
    if($failed)
    {
       #collection  failed
       return $failed
    }
    emdcommon_ocm::EMD_PERL_DEBUG("Finished collection of WLS inventory");
  }
  my $numRows = 0;
  foreach my $compKey (@compsList)
  {
    my $prod = $products{$compKey};
    foreach my $index (keys %{$prod->{DEP_ON_LIST}})
    {
      my $dn = $prod->{NAME};
      my $dv = $prod->{VER};
      my $rn = $prod->{DEP_ON_LIST}->{$index}->{COMP}->{NAME};
      my $rv = $prod->{DEP_ON_LIST}->{$index}->{COMP}->{VER};
      $metricInfo->{ROWS}->{$numRows++} = bless(
                   {_DEP_NAME=>$dn,
                    _DEP_VER=>defined($dv)?$dv:"0.0.0.0",
                    _REF_NAME=>$rn,
                    _REF_VER=>defined($rv)?$rv:"0.0.0.0"
                   },"CompDepClass");
     #in case of .dat file there can be componenent dependencies among these dependent 
     # components also
     my $depComp = $prod->{DEP_ON_LIST}->{$index}->{COMP};
     foreach my $idx (keys %{$depComp->{DEP_ON_LIST}})
     {
      $metricInfo->{ROWS}->{$numRows++} = bless(
                   {_DEP_NAME=>$rn,
                    _DEP_VER=>"0.0.0.0",
                    _REF_NAME=>$depComp->{DEP_ON_LIST}->{$idx}->{NAME},
                    _REF_VER=>"0.0.0.0"
                   },"CompDepClass");
     }
    }
  }
  $metricInfo->{STATUS}   = COLLECTED;
  return OH_Error::SUCCESS;
}


#******** SUB getLocale()******************************************************
# Returns the locale of the WLS installation
# INPUT ARGS
#       1. WLS Object ref
# RETURNS
#       1. String representing the LOCALE of the installation
#
#******************************************************************************
sub getLocale()
{
    # -- For 9.* get it from ${WLS_INSTALL_DIR}/common/lib/platform.properties
    # -- For 10.* get it from ${WL_HOME}/.product.properties
        # -- before and 10.3.1
        # -- 10.3.2
        # -- 10.3.3 onwards
}


###############################################################################
#                 INTERNAL SUBROUTINES                                        #
###############################################################################


#******* collectInfo **********************************************************
# this parses the registry.dat/xm and patch-registry files and returns the 
#information holding object
#
#******************************************************************************
sub collectInfo
{
    my $self = shift;
    initVariables(); #initialize all global variables
    my %args = @_;
    my $warnings = $args{WARNINGS};
    my ($mwHome,$homeLoc);
    if(defined($args{MW_HOME}))#incase of discovery we pass just MW_HOME
    {
      $mwHome   = $args{MW_HOME};
    }
    else #usual metric collection case
    {
      $mwHome = $self->{MW_HOME};
      $homeLoc = $self->{HOME_LOC};
    }

    my ($registryFile, $retVal, $msg);
    #In platforms that doesn't support fork, its difficult to parse encrupted dat file using a chile process to decrypt. It would have mattered if we were collecting WLS homes ourselves but from discovery perspective, we don't lose anything by reading from xml file than from dat file. On windows, parsing the XML files directly to avoid confusing error msgs in agent logs. BUG 13626628
    if ($^O =~ /win/i)
    { $retVal = 1;} #Go with XML parsing always
    else
    {
      $registryFile = File::Spec->catfile($mwHome,REG_DAT_FILE);
      $retVal = parseRegistryDat($registryFile);
    }
    # parser returns zero on success and non zero value on failure
    if($retVal)#failed to parse DAT files
    {
        $msg = "Failed to parse $registryFile.. Trying XML file now";
        emdcommon_ocm::EMD_PERL_WARN($msg);
        $registryFile = File::Spec->catfile($mwHome,REG_XML_FILE);
        initVariables();
        $retVal= parseRegistryXml($registryFile);
        if($retVal)
        {
          return $retVal;
        }
        emdcommon_ocm::EMD_PERL_DEBUG("Successfully parsed $registryFile");
    }
    # if we are here - that means parsing is successfull
    # Now process the products hash, replace install loc of TL component with the install component of WLS
    foreach my $prodKey (keys %products)
    {
      my $prodInfo = $products{$prodKey};
      if(defined($prodInfo->{IS_TOP_LEVEL}) && ($prodInfo->{IS_TOP_LEVEL} eq 'Y'))
      {

        #check if any weblogic servers are there and replace the install location
        # of the TL product with install location of the WLServer
        foreach my $index (keys %{$prodInfo->{DEP_ON_LIST}})
        {
         # replace time stamps of dependent components with timestamps of the TL component
         my $compKey = "$prodInfo->{DEP_ON_LIST}->{$index}->{COMP}->{NAME}|$prodInfo->{DEP_ON_LIST}->{$index}->{COMP}->{VER}";
         my $compRef = $products{$compKey};
         $compRef->{TIMESTAMP}= $prodInfo->{TIMESTAMP};
         
         if($prodInfo->{DEP_ON_LIST}->{$index}->{COMP}->{NAME} eq "WebLogic Server")
         {
           my $webLogicRef = $products{"WebLogic Server|$prodInfo->{DEP_ON_LIST}->{$index}->{COMP}->{VER}"};
           if(not defined ($webLogicRef))
           {
             my  $msg = "WebLogic Server|$prodInfo->{DEP_ON_LIST}->{$index}->{VER} which is referenced by $prodInfo->{NAME} does not exist";
              emdcommon_ocm::EMD_PERL_WARN($msg);
              $warnings->addWarning($msg);
           }
           else
           {
             push(@wlsHomes, $webLogicRef);
             # replace the installed location
             if(defined ($webLogicRef->{INST_LOC}))
             {
               $prodInfo->{INST_LOC} = removeTrailingSlash($webLogicRef->{INST_LOC});
             }
             else
             {
              # 9.* case weblogi servers install location should be put with top level component's
              $webLogicRef->{INST_LOC} = $prodInfo->{INST_LOC};
             }
             #else keep what TL product has
           }
         }#end of if 
        }#end of foreach
      }#end of if
    }# end of foreach

    #now get the components in the current install location
   if(defined ($homeLoc))
   { 
    my $TLCompsFound = 0; 
    foreach my $prodKey (keys %products)
    {
      my $prodInfo = $products{$prodKey};
      if(defined($prodInfo->{IS_TOP_LEVEL}) && ($prodInfo->{IS_TOP_LEVEL} eq 'Y'))
      {
       if($prodInfo->{INST_LOC} eq $homeLoc)
       {
        ## get the components in this
        my $compKey = join("|",$prodInfo->{NAME},$prodInfo->{VER});
        push( @compsList, $compKey );
        %comps = ();
        emdcommon_ocm::EMD_PERL_DEBUG("Found TL COMP $prodInfo->{NAME}|$prodInfo->{VER}");
        my $depList = $prodInfo->{DEP_ON_LIST};
        foreach my $index (keys %{$depList})
        {
          my $comp = $depList->{$index}->{COMP};
          my $compKey = join("|",$comp->{NAME},$comp->{VER});
          
          if ( ! exists($comps{$compKey}) ) #TO AVOID DUPLICATES
          {
            push( @compsList, $compKey );
            $comps{$compKey} = 1;
          }
          else
         {
           my $msg = "Duplicate component found $compKey in  $prodKey";
           emdcommon_ocm::EMD_PERL_WARN($msg);
           $warnings->addWarning($msg);
 
         }
          emdcommon_ocm::EMD_PERL_DEBUG("Found COMP $comp->{NAME} | $comp->{VER}");
        }
       }
        # put this in the toplevel component hash
        $TLComps{$TLCompsFound}->{NAME} = $prodInfo->{NAME};
        $TLComps{$TLCompsFound}->{VER} = $prodInfo->{VER};
        $TLComps{$TLCompsFound}->{HOME_LOC} = $prodInfo->{INST_LOC};
        $TLCompsFound++;
      }
    }
   }
    if(ref($self) eq "OH_WLS")
    {
     $self->{COLLECTED} = "T";
    }
    return OH_Error::SUCCESS;

    #parsing successful
}

{ #START OF PATCH REGISTRY BLOCKk
my $inProduct = "F";
my $inName    = "F";
my $inVersion = "F";
my $inPatch   = "F";
my $inId     = "F";
my $inTimeStamp = "F";
my $inProfile  = "F";
my $prodRef;
my $currName;
my $currVer;
my $currId;


sub parsePatchRegistryXml
{
  my $file = shift;
  emdcommon_ocm::EMD_PERL_DEBUG("$file EXISTS and READABLE .. proceeding");
  ## -- parse the file
  if (open(FILE, $file))
   {
     close FILE;
     my $p2 = new XML::Parser(ErrorContext => 2);
     $p2->setHandlers(Start =>  \&startPatchReg,
                      End   =>  \&endPatchReg,
                      Char  =>  \&charPatchReg );

     eval {$p2->parsefile($file)};
     if($@)
      {
        my $msg = "Failed to parse $file:$@";
        emdcommon_ocm::EMD_PERL_WARN($msg);
        $warnings->addWarning($msg);
      }
   }
  else
   {
     my $msg = "Failed to open $file";
     emdcommon_ocm::EMD_PERL_WARN($msg);
     warnings->addWarning($msg);
   }
  return OH_Error::SUCCESS;
}



  sub startPatchReg
  {
    my $p = shift;
    my $el = shift;
    #print "__ $el __ \n";
    if ($el eq "product")
    {
      $inProduct = "T";

    }
    elsif($el eq "name")
    {
      $inName = "T";

    }
    elsif($el eq "version")
    {
      $inVersion = "T";
    }
    elsif($el eq "patchInstallEntry")
    {
      $inPatch = "T";
    }
    elsif($el eq "id")
    {
     $inId = "T";
    }
    elsif($el eq "timestamp")
    {
      $inTimeStamp ="T";
    }
    elsif($el eq "profile")
    {
      $inProfile = "T"
    }
  }
  sub endPatchReg
  {
    my $p = shift;
    my $el = shift;
    #print "*__ $el __* \n";
    if ($el eq "product")
    {
      $inProduct = "F";
    }
    elsif($el eq "name")
    {
      $inName = "F";
    }
    elsif($el eq "version")
    {
      $inVersion = "F";
    }
    elsif($el eq "patchInstallEntry")
    {
      $inPatch = "F";
    }
    elsif($el eq "id")
    {
     $inId = "F";
    }
    elsif($el eq "timestamp")
    {
      $inTimeStamp ="F";
    }
    elsif($el eq "profile")
    {
      $inProfile = "F"
    }
  }

  sub charPatchReg
  {
    my $p = shift;
    my $info = shift;
    $info =~ s/\n/ /g;
#    print "$info \n";
    if($inProduct eq "T")
    {
      if($inName eq "T")
      {
        $currName = $info;
      }
      elsif($inVersion eq "T")
      {
        $currVer = $info;
        $currVer =~ s/\s//g;
        if($currVer)
        {
         my $compVerKey = "$currName"."|"."$currVer";
         $prodRef = $products{$compVerKey};
         if(not defined($prodRef))
         {
           my $msg = "Patches have been applied on $currName $currVer which doesn't exist";
           emdcommon_ocm::EMD_PERL_WARN($msg);
           $warnings->addWarning($msg);
         }
        }
      }
      elsif($inPatch eq "T")
      {


        if( $inId eq "T")
        {
           $currId = $info;

        }
        elsif($inTimeStamp eq "T")
        {
          $prodRef->{PATCH_LIST}->{$currId}->{TIMESTAMP} = $info;
        }
        elsif($inProfile eq "T")
        {
          $prodRef->{PATCH_LIST}->{$currId}->{PROFILE} = $info;
        }
     }
    }
  }

}#END OF PARSE PATCH REGISTRY BLOCK

{# START OF PARSE REGISTRY XML BLOCK

my $currProduct;
my $prodRef;
my $currJavaComp;
my $currTopComponent;
my ($inProduct, $inRelease, $inComp, $depFound, $compFound, $inComponent, $inJavaInst);
sub parseRegistryXml
{
  my $file = shift;
  if(my $retVal = OH_Utilities::checkPermissions($file))
  {
    $error->setError( CODE  => $retVal,
                      ARG   => $file);
    return $retVal; 
  }
  emdcommon_ocm::EMD_PERL_DEBUG("$file exists and is readable .. proceeding");
  ## -- parse the file
  if (open(FILE, $file))
  {
     close FILE;
     initVariables();
     my $p2 = new XML::Parser(ErrorContext => 2);
     emdcommon_ocm::EMD_PERL_DEBUG("Starting parsing of $file");
     $p2->setHandlers(Start => \&xmlStart,
                       End   => \&xmlEnd);

     eval {$p2->parsefile($file)};

     if ($@)
     {
         emdcommon_ocm::EMD_PERL_DEBUG("Failed to parse using default encoding:$@");
         foreach $_ (@encTable)
          {
             initVariables();
             emdcommon_ocm::EMD_PERL_DEBUG("Trying with $_ eincoding..");
             eval '$p2->parsefile($file, ProtocolEncoding => $_)';
             if ($@)
             {
                emdcommon_ocm::EMD_PERL_DEBUG("Failed again :$@");
             }
             else
             {
              last;
             }
           }
      }
      if ($@)
       {
            $error->setError( CODE  => OH_Error::XML_PARSE_ERR,
                              ARG   => $file);
            return OH_Error::XML_PARSE_ERR;
       }
      emdcommon_ocm::EMD_PERL_DEBUG("Finished parsing of $file");
      return OH_Error::SUCCESS;
   }
   else
   {
       $error->setError( CODE  => OH_Error::FILE_CANNOT_OPEN,
                         ARG   => $file);
       return OH_Error::FILE_CANNOT_OPEN;
   }
}



sub xmlStart
{
  my $p = shift;
  my $el = shift;
  #print " __ $el __ \n";
  if($el eq "product")
  {
    $inProduct = "T";
    my %atts = @_;
    #this is a new product
    $currProduct = {};
    $currProduct->{NAME} = $atts{name};

  }
  elsif($el eq "release")
  {
    if($inProduct)
    {
      $compFound = 0;
      $inRelease = "T";
      my %atts = @_;
      if($atts{Status} eq "installed")
      {
          my $ver = "$atts{level}.$atts{ServicePackLevel}.$atts{PatchLevel}";
          # we have both component and version
          if(exists($products{"$currProduct->{NAME}|$ver"}))
          {
            my $msg = " DUPLICATE prodct found ";
            emdcommon_ocm::EMD_PERL_WARN($msg);
            $warnings->addWarning($msg);
          }
          else
          {
            $products{"$currProduct->{NAME}|$ver"} = $currProduct;
          }
          $currProduct->{VER} = $ver;
          $currProduct->{TIMESTAMP} = $atts{InstallTime};
          $currProduct->{INST_LOC} = removeTrailingSlash($atts{InstallDir});
          $currProduct->{IS_TOP_LEVEL} = 'Y';

      }
    }
  }
  elsif($el eq "component")
  {
    my %atts = @_;
    if(exists ($atts{version}))
    {
      # that means this is a version component
      #$inTopComponent = "T";
      $depFound = 0;
      $currTopComponent = $products{"$atts{name}|$atts{version}"};
      if(not defined ($currTopComponent))
      {
        $products{"$atts{name}|$atts{version}"}=$currTopComponent = {};
      }
      else
      {
        my $msg= "DUPLICATE product $atts{name}|$atts{version} found ";
        emdcommon_ocm::EMD_PERL_WARN($msg);
      }
      $currTopComponent->{NAME} = $atts{name};
      $currTopComponent->{VER}  = $atts{version};
      $currTopComponent->{INST_LOC} = 
              OH_Utilities::removeTrailingSlash($atts{InstallDir});
      #put it in the tl component also
      $currProduct->{DEP_ON_LIST}->{$compFound}->{COMP}->{NAME}= $atts{name};
      $currProduct->{DEP_ON_LIST}->{$compFound}->{COMP}->{VER}= $atts{version};
      if(defined($atts{InstallDir}))
      {
       $currProduct->{DEP_ON_LIST}->{$compFound}->{INST_LOC}=
              OH_Utilities::removeTrailingSlash($atts{InstallDir});
      }
      $compFound++;
    }
    else
    {
      # that means these are versionless components
      $currTopComponent->{DEP_ON_LIST}->{$depFound++}->{COMP}->{NAME} = $atts{name};


    }
  }
  elsif($el eq "java-installation")
  {
    $inJavaInst = "T";
    my %atts = @_;
    my ($name, $ver) = ($atts{Name},defined($atts{version})?$atts{version}:"0.0.0.0");
    $currJavaComp = $products{"$name|$ver"};
    if(not defined ($currJavaComp))
      {
        $products{"$name|$ver"}= $currJavaComp = {};
      }
    else
      {
        my $msg= "DUPLICATE product $name|$ver found ";
        emdcommon_ocm::EMD_PERL_WARN($msg);
        $warnings->addWarning($msg);
      }
    $currJavaComp->{NAME} = $name;
    $currJavaComp->{VER} = $ver;
    
  }
  elsif($el eq "dependent-product")
  {
    if($inJavaInst eq "T")
    {
      my %atts = @_;
      my ($name,$ver) = ($atts{Name}, $atts{Version});
      my $prodRef = $products{"$name|$ver"};
      if(not defined($prodRef))
      {
       my $msg = "$currJavaComp->{NAME}|$currJavaComp->{VER} is depended upon by $name|$ver which doesn't exist";
       emdcommon_ocm::EMD_PERL_WARN($msg);
       $warnings->addWarning($msg);
      }
      else
      {
        my $sizeOfDepList = scalar(keys %{$prodRef->{DEP_ON_LIST}});
        $prodRef->{DEP_ON_LIST}->{$sizeOfDepList}->{COMP}->{NAME}= $currJavaComp->{NAME};
        $prodRef->{DEP_ON_LIST}->{$sizeOfDepList}->{COMP}->{VER}= $currJavaComp->{VER};
        
      }
    }
  }
}

sub xmlEnd
{
  my $p = shift;
  my $el = shift;
  #print ("* __ $el __* \n");
  if($el eq "product")
  {
    $inProduct = "F";
  }
  elsif($el eq "release")
  {
    $inRelease = "F";
  }
  elsif($el eq "component")
  {
    $inComponent = "F";
  }
  elsif($el eq "java-installation")
  {
    $inJavaInst = "F";
    $currJavaComp = undef;
  }

}


}# END OF PARSE REGISTRY XML BLOCK


sub parseRegistryDat
{
    ## -- check for file existance and redability
    my $file = shift;
    if( my $retVal = OH_Utilities::checkPermissions($file))
    {
     $error->setError(CODE => $retVal,
                        ARG  => $file) ;
     return $retVal;
    }
     emdcommon_ocm::EMD_PERL_DEBUG("$file Exists and is readable .. proceeding");
     ## -- parse the file
     my $childPid = undef;
     if (open(FILE, $file))
      {
        close FILE;
        my $p2 = new XML::Parser(ErrorContext => 2);
        initVariables();
        # .dat file is an encrypted file. First we need to convert it into plain text
        $childPid = open(DAT_2_XML,"-|");
        unless($childPid)#child
        {
          &decrypt($file);
          exit(0);
        }
        else
        {
        emdcommon_ocm::EMD_PERL_DEBUG("Starting parsing of $file");
        $p2->setHandlers(Start => \&datStart,
                           End => \&datEnd);

        eval {$p2->parse(*DAT_2_XML)};

        if ($@)
         {
            emdcommon_ocm::EMD_PERL_DEBUG("Failed to parse using default encoding:$@");
            foreach $_ (@encTable)
            {
                initVariables();
                emdcommon_ocm::EMD_PERL_DEBUG("Trying with $_ eincoding..");
                eval '$p2->parse(*DAT_2_XML, ProtocolEncoding => $_)';

                if ($@)
                {
                  emdcommon_ocm::EMD_PERL_DEBUG("Failed..:$@");

                }
                else
                {
                 #succees
                 last;
                }
            }
         }
        if ($@)
         {
            emdcommon_ocm::EMD_PERL_DEBUG("Failed to parse $file");
            return OH_Error::XML_PARSE_ERR;# so that we proceed to .xml parsing
          }
       wait; 
       close(DAT_2_XML);
        emdcommon_ocm::EMD_PERL_DEBUG("Successfully collected $file");
        return OH_Error::SUCCESS;;
     }
    }
   else
    {
      emdcommon_ocm::EMD_PERL_DEBUG("Failed to open $file");
      return OH_Error::FILE_CANNOT_OPEN;# so that we proceed to .xml parsing
    }
}

#start tag handler when parsing registry.dat
sub datStart
{
  my $p = shift;
  my $el = shift;
  #print "__ $el ___ \n";

  if ($el eq "product-alias")
  {
    my %atts = @_;
    my $status = $atts{status};
    if(($status) && ($status eq "installed"))
    { #this is an installed one
      my ($name, $ver) = ($atts{id},$atts{version});
      my  $prodInfo = $products{$name."|".$ver};
      emdcommon_ocm::EMD_PERL_DEBUG("TL PRODUCT $name|$ver FOUND");
      if(not defined($prodInfo))
      {
        $products{$name."|".$ver} = $prodInfo = {};
      }
      $currProduct = $prodInfo;# TO store install directory and time later
      $prodInfo->{NAME} = $name;
      $prodInfo->{VER}  = $ver;
      $prodInfo->{IS_TOP_LEVEL} = 'Y';
      $inProductAlias = "T";
   }
 }
 elsif ($el eq "product-ref")
 {
     if($inProductAlias eq "T")
     {
       my %atts = @_;
       my ($name, $ver) = ($atts{product},$atts{release});
       my  $prodInfo = $products{"$name|$ver"};
       emdcommon_ocm::EMD_PERL_DEBUG("Proudct $name|$ver found");
       if(not defined($prodInfo))
       {
        $products{"$name|$ver"} = $prodInfo = {};
       }
       $prodInfo->{NAME} = $name;
       $prodInfo->{VER}  = $ver;
       $currProduct->{DEP_ON_LIST}->{$productsFound}->{COMP}->{NAME}= $name;
       $currProduct->{DEP_ON_LIST}->{$productsFound}->{COMP}->{VER}= $ver;
       $productsFound++;
     }
 }
 elsif($el eq "install-info")
 {# these are the install dir and timestamp of the product
   if(($inProductAlias eq "T") ||($inProduct eq "T"))
   {
     my %atts= @_;
     $currProduct->{INST_LOC} = removeTrailingSlash($atts{"install-dir"});
     $currProduct->{TIMESTAMP} = $atts{"install-date"};
     emdcommon_ocm::EMD_PERL_DEBUG("Updating install info for $currProduct->{NAME}|$currProduct->{VER}");
   }
 }
 elsif($el eq "product")
 {
     my %atts = @_;
     my $status = $atts{status};
     if(($status) && ($status eq "installed"))
     {
       $currProduct = $products{"$atts{id}|$atts{release}"};
       $inProduct = "T";
       emdcommon_ocm::EMD_PERL_DEBUG("Getting comps for product $atts{id}|$atts{release}");
     }
 }
 elsif($el eq "jvm-ref")
 {
     if($inProduct eq "T")
     {
       my %atts= @_;
       $currProduct->{DEP_ON_LIST}->{$componentsFound++}->{COMP}->{NAME}= $atts{id};
       emdcommon_ocm::EMD_PERL_DEBUG("Found jvm comp $atts{id}");
     }
 }
 elsif($el eq "component")
 {
     if($inProduct eq "T")
     {
       my %atts = @_;
       my $status = $atts{status};
       if(($status) && ($status eq "installed"))
       {
         my $comp = $atts{id};
         my $compFullName = "$comp";#TODO ($currProduct->{NAME} $currProduct->{VER})";
         my $compInfo ={};# $comps{$compKey};
         emdcommon_ocm::EMD_PERL_DEBUG("Found component $comp for product ");
         $compInfo->{NAME} = $compFullName;
         $currComp = $compInfo;
         $inComponent = "T";
         $currProduct->{DEP_ON_LIST}->{$componentsFound++}->{COMP}=$compInfo;
       }
     }
 }
 elsif($el eq "req-group")
 {
     $inReqGroup = "T";
     $depFound = 0;
 }
 elsif($el eq "dependency")
 {
     if(($inComponent eq "T") && ($inReqGroup eq "T"))
     {
       my %atts = @_;
       $currComp->{DEP_ON_LIST}->{$depFound++}->{NAME} = "$atts{component}";#TODO ($atts{product} $atts{release})";
       emdcommon_ocm::EMD_PERL_DEBUG("Found dep comp $atts{component} for $currComp->{NAME}");
     }
 }

}

#end tag handler when parsing registry.dat
sub datEnd
{
  my $p = shift;
  my $el = shift;
  #print "* __ $el __* \n";
  
if ($el eq "product-alias")
  {
    #$currProduct = ();
    $inProductAlias = "F";
  }
  elsif ($el eq "product")
  {
    #$currProduct = ();
    $inProduct = "F";
    $componentsFound = 0;
  }
  elsif($el eq "component")
  {
    #$currComp = undef;
    $inComponent = "F";
    $depFound = 0;
  }
  elsif($el eq "req-group")
  {
    $inReqGroup = "F";
    $depFound = 0;
  }
}



#******************************************************************************
# Takes an input and output file name as arguments
# Encrypt / Decrypts data in input file and writes
# encrypted/decrypted data to output file.
#
sub decrypt
{
  my $input_file = shift;
  open(INFILE, $input_file);
  while(<INFILE>)
  {
    my $str = $_;
    my $seed = seed_string(length($str));
    my $crypt = $str ^ $seed;
    print "$crypt";
  }
  close (INFILE);
}

#******************************************************************************
# a seed string of the same length as the
# message is generated.. thats than
# XOR'ed, character-by-character, on the
# message, the result is mixed but when
# you XOR this mix with the same seed string,
# you get the original message back
# char(85) is U 
#

sub seed_string {
  my $len = shift;
  my $ret;
  for (1 .. $len) { $ret .= chr(85) }
  return $ret;
}

1;
