#!/usr/local/bin/perl
#
# $Header: emagent/sysman/admin/scripts/adrviewer.pl /main/20 2012/05/21 11:56:03 fagonzal Exp $
#
# adrviewer.pl
#
# Copyright (c) 2006, 2012, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      adrviewer.pl - adrviewer OS Fetchlet
#
#    DESCRIPTION
#      adrviewer is to access adr repository.
#
#      Error codes
#        DIU-59001: uidrvpath does not exist
#        DIU-59003: Can't open path
#        DIU-59004: Cannot chdir
#        DIU-59005: Invalid command
#        DIU-59006: Error invoking executable
#
#    MODIFIED   (MM/DD/YY)
#    fagonzal    05/15/12 - Allowing uidrvhome and uidrvlib to be passed in
#                           separately
#    fagonzal    05/15/12 - Adding library path to trace output
#    fagonzal    03/15/12 - bug 13845170
#    fagonzal    02/27/12 - bug 13777673
#    fagonzal    01/11/12 - Dumping trace info in case of error
#    fagonzal    08/12/11 - bug 12870888
#    fagonzal    06/22/11 - bug 12680446 - passing in OCM endpoint (repeater
#                           url)
#    fagonzal    05/31/11 - bug 12433132
#    fagonzal    05/02/11 - Encoding strings as UTF-8 (bug 9181659)
#    fagonzal    04/28/11 - Deriving JAVA_HOME from ORACLE_HOME, passing in
#                           agent ORACLE_HOME
#    fagonzal    04/05/11 - Setting LD_LIBRARY_PATH_64 even when clear
#    fagonzal    03/15/11 - Improving error handling in the case where there is
#                           a problem invoking uidrvci
#    fagonzal    02/28/11 - bug 10248915
#    fagonzal    02/28/11 - bug 11797970
#    fagonzal    02/06/11 - Fixing SshWrapper,CopyPkgFile invocation
#    fagonzal    01/14/11 - Calling new Java upload program
#    fagonzal    11/08/10 - Using temp directory for storing package files
#    fagonzal    04/02/10 - Cleaning up script to support Exadata
#    fagonzal    04/01/10 - Renaming OCMAccess
#    fagonzal    03/22/10 - Cleaning up temp directory discovery
#    fagonzal    03/10/10 - Removing quickPackage command
#    fagonzal    02/19/10 - Setting LD_LIBRARY_PATH_64 for SPARC64
#    fagonzal    02/11/10 - Changing location of OCM jars
#    fagonzal    01/12/10 - bug 9258685
#    fagonzal    12/23/09 - Adding OH/bin to path before invoking uidrvci
#    fagonzal    10/20/09 - Fixing java invocation, classpath, for package
#                           upload command
#    fagonzal    09/23/09 - Adding support for uidrvci in OIC
#    balnaff     01/28/09 - bug 8205527
#    fagonzal    01/15/09 - Adding support for passing ORACLE_SID to work
#                           around bug 7140727
#    fagonzal    01/08/09 - Reverting to use open2, as open3 for some reason is
#                           unreliable on NT
#    balnaff     12/17/08 - getting ORACLE_CONFIG_DIRECTORY
#    fagonzal    12/04/08 - Using system-dependent lib path, logging stderr output
#    fagonzal    11/24/08 - Adding support for passing correlation keys
#    fagonzal    08/28/08 - Logging arguments for ARGLIST case
#    balnaff     08/21/08 - 
#    fagonzal    08/13/08 - Adding option to pass oracle home, ADR base and ADR
#                           home through STDIN
#    mappusam    06/27/08 - bug-6965447 fix
#    fagonzal    04/01/08 - Removing remote correlation key check before
#                           creating remote package
#    dchakumk    03/10/08 - XbranchMerge dchakumk_6752_xbmsrc from
#                           st_emdbsa_11.1
#    balnaff     01/25/08 - bug 6732588
#    balnaff     01/24/08 - bug 6732588
#    fagonzal    12/19/07 - Fixing bug that prevented sweep command from
#                           executing
#    jsoule      08/29/07 - remove unnecessary EM_TARGET_ADR_BASE parameter
#    balnaff     07/01/07 - adding ml creds
#    balnaff     05/15/07 - 
#    loliu       05/21/07 - Disabled output TRACE for showXMLTrace and
#                           showTraceSection
#    jsoule      04/20/07 - enable package name, description
#    jsoule      04/12/07 - use backslash to find uidrvci.exe
#    keiwong     04/03/07 - fix bug 5898320
#    loliu       03/25/07 - Changed adrviewer.pl to use target ADR base and
#                           home from environment
#    jsoule      03/15/07 - copy temp file creation
#    jsoule      03/09/07 - add modified createPackageCorrelated
#    jsoule      03/03/07 - add remote packaging special cases
#    keiwong     01/09/07 - fix bug 5750178
#    keiwong     12/04/06 - fix bug 5678573
#    hopark      10/25/06 - add package state change
#    hopark      10/10/06 - add upload
#    hopark      10/05/06 - add quick packaging
#    hopark      09/19/06 - return manifest
#    hopark      07/30/06 - change adrbase
#    hopark      05/09/06 - Creation
# 

use Time::Local;
use IPC::Open2;
use File::Spec;
use File::Temp qw/ tempfile tempdir /;
use IO::Select;
use Net::Domain qw/ hostname /;

# declare stdinArgs early.
my %stdinArgs;

my $traceFile;
my @traceDump;

my $traceLevel = $ENV{EMAGENT_PERL_TRACE_LEVEL};

# Returns the path to the trace log file.
#
# First looks at the ADRVIEWER_LOG_DIR environment variable. If this variable
# is not defined, uses $ORACLE_HOME/log. If this directory does not exist,
# it defaults to the user home directory. If the user home directory is also 
# not defined, logging will be disabled.
sub getTraceFilePath() {
    my $traceFileDir = $ENV{ADRVIEWER_LOG_DIR};
    if ($traceFileDir eq "" || !-e $traceFileDir) {
        $traceFileDir = $orahome.$separator."log";
        if (!-e $traceFileDir) {
            $traceFileDir = $ENV{HOME};
            if ($NT) {
                $traceFileDir = $ENV{USERPROFILE};
            }
        }
    }

    if ($traceFileDir eq "" || !-e $traceFileDir) {
        return ""; 
    } else {
        return $traceFileDir.$separator."adrviewer.log";
    }
}

#
# Writes a trace file entry (if enabled), and adds the entry to the 
# internal trace dump
#
sub TRACE {
    my @args = @_;

    # Add the trace to the internal trace dump
    push(@traceDump, join(" ", @args));

    traceNoDump(@args);
}

#
# Writes a trace file entry (if enabled), without adding it to the 
# internal trace dump
#
sub traceNoDump {
    my @args = @_;

    # Write to trace file, if enabled
    if ($traceLevel ne "" && $traceLevel <= 1 && $traceFile ne "") {
        open(LOG, ">>", $traceFile);
        print LOG localtime(time) . "  - ";
        print LOG @args;
        close(LOG);
    }
}

sub isPerlDebugEnabled {
    my $trace_level = $ENV{EMAGENT_PERL_TRACE_LEVEL};
    return $trace_level ne "" && $trace_level == 1;
}

sub parse_csv {
    my $text = shift;      # record containing comma-separated values
    my @new  = ();
    push(@new, $+) while $text =~ m{
        # the first part groups the phrase inside the quotes.
        # see explanation of this pattern in MRE
        "([^\"\\]*(?:\\.[^\"\\]*)*)",?
           |  ([^,]+),?
           | ,
    }gx;
    push(@new, undef) if substr($text, -1,1) eq ',';
    return @new;      # list of values that were comma-separated
}

sub trim {
    my $string = shift(@_);
    $string =~ s/\s//g;
    return $string;
}

# Parse the input arguments and put them into a hash
sub parseInputArguments {
  my @inputArguments = @_;
  my %r;
  while (@inputArguments) {
    my $line = shift(@inputArguments);
    if ($line =~ /(.*)=(.*)/) {
      if ($2 eq "__BeginProp__") {
        while (@inputArguments) {
          $line = shift(@inputArguments);
          if ($line ne "__EndProp__\n") {
            $r{"$1"} .= "$line";
          } else {
            last;
          }
        }
      } else {
        $r{"$1"} = "$2";
      }
    }
  }
  return %r;
}

# Returns the name of the shared library path environment variable
# for the current platform
sub getLibPathName {
    my $lib_path_name = "LD_LIBRARY_PATH";
    my $os = $^O;

    if ($os eq "hpux") {
        $lib_path_name = 'SHLIB_PATH' ;
        delete $ENV{LD_LIBRARY_PATH};
    } elsif ( $os eq "aix" ) {
        $lib_path_name = 'LIBPATH' ;
    }

    return $lib_path_name;
}

# Returns the folder location to store package files in
sub getPackageFolder {
  my $packageFolder = tempdir();
  TRACE("packageFolder: $packageFolder\n");
  return $packageFolder;
}

# Create a temporary file
# return fileHandle and fileName
sub createTempFile {
  my ($suffix) = @_;
  my $dir = tempdir(CLEANUP => 1);
  my $fileHandle;
  my $tempFilename;

  if (defined($suffix)) {
    ($fileHandle, $tempFilename) = 
      tempfile(DIR => $dir, SUFFIX => $suffix );
  } else {
    ($fileHandle, $tempFilename) = tempfile(DIR => $dir);
  }

  TRACE("tempFilename: $tempFilename\n");

  return ($fileHandle, $tempFilename);
}

sub invokeCmd {
    @fields = parse_csv($arglist);
    $cmd = shift(@fields);
    TRACE("cmd: $cmd\n");

    if ($cmd eq "checkFileExist") {
        $path = shift(@fields);
        $result = 0;
        if (-e $path) {
            $result = 1;
        }
        print $result;
        return;
    }

    if ($cmd eq "readFile") {
        $path = shift(@fields);
        open F, "< $path" or die "ERROR,DIU-59003: Can't open $path : $!";
        my @f = <F>;
        close F;      
        print @f;
        return;
    }

    if ($cmd eq "copyPackageFile") {
        copyPackageFile(@fields);
        return;
    }

    if ($cmd eq "uploadPhysicalPackage") {
        uploadPackage(@fields);
        return;
    }

    if ($cmd eq "setPackageUploadedState") {
        setPackageUploadedState(@fields);
        return;
    }

    unshift(@fields,$cmd);

    # Special case command processing for correlation key APIs
    if ($cmd eq "getRemoteCorrKeysFromPackage") {
        # create temp file for ips to populate with correlated-key data
        my $corrKeysXMLFile = createTempFile(".xml");
        push(@fields, $corrKeysXMLFile);
    } elsif ($cmd eq "createPackageCorrelated") {
        invokeCreatePackageCorrelated(@fields);
        return; # do not call invokeADR
    } elsif ($cmd eq "useRemoteCorrKeysFromPackage" ||
             $cmd eq "checkRemoteCorrKeysFromPackage") {
        push (@fields, createCorrKeysTempFile());
    }

    invokeADR(@fields);
}

# Parses a properties file (used to get the agent classpath from emd.properties)
#
# Copied from emagent/scripts/unix/SecureUtil.pm
sub parsePropertiesFile {
  my($fname) = @_;
  my %lprop;

  if (! -T $fname ) {
    print "File $fname is not a text file\n";
    next;
  }

  open(FILE,$fname) or die "Can not read file: $fname\n$!\n";
  while (<FILE>) {
    # Remove leading and trailing whitespaces
    s/^\s+|\s+$//;
    s/#.*$//g;

    # Validate each non-empty line
    if (! /^$/) {
      my($name,$value) = /([^=]+)\s*=\s*(.+)/;
      if (defined($name) && defined($value)) {
         $name  =~ s/^\s+|\s+$//g;
         $value =~ s/^\s+|\s+$//g;
         $lprop{$name} = $value;
      }
    }
  }
  close(FILE);

  # Return success
  return %lprop;
}

sub copyPackageFile() {
    my @fields = @_;
    my $remotePkgFile = shift(@fields);

    TRACE("remotePkgFile: $remotePkgFile\n");

    # Setting up classpath for invoking CopyPkgFile.class
    my $classpath = $emdroot.$separator."lib".$separator."jsch.jar";
    $classpath .= $classpath_separator;
    $classpath .= $agentlibpath."swb.jar";
    TRACE("classpath: $classpath\n");

    # Constructing the java command, which takes four arguments:
    # username, hostname, localDirectory and remoteFile
    my $packageFolder = getPackageFolder();
    TRACE("packageFolder: $packageFolder\n");

    my $exec = "$javabinary -classpath $classpath oracle.sysman.gcagent.addon.swb.CopyPkgFile $sshUsername $sshHostname $packageFolder $remotePkgFile";
    TRACE("exec: $exec\n");

    open(LOGFILE, ">>$traceFile");
    open(SAVEERR, ">&STDERR");
    open(STDERR, ">>&LOGFILE");
    open(SAVEOUT, ">&STDOUT");
    open(STDOUT, ">>&LOGFILE");

    open(INPUT, "| $exec");
    print INPUT "$sshPassword\n";
    close(INPUT);
    
    my $exitcode = $? >> 8;
    TRACE("exitcode: $exitcode\n");

    close(STDOUT);
    open(STDOUT, ">&SAVEOUT");
    close(STDERR);
    open(STDERR, ">&SAVEERR");

    close(LOGFILE);

    my ($volume,$directories,$file) = File::Spec->splitpath($remotePkgFile);
    my $localPkgFile = File::Spec->catdir($packageFolder, $file);
    TRACE("localPkgFile: $localPkgFile\n");
    print STDOUT $localPkgFile;

    return $exitcode;
}

sub uploadPackage() {
    my @fields = @_;
    my $srNumber = shift(@fields);
    my $pkgFile = shift(@fields);
    my $mosUsername = shift(@fields);
    my $mosPassword = shift(@fields);
    my $repeaterUrl = shift(@fields);

    TRACE("srNumber: $srNumber\n");
    TRACE("pkgFile: $pkgFile\n");
    TRACE("repeaterUrl: $repeaterUrl\n");

    # Set up the classpath for invoking UploadPackage.class
    my $ocmlibpath = $myorahome.$separator."oui".$separator."jlib".$separator;
    my $ocmjlibpath = $ocmlibpath.$separator."jlib".$separator;
    TRACE("ocmlibpath: $ocmlibpath\n");
    TRACE("ocmjlibpath: $ocmjlibpath\n");
    
    # Set up the boot classpath (workaround for bug 13852758)
    my $classpath = $ocmjlibpath.$separator."emocmcommon.jar";
    my $osInfoCommand = "$javabinary -cp $classpath OsInfo";
    TRACE("classpath: $classpath\n");
    TRACE("osInfoCommand: $osInfoCommand\n");

    my $os = `$javabinary -cp $classpath OsInfo`;
    TRACE("os: $os\n");

    my $bootClasspathOption = "";
    if ($os =~ m/(s390|s390x)/) {
      my $bootClasspath = $ocmjlibpath."jcert.jar";
      $bootClasspath .= $classpath_separator;
      $bootClasspath .= $ocmjlibpath."jnet.jar";
      $bootClasspath .= $classpath_separator;
      $bootClasspath .= $ocmjlibpath."jsse.jar";

      $bootClasspathOption = "-Xbootclasspath/a:$bootClasspath";
    }
    TRACE("bootClasspathOption: $bootClasspathOption\n");

    $classpath = $ocmlibpath."emocmutl.jar";
    $classpath .= $classpath_separator;
    $classpath .= $agentlibpath."swb.jar";
    TRACE("classpath: $classpath\n");

    open(LOGFILE, ">>$traceFile");
    open(SAVEERR, ">&STDERR");
    open(STDERR, ">>&LOGFILE");

    # Construct the java command 
    my $exec = "$javabinary $bootClasspathOption -classpath $classpath oracle.sysman.gcagent.addon.swb.UploadPackage";
    TRACE("exec: $exec\n");

    my $pid;
    eval {
        $pid = open2(*BUFP, *INPUT, $exec);
    };

    if ($@) {
        TRACE("ERROR: $@\n");
        print "ERROR,DIU-59006: Error invoking UploadPackage.\n";
        return $?;
    }

    print INPUT "ORACLE_HOME=$myorahome\n";
    print INPUT "PACKAGE_FILE=$pkgFile\n";
    print INPUT "MOS_USERNAME=$mosUsername\n";
    print INPUT "MOS_PASSWORD=$mosPassword\n";
    print INPUT "SR_NUMBER=$srNumber\n";
    print INPUT "REPEATER_URL=$repeaterUrl\n";
    print INPUT "\n"; # bug 13777673. Closing STDIN is not recognized on Windows
    close(INPUT);

    while (<BUFP>) {
      my $line = $_;

      # On Windows, a CR char is seen before the newline
      # Working around this by stripping each line of whitespace
      $line =~ s/^\s+//; # remove leading spaces
      $line =~ s/\s+$//; # remove trailing spaces (including newlines)
      $line .= "\n";     # add a newline

      traceNoDump("output: $line");
      print $line;
    }
    close(BUFP);

    # need to avoid running out of resource in NT
    waitpid($pid, 0);

    my $exitcode = $? >> 8;
    TRACE("exitcode: $exitcode\n");

    close(STDERR);
    open(STDERR, ">&SAVEERR");

    return $exitcode;
}

#
# createCorrKeysTempFile
#  Pull the correlation keys from stdin arguments, populate a temp file with
#  them.  Return the filename of the temp file.
#
sub createCorrKeysTempFile() {
    # populate temp file with correlated-key data for ips
    $corrKeysXMLFile = createTempFile(".xml");
    open(F, ">$corrKeysXMLFile")
        or die "ERROR,DIU-59003: Can't open temp file : $!";
    print {F} ($corrkey);
    close(F);
    return $corrKeysXMLFile;
}

#
# invokeCreatePackageCorrelated
#  This routine intercepts a createPackageCorrelated invocation from the OMS
#  This processing is necessary to avoid making many OMS->Agent calls.
#
# $createCmd should be createPackageCorrelated
# $name name of the main package
#
sub invokeCreatePackageCorrelated() {
    my ($createCmd, $name) = @_;
    my $output = '';
    my $corrKeysXMLFile = createCorrKeysTempFile();

    # redirect printing to the trace file -- this suppresses invokeADR output.
    # if tracing is disabled, attempt to trace to /dev/null
    if (isPerlDebugEnabled() || !open(LOG, ">/dev/null")) {
      open(LOG, ">>$traceFile");
    }
    select(LOG);

    # correlated keys apply; continue with the create
    invokeADR($createCmd, $name, 
        "correlated_package on instance $ENV{ORACLE_SID}");
    $output = $results[0];
    TRACE("output: $output\n");

    if (!($results[0] =~ m/^ERROR/)) {
        # no error indicates success;
        # output has now been determined, unless errors below
        my ($relation, $corrPackageID, $corrADRHome) = parse_csv($results[0]);
        invokeADR("useRemoteCorrKeysFromPackage",
            $corrPackageID, $corrKeysXMLFile);

        if ($results[0] =~ m/^ERROR/) {
            # error in useRemoteCorrKeysFromPackage"; report it and clean up
            TRACE("error populating package; cleaning up...\n");
            $output = $results[0];
            invokeADR("deletePackage", $corrPackageID);
        }
    }

    # restore printing to STDOUT
    select(STDOUT);

    # the result
    print $output;

    # Dump out trace info if an error occurred
    if ($output =~ m/^ERROR/) {
        for my $line (@traceDump) {
            print "TRACE,$line";
        }
    }
}

sub setPackageUploadedState() {
    my $pkgId = shift(@_);
    my $seq = shift(@_);
    
    my @fields = ();
    push(@fields, "updatePackageUploadStatus");
    push(@fields, $pkgId);
    push(@fields, $seq);
    push(@fields, 6); # Package uploaded state
    invokeADR(@fields);
    
    # format time in  2006-10-25 16:16:01 GMT-07:00
    ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime(time);
    my $offset  = sprintf "%.1f", (timegm(localtime) - time) / 3600;
    my $minutes = sprintf "%02d", abs( $offset - int($offset) ) * 60;
    $timestr =
        sprintf ("%4d-%02d-%02d %02d:%02d:%02d GMT%+03d:%02d",
            $year+1900,$mon+1,$mday,$hour,$min,$sec,int($offset), $minutes);
    @fields = ();
    push(@fields, "setPackageUploadTime");
    push(@fields, $pkgId);
    push(@fields, $seq);
    push(@fields, $timestr);
    invokeADR(@fields);
}

#
#  Function:    sweepADR
#  Description: check whether ADR should be swept and do so when necessary.
#  Arguments:   $_[0] = target's adrbase
#               $_[1] = target's adrhome
#               $_[2] = target's oracle_home
#               $_[3] = arg to adrci sweep directive
#
sub sweepADR {
  my $adr_base   = shift(@_);
  my $adr_home   = shift(@_);
  my $uidrv_home = shift(@_);
  my $sweep      = shift(@_);

  my $adr_homepath = "";
  if ($sweep && $adr_base)
  {
    $regexp = quotemeta("$adr_base$separator");
    if ($adr_home =~ /$regexp(.*)/)
    {
      $adr_homepath = $1;
      $adrciexe  = "$uidrv_home$separator".
                   ($NT ? "adrci.exe" : "adrci");
      $adrscript = "exec=\"set base $adr_base;".
                          "set homepath $adr_homepath;".
                          "sweep $sweep\"";
      TRACE("Sweeping ADR with [$adrciexe $adrscript]\n");
      if (system("$adrciexe $adrscript"))
      {
        # TODO: this should be error level
        TRACE("ADR sweep failed; error code: ".($?>>8).
              " signal: ".($?&255)." error: $!\n");
      }
    }
  }
}

sub invokeADR() {
    my @fields = @_;
    my $cmd = shift(@fields);
    my $arguments = "$cmd";

    foreach (@fields) {
        $arg = $_;
        $arguments .= ",";

        if ($arg eq "_TEMP_") {
            $arg = getPackageFolder();
        }

        if ($arg =~ m/(,)/) {
            $arguments .= "\"" . $arg . "\"";
        } else {
            $arguments .= $arg;
        }
    }
    TRACE("arguments: $arguments\n");

    # identify the XML file for getRemoteCorrKeysFromPackage
    my $corrKeysXMLFile = '';

    if ($cmd eq "getRemoteCorrKeysFromPackage") {
        $corrKeysXMLFile = $fields[$#fields];
    }

    # Bug 11797970. IPS packaging changes the cwd, and reverts back to the 
    # original directory. Because of this, we need to be in a directory that 
    # the current user can access.
    chdir($uidrvhome) || die "ERROR,DIU-59004: Cannot chdir $uidrvhome: $!";

    my $uidrv = "./uidrvci";
    if ($NT) {   
        $uidrv = ".\\uidrvci.exe";
    }

    if (!-e $uidrv) {
        my $uidrvpath = $uidrvhome.$separator.$uidrv;
        print "ERROR,DIU-59001: $uidrvpath does not exist\n";
        return;
    }

    my $exec = "$uidrv -uidrv_adrbase=$target_adrbase -target_adrhome=$target_adrhome";
    TRACE("exec: $exec\n");

    @results = ();

    my $pid;
    eval {
        $pid = open2(*OUTPUT, *INPUT, $exec);
    };

    if ($@) {
        TRACE("ERROR: $@\n");
        print "ERROR,DIU-59006: Error invoking uidrvci.\n";
        return $?;
    }

    print INPUT "$arguments\n";

    print INPUT "\n"; # need this for uidrvci to exit
    close(INPUT);

    # Trace output by default, except trace file maps and sections
    $traceOutput = 1;
    if (($cmd eq "showXMLTraceMap") || ($cmd eq "showTraceSection")) {
        $traceOutput = 0;
    }

    # Flag to track whether to dump tracing info
    my $dumpTrace = 0;

    while (<OUTPUT>) {
        my $line = $_;

        # Set dump trace flag if an error occurred
        if ($line =~ m/^ERROR/) {
            $dumpTrace = 1;
            $traceOutput = 1;
        }
 
        # bug 9258685. On Windows, a CR char is seen before the newline
        # Working around this by stripping each line of whitespace
        $line =~ s/^\s+//; # remove leading spaces
        $line =~ s/\s+$//; # remove trailing spaces (including newlines)
        $line .= "\n";     # add a newline

        if ($traceOutput) {
            traceNoDump("output: $line");
        }

        # Special case output for getRemoteCorrKeysFromPackage
        # On success, XML file contents are returned
        if ($cmd eq "getRemoteCorrKeysFromPackage" && # command
            $line == 1 &&                             # success
            $corrKeysXMLFile) {                       # unspooled file
            if (!open(F, "< $corrKeysXMLFile")) {
                print "ERROR,DIU-59003: Can't open temp correlation key file : $!";
            } else {
                while (my $fline = <F>) {
                    print $fline;
                }
                close(F);
            }

            # we tried once; don't try again
            $corrKeysXMLFile = '';
        } else {
            print $line;
        }

        push(@results, $line);
    }

    close(OUTPUT);

    # If the dump trace flag is set, dump the trace at the end of the output
    if ($dumpTrace) {
        for my $line (@traceDump) {
            print "TRACE,$line";
        }
    }

    # need to avoid running out of resource in NT
    waitpid($pid, 0);

    my $ec = $?;
    my $exitcode = $ec >> 8;
    my $sig = $ec & 127;
    my $has_coredump = $ec & 128;
    return $exitcode;
}

$NT = 0;
if ($^O =~ /MSWin32/i){
    $NT = 1;
}

$separator = $NT ? "\\" : "\/";
$classpath_separator = $NT ? ";":":";

# All strings are UTF-8
use open IO => ":encoding(utf-8)";

# Grab all the input arguments from stdin and parse them
@inputArguments = <STDIN>;
%stdinArgs = parseInputArguments(@inputArguments);

# Retrieve input variables from stdin if present
$uidrvhome = $stdinArgs{"UIDRV_HOME"};
$uidrvlib = $stdinArgs{"UIDRV_LIB"};
$orahome = $stdinArgs{"ORACLE_HOME"};
$myorahome = $stdinArgs{"AGENT_ORACLE_HOME"};
$emdroot = $stdinArgs{"AGENT_EMD_ROOT"};
$target_adrbase = $stdinArgs{"ADR_BASE"};
$target_adrhome = $stdinArgs{"ADR_HOME"};
$arglist = $stdinArgs{"ARGLIST"};
$corrkey = $stdinArgs{"CORRKEY"};
$isExadataCell = $stdinArgs{"EXADATA"};
$sshUsername = $stdinArgs{"SSH_USERNAME"};
$sshPassword = $stdinArgs{"SSH_PASSWORD"};
$sshHostname = $stdinArgs{"SSH_HOSTNAME"};

# Workaround for bug 7140727
$sid = $stdinArgs{"SID"};
$ENV{ORACLE_SID} = $sid;

$traceFile = getTraceFilePath();

if (!$orahome) {
  $orahome = $ENV{"ORACLE_HOME"};
} else {
  $ENV{"ORACLE_HOME"} = $orahome;
}

TRACE("NT: $NT\n");

TRACE("uidrvhome: $uidrvhome\n");
TRACE("uidrvlib: $uidrvlib\n");
TRACE("orahome: $orahome\n");
TRACE("myorahome: $myorahome\n");
TRACE("emdroot: $emdroot\n");
TRACE("arglist: $arglist\n");
TRACE("corrkey: $corrkey\n");
TRACE("exadata: $isExadataCell\n");
TRACE("sshHostname: $sshHostname\n");
TRACE("sid: $sid\n");

# Determine the agent lib directory
$agentlibpath = $emdroot.$separator."jlib".$separator;
TRACE("agentlibpath: $agentlibpath\n");

# Determine the JAVA_HOME and the Java binary location
$javahome = $myorahome.$separator."jdk".$separator."jre";
$javabinary = $javahome.$separator."bin".$separator."java";
TRACE("javahome: $javahome\n");
TRACE("javabinary: $javabinary\n");

$ENV{JAVA_HOME} = $javahome;

# Determine uidrvci location if not set
# Also determine location of uidrvci dependent libraries
if ($uidrvhome) {
  if (!$uidrvlib) {
    $uidrvlib = $uidrvhome;
  }
} else {
  $uidrvhome = File::Spec->catdir($orahome, "bin");
  $uidrvlib = File::Spec->catdir($orahome, "lib");
}

# Determine uidrvci & ADR location on Exadata cell
if ($isExadataCell) {
  $uidrvhome = $ENV{OSS_BIN};

  $target_adrbase = $ENV{"ADR_BASE"};
  $target_adrhome = 
    File::Spec->catdir($target_adrbase, "diag", "asm", "cell", hostname());
}

TRACE("uidrvhome: $uidrvhome\n");
TRACE("uidrvlib: $uidrvlib\n");
TRACE("target adrbase: $target_adrbase\n");
TRACE("target adrhome: $target_adrhome\n");

@results = ();

if (!$isExadataCell) {
  if ($NT) {
    my $ldpath = $ENV{'PATH'};
    $ldpath = "$uidrvhome;$uidrvlib;$ldpath";
    $ENV{'PATH'} = $ldpath;
  } else {
    my $libPathName = getLibPathName();
    my $currentLibPath = $ENV{$libPathName};
    my $newLibPath = "$uidrvlib:$currentLibPath";
    $ENV{$libPathName} = $newLibPath;
    TRACE("newLibPath: $newLibPath\n");

    # Modify LD_LIBRARY_PATH_64. Don't rely on this being set, since a sudo
    # environment might clear it
    my $libPath64 = $ENV{'LD_LIBRARY_PATH_64'};
    my $newLibPath64 = "$uidrvlib:$libPath64";
    $ENV{'LD_LIBRARY_PATH_64'} = $newLibPath64;
    TRACE("newLibPath64: $newLibPath64\n");

    my $path = $ENV{'PATH'};
    $path = "$uidrvhome:$path";
    $ENV{'PATH'} = $path;
    TRACE("path: $path\n");
  }
}

$arglist =~ s/&#061/=/g;
$arglist =~ s/&#044/,/g;
$arglist =~ s/&#034/"/g;
$arglist =~ s/&#039/'/g;

invokeCmd;

exit;

