#!/usr/local/bin/perl
# 
# $Header: emagent/sysman/admin/scripts/ccs/ecmCcsSyncFile.pl /main/4 2012/06/04 03:21:51 sjshaw Exp $
#
# ecmCcsSyncFile.pl
# 
# Copyright (c) 2011, 2012, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      ecmCcsSyncFile.pl - Base script for prepartion of CCS File Sync operation
#
#    DESCRIPTION
#      <short description of component this file declares/defines>
#
#    NOTES
#      <other useful comments, qualifications, etc.>
#
#    MODIFIED   (MM/DD/YY)
#    sjshaw      05/07/12 - fixed source path for backup (copy file from orig dest path)
#    skundalk    03/30/12 - making use of use emdcommon_ocm; for using
#                           emdcommon_ocm::EMD_PERL_DEBUG
#    skundalk    03/23/12 - XbranchMerge skundalk_sync_minor from
#                           st_emgc_pt-12.1.0.2
#    skundalk    03/14/12 - adding comments
#    skundalk    01/27/12 - updating directory check code
#    skundalk    12/27/11 - changes to detect conflicts
#    skundalk    11/10/11 - fixes
#    smirajka    09/29/11 - Creation
# 

package ecmCcsSyncFile;

require "ccs/ecmFileUtils.pl";

#emdcommon_ocm is required to use the debuging calls to EMD_PERL_DEBUG
# the file emd_common.pl has the implementation of EMD_PERL_DEBUG method but
# it cannot be used directly using 'require' in a perl package.
# PERL gets confused and the EMD_PERL_DEBUG call will not work as required.

use emdcommon_ocm;

use strict;
use warnings;
use Env;
use File::Spec;
use File::Copy;
use File::Path;
use Digest::MD5 qw( md5_hex );
use Encode;
binmode STDOUT,":utf8";

use constant READ                           => 'READ';
use constant NOREAD                         => 'NOREAD';
use constant WRITE                          => 'WRITE';
# when the directory does not have executable permissions the permission is NOWRITE.
use constant NOWRITE                        => 'NOWRITE';
use constant ABSENT                         => 'ABSENT';
use constant NOT_EMPTY                      => 'NOT_EMPTY'; 
# constant for value YES
use constant Y                              => 'Y';
# constant for value NO
use constant N                              => 'N';

use constant DIR_SEPERATOR                  => '/';
use constant TYPE                           => 'F';
use constant EXPR                           => 'EXPR';
use constant SOC                            => 'SOC';
use constant OMS_CHKSUM                     => 'OMS_CHKSUM';
# global flag if set as "Y" missing directories will be created."
use constant CREATE_DEST_DIR                => 'CREATE_DEST_DIR';
use constant BACKUP_DIR_OK                  => 'BACKUP_DIR_OK';
 
# Constant with value as "Y" or "N" if the file not readable,
#either becuase the file does not have READ permission or the 
#directory does not have read permission beause of which file
#cannot be read.
use constant DEST_FILE_READ_CONFLICT        => 'DEST_FILE_READ_CONFLICT_';

use constant OMS_BASE_PATH                  => 'OMS_BASE_PATH';
use constant OMS_ENCODING                   => 'OMS_ENCODING';
use constant DEST_DIR_STATUS_FOR            => 'DEST_DIR_STATUS_FOR_';
use constant DEST_FILE_STATUS_FOR           => 'DEST_FILE_STATUS_FOR_';
use constant DEST_FILE_CHECKSUM_FOR         => 'DEST_FILE_CHECKSUM_FOR_';
use constant DEST_FILE_CHECKSUM_MATCH       => 'DEST_FILE_CHECKSUM_MATCH_';
use constant NEW_DEST_DIR_CREATED_FOR_      => 'NEW_DEST_DIR_CREATED_FOR_';
use constant NEW_BACKUP_DIR_CREATED_FOR_    => 'NEW_BACKUP_DIR_CREATED_FOR_';
use constant ERROR                          => 'ERROR';
use constant DIE_MSG                        => 'DIE_MSG';

#####################################################################################
#   Main function
#####################################################################################
emdcommon_ocm::EMD_PERL_DEBUG("################## File synchronization ######################");

# Assign passed parameters to corrosponding variables
my ($basePath, $encoding, $origBasePath, $backupDir, $ccsDef) = @ARGV;

# Define result hash
my %resultHash;
  
# If encoding is not defined then set it to null
if (! defined ($encoding))
{
    $encoding = "";
}

# if backup dir is "-" then set it to empty string.
if (! defined ($backupDir) || $backupDir eq "-") 
{
    $backupDir = "";
}

# Initialize result hash
$resultHash{OMS_BASE_PATH}     = $basePath;
$resultHash{OMS_ENCODING}      = $encoding;


##### Main program execution
#if only one argument is passed then that will be assumed as the back base path and will be
# checked for backup ready. 

if (scalar(@ARGV) == 2) {
   # basePath will contain the directory path to check
   # encoding will contain value for checking if the directory is WRITE and empty.
   # encoding value 1= check WRITE permissions only.
   # encoding value 2= check WRITE permission and directory is empty.
   
   
   emdcommon_ocm::EMD_PERL_DEBUG("Passed Arguments are: basePath= $basePath. Checking backup directory.");
   eval {
      &checkBackupDir($basePath, $encoding);      
   };
   if ($@)
   {
       emdcommon_ocm::EMD_PERL_DEBUG("checkBackupDir() function failed, cause: $@");
       $resultHash{DIE_MSG} = $@;
   }
}
else {
   emdcommon_ocm::EMD_PERL_DEBUG("Passed Arguments are: basePath= $basePath, Encoding= $encoding, origBasePath= $origBasePath, BackupDir= $backupDir, CCSDEF= $ccsDef");
   # Call prepare() function 
   eval 
   { 
      my $goAhead = 1;
      
      if ($backupDir ne "") {
         # check if backup dir is writable (empty backup dir is permitted)
         my $backupDirStatus = &checkBackupDir($backupDir, 1);
         
         # absence is permitted, but existing dir must be writable
         if (($backupDirStatus ne ABSENT) && ($backupDirStatus ne WRITE)) {
            $goAhead = 0;
         }      
      }

      # if alternative destination dir is specified, check and create as necessary
      if ($basePath ne $origBasePath) {
         &createDirectory($basePath);
      }
      
      if ($goAhead == 1) {
         &prepare($basePath, $encoding, $ccsDef);
      
         &checkAndCreateDir($basePath, $backupDir, $ccsDef);
      }
   };
   if ($@)
   {
       emdcommon_ocm::EMD_PERL_DEBUG("Failure in script, cause: $@");
       $resultHash{DIE_MSG} = $@;
   }
   
}
    

# Print result Hash on STDOUT sorted by Key name
# format is: <key name> = <value>
foreach my $key (sort (keys %resultHash))
{
    print "$key" . " = " . "$resultHash{$key}\n";
}


#####################################################################################
#   End of Main function
#####################################################################################

#---------------------------------------------------------------------------------------------------------
# FUNCTION : prepare($basePath, $encoding, $ccsDef)
#if ($backupDir ne "")
# DESC     :Subroutine for prepartion of File Sync.
#           It checks directory presence.
#           It checks presence of File. If present calculates checksum. 
# ARGS     :
#           $basePath       - Base path as provided in CCS
#           $encoding       - CCS Encoding
#           $ccsDef         - String containing preformatted CCS definition (eg. F=a/b/c.txt, SOC=Y, CHKSUM=1as34|F=e/f.txt, SOC=Y, CHKSUM=8sh67)
# Returns   :
#---------------------------------------------------------------------------------------------------------

sub prepare
{

    # Assign passed parameters to corrosponding variables
    my ($basePath, $encoding, $ccsDef) = @_;
    
    # Get CCS definition Array Reference from passed definition string
    my $ccsDef_ref = ecmCcsUtils::getDefinitions($ccsDef);
    
    # Create definitions array (containing hash elements)
    my @defArray = @$ccsDef_ref;
    
    #if conflict found, stop processing after all the files are checked for conflicts.
    my $conflict = N;
    #stop processing if the flag value is Y
    my $socFlag = N;
    
    # Iterate for each element which is a hash ref (F=a/b/c.txt, SOC=Y, CHKSUM=1as34) of definitions array
    foreach my $currentDef_ref (@defArray)
    {
        # Recreate hash containing individual definition details
        my %currentDef  = %$currentDef_ref;
        
        my $filePath    = $currentDef{EXPR};
        my $soc         = $currentDef{SOC};
        my $oms_chk_sum = $currentDef{OMS_CHKSUM};

        # Get Directory status and populate result hash
        my $directoryStatus = &getDirectoryStatus($basePath, $filePath);
        
        $resultHash{SOC."_".$filePath} = $soc;
        $resultHash{OMS_CHKSUM."_".$filePath} = $oms_chk_sum;
        
                
        $resultHash{DEST_DIR_STATUS_FOR.$filePath} = $directoryStatus;
    
        # Set default value for DEST_FILE_CONFLICT
        $resultHash{DEST_FILE_READ_CONFLICT.$filePath} = N;
            
        # If directory has write permissions then get file status
        if($directoryStatus eq WRITE)
        {
            emdcommon_ocm::EMD_PERL_DEBUG("Destination Directory for $filePath, has Write permissions");
            
            # Get File Status and populate result hash
            my $fileStatus = &getFileStatus($basePath, $filePath);
            $resultHash{DEST_FILE_STATUS_FOR.$filePath} = $fileStatus;
            
            # If file has write permission, then get file checksum 
            if($fileStatus eq WRITE)
            {
		# Always run checksum against original file destination path
                my $fileChecksum = &getFileChecksum($origBasePath, $filePath, $encoding);
                # Populate result hash                    
                $resultHash{DEST_FILE_CHECKSUM_FOR.$filePath} = $fileChecksum;
                
                # Compare OMS file checksum with actual file checksum
                if ($oms_chk_sum eq $fileChecksum)
                {
                	# If both checksum match then populate result hash for corresponding file path as 'YES'
                	$resultHash{DEST_FILE_CHECKSUM_MATCH.$filePath} = Y;
                }
                else
                {
                	# If both checksum does NOT match then populate result hash for corresponding file path as 'NO'
                	$resultHash{DEST_FILE_CHECKSUM_MATCH.$filePath} = N;
                	if ($soc eq Y) {
                	   $socFlag = Y;
                	}
                }
            }
            elsif ($fileStatus eq ABSENT)
            {
            	# Set to no conflict
                $resultHash{DEST_FILE_READ_CONFLICT.$filePath} = N;
                if ($soc eq Y) {
                     $socFlag = N;
                }
            }
            elsif ($fileStatus eq READ)
            {
            	$resultHash{DEST_FILE_READ_CONFLICT.$filePath} = Y;
            }
        }
        # IF directory is absent 
        elsif ($directoryStatus eq ABSENT)
        {
            # Set to no conflict
            $resultHash{DEST_FILE_READ_CONFLICT.$filePath} = N;
        }
        # If directory has only read permissions
        elsif(($directoryStatus eq READ))
        {
            $resultHash{DEST_FILE_READ_CONFLICT.$filePath} = Y;
        }
        
        #set the conflict flag if there were conflict
        if ($resultHash{DEST_FILE_READ_CONFLICT.$filePath} eq Y)
        {
           $conflict = Y;
        }
   }
   
   #check if stop on conflict is required.
   if ($socFlag eq Y)
   {
      my $msg = "Stopped processing because stop on conflict flag was raised.";
      $resultHash{ERROR} = $msg;
            
      die "$msg";
   }
   
   #check if stop on conflict is required.
   if ($conflict eq Y)
   {
      my $msg = "Stopped processing because file/s were not readable.";
      $resultHash{ERROR} = $msg;
            
      die "$msg";
   }
}



# TODO Rewrite Declaration comments
#---------------------------------------------------------------------------------------------------------
# FUNCTION : checkAndCreateDir($basePath, $backupDir, $ccsDef)
#
# DESC     :Subroutine for checking status of $resultHash{DEST_FILE_CONFLICT.$filepath}. If it is YES then dont create directories.
#           If it is NO then allow creation of directories if they are absent.
# ARGS     :
#           $basePath       - Base path as provided in CCS
#           $backupDir      - Backup directory path
#           $ccsDef         - String containing preformatted CCS definition (eg. F=a/b/c.txt, SOC=Y, CHKSUM=1as34|F=e/f.txt, SOC=Y, CHKSUM=8sh67)
# Returns   :
#---------------------------------------------------------------------------------------------------------

sub checkAndCreateDir
{

    # Assign passed parameters to corrosponding variables
    my ($basePath, $backupDir, $ccsDef) = @_;
    # Get CCS definition Array Reference from passed definition string
    my $ccsDef_ref = ecmCcsUtils::getDefinitions($ccsDef);
    
    # Create definitions array (containing hash elements)
    my @defArray = @$ccsDef_ref;
    
    # Set default by marking the directory creation operation as go
    $resultHash{CREATE_DEST_DIR} = Y;
    
#    # Iterate for each element which is a hash ref (F=a/b/c.txt, SOC=Y, CHKSUM=1as34) of definitions array
#    # to find file conflict flag and update go no go for CREATE_DEST_DIR
#    foreach my $currentDef_ref (@defArray)
#    {
#        # Recreate hash containing individual definition details
#        my %currentDef  = %$currentDef_ref;
#        my $soc         = $currentDef{SOC};
#        my $filePath    = $currentDef{EXPR};
#        
#        # If $resultHash{DEST_FILE_CONFLICT.$filePath} = YES; AND StopOnConflict is Y then mark it as NO GO and exit out of loop
#        if (($resultHash{DEST_FILE_CONFLICT.$filePath} eq Y) && ($soc eq Y ) )
#        {
#            # Mark the directory creation operation as no go
#            $resultHash{CREATE_DEST_DIR} = N;
#        	last;
#        }
#    }
    
    
    # Set default value so that if backup directory is not specified  the processing
    # will still continue.
    $resultHash{BACKUP_DIR_OK} = Y;
    
    # Create backup directory only if there is no conflict for destination file/directory && backup directory is provided by user
    if ($backupDir ne "")
    {
       # check if the backup directory is writable (empty dir is permitted)
       my $bckDirStatus = &checkBackupDir($backupDir, 1);
       
       # directory exists and is writable, or is absent and will be created
       if (($bckDirStatus eq ABSENT) || ($bckDirStatus eq WRITE)) {
       
          #checkBackupDir(..) updates the following flag so resetting it.
          $resultHash{BACKUP_DIR_OK} = Y;
          
          # Iterate for each element which is a hash ref (F=a/b/c.txt, SOC=Y, CHKSUM=1as34) of definitions array
          foreach my $currentDef_ref (@defArray)
          {
              # Recreate hash containing individual definition details
              my %currentDef  = %$currentDef_ref;
              
              my $filePath    = $currentDef{EXPR};
              
              # Create backup Directory
              my $directoryStatus = &createDirectory($backupDir, $filePath);
              
              if ($directoryStatus eq WRITE)
              {
                 $resultHash{NEW_BACKUP_DIR_CREATED_FOR_.$filePath}  = Y;
              }
              # else set result to false and exit
              else
              {
                 $resultHash{NEW_BACKUP_DIR_CREATED_FOR_.$filePath}  = N;
                 $resultHash{BACKUP_DIR_OK} = N;
                 last;
              }
              
              # Copy destination file to backup directory if BACKUP_DIR_OK is YES
              if ($resultHash{BACKUP_DIR_OK} eq Y)
              {
                  my $cleanBasePath      = ecmFileUtils::removeLastSlash($basePath);
                  my $cleanOrigBasePath  = ecmFileUtils::removeLastSlash($origBasePath);
                  my $cleanBackupDir     = ecmFileUtils::removeLastSlash($backupDir);
                  my $cleanFilePath      = ecmFileUtils::removeFirstSlash($filePath);
                 
                  # Form destination and backup filepath
                  my $destFilePath   = $cleanOrigBasePath  . DIR_SEPERATOR . $cleanFilePath;
                  my $backupFilepath = $cleanBackupDir . DIR_SEPERATOR . $cleanFilePath ;
                  emdcommon_ocm::EMD_PERL_DEBUG("Using Destination File Path: $destFilePath,  Backup File Path: $backupFilepath");
                  
                  if (-e $destFilePath) {
                     # Copy file from destination to backup directory
                     if (!copy ($destFilePath, $backupFilepath)) {
                        $resultHash{ERROR} = "Copy file to backup directory failed for $filePath";
                        die "Copy file to backup directory failed for $filePath : $!"; 
                     } 
              
                     #  TODO:: write checksum verification for copied backup file
                  }
                  else {
                     emdcommon_ocm::EMD_PERL_DEBUG("Destination File Path: $destFilePath does not exist. Skiping backup for this file.");
                  }
              }
          }    
       }
       else {
          $resultHash{BACKUP_DIR_OK} = $bckDirStatus;
          $resultHash{ERROR} = "Backup directory check failed. Directory status: $bckDirStatus";
       }
    }   # if loop end
    
    
    # Create destination directory only if there is no conflict for any destination/backup file or directory
    if (($resultHash{CREATE_DEST_DIR} eq Y) && ($resultHash{BACKUP_DIR_OK} eq Y))
    {
        # Iterate for each element which is a hash ref (F=a/b/c.txt, SOC=Y, CHKSUM=1as34) of definitions array
        foreach my $currentDef_ref (@defArray)
        {
            # Recreate hash containing individual definition details
            my %currentDef  = %$currentDef_ref;
            my $filePath    = $currentDef{EXPR};
            
            # Get destination directory status
            my $directoryStatus = &getDirectoryStatus($basePath, $filePath);
            
            # If directory is absent then create it
            if ($directoryStatus eq ABSENT)
            {
                # Create destination Directory
                my $newDestDirStatus = &createDirectory($basePath, $filePath);
                
                # Set result hash
                if ($newDestDirStatus eq WRITE)
                {
                	$resultHash{NEW_DEST_DIR_CREATED_FOR_.$filePath}  = Y;
                }
                # else set result to false and exit
                else
                {
                	$resultHash{NEW_DEST_DIR_CREATED_FOR_.$filePath}  = N;
                    last;
                }
            }
        }
    }
}



#---------------------------------------------------------------------------------------------------------
# FUNCTION : getDirectoryStatus($basePath, $filePath)
#
# DESC     :Subroutine for checking directory present.
#           This function creates complete directory path by appending base path and relative path (excluding file name)
#           It verifies presence of destination directory. 
# ARGS     :
#           $basePath       - Base path as provided in CCS
#           $filePath       - File relative path as provided in CCS
# Returns   :
#           $dirStatus      - Directory status read/noread/write/absent
#---------------------------------------------pushFile------------------------------------------------------------

sub getDirectoryStatus
{
    # Assign passed parameters to corrosponding variables
    my ($basePath, $filePath) = @_;

    $basePath = ecmFileUtils::removeLastSlash($basePath);
    $filePath = ecmFileUtils::removeFirstSlash($filePath);

    emdcommon_ocm::EMD_PERL_DEBUG("Using basePath: $basePath  filePath: $filePath");
    
    my $dirPath = $basePath;
    
    if ($filePath ne "") {
      # Join base path and file path to form destination path to be checked
      #  my $destinationDir = $basePath . DIR_SEPERATOR . $filePath;
      my $destinationDir = File::Spec->catpath("", $basePath, $filePath);
      
      emdcommon_ocm::EMD_PERL_DEBUG("Using destination directory: $destinationDir");

      # Split destination path 
      my ($volume, $directories, $file) = File::Spec->splitpath($destinationDir);
          
      # Concat volume and directories
      $dirPath = $volume . $directories; 
    } 
    
    my $dirStatus;
    # Check if destination directory exists and has write permission
    if((-d $dirPath) && (-w $dirPath) && (-r $dirPath) && (-x $dirPath))
    {
        emdcommon_ocm::EMD_PERL_DEBUG("Destination directory: $dirPath exists, has RWX permission.");
        $dirStatus = WRITE;
    }
    elsif ((-d $dirPath) && (-w $dirPath)) {
        emdcommon_ocm::EMD_PERL_DEBUG("Destination directory: $dirPath exists, has Write permission but cannot be accessed");
        $dirStatus = NOWRITE;
    }
    elsif((-d $dirPath) && (-r $dirPath))
    {
        emdcommon_ocm::EMD_PERL_DEBUG("Destination directory: $dirPath exists and has Read permission");
        $dirStatus = READ;
    }
    elsif((-d $dirPath))
    {
        emdcommon_ocm::EMD_PERL_DEBUG("Destination directory: $dirPath exists but not accessible");
        $dirStatus = NOREAD;
    }
    else
    {
        emdcommon_ocm::EMD_PERL_DEBUG("Destination directory: $dirPath does NOT exist");
        $dirStatus = ABSENT;
    }

    # Return directory status
    emdcommon_ocm::EMD_PERL_DEBUG("Directory status: $dirStatus");
    return $dirStatus;    
}

#---------------------------------------------------------------------------------------------------------
# FUNCTION : checkBackupDir($dirPath, $dirCheck)
#pushFile
# DESC     : Check if the backup directory is empty and is writable.
# 
# ARGS     :
#           $dirPath - backup directory path
#           $dirCheck - What to check for the directory. 
#               value  1= Only check WRITE permission
#               value  2= Check WRITE permission and if directory is empty.             
#           
# Returns   :
#           $DirStatus     - Directory status notEmpty/read/noread/write/absent
#---------------------------------------------------------------------------------------------------------

sub checkBackupDir
{
   my ($dirPath, $dirCheck) = @_;
   my $dirStatus = getDirectoryStatus($dirPath, "");
   
   if ($dirStatus eq WRITE && $dirCheck == 2) {
      if (opendir(my $dirHandle, $dirPath)) { 
         my @files = readdir($dirHandle);
         my $cnt = scalar(@files); 
      
         if (($cnt <= 2) && ($cnt != 0)) {
            foreach (@files) {
               if ($_ ne "." && $_ ne "..") {
                  $dirStatus = NOT_EMPTY;
                  $resultHash{ERROR} = "Backup directory $dirPath not empty.";
                  last;
               }
            } 
         }
         elsif ($cnt > 2) {
            $dirStatus = NOT_EMPTY;
            $resultHash{ERROR} = "Backup directory $dirPath not empty.";
         }
         
         closedir $dirHandle;
      }
      else {
         $resultHash{ERROR} = "Can not open directory $dirPath.";
         die "Can not open directory $dirPath: $!";
      }
   }
   # absence is allowed, but if present, dir must be writable
   elsif (($dirStatus ne ABSENT) && ($dirStatus ne WRITE)) {
      $resultHash{ERROR} = "Backup directory $dirPath not accessible.";
   }
   
   $resultHash{BACKUP_DIR_OK} = $dirStatus;
   
   return $dirStatus
}


#---------------------------------------------------------------------------------------------------------
# FUNCTION : getFileStatus($basePath, $filePath)
#
# DESC     :Subroutine for checking file present.
#           This function creates complete file path by appending base path and relative path (including file name)
#           It verifies presence of file. 
# ARGS     :
#           $basePath       - Base path as provided in CCS
#           $filePath       - File relative path as provided in CCS
# Returns   :
#           $fileStatus     - Directory status read/noread/write/absent
#---------------------------------------------------------------------------------------------------------

sub getFileStatus
{
    # Assign passed parameters to corrosponding variables
    my ($basePath, $filePath) = @_;

    $basePath = ecmFileUtils::removeLastSlash($basePath);
    $filePath = ecmFileUtils::removeFirstSlash($filePath);

    emdcommon_ocm::EMD_PERL_DEBUG("Using basePath: $basePath  filePath: $filePath");

    # Join base path and file path to form complete file path to be checked
    my $completeFilePath = $basePath . DIR_SEPERATOR . $filePath;
    emdcommon_ocm::EMD_PERL_DEBUG("Using file path: $completeFilePath");

    my $fileStatus;
    # Check if file exists and has write permission
    if((-e $completeFilePath) && (-w $completeFilePath) && (-r $completeFilePath))
    {
        emdcommon_ocm::EMD_PERL_DEBUG("File: $completeFilePath exists and has Write permission");
        $fileStatus = WRITE;
    }
    elsif((-e $completeFilePath) && (-r $completeFilePath))
    {
        emdcommon_ocm::EMD_PERL_DEBUG("File: $completeFilePath exists and has Read permission");
        $fileStatus = READ;
    }
    elsif(-e $completeFilePath) 
    {
        emdcommon_ocm::EMD_PERL_DEBUG("File: $completeFilePath exists but not accessible");
        $fileStatus = NOREAD;
    }
    else
    {
        emdcommon_ocm::EMD_PERL_DEBUG("File: $completeFilePath does NOT exist");
        $fileStatus = ABSENT;
    }
    # Return file status
    emdcommon_ocm::EMD_PERL_DEBUG("File status: $fileStatus");
    return $fileStatus;    
}


#---------------------------------------------------------------------------------------------------------
# FUNCTION : getFileChecksum($basePath, $filePath, $encoding)
#
# DESC     :Subroutine for calculating file hash.
#           This function creates complete file path by appending base path and relative path (including file name)
#           It verifies presence of file. 
# ARGS     :
#           $basePath       - Base path as provided in CCS
#           $filePath       - File relative path as provided in CCS
#           $encoding       - Encoding format as stored in OMS
# Returns   :
#           $md5            - File Checksum
#---------------------------------------------------------------------------------------------------------

sub getFileChecksum
{
    # Assign passed parameters to corrosponding variables
    my ($basePath, $filePath, $encoding) = @_;

    $basePath = ecmFileUtils::removeLastSlash($basePath);
    $filePath = ecmFileUtils::removeFirstSlash($filePath);

    emdcommon_ocm::EMD_PERL_DEBUG("Using basePath: $basePath  filePath: $filePath");

    # Join base path and file path to form complete file path to be checked
    my $completeFilePath = $basePath . DIR_SEPERATOR . $filePath;
    emdcommon_ocm::EMD_PERL_DEBUG("Using complete file path: $completeFilePath");

    my $INPUT_FILE;
    my $fileData = "";
    my $md5      = "";
    my $filesize = "";
                    
    # Read file and calculate md5 checksum
    eval 
    {
        $encoding = ecmFileUtils::getValidEncoding($encoding, $filePath);
        
        if (defined($encoding) && $encoding ne "") 
        {
            open ($INPUT_FILE, "<:encoding($encoding)", $completeFilePath);  
        }
        else
        {
            open ($INPUT_FILE, "< $completeFilePath");
        }
                        
        my $currentLineTerminator = $/;
        undef $/;
        $fileData = <$INPUT_FILE>;
        Encode::_utf8_on($fileData);
        $/ = $currentLineTerminator;
        close($INPUT_FILE);

        # Get md5 checksum of file
        $md5 = md5_hex(encode_utf8($fileData));
    };    
    if ($@)
    {
        emdcommon_ocm::EMD_PERL_DEBUG("File Read failed: $@");
        my $errorMsg = "Failed to evaluate checksum for $filePath. $@";
        $resultHash{ERROR.$filePath} = $errorMsg;
        die $errorMsg;
    }
    
    # Return file checksum
    emdcommon_ocm::EMD_PERL_DEBUG("File checksum: $md5");
    return $md5;
}


#---------------------------------------------------------------------------------------------------------
# FUNCTION : createDirectory($basePath, $filePath)
#
# DESC     :Subroutine for creating new directory.
#           This function creates new directory as given in base path and file path (excluding file name)
#           It verifies presence directory got created.
#           This function should be called only after determining directory is absent via getDirectoryStatus() 
# ARGS     :
#           $basePath       - Base path as provided in CCS
#           $filePath       - File relative path as provided in CCS
# Returns   :
#           $dirStatus      - Status of directory created read/write/absent
#---------------------------------------------------------------------------------------------------------

sub createDirectory
{
    # Assign passed parameters to corrosponding variables
    my ($basePath, $filePath) = @_;
    $basePath = ecmFileUtils::removeLastSlash($basePath);
    $filePath = ecmFileUtils::removeFirstSlash($filePath);

    emdcommon_ocm::EMD_PERL_DEBUG("Using basePath: $basePath  filePath: $filePath");

    # Join base path and file path to form destination path to be checked
    my $destinationDir = $basePath . DIR_SEPERATOR . $filePath;
    emdcommon_ocm::EMD_PERL_DEBUG("Using destination directory: $destinationDir");

    # Split destination path 
    my ($volume, $directories, $file) = File::Spec->splitpath($destinationDir);

    # Concat volume and directories
    my $dirPath = $volume . $directories;
    
    # Get Directory status
    my $directoryStatus = &getDirectoryStatus($basePath, $filePath);
    
    # Create new directory only if it is absent
    if ($directoryStatus eq ABSENT)
    {
    	eval
        {
            mkpath($dirPath);
            emdcommon_ocm::EMD_PERL_DEBUG("NEW_DIR_CREATED_FOR_.$filePath");
            
            # Get Directory status for newly created directory
            $directoryStatus = &getDirectoryStatus($basePath, $filePath);
        };
        # Throw error if eval fails
        if ($@)
        {
            emdcommon_ocm::EMD_PERL_DEBUG("Directory creation failed: $@");
            my $errorMsg = "Failed to create directory for $filePath.";
            $resultHash{ERROR} = $errorMsg;
            die $errorMsg;
        }
    }
    
            
#    # Create new directory only if it is absent
#    if (!(-d $dirPath))
#    {
#        eval
#        {
#            mkpath($dirPath);
#            emdcommon_ocm::EMD_PERL_DEBUG("NEW_DIR_CREATED_FOR_.$filePath");
#        };
#        # Throw error if eval fails
#        if ($@)
#        {
#            emdcommon_ocm::EMD_PERL_DEBUG("Directory creation failed: $@");
#            my $errorMsg = "Failed to create directory for $filePath. $@";
#            $resultHash{ERROR.$filePath} = $errorMsg;
#            die $errorMsg;
#        }	
#    }
#   
#    
#    # Verify new directory got created with correct permissions
#    my $dirStatus;
#    if((-d $dirPath) && (-w $dirPath))
#    {
#        emdcommon_ocm::EMD_PERL_DEBUG("Destination directory: $dirPath exists and has Write permission");
#        $dirStatus = WRITE;
#    }
#    elsif((-d $dirPath) && (-r $dirPath))
#    {
#        emdcommon_ocm::EMD_PERL_DEBUG("Destination directory: $dirPath exists and has Read permission");
#        $dirStatus = READ;
#    }
#    else
#    {
#        emdcommon_ocm::EMD_PERL_DEBUG("Destination directory: $dirPath does NOT exist");
#        $dirStatus = ABSENT;
#    }

    # Return directory status
    emdcommon_ocm::EMD_PERL_DEBUG("Directory status: $directoryStatus");
    return $directoryStatus;   
}


1;
