# $Header: emagent/sysman/admin/scripts/ccs/ecmCcsUtils.pl /main/10 2012/07/23 11:08:03 bdknowlt Exp $
#
# Copyright (c) 2010, 2012, Oracle and/or its affiliates. All rights reserved. 
#
#    DESCRIPTION
#      Utility methods for ccs collection. Contains method for collecting
# data from files, OS commands and database queries. 
#
#    MODIFIED   (MM/DD/YY)
#       bdknowlt 07/13/12 - adding raw file reads and STDOUT writes to support
#                           any file style on any platform
#       skundalk 03/26/12 - using agent debug methods along with ccs specific
#                           logging
#       skundalk 03/07/11 - adding error printing method
#       skundalk 02/17/11 - making changes to use err filters on metrics
#       skundalk 01/12/11 - adding some more debugging and error handling.
#       skundalk 12/10/10 - adding documentation
#       skundalk 11/12/10 - removing unnecessary columns from ColumnDescriptor
#                           element
#       skundalk 10/21/10 - fix for 10206413. Avoid duplicate file load
#       skundalk 10/05/10 - encoding and database target type changes
#       skundalk 09/09/10 - enable target property substitutio code and
#                           updating script include path
#       skundalk 06/25/10 - ccs script files
#      
##*********************************************************************
package ecmCcsUtils;

use Digest::MD5 qw( md5_hex );
use Encode;
use IPC::Cmd qw(run);
use DBI;

# the working directory is scripts for perl execution.
require "ccs/ecmFileUtils.pl";


#use FindBin qw($Bin);
#use lib $Bin;

#emdcommon_ocm is required to use the debuging calls to EMD_PERL_DEBUG
# the file emd_common.pl has the implementation of EMD_PERL_DEBUG method but
# it cannot be used directly using 'require' in a perl package.
# PERL gets confused and the EMD_PERL_DEBUG call will not work as required.

use emdcommon_ocm;

use strict;

# if the value of CCSLOG variable is 1 the debugging information
# will be logged to ecmCCS.log file in the directory specified
# in the variable TMPDIR
my $debug = $ENV{'CCSLOG'};
my $TMPDIR = $ENV{'TMPDIR'};

# characters used in ccs string representation.
my @RESERVED_CHARS = (",","|", "=");

my $UNDER_SCORE = "_";
my $PIPE = "|";
    
my $DEFAULT_COMMAND_TIMEOUT = 60;
my $DEFAULT_LONG_READ_LENGTH = 80;

# global variable to keep track of expressions getting evaluated.
my $expressionOrder =0;
    
# expression type    
my $TYPE_FILE = "F";
my $TYPE_CMD = "O";
my $TYPE_DB = "D";
    
# type of expression F for file, C for command and D for database    
my $KEY_TYPE = "TYPE";
# expression for the ccs item.
my $KEY_EXPRESSION = "EXPR";
# encoding for the files.
my $KEY_ENCODING = "ENCODING";
# name for the expression
my $KEY_NAME = "NAME";

# ignore command output. Used in case of Pre and Post commands. 
my $KEY_IGNORE_OUTPUT = "IGNORE_OUTPUT"; 

my $KEY_COMMAND_TIMEOUT = "COMMAND_TIMEOUT";

my $KEY_LONG_READ_LENGTH = "LONG_READ_LENGTH";
my $ROW_HEADER = "[row]\n";

# Error string prefix.
my $ERROR_MSG_PREFIX = "ECM_CCS_ERROR";


#==============================================================================
# logDebugMsg(<message>)
#
# log the message to log file.
# Arguments:
#   message - message to be logged.
# Returns:
#   none
#==============================================================================
sub logDebugMsg($) {
    ecmCcsUtils::logDebug("ecmCcsUtils.pl", $_[0]);
    
    return;
}

#======================================================================
# logDebug(<logInitiator>,<msg>)
#
# Log a debug message to agent log file emagent_perl.trc. 
# If TMPDIR and CCSLOG environment variables are enabled then the ccs debug messages
# will be writen to ecmccs.log files under TMPDIR.

#NOTE: it is observed that the agent logs don't get enabled in some views and it is helpful
# to use the ccs specific logging.
#
# Arguments:
#  logInitiator - file name that is logging the message.
#  msg - message to log.
#======================================================================
sub logDebug($;$) {
    my ($initiatorFile, $msg) = @_;
    
    #logging message using agent methods.
    emdcommon_ocm::EMD_PERL_DEBUG("[$initiatorFile]:$msg");
    
    # following logic writes log messages to ccs specific log file
    # that can be created anywhere and only CCS logs can be seen in it.
    # enable this log if you if you see lot of noise and rollover in 
    # agent default log files.
    if (-e $TMPDIR) {
        my $logfile="$TMPDIR/ecmccs.log";
    
        if (1 == ($debug & 1)) {
            open(OUTPUT, ">> $logfile");

            # use binmode to prevent wide character in print error
            # if someone wants to log multi-byte data
            binmode(OUTPUT, ":encoding(UTF-8)");

            printf(OUTPUT "%s\n", localtime()." [DEBUG] File: [".$initiatorFile."] Message: ".$msg);
            close(OUTPUT);
        }
    }    

    return;
}

#======================================================================
# printError(msg)
#
# Log a error message to STDOUT. This error message will be consumed by
# the "errStartsWith" filter on the metrics.
# The message is prefixed with $ERROR_MSG_PREFIX
#
# Arguments:
#  errorMsg - error message to log to stdout..
#======================================================================
sub printError($) {
    my ($errorMsg) = @_;
    
    my $errorString = "";
    $errorString = $errorString.length($ERROR_MSG_PREFIX).$UNDER_SCORE.$ERROR_MSG_PREFIX.$PIPE;
    $errorString = $errorString.length($errorMsg).$UNDER_SCORE.$errorMsg.$PIPE;
       
    $errorString = $errorString."\n";
    
    logDebugMsg("$errorString");
    
    print STDOUT $errorString;
}

#==============================================================================
# PUBLIC <HASH> getDefinitions(<String:ccsExpressions>)
#
# method to convert the string representation of CCS expression into individual items.
# Arguments:
#   <String:ccsExpressions> String containing resource expressions seperated by "|" (pipe) and 
#   resource expression properties by "," (comma).
# Returns:
#   Array of Hash containing individual expressions
# example ({type=file, name="abc/config.xml"}, {type=cmd, name=ls -l}) 
#==============================================================================
my $g_data = "";
my $g_index = 0;
my $g_dataLength = 0;

my $g_defData = "";
my $g_defIndex = 0;
my $g_defLength = 0;

sub getDefinitions($) {
    $g_data = $_[0];
    $g_index = 0;
    $g_dataLength = length($g_data); 
    
    my $defDelimiter = "|";
    my $propertyDelimiter = "=";
    my $itemDelimiter = ",";
    
    #breaking the string representation in definition strings.
    my @strDefinitions = ();
    my $defStartIndex = 0;
    
    my @definitions = ();
    
    while ($g_index < $g_dataLength) {
        my $tmpChar = nextChar();
        logDebugMsg("tmpChar=[$tmpChar], g_index=$g_index, g_dataLength=$g_dataLength, defStartIndex=$defStartIndex");
        
        
        if ($tmpChar eq "\\") {
            nextChar(); # skip a character.
        }
        elsif (($tmpChar eq $defDelimiter) || $g_index == $g_dataLength) { # check for | or End of data.
           my $charCount;
           
           if ($g_index == $g_dataLength && ($tmpChar ne $defDelimiter)) {
               $charCount = $g_index - $defStartIndex;
           }
           else {
               $charCount = $g_index - 1 - $defStartIndex;               
           }
        
           my $defStr = substr($g_data, $defStartIndex, ($charCount));
           
           push (@strDefinitions, $defStr);
           $defStartIndex = $g_index;
        }
    }
    
    foreach my $currentDef (@strDefinitions) {
        logDebugMsg("Seperating properties from definition $currentDef");
        
        $g_defIndex = 0;
        $g_defData = $currentDef;
        $g_defLength = length($g_defData);
        
        my $defOk = 0;
        my %defProperties = ();
        
        while ($g_defIndex < $g_defLength) {
              
            my $propertyName = getStringTill($propertyDelimiter, 0);
            my $propetyValue = getStringTill($itemDelimiter, 1);
            
            if ($propertyName eq $TYPE_FILE || $propertyName eq $TYPE_CMD || $propertyName eq $TYPE_DB) {
                $defOk = 1;
                $defProperties{$KEY_TYPE} = $propertyName;
                $defProperties{$KEY_EXPRESSION} = $propetyValue;
            }
            else {
                $defProperties{$propertyName} = $propetyValue;
            }
        }
        
        if ($defOk == 0) {
            die "ERROR: CCS string representation cannot be parsed. Expression type not found in $currentDef";
        }
        
        push (@definitions, \%defProperties);
    } 
    
    
    return \@definitions;
}



#==============================================================================
# nextChar()
#
# Get the next character after the current index. The method returns next 
# character from the string stored in variable $g_data.
#
# Arguments:
#   none
# Returns:
#  If the data index is less than the data length then next character will be returned.
# else a empty string will be returned. 
#==============================================================================
sub nextChar() {
    my $character = "";
    if ($g_index < $g_dataLength) {
        $character = substr($g_data, $g_index, 1);
        $g_index++;
    }
    
    return $character;
}


#==============================================================================
# <String:subString> getStringTill(<String:searchChar>, <Number:criteria>)
#
# Get the characters till the indicated character. The search character is not
# included in the return String.

# Note: Method searches $g_defData string data for the seach character.
#
# Arguments:
#   <String:searchChar> - String representing the search character.
#   <Number:criteria> - If 0 the search character is needed to stop searching, if not found the script will DIE.
#   if 1 the string from StartIndex to end will be returned if the search character is not found. 
# Returns:
#   <String:subString> - sub string till the character index found in the string.
#==============================================================================
sub getStringTill($;$) {
    my $endChar = $_[0];
    my $isOptional = $_[1];
    
    my $continue = 1;
    my $startIndex = $g_defIndex;
    my $result = "";
    
    while (($g_defIndex < $g_defLength) && ($continue == 1)) {
        my $tempChar = nextDefChar();
         
        if ($tempChar eq "\\") {
           # escape character.
           nextDefChar();
        }    
        elsif ($tempChar eq $endChar) {
            $continue = 0;
            $result = substr($g_defData, $startIndex, ($g_defIndex - 1 - $startIndex));
        }    
     }
     
     if ($continue == 1 && $isOptional == 1) {
         $result = substr($g_defData, $startIndex, ($g_defLength - $startIndex));
         logDebugMsg("String $result collected till EOL.");
     }
     elsif ($continue == 1) {
         logDebugMsg("ERROR: Property data malformed. $endChar not found after $startIndex.");
         die "ERROR: Property data malformed. $endChar not found after $startIndex.";
     }
     else {
         logDebugMsg("String $result collected till $g_defIndex.");
     }
         
    return $result;
} 

#==============================================================================
# <String:nextCharacter> nextDefChar()
#
# Get the next character after the current index. The method returns next 
# character from the string stored in variable $g_defData.
#
# Arguments:
#   none
# Returns:
#  If the data index is less than the data length then next character will be returned.
# else a empty string will be returned. 
#==============================================================================
sub nextDefChar() {
    my $character = "";
    
    if ($g_defIndex < $g_defLength) {
        $character = substr($g_defData, $g_defIndex, 1);
        $g_defIndex++;
    }
    
    return $character;
} 

#==============================================================================
# <updatedData> escapeChars(<String:data>, <String:charArray>)
#
# Escape characters from <data> string with "\" (backslash). The characters are escaped according to the array order. 
#
# Arguments:
#   <data> - String data containing characters that need to be escaped.
#   <charArray> - characters that need to be escaped. 
# Returns:
#  data with characters escaped. 
#==============================================================================
sub escapeChars($;$) {
    my $data = $_[0];
    my $refChar = $_[1];
    
    $data =~ s/\\/\\\\/g;
    
    my @charArray = @$refChar;
    foreach my $char (@charArray) {
        my $regExChar = ecmFileUtils::makeRegex($char);
        $data =~ s/$regExChar/\\$char/g;
    } 
    
    return $data;
}

#==============================================================================
# <updatedData> unEscapeChars(<String:data>, <String:charArray>)
#
# unEscape characters from <data> string which were escaped with "\" (backslash). 
# The characters are unescaped according to the array order. 
#
# Arguments:
#   <data> - String data containing characters that need to be unescaped.
#   <charArray> - characters that need to be unescaped. 
# Returns:
#  data with characters escaped. 
#==============================================================================
sub unEscapeChars($;$) {
   my $data = $_[0];
    my $refChar = $_[1];
    
    my @charArray = @$refChar;
    foreach my $char (@charArray) {
        my $regExChar = ecmFileUtils::makeRegex($char);
        $data =~ s/\\$regExChar/$char/g;
    }
    
    $data =~ s/\\\\/\\/g; 
    
    return $data; 
}

#==============================================================================
# void getData(<String: basePath>, <String:ccsStringRep>, <String:dbUrl>,
#              <String:dbUser>, <String:dbPassword>, <String:targetProperties> )
#
# Print the data from file/os command/database query as per the ccs to STDOUT. 

# Arguments:
#   <String:basePath> - Base path for file definitions.
#   <String:ccsStringRep> - representation of ccs in string. 
#   <String:encoding> - encoding to be used for all files.
#   <String:dbUrl> - database url to connect to database.
#   <String:dbUser> - database user.
#   <String:dbPassword> - database user password.
#   <String:targetProperties> - string representation of target properties.
# Returns:
#  data for the definition as per OS_LINE_TOKEN format. 
#==============================================================================
my %g_absFilePathHash;

sub getData($;$;$;$;$;$) {
    my $basePath = $_[0];
    my $ccsExpr =  $_[1];
    my $encoding = $_[2];
    
    my $dbUrl =  $_[3];
    my $dbUser = $_[4];
    my $dbPassword = $_[5];
    my $targetProperties = $_[6];
    
    # initailize the order counter.
    $expressionOrder = 0;
    
    # initialize absolute file path hash.
    %g_absFilePathHash = ();
    
    my $defRef = getDefinitions($ccsExpr);
    
    my @defArray = @$defRef;
    my $ccsData = "";
    
    my $configItems = 0;
    
    $basePath = ecmFileUtils::substituteVars($basePath, $targetProperties);
    logDebugMsg("getData: using basepath:$basePath");
    
    foreach my $currentDefRef (@defArray) {
        my %currentDef = %$currentDefRef;
        my $type = $currentDef{$KEY_TYPE};
        
        my $defData = "";
        
        if ($type eq $TYPE_FILE) {
            getFileData($basePath, $currentDefRef, $encoding);
            $configItems++;
        }
        elsif ($type eq $TYPE_CMD) {
            getCmdData($currentDefRef, $basePath);
            $configItems++;
        }
        elsif ($type eq $TYPE_DB) {
            if (!defined($dbUrl) || $dbUrl eq "") {
               $dbUrl = ecmFileUtils::getDbUrlFromProperties($targetProperties);
            }
            if ($dbUrl ne "") {
               getDbData($currentDefRef, $dbUrl, $dbUser, $dbPassword);
               $configItems++;
            }
        } 
        else {
            logDebugMsg("ERROR: Invalid CCS expressions. $type");
        }
        
        if ($configItems < 1) {
            # this may happen if a CCS only contains SQL queries and no
            # JDBC URL has been specified
            logDebug("ERROR: No data collected.");
            die "ERROR: No data collected.";
        }
        
        $ccsData = $ccsData.$defData;
    }
}

#==============================================================================
# <void> getFileData(<String:basePath>, <Reference:fileDefHash>, <String:encoding>)
#
# Prints data from file as per the ccs definition to STDOUT. The data is in the 
# OS_LINE_TOKEN format.
#
# Arguments:
#   <String:basePath> - Base path for file definitions.
#   <Reference:fileDefHash> - file definition Hash containing data to fetch data from a file.
#   <String:encoding> - encoding to be used for all files. 
# Returns:
#   Nothing.
#==============================================================================
sub getFileData($;$;$) {
    my $basePath = $_[0];
    my $defRef =  $_[1];
    my $encoding = $_[2];
    
    my $DIR_SEPARATOR= '/';
    
    my $errorMsg= "";
    my $error = 0;
    
    my %def = %$defRef;
    
    my $expression = $def{$KEY_EXPRESSION};
    my $currentEncoding = $def{$KEY_ENCODING};
    my $datasourceName = $def{$KEY_NAME};
    
    # unescape so that special characters are handled properly - otherwise,
    # files with special characters (e.g., "a|b.txt") will not get 
    # collected successfully
    $expression = unEscapeChars($expression, \@RESERVED_CHARS);
    $datasourceName = unEscapeChars($datasourceName, \@RESERVED_CHARS);

    # use windows separator if OS is windows
    if ($^O eq "MSWin32") {
        $DIR_SEPARATOR= '\\';
    }
    
    if ($currentEncoding ne "") {
     $encoding = $currentEncoding;
    }
    
    logDebugMsg("getFileData: using encoding:$encoding");
    
    my @filePaths = ();
    
    if (defined($expression) && $expression ne "") {
        if ($expression =~ m/\*|\?/) {
        	   logDebugMsg("getFileData: expanding regEx:$expression");
            eval {
                my $filePathArray = ecmFileUtils::findFiles($basePath, $expression);
                @filePaths = @$filePathArray;
            };    
            
            if (defined($@) && $@ ne "")  {
                $error = 1;
                $errorMsg = "Failed to expand wildcard expression. Cause- $@";                   
            }
        }
        else {
             logDebugMsg("getFileData: file expression without regEx. relativePath:$expression");
        	
             my $filePath = ecmFileUtils::removeLastSlash($expression);
             my $currentBasePath = $basePath;
             $currentBasePath = ecmFileUtils::removeLastSlash($currentBasePath);
             
             my $absolutePath = $currentBasePath.$DIR_SEPARATOR.$filePath;
             
             push (@filePaths, $absolutePath);
        }
        
        if ($error == 0) {
            foreach my $currentFile (@filePaths) {
                my $fileAlreadyLoaded = $g_absFilePathHash{$currentFile};
                my $proceed = 1;
                
                if (defined($fileAlreadyLoaded) && $fileAlreadyLoaded ne "") {
                   logDebugMsg("getFileData: File already loaded by previous expressions skipping it. $currentFile");
                   $proceed = 0;
                }
                
                if (-e $currentFile && -r $currentFile && $proceed == 1) {
                	  logDebugMsg("getFileData: Reading file $currentFile.");
                	  
                	  $encoding = ecmFileUtils::getValidEncoding($encoding, $currentFile);
                	   
                    my $INPUT_FILE;
                    
                    my $fileData = "";
                    my $md5 = "";
                    my $filesize = "";
                    
                    eval {
                        # use the ":raw" layer to preserve /r characters.  Otherwise
                        # PERL removes the /r characters from the input data automatically.
                        # Use the specified endoding as well to obtain a logical (Unicode)
                        # string instead of a byte string once the read is complete
                        open ($INPUT_FILE, "< $currentFile");
                        if (defined($encoding) && $encoding ne "") {
                            binmode($INPUT_FILE, ":raw:encoding($encoding)");
                        }
                        else {
                            binmode($INPUT_FILE, ":raw:encoding(UTF-8)");
                        }

                        my $currentLineTerminator = $/;
                        undef $/;
                        
                        $fileData = <$INPUT_FILE>;

                        $/ = $currentLineTerminator;
                                        
                        close($INPUT_FILE);
 
                        $md5 = md5_hex(encode_utf8($fileData));
                        $filesize = -s $currentFile;
                    };    
                    
                    if (defined($@) && $@ ne "")  {
                        $errorMsg = "Failed to get data for file. Cause- $@";
                    }
                    
                    # evaluate relative path.
                    my $relativePath = ".";
                    if (length($currentFile) > length($basePath)) {
                       $relativePath = substr($currentFile, length($basePath));
                    	  $relativePath = ecmFileUtils::removeFirstSlash($relativePath);
                    	  if ($relativePath eq "") {
                    		  $relativePath = "."
                    	  }
                    }
                    
                    printData($relativePath, $TYPE_FILE, ++$expressionOrder, $expression, $currentFile, $filesize, $md5, $fileData, $errorMsg);
                    
                    #update hash so the duplicate file load does not happen.
                    $g_absFilePathHash{$currentFile} = $currentFile;
                    
                }
                else {
                   if ($proceed == 0) {
                     logDebugMsg("getFileData: File $currentFile skipped.");
                   }
                   else {
                      logDebugMsg("getFileData: Can not read file $currentFile.");  
                   } 
                }
            }
        } 
        else {
            printData($datasourceName, $TYPE_FILE, ++$expressionOrder, $expression, "", "", "", "", $errorMsg);
        }   
    }
    
}

#==============================================================================
# <void> printData(<dataSourceName>,<type>,<order>,<expression>,<fullPath>,
#                  <size>,<hash>,<data>,<errorMsg>)
#
# Construct the data string as per the OS Fetchlet and print it to STDOUT. 

# Arguments:
#   <dataSourceName> - name of the ccs definition.
#   <type> - type of definition F,C or D
#   <order> - order of the definition
#   <expression> - relative path, regex, command, databse query
#   <fullPath> - fullPath for files;
#   <size> - data size in bytes 
#   <hash> - md5 hash for data
#   <data> - data contents
#   <errorMsg> - error message if data collection for the definition was not successful. 

# Returns:
#  nothing. 
#==============================================================================
sub printData {
    my ($dataSourceName, $type, $order, $expression, $fullPath, $size, $hash, $data, $errorMsg) = @_;
    
    my $ccsDefData = "";

    $ccsDefData = $ccsDefData.length($dataSourceName).$UNDER_SCORE.$dataSourceName.$PIPE;
    $ccsDefData = $ccsDefData.length($type).$UNDER_SCORE.$type.$PIPE;
    $ccsDefData = $ccsDefData.length($order).$UNDER_SCORE.$order.$PIPE;
    $ccsDefData = $ccsDefData.length($expression).$UNDER_SCORE.$expression.$PIPE;
    $ccsDefData = $ccsDefData.length($fullPath).$UNDER_SCORE.$fullPath.$PIPE;
    $ccsDefData = $ccsDefData.length($size).$UNDER_SCORE.$size.$PIPE;
    $ccsDefData = $ccsDefData.length($hash).$UNDER_SCORE.$hash.$PIPE;

    # use length here because the agent fetchlet expects the number of 
    # characters, not the number of bytes
    $ccsDefData = $ccsDefData.length($data).$UNDER_SCORE.$data.$PIPE;
    
    $ccsDefData = $ccsDefData.length($errorMsg).$UNDER_SCORE.$errorMsg.$PIPE;
    
    # add the appropriate line termination sequence.  The data will
    # be printed in binary mode, so "\r" will not be added in 
    # automatically by PERL, so it must be added explicitly
    if ($^O eq "MSWin32") {
        $ccsDefData = $ccsDefData."\r\n";
    }
    else {
        $ccsDefData = $ccsDefData."\n";
    }

    # convert logical (Unicode) string to byte string for output
    $ccsDefData = Encode::encode("UTF-8", $ccsDefData);

    # output the data in ":raw" mode to prevent PERL 
    # from adding "\r" in front of newline ("\n") characters
    binmode(STDOUT, ":raw");

    print STDOUT $ccsDefData;
}

#==============================================================================
# <void> getFileCmdData(<Reference:cmdDefHash>, <String:basePath>)
#
# Print the data from execution of a OS command as per the ccs definition.
# The data is printed on the STDOUT

# Arguments:
#   <Reference:cmdDefHash> - command definition Hash containing data to excute OS command.
#   <String:basePath> - base path configured for the ccs. 
# Returns:
#  nothing 
#==============================================================================
sub getCmdData($) {
    my $defRef =  $_[0];
    my $basePath = $_[1];
    
    my %def = %$defRef;
    
    my $ccsDefData = "";
    
    my $expression = $def{$KEY_EXPRESSION};
    
    $expression = unEscapeChars($expression, \@RESERVED_CHARS);
    
    my $encoding = $def{$KEY_ENCODING};
    my $ignoreOP = $def{$KEY_IGNORE_OUTPUT};
    my $timeOut =  $def{$KEY_COMMAND_TIMEOUT};
    my $datasourceName = $def{$KEY_NAME};
    
    my $errorMsg = "";
    
    $expression =~ s/\$/\\\$/g;
    
    if (!defined($timeOut) || $timeOut == 0) {
        $timeOut = $DEFAULT_COMMAND_TIMEOUT;
    }
    
    logDebugMsg("getCmdData: using expression:$expression");
    
    $SIG{ALRM} = \&commandTimedOut;
    
    eval {
        alarm($timeOut);
        
        if (defined($basePath) && $basePath ne "") {
           logDebugMsg("Changed working directory to $basePath.");
           chdir $basePath;
        }
        else {
          logDebugMsg("Working directory not changed for commands"); 
        }
        
        my($success, $errorCode, $stdOutErr, $stdout, $stderr) =  run(command => $expression);
        alarm(0);
        
        if (defined($ignoreOP) && lo($ignoreOP) eq "y") {
            logDebugMsg("Pre/Post command. Output will not be collected for the command.");
        }
        else {
        
            my $defData = join('', @$stdout);
            my $size = length($defData);
            
            Encode::_utf8_on($defData);
            my $md5 = md5_hex(encode_utf8($defData));
            
            printData($datasourceName, $TYPE_CMD, ++$expressionOrder, $expression, "", $size, $md5, $defData, "");
        }
    
    };
      
    if (defined($@) && $@ ne "")  {
        $errorMsg = "Failed to execute command. Cause- $@";
        printData($datasourceName,$TYPE_CMD, ++$expressionOrder, $expression, "", "", "", "", $errorMsg)
    }  
    
}

#==============================================================================
# commandTimedOut()
#
# Raise a exception so that processing stops. 
# Return data from execution of a OS command as per the ccs definition. 
#==============================================================================
sub commandTimedOut {
    die "Command time out reached.";
}

#==============================================================================
# <returnData> getDbData(<Reference:DbDefHash>, <DbUrl>, <DbUser>, <DbPassword>)
#
# return data from execution of a SQL query as per the ccs definition. 
# Arguments:
#   <Reference:DbDefHash> - Database SQL query definition hash.
#   <DbUrl> - Database connection url according to PERL syntax. 
#         example: dbi:Oracle:HOST=localhost;SID=I18N;PORT=1521
#   <DbUser> - Database user role to use.
#   <DbPassword> - Database user password.
# Returns:
#  data for the definition as per OS_LINE_TOKEN format. 
#==============================================================================
sub getDbData($;$;$;$) {
    my $defRef =  $_[0];
    my $dbUrl = $_[1];
    my $dbUser = $_[2];
    my $dbPassword = $_[3];
    
    my %def = %$defRef;
    
    my $ccsDefData = "";
    my $defData = "";
    my $errString = "";
    my $error = 0;
    
    my $expression = $def{$KEY_EXPRESSION};
    $expression = unEscapeChars($expression, \@RESERVED_CHARS);
    
    my $datasourceName = $def{$KEY_NAME};
    my $errorMsg = "";
    my $md5 = "";
    my $size = "";
    
    my $longReadLen = $def{$KEY_LONG_READ_LENGTH};
    
    if (!defined($longReadLen)) {
        $longReadLen = $DEFAULT_LONG_READ_LENGTH;
    }
    
    logDebugMsg("using DB url $dbUrl");
    
    eval {
        my $dbHandle = DBI->connect($dbUrl, $dbUser, $dbPassword, {RaiseError =>0, PrintWarn=>0, PrintError => 0, AutoCommit => 0 });
        
        if ($DBI::errstr ne "") {
            $errString = $errString.$DBI::errstr;
            $error = 1;
        }
        
        if ($error != 1) {
            $dbHandle->{LongTruncOk} = 1;
            $dbHandle->{LongReadLen} = $longReadLen;
            
            my $queryHandle = $dbHandle->prepare($expression);
            
            if (defined($queryHandle)) {
               $queryHandle->execute();  
               
               #execution of query failed.
               if ($DBI::errstr ne "") {
                  $errString = $errString.$DBI::errstr;
                  $error = 1;
               } 
            }
            else {
               my $dbErrMsg = $dbHandle->errstr();
               $errString = $errString."Failed to prepare sql query. $dbErrMsg";
               $error = 1;
            }
            
            if ($error != 1) {
                while (my $hashRef = $queryHandle->fetchrow_hashref()) {
                    my %row = %$hashRef;
                    
                    $defData = $defData.$ROW_HEADER;
#                    $DB::single=1;
                    foreach my $columnName (keys(%row)) {
                        $defData = $defData.length($columnName).'_'.$columnName.'=';
                        
                        my $colData = $row{$columnName};
                        
                        if ($colData eq "") {
                           $defData = $defData."0_\n"; 
                        }
                        else {
                           Encode::_utf8_on($colData);
                           $defData = $defData.length($colData).'_'.$colData."\n";
                        }
                    }                        
                }
                
                if ($DBI::errstr ne "") {
                    $queryHandle->finish();
                    
                    $errString = $errString.$DBI::errstr;
                    $error = 1;
                }
            }
            
            $dbHandle->disconnect;
            
            $size = length($defData);
            $md5 = md5_hex(encode_utf8($defData));
        }
    };
    
    if ($error == 1) { 
       $errorMsg = "Failed to execute query. Cause-$errString"; 
       logDebugMsg("ERROR: $errorMsg");
    }
    elsif ($@ ne "") {
       $errorMsg = "Failed to execute query. Runtime execution cause-$@";
       $error = 1;
       logDebugMsg("ERROR: $errorMsg");
    }
    
     
     if ($error == 1) {
         logDebugMsg("printing with error message."); 
         printData($datasourceName, $TYPE_DB, ++$expressionOrder, $expression, "", "", "", "", $errorMsg)
     }
     else {
         logDebugMsg("printing successful data.");
         printData($datasourceName, $TYPE_DB, ++$expressionOrder, $expression, "", $size, $md5, $defData, "")
     }
}

#==============================================================================
# <returnData> getStdin()
#
# Process the STDIN and return the data. The call will be blocked for a minute
# before comming out of the method.
 
# Arguments:
# None
# Returns:
#  Data captured from STDIN. CHOMP is performed on the data before returning it. 
#==============================================================================
sub getStdin() {
   
   logDebugMsg("Getting data from STDIN");
   
   my $stdinData = "";
   
   $SIG{ALRM} = \&stdinTimedOut;
    
    eval {
        alarm(30);
        
        $stdinData = <STDIN>;
        logDebugMsg("data before chomp [$stdinData]");
        
        chomp ($stdinData);
        alarm(0);
    };
      
    if (defined($@) && $@ ne "")  {
        logDebugMsg("Failed to read stdin. Cause- $@");
    }
    
    logDebugMsg("STDIN captured [$stdinData]");
    
    return $stdinData;  
}

#==============================================================================
# stdinTimedOut()
#
# Callback method for comming out of a blocked call. 
#==============================================================================
sub stdinTimedOut {
    logDebugMsg("STDIN timeout reached.");
}
    
###################################################################
# Main 
###################################################################

1;
