# $Header: emagent/sysman/admin/scripts/ccs/ecmFileUtils.pl /main/7 2012/06/11 06:16:51 bdknowlt Exp $
#
# Copyright (c) 2010, 2012, Oracle and/or its affiliates. All rights reserved. 
#
#    DESCRIPTION
#      Utility methods for file handling.
#
#    MODIFIED   (MM/DD/YY)
#       skundalk 06/07/11 - adding support for multiple target properties in
#                           basepath
#       skundalk 12/21/10 - logging target property messages
#       skundalk 10/05/10 - encoding changes
#       skundalk 09/27/10 - adding logging and fixing regex link problem
#       skundalk 09/09/10 - enable target property substitution code and
#                           updating script include path
#       skundalk 06/25/10 - ccs script files
#      
##*********************************************************************
package ecmFileUtils;

use File::Find;
use strict;

require "ccs/ecmPropertyInstance.pl";

my $LOCAL_PROPERTIES = "LOCAL_PROPERTIES";

#encoding constants
# use UTF-8 for xml files and target locale for rest of the files.
my $ENCODING_ORACLE_DEFAULT = "ORACLE_DEFAULT";
# use target locale for all files
my $ENCODING_TARGET_LOCALE = "TARGET_LOCALE";

#==============================================================================
# logDebugMsg(<message>)
#
# log the message to log file.
# Arguments:
#   message - message to be logged.
# Returns:
#   none
#==============================================================================
sub logDebug($) {
    ecmCcsUtils::logDebug("ecmFileUtils.pl", $_[0]);
    
    return;
}

#==============================================================================
# <String:normPath> normalizeSlash(<String:path>)
#
# method to convert all "\" to "/".
# This method will mostly be used for windows style path.

# Arguments:
#   <String:path> containing back slashes.
# Returns:
#   String containing all forward slashes. 
#==============================================================================
sub normalizeSlash($) {
    my $path = $_[0];
    $path =~ s/\\/\//g;
    
    return $path;
}

#==============================================================================
# <String:updFilePath> removeLastSlash(<String:filePath>)
#
# method to remove "/" and any spaces after it from RHS if present.  
#
# Arguments:
#   <String:filePath> - String ending with "/" or "/" and spaces.
# Returns:
#   String after removing "/" or "/" and spaces from RHS.
#==============================================================================
sub removeLastSlash($) {
    my $path = $_[0];
    $path =~ s/\/\s*$//;
    
    return $path;
}

#==============================================================================
# <String:updFilePath> removeFirstSlash(<String:filePath>)
#
# method to remove "/" from LHS if present. If there are more than one "/"
# on LHS all of them will be removed till a non slash character is found.
#
# Arguments:
#   <String:filePath> - String starting with "/" .
# Returns:
#   String after removing "/" from LHS.
#==============================================================================
sub removeFirstSlash($) {
    my $path = $_[0];
    $path =~ s/^\/*//;
    
    return $path;
}

#==============================================================================
# <String:escapedData> makeRegex(<String:data>)
#
# method to make the string a proper regular expression. Escaping regular 
# expression special characters.
#
# Arguments:
#   String which needs to be converted to regex.
# Returns:
#   updated regex .
#==============================================================================
sub makeRegex($){
    my $exp = $_[0];
    
    $exp =~ s/\\/\\\\/g;
    $exp =~ s/\[/\\[/g;
    $exp =~ s/\./\\./g;
    $exp =~ s/\|/\\|/g;
    $exp =~ s/\+/\\+/g;
    $exp =~ s/\(/\\(/g;
    $exp =~ s/\)/\\)/g;
    $exp =~ s/\^/\\^/g;
    $exp =~ s/\$/\\\$/g;
    $exp =~ s/\{/\\{/g;
    
    $exp =~ s/\//\\\//g;
    $exp =~ s/\*/\.\*/g;
    $exp =~ s/\?/\.\?/g;
    
    
    return $exp;
}

#==============================================================================
# <ArrayRef:absFilePath> findFiles(<basePath>,<relativeFilePath>)
#
# method to expand the regular expression into absolute file name.
#   
#
# Arguments:
#   <basePath> = Base Path without any regular expression.
#   <relativeFilePath> = Regular exrepssion for the file name.
# Returns:
#   Array reference containing the absolute path to the expanded files.
#==============================================================================
sub findFiles($;$) {
    my $basePath = $_[0];
    my $filePath = $_[1];
    my $dirSeparator = '\/';
    
    $filePath = removeLastSlash($filePath);
    $basePath = removeLastSlash($basePath);
    
    logDebug("DEBUG:using basePath: $basePath  filePath: $filePath");
    
    my @regexArray = split(/$dirSeparator/, $filePath);
    
    my $fileName = pop (@regexArray);
    
    my $depth = scalar(my @basePathTokens = split(/$dirSeparator/, $basePath));
    
    my $currentRegEx = "";
    my @directories = ($basePath);
    
    my $continue = 1;
    my $counter = 0;
    my $dirCount = scalar @regexArray;
    
    while ($counter < $dirCount && $continue == 1) {
        my $token = $regexArray[$counter];
        logDebug("DEBUG:Evaluating regex: $token");
         
        if ($token =~ m/(\s)*\*\*(\s)*/) {  # match spaces**spaces or **
            logDebug("DEBUG:recursive directory regex found.");
            $depth = -1;
            $continue = 0; 
        }
        else {
            $currentRegEx = '^'.makeRegex($token);
            my $refNewDirs = checkDir($currentRegEx, $depth, \@directories, 1);
            @directories = @ {$refNewDirs};
            my $dirCount = scalar @directories;
            
            if ($dirCount == 0) {
                $continue = -1; # do not continue 
            }
            $depth++;
        }
        
        $counter++;
    }
    
    my @files = ();
    
    if ($continue != -1) {
        $currentRegEx = '^'.makeRegex($basePath).$dirSeparator.makeRegex($filePath).'$';
        
        logDebug("DEBUG:completePathRegex:$currentRegEx"); 
        
        my $refFiles = checkDir($currentRegEx, $depth, \@directories, 0);
        @files = @ {$refFiles};
    } 
    
    logDebug("DEBUG:Number of files evaluated from regex:".scalar(@files).".");
       
    #sorting the files alphabetically. This is needed because find()
    #can traverse the tree differently(may be based on inodes) found this on farm runs. 
    
    my @sortedFiles = sort {uc($a) cmp uc($b)} @files;
    
    return \@sortedFiles;
}

#==============================================================================
# PRIVATE <ArrayRef:absPathResources> checkDir(<String:regEx>,
#                                              <Number:depth>, 
#                                              <ArrayRef:directories>,
#                                              <Number:resourceType>)
#
# method to check directories for regular expression matching directories or files.
#   
#
# Arguments:
#   <String:regEx> = Regular exressions that is to be matched with directories or files.
#   <Number:depth> = Depth till which a directory traversal will be performed.
#   <ArrayRef:directories> = Array of directory entries that will be searched.
#   <Number:resourceType> = if 1 then the method returns directories matching the Arg1 regex.
#            if 0 the method returns files matching the Arg1 regex.
# Returns:
#   Array containing the absolute path to the expanded files or directories.
#==============================================================================
my $g_regex = "";
my $g_depth = "";
my @g_resources = ();
my $g_returnDirs = 0;

sub checkDir($;$;$;$) {
    $g_regex = $_[0];
    $g_depth = $_[1];
    my $refDir = $_[2];
    $g_returnDirs = $_[3];
    
    my @directories = @ {$refDir};
    
    @g_resources = ();
    
    find({ wanted => \&process, follow => 1, follow_skip => 2}, @directories);
    
    my @resources = (@g_resources);
    
    return \@resources;
}

#==============================================================================
# PRIVATE process()
#
# Call back method for finding files or directories. 

# Note: Comment the debug log messages when done troubleshooting.  
#
# Arguments:
#   none
# Returns:
#   none
#==============================================================================
sub process {
    my @tokens = split(/\//, $File::Find::name);
    my $currentDepth = scalar (@tokens);
    my $collect = 0;
    
#    logDebug("DEBUG:processing resource:$File::Find::name");
    
    if ($g_returnDirs == 1 && (-d $File::Find::name)) {
#      logDebug("DEBUG:currentDepth: $currentDepth, gDepth: $g_depth, returnDir:$g_returnDirs ");
      
        my $dirName = $_;
        if ($currentDepth != $g_depth) { # do not match current listing directory.
           if ($dirName =~ m/$g_regex/) { 
              $collect = 1;
           }
        
           if ($g_depth != -1) {
             $File::Find::prune = 1;
           }         
        }
    }
    elsif ($g_returnDirs == 0) {
        if ((-f $File::Find::name) && $File::Find::name =~ m/$g_regex/) {
            $collect = 1;
        }
        elsif ($g_depth != -1 && (-d $File::Find::name) && $currentDepth != $g_depth) { 
         # do not match current listing directory.
         # if not recursive directory search then stop descending into lower directories.
            $File::Find::prune = 1;
        }
    }
    
    if ($collect == 1) { 
        push (@g_resources, $File::Find::name);
    }
}

#==============================================================================
# <updatedBasePath> substituteVars(<basePath>, <propertyInstanceString>)
#
# Substitue the variables in <basePath> with its value from the <PropertyInstanceString>. 
#
# Arguments:
#   <basePath> - Base path for file definitions.
#   <propertyInstanceString> - String representation of target and agent properties. 
# Returns:
#  Updated basePath with substituted variables. 
#==============================================================================

my $data = "";
my $index = 0;
my $dataLength = 0;
my $varStartChar = "{";
my $varEndChar = "}";

sub substituteVars($;$) {
    my $basePath = $_[0];
    my $propertyInstanceStr = $_[1];
    
    my $updatedBasePath = $basePath;
    
    my $varRef = getVariableNames($basePath);
    my @varNames = @$varRef;
    
    if (scalar(@varNames) > 0) {
        my $mapRef = ecmPropertyInstance::processProperties($propertyInstanceStr);
        
        my %propertyMap = %$mapRef;
        
        my $propertyValue;
        
        my $refProperties = $propertyMap{$LOCAL_PROPERTIES};
        
        foreach my $varName (@varNames) {
           if (defined($refProperties)) {
            my %properties = %$refProperties;
        
            $propertyValue = $properties{$varName};
           }    
        
           if (!defined($propertyValue)) {
              logDebug("ERROR: Target Property \"$varName\" missing from string representation. ");
              die "ERROR: Target Property \"$varName\" missing from string representation. "; 
           }
           else {
               my $startChar = makeRegex($varStartChar);
               my $endChar = makeRegex($varEndChar);
               my $rxVarName = makeRegex($varName);
               
               my $search = $startChar.$rxVarName.$endChar;
               
               $updatedBasePath =~ s/$search/$propertyValue/g;
               
               logDebug("Updated basePath $updatedBasePath");
           }
        }
        
    }
    else {
        logDebug("Target Property not found in base path. No substitution done.");
    }
    
    return $updatedBasePath;
}


#==============================================================================
# <variableNameArrayReference> getVariableNames(<basePath>)
#
# Extract the variable name from the base path. Vairable name will be prefixed
# by $varStartChar and suffixed by $varEndChar. 
# Arguments:
#   <basePath> - Base path for file definitions.
# Returns:
#  Array containing Variable name if any, else the return EMPTY array. 
#==============================================================================
sub getVariableNames($) {
    my $basePath = $_[0];
    
    $dataLength = length($basePath);
    $index = 0;
    $data = $basePath;
    logDebug("Processing basepath $basePath for variable names");
    
    my @variableNames;
    my $variableName;
    while ($index < $dataLength) {
        my $tempChar = nextChar();
        # logDebug("Next character in basepath:$tempChar");
        
        if ($tempChar eq $varStartChar) {
            $variableName = getStringTill($varEndChar);
            logDebug("Target property found in base path:$variableName");
            push(@variableNames, $variableName);
        }
    }
    
    return \@variableNames;
}

#==============================================================================
# nextChar()
#
# Get the next character after the current index. The method returns next 
# character from the string stored in variable $data.
#
# Arguments:
#   none
# Returns:
#  If the data index is less than the data length then next character will be returned.
# else a empty string will be returned. 
#==============================================================================
sub nextChar() {
    my $character = "";
    if ($index < $dataLength) {
        $character = substr($data, $index, 1);
        $index++;
    }
    
    return $character;
}

#==============================================================================
# <String:subString> getStringTill(<String:endCharacter>)
#
# Get the characters till the indicated character. The search character is not
# included in the return String.
# The method searches the $data variable for the string from the 
# current $index value.
#
# Note: if the endCharacter is not found the method dies.

# Arguments:
#   <String:endCharacter> - character at which the search should stop.
# Returns:
#   sub string till the endCharacter.
#==============================================================================
sub getStringTill($) {
    my $endChar = $_[0];
    my $continue = 1;
    my $startIndex = $index;
    my $result = "";
    
     while (($index < $dataLength) && ($continue == 1)) {
         my $tempChar = nextChar();
         
         if ($tempChar eq "\\") {
            # escape character.
            nextChar();
        }    
        elsif ($tempChar eq $endChar) {
            $continue = 0;
            $result = substr($data, $startIndex, ($index - 1 - $startIndex));
        }    
     }
     
     if ($continue == 1) {
         logDebug("ERROR: Property data malformed. $endChar not found after $startIndex.");
         die "ERROR: Property data malformed. $endChar not found after $startIndex.";
     }
     else {
         logDebug("String $result collected till $index.");
     }
         
    return $result;
}    

#==============================================================================
# <String:encoding> getValidEncoding(<String:ccsEncoding>, <String:fileName>)
#
# Check the encoding string and return the proper encoding to be used for reading
# the file.
# if the encoding is TARGET_LOCAL then empty string is returned and the file is 
# to be read using the LOCALE encoding set.
# if the encoding is ORACLE_DEFAULT then all .xml files will be read with UTF-8
# encoding and all non xml files will be read with the LOCALE encoding set
# on the target machine. 
# 
# Arguments:
#   <String:ccsEncoding> - encoding specified for the file definition.
#   <String:fileName> - file name.
# Returns:
#   sub string till the endCharacter.
#==============================================================================
sub getValidEncoding($;$) {
   my $encoding = $_[0];
   my $fileName = $_[1];
   
   logDebug("getValidEncoding: checking encoding: $encoding for file:$fileName.");
   
   my $validEncoding = $encoding;
   
   if ($encoding eq $ENCODING_ORACLE_DEFAULT) {
      if ($fileName =~ m/\.xml$/i) {
         $validEncoding = "UTF-8";
      }   
      else {
         $validEncoding = "";
      }
   }
   elsif ($encoding eq $ENCODING_TARGET_LOCALE) {
      $validEncoding = "";
   }
   
   logDebug("getValidEncoding: using encoding:$validEncoding for file:$fileName.");
   
   return $validEncoding;
}

#==============================================================================
# <String:dbUrl> getDbUrlFromProperties(<String:targetPropertyStr>)
#
# Check the target properties for "SID", "Port" and "MachineName"
# get the values and create perl specific dbi url.
# example: dbi:Oracle:HOST=localhost;SID=I18N;PORT=1521
# 
# Note: The method dies if any of the needed properties do not have values set.

# Arguments:
#   <String:targetPropertyStr> - target properties string representation.
# Returns:
#   dbUrl created from target properties.
#==============================================================================
sub getDbUrlFromProperties($) {
   my $rtnVal = "";
   my $propertyInstanceStr = $_[0];
   
   my $NAME_SID = "SID";
   my $NAME_PORT = "Port";
   my $NAME_MACHINE = "MachineName";
   
   my $mapRef = ecmPropertyInstance::processProperties($propertyInstanceStr);
        
   my %propertyMap = %$mapRef;
  
   my $refProperties = $propertyMap{$LOCAL_PROPERTIES};
  
   my %properties = %$refProperties;
  
   my $dbSid = $properties{$NAME_SID};
   my $dbPort = $properties{$NAME_PORT};
   my $dbMachine = $properties{$NAME_MACHINE};
   
   logDebug("getDbUrlFromProperties: dbSid:$dbSid, dbPort:$dbPort, dbMachine:$dbMachine");
   
   if (!defined($dbSid) || $dbSid eq "" || !defined($dbPort) || $dbPort eq "" ||
                    !defined($dbMachine) || $dbMachine eq "")
   {
      if ((!defined($dbSid) || $dbSid eq "") && (!defined($dbPort) || $dbPort eq "") &&
                    (!defined($dbMachine) || $dbMachine eq ""))
      {
         # no JDBC URL specified - in this case, the query will be skipped, but the
         # rest of the collection will proceed
         $rtnVal = "";
      }
      else {
         logDebug("ERROR: Can not create a database url for connection.");
         die "ERROR: Invalid values. Can not create a database url for connection. 
                    dbSid:$dbSid, dbPort:$dbPort, dbMachine:$dbMachine";
      }
   }
   else {
      #dbi:Oracle:HOST=localhost;SID=I18N;PORT=1521
      $rtnVal = "dbi:Oracle:HOST=".$dbMachine.";SID=".$dbSid.";PORT=".$dbPort;
   }

  return $rtnVal;
}

###################################################################
# Main 
###################################################################

1;