####################################################################
# Copyright (c) 2011, 2012, Oracle and/or its affiliates. All rights reserved. 
#
#     Name: getpatchinv.pl
#
# How to run: '
#  'getpatchinv.pl -homes /path/homefile.txt -out /path/outputfile
#                  -jre <path to executable jre>
#
# the script returns zero on success, non-zero on failure.
#
# This script requires the following:
# a)  'perl' must be existed in Oracle Home
# b) 'opatch lspatches' command is available under ORACLE_HOME/OPatch
# c) input file given via the -homes cmd-line arg
#
# You can pass in -jre <path to executable jre> so OPatch will run
# using your given JRE instead of OPatch's searching for it
#
#
# <homefile.txt> is a text file, each line contains the path to the Oracle Home
# <outputfile> is where the output of the script goes to.
#    if you specify an existing file, the script will overwrites
#             its previous content
#    the output file has following format:
#       ORACLE_HOME=<path to home>
#       <list of patch>
#    where
#       <path to home> is what you put in he <homefile.txt>
#       <lis of patch> is the list of patches, each patch is on a new line
#    if there is no patch in the given Oracle Home, the <list of patch>
#    will be empty
#    if given Oracle Home is not valid (can' find it, can' run OPatch...), it
#    will not be present in the output file.
#
#   Sample homefile.txt
#
#   /ua0/homes/dbhome1
#   /ua0/homes/dbhome2
#   /ua0/homes/dbhome3
#
#   Sample output file
#
#   ORACLE_HOME=/ua0/homes/dbhome1
#   123:Agent patch for memory leak
#   124:Another agent patch
#   ORACLE_HOME=/ua0/homes/dbhome2
#
#   Explanation of above output
#   dbhome1 has 2 patches installed. The "Agent patch for memory leak"
#   and "Another agent patch" is the patch description
#
#   dbhome2 has no patch installed
#   dbhome3 doesn't exist, or the script can' run OPatch, so it is removed from
#    the output
#
# Bugs
# <if you know of any bugs, put them here>
#
#    MODIFIED   (MM/DD/YY)
#    ysarig     01/24/12 - Restruct to always use opatch from the agent home
#    Bin        12/24/11 - Implement it
#    Phi        12/24/11 - Initial creation
#
#####################################################################

use English;
use strict;
use Cwd;
use File::Basename;
use File::Spec;


my $BASE = $ENV{'PWD'};

#
# Usage Function
#
sub usage
{
  printf("Usage:\n"); 
  printf("\t$PROGRAM_NAME -homes /path/homefile.txt -out /patch/outoutfile -jre <path to executable jre>\n\n");
  printf("\thomes\n");
  printf("\tThe homes is a mandtory option which points a text file,\n");
  printf("\teach line contains the path to the Oracle Home.\n\n");
  printf("\tout\n");
  printf("\tThe out is a mandtory option which points where the output of the script goes to.\n\n");
  printf("\tjre\n");
  printf("\tThe jre is optional option, which points the path of executable jre location,\n");
  printf("\tso OPatch will run using your give JRE instead of OPatch searching for it.\n\n");
  exit 1;
}

#
# Detect system platform
#

sub isWindows
{
  return ($OSNAME =~ m#Win#);
}

#
# Make sure there is an opatch installed in ORACLE_HOME
#
my $ORACLE_HOME = $ENV{'ORACLE_HOME'};
my $OPATCH = File::Spec->catfile($ORACLE_HOME, "opatch", "OPatch");
if( ! -d $OPATCH)
{
  printf("Patch Inventory Command can't find OPatch in Oracle Home $ORACLE_HOME.\n");
  exit 2;
}

my $ARG_NUM = @ARGV;

#
# Get all arguments
#

my $HOMES_FILE = "";
my $OUT_FILE = "";
my $JRE = "";

for(my $i = 0; $i < $ARG_NUM; ++$i)
{
  if($ARGV[$i] eq "-homes")
  {
     $i++;
     $HOMES_FILE = $ARGV[$i];
  }
  elsif($ARGV[$i] eq "-out")
  {
    $i++;
    $OUT_FILE = $ARGV[$i];
  }
  elsif($ARGV[$i] eq "-jre")
  {
    $i++;
    $JRE=$ARGV[$i];
  }
}


#
# Validate mandtory option
#
if (!$HOMES_FILE || !$OUT_FILE)
{
  &usage;
}

if (! -f $HOMES_FILE)
{
  printf("The homes option's value '%s' is not valid.\n",$HOMES_FILE);
  exit 3;
}

#
# Validate jre option
#

if($JRE)
{
  my $JAVA_BIN = File::Spec->catfile($JRE,"","bin");
  my $JAVA;
  if( isWindows() )
  {
    $JAVA = File::Spec->catfile($JAVA_BIN,"","java.exe");
  }
  else
  {
    $JAVA = File::Spec->catfile($JAVA_BIN,"","java");
  }
  if ( ! -x $JAVA)
  {
    printf("$JRE is not a valid jre executable location, please set the correct jre location.\n");
    exit 4;
  }
}

#
# Validate the the opatch in ORACLE_HOME has a lspatches command
#
my $opatch;
my $lspatches;

if( isWindows() )
{
  $opatch = File::Spec->catfile($OPATCH,"opatch.bat");
}
else
{
  $opatch = File::Spec->catfile($OPATCH,"opatch");
}

if($JRE)
{
  $lspatches = `$opatch -help -oh $ORACLE_HOME -jre $JRE`;
}
else
{
  $lspatches = `$opatch -help -oh $ORACLE_HOME`;
}

if ( $? || ! ($lspatches =~ /lspatches/) )
{
  printf("Patch Inventory Command can't find 'opatch lspatches' in '$opatch'.\n");
  exit 5;
}


#
# Read Oracle Home from homs_file
# And For each oracle home, do 'opatch lspatches' command
#

open (HOMES_IN, $HOMES_FILE) or die "can't open the Oracle Home List file $HOMES_FILE";
if (! open (OUT_IN, ">$OUT_FILE"))
{
   printf("The out option's value '%s' is not valid.", $OUT_FILE);
   close HOMES_IN;
   exit 6;
}

print OUT_IN "----- START OF REPORT -----\n";

my @HOMES = <HOMES_IN>;
foreach my $home (@HOMES)
{
  chomp $home;
  if( ! -d $home )
  {
    printf("Oracle Home %s is not a directory, will omit it.\n", $home);
  }
  else
  {
    print OUT_IN "--------------------------------\n";
    print OUT_IN "ORACLE_HOME=$home\n";
    my $output = "";
    if($JRE)
    {
       $output=`$opatch lspatches -oh $home -jre $JRE`;
    }
    else
    {
       $output=`$opatch lspatches -oh $home`;
    }
    print OUT_IN $output;
#    if( ! $? && !($output =~ /no Interim/) )
#    {
#      print OUT_IN $output;
#    }
  }
}

close HOMES_IN;
print OUT_IN "----- END OF REPORT -----\n";
close OUT_IN;
print "Patch Inventory Command Completeed success.";
exit 0;
