# $Header: emagent/sysman/admin/scripts/hostServiceTags.pm /main/2 2010/10/15 09:55:17 nparaddi Exp $
#
# Package : hostServiceTags.pm
#
# Copyright (c) 2009, 2010, Oracle and/or its affiliates. All rights reserved. 
#
#   NAME
#       hostServiceTags.pm
#
#   DESCRIPTION
#       Service Tag Utilities
#
#   NOTES
#
#   MODIFIED     (MM/DD/YY)
#      nparaddi   10/05/10 - change semd_common semd_common_ocm
#      ndutko     04/05/10 - Strip off the TZ as the servicetag information is
#                            always in GMT and the data definition for the
#                            INSTALLATION_TIME is a TIMESTAMP (no TZ)
#      ndutko     02/25/10 - correct problem where /usr/bin/grep does not exist
#      name       12/01/09 - Creation
#
##*************************************************************

package hostServiceTags;

use strict;
use FileHandle;
use ias::simpleXPath;
require "semd_common_ocm.pl";

use Carp;
require 5.005;

#require 5.6.1;
require Exporter;

#******************************************
#     Global Variables
#******************************************

use constant NIL              => '';
use constant FALSE            => 'FALSE';
use constant TRUE             => 'TRUE';
use constant SUFFIX           => '.xml';
use constant AGENT_URN        => 'agent_urn';
use constant AGENT_VERSION    => 'agent_version';
use constant REGISTRY_VERSION => 'registry_version';
use constant PRODUCT_URN      => 'product_urn';
use constant INSTANCE_URN     => 'instance_urn';
use constant SERVICE_TAG      => 'service_tag';
use constant LINUXPATH        => '/opt/sun/servicetag/bin/stclient';
use constant LINUXREGEX       => 'linux';
use constant WINDOWSPATH  => 'C:\Program Files\Sun\servicetag\stclient.exe';
use constant WINDOWSREGEX => 'mswin';
use constant WINDOWSREGISTRY => ("HKEY_LOCAL_MACHINE/SOFTWARE/Sun Microsystems/ServiceTags/stclient.exe"
				,"HKEY_LOCAL_MACHINE/SOFTWARE/Microsoft/Windows/CurrentVersion/App Paths/sunstclient.exe"
				,"HKEY_LOCAL_MACHINE/SOFTWARE/Microsoft/Windows/CurrentVersion/App Paths/stclient.exe");
use constant SOLARISPATH  => '/usr/bin/stclient';
use constant SOLARISREGEX => 'solaris';

# to match 1.1.4 and greater in the 1.1.x release series
# and 1.2 and greater
# and 10.x and greater
use constant VERSIONREGEX => '(^1\.1\.[4-9]|^[1-9]\.[2-9]|^[1-9][0-9])';
use constant REGISTRY     => 'registry';
use constant INSTANCE     => 'instance';
use constant DELIM        => '|';
use constant DELIM_SPLIT  => '\|';
use constant DATASOURCE  => 'SERVICE_TAG';

#******************************************
#   Export Subroutines
#******************************************

@hostServiceTags::ISA    = ('Exporter');
@hostServiceTags::EXPORT = qw(
  &getAgentXml
  &getServiceTagsXml
  &getSystemSerialNumber
  &getAgentUrn
  &getAgentVersion
  &getRegistryVersion
  &getServiceTagEmResults
);

#******************************************
#     Global Variables
#******************************************

use constant NIL => '';
use constant SEP => ':';

#******************************************
#     Exported Subroutines
#******************************************
sub getAgentXml {
	my $stClientPath = getStClientPath();
	if ( $stClientPath eq NIL ) {
		return NIL;
	}

	# get the agent data
	my $agentXml = `$stClientPath -E`;

	return $agentXml;
}

sub getServiceTagsXml {
	my $stClientPath = getStClientPath();
	if ( $stClientPath eq NIL ) {
		return NIL;
	}

	# get the serviceTags data
	my $serviceTagsXml = `$stClientPath -x`;

	return $serviceTagsXml;
}

sub getServiceTagEmResults
{
    my @results;

    my $xml = getServiceTagsXml();
    if ($xml eq NIL) {
        return @results;
    }

    my $productUrn = NIL;
    my $productName = NIL;
    my $vendorName = NIL;
    my $version = NIL;
    my $installDtt = NIL;
    my $installLoc = NIL;
    my $vendorSpecificInfo = NIL;
    my $virtMachine = NIL;
    my $softwareArch = NIL;
    my $productParentUrn = NIL;
    my $productParentName = NIL;
    my $mediaType = NIL;
    my $yearMonthDay = NIL;
    my $hourMinSec = NIL;

    my @serviceTags = simpleXPathQueryForTextNodes($xml, "registry/service_tag");
    for my $serviceTag (@serviceTags) {
        for my $childNode (@{$$serviceTag{'children'}}) {
            if ($$childNode{'tagName'} eq "") {
                next;
            }
            my $tagName = $$childNode{'tagName'};

            my $tagValue = "";
            my @children = $$childNode{'children'};
            for my $childRef (@children) {
                $tagValue = getStringValue($childNode);
            }
            if ($tagName eq "product_name") {
                $productName = $tagValue;
            } elsif ($tagName eq "product_version") {
                $version = $tagValue;
            } elsif ($tagName eq "product_urn") {
                $productUrn = $tagValue;
            } elsif ($tagName eq "product_parent_urn") {
                $productParentUrn = $tagValue;
            } elsif ($tagName eq "product_parent") {
                $productParentName = $tagValue;
            } elsif ($tagName eq "product_defined_inst_id") {
                $vendorSpecificInfo = $tagValue;
            } elsif ($tagName eq "product_vendor") {
                $vendorName = $tagValue;
            } elsif ($tagName eq "platform_arch") {
                $softwareArch = $tagValue;
            } elsif ($tagName eq "timestamp") {
                ($yearMonthDay, $hourMinSec) = split(" ", $tagValue);
                $installDtt = "$yearMonthDay $hourMinSec";
            } elsif ($tagName eq "container") {
                $virtMachine = $tagValue;
            } elsif ($tagName eq "source") {
                $mediaType = $tagValue;
            }
        }
        my $emResult = "em_result="
            . $productUrn
            . DELIM . $productName
            . DELIM . $vendorName
            . DELIM . $version
            . DELIM . $installDtt
            . DELIM . $installLoc
            . DELIM . $productName
            . DELIM . $vendorSpecificInfo
            . DELIM . $virtMachine
            . DELIM . $softwareArch
            . DELIM . $productParentUrn
            . DELIM . $productParentName
            . DELIM . $mediaType
            . DELIM . DATASOURCE . "\n";

        push(@results, $emResult);
    }
    return @results;
}


sub getSystemSerialNumber {
	return getXMLElementValue( getAgentXml(), "agent/system_info",
							   "serial_number" );
}

sub getAgentUrn {
	return getXMLElementValue( getAgentXml(), "agent", "agent_urn" );
}

sub getAgentVersion {
	return getXMLElementValue( getAgentXml(), "agent", "agent_version" );
}

sub getRegistryVersion {
	return getXMLElementValue( getAgentXml(), "agent", "registry_version" );
}

#******************************************
#     Private Subroutines
#******************************************

sub getXMLElementValue {
	my $xml = $_[0];
	if ( $xml eq NIL ) {
		return NIL;
	}
	my @agents = simpleXPathQueryForTextNodes( $xml, $_[1] );

	my $targetTagName = $_[2];

	for my $agent (@agents) {

		for my $childNode ( @{ $$agent{'children'} } ) {

			if ( $$childNode{'tagName'} eq "" ) {
				next;
			}

			my $tagName  = $$childNode{'tagName'};
			my $tagValue = "";
			my @children = $$childNode{'children'};

			for my $childRef (@children) {
				$tagValue = getStringValue($childNode);
			}
			if ( $tagName eq $targetTagName ) {
				return $tagValue;
			}
		}
	}
	return NIL;
}

# helper function to extract element value
sub getStringValue {
	my $nodeRef = $_[0];

	if ( ${$nodeRef}{'type'} eq 'text' ) {
		return ${$nodeRef}{'text'};
	}

	my @childText = map( getStringValue($_), @{ $$nodeRef{'children'} } );
	return join( '', @childText );
}

sub getWinInstLocation {
	no strict;
	eval 'use Win32::TieRegistry;';
	$Registry->Delimiter("/");
	my $instLoc =
	  $Registry->{"$_[0]"};
	return $instLoc->{"/"};
}

sub getStClientPath() {

	# determine operating system and set os dependent parameters
	my $stclientPath = NIL;
	if ( get_osType() eq 'LNX' ) {

		# try default location first since it's faster
		if ( locationCheck(LINUXPATH) ne NIL ) {
			return LINUXPATH;
		}

		# try to find the stclient location using rpm commands
		my $stclientLocation;
                my @svctagLocation = ();
                my @results = `/bin/rpm -qa`;
                my $loc;
                for $loc (@results)
                {
                   chomp($loc);
                   if ($loc =~ /servicetag/i) { push (@svctagLocation, $loc); }
                }
		if ( @svctagLocation == 1 ) {

			# only 1 matching pkg
			$loc = getClientPath($svctagLocation[0]);
			return locationCheck($loc);
		} else {
			# more than one match
			for my $path (@svctagLocation) {
				chomp($path);
				my $loc = getClientPath($path);
				if ( locationCheck($loc) ne NIL ) {

					# found a valid client
					return $loc;
				}
			}
		}

		# couldn't find stclient
		return NIL;

	} elsif ( get_osType() eq 'SOL' ) {

		# try default location first
		if ( locationCheck(SOLARISPATH) ne NIL ) {
			return SOLARISPATH;
		}

		# look in contents to find the path
                my $loc = NIL;
                if (-e '/var/sadm/install/contents')
                {
                    my $tmpPath;
                    open SADM,"</var/sadm/install/contents";
                    while ($tmpPath=<SADM> and $loc ne NIL)
                    {
                        chomp($tmpPath);
                        if ($tmpPath =~ /bin\/stclient /)
                        {
                            $loc = $tmpPath;
                        }
                    }
                    close SADM;
                }

		# should be first in the split array
		my @locSplit = split(/ /, $loc );
		chomp( $locSplit[0] );
		return locationCheck( $locSplit[0] );
	} elsif ( get_osType() eq 'WIN' ) {
		# try default location first since it's faster
		if ( locationCheck(WINDOWSPATH) ne NIL ) {
			return locationCheck(WINDOWSPATH);
		}

		# get install location from registry
		for my $reg (WINDOWSREGISTRY) {
			my $regvalue = getWinInstLocation($reg);
			chomp($regvalue);
			if ( locationCheck($regvalue) ne NIL ) {
				return locationCheck($regvalue);
			}

		}
	}

	# probably not a supported OS
	return NIL;
}

sub getClientPath()
{
    my $path = $_[0];
    my @results = `/bin/rpm -ql $path`;
    for my $loc (@results)
    {
        chomp($loc);
        if ($loc =~ /bin\/stclient$/)
        {
            return $loc;
        }
    }
    return NIL;
}

sub locationCheck() {
	my $stclientPath = $_[0];
	chomp($stclientPath);

	# make sure stclient exists
	if ( -e $stclientPath ) {

		# version check
		#return  versionCheck();
		return versionCheck($stclientPath);
	}
	return NIL;
}

sub versionCheck() {

	# version check
	my $stclientPath = $_[0];

	# add quotes if windows
	if ( get_osType() eq 'WIN' ) {
		$stclientPath = '"' . $stclientPath . '"';
	}
	my $stVersion = `$stclientPath -v`;
	if ( $stVersion =~ /${\(VERSIONREGEX)}/ ) {
		return $stclientPath;
	}
	return NIL;
}

