#!/usr/local/bin/perl
# 
# $Header: emagent/sysman/admin/scripts/import_cert_to_agent_trust.pl /st_emagent_10.2.0.1.0/3 2010/01/09 03:46:18 sparmesw Exp $
#
# import_cert_to_agent_trust.pl
# 
# Copyright (c) 2009, 2010, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      import_cert_to_agent_trust.pl - <one-line expansion of the name>
#
#    DESCRIPTION
#      Import trust certificate to EMAgent's AgentTrust.jks
#
#      Requirements:
#         Requires ORACLE_HOME and either EMSTATE or EMDROOT env variables set
#
#      Usage:
#         import_cert_to_agent_trust.pl "-password <pwd>"
#                          
#                         OR
#
#         import_cert_to_agent_trust.pl 
#            <Enter <pwd> on stdin when prompted>
#
#       Optional Usage: (Additional arguments supported)
#
#         import_cert_to_agent_trust.pl "-password <pwd> [
#                                        -keystore_file <TruststoreFile>
#                                        -alias <Alias of cert to import>
#                                        -trust_certs_loc <Cert file to import>]"
#
#    NOTES
#      <other useful comments, qualifications, etc.>
#
#    MODIFIED   (MM/DD/YY)
#    sparmesw    01/07/10 - Bug 9263989 on NT
#    sparmesw    01/04/10 - Handle call from install script
#    sparmesw    12/07/09 - Creation
# 

use IPC::Open2;
require "emd_common.pl";
#Code to read passwd from stdin is used from below
require "$ENV{ORACLE_HOME}/bin/EmctlCommon.pm";
require "$ENV{ORACLE_HOME}/bin/SecureUtil.pm";

my $alias;
my $password;
my $trust_certs_loc;
my $keystoreFilename;
my $oraHome = $ENV{ORACLE_HOME};
my $state_root;
my $isVerbose;

if ( $ENV{EMSTATE} eq "" ) {
    $state_root = $ENV{EMDROOT};
} else {
    $state_root = $ENV{EMSTATE};
}


# Define defaults for input params
# Default keystore is the agent's jks based trust store.
$keystoreFilename = $state_root."/sysman/config/montrust/AgentTrust.jks";
$trust_certs_loc = $state_root."/sysman/config/b64WLSDemoTrustCACert.txt";

$alias = "wlscertgencab";
$isVerbose = "false";

#--------------------------------------------------------------------------------------------------
EMD_PERL_DEBUG("\nimport_cert_to_agent_trust.pl: Step#1 - Collecting INPUT parameters....");
#--------------------------------------------------------------------------------------------------

$i=0;
$cmdString = "";
foreach (@ARGV)
{
  $cmdString = $cmdString . " " .$ARGV[$i];
  $i++;
}

foreach $t (split("-",$cmdString))
{
    my @arg = split(" ", $t,2);
    if ($arg[0] eq "keystore_file")
    {
        $keystoreFilename = trim($arg[1]);
    }
    elsif ($arg[0] eq "trust_certs_loc")
    {
        $trust_certs_loc = trim($arg[1]);
    }
    elsif ($arg[0] eq "alias")
    {
        $alias = trim($arg[1]);
    }
    elsif ($arg[0] eq "password")
    {
        $password = trim($arg[1]);
    }
    elsif ($arg[0] eq "verbose")
    {
        $isVerbose = trim($arg[1]);
    }
    elsif ($arg[0] eq "help")
    {
        usage();
        exit 1;
    }
}

if (-t STDIN)
{
   if (!$password)
   {
     $password = EmctlCommon::promptUserPasswd ("Password: ");
   }
} else {
   if (!$password)
   {
     $password = <STDIN>;
     chomp($password);
   }
}


EMD_PERL_DEBUG("\nimport_cert_to_agent_trust.pl: INPUT => Oracle Home = $oraHome");
EMD_PERL_DEBUG("\nimport_cert_to_agent_trust.pl: INPUT => keystore_file = $keystoreFilename");
EMD_PERL_DEBUG("\nimport_cert_to_agent_trust.pl: INPUT => alias = $alias");
EMD_PERL_DEBUG("\nimport_cert_to_agent_trust.pl: INPUT => trust_certs_loc = $trust_certs_loc");

if ($isVerbose eq "true")
{
  print "\n Command argument string = $cmdString";
  print("\n Oracle Home = $oraHome");
  print("\n EM state dir = $state_root");
  print("\n keystore_file = $keystoreFilename");
  print("\n trust_certs_loc = $trust_certs_loc");
  print("\n alias = $alias");
}

chomp($os = $^O);

#--------------------------------------------------------------------------------------------------------
EMD_PERL_DEBUG("\nimport_cert_to_agent_trust.pl: Step#2 - Setting Execution Environment Variables...");
#--------------------------------------------------------------------------------------------------------

my $oraHome = $ENV{ORACLE_HOME};
my $command;
if (!$oraHome)
{
  $command = "jdk/bin/keytool ";
} else {
  $command = $oraHome."/jdk/bin/keytool ";
}

$command = $command." -import -alias ".$alias." -file ".$trust_certs_loc." -keystore ".$keystoreFilename." -noprompt 2>&1";

#-------------------------------------------------------------------------------------------------------------
EMD_PERL_DEBUG("\nimport_cert_to_agent_trust.pl: Step#3 - Executing the following Command... \n$command\n");
#-------------------------------------------------------------------------------------------------------------
if ($isVerbose eq "true")
{
  print("\n\n Executing the following Command: \n$command\n\n");
}

local(*READ, *WRITE);
my $pid = open2(\*READ, \*WRITE, $command);
print WRITE "$password\n";
close(WRITE);

my @cmdOutput;

while(<READ>) {
	push(@cmdOutput, $_);
}
close(READ);
waitpid($pid, 0);


#-------------------------------------------------------------------------------------------------------------
EMD_PERL_DEBUG("\nimport_cert_to_agent_trust.pl: Step#4 - Execution Complete. Composing result.\n");
#-------------------------------------------------------------------------------------------------------------
#Variable to store any other unknown error messages
my $errorLine; 

my $keytool_prompt = "Enter keystore password:";
my $line;
foreach  my $line1 (@cmdOutput) {
	EMD_PERL_DEBUG("import_cert_to_agent_trust.pl!!! .. ONE_LINE: ".$line);
        my $idx = index($line1,$keytool_prompt);
        my $len = length($keytool_prompt);
        if ($idx >=0)
        {
           $line = substr($line1,$len);
        } else {
           $line = $line1;
        }
	if ($line =~ m/FileNotFoundException/ )	{
		EMD_PERL_ERROR("import_cert_to_agent_trust.pl!!! .. could not load trust certificate: ".$line);
		printMessage("FILE_NOT_FOUND", $line, @cmdOutput);
		exit(1);
	}elsif ($line =~ m/password is too short/ ) {
		EMD_PERL_ERROR("import_cert_to_agent_trust.pl!!! .. AuthenticationException Occured: ".$line);
		printMessage("AUTHENTICATION_FAILED", $line, @cmdOutput);
		exit(1);
	}elsif ($line =~ m/Exception/ ) {
		$errorLine = $line;
	}
	if ($line =~ m/was added/ ) {
		EMD_PERL_DEBUG("import_cert_to_agent_trust.pl!!! .. Operation Successful.");
		printMessage("SUCCESS", $line, @cmdOutput);
		exit(0);
	} else {
           if ($line =~ m/already exists/ ) 
           {
		EMD_PERL_DEBUG("import_cert_to_agent_trust.pl!!! .. Operation Successful as alias already exists in keystore.");
		printMessage("SUCCESS", $line, @cmdOutput);
		exit(0);
           }
	}
}

printMessage("FAILED", $errorLine, @cmdOutput);
exit(1);

sub printMessage()
{
	(my $ExitStatus, my $StatusMessage, my @OutputTrace) = @_;
	EMD_PERL_DEBUG("import_cert_to_agent_trust.pl: Message: ".@OutputTrace);
        if ($isVerbose eq "true")
        {
           print("Keytool output: \n");
           foreach  my $line1 (@OutputTrace) {
	      print("$line1\n");
           }
        }
        print "\nMessage   : $StatusMessage";
	print "ExitStatus: $ExitStatus\n\n";
}

sub usage
{
    print <<_EOF_
\n     Usage:  import_cert_to_agent_trust.pl "password <pwd to AgentTrust.jks> [-alias <Alias of cert to import> -trust_certs_loc <Cert file to import>]"

           NOTE:  If Password is NOT specified on the commandline it needs to be specified on the stdin when prompted.\n
_EOF_

}

#trims the space
sub trim {
    my @out = @_;
    for (@out) {
        s/^\s+//;
        s/\s+$//;
    }
    return wantarray ? @out : $out[0];
}
