#!/usr/local/bin/perl
# 
# $Header: emagent/sysman/admin/scripts/jobutil/JobDiagUtil.pm /main/14 2012/03/16 09:26:45 rdabbott Exp $
#
# JobDiagUtil.pm
# 
# Copyright (c) 2010, 2012, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      JobDiagUtil.pm - Perl helper APIs for diagnosability
#
#    DESCRIPTION
#      APIs to help job and paf integrators make use of the diag framework
#
#    APIs
#      getIncidentDumpFileDir
#          return the dir into which incident dump files should go
#      getIncidentExitCode
#          return the "magic" incident exit code
#      triggerIncidentWithFilesNLS
#          trigger an incident w/ a description and list of related dump files.
#      triggerIncidentWithFiles
#          (Deprecated) 
#      triggerIncidentByID
#          trigger an incident.  Include a local root incident.
#      triggerIncident
#          trigger an incident.  Uses optional named params.
#      associateRemoteLogFiles
#          associate logfiles with tihs step.
#      associateRemoteLogDirFiles
#          associate some logfiles in a directory with tihs step.
#      getLoggingLevel
#          return the current logging level
#      logIncident
#          log a system visible incident object message 
#      logSevere
#          log a user visible severe oblect message
#      logWarning
#          log a user visible warning oblect message
#      logInfo
#          log a user visible info oblect message
#      logFine
#          log a user visible fine level oblect message
#      log
#          log an object message at level with more options
#      printCommandBlock
#          helper API to print a command block
#
#    ENV SETTINGS
#      Some APIs rely on or set values into the environment
#      See "ENV Variables" for details.
#
#    OPEN ISSUES
#      getIncidentDumpFileDir     - need to fix algorithm for this
#      getLoggingLevel            - nyi
#
#    NOTES
#      Project 29520
#
#    TEST SUITES
#      emcore/test/src/emdrep/olog/tvmrolog.tsc
#      emcore/test/src/emdrep/jobs/unittests/diag/tvmrjdg.tsc
#      emcore/test/src/emdrep/jobs/systemtest/jdiag/tvmrjjd.tsc
#
#    MODIFIED   (MM/DD/YY)
#    rdabbott    01/20/12 - fix 13603451: create dir on windows
#    rdabbott    07/21/11 - fix 12703055: escape non ascii chars
#    jsoule      07/06/11 - change magic incident exit code to a valid one
#    rdabbott    06/23/11 - require msgid and res bundle
#    rdabbott    06/17/11 - add rb override
#    rdabbott    05/04/11 - fix 10191930: integrator defined problem key
#    rdabbott    03/14/11 - Fix 10350922: no eoln after command block end tag
#    rdabbott    11/24/10 - bug 10317022: fix dir creation
#    lsatyapr    10/02/10 - All cmdblk output to stderr
#    rdabbott    09/23/10 - make the diag dump dir unique per step
#    rdabbott    07/06/10 - default to user visible for logging apis
#    rdabbott    06/25/10 - sys_incident -> incident
#    rdabbott    05/08/10 - add integrator doc
#    rdabbott    04/26/10 - use new constant values
#    rdabbott    03/26/10 - test incident w files
#    rdabbott    03/22/10 - use step target for associated incidents
#    rdabbott    03/19/10 - add incident level logging
#    rdabbott    03/12/10 - file times
#    rdabbott    03/09/10 - impl logging apis
#    rdabbott    03/09/10 - fill in IMPL
#    rdabbott    02/23/10 - review
#    rdabbott    02/04/10 - Creation
# 

package JobDiagUtil;


use strict;

use Carp;
use Cwd;
use File::stat;
use File::Spec;
use File::Path;

#use Sys::hostname;
#use Time::localtime;


require 5.005;


@JobDiagUtil::EXPORT = qw(
  &getIncidentDumpFileDir
  &getIncidentExitCode
  &triggerIncidentWithFilesNLS
  &triggerIncidentWithFiles
  &triggerIncidentByID
  &triggerIncident
  &associateRemoteLogFiles
  &associateRemoteLogDirFiles
  &getLoggingLevel
  &logIncident
  &logSevere
  &logWarning
  &logInfo
  &logFine
  &log
  &printCommandBlock
  );


#******************************************
#     ENV Variables
#******************************************

# Environment variables used by this script

# The "primary" correlation key for incidents
use constant ENV_ECID                  => 'EM_JOB_DIAG_ECID';

# The problem key for incidents
use constant ENV_PROBLEM_KEY           => 'EM_JOB_DIAG_PROBLEM_KEY';

# The location for incident dump scripts to be put
use constant ENV_DUMP_DIR              => 'EM_JOB_DIAG_DUMP_DIR';

# The logging level for this step
use constant ENV_LOG_LEVEL             => 'EM_JOB_DIAG_LOG_LEVEL';

# The exit code which would trigger an incident
use constant MAGIC_INCIDENT_EXIT_CODE  => 42;

#******************************************
#     Global Variables
#******************************************

use constant SEP => ':';

use constant START_CMD_BLOCK => '$$$--*$$';
use constant END_CMD_BLOCK   => '$$$*--$$';

# logging levels
use constant INCIDENT_LEVEL => '4';
use constant SEVERE_LEVEL   => '3';
use constant WARNING_LEVEL  => '2';
use constant INFO_LEVEL     => '1';
use constant FINE_LEVEL     => '0 '; # add a trailing space so it tests as true

# visibility
use constant VISIBLE_SUPPORT_ONLY  => '1';
use constant VISIBLE_USER          => '2';
use constant VISIBLE_SYSTEM        => '3';

#******************************************
#     Internal Subroutines (helpers)
#******************************************

# indent printing of command blocks for readability
use constant INDENT   => '   ';

# pass a null arg to pl/sql call
use constant NULL_ARG => 'null';

# print some command block text w/o newline
#
# Arg: List of elements to print
sub printCb
{
    print STDERR @_;
# print command blocks to stdout
#   print @_ if index(@_[0], '$$$') < 0;
}

# print one line for a command block
# Always append a newline
#   No checks on passed values
#
# Arg: List of elements to print on one line
sub printCbLn
{
    printCb @_, "\n";
}

# encode passed text it contains spl XML chars
#   Wrap any "unusual" character in CDATA
sub escapeXML
{
    $_ = shift;
    $_ = "<![CDATA[$_]]>" if not ( $_ =~ '^[A-Za-z_0-9%.* \n\t/\-]*$' );
    return $_;
}

# print one command block arg
#   (do nothing if no args supplied)
#
# Arg: For records or arrays, prefix with 
#     record, <count>, <value>, <value>, ...
#     array,  <count>, <value>, <value>, ...
# For scalar, 
#     scalar, <value>
# OR just
#     <value>
#
# Return: Remaining args, if any
sub printCbArg
{
    my $indent = shift or die("must provide indent");
    my $arg    = shift or return;

    if ( "record" eq $arg or "array" eq $arg )
    {
        my $count = shift;
        printCbLn "$indent<$arg>";

        for (my $i = 1; $i <= $count; $i++) 
        {
            my $value = shift or die("must provide value[$i] for $indent$arg");
            @_ = &printCbArg($indent . INDENT, $value, @_);
        }

        printCbLn "$indent</$arg>";
    }
    else
    {
        if ( /^scalar/ )  # do not expect this
        {
            $arg = shift or die("must provide scalar value");
        }

        printCbLn "$indent<scalar>" . escapeXML($arg) . "</scalar>";
    }

    # return remaining args for next pass
    return @_;
}

# Print a command block.
# This API prints a command block into the output.  
# If the remoteOp command that launched this process is configured 
# to accept command blocks, it would be intepreted.
# If the procName specified is registered as a valid command block, 
# and if the arg list passed here matches the expected arg list, 
# and if the user running the command has privilage to run procName,
# then procName would be run in pl/sql in the repository.
#
# Args: procName
#       [procArgs]
# To specify optional procArgs:
# For records or arrays, 
#     record, <count>, <value>, <value>, ...
#     array,  <count>, <value>, <value>, ...
# For scalar, 
#     scalar, <value>
# OR just
#     <value>
#
# Return: none
sub printCommandBlock 
{
    my $prcName = shift or die("must specify a procedure name");
    $prcName = escapeXML($prcName);

    printCbLn START_CMD_BLOCK;
    printCbLn "<commandBlock>";
    printCbLn INDENT . "<executeProc name=\"$prcName\">";

    # assume step id is always first
    &printCbArg ( INDENT . INDENT, '%job_step_id%' );

    while ( @_ = &printCbArg(INDENT . INDENT, 
                             @_) )
    {
    }

    printCbLn INDENT . "</executeProc>";
    printCbLn "</commandBlock>";
    printCb   END_CMD_BLOCK;   # bug 10350922: no EOLN for END_CMD_BLOCK
}


# Get a command block arg.
#
# Args: argValue
#
# return the arg or NULL_ARG if the arg is not set
sub getCommandBlockArg
{
    my $arg = shift;
    $arg = NULL_ARG if (!$arg);
    return $arg;
}


# Convert the params into a command block array 
#
# Args: elements of the array
#
# return the array or NULL_ARG if no args
sub getArray
{
    return ("array", 0) if not $_[0];
    my $count = scalar(@_);

#print "ARRAY($count)=@_\n";
    return ("array", $count, @_);
}


# Get the current logged in user 
# getlogin should normally work, but if not, this also checks 
# for $USER and $USERNAME
#
# Args: none
#
# return the user
sub getCurrentUser
{
    # for some reason getlogin does not work when run as a step
    # perhaps this is related to how the agent launches the process?
    # try env for user or username as a backup plan
    my $user = getlogin;
    $user =  $ENV{'USER'}     if not $user;
    $user =  $ENV{'USERNAME'} if not $user;

    return $user;
}


# Get the home dir for the current user
#
# return this user's home dir
sub getHomeDir
{
# we do not have File::HomeDir installed :(
# return File::HomeDir->my_home;

# Next best thing is the env vars - several sites mention HOME and USERPROFILE
    return "$ENV{HOME}"        if "$ENV{HOME}";
    return "$ENV{USERPROFILE}" if "$ENV{USERPROFILE}";

# websites also mention HOMEDRIVE and HOMEPATH for some versions of windows
    if ( "$ENV{HOMEDRIVE}" and "$ENV{HOMEPATH}" ) 
    {
        # only create the path if the pieces exist, 
        # otherwise we could create "/" as the path
        return File::Spec->catpath( "$ENV{HOMEDRIVE}", "$ENV{HOMEPATH}", '',);
    }

    return "";
}


# Get the target name, target type and hostname.
#
# Note: We need to get the actual host.
#       If this is a rac or cluster, the master agent could change 
#       and we would not find the file.
#
# Args: none
#
# return %target_name%, "%target_type%', hostname
#        is a problem getting the hostname, pass NULL_ARG for the hostname
#
sub getTargetInfo
{
# Sys/hostname.pm not found ?!    
#   my $host = &Sys::hostname;

    my $host = `hostname`;
    $host =~ s/\s+$//;

    my $user = &getCurrentUser();

    return ('%target_name%', 
            '%target_type%', 
            &getCommandBlockArg($host),
            &getCommandBlockArg($user));
}


# Force the agent to trigger an incident.
#
# params: problemKey   - unique problem key (for problem correlation)
#                        Must be defined by the caller
#
# return: this command exits the script and does NOT return to the caller
sub triggerAgentIncident
{
    my ($problemKey) = @_;

    $ENV{ENV_PROBLEM_KEY} = $problemKey;

    # for the agent to trigger an incident
    exit MAGIC_INCIDENT_EXIT_CODE;
}


# return the full path to the file(s) passed in
#
# params: file1 relative (or full) path to file1
#         file2 relative (or full) path to file2
#         ...
#
# return: the full path to the files passed in
sub getFullPath
{
    my @fullPath;
    my $found;
# print "getFullPath: @_\n";

    while ( $_ = shift )
    {
        if ( ! -e "$_" )
        {
            print STDERR "WARNING: ignoring file $_: not found\n";
            next;
        }

        if ( -z "$_" )
        {
            print STDERR "WARNING: file $_: is empty\n";
        }

# print "getFullPath: checking $_\n";
        #TODO: Use File::Spec->rel2abs($_) instead?
        $found = &Cwd::abs_path($_);
        if ( $found )
        {
            push ( @fullPath, $found );
        }
        else
        {
            print STDERR "WARNING: unable to determine path to file $_\n";
        }
    }

# print "getFullPath: @fullPath\n";
    return @fullPath;
}


# Get file info based on passed in file list
#
# Args: list of files
#
# return array info, list of full path to the files
sub getFileInfo
{
    my @files = &getFullPath(@_);

    return &getArray(@files);
}


# Get message info based on args
#
# Args: message (required)
#       resource_id
#
# return array info, list of full path to the files
sub getMessageInfo
{
    my $message   = shift;
    my $res_id    = shift;
    my $res_bndl  = shift;

    if ( ! $message )
    {
        print STDERR "ERROR: No message specified for logging\n";
    }

    return ( "record", 
             4,
             &getCommandBlockArg($message),
             &getCommandBlockArg($res_id),
             &getCommandBlockArg($res_bndl),
             &getArray(@_) );
}


# Return the absolute time.
#
# If a relative time is specified (+/- or p/n some number of minutes)
#     return the specified delta from "now"
# otherwise return the time passed in (assume it is the absolute time already.
#
# Arg: The absolute time in epoch format or +/- offset time (in seconds)
#
# return the absolute (cannonical) epoch time
sub absoluteTime
{
    my $time = shift or return;

    substr($time, 0, 1);

    $_ = substr($time, 0, 1);

    if ( /[-\+np]/ )
    {
        # user specified an offset, return the current time +/- that offset
        return $time + time();
    }

    return $time;
}


# Return the files within the start / end time range
#
# Both start and end time must be specified.
# If start is 0, all files pass the start time check
# If end   is 0, all files pass the end time   check
#
# Arg: startTime, endTime, files
#
# return the files that are within the time range
sub checkTimes
{
    my $start = shift;
    my $end   = shift;

    my @files;

    foreach my $file (@_)
    {
        my $mod = &stat($file)->mtime;

        # remove files from the file list that are before start
        next if ( $start and $mod < $start );

        # remove files from the file list that are after end
        next if ( $end   and $end < $mod );

        # keep files that pass the time tests
        push (@files, $file);
    }

    return @files;
}


# private global storing the dumpDir location
my $g_dumpDir = ' ';

# Try to create a $dir/$subDir as a user writable dir
# Return the created dir, or null if unable to create it
#
# params: dir    the parent directory
#         subDir the sub directory (optional)
#         world  r if the dir should be world readable
#                w if the dir should be world writable
#
# return: the sub dir if created, otherwise null 
#
# called from getIncidentDumpFileDir
sub mkUserDir 
{
    my $pDir   = shift or return;
    my $subDir = shift;
    my $world  = shift;
    my $mask   = oct(1700);

# print "world = $world\n";
    $mask = oct(1755) if 'r' eq $world;
    $mask = oct(1777) if 'w' eq $world;

# print "mask = $mask\n";
# print "Trying $pDir\n";

    # if we are creating a sub dir, check if we have a valid parent dir
    # if we are not creating a sub dir, no need to check parent dir
    if ( (not $subDir) or (-d "$pDir/") )
    {
        # parent dir exists - create sub dir
        $pDir = "$pDir/$subDir" if ($subDir);
# print "check pDir $pDir\n";

        if ( ! -d "$pDir/" )
        {
            mkpath "$pDir" or return;
            chmod $mask, "$pDir";
        }

        return $pDir if ( -d "$pDir/" );
    }

    # no luck - try something else
    return;
}


# Try to create a dummp dir "$parentDir/emdiag/$user/jobs/p_$$/" 
# Return the created dir, or null if unable to create it
#
# params: parentDir the parent dir
#         user      the current user (should be run as user for sudo / pdp)
#
# return: the sub dir if created, otherwise null 
#
# called from getIncidentDumpFileDir
sub createDumpDir 
{
    my $pDir = shift or return;
    my $user = shift or return;

# print "Trying $pDir\n";

    # is this a valid parent dir?
    if ( "$pDir" and -d "$pDir/" )
    {
# print "Dir exists!\n";
        # parent dir exists - create sub dirs
        my $sub  = "jobs/p_$$";

        $pDir    = &mkUserDir($pDir, "emdiag", 'w');
        $pDir    = &mkUserDir($pDir, $user,    'r') if $pDir;
        $pDir    = &mkUserDir($pDir, $sub)          if $pDir;

        return $pDir;
    }

    # no luck - try something else
    return;
}


#******************************************
#     Exported Subroutines
#******************************************


# returns the directory where incident dump files should be created.
# copy or create dump files in this directory before triggering an incident
#
# params: NONE
#
# return: the directory 
#
# See also: triggerIncidentWithFiles/NLS
sub getIncidentDumpFileDir 
{
# print "in getIncidentDumpFileDir\n";
# print "g_dumpDir = |$g_dumpDir|\n";
    # return g_dumpDir if already set
    return $g_dumpDir if ($g_dumpDir ne ' ');

    # Try to get the dump dir from the environemnt
    # (The agent SHOULD have set this for us - see bug 10162841)
    my $agPath = "$ENV{EM_DIAG_DUMP_DIR}";
    if ( $agPath and &mkUserDir($agPath) )
    {
# print "mkpath $agPath worked!\n";
        return $g_dumpDir = $agPath;
    }

    my $user = &getCurrentUser();
# print "user = |$user|\n";

    # Otherwise, find a good dump dir on our own 
    foreach my $dir ("$ENV{EM_DIAG_DUMP_ROOT}", 
                     "&getHomeDir()",
                     "$ENV{TEMP}", 
                     "$ENV{TMP}", 
                     "/tmp" )
    {
# print "About to try |$dir|\n";

        $g_dumpDir = &createDumpDir ( $dir, $user );
# print "g_dumpDir = |$g_dumpDir|\n";
        return $g_dumpDir if $g_dumpDir;
    }

    # last resort: use .
    $g_dumpDir = ".";
    return $g_dumpDir;
}

# return the "magic" incident exit code
sub getIncidentExitCode
{
    return MAGIC_INCIDENT_EXIT_CODE;
}

# trigger an incident at the agent.
#
# Force the agent to trigger an incident.
# Optionally associate a target incident and or dump files.
# The target associated with the step is assumed to be the incident target.
#
# This API uses NAMED parameters.  Pass in a hash with these params.
#
# NOTE: problemMsgId should be in a range reserved in
#       emdev/logging/reserved_message_numbers.txt
#       problemMsgBundle must have the following entry:
#       { "oracle.core.ojdl.logging.MessageIdKeyResourceBundle", "" },
#       See diag spec for more details.
#
# params: problemMsgId       - problem message id (for problem description)
#                              This is a unique message id for this problem.
#                              It will be part of the default problem key.
#                              Each error condition should have its own
#                              message id.
#                              Required
#         problemMsgBundle   - problem message bundle (for MsgId translation)
#                              Required
#         problemKey         - optional unique problem key 
#                              Default: problemMsgId 
#                              If incidentID is specified based on a target
#                              incident, the problemKeys should match.
#         correlationKeysRef - ref to a list of correlation keys and values
#                              correlation keys are name-value pairs like 
#                              'ECID', $ecid or 'SESSION', $mySession
#                              These should be in a flat list like this:
#                              ['ECID', $ecid, 'SESSION', $mySession]
#         dumpFilesRef       - ref to a list of dump files to associate 
#                              with this incident
#         incidentID         - id of incident triggered at a managed target
#                              only problemKey is meaningful with incidentID
#
# Examples:
#    triggerIncident({problemMsgId       => 'EM-12345',
#                     problemMsgBundle   => 'oracle.sysman.resource...',
#                     correlationKeysRef => ['ECID', $ecid, 'Key2', $value2],
#                     dumpFilesRef       => [file1]
#                    });
#    triggerIncident({problemKey         => 'ABC1',
#                     dumpFilesRef       => [file1, file2]
#                    });
#    triggerIncident({problemKey         => 'ABC2',
#                     incidentID         => '123
#                    });
#
# return: this command exits the script and does NOT return to the caller
sub triggerIncident
{
    my ($args) = @_;
#print "triggerIncident()\n";
#foreach my $k (keys %$args)
#{
#    print "\t$k => $args->{$k}\n";
#}

    if ( exists $args->{incidentID} )
    {
        &printCommandBlock("EM_JOB_DIAG.associate_remote_incident",
                           &getCommandBlockArg($args->{problemKey}),
                           &getCommandBlockArg($args->{incidentID}),
                           &getTargetInfo);
    }

    if ( exists $args->{dumpFilesRef} )
    {
        &printCommandBlock("EM_JOB_DIAG.associate_remote_incident_wf",
                           &getCommandBlockArg($args->{problemMsgId}),
                           &getCommandBlockArg($args->{problemMsgBundle}),
                           &getCommandBlockArg($args->{problemKey}),
                           &getArray          (@{$args->{correlationKeysRef}}),
                           &getArray          (@{$args->{dumpFilesRef}}),
                           NULL_ARG,    # incident id to be filled in by PBS
                           &getTargetInfo);
    }

    my $problemKey = $args->{problemKey} or $args->{problemMsgId};

    # verify args by first deleting expected ones
    delete $args->{problemMsgId};
    delete $args->{problemMsgBundle};
    delete $args->{problemKey};
    delete $args->{correlationKeysRef};
    delete $args->{dumpFilesRef};
    delete $args->{incidentID};

    my $warn = "WARNING: ignoring unexpected args to triggerIncident()\n";
    foreach my $k (keys %$args)
    {
        print STDERR "$warn    $k => $args->{$k}\n";
        $warn = "";
    }

    &triggerAgentIncident($problemKey);
}


# trigger an incident at the agent.
#
# Force the agent to trigger an incident and associate it with this one.
#
# params: problemKey       - problemKey (for problem description)
#         incidentID       - incident ID from the target
#
# return: this command exits the script and does NOT return to the caller
#
# Examples:
#    triggerIncidentByID('ABC2', 123);
#
sub triggerIncidentByID 
{
    my ($problemKey, $incidentID) = @_;
    &triggerIncident ({problemKey => $problemKey,
                       incidentID => $incidentID
                      });
}


# trigger an incident at the agent with an NLS description.
# (Recommended API)
#
# Force the agent to trigger an incident and associate the dumpfiles with it.
# The dump files must be in the incident dump file dir.
#
# NOTE: to set a problemKey or correlationKey use triggerIncident
#
# params: problemMsgId     - error message id (for problem description)
#         problemMsgBundle - problem message bundle (for MsgId translation)
#         dumpFiles        - list of dump files to associate with this incident
#
# return: this command exits the script and does NOT return to the caller
#
# Examples:
#    triggerIncidentWithFilesNLS('EM-12345',
#                                'oracle.sysman.resource...',
#                                file1,
#                                file2);
#    triggerIncident({problemMsgId       => 'EM-12345',
#                     problemMsgBundle   => 'oracle.sysman.resource...',
#                     correlationKeysRef => ['ECID', $ecid, 'KEY_2', $value2],
#                     dumpFilesRef       => [file1, file2]
#                    });
#
# Note: the PBS will catch this, log the files with the Incident Fwk,
#       get and incident number in return and call the repository API
#       with that incident number.
#
# See also: getIncidentDumpFileDir 
sub triggerIncidentWithFilesNLS
{
    my $problemMsgId     = shift;
    my $problemMsgBundle = shift;
    &triggerIncident ({problemMsgId       => $problemMsgId,
                       problemMsgBundle   => $problemMsgBundle,
                       dumpFilesRef       => [@_]
                      });
}


# trigger an incident at the agent.
# (Deprecated API, use triggerIncidentWithFilesNLS)
#
# Force the agent to trigger an incident and associate the dumpfiles with it.
# The dump files must be in the incident dump file dir.
#
# params: problemKey - unique problem key.  Must be defined by the caller.
#         dumpFiles  - list of dump files to associate with this incident
#
# return: this command exits the script and does NOT return to the caller
#
# Note: the OMS will catch this, log the files with the Incident Fwk,
#       get and incident number in return and call the repository API
#       with that incident number.
#
# See also: getIncidentDumpFileDir 
sub triggerIncidentWithFiles 
{
    my $problemKey = shift;
    &triggerIncident ({problemKey   => $problemKey,
                       dumpFilesRef       => [@_]
                      });
}


# Associate one or more remote log files with this step.
#
# The job framework will associate the specified log files with this step.
# For example, if the step runs a restore operation, the resulting logfile
# might be associated with this step.
#
# params: remoteFilePath  - the full path to a file
#         remoteFilePath2 - the full path to a file
#         remoteFilePath3 - the full path to a file
#         ...
#
# return: NONE
sub associateRemoteLogFiles
{
    &printCommandBlock("EM_JOB_DIAG.associate_remote_log_files",
                       &getFileInfo(@_),
                       &getTargetInfo);
}


# Associate some remote log files in a directory with this step.
#
# All log files matching filePattern in remoteDirPath within the time window
# will be associated with this step.
#
# Note: the filePattern is used for a grep on the file name.
#       'foo' would match any file that contains 'foo' in it.
#       To find files starting with foo, use '^foo'.
#       Standard perl regex can be used in the filePattern.
# Note: the startTime and endTime are specified in epoch time, the value 
#       returned by a call to time().
# Note: times may also be specified as an offset (in seconds) of the 
#       current time.  +/-<some_number>.  
#
# params: remoteDirPath - (required) the full path to the directory
#         filePattern   - (optional) the file pattern to match
#         startTime     - (optional) find files modified after startTime
#         endTime       - (optional) find files modified before endTime
#
# return: the number of files found
sub associateRemoteLogDirFiles  
{
    my ( $remoteDirPath, $filePattern, $startTime, $endTime ) = @_;

    if ( ! -e $remoteDirPath )
    {
        print STDERR "WARNING: dir $remoteDirPath: not found\n";
        return;
    }

    if ( ! -d $remoteDirPath )
    {
        print STDERR "WARNING: $remoteDirPath: exists but is not a directory\n";
        return;
    }

    # create a list of all files in remoteDirPath

    opendir(DIR, $remoteDirPath);
    my @files = readdir(DIR);
    closedir(DIR);
    if ( $filePattern )
    {
        @files = grep(/$filePattern/, @files);
    }

    map {$_ = File::Spec->catfile($remoteDirPath, $_)} @files;

    if ( $startTime )
    {
        # check format of startTime
        $startTime = &absoluteTime($startTime);
        $endTime   = &absoluteTime($endTime);
        $endTime   = '0' if (!$endTime);

        @files = &checkTimes($startTime, $endTime, @files);
    }

    if ( ! @files )
    {
        my $match;
        $match = " matching $filePattern" if ( $filePattern );
        $match = "$match in time window"  if ( $startTime );
        print STDERR "WARNING: $remoteDirPath: exists, but contains no files$match\n";
        return;
    }

    @files = sort(@files);

    &associateRemoteLogFiles(@files);

    return 0;
}


# returns the log level enabled for this step.
#
# returns one of:
#     INCIDENT_LEVEL
#     SEVERE_LEVEL
#     WARNING_LEVEL
#     INFO_LEVEL
#     FINE_LEVEL
#
# params: NONE
#
# return: the log level 
sub getLoggingLevel
{
    my $logLevel = $ENV{ENV_LOG_LEVEL};

    # return the logging level from the environment or a default
    return $logLevel if ($logLevel);

    return FINE_LEVEL;
    # todo: ? return SEVERE_LEVEL;
}


# Convert string log level to int.
#
# params: logLevel        - the log level for this message
#                           INCIDENT_LEVEL
#                           SEVERE_LEVEL
#                           WARNING_LEVEL
#                           INFO_LEVEL
#                           FINE_LEVEL
#
# return: int value
sub getLevel
{
    my $logLevel  = uc shift;

    if ( $logLevel =~ /^[+-]?\d+$/ 
    and FINE_LEVEL <= $logLevel and $logLevel <= INCIDENT_LEVEL)
    {
        # No conversion necessary - already valid
        return $logLevel;
    }

    if ( 'INCIDENT_LEVEL'   eq $logLevel )
    {
        return INCIDENT_LEVEL;
    }
    elsif ( 'SEVERE_LEVEL'  eq $logLevel )
    {
        return SEVERE_LEVEL;
    }
    elsif ( 'WARNING_LEVEL' eq $logLevel )
    {
        return WARNING_LEVEL;
    }
    elsif ( 'INFO_LEVEL'    eq $logLevel )
    {
        return INFO_LEVEL;
    }
    elsif ( 'FINE_LEVEL'    eq $logLevel )
    {
        return FINE_LEVEL;
    }
    elsif ( FINE_LEVEL      eq $logLevel )
    {
        return FINE_LEVEL;
    }

    print STDERR "WARNING: logLevel=$logLevel.\n";
    print STDERR "Expected one of 'INCIDENT_LEVEL', 'SEVERE_LEVEL', 'WARNING_LEVEL', 'INFO_LEVEL' or 'FINE_LEVEL'\n";

    return FINE_LEVEL;
}


# Convert string visibility to int.
#
# params: visible         - who can see this message? 
#                           VISIBLE_SUPPORT_ONLY
#                           VISIBLE_USER
#                           VISIBLE_SYSTEM
#
# return: int value
sub getVisibility
{
    my $visible   = uc shift;

    if ( $visible =~ /^[+-]?\d+$/ 
    and VISIBLE_SUPPORT_ONLY <= $visible and $visible <= VISIBLE_SYSTEM )
    {
        # No conversion necessary - already valid
        return $visible;
    }

    if ( 'VISIBLE_SUPPORT_ONLY' eq $visible )
    {
        return VISIBLE_SUPPORT_ONLY;
    }
    elsif ( 'VISIBLE_USER' eq $visible )
    {
        return VISIBLE_USER;
    }
    elsif ( 'VISIBLE_SYSTEM' eq $visible )
    {
        return VISIBLE_SYSTEM;
    }

    print STDERR "WARNING: visible=$visible.\n";
    print STDERR "Expected one of 'VISIBLE_SUPPORT_ONLY', 'VISIBLE_USER' or 'VISIBLE_SYSTEM'\n";
    $visible = VISIBLE_SUPPORT_ONLY;
}


# Log a message at logLevel.
#
# NOTE: This API is provided for completeness.  
# It is preferable to use the level specific APIs.
#
# The log string will be stored at logLevel
# as part of the object logging for this step.
#
# If visible is set to 'VISIBLE_SYSTEM', the message will also be stored in the 
# system error table.
#
# The Oracle Standard is that messages at warning and above be translated.
# 
# params: errorCode       - the ORA error number or 'null'
#         logLevel        - the log level for this message
#                           INCIDENT_LEVEL
#                           SEVERE_LEVEL
#                           WARNING_LEVEL
#                           INFO_LEVEL
#                           FINE_LEVEL
#         visible         - who can see this message? 
#                           VISIBLE_SUPPORT_ONLY
#                           VISIBLE_USER
#                           VISIBLE_SYSTEM
#         message         - the user visible message
#         messageId       - the message id for a translated message
#         messageBundle   - the bundle for a translated message
#         messageParams   - the replacement params for a translated message
#
# return: NONE
sub log
{
    my $errorCode = shift;
    my $logLevel  = shift;
    my $visible   = shift;

    $logLevel     = &getLevel($logLevel);
    $visible      = &getVisibility($visible);

    &printCommandBlock("EM_JOB_DIAG.log",
                       &getCommandBlockArg($errorCode),
                       $logLevel,
                       $visible,
                       &getMessageInfo(@_));
}


# Log a VISIBLE_SYSTEM incident / system error messages.
#
# NOTE: This API is provided for completeness.  
# It is preferable to use the triggerIncident APIs when incidents arise.
#
# The log string will be stored at incident level
# as part of the object logging for this step and in the system error table.
#
# The message should provide translation information.
# The Oracle Standard is that messages at warning and above be translated.
# 
# params: message         - the user visible message
#         messageId       - the message id for a translated message
#         messageBundle   - the bundle for a translated message
#         messageParams   - the replacement params for a translated message
#
# return: NONE
sub logIncident
{
    &log(NULL_ARG, INCIDENT_LEVEL, VISIBLE_SYSTEM, @_);
}


# Log a VISIBLE_USER message at logLevel = SEVERE_LEVEL.
#
# params: message         - the user visible message
#         messageId       - the message id for a translated message
#         messageBundle   - the bundle for a translated message
#         messageParams   - the replacement params for a translated message
#
# See log() for details.
sub logSevere
{
    &log(NULL_ARG, SEVERE_LEVEL, VISIBLE_USER, @_);
}

# Log a VISIBLE_USER message at logLevel = WARNING_LEVEL.
#
# params: message         - the user visible message
#         messageId       - the message id for a translated message
#         messageBundle   - the bundle for a translated message
#         messageParams   - the replacement params for a translated message
#
# See log() for details.
sub logWarning
{
    &log(NULL_ARG, WARNING_LEVEL, VISIBLE_USER, @_);
}

# Log a VISIBLE_USER message at logLevel = INFO_LEVEL.
#
# params: message         - the user visible message
#         messageId       - the message id for a translated message
#         messageBundle   - the bundle for a translated message
#         messageParams   - the replacement params for a translated message
#
# See log() for details.
sub logInfo
{
    &log(NULL_ARG, INFO_LEVEL, VISIBLE_USER, @_);
}

# Log a VISIBLE_USER message at logLevel = FINE_LEVEL (debug).
#
# params: message         - the user visible message
#         messageId       - the message id for a translated message
#         messageBundle   - the bundle for a translated message
#         messageParams   - the replacement params for a translated message
#
# See log() for details.
sub logFine
{
    &log(NULL_ARG, FINE_LEVEL, VISIBLE_USER, @_);
}

