#!/usr/local/bin/perl
# 
# $Header: emagent/sysman/admin/scripts/jobutil/JobUtil.pm /main/2 2011/01/18 00:09:47 amathur Exp $
#
# JobUtil.pm
# 
# Copyright (c) 2010, 2011, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      JobUtil.pm - Perl helper APIs for job system
#
#    DESCRIPTION
#      Common APIs for Job and PAF Integrators
#
#    APIs
#      getInputStreamParamsMap
#          Get the parameters passed on the STDIN as a map
#
#      extractParamAndValue
#          Parse a line and split into param name and param value based on a delimiter
#
#      formSQLPlusConnectString
#          Form the Connection string to logged into SQL*Plus
#
#    OPEN ISSUES
#      None
#
#    NOTES
#      Project 28096
#
#
#    MODIFIED   (MM/DD/YY)
#    amathur     02/04/10 - Creation
# 

package JobUtil;

use strict;
use JobDiagUtil;

require 5.005;

@JobUtil::EXPORT = qw (
    &processExecReturn
    &getInputStreamParamsMap
    &extractParamAndValue
    &formSQLPlusConnectString
);

# array of the possible clauses of password-quoting failure from MGMT_JOB_FUNCTIONS
use constant INVALID_PASSWORD_REASON => [
    "empty password",
    "password is too long",
    "neither quoted nor unquoted password can contain double quotation or nul",
    "unmatched quotation mark" 
];

# Process the return of a process and exit from the script or program
#
# Arg: <operation name> <exit code> <error received>
#   requires the operation name for logging prupose
#   requires the exit code of the process
#   requires the error string generated by the process. Usually this can be caught by "$!"
#
# Sample: processExecReturn("zip_creation", 9, "$!");
#
sub processExecReturn
{
    my($operation, $syscode, $error) = @_;
    
    JobDiagUtil::logFine("[$operation] Return code received : $syscode : error string : $error");
    
    $syscode = 0xFFFF & $syscode;
    if ($syscode != 0)
    {
        my $sig = 0xFF & $syscode;
        my $ret = $syscode >> 8;

        if ($sig != 0)
        {
            print "Signal received : $sig. \n";
            exit (- $sig);
            
        }
        else
        {
            print "Process returned : $ret. $error \n";
            exit $ret;
            
        }
    }
}


# Get the parameters passed on the STDIN as a map
# 
# Arg: <delimiter> <EOS String>
#   requires the delimiter string or character between param name and param value
#   requires End Of Stream string to stop parsing
#
# Sample: getInputStreamParamsMap("=","__END_OF_STREAM__")
#
sub getInputStreamParamsMap
{
    my $delim     = shift or die("Require delimiter.");;
    my $EOS       = shift or die("Require end-of-stream string.");
    my %paramsMap = ();

    while(<STDIN>)
    {
        # truncate the line
        chomp;

        # trim the line
        $_ = trim($_);
        
        # end the loop
        last if (/^$EOS$/);

        # skip is blank line
        next if (/^$/);
        
        # enable to debug
        # print "I read==".$_."==\n";

        # extract the paramname and param value
        my ($paramName, $paramValue) = extractParamAndValue($_, $delim);

        # add to map
        $paramsMap{$paramName} = $paramValue;
    }

    return %paramsMap;
}

# Parse a line and split into param name and param value based on a delimiter
#
# Arg: <line> <delimiter>
#    requires to line to parse
#    requires the delimiter string or charaxter to search in the line. Only first instance of the delimiter is checked
#    If the delimter is not provided, then "=" is used as the delimiter
#
# Sample: extractParamAndValue("paramname=paramvalue","=");
#
sub extractParamAndValue
{
    my $line  = shift or die("Require input line.");
    my $delim = shift or "=";

    my $name  = "";
    my $value = "";

    my $pos = index($line, $delim);
    if ($pos != -1)
    {
        $name = substr($line, 0, $pos);
        $value = substr($line, $pos + 1);
    }

    return ($name, $value);
}


# Form the Connect string to logged into SQL*Plus
#
# Arg: <username> <password> [dbrole]
#    requires SQL*Plus username
#    requires SQL*Plus password
#    if dbrole is provided, then it is converted to upper case
#
# Sample:
#    with DB Role: formSQLPlusConnectionString("sys","syspwd","sys")
#    without DB Role: formSQLPlusConnectionString("sysman","sysman")
#
sub formSQLPlusConnectString
{
    my $userName = shift or die("Require SQL username.");
    my $password = shift or die("Require SQL password.");
    my $role     = shift or "";

    $role     = fixDBRole($role);
    $password = quotePassword($password);

    return "connect ".$userName."/".$password.$role;
}

# Append the "AS" keyword to the DB Role and upper case the DB Role
# 
# Arg: [dbrole]
#    if dbrole is not provided, an empty string is considered and returned
#
# Sample: fixDBRole("sysman")
#
sub fixDBRole
{
    my $role = shift or return "";

    if ($role =~ /^$/ || $role =~ /normal/i)
    {
        return "";
    }
    else
    {
        $role = uc $role;
        return " AS ".$role;
    }
}

# Escape and quote the password for consumption by SQL*Plus
# Note: replicating logic from MGMT_JOB_FUNCTIONS.quote_password_scalar()
# Internal API, not exposed to outside world
sub quotePassword
{
    my $err_num = 0;
    my $password    = shift or goto ERROR_HANDLER;

    # no password or zero length password
    if ($password =~ /^$/)
    {
        $err_num = 0;
        goto ERROR_HANDLER;
    }

    my $len = length($password);
    my $str_trim = substr($password, 1, $len - 2);

    # cannot contain double quotations or null
    if ( (index($str_trim, "\"") != -1)
         || (index($str_trim, chr(0)) != -1) )
    {
        $err_num = 2;
        goto ERROR_HANDLER;
    }

    my $str_start = substr($password, 0, 1);
    my $str_end   = substr($password, $len - 1);

    #if first and last characters are quotes
    if ( ($str_start eq "\"") 
         && ($str_end eq "\"") )
    {
        # if length is less than 3
        if ($len < 3)
        {
            if ($len == 1)
            {
                # too small, has one single quote mark
                $err_num = 3;
            }
            else
            {
                # too small, has no data bewtween quotes
                $err_num = 0;
            }
            goto ERROR_HANDLER;
        }
        else
        {
            # return the same password, it is already quoted
            return $password;
        }
    }
    else
    {
        # first and last characters are not null
        if ( ($str_start ne "\"")
             && ($str_end ne "\""))
        {
            # quote and return
            return "\"".$password."\"";
        }
        else
        {
            # password needs to be quoted both at the beginning and the end
            $err_num = 3;
            goto ERROR_HANDLER;
        }
    }

    # placate compiler
    return "";

ERROR_HANDLER:
    my $error_string = "Invalid database password: ".INVALID_PASSWORD_REASON->[$err_num];
    JobDiagUtil::logSevere("Unable to quote password : $error_string");
    die($error_string);
}


# trim left and right spaces from a line
sub trim
{
    my $string = shift or return "";

    $string =~ s/^\s+//;;
    $string =~ s/\s+$//;

    return $string;
}

1;
