#!/usr/local/bin/perl
# 
# $Header: emagent/sysman/admin/scripts/jobutil/filetransfer.pl /main/1 2011/01/18 00:09:47 amathur Exp $
#
# filetransfer.pl
# 
# Copyright (c) 2011, Oracle and/or its affiliates. All rights reserved. 
#
#    NAME
#      filetransfer.pl 
#
#    DESCRIPTION
#      Scripts to assist in the file tranfer operation
#
#    NOTES
#      Called from the FileTransfer job type.
#      Requires: JobDiagUtil.pl
#      Requires: JobUtil.pl
#      Reads all parameters only from the input stream
#
#    MODIFIED   (MM/DD/YY)
#    amathur     01/10/11 - code for filetransfer job type
#    amathur     01/10/11 - Creation
# 
use strict;
use JobDiagUtil;
use JobUtil;
use IO::File;
use File::Path;

# constant to indicate stop reading of the input stream
use constant EOS        => "__EM_JOB_EOS__";

# constant to indicate delimition between name and value pairs on the input stream
use constant DELIMITER  => "=";

# constant to denote a destination file
use constant DEST_FILE  => "__EM_JOB_DEST_FILE__";

# constant to denote a destination directory
use constant DEST_DIR   => "__EM_JOB_DEST_DIR__";

# constant to denote source path which needs to be zipped
use constant SRC_PATH   => "__EM_JOB_SRC_PATH__";

# constant to denote the temporary location where zip can be created
use constant STAGE_LOC  => "__EM_JOB_STAGE_LOC__";

# constant to denote an archive file
use constant ARCH_FILE  => "__EM_JOB_ARCH_FILE__";

# constant to denote zip executable
use constant ZIP_EXEC   => "__EM_JOB__ZIP_EXEC__";

# constant to denote the unzip executable
use constant UNZIP_EXEC => "__EM_JOB_UNZIP_EXEC__";

# constant to denote whether the extraction of archive file would be destructive or not
use constant OVERWRITE_FILES => "__EM_JOB_OVERWRITE_FILES__";

# operational code for deleting a file
use constant OPCODE_DEL_FILE   => "-delFile";

# operational code for deleting a directory
use constant OPCODE_DEL_DIR    => "-delDir";

# operational code for creating a directory
use constant OPCODE_CRT_DIR    => "-createDir";

# operational code for zipping a directory
use constant OPCODE_ZIP_DIR    => "-zipDir";

# operational code for unzipping a file
use constant OPCODE_UNZIP_FILE => "-unzipFile";


# get the operation code
my $operationalCode = "$ARGV[0]";

# get the parameters -- mandatorily all params need to be passed on the Input Stream
my %paramsMap = getInputParams();

# call sub routines based on the op-code
if ($operationalCode eq OPCODE_DEL_FILE)
{
    delFile();
}
elsif ($operationalCode eq OPCODE_DEL_DIR)
{
    delDirectory();
}
elsif ($operationalCode eq OPCODE_CRT_DIR)
{
    createDirectory();
}
elsif ($operationalCode eq OPCODE_ZIP_DIR)
{
    zip();
}
elsif ($operationalCode eq OPCODE_UNZIP_FILE)
{
    unzip();
}

# get the input stream consumables in a hashmaps
sub getInputParams
{
    return JobUtil::getInputStreamParamsMap(DELIMITER, EOS);
}

# delete a file
sub delFile
{
    my $destFilepath = $paramsMap{+DEST_FILE};

    JobDiagUtil::logFine("File to delete : $destFilepath");

    if (-e $destFilepath)
    {
        if (-w $destFilepath)
        {
            unlink "$destFilepath" ;
            if (! -e $destFilepath)
            {
                print "Deleted file : $destFilepath\n";
            }
        }
        else
        {
            JobDiagUtil::logSevere("$destFilepath is not writable and hence, cannot delete it.");
        }
    }
    else
    {
        JobDiagUtil::logFine("$destFilepath does not exists.");
    }
}

# delete a directory
sub delDirectory
{
    my $destPath  = $paramsMap{+DEST_DIR};

    JobDiagUtil::logFile("Trying to delete directory : $destPath");

    if (-e $destPath)
    {
        rmtree($destPath, 1, 0);
        if (! -d $destPath)
        {
            print "Deleted directory and contents from $destPath \n";
        }
        else
        {
            JobDiagUtil::Severe("Unable to delete directory $destPath");
        }
    }
}

# internal not exposed using command line param
sub createDirectoryInt
{
    my $stagingArea = shift;

    JobDiagUtil::logFine("Trying to create directory $stagingArea");

    if (! -d $stagingArea)
    {
        mkpath "$stagingArea";
        chmod 01750, "$stagingArea";
        if (-d $stagingArea)
        {
            print "Created $stagingArea \n";
        }
        else
        {
            die "Failed to create $stagingArea";
        }
    }
    else
    {
        if (! -w $stagingArea)
        {
           die "$stagingArea is not writable.";
        }
    }
}

# create a directory
sub createDirectory
{
    my $stagingArea  = $paramsMap{+STAGE_LOC};

    createDirectoryInt($stagingArea);
}


# zip a directory and create the zip file in a staging location
sub zip
{
    my $sourcePath  = $paramsMap{+SRC_PATH};
    my $stagingArea = $paramsMap{+STAGE_LOC};
    my $archFile    = $paramsMap{+ARCH_FILE};
    my $zipExec     = $paramsMap{+ZIP_EXEC};

    # check that the source directory for zip is present and readable.
    if (!(-d $sourcePath && -r $sourcePath))
    {
        die "$sourcePath does not exists or is not readable.";
    }
    chdir "$sourcePath";

    # check if staging area is created, if not create it
    createDirectoryInt($stagingArea);
    
    my $archFilePath = $stagingArea . "/$archFile";
    my @cmd = ($zipExec, "-rv", $archFilePath, ".");

    # Print diagnostics
    JobDiagUtil::logFine <<DEBUG_ZIP_PRINTS;
        SRC_PATH : $sourcePath
        STAGING_AREA : $stagingArea
        ARCH_FILE : $archFile
        ZIP_EXEC : $zipExec
        ARCH_FILE_PATH : $archFilePath
        ZIP_CMD : @cmd
DEBUG_ZIP_PRINTS

    print "Executing command @cmd\n";
    my $syscode = system(@cmd);
    my $error = $!;
    JobUtil::processExecReturn("CREATE_ZIP", $syscode , $error);
}


# unzip the arhive in a destination location
sub unzip
{
    my $destPath     = $paramsMap{+DEST_DIR};
    my $destArchFile = $paramsMap{+ARCH_FILE};
    my $unzipExec    = $paramsMap{+UNZIP_EXEC};
    my $overWrite    = $paramsMap{+OVERWRITE_FILES};

    # if the archive file does not exist or is not readable, then error out
    if (! (-e $destArchFile) && (-r $destArchFile) )
    {
        die "$destArchFile is either does not exists or is not readable.\n";
    }

    my @cmd = ($unzipExec, "-d", $destPath);
    if ($overWrite eq "true")
    {
        push(@cmd, "-o");
    }
    push(@cmd, $destArchFile);

    JobDiagUtil::logFine <<DEBUG_UNZIP_PRINTS;
        OVERWRITE : $overWrite
        DEST_PATH : $destPath
        DEST_ARCH_FILE : $destArchFile
        UNZIP_EXEC : $unzipExec
        UNZIP_CMD : @cmd
DEBUG_UNZIP_PRINTS

    print "Executing command @cmd\n";
    my $syscode = system(@cmd);
    my $error = $!;
    JobUtil::processExecReturn("EXTRACT_ZIP", $syscode, $error);
}
